#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import ConfigMount
import LagIntfCli
import LazyMount
import Tac
import CliPlugin.BfdGlobalConfigMode as BfdGlobalConfigMode
import CliPlugin.IntfCli as IntfCli
import CliPlugin.BfdCli as BfdCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.VirtualIntfRule import IntfMatcher

perlinkMode = Tac.Type( "Bfd::PerlinkMode" )
allIntfStatusDir = None
bfdConfigIntf = None
bridgingHwCapabilities = None
bfdStatusPeer = None
bfdConfigGlobal = None

def IntfId( intfName ):
   return Tac.Value( 'Arnet::IntfId', intfName )

class LagBfdIpIntf ( IntfCli.IntfDependentBase ):
   def __init__( self, intf, sysdbRoot, createIfMissing=True ):
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   def setDefault( self ):
      intf = IntfId( self.intf_.name )
      if intf in bfdConfigIntf.perLink:
         del bfdConfigIntf.perLink[ intf ]
      if intf in bfdConfigIntf.neighborAddrIntfConfig:
         del bfdConfigIntf.neighborAddrIntfConfig[ intf ]
      if intf in bfdConfigIntf.rfc7130Interop:
         del bfdConfigIntf.rfc7130Interop[ intf ]

def addOrRemoveWarningForPerlink( mode, perlink, add=True ):
   if add:
      if perlink == perlinkMode.rfc7130 and \
         not bridgingHwCapabilities.rfc7130Supported and \
         ( not BfdCli.rfc7130SupportedWarning.get( mode.intf.name, False ) or \
         ( bfdConfigIntf.perLink.get( mode.intf.name ) != perlink ) ):
         BfdCli.rfc7130SupportedWarning[ mode.intf.name ] = True
         mode.addWarning( "bfd per-link rfc7130 configuration will be ignored "
                 "because the hardware platform does not support rfc7130." )
         return
      if perlink == perlinkMode.rfc7130 and \
         mode.intf.name not in bfdConfigIntf.neighborAddrIntfConfig and \
         ( not BfdCli.rfc7130BfdNeighborWarning.get( mode.intf.name, False ) or \
         ( bfdConfigIntf.perLink.get( mode.intf.name ) != perlink ) ):
         BfdCli.rfc7130BfdNeighborWarning[ mode.intf.name ] = True
         mode.addWarning( "bfd per-link rfc7130 configuration will be ignored "
                          "until bfd neighbor is configured on interface %s."
                          % mode.intf.name )
         return
      intfStatus = allIntfStatusDir.intfStatus.get( mode.intf.name )
      if intfStatus and intfStatus.forwardingModel != "intfForwardingModelRouted" \
         and perlink == perlinkMode.legacy \
         and ( not BfdCli.routedPortWarning.get( mode.intf.name, False ) or \
               ( bfdConfigIntf.perLink.get( mode.intf.name ) != perlink ) ):
         BfdCli.routedPortWarning[ mode.intf.name ] = True
         mode.addWarning( "bfd per-link configuration will be ignored "
                          "while interface %s is not a routed port." % \
                          ( mode.intf.name ) )
         return
      modeStr = " rfc7130" if perlink == perlinkMode.rfc7130 else ""
      peerStatus = bfdStatusPeer.intfToPeerMap.get( mode.intf.name )
      if peerStatus and peerStatus.peerList\
         and ( not BfdCli.existingSessionWarning.get( mode.intf.name, False ) or \
               ( bfdConfigIntf.perLink.get( mode.intf.name ) != perlink ) ):
         BfdCli.existingSessionWarning[ mode.intf.name ] = True
         mode.addWarning( "bfd per-link%s configuration will supersede "
                          "existing BFD session on interface %s." % \
                          ( modeStr, mode.intf.name ) )
   else:
      BfdCli.rfc7130SupportedWarning[ mode.intf.name ] = False
      BfdCli.rfc7130BfdNeighborWarning[ mode.intf.name ] = False
      if perlink == perlinkMode.rfc7130:
         if bfdConfigIntf.perLink.get( mode.intf.name ) == perlinkMode.rfc7130:
            BfdCli.existingSessionWarning[ mode.intf.name ] = False
      else:
         BfdCli.routedPortWarning[ mode.intf.name ] = False
         BfdCli.existingSessionWarning[ mode.intf.name ] = False

#------------------------------------------------------------------------------
# "[no|default] bfd per-link [rfc-7130]"
# command in Port-Channel config-if mode
#------------------------------------------------------------------------------

def perlinkFunctionCmd( mode, no=False, rfc7130=None ):
   intf = IntfId( mode.intf.name )
   if not no:
      if rfc7130:
         perlink = perlinkMode.rfc7130
      else:
         perlink = perlinkMode.legacy
      addOrRemoveWarningForPerlink( mode, perlink, add=True )
      bfdConfigIntf.perLink[ intf ] = perlink
   else:
      if rfc7130:
         perlink = perlinkMode.rfc7130
      else:
         perlink = perlinkMode.none
      if intf in bfdConfigIntf.perLink:
         addOrRemoveWarningForPerlink( mode, perlink, add=False )
         if perlink == perlinkMode.rfc7130:
            if bfdConfigIntf.perLink[ intf ] == perlink:
               del bfdConfigIntf.perLink[ intf ]
         else:
            del bfdConfigIntf.perLink[ intf ]

class BfdPerLinkCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd per-link [ rfc-7130 ]'
   noOrDefaultSyntax = syntax
   data = {
      'bfd': BfdCli.matcherBfd,
      'per-link': 'Configure BFD per-link',
      'rfc-7130': 'RFC 7130 compliant',
   }

   @staticmethod
   def handler( mode, args ):
      perlinkFunctionCmd( mode, rfc7130='rfc-7130' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      perlinkFunctionCmd( mode, no=True, rfc7130='rfc-7130' in args )

lagIntfConfigModelet = LagIntfCli.LagIntfConfigModelet
lagIntfConfigModelet.addCommandClass( BfdPerLinkCmd )

intfMatcher = IntfMatcher()
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfMatcher |= LagIntfCli.subMatcher

class BfdLagStaticSessionCmd( CliCommand.CliCommandClass ):
   _baseSyntax = 'session PEERADDR INTF VRF'
   syntax = 'test ' + _baseSyntax
   noOrDefaultSyntax = syntax

   data = {
            'test': 'BFD test harness',
            'session': BfdCli.matcherSession,
            'PEERADDR': IpGenAddrMatcher( helpdesc='IPv4/IPv6 Neighbor address' ),
            'INTF': intfMatcher,
            'VRF': CliMatcher.DynamicNameMatcher( BfdCli.getVrfNames, 'VRF name' ),
          }

   handler = BfdCli.addOrRemoveBfdSession
   noOrDefaultHandler = handler
   hidden = True

BfdGlobalConfigMode.RouterBfdMode.addCommandClass( BfdLagStaticSessionCmd )

class BfdLagStaticSessionDeprecatedCmd( BfdLagStaticSessionCmd ):
   syntax = 'bfd ' + BfdLagStaticSessionCmd._baseSyntax
   noOrDefaultSyntax = syntax
   data = BfdLagStaticSessionCmd.data.copy()
   data.update( { 'bfd': BfdCli.matcherBfd } )
   hidden = True
   handler = BfdLagStaticSessionCmd.handler
   noOrDefaultHandler = BfdLagStaticSessionCmd.noOrDefaultHandler

BfdGlobalConfigMode.RouterBfdMode.addCommandClass( BfdLagStaticSessionDeprecatedCmd )

#------------------------------------------------------------------------------
# "[no|default] bfd rfc-7130 interop"
# command in Port-Channel config-if mode
#------------------------------------------------------------------------------
def setRfc7130Interop( mode, args ):
   intf = IntfId( mode.intf.name )
   bfdConfigIntf.rfc7130Interop[ intf ] = True

def noRfc7130Interop( mode, args ):
   intf = IntfId( mode.intf.name )
   if intf in bfdConfigIntf.rfc7130Interop:
      del bfdConfigIntf.rfc7130Interop[ intf ]

class BfdRfc7130InteropCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd rfc-7130 interop'
   noOrDefaultSyntax = 'bfd rfc-7130 ...'
   data = {
      'bfd': BfdCli.matcherBfd,
      'rfc-7130': 'RFC 7130 compliant',
      'interop': ' Set BFD rfc-7130 interop mode'
   }

   handler = setRfc7130Interop
   noOrDefaultHandler = noRfc7130Interop

lagIntfConfigModelet = LagIntfCli.LagIntfConfigModelet
lagIntfConfigModelet.addCommandClass( BfdRfc7130InteropCmd )

#------------------------------------------------------------------------------
# "[no|default] bfd neighbor <A.B.C.D>"
# command in Port-Channel config-if mode
#------------------------------------------------------------------------------
def bfdNeighFunctionHelper( mode, no=False, address=None ):
   if address is None:
      address = Tac.newInstance( 'Arnet::IpGenAddr', '' )
   
   intf = IntfId( mode.intf.name )
   if not address.isAddrZero:
      if no and ( intf not in bfdConfigIntf.neighborAddrIntfConfig ):
         return
      if intf not in bfdConfigIntf.neighborAddrIntfConfig:
         bfdConfigIntf.neighborAddrIntfConfig.newMember( intf )
      neighAddrConf = bfdConfigIntf.neighborAddrIntfConfig[ intf ]
      addrFamily = address.af
      if addrFamily == 'ipv4':
         if no:
            if address == neighAddrConf.neighborAddrV4:
               neighAddrConf.neighborAddrV4 = Tac.Value( 'Arnet::IpGenAddr', '' )
            else:
               mode.addWarning( "bfd neighbor %s not configured on intf %s."
                 % ( address.stringValue, mode.intf.name ) )
         else:
            neighAddrConf.neighborAddrV4 = address
               
      elif addrFamily == 'ipv6':
         if no:
            if address == neighAddrConf.neighborAddrV6:
               neighAddrConf.neighborAddrV6 = Tac.Value( 'Arnet::IpGenAddr', '' )
            else:
               mode.addWarning( "bfd neighbor %s not configured on intf %s."
                 % ( address.stringValue, mode.intf.name ) )
         else:
            neighAddrConf.neighborAddrV6 = address

      if neighAddrConf.neighborAddrV4.isAddrZero and \
         neighAddrConf.neighborAddrV6.isAddrZero:
         del bfdConfigIntf.neighborAddrIntfConfig[ intf ]
   else:
      # if no addr is given for "bfd neighbor" command, we remove configured
      # bfd neighbor address for both V4 and V6
      if intf in bfdConfigIntf.neighborAddrIntfConfig:
         del bfdConfigIntf.neighborAddrIntfConfig[ intf ]
   
def neighAddrFunctionCmd( mode, args ):
   bfdNeighFunctionHelper( mode, address=args[ 'PEERADDR' ] )

def neighAddrNoFunctionCmd( mode, args ):
   bfdNeighFunctionHelper( mode, address=args.get( 'PEERADDR' ), no=True )

class BfdLagNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd neighbor PEERADDR'
   noOrDefaultSyntax = 'bfd neighbor [ PEERADDR ]'
   data = {
      'bfd': BfdCli.matcherBfd,
      'neighbor': 'Configure BFD neighbor ip address',
      'PEERADDR': IpGenAddrMatcher( helpdesc='IPv4 address' ),
   }

   handler = neighAddrFunctionCmd
   noOrDefaultHandler = neighAddrNoFunctionCmd

lagIntfConfigModelet.addCommandClass( BfdLagNeighborCmd )

def Plugin( entityManager ):
   global allIntfStatusDir
   global bfdConfigIntf
   global bridgingHwCapabilities
   global bfdStatusPeer
   global bfdConfigGlobal

   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   bfdConfigIntf = ConfigMount.mount( entityManager, 'bfd/config/intf', 
                                      'Bfd::ConfigIntf', 'w')
   IntfCli.Intf.registerDependentClass( LagBfdIpIntf )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   bfdStatusPeer = LazyMount.mount( entityManager, 'bfd/status/peer',
                                    'Bfd::StatusPeer', 'r')
   bfdConfigGlobal = ConfigMount.mount( entityManager, 'bfd/config/global',
                                        'Bfd::ConfigGlobal', 'w' )
