#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import ArPyUtils
import CliCommand, CliMatcher, ShowCommand
import CommonGuards
import Tac, BasicCli, EventMonCli
import sys
from TableOutput import Format, createTable
from EthIntfCli import EthPhyAutoIntfType
from Intf.IntfRange import IntfRangeMatcher
from LagIntfCli import LagAutoIntfType
import CliPlugin.TechSupportCli

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2016-11-28 15:23:24',
   lambda: [ 'show event-monitor lacp verbose match-time last-hour' ] )

def showLacpTable( mode, args ):
   if 'verbose' in args:
      showLacpTableVerbose( mode, args )
   else:
      EventMonCli.showTable( mode, args, "lacp" )

def showLacpTableVerbose( mode, args ):
   lacpEventMonReason = Tac.Value( "Lacp::LacpEventMonReason" )
   t = showLacpTableFormat()

   with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) as capturedOut:
      EventMonCli.showTable( mode, args, "lacp" )
      sys.stdout.flush()

   output = capturedOut.contents().split( '\n' )[ : -1 ]
      
   for entry in output:
      originalFields = entry.split( "|" )
      finalFields = {}
      fieldHeaders = [ "time", "intf", "portChannel",  "sm", "state", 
                       "reason", "portEn", "lacpEn", "currEx", "lacpduRx", 
                       "portMoved", "selected", "partnerSync", "count" ]
      if len( originalFields ) != len( fieldHeaders ):
         # Output does not contain proper fields, so it is most likely an
         # error message so we will print the error message and exit
         print '\n'.join( output )
         return
      for field, header in zip( originalFields, fieldHeaders ):
         if header == "reason":
            key = int( field )
            if finalFields[ "sm" ] == "rx":
               finalFields[ header ] = lacpEventMonReason.rxSmReason[ key ]
            elif finalFields[ "sm" ] == "mux":
               finalFields[ header ] = lacpEventMonReason.muxSmReason[ key ]
         elif header == "intf":
            finalFields[ header ] = field.replace( "Ethernet", "Et" )
         elif header == "portChannel":
            if field == "unknown":
               finalFields[ header ] = ""
            else:
               finalFields[ header ] = field.replace( "Port-Channel", "Po" )
         elif header == "state":
            if finalFields[ "sm" ] == "rx":
               finalFields[ header ] = field.replace( "rxSm", "" )
            elif finalFields[ "sm" ] == "mux":
               finalFields[ header ] = field.replace( "muxSm", "" )
         else:
            finalFields[ header ] = field
      t.newRow( finalFields[ "time" ], finalFields[ "intf" ], 
                finalFields[ "portChannel" ], finalFields[ "state" ], 
                finalFields[ "reason" ] )

   print t.output()

def showLacpTableFormat():
   t = createTable( [ "Time", "Port", "LAG", "State", "Reason" ] )
   fHeading = Format( justify="left", isHeading=True, noBreak=False )
   fHeading.noPadLeftIs( True )
   fState = Format( justify="left", noBreak=False )
   fState.noPadLeftIs( True )
   fReason = Format( justify="left", noBreak=False, wrap=True, minWidth=20 )
   fReason.noPadLeftIs( True )
   t.formatColumns( fHeading, fHeading, fHeading, fState, fReason )
   return t

# LACP history table can be grouped by interface or state-machine
lacpGroupMap = { 'member':'intf', 'state-machine':'statemachine', 
                 'port-channel':'portchannel' }

class EventMonitorLacpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show event-monitor lacp [ verbose ] '
              '[ { ( group-by GROUPBY ) | ( limit LIMIT ) | ' 
              '( match-time TIME ) | '
              '( match-interface ( ETH | LAG ) ) | '
              '( match-sm MATCH_SM ) } ]' )
   data = {
      'event-monitor' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'event-monitor',
            helpdesc='Show event monitor logs' ),
         guard=CommonGuards.standbyGuard ),
      'lacp' : 'Monitor LACP table events',
      'verbose' : 'Print verbose output',
      'group-by' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'group-by',
            helpdesc='Group the results by attribute' ),
         maxMatches=1 ),
      'GROUPBY' : CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'member': 'Group by member',
            'port-channel': 'Group by port-channel',
            'state-machine': 'Group by state machine',
         } ),
         maxMatches=1 ),
      'limit' : CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 'limit',
            helpdesc='Limit the number of messages' ),
         maxMatches=1 ),
      'LIMIT' : CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 15000, helpdesc='Message limit' ),
         maxMatches=1 ),
      'match-time' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'match-time',
            helpdesc='Filter results by time' ),
         maxMatches=1 ),
      'TIME' : CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'last-minute': 'Match for the last minute',
            'last-hour': 'Match for the last hour',
            'last-day': 'Match for the last day',
            'last-week': 'Match for the last week',
         } ),
         maxMatches=1 ),
      'match-interface': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'match-interface',
            helpdesc='Filter results by intf' ),
         maxMatches=2 ),
      'ETH' : CliCommand.Node(
         matcher=IntfRangeMatcher( explicitIntfTypes=set( [ EthPhyAutoIntfType ] ) ),
         maxMatches=1 ), 
      'LAG' : CliCommand.Node(
         matcher=IntfRangeMatcher( explicitIntfTypes=set( [ LagAutoIntfType ] ) ),
         maxMatches=1 ),
      'match-sm' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'match-sm',
            helpdesc='Filter results by state machine name' ),
         maxMatches=1 ),
      'MATCH_SM' : CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'mux': 'Mux state machine',
            'rx': 'Receive state machine',
         } ),
         maxMatches=1 ),
   }
   handler = showLacpTable
   privileged = True

BasicCli.addShowCommandClass( EventMonitorLacpCmd )
