#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Contains the CLI model for L2RIB "show commands" ( flood-set/host table ).
import Ethernet
import Tac
from CliModel import (
   Model,
   Int,
   Enum,
   Bool,
   Str,
   List,
   DeferredModel,
   Submodel )
from ArnetModel import (
   IpGenericAddr,
   IpGenericPrefix,
   MacAddress )
from IntfModels import Interface
from CliPlugin.TunnelModels import TunnelId
from CliPlugin.BridgingCliModel import _BaseTableEntry
from TableOutput import createTable, Format

# Keep a global instance to get the string representation for host
# entry type.
hostEntryTypeHelper = Tac.newInstance( 'Bridging::EntryTypeHelper' )

class TableSummary( DeferredModel ):
   'Describes host entries from one source'
   name = Str( help='Table name' )
   hosts = Int( help='Number of hosts in table' )

class Summary( DeferredModel ):
   'Descibes host entries from all known sources.'
   tables = List( valueType=TableSummary, help='Number of entries in each table' )

class L2RibObject( Model ):
   # This is the base class for all L2RIB objects ( Dest, LoadBalance
   # and Label )
   spacePerLevel = Int( help='Number of spaces to indent per level',
                        default=3 )
   # There is no index for CiDests
   index = Int( help='Object table key', optional=True )
   level = Int( help='Level in the chain of objects' )
   def getStr( self, detail ):
      assert False, "Derived class must implement this for rendering"

class LoadBalance( L2RibObject ):
   """Describes a load balance entry attributes and its strep."""
   num = Int( help='Number of load balance objects' )

   def getStr( self, detail ):
      # detail=False: "   Load Balance entry: 2-way"
      # detail=True:  "   Load Balance entry 10: 2-way"
      content = ''
      if self.level:
         content = content + ' ' * self.spacePerLevel * self.level
      content = content + 'Load Balance entry'
      # Add index to content if detail output is required.
      if detail:
         assert self.index, "Unknown load balance table index"
         content = content + ( ' %d' % self.index )
      content = content + ': %d-way' % self.num
      return content

class Label( L2RibObject ):
   """Describes L2Rib::Label attributes and its strep."""
   label = Int( help='MPLS Label' )
   def getStr( self, detail ):
      content = ''
      if self.level:
         content = content + ' ' * self.spacePerLevel * self.level
      content = content + 'Label entry'
      # Add index to content if detail output is required.
      if detail:
         assert self.index, "Unknown label table index"
         content = content + ( ' %d' % self.index )
      content = content + ': %d' % self.label
      return content

class Dest( L2RibObject ):
   """Describes a Destination entry attributes and its strep."""
   destType = Enum( values=[ 'MPLS',
                             'Tunnel',
                             'VXLAN',
                             'CPU',
                             'Interface' ],
                    help='Destination type' )
   mplsLabel = Int( help='MPLS Label', optional=True )
   tunnelId = Submodel( valueType=TunnelId, help="Tunnel Identifier", optional=True )
   tunnelEndPoint = IpGenericPrefix( help="IP prefix of the MPLS tunnel endpoint",
                                     optional=True )
   color = Int( help="Color of SR-TE policy tunnel", optional=True )
   vtepAddr = Submodel( valueType=IpGenericAddr,
                        help="IP address of VXLAN tunnel endpoint", optional=True )
   interface = Interface( help="Local interface", optional=True )

   def getStr( self, detail ):
      def getTep():
         if not self.tunnelEndPoint:
            return None
         tep = str( self.tunnelEndPoint )
         if self.tunnelId and self.tunnelId.type == 'SR-TE Policy':
            tep += ', color %d' % self.color
         return tep
      content = ''
      if self.level:
         content = content + ' ' * self.spacePerLevel * self.level
      if self.destType == 'MPLS':
         underlayIntf = self.tunnelId.renderStr() if self.tunnelId else \
                        self.interface.stringValue # pylint: disable-msg=E1101
         assert underlayIntf is not None, "Malformed MPLS destination"
         tep = getTep()
         tunnelInfo = 'MPLS %s' % underlayIntf if not tep else \
                      'MPLS %s, TEP %s' % ( underlayIntf, tep )
         content = content + \
                   '%s, %d' % ( tunnelInfo, self.mplsLabel )
      elif self.destType == 'Tunnel':
         underlayIntf = self.tunnelId.renderStr() if self.tunnelId else \
                        self.interface.stringValue # pylint: disable-msg=E1101
         assert underlayIntf is not None, "Malformed Tunnel destination"
         tep = getTep()
         tunnelInfo = 'Tunnel %s' % underlayIntf if not tep else \
                      'Tunnel %s, TEP %s' % ( underlayIntf, tep )
         content = content + tunnelInfo
      elif self.destType == 'VXLAN':
         content = content + \
                     'VTEP %s' % self.vtepAddr.formatStr().stringValue
      elif self.destType == 'Interface':
         content = content + self.interface.stringValue # pylint: disable-msg=E1101
      elif self.destType == 'CPU':
         content = content + 'CPU'
      else:
         raise NotImplementedError
      return content

class HostEntry( _BaseTableEntry ):
   """Base type for input and output host entries used in HostTableInput(
   Output ) Models."""
   seqNo = Int( help='Sequence number' )
   pref = Int( help='Preference for the host entry' )
   # TODO: entryType must be an Enum which aligns with
   # Bridging::EntryType Tac::Enum. The mapping must belong in
   # BridgingCliModel.py.
   entryType = Str( help='Host entry type' )
   dests = List( valueType=L2RibObject,
                 help='collection of destination objects' )
   def getStr( self ):
      inputFmt = '%s, VLAN %d, seq %d, pref %d, %s'
      return inputFmt % ( self.macToString(),
                          self.vlanId,
                          self.seqNo,
                          self.pref,
                          hostEntryTypeHelper.entryTypeStr( self.entryType ) )

class HostEntryAndSource( HostEntry ):
   source = Str( help='host entry source' )
   def getStr( self ):
      outputFmt = '%s, VLAN %d, seq %d, pref %d, %s, source: %s'
      return outputFmt % ( self.macToString(),
                           self.vlanId,
                           self.seqNo,
                           self.pref,
                           self.entryType,
                           self.source )

class HostTable( Model ):
   """Base class for rendering host table ( input and output )."""
   hosts = List( valueType=HostEntry,
                 help='Host table entries' )
   invalidHosts = List( valueType=HostEntry,
                        help='Invalid host table entries' )
   _detail = Bool( help='Detailed output of host entry',
                   optional=True )

   def render( self ):
      self._renderHosts( self.hosts )
      if self.invalidHosts:
         print 74 * '-'
         print "Dangling HostTable Entries"
         print 74 * '-'
      self._renderHosts( self.invalidHosts )

   def _renderHosts( self, hosts ):
      for host in hosts:
         print host.getStr()
         for dest in host.dests:
            print dest.getStr( self._detail )

class FloodSet( Model ):
   vlanId = Int( help='Vlan Id' )
   macAddr = MacAddress( help ='Flood set MAC address' )
   floodType = Enum( values=[ 'Any', 'All' ], help='Flood type' )
   dests = List( valueType=Dest, help='Flood destination' )

class FloodSetSummary( Model ):
   tableName = Str( help="L2Rib Flood Set table name" )
   source = Str(help="L2Rib Flood Set table source", optional=True )
   floodSets = List( valueType=FloodSet,
                    help='Vlan flood set summary' )

   def getStr( self ):
      fsSummaryStr = "L2 RIB %s flood set: \n" % self.tableName
      if self.source is not None:
         fsSummaryStr += "Source: %s\n" % self.source
      table = createTable( ( "Vlan", "Address", "Type", "Destination" ),
                           tableWidth=100 )
      fh = Format()
      fh.noPadLeftIs( True )
      fh.padLimitIs( True )
      table.formatColumns( fh, fh, fh, fh )
      for floodSet in self.floodSets:
         destStr = ''
         first = True
         for dest in floodSet.dests:
            if not first:
               destStr += '\n'
            first = False
            destStr += dest.getStr( detail=False )
         macAddrStr = Ethernet.convertMacAddrCanonicalToDisplay(
            floodSet.macAddr.stringValue )
         table.newRow( floodSet.vlanId, macAddrStr, floodSet.floodType,
                       destStr )
      fsSummaryStr += table.output()
      return fsSummaryStr

class FloodSetSummaryColl( Model ):
   floodSetSummaries = List( valueType=FloodSetSummary,
                             help='Vlan flood set summary collection' )

   def render( self ):
      fsSummaryStr = ''
      for fsSummary in self.floodSetSummaries:
         fsSummaryStr += fsSummary.getStr() + '\n'
      print fsSummaryStr
