#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#pylint: disable=import-error, no-member, ungrouped-imports

import CliCommand
import CliMatcher
import SmashLazyMount
import Tac, BasicCli, ShowCommand
import Tracing
from TypeFuture import TacLazyType
from ArnetModel import IpGenericAddr
from CliPlugin import TechSupportCli, MacAddr, EthIntfCli, LagIntfCli
from Ethernet import convertMacAddrToDisplay
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
from L2RibLib import traverseAndValidateMultiTable
from TunnelCli import (
   getTunnelIdModel,
   getEndpointFromTunnelId,
   getColoredEndpointFromTunnelId )
from L2RibModel import (
   Summary,
   Dest,
   HostEntry,
   HostEntryAndSource,
   Label,
   LoadBalance,
   HostTable,
   FloodSet,
   FloodSetSummary,
   FloodSetSummaryColl )
import sys

FecIdIntfId = TacLazyType( "Arnet::FecIdIntfId" )
FecIdType = TacLazyType( "Smash::Fib::FecId" )

t0 = Tracing.trace0
storedEntityManager = None
dynTable = None
bgpTable = None
l2RibSourceHostModel = None
cppPlugin = None

class AliasResolve( object ):
   def __init__( self ):
      self.back = Tac.newInstance( 'L2RibCli::AliasResolver' )
      self.enumDecoder = Tac.Value( 'L2Rib::SourceDecoder' )

   def entryIs( self, commandToken, prettyName, sourceName, tableName, *args ):
      '''set entry. args are additional command aliases'''
      entry = self.back.entryIs( sourceName or "",
                                 commandToken or "",
                                 prettyName or "",
                                 tableName or "" )
      for alias in args:
         entry.commandAlias[ alias ] = True
         self.back.lookup[ alias ] = entry

   def _resolve( self, target, alias=None ):
      '''use key in appropriate list. If no alias provided, return full list.
         If alias provided but not found, return alias'''
      if alias:
         return getattr( self.back.lookup.get( alias.lower(), None ),
                         target, None ) or alias
      return [ getattr( entry, target, None )
               for entry in self.back.repo.itervalues()
               if getattr( entry, target, False ) ]

   def commandToken( self, alias=None ):
      '''returns command token associated with alias,
      or list of all tokens if no alias. Return None if alias not found'''
      return self._resolve( 'commandToken', alias )

   def prettyName( self, alias=None ):
      '''returns display name associated with alias'''
      return self._resolve( 'prettyName',  alias )

   def sourceName( self, alias=None ):
      '''return associated host source name'''
      return self._resolve( 'source', alias )

   def decodedEnum( self, alias=None ):
      '''return associated source name decoded with Tac SourceDecoder '''
      return self.enumDecoder.sourceFromEnum( self._resolve( 'source', alias ) )

   def commandAlias( self, alias=None ):
      '''return associated alias for commands'''
      r = self._resolve( 'commandAlias', alias )
      if isinstance( r, str ):
         return r
      return r.keys()

   def tableName( self, alias=None ):
      '''return associated table name'''
      return self._resolve( 'tableName', alias )

# Populate lookup tables
aliasResolve = AliasResolve()
aliasResolve.entryIs( 'local-dynamic', 'Local Dynamic', 'sourceLocalDynamic',
                      'localDynamic', 'dynamic', 'localDynamic' )
aliasResolve.entryIs( 'static', 'Local Static', 'sourceLocalStatic', 'static' )
aliasResolve.entryIs( 'bgp', 'BGP', 'sourceBgp', 'bgp' )
aliasResolve.entryIs( 'mlag', 'MLAG', 'sourceMlag', 'mlag' )
aliasResolve.entryIs( 'router', 'Router Mac', 'sourceRouterMac', 'router' )
aliasResolve.entryIs( 'dot1x', 'Dot1x', 'sourceDot1x', 'dot1x' )
aliasResolve.entryIs( 'mss', 'Mss', 'sourceMss', 'mss' )
aliasResolve.entryIs( 'vxlan-control-service', 'Vxlan-Control-Service',
                      'sourceVcs', 'vcs' )
aliasResolve.entryIs( 'vxlan-static', 'VXLAN Static', 'sourceVxlanStatic', 
                      'vxlanStatic' )
aliasResolve.entryIs( 'vxlan-dynamic', 'VXLAN Dynamic', 'sourceVxlanDynamic',
                      'vxlanDynamic' )
aliasResolve.entryIs( None, 'TestProto1', 'sourceTestProto1', None )
aliasResolve.entryIs( None, 'TestProto1', 'sourceTestProto2', None )
aliasResolve.entryIs( None, 'TestProto1', 'sourceTestProto3', None )

# preserve access to deprecated structures for compatibility with existing packages
# use aliasResolve instead of these
class DepDict( dict ):
   def __init__( self, keyfunc, valuefunc ):
      self.valuefunc = valuefunc
      self.keyfunc = keyfunc
   def get( self, key, default=None ):
      return self.valuefunc( key )
   def __getitem__( self, key ):
      return self.valuefunc( key )
   def keys( self ):
      return self.keyfunc()

sourceAliases = DepDict( aliasResolve.commandAlias, aliasResolve.commandToken )
altsByTokan = DepDict( aliasResolve.commandToken, aliasResolve.commandAlias )
tableDisplayNames = DepDict( aliasResolve.tableName, aliasResolve.commandToken )
hostSourceNames = DepDict( aliasResolve.sourceName, aliasResolve.prettyName )

def extractSource( args ):
   if args[ 'SOURCE' ] == 'all':
      return 'all'
   return aliasResolve.commandToken( args[ 'SOURCE' ] )

def l2RibCliHostModel( sourceDirEntry=None ):
   """Return L2RibHostModel model as found from l2RibSourceHostModel dict
   if one exits. Else we construct one from sourceDirEntry."""

   sourceName = sourceDirEntry.source if sourceDirEntry else 'output'
   if not sourceDirEntry:
      multiTableMountInfo = { 'lb' : ( "bridging/l2Rib/lbOutput",
                                       "L2Rib::LoadBalanceOutput" ),
                              'dest' : ( "bridging/l2Rib/destOutput",
                                         "L2Rib::DestOutput" ),
                              'label' : ( "bridging/l2Rib/labelOutput",
                                          "L2Rib::LabelOutput" ) }
   elif sourceDirEntry.multiTable:
      multiTableMountInfo = { 'lb' : ( sourceDirEntry.lbTableSmashPath,
                                       "L2Rib::LoadBalanceTable" ),
                              'dest' : ( sourceDirEntry.destTableSmashPath,
                                         "L2Rib::DestTable" ),
                              'label' : ( sourceDirEntry.labelTableSmashPath,
                                          "L2Rib::LabelTable" ) }
   else:
      multiTableMountInfo = {}

   global l2RibSourceHostModel
   if not l2RibSourceHostModel:
      l2RibSourceHostModel = dict()

   if l2RibSourceHostModel.get( sourceName ):
      return l2RibSourceHostModel[ sourceName ]

   lbTable = SmashLazyMount.mount(
      storedEntityManager, multiTableMountInfo[ 'lb' ][ 0 ],
      multiTableMountInfo[ 'lb' ][ 1 ], SmashLazyMount.mountInfo( 'reader' ) ) \
      if multiTableMountInfo.get( 'lb', None ) else None
   destTable = SmashLazyMount.mount(
      storedEntityManager, multiTableMountInfo[ 'dest' ][ 0 ],
      multiTableMountInfo[ 'dest' ][ 1 ], SmashLazyMount.mountInfo( 'reader' ) ) \
      if multiTableMountInfo.get( 'dest', None ) else None
   labelTable = SmashLazyMount.mount(
      storedEntityManager, multiTableMountInfo[ 'label' ][ 0 ],
      multiTableMountInfo[ 'label' ][ 1 ], SmashLazyMount.mountInfo( 'reader' ) ) \
      if multiTableMountInfo.get( 'label', None ) else None
   l2RibSourceHostModel[ sourceName ] = L2RibHostModel( sourceName, destTable,
                                                        labelTable, lbTable )
   return l2RibSourceHostModel[ sourceName ]

def floodSetTypeStr( h ):
   if h == 'floodSetTypeAll':
      return 'All'
   elif h == 'floodSetTypeAny':
      return 'Any'

   return 'Invalid'

def filterOutOfShow( h, mac, vlan, intf, vtep ):
   if vlan and h.vlanId != vlan:
      return True
   if mac and convertMacAddrToDisplay( h.macAddr ) != convertMacAddrToDisplay( mac ):
      return True
   if intf:
      if h.ciDestType != 'destTypeIntf' or str( h.ciIntf ) != str( intf ):
         return True
   if vtep:
      if h.ciDestType != 'destTypeVxlan':
         return True

      match = False
      for i in range( 0, len( h.ciDest.vtepAddr ) ):
         if h.ciDest.vtepAddr[ i ] == vtep:
            match = True
      return not match
   return None

def allSourceDirEntries():
   allSde = dict()
   for sdeName in cppPlugin.sourceDir.entityPtr:
      sde = cppPlugin.sourceDir.entityPtr[ sdeName ]
      if sde.initialized:
         allSde[ sdeName ] = sde

   for sdeName in cppPlugin.agentSourceDir.source.keys():
      sde = cppPlugin.agentSourceDir.source[ sdeName ]
      if sde.initialized:
         assert sde.name not in allSde.keys()
         allSde[ sde.name ] = sde

   return allSde

def updateHostTableModel( hostTable, hostTableModel, cliHostModel,
                          mac, vlan, intf, vtep, detail, source ):
   """Walk L2RIB Host Table ( Input or Output ) and add hosts to hostTableModel."""
   # note that these are deliberately not sorted, becasue sorting them
   # in python becomes very expensive when the tables gets big.  Doing
   # this properly really needs a C++ CLI implementation, like we have
   # for 'show ip route'
   for host in hostTable.host.values():
      if filterOutOfShow( host, mac, vlan, intf, vtep ):
         continue
      ( valid, hostModel ) = cliHostModel.getHostModel( host, detail, source )
      if valid:
         hostTableModel.hosts.append( hostModel )
      else:
         hostTableModel.invalidHosts.append( hostModel )

def showL2RibInput( mode, source, mac=None, vlan=None, intf=None, vtep=None,
                    detail=None ):
   sourceName = aliasResolve.tableName( source )
   hostTableModel = HostTable()
   hostTableModel._detail = True if detail else False # pylint: disable-msg=W0212

   allSde = allSourceDirEntries()
   if( sourceName not in allSde.keys() and sourceName != 'all' ):
      return hostTableModel

   for sdeName, sde in allSde.iteritems():
      # When hostTableSmashPath is not a valid string, skip the sde to prevent
      # ConfigAgent from crashing.
      if( sourceName in [ 'all', sdeName ] and sde.hostTableSmashPath ):
         cliHostModel = l2RibCliHostModel( sde )
         hostTable = SmashLazyMount.mount( storedEntityManager,
                                           sde.hostTableSmashPath,
                                           'L2Rib::HostInput',
                                           SmashLazyMount.mountInfo( 'reader' ) )
         # When 'all' input tables have to be listed, tag the source
         # name with the host entry to identify the source.
         source = sdeName if sourceName == 'all' else None
         updateHostTableModel( hostTable, hostTableModel, cliHostModel,
                               mac, vlan, intf, vtep, detail, source )
   return hostTableModel

def showL2RibOutput( mode, mac=None, vlan=None, intf=None, vtep=None, detail=None ):
   cliHostModel = l2RibCliHostModel()
   hostTableModel = HostTable()
   hostTableModel._detail = True if detail else False # pylint: disable-msg=W0212
   updateHostTableModel( cppPlugin.outputTable, hostTableModel, cliHostModel,
                         mac, vlan, intf, vtep, detail, None )
   return hostTableModel

def getDestModel( dest, level=0, detail=False ):
   """Construct Dest L2RibObject using L2Rib::Dest Tac model."""
   destModel = Dest()
   destModel.level = level
   destModel.index = dest.key if detail else None
   def getSrTeDest( intfId ):
      # Given a FecIdIntfId interface, this routine returns a 3-tuple ( tunnelId,
      # endpoint and color ) for the SR-TE policy tunnel table entry.
      if not FecIdIntfId.isFecIdIntfId( intfId ):
         return None, None, None
      fecId = FecIdIntfId.intfIdToFecId( intfId )
      tunnelId = FecIdType.fecIdToTunnelId( fecId )
      endpoint, color = getColoredEndpointFromTunnelId( tunnelId )
      if endpoint:
         return tunnelId, endpoint, color
      return None, None, None
   if dest.destType == 'destTypeMpls':
      destModel.mplsLabel = dest.mpls.label
      # MPLS destination may include a tunnel interface or physical
      # interface. Populate the relevant field in the destination
      # model.
      if dest.mpls.tunnelId:
         destModel.tunnelId = getTunnelIdModel( dest.mpls.tunnelId )
         destModel.tunnelEndPoint = getEndpointFromTunnelId( dest.mpls.tunnelId )
      else:
         # See if the IntfId is a FecId/SR-TE policy tunnel. If so, then the
         # CLI model will contain the SR-TE policy tunnel information instead of
         # the policy FEC interface id.
         tunnelId, endpoint, color = getSrTeDest( dest.mpls.intfId )
         if tunnelId:
            destModel.tunnelId = getTunnelIdModel( tunnelId )
            destModel.tunnelEndPoint = endpoint
            destModel.color = color
         else:
            destModel.interface = dest.mpls.intfId
      destModel.destType = 'MPLS'
   elif dest.destType == 'destTypeVxlan':
      destModel.vtepAddr = IpGenericAddr( ip=dest.vxlan.vtepAddr )
      destModel.destType = 'VXLAN'
   elif dest.destType == 'destTypeTunnel':
      # Tunnel destinations can be overloaded to an interface destination.
      destModel.destType = 'Tunnel'
      if dest.tunnel.tunnel:
         destModel.tunnelEndPoint = getEndpointFromTunnelId( dest.tunnel.tunnel )
         destModel.tunnelId = getTunnelIdModel( dest.tunnel.tunnel )
      else:
         # See if the IntfId is a FecId/SR-TE policy tunnel
         tunnelId, endpoint, color = getSrTeDest( dest.tunnel.intfId )
         # CLI model will contain the SR-TE policy tunnel information instead of
         # the policy FEC interface id.
         if tunnelId:
            destModel.tunnelId = getTunnelIdModel( tunnelId )
            destModel.tunnelEndPoint = endpoint
            destModel.color = color
         else:
            destModel.interface = dest.tunnel.intfId
   elif dest.destType == 'destTypeIntf':
      destModel.interface = dest.intf.intfId
      destModel.destType = 'Interface'
   elif dest.destType == 'destTypeCpu':
      destModel.destType = 'CPU'
   else:
      raise NotImplementedError
   return destModel

def getFloodSetSummaryModel( table, tableName, vlan=None, vtepAddr=None, intfId=None,
                        label=None ):
   assert table
   fsSummary = FloodSetSummary()
   fsSummary.tableName = tableName
   printVlans = table.vlanFloodSet.keys()
   if vlan:
      if vlan not in printVlans:
         return fsSummary
      printVlans = [ vlan ]

   for vlanId in sorted( printVlans ):
      vfs = table.vlanFloodSet.get( vlanId )
      if not vfs or vfs.vlanId != vlanId:
         continue

      for macAddr in sorted( vfs.floodSet ):
         fs = vfs.floodSet.get( macAddr )
         if ( not fs or fs.macAddr != macAddr or
              fs.floodSetType == 'floodSetTypeInvalid' ):
            continue

         floodSet = FloodSet()
         floodSet.vlanId = vlanId
         floodSet.macAddr = macAddr
         if fs.floodSetType == 'floodSetTypeAll':
            floodSet.floodType = 'All'
         else:
            floodSet.floodType = 'Any'

         for d in fs.destSet.keys():
            if( vtepAddr and
                ( d.destType != 'destTypeVxlan' or
                  d.vxlan.vtepAddr != vtepAddr ) ):
               continue
            elif( intfId and
                  ( d.destType != 'destTypeIntf' or
                    d.intf.intfId != intfId ) ):
               continue
            elif( label and
                  ( d.destType != 'destTypeMpls' or
                    d.mpls.label != label ) ):
               continue
            dest = getDestModel( d )
            floodSet.dests.append( dest )

         if floodSet.dests:
            fsSummary.floodSets.append( floodSet )
   return fsSummary

enumDecoder = Tac.Value( 'L2Rib::SourceDecoder' )

def getInputFloodSetSummaryColl( sourceName, vlan=None, vtep=None ):
   allSde = allSourceDirEntries()
   fsSummaryColl = FloodSetSummaryColl()
   for sdeName in allSde:
      if( sourceName == 'all' or sourceName == sdeName ):
         sde = allSde[ sdeName ]

         tableName = "%s Input" % enumDecoder.sourceStr( sde.source )
         fsSummary = getFloodSetSummaryModel( sde, tableName, vlan, vtep,
                                              intfId=None, label=None )
         fsSummaryColl.floodSetSummaries.append( fsSummary )
   return fsSummaryColl

def showL2RibInputFloodSet( mode, source, vlan=None, vtep=None ):
   sourceName = aliasResolve.tableName( source )

   allSde = allSourceDirEntries()
   if( sourceName not in allSde.keys() and sourceName != 'all' ):
      mode.addError( "No input table %s." % source )
      return None

   return getInputFloodSetSummaryColl( sourceName, vlan, vtep )

def fsSourceStr( source ):
   prettyNames = list()
   for s in aliasResolve.sourceName():
      if source & aliasResolve.decodedEnum( s ):
         prettyNames.append( aliasResolve.prettyName( s ) )

   if not prettyNames:
      return 'None'

   return ', '.join( prettyNames )

def getOutputFloodSetSummaryColl( vlan=None, vtep=None ):
   tableName = "Output"
   fsSummaryColl = FloodSetSummaryColl()
   fsSummary = getFloodSetSummaryModel( cppPlugin.outputFloodSetTable, tableName,
                                        vlan, vtep,
                                        intfId=None, label=None )
   fsSummary.source = fsSourceStr( cppPlugin.outputFloodSetTable.source )
   fsSummaryColl.floodSetSummaries.append( fsSummary )
   return fsSummaryColl

def showL2RibOutputFloodSet( mode, vlan=None, vtep=None ):
   return getOutputFloodSetSummaryColl( vlan, vtep )

def showL2RibSummary( mode, args ):
   cppPlugin.showL2RibSummary( sys.stdout.fileno(), mode.session_.outputFormat() )
   return Summary

def getLabelModel( label, level, detail ):
   labelModel = Label()
   labelModel.level = level
   labelModel.index = label.key if detail else None
   labelModel.label = label.label
   return labelModel

def getLbModel( lb, level, detail ):
   lbModel = LoadBalance()
   lbModel.num = len( lb.next )
   lbModel.level = level
   lbModel.index = lb.key if detail else None
   return lbModel

class L2RibHostModel( object ):
   def __init__( self, tableName, destTable=None, labelTable=None, lbTable=None ):
      """Provides methods to populate L2RibModel.HostTable."""
      self.tableName = tableName
      self.destTable = destTable
      self.labelTable = labelTable
      self.lbTable = lbTable

   def getHostModel( self, host, detail=True, source=None ):
      """Returns L2RibModel.HostEntry CLI model from host."""
      hostModel = HostEntryAndSource() if self.tableName == 'output' or \
                  source is not None else HostEntry()
      hostModel.macAddress = host.key.macaddr
      hostModel.vlanId = host.key.vlanId
      hostModel.seqNo = host.seqNo
      hostModel.pref = host.preference
      hostModel.entryType = host.entryType
      if self.tableName == 'output':
         hostModel.source = aliasResolve.prettyName( host.source )
      elif source is not None:
         hostModel.source = aliasResolve.prettyName( source )
      retVal = self._updateDestModel( host, hostModel, detail )
      return ( retVal, hostModel )

   def _updateDestModel( self, host, hostModel, detail ):
      """Populate hostModel.dests for Ci/multi-table destination"""
      return self._updateCiDests( host, hostModel ) \
         if host.ciDest != Tac.Value( "L2Rib::CiDest" ) else \
            self._updateDests( host.next, hostModel, 1, detail )

   def _updateCiDests( self, host, hostModel ):
      """Construct Dest Models for Ci destination objects"""
      if host.ciDest.destType == 'destTypeIntf':
         dest = Dest()
         dest.level = 1
         dest.destType = 'Interface'
         dest.interface = host.ciDest.intf
         hostModel.dests.append( dest ) # pylint: disable-msg=E1101
         return True
      if host.ciDest.destType == 'destTypeVxlan':
         for vtepAddr in sorted( host.ciDest.vtepAddr.values() ):
            dest = Dest()
            dest.level = 1
            dest.destType = 'VXLAN'
            dest.vtepAddr = IpGenericAddr( ip=vtepAddr )
            hostModel.dests.append( dest ) # pylint: disable-msg=E1101
         return True
      return False

   def _updateDests( self, nextObj, hostModel, startLevel, detail ):
      def _updateDestAction( objTuple, entry, level ):
         getEntryModel = None
         if objTuple.tableType == 'tableTypeDest':
            getEntryModel = getDestModel
         elif objTuple.tableType == 'tableTypeLabel':
            getEntryModel = getLabelModel
         elif objTuple.tableType == 'tableTypeLoadBalance':
            getEntryModel = getLbModel
         else:
            assert False, "Unexpected table type" % objTuple.tableType
         hostModel.dests.append( getEntryModel( entry, level, detail ) )
      return traverseAndValidateMultiTable( nextObj, self.destTable,
                                            self.labelTable, self.lbTable,
                                            startLevel, _updateDestAction )

# ----------------------------------------------------------------------------------

# sources with helptext
sourceMatchConf = { 'all':'Show all L2 RIB input tables' }
for tk in aliasResolve.commandToken():
   sourceHelpDesc = 'Show L2 RIB {} input tables'.format(
         aliasResolve.prettyName( tk ) )
   sourceMatchConf[ tk ] = sourceHelpDesc
   aliasList = aliasResolve.commandAlias( tk )
   if isinstance( aliasList, list ):
      sourceMatchConf.update( { alias:sourceHelpDesc
                                for alias in aliasList } )

class InputMatcher( CliCommand.CliExpression ):
   expression = 'input SOURCE'
   data = { 'input':'Show L2 RIB input tables',
            'SOURCE':CliMatcher.EnumMatcher( sourceMatchConf ) }

# ----------------------------------------------------------------------------------

# data for common tokens
l2RibHelpDesc = 'Show L2 RIB information'
outputHelpDesc = 'Show L2 RIB output tables'
floodSetHelpDesc = 'Show L2Rib floodset from input source'

# adapters to connect old-parser handlers to new parser
def showL2RibInputAdapter( mode, args ):
   # used 'get' where optional AND default value is NONE. [] otherwise for assert.
   return showL2RibInput( mode=mode, source=extractSource( args ),
                          detail=( 'detail' in args.keys() ),
                          vlan=args.get( 'VLAN' ),
                          mac=args.get( 'MAC' ),
                          vtep=args.get( 'VTEP' ),
                          intf=( args.get( 'PHY' ) or args.get( 'VIRT' ) ) )
def showL2RibInputFloodSetAdapter( mode, args ):
   # used 'get' where optional AND default value is NONE. [] otherwise for assert.
   return showL2RibInputFloodSet( mode=mode, source=extractSource( args ),
                                  vlan=args.get( 'VLAN' ),
                                  vtep=args.get( 'VTEP' ) )
def showL2RibOutputAdapter( mode, args ):
   return showL2RibOutput( mode=mode,
                           detail=( 'detail' in args.keys() ),
                           vlan=args.get( 'VLAN' ),
                           mac=args.get( 'MAC' ),
                           vtep=args.get( 'VTEP' ),
                           intf=( args.get( 'PHY' ) or args.get( 'VIRT' ) ) )
def showL2RibOutputFloodSetAdapter( mode, args ):
   return showL2RibOutputFloodSet( mode=mode,
                                   vlan=args.get( 'VLAN' ),
                                   vtep=args.get( 'VTEP' ) )

# entry filter tokens, grouped appropriately
class FloodSetFilter( CliCommand.CliExpression ):
   expression = '[ ( vlan VLAN ) ] [ ( vtep VTEP ) ]'
   data = { 'vlan':'Filter by VLAN id',
            'VLAN':CliMatcher.IntegerMatcher( 1, 4094, helpdesc='Choose VLAN ID' ),
            'vtep':'Filter by VTEP',
            'VTEP': IpGenAddrMatcher.IpGenAddrMatcher( 
               helpdesc="IP address of VTEP" ),
           }
class InputFilter( CliCommand.CliExpression ):
   expression = ( '[ FS_FILTER ]' +
                '[ ( mac MAC ) ] [ ( interface ( PHY | VIRT ) ) ] [ detail ]' )
   data = { 'FS_FILTER':FloodSetFilter,
            'mac':'Filter by MAC address',
            'MAC':MacAddr.MacAddrMatcher(),
            'interface':'Filter by destination interface',
            'PHY':EthIntfCli.EthPhyIntf.ethMatcher,
            'VIRT':LagIntfCli.EthLagIntf.matcher,
            'detail':'More comprehensive output' }

# ----------------------------------------------------------------------------------
# add commands
# ----------------------------------------------------------------------------------

# show l2rib summary
class L2RibSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib summary'
   data = { 'l2Rib':l2RibHelpDesc,
            'summary':'Show L2 Rib summary of known hosts and sources' }
   handler = showL2RibSummary
   cliModel = Summary
BasicCli.addShowCommandClass( L2RibSummaryCmd )

# input tables
class L2RibInputCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib INPUT [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'INPUT':InputMatcher,
            'FILTERS': InputFilter }
   handler = showL2RibInputAdapter
   cliModel = HostTable
BasicCli.addShowCommandClass( L2RibInputCmd )

# input floodset tables
class L2RibInputFloodSetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib INPUT floodset [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'INPUT':InputMatcher,
            'floodset':floodSetHelpDesc,
            'FILTERS':FloodSetFilter }
   handler = showL2RibInputFloodSetAdapter
   cliModel = FloodSetSummaryColl
BasicCli.addShowCommandClass( L2RibInputFloodSetCmd )

# output tables
class L2RibOutputCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib output [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'output':outputHelpDesc,
            'FILTERS':InputFilter }
   handler = showL2RibOutputAdapter
   cliModel = HostTable
BasicCli.addShowCommandClass( L2RibOutputCmd )

# output floodset tables
class L2RibOutputFloodSetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2Rib output floodset [ FILTERS ]'
   data = { 'l2Rib':l2RibHelpDesc,
            'output':outputHelpDesc,
            'floodset':floodSetHelpDesc,
            'FILTERS':FloodSetFilter }
   handler = showL2RibOutputFloodSetAdapter
   cliModel = FloodSetSummaryColl
BasicCli.addShowCommandClass( L2RibOutputFloodSetCmd )

#----------------------------------------------------------------------------------
# register 'show l2rib input all', 'show l2rib input all floodset',
# 'show l2rib output', 'show l2rib output floodset' into 'show tech-support
# extended evpn'.
#-----------------------------------------------------------------------------------
def _showTechEvpnCmds():
   return [
            'show l2rib input all',
            'show l2rib input all floodset',
            'show l2rib output',
            'show l2rib output floodset',
          ]
TechSupportCli.registerShowTechSupportCmdCallback( '2017-11-03 12:06:06',
                                                   _showTechEvpnCmds,
                                                   extended='evpn' )

def Plugin( entityManager ):
   global storedEntityManager, cppPlugin
   storedEntityManager = entityManager
   cppPlugin = Tac.newInstance( 'L2RibCli::Helper',
                                entityManager.cEntityManager(),
                                aliasResolve.back )
