
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliParser
import LazyMount
from L2ProtocolModel import ( L2ProtocolForwardingProfile,
      L2ProtocolForwardingProfiles, L2ProtocolForwardingProfileSummary,
      L2ProtocolForwardingProfileSummaries, L2ProtocolForwardingInfo,
      L2ProtocolForwardingInterface, L2ProtocolForwardingInterfaces )
from EbraLib import l2ProtocolFromMacAddress
from EthIntfCli import EthPhyAutoIntfType
from Intf.IntfRange import IntfRangeMatcher
import ShowCommand,  CliMatcher, CliCommand

# Module globals set up by the Plugin function below
l2PtProfileConfig = None
l2PtIntfConfig = None
bridgingHwCapabilities = None
l2ProtocolFwdIntfStatus = None

def l2ProtocolModeGuard( mode, token ):
   if bridgingHwCapabilities.l2ProtocolTransparencySupported:
      return None
   return CliParser.guardNotThisPlatform

matcherL2Protocol = CliMatcher.KeywordMatcher( 'l2-protocol',
                                      helpdesc='L2 Protocol information' )
nodeL2Protocol = CliCommand.Node( matcher=matcherL2Protocol,
                                  guard=l2ProtocolModeGuard )
matcherForwarding = CliMatcher.KeywordMatcher( 'forwarding',
                                      helpdesc='L2 forwarding information' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile',
                                   helpdesc='Forwarding profile information' ) 

def populateL2ProtocolForwardingProfiles( profileNames, profiles, summary=None ):
   for profileName in profileNames:
      l2PtProfile = l2PtProfileConfig.profile.get( profileName )
      profile = L2ProtocolForwardingProfileSummary() if summary else \
                L2ProtocolForwardingProfile()
      profiles.profiles[ profileName ] = profile
      # Populate profile info
      if not summary:
         for protocol, action in l2PtProfile.protocolToAction.iteritems():
            info = L2ProtocolForwardingInfo()
            info.protocol = l2ProtocolFromMacAddress( protocol.addr )
            info.tagFormat = protocol.tagFormat
            info.action = action
            profile.protocolInfo.append( info )
   if summary:
      # Populate active interfaces
      for intf, intfStatus in l2ProtocolFwdIntfStatus.intfStatus.iteritems():
         profile = profiles.profiles.get( intfStatus.l2PtProfileName )
         if profile:
            profile.activeIntfs.append( intf )
      # Populate configured interfaces
      for intf, intfProfileName in l2PtIntfConfig.intfToProfile.iteritems():
         profile = profiles.profiles.get( intfProfileName )
         if profile:
            profile.configuredIntfs.append( intf )

def showL2ProtocolForwardingProfile( mode, args ):
   profileName = args.get( 'PROFILE' )
   summary = 'summary' in args
   profiles = L2ProtocolForwardingProfileSummaries() if summary else \
              L2ProtocolForwardingProfiles()
   if profileName and profileName not in l2PtProfileConfig.profile:
      return profiles
   profileNames = [ profileName
                  ] if profileName else l2PtProfileConfig.profile
   populateL2ProtocolForwardingProfiles( profileNames, profiles, summary )
   return profiles

def l2ProtocolForwardingInfos( protocolInfos ):
   l2ProtocolFwdInfos = []
   for l2ProtocolFwdInfo in protocolInfos.itervalues():
      info = L2ProtocolForwardingInfo()
      l2ProtocolMatch = l2ProtocolFwdInfo.l2ProtocolMatch
      info.protocol = l2ProtocolFromMacAddress( l2ProtocolMatch.addr )
      info.tagFormat = l2ProtocolMatch.tagFormat
      info.action = l2ProtocolFwdInfo.action.action
      l2ProtocolFwdInfos.append( info )
   return l2ProtocolFwdInfos

def l2ProtocolForwardingInterfaces( intfNames, detail ):
   interfaceInfo = L2ProtocolForwardingInterfaces()
   if detail:
      interfaceInfo.detailIs( True )
   for intf in intfNames:
      # Get the L2ProtocolFwdStatus, if exists
      l2ProtocolFwdStatus = l2ProtocolFwdIntfStatus.intfStatus.get( intf )
      if not l2ProtocolFwdStatus:
         continue

      interfaceProfileInfo = L2ProtocolForwardingInterface()
      interfaceInfo.interfaces[ intf ] = interfaceProfileInfo
      interfaceProfileInfo.profileName = l2ProtocolFwdStatus.l2PtProfileName

      if detail:
         # Populate the protocol information
         interfaceProfileInfo.protocolInfo = l2ProtocolForwardingInfos(
            l2ProtocolFwdStatus.seqToProtocolInfo )
         # Populate the PW protocol information
         interfaceProfileInfo.protocolInfo += l2ProtocolForwardingInfos(
            l2ProtocolFwdStatus.pseudoWireProtocolInfo )

   return interfaceInfo

#-------------------------------------------------------------------------------
# "show l2-protocol forwarding interface [intfs] [detail]"
#-------------------------------------------------------------------------------

def showL2ProtocolForwardingInterface( mode, args ):
   intfs = args.get( 'INTERFACES' )
   detail = 'detail' in args
   if intfs:
      intfs = [ intf for intf in intfs.intfNames()
                if intf.startswith( 'Ethernet' ) ]
   else:
      intfs = l2ProtocolFwdIntfStatus.intfStatus
   return l2ProtocolForwardingInterfaces( intfs, detail )

profileNameRule = CliMatcher.DynamicNameMatcher(
   lambda mode: l2PtProfileConfig.profile,
   helpdesc='L2 protocol forwarding profile name' )

explicitIntfTypes = ( EthPhyAutoIntfType, )

class L2ProtocolForwardingInterfacesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2-protocol forwarding interfaces [ INTERFACES ] [ detail ]'
   data = {
      'l2-protocol': nodeL2Protocol,
      'forwarding': matcherForwarding,
      'interfaces': 'Interface-specific details',
      'INTERFACES': IntfRangeMatcher( explicitIntfTypes=explicitIntfTypes ),
      'detail': 'Display information in detail',
   }
   handler = showL2ProtocolForwardingInterface
   cliModel = L2ProtocolForwardingInterfaces

BasicCli.addShowCommandClass( L2ProtocolForwardingInterfacesCmd )

class L2ProtocolForwardingProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2-protocol forwarding profile [ PROFILE ]'
   data = {
      'l2-protocol': nodeL2Protocol,
      'forwarding': matcherForwarding,
      'profile': matcherProfile,
      'PROFILE': profileNameRule,
   }
   handler = showL2ProtocolForwardingProfile
   cliModel = L2ProtocolForwardingProfiles

BasicCli.addShowCommandClass( L2ProtocolForwardingProfileCmd )

class L2ProtocolForwardingProfileSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show l2-protocol forwarding profile [ PROFILE ] summary'
   data = {
      'l2-protocol': nodeL2Protocol,
      'forwarding': matcherForwarding,
      'profile': matcherProfile,
      'PROFILE': profileNameRule,
      'summary': 'Summary',
   }
   handler = showL2ProtocolForwardingProfile
   cliModel = L2ProtocolForwardingProfileSummaries

BasicCli.addShowCommandClass( L2ProtocolForwardingProfileSummaryCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global l2PtProfileConfig, l2PtIntfConfig
   global bridgingHwCapabilities
   global l2ProtocolFwdIntfStatus
   l2PtProfileConfig = LazyMount.mount( entityManager,
         "l2protocolforwarding/profileconfig",
         "Ebra::L2Pt::L2PtProfileConfig", "r" )
   l2PtIntfConfig = LazyMount.mount( entityManager,
         "l2protocolforwarding/intfconfig",
         "Ebra::L2Pt::L2PtIntfConfig", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
         "bridging/hwcapabilities",
         "Bridging::HwCapabilities", "r" )
   l2ProtocolFwdIntfStatus = LazyMount.mount( entityManager,
                              "l2protocolforwarding/intfstatusnew",
                              "Ebra::L2Pt::L2ProtocolFwdIntfStatus", "r" )
