# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Enum, Dict, List, Str, Bool
from IntfModel import Interface
from EbraLib import ( tacL2PtProtocol, tacL2PtTagFormat, tacL2PtAction,
                      l2PtProtocolToCliToken )
from Intf import IntfRange
import Arnet
import TableOutput

class L2ProtocolForwardingInfo( Model ):
   protocol = Enum( values=tacL2PtProtocol.attributes,
                    optional=True,
                    help="L2 Protocol" )
   tagFormat = Enum( values=tacL2PtTagFormat.attributes,
                     optional=True,
                     help="VLAN tag format" )
   action = Enum( values=( tacL2PtAction.forward, ),
                  help="Action to be performed on protocol packets" )

class L2ProtocolForwardingProfile( Model ):
   protocolInfo = List( valueType=L2ProtocolForwardingInfo,
                           optional=True,
                           help="L2 protocol forwarding information " )
   def render( self ):
      for info in self.protocolInfo:
         if info.tagFormat and info.tagFormat != tacL2PtTagFormat.all:
            print "  %s: %s %s" % ( l2PtProtocolToCliToken[ info.protocol ].upper(),
                                    info.tagFormat.title(), info.action.title() )
         else:
            print "  %s: %s" % ( l2PtProtocolToCliToken[ info.protocol ].upper(),
                                 info.action.title() )

class L2ProtocolForwardingProfiles( Model ):
   profiles = Dict( valueType=L2ProtocolForwardingProfile,
                    help="A mapping of L2 protocol forwarding profile name to "
                         "profile information" )
   def render( self ):
      for profileName, profile in self.profiles.iteritems():
         print "Profile: %s" % profileName
         profile.render()

class L2ProtocolForwardingProfileSummary( Model ):
   configuredIntfs = List( valueType=Interface,
                           optional=True,
                           help="Interfaces on which profile is configured" )
   activeIntfs = List( valueType=Interface,
                       optional=True,
                       help="Interfaces on which profile is active" )
   def render( self ):
      print '  Configured:',
      print ','.join( IntfRange.intfListToCanonical( self.configuredIntfs ) )
      print '  Active:',
      print ','.join( IntfRange.intfListToCanonical( self.activeIntfs ) )

class L2ProtocolForwardingProfileSummaries( Model ):
   profiles = Dict( valueType=L2ProtocolForwardingProfileSummary,
                    help="A mapping of L2 protocol forwarding profile name to "
                         " interfaces on which it is active/configured." )
   def render( self ):
      for profileName, profile in self.profiles.iteritems():
         print "Profile: %s" % profileName
         profile.render()

class L2ProtocolForwardingInterface( Model ):
   profileName = Str( help="Active L2 protocol forwarding profile" )
   protocolInfo = List( valueType=L2ProtocolForwardingInfo,
                        help="L2 protocol forwarding information",
                        optional=True )

class L2ProtocolForwardingInterfaces( Model ):
   interfaces = Dict( valueType=L2ProtocolForwardingInterface,
                      keyType=Interface,
                      help="L2 protocol forwarding information for interfaces" )
   _detail = Bool( help="User requested detailed L2 protocol forwarding information",
                   optional=True )

   def detailIs( self, detail ):
      self._detail = detail

   def _mapIntfProtocolTagToAction( self ):
      intfProtocolTagToAction = {}
      for intfName, intfInfo in self.interfaces.iteritems() :
         protocolInfo = intfInfo.protocolInfo
         for info in protocolInfo:
            if info.tagFormat in [ tacL2PtTagFormat.all,
                                   tacL2PtTagFormat.tagged ]:
               intfProtocolTagToAction[ ( intfName, info.protocol, 'T' ) ] = \
                                                         info.action.upper()[ 0 ]
            if info.tagFormat in [ tacL2PtTagFormat.all,
                                   tacL2PtTagFormat.untagged ]:
               intfProtocolTagToAction[ ( intfName, info.protocol, 'U' ) ] = \
                                                         info.action.upper()[ 0 ]
      return intfProtocolTagToAction

   def render( self ):
      fc = TableOutput.Format( justify='center' )
      fc.padLimitIs( True )
      fl = TableOutput.Format( justify='left' )
      fl.padLimitIs( True )

      # This means that user asked for detailed information
      if self._detail:
         intfProtocolTagToAction = self._mapIntfProtocolTagToAction()

         headers = [ "Interface", "Profile" ]
         tagFormats = ( 'T', 'U' )
         formats = [ fl, fl ]

         protocols = sorted( tacL2PtProtocol.attributes )
         for protocol in protocols:
            headers += [ ( l2PtProtocolToCliToken[ protocol ].upper(), tagFormats ) ]
            formats += [ fc ] * len( tagFormats )

         table = TableOutput.createTable( headers )
         table.formatColumns( *formats )
         for intfName in Arnet.sortIntf( self.interfaces ):
            profileName = self.interfaces[ intfName ].profileName
            entry = [ intfName, profileName ]
            for protocol in protocols:
               for tagFormat in tagFormats:
                  entry.append( intfProtocolTagToAction.get( ( intfName, protocol,
                                                               tagFormat ), '-' ) )
            table.newRow( *entry )
         print 'Tagging Types: T: tagged U: untagged'
         print 'Actions: F: forward'
      else:
         headers = [ 'Interface', 'Profile' ]
         table = TableOutput.createTable( headers )
         table.formatColumns( fl, fl )
         for intfName in Arnet.sortIntf( self.interfaces ):
            table.newRow( intfName, self.interfaces[ intfName ].profileName )
      print table.output()
