# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import BasicCli, IntfCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import ConfigMount, LazyMount
from CliMode.L2Protocol import L2ProtocolModeBase, L2ProtocolProfileModeBase
from EbraLib import ( tacL2PtAction, tacL2PtTagFormat, tacL2PtProtocol,
                      l2ProtocolMacAddress, cliTokenToL2PtProtocol )

__defaultTraceHandle__ = Tracing.Handle( 'L2PtCli' )
t0 = Tracing.trace0

# Module globals set up by the Plugin function below
l2PtProfileConfig = None
l2PtIntfConfig = None
bridgingHwCapabilities = None

#-------------------------------------------------------------------------------
# The "config-l2-protocol" mode.
#-------------------------------------------------------------------------------
class L2ProtocolMode( L2ProtocolModeBase, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'L2 Protocol Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      L2ProtocolModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# Used in global mode to guard profile creation
def l2ProtocolModeGuard( mode, token ):
   if bridgingHwCapabilities.l2ProtocolTransparencySupported:
      return None
   return CliParser.guardNotThisPlatform

# Used in interface mode to guard profile attachment to the interface
def l2ProtocolFwdModeGuard( mode, token ):
   if l2ProtocolModeGuard( mode, token ) is None:
      if mode.intf.isSubIntf():
         if bridgingHwCapabilities.l2ProtocolForwardingSubIntfSupported:
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform

def gotoL2ProtocolMode( mode, args ):
   childMode = mode.childMode( L2ProtocolMode )
   mode.session_.gotoChildMode( childMode )

#--------------------------------------------------------------------------------
# l2-protocol
#--------------------------------------------------------------------------------
class L2ProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'l2-protocol'
   data = {
      'l2-protocol': CliCommand.guardedKeyword( 'l2-protocol', 
                        helpdesc='Enter L2 Protocol configuration mode',
                        guard=l2ProtocolModeGuard ),
   }
   handler = gotoL2ProtocolMode

BasicCliModes.GlobalConfigMode.addCommandClass( L2ProtocolCmd )

#-------------------------------------------------------------------------------
# The "forwarding profile <profileName>" mode.
#-------------------------------------------------------------------------------
class L2ProtocolProfileModeContext( object ):
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profileName_ = profileName
      # Current profile
      profile = l2PtProfileConfig.profile.get( profileName )
      # Profile undergoing editing
      self.editProfile_ = Tac.newInstance( 'Ebra::L2Pt::L2PtProfile',
                                           self.profileName_ )
      if profile:
         self.copyProfile( self.editProfile_, profile )
         self.editProfile_.version = profile.version

   def copyProfile( self, dstProfile, srcProfile ):
      copier = Tac.newInstance( 'Cli::Session::EntityCopy' )
      copier.handler = Tac.newInstance( "Ebra::L2Pt::L2PtProfileCopyHandler" )
      copier.handler.handleEntity( copier, dstProfile, srcProfile, '' )

   def editProfile( self ):
      return self.editProfile_

   def profileName( self ):
      return self.profileName_

   def commit( self ):
      profile = l2PtProfileConfig.profile.newMember( self.profileName_ )
      if profile.version != self.editProfile_.version:
         self.mode.addWarning( 'Version mismatch possibly due to profile being '
                               'edited in another CLI session. Aborting change.')
         return
      self.copyProfile( profile, self.editProfile_ )
      t0( 'Profile Version = %d' % profile.version )

class L2ProtocolProfileMode( L2ProtocolProfileModeBase, BasicCli.ConfigModeBase ):
   name = "L2 Protocol Forwarding Profile Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, profileName ):
      self.l2ProtocolProfileModeContext = L2ProtocolProfileModeContext( self,
                                                                        profileName )
      self.profileName_ = profileName
      L2ProtocolProfileModeBase.__init__( self, param=self.profileName_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.l2ProtocolProfileModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      context = self.l2ProtocolProfileModeContext
      self.l2ProtocolProfileModeContext = None
      context.commit()

def gotoL2PtProfileMode( mode, args ):
   childMode = mode.childMode( L2ProtocolProfileMode, 
                               profileName=args[ 'PROFILENAME' ] )
   mode.session_.gotoChildMode( childMode )

def deleteL2PtProfile( mode, args ):
   # Delete L2Pt profile from config
   del l2PtProfileConfig.profile[ args[ 'PROFILENAME' ] ]

matcherL2ProfileName = CliMatcher.DynamicNameMatcher(
                         lambda mode: l2PtProfileConfig.profile,
                         'L2 protocol forwarding profile name',
                         pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

#--------------------------------------------------------------------------------
# [ no | default ] forwarding profile PROFILENAME
#--------------------------------------------------------------------------------
class ForwardingProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding profile PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'forwarding': '',
      'profile': 'Enter forwarding profile configuration mode',
      'PROFILENAME': matcherL2ProfileName,
   }
   handler = gotoL2PtProfileMode
   noOrDefaultHandler = deleteL2PtProfile

L2ProtocolMode.addCommandClass( ForwardingProfileProfilenameCmd )

protocols = dict( zip( cliTokenToL2PtProtocol, cliTokenToL2PtProtocol ) )
supportedActions = { tacL2PtAction.forward : tacL2PtAction.forward }
tagFormats = { tacL2PtTagFormat.tagged : tacL2PtTagFormat.tagged,
               tacL2PtTagFormat.untagged : tacL2PtTagFormat.untagged }

def getProtocols( mode ):
   if bridgingHwCapabilities:
      if bridgingHwCapabilities.l2ProtocolTransparencySubset:
         return { tacL2PtProtocol.lacp : 'lacp', tacL2PtProtocol.lldp : 'lldp',
                  tacL2PtProtocol.stp : 'stp' }
   return protocols

def getTagFormats( mode ):
   if bridgingHwCapabilities:
      if bridgingHwCapabilities.l2ProtocolTransparencySubset:
         # we only support tacL2PtTagFormat.all
         return {}
   return tagFormats

dynProtoMatcher = CliMatcher.DynamicKeywordMatcher( getProtocols )
dynTagMatcher = CliMatcher.DynamicKeywordMatcher( getTagFormats )

def configL2ProtocolFwdProtocol( mode, args ):
   protocol = args[ 'PROTOCOL' ]
   action = args[ 'ACTION' ]
   tagFormat = args.get( 'TAGFORMAT' )
   macAddr = l2ProtocolMacAddress( cliTokenToL2PtProtocol[ protocol ] )
   if not tagFormat:
      tagFormat = tacL2PtTagFormat.all
   l2PtProtocolInfo = Tac.newInstance( "Ebra::L2Pt::L2ProtocolInfo", macAddr,
                                       tagFormat, 0 )
   l2ProtocolMatch = Tac.newInstance( "Ebra::L2Pt::L2ProtocolMatch", macAddr,
                                      tagFormat, 0 )
   profile = mode.l2ProtocolProfileModeContext.editProfile()

   if CliCommand.isNoOrDefaultCmd( args ):
      del profile.protocolToAction[ l2PtProtocolInfo ]
      seq = profile.protocolInfoToSeq.get( l2ProtocolMatch )
      if seq is not None:
         del profile.seqToProtocolInfo[ seq ]
         del profile.protocolInfoToSeq[ l2ProtocolMatch ]
   else:
      if( len( profile.seqToProtocolInfo.keys() ) == 0 ):
         seq = 10
      else:
         seq = sorted( profile.seqToProtocolInfo.keys() )[-1] + 10
      if l2ProtocolMatch not in profile.protocolInfoToSeq:
         l2ProtocolFwdAction = Tac.newInstance( "Ebra::L2Pt::L2ProtocolFwdAction",
                                                action )
         l2ProtocolFwdInfo = Tac.newInstance( "Ebra::L2Pt::L2ProtocolFwdInfo",
                                              l2ProtocolMatch, l2ProtocolFwdAction )
         profile.protocolToAction[ l2PtProtocolInfo ] = action
         profile.protocolInfoToSeq[ l2ProtocolMatch ] = seq
         profile.seqToProtocolInfo[ seq ] = l2ProtocolFwdInfo

#------------------------------------------------------------------------
# [ no | default ] PROTOCOL [ TAGFORMAT ] ACTION in L2ProtocolProfileMode
#------------------------------------------------------------------------
class L2ProtocolFwdProtocolConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'PROTOCOL [ TAGFORMAT ] ACTION'
   noOrDefaultSyntax = syntax
   data = {
      'PROTOCOL': dynProtoMatcher,
      'TAGFORMAT': dynTagMatcher,
      'ACTION': CliMatcher.EnumMatcher( supportedActions ),
   }
   handler = configL2ProtocolFwdProtocol
   noOrDefaultHandler = configL2ProtocolFwdProtocol
L2ProtocolProfileMode.addCommandClass( L2ProtocolFwdProtocolConfigCmd )

class L2ProtocolIntfProfileModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Ethernet' )

IntfCli.IntfConfigMode.addModelet( L2ProtocolIntfProfileModelet )

def setL2PtProfile( mode, args ):
   intf = mode.intf
   profileName = args.get( 'PROFILENAME' )
   if CliCommand.isNoOrDefaultCmd( args ):
      if profileName:
         if intf.name in l2PtIntfConfig.intfToProfile and \
               l2PtIntfConfig.intfToProfile[ intf.name ] == profileName:
            del l2PtIntfConfig.intfToProfile[ intf.name ]
      else:
         del l2PtIntfConfig.intfToProfile[ intf.name ]
   else:
      l2PtIntfConfig.intfToProfile[ intf.name ] = profileName

# The guard is configured with "forwarding" token so that we don't
# interfere with other Intf mode commands that start with "l2-protocol"
#------------------------------------------------------------------------
# [ no | default ] l2-protocol forwarding profile PROFILENAME
#------------------------------------------------------------------------
class L2ProtocolProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'l2-protocol forwarding profile PROFILENAME'
   noOrDefaultSyntax = 'l2-protocol forwarding profile [ PROFILENAME ]'
   data = {
      'l2-protocol': 'Set L2 protocol characteristics of the interface',
      'forwarding': CliCommand.guardedKeyword( 'forwarding',
                        helpdesc='',
                        guard=l2ProtocolFwdModeGuard ), 
      'profile': 'Enter forwarding profile configuration mode',
      'PROFILENAME': matcherL2ProfileName,
   }
   handler = setL2PtProfile
   noOrDefaultHandler = setL2PtProfile

L2ProtocolIntfProfileModelet.addCommandClass( L2ProtocolProfilenameCmd )

#---------------------------------------------------------------------------------
# 1) To delete L2 Protocol forwarding config when the corresponding interface gets
#    deleted.
# 2) Support for default interface command.
#---------------------------------------------------------------------------------
class L2ProtocolFwdIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "L2ProtocolFwdIntfJanitor: interface", self.intf_.name, "going away" )
      del l2PtIntfConfig.intfToProfile[ self.intf_.name ]


def Plugin( entityManager ):
   global l2PtProfileConfig
   global l2PtIntfConfig
   global bridgingHwCapabilities

   l2PtProfileConfig = ConfigMount.mount( entityManager,
         "l2protocolforwarding/profileconfig",
         "Ebra::L2Pt::L2PtProfileConfig", "w" )
   l2PtIntfConfig = ConfigMount.mount( entityManager,
         "l2protocolforwarding/intfconfig",
         "Ebra::L2Pt::L2PtIntfConfig", "w" )

   IntfCli.Intf.registerDependentClass( L2ProtocolFwdIntfJanitor )
   mount = LazyMount.mount
   bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                                   "Bridging::HwCapabilities", "r" )
