#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliToken
import CliParser
import ShowCommand
from CliMatcher import KeywordMatcher
import Tac
from MrouteCli import isMatchingSG
from Arnet.NsLib import runMaybeInNetNs
from IpLibConsts import DEFAULT_VRF
from DeviceNameLib import kernelIntfToEosIntf
import LazyMount, Cell
from Arnet.NsLib import DEFAULT_NS
import Arnet
import CliPlugin.VrfCli as VrfCli
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliToken.McastCommon import multicastMatcherForShow
from McastCommonCliLib import ( mcastRoutingSupported,
                                mcastRoutingSupportedGuard,
                                mcast6RoutingSupportedGuard,
                                mcastGenRoutingSupportedGuard )
from RouterMulticastCliLib import AddressFamily
import struct, socket
import re
import SmashLazyMount
from collections import defaultdict
allVrfStatusLocal = None
bessAgentStatus = None
_entityManager = None

def mcastFibIpv4SupportedGuard( mode=None, token=None ):
   if mcastRoutingSupported( _entityManager.root(), routingHardwareStatus=None ):
      return None
   return CliParser.guardNotThisPlatform

def nsFromVrf( vrfName=DEFAULT_VRF ):
   ns = DEFAULT_NS
   if vrfName == DEFAULT_VRF:
      return ns
   elif vrfName in allVrfStatusLocal.vrf:
      ns = allVrfStatusLocal.vrf[ vrfName ].networkNamespace
      return ns
   else:
      return None

def ntoh( s, af=AddressFamily.ipv4 ):
   # the /proc/net/ip_mr_* files give a hex value of a network-byte-order
   # value.  int() turns it into a 32-bit unsigned value; struct.pack turns
   # it into the 4 bytes that socket.inet_ntoa() wants.
   if af == AddressFamily.ipv4:
      addr = int( s, 16 )
      return socket.inet_ntoa( struct.pack( "I", addr ) )
   elif af == AddressFamily.ipv6:
      ip = s.replace( ':', '' )
      addr = ip.decode( 'hex' )
      v6Addr = socket.inet_ntop( socket.AF_INET6, struct.pack( '>4I',
         *struct.unpack( '>4I', addr ) ) )
      return v6Addr

def readKernelVifs( vrfName=DEFAULT_VRF, af=AddressFamily.ipv4 ):
   if vrfName is not DEFAULT_VRF:
      ns = nsFromVrf( vrfName )
      if af == AddressFamily.ipv4:
         vifOutput = runMaybeInNetNs( ns , [ 'cat', '/proc/net/ip_mr_vif' ],
               asRoot=True, stdout=Tac.CAPTURE )
      elif af == AddressFamily.ipv6:
         vifOutput = runMaybeInNetNs( ns, [ 'cat', '/proc/net/ip6_mr_vif' ],
               asRoot=True, stdout=Tac.CAPTURE )

      lines = vifOutput.rstrip().split( '\n' )
   else:
      if af == AddressFamily.ipv4:
         lines = file( '/proc/net/ip_mr_vif' ).readlines()
      elif af == AddressFamily.ipv6:
         lines = file( '/proc/net/ip6_mr_vif' ).readlines()
   return lines

def readKernelRoutes( vrfName=DEFAULT_VRF, af=AddressFamily.ipv4 ):
   if vrfName is not DEFAULT_VRF:
      ns = nsFromVrf( vrfName )
      if af == AddressFamily.ipv4:
         mrOutput = runMaybeInNetNs( ns , [ 'cat', '/proc/net/ip_mr_cache' ],
               asRoot=True, stdout=Tac.CAPTURE )
      elif af == AddressFamily.ipv6:
         mrOutput = runMaybeInNetNs( ns, [ 'cat', '/proc/net/ip6_mr_cache' ],
               asRoot=True, stdout=Tac.CAPTURE )
      lines = mrOutput.rstrip().split( '\n' )
   else:
      if af == AddressFamily.ipv4:
         lines = file( '/proc/net/ip_mr_cache' ).readlines()
      elif af == AddressFamily.ipv6:
         lines = file( '/proc/net/ip6_mr_cache' ).readlines()
   return lines

def showMfibSoftware( detail=False, source=None, group=None,
                        vrfName=DEFAULT_VRF, af=AddressFamily.ipv4 ):
   """Display the contents of the kernel multicast fib (the "software
      mfib"), optionally limiting output to matching sources and
      groups."""
   if( ( af == AddressFamily.ipv4 and bessAgentStatus.v4Enabled ) or \
       ( af == AddressFamily.ipv6 and bessAgentStatus.v6Enabled ) ):
      showMfibSoftwareBess( detail=detail, source=source, group=group,
                            vrfName=vrfName, af=af )
   else:
      showMfibSoftwareKernel( detail=detail, source=source, group=group,
                              vrfName=vrfName, af=af )

def showMfibSoftwareBess( detail=False, source=None, group=None,
                          vrfName=DEFAULT_VRF, af=AddressFamily.ipv4 ):

   class ShowMfibCounterInfo:

      def __init__( self ):
         self.byteCount = None
         self.packetCount = None
         self.rpfFailureCount = None
         self.packetDroppedCount = None
         self.packetQueuedCount = None
   if source:
      source = str( source )
   if group:
      group = str( group )

   path = Tac.Type( "Sfe::Multicast::MrouteCounterTable" ).mountPath( af, vrfName )
   bessMrouteCounterTable = SmashLazyMount.mount(
         _entityManager,
         path,
         'Sfe::Multicast::MrouteCounterTable',
         SmashLazyMount.mountInfo( 'reader' ) )

   # Traverse through bessCounterInfo for counters related to Mfib
   showMfibCounterInfoDict = defaultdict( ShowMfibCounterInfo )

   if source and group:
      source = Arnet.IpGenPrefix( source )
      group = Arnet.IpGenPrefix( group )
      count = bessMrouteCounterTable.mroute.get(
            Tac.Value( "Routing::Multicast::Fib::IpGenRouteKey",
               source, group ) )
      if not count:
         return
      showMfibCounterInfo = showMfibCounterInfoDict[ ( source, group ) ]
      showMfibCounterInfo.packetCount = count.pktsIn
      showMfibCounterInfo.byteCount = count.bytesIn
      showMfibCounterInfo.rpfFailureCount = count.wrongVif
      showMfibCounterInfo.packetDroppedCount = count.pktsDropped
      showMfibCounterInfo.packetQueuedCount = count.pktsQueued
   else:
      for routeKey, count in bessMrouteCounterTable.mroute.iteritems():
         # Print only matching group entries if specified
         if group and str( routeKey.g.ipGenAddr ) != group:
            continue

         showMfibCounterInfo = showMfibCounterInfoDict[ ( routeKey.s, routeKey.g ) ]
         showMfibCounterInfo.packetCount = count.pktsIn
         showMfibCounterInfo.byteCount = count.bytesIn
         showMfibCounterInfo.rpfFailureCount = count.wrongVif
         showMfibCounterInfo.packetDroppedCount = count.pktsDropped
         showMfibCounterInfo.packetQueuedCount = count.pktsQueued

   for rKey, counterInfo in showMfibCounterInfoDict.iteritems():
      g = str( rKey[ 1 ].ipGenAddr ).upper()
      s = str( rKey[ 0 ].ipGenAddr ).upper()
      print g, s
      print "\tPackets Received:", counterInfo.packetCount
      print "\tBytes Received  :", counterInfo.byteCount
      print "\tRPF Failures    :", counterInfo.rpfFailureCount
      if detail:
         print "\tPackets Queued/Dropped :%s/%s" % \
               ( counterInfo.packetQueuedCount, counterInfo.packetDroppedCount )

def showMfibSoftwareKernel( detail=False, source=None, group=None,
                        vrfName=DEFAULT_VRF, af=AddressFamily.ipv4 ):
   """Display the contents of the kernel multicast fib (the "software
      mfib"), optionally limiting output to matching sources and
      groups."""
   ns = nsFromVrf( vrfName )
   if not ns:
      return
   vifs = { -1: 'Unresolved' }
   if source:
      source = str( source )
   if group:
      group = str( group )
   vifLines = readKernelVifs( vrfName, af )
   for line in vifLines[ 1: ]:
      ( vifi, intf ) = line.split()[ 0:2 ]
      # This is lazy: the right way is to make a list of kernel
      # interface names from interfaces/status/all and use that
      # mapping.
      vifs[ int( vifi ) ] = kernelIntfToEosIntf( intf )

   lines = readKernelRoutes( vrfName, af )
   noPktInfo = False
   headerLine = lines[ 0 ].strip()
   if re.search( r'Group\s+Origin\s+Iif\s+Pkts\s+Bytes\s+Wrong\s+Oifs',
         headerLine ):
      noPktInfo = True

   for line in lines[ 1: ]:
      vals = line.split()
      if noPktInfo:
         ( g, s, ivif, pkts, byts, rpffail ) = vals[ 0:6 ]
         ovifs = vals[ 6: ]
         pktsdropped = "Unknown"
         pktsqed = "Unknown"
      else:
         ( g, s, ivif, pkts, pktsdropped, pktsqed, byts, rpffail ) = vals[ 0:8 ]
         ovifs = vals[ 8: ]
      s = ntoh( s, af )
      g = ntoh( g, af )
      # Print only entries that match the source and group that the
      # user specified.
      if not isMatchingSG( s, g, source, group ):
         continue

      iif = vifs.get( int( ivif ), 'bad vifi ' + ivif )
      oifs = [ vifs.get( int( ov.split( ':' )[ 0 ] ), 'bad ovif ' + ov ) for
                ov in ovifs ]
      print g.upper(), s.upper()
      print "\t", iif, "(iif)"
      for o in oifs:
         print "\t", o

      print "\tPackets Received:", pkts
      print "\tBytes Received  :", byts
      print "\tRPF Failures    :", rpffail
      if detail:
         print "\tPackets Queued/Dropped :", pktsqed, "/", pktsdropped

def showIpv4MfibSoftware( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   group = args.get( 'GROUP' )
   source = args.get( 'SOURCE' )
   detail = 'detail' in args

   return showMfibSoftware( detail, source, group, vrfName,
         af=AddressFamily.ipv4 )

def showIpv6MfibSoftware( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   group = args.get( 'GROUP' )
   source = args.get( 'SOURCE' )
   detail = 'detail' in args

   return showMfibSoftware( detail, source, group, vrfName,
         af=AddressFamily.ipv6 )

vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='VRF name' )

groupIpv4AddrMatcher = IpAddrMatcher( helpdesc='Group address' )
srcIpv4AddrMatcher = IpAddrMatcher( helpdesc='Source address' )
groupIpv6AddrMatcher = Ip6AddrMatcher( helpdesc='Group address' )
srcIpv6AddrMatcher = Ip6AddrMatcher( helpdesc='Source address' )

softwareKwMatcher = KeywordMatcher( 'software', helpdesc='Software multicast FIB' )
detailKwMatcher = KeywordMatcher(
      'detail',
      helpdesc='Display information in detail' )
ipv4Node = CliCommand.Node( matcher=CliToken.Ipv4.ipv4MatcherForShow,
                            guard=mcastRoutingSupportedGuard )
ipv6Node = CliCommand.Node( matcher=CliToken.Ipv6.ipv6MatcherForShow,
                            guard=mcast6RoutingSupportedGuard )

fibKwNode = CliCommand.guardedKeyword(
      'fib',
      helpdesc='Multicast FIB information',
      guard=mcastGenRoutingSupportedGuard )

mfibKwDeprecatedNode = CliCommand.guardedKeyword(
      'mfib',
      helpdesc='Multicast FIB',
      guard=mcastRoutingSupportedGuard,
      deprecatedByCmd='show multicast fib ipv4' )

#-------------------------------------------------------------------------------
# show multicast fib ipv4 [ VRF ] software [ GROUP [ SOURCE ] ] [ detail ]
#-------------------------------------------------------------------------------
class MulticastFibIpv4SoftwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show multicast fib ipv4 [ VRF ] software [ GROUP [ SOURCE ] ]
            [ detail ]'''
   data = {
      'multicast' : multicastMatcherForShow,
      'fib' : fibKwNode,
      'ipv4' : ipv4Node,
      'VRF' : vrfExprFactory,
      'software' : softwareKwMatcher,
      'GROUP' : groupIpv4AddrMatcher,
      'SOURCE' : srcIpv4AddrMatcher,
      'detail' : detailKwMatcher
   }
   handler = showIpv4MfibSoftware

BasicCli.addShowCommandClass( MulticastFibIpv4SoftwareCmd )

#-------------------------------------------------------------------------------
# show multicast fib ipv6 [ VRF ] software [ GROUP [ SOURCE ] ] [ detail ]
#-------------------------------------------------------------------------------
class MulticastFibIpv6SoftwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show multicast fib ipv6 [ VRF ] software [ GROUP [ SOURCE ] ]
            [ detail ]'''
   data = {
      'multicast' : multicastMatcherForShow,
      'fib' : fibKwNode,
      'ipv6' : ipv6Node,
      'VRF' : vrfExprFactory,
      'software' : softwareKwMatcher,
      'GROUP' : groupIpv6AddrMatcher,
      'SOURCE' : srcIpv6AddrMatcher,
      'detail' : detailKwMatcher
   }
   handler = showIpv6MfibSoftware

BasicCli.addShowCommandClass( MulticastFibIpv6SoftwareCmd )

#-------------------------------------------------------------------------------
# Legacy:
# show ip mfib [ VRF ] software [ GROUP [ SOURCE ] ] [ detail ]
#-------------------------------------------------------------------------------
class IpMfibSoftwareLegacyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip mfib [ VRF ] software [ GROUP [ SOURCE ] ]
            [ detail ]'''
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'mfib' : mfibKwDeprecatedNode,
      'VRF' : vrfExprFactory,
      'software' : softwareKwMatcher,
      'GROUP' : groupIpv4AddrMatcher,
      'SOURCE' : srcIpv4AddrMatcher,
      'detail' : detailKwMatcher
   }
   handler = showIpv4MfibSoftware

BasicCli.addShowCommandClass( IpMfibSoftwareLegacyCmd )


def Plugin( entityManager ):

   global allVrfStatusLocal, bessAgentStatus, _entityManager
   _entityManager = entityManager

   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   bessAgentStatus = LazyMount.mount( entityManager,
                                      "bess/agentstatus",
                                      "McastCommon::BessAgentStatus", "r" )
