#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Multicast Routing related commands.
# Configure commands:
# [no] ip mfib cache-entries unresolved max
# [no] ip mfib packet-buffers unresolved max
#
# Show/Enable-level commands:
# -  show ip mfib software
#-------------------------------------------------------------------------------

import BasicCli
import CliCommand
from CliMatcher import KeywordMatcher, IntegerMatcher
import CliToken.Clear
import CliToken.Ip
from IpLibConsts import DEFAULT_VRF
import RouterMulticastCliLib
import Tracing
from McastCommonCli import mcastRoutingSupportedGuard
from RouterMulticastCliLib import configGetters
from McastCommonCliLib import AddressFamily

traceHandle = Tracing.Handle( "KernelMfibCli" )
t0 = traceHandle.trace0

KernelMfibConfigColl = 'Routing::KernelMfib::ConfigColl'

( kernelMfibConfigColl,
  kernelMfibConfigCollFromMode,
  kernelMfibConfig,
  kernelMfibConfigFromMode ) = configGetters( KernelMfibConfigColl )

def _getKernelMfibConfigFromMode( mode, args ):
   # AF is indicated by config mode; for legacy commands, ipunknown indicates ipv4
   af = getattr( mode, 'af', AddressFamily.ipv4 )
   if af == AddressFamily.ipunknown:
      af = AddressFamily.ipv4
   vrfName = getattr( mode, 'vrfName', DEFAULT_VRF )
   return kernelMfibConfig( af, vrfName )

mfibNode = CliCommand.guardedKeyword(
      'mfib',
      helpdesc='Multicast FIB',
      guard=mcastRoutingSupportedGuard )

cacheEntriesKwMatcher = KeywordMatcher( 'cache-entries',
      helpdesc='MFIB cache entries' )
unresolvedKwMatcher = KeywordMatcher(
      'unresolved',
      helpdesc='Unresolved MFIB route' )
maxCacheKwMatcher = KeywordMatcher(
      'max',
      helpdesc='Maximum number unresolved MFIB entries' )
maxUnresCacheEntriesMatcher = IntegerMatcher( 10, 1000000,
      helpdesc='Maximum number of unresolved MFIB cache entries' )

def noMaxUnresCacheEntries( mode, args ):
   mfibConfig = _getKernelMfibConfigFromMode( mode, args )
   mfibConfig.maxUnresolvedCacheEntries = \
      mfibConfig.maxUnresolvedCacheEntriesDefault

def setMaxUnresCacheEntries( mode, args ):
   maxUnresCacheEntries_ = args.get( 'MAX_ENTRIES' )
   config = _getKernelMfibConfigFromMode( mode, args )
   config.maxUnresolvedCacheEntries = maxUnresCacheEntries_

#-------------------------------------------------------------------------------
# (config-router-multicast-*-ipv4) unresolved cache-entries max <10-1000000>
#-------------------------------------------------------------------------------
class UnresCacheEntriesMaxCmd( CliCommand.CliCommandClass ):
   syntax = 'unresolved cache-entries max MAX_ENTRIES'
   noOrDefaultSyntax = 'unresolved cache-entries max ...'
   data = {
      'unresolved' : unresolvedKwMatcher,
      'cache-entries' : cacheEntriesKwMatcher,
      'max' : maxCacheKwMatcher,
      'MAX_ENTRIES' : maxUnresCacheEntriesMatcher
   }

   handler = setMaxUnresCacheEntries
   noOrDefaultHandler = noMaxUnresCacheEntries

RouterMulticastCliLib.RouterMulticastIpv4Modelet.addCommandClass(
      UnresCacheEntriesMaxCmd )

#-------------------------------------------------------------------------------
# Deprecated:
# (config-[vrf]) [no] ip mfib cache-entries unresolved max <10-1000000>
#-------------------------------------------------------------------------------
class IpMfibCacheEntriesUnresMaxLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip mfib cache-entries unresolved max MAX_ENTRIES'
   noOrDefaultSyntax = 'ip mfib cache-entries unresolved ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'mfib' : mfibNode,
      'cache-entries' : cacheEntriesKwMatcher,
      'unresolved' : unresolvedKwMatcher,
      'max' : maxCacheKwMatcher,
      'MAX_ENTRIES' : maxUnresCacheEntriesMatcher
   }

   handler = setMaxUnresCacheEntries
   noOrDefaultHandler = noMaxUnresCacheEntries

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass(
      IpMfibCacheEntriesUnresMaxLegacyCmd )

BasicCli.GlobalConfigMode.addCommandClass( IpMfibCacheEntriesUnresMaxLegacyCmd )

pktBuffersKwMatcher = KeywordMatcher(
      'packet-buffers',
      helpdesc='Maximum number unresolved multicast packets queued' )
maxUnresPktMatcher = IntegerMatcher( 3, 1000000, 
      helpdesc='Maximum number of unresolved multicast packets queued' )
maxPktKwMatcher = KeywordMatcher(
      'max',
      helpdesc='Maximum number unresolved packets queued' )

def noMaxUnresPktBuffers( mode, args ):
   mfibConfig = _getKernelMfibConfigFromMode( mode, args )
   mfibConfig.maxUnresolvedPktBuffers = mfibConfig.maxUnresolvedPktBuffersDefault

def setMaxUnresPktBuffers( mode, args ):
   maxUnresPktBuffers_ = args.get( 'MAX_BUFFERS' )
   mfibConfig = _getKernelMfibConfigFromMode( mode, args )
   mfibConfig.maxUnresolvedPktBuffers = maxUnresPktBuffers_

#-------------------------------------------------------------------------------
# (config-router-multicast-*-ipv4) unresolved packet-buffers max <3-1000000>
#-------------------------------------------------------------------------------
class UnresPacketBuffersMaxCmd( CliCommand.CliCommandClass ):
   syntax = 'unresolved packet-buffers max MAX_BUFFERS'
   noOrDefaultSyntax = 'unresolved packet-buffers max ...'
   data = {
      'unresolved' : unresolvedKwMatcher,
      'packet-buffers' : pktBuffersKwMatcher,
      'max' : maxPktKwMatcher,
      'MAX_BUFFERS' : maxUnresPktMatcher
   }

   handler = setMaxUnresPktBuffers
   noOrDefaultHandler = noMaxUnresPktBuffers

RouterMulticastCliLib.RouterMulticastIpv4Modelet.addCommandClass(
      UnresPacketBuffersMaxCmd )

#-------------------------------------------------------------------------------
# Deprecated:
# (config-[vrf]) [no] ip mfib packet-buffers unresolved max <3-1000000>
#-------------------------------------------------------------------------------
class IpMfibPacketBuffersUnresMaxLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip mfib packet-buffers unresolved max MAX_BUFFERS'
   noOrDefaultSyntax = 'ip mfib packet-buffers unresolved max ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'mfib' : mfibNode,
      'packet-buffers' : pktBuffersKwMatcher,
      'unresolved' : unresolvedKwMatcher,
      'max' : maxPktKwMatcher,
      'MAX_BUFFERS' : maxUnresPktMatcher
   }

   handler = setMaxUnresPktBuffers
   noOrDefaultHandler = noMaxUnresPktBuffers

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass(
      IpMfibPacketBuffersUnresMaxLegacyCmd )
BasicCli.GlobalConfigMode.addCommandClass( IpMfibPacketBuffersUnresMaxLegacyCmd )


class RouterMulticastListener( RouterMulticastCliLib.RouterModeCallbackBase ):
   def modeDeleted( self, **kwargs ):
      for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
         configColl = kernelMfibConfigColl( af )
         for vrfName, config in configColl.vrfConfig.iteritems():
            if vrfName == DEFAULT_VRF:
               config.reset()
            else:
               del configColl.vrfConfig[ vrfName ]

   def vrfModeEntered( self, vrfName, **kwargs ):
      for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
         kernelMfibConfig( af, vrfName )

   def vrfModeDeleted( self, vrfName, **kwargs ):
      for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
         configColl = kernelMfibConfigColl( af )
         if vrfName == DEFAULT_VRF:
            configColl.vrfConfig[ DEFAULT_VRF ].reset()
         else:
            del configColl.vrfConfig[ vrfName ]

   def afModeDeleted( self, vrfName, af, **kwargs ):
      kernelMfibConfig( af, vrfName ).reset()


def Plugin ( entityManager ):
   configTypes = [ KernelMfibConfigColl ]

   RouterMulticastCliLib.doConfigMounts( entityManager, configTypes )
   RouterMulticastCliLib.RouterMulticastMode.registerCallback(
         RouterMulticastListener() )
