# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import os
import sys

import AgentCommandRequest
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliToken.Ip
import ConfigMount
from CliPlugin import AgentLibShowCommands, ShowTaskSchedulerModel
from CliPlugin.IraRouterKernel import RouterKernelConfigMode
from CliPlugin.IraRouterKernel import routerKernelVrfHook
import CliPlugin.IraIpCli as IraIpCli
import CliPlugin.IraIp6Cli as IraIp6Cli
import CliPlugin.VrfCli as VrfCli
import CliPlugin.TechSupportCli as TechSupportCli
from CliMode.KernelFib import RouterKernelAfMode
from IpLibConsts import DEFAULT_VRF
import KernelFibAgent
import LazyMount
import ShowCommand
import SmashLazyMount
import Tracing
import Tac

traceHandle = Tracing.Handle( 'KernelFibCli' )
t5 = traceHandle.trace5 # Info

defaultKernelRoutePreference = 40

myEntManager = None
agentCmdConfig = None
routingHardwareStatus = None
kernelfibConfig = None
kernelfibConfig6 = None
vrfNameStatus = None
unprogrammedRoutePrinter = None

matcherKernelfib = CliMatcher.KeywordMatcher( 'kernelFib',
      helpdesc='Show detailed state of KernelFib agent' )
matcherKernel = CliMatcher.KeywordMatcher( 'kernel',
      helpdesc='Show only kernel routes' )
matcherUnprogrammed = CliMatcher.KeywordMatcher( 'unprogrammed',
      helpdesc='Uprogrammed routes information' )

#-------------------------------------------------------------------------------
# Create per-vrf config ( called when vrf is created/ router-kernel..mode 
# is entered)
#-------------------------------------------------------------------------------
def _routeVrfCreation( vrfName, status ):
   if status:
      kernelfibConfig.vrfConfig.newMember( vrfName )
      kernelfibConfig6.vrfConfig.newMember( vrfName )
   else:
      del kernelfibConfig.vrfConfig[ vrfName ]
      del kernelfibConfig6.vrfConfig[ vrfName ]

# Register callback handler for creating/deleting vrf config
routerKernelVrfHook.addExtension( _routeVrfCreation )

#-------------------------------------------------------------------------------
# router kernel - address family config mode. 
#-------------------------------------------------------------------------------
class RouterKernelAfConfigMode( RouterKernelAfMode, BasicCli.ConfigModeBase ):
   name = 'Kernel route address family configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = parent.vrfName
      RouterKernelAfMode.__init__( self, ( self.vrfName, addrFamily ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def clearConfig( self ):
      setKernelRouteAdminDistance( self, self.addrFamily, self.vrfName )

#-------------------------------------------------------------------------------
# [ no | default ] address-family ADDR_FAMILY
#-------------------------------------------------------------------------------
class RouterKernelAfConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family ADDR_FAMILY'
   noOrDefaultSyntax = syntax
   data = {
      'address-family' : 'Address family configuration options',
      'ADDR_FAMILY' : CliMatcher.EnumMatcher( {
         'ipv4' : 'IPv4 related',
         'ipv6' : 'IPv6 related'
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      addrFamily = args[ 'ADDR_FAMILY' ]
      childMode = mode.childMode( RouterKernelAfConfigMode, addrFamily=addrFamily )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      addrFamily = args[ 'ADDR_FAMILY' ]
      childMode = mode.childMode( RouterKernelAfConfigMode, addrFamily=addrFamily )
      childMode.clearConfig()

RouterKernelConfigMode.addCommandClass( RouterKernelAfConfigModeCmd )

# Set / Reset the preference values for kernel routes
def setKernelRouteAdminDistance( mode, addrFamily, vrfName, distance=None ):
   if addrFamily == 'ipv4':
      config = kernelfibConfig
   elif addrFamily == 'ipv6':
      config = kernelfibConfig6
   else:
      mode.addError( 'Un-supported address family %s.' % addrFamily )
      return

   if vrfName not in config.vrfConfig:
      mode.addError( 'No such VRF %s.' % vrfName )
      return

   if distance:
      config.vrfConfig[ vrfName ].preference = distance
   else:
      config.vrfConfig[ vrfName ].preference = defaultKernelRoutePreference

#--------------------------------------------------------------------------------
# [ no | default ] distance DISTANCE
#--------------------------------------------------------------------------------
class DistanceDistanceCmd( CliCommand.CliCommandClass ):
   syntax = 'distance DISTANCE'
   noOrDefaultSyntax = syntax
   data = {
      'distance' : 'Set the admin distance for non EOS routes',
      'DISTANCE' : CliMatcher.IntegerMatcher( 1, 255, helpdesc='Distance value' ),
   }

   # Set the kernel route preference to specified value
   @staticmethod
   def handler( mode, args ):
      distance = args[ 'DISTANCE' ]
      assert mode.vrfName
      assert mode.addrFamily
      t5( 'Setting kernel route preference for %s (%s) to %d' % \
        ( mode.vrfName, mode.addrFamily, distance ) )
      setKernelRouteAdminDistance( mode, mode.addrFamily, mode.vrfName, distance )

   # Reset the kernel route preference to default
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      assert mode.vrfName
      assert mode.addrFamily
      t5( 'Setting kernel route preference for %s (%s) to default' % \
        ( mode.vrfName, mode.addrFamily ) )
      setKernelRouteAdminDistance( mode, mode.addrFamily, mode.vrfName )

RouterKernelAfConfigMode.addCommandClass( DistanceDistanceCmd )

#--------------------------------------------------------------------------------
# show ip route [ VRF ] kernel unprogrammed
#--------------------------------------------------------------------------------
class IpRouteKernelUnprogrammedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] kernel unprogrammed'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'route' : IraIpCli.routeMatcherAfterShowIp,
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Display VRF state',
                                     inclDefaultVrf=True ),
      'kernel' : matcherKernel,
      'unprogrammed' : matcherUnprogrammed,
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
      if vrfNameStatus.vrfIdExists( vrfName ) :
         vrfId = vrfNameStatus.nameToIdMap.vrfNameToId[ vrfName ]
         fd = sys.stdout.fileno()
         fmt = mode.session_.outputFormat()
         unprogrammedRoutePrinter.printRoutes( vrfId, False, fd, fmt )

BasicCli.addShowCommandClass( IpRouteKernelUnprogrammedCmd )

#--------------------------------------------------------------------------------
# show ipv6 route [ VRF ] kernel unprogrammed
#--------------------------------------------------------------------------------
def ipV6Vrf( mode, token ):
   if routingHardwareStatus.vrfCapability.ipv6EnabledDefault is False:
      return CliParser.guardNotThisPlatform
   return None

class Ipv6RouteKernelUnprogrammedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] kernel unprogrammed'
   data = {
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'route' : IraIp6Cli.routeAfterShowIpv6Matcher,
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Display VRF state',
                                     guard=ipV6Vrf,
                                     inclDefaultVrf=True ),
      'kernel' : matcherKernel,
      'unprogrammed' : matcherUnprogrammed,
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
      if vrfNameStatus.vrfIdExists( vrfName ) :
         vrfId = vrfNameStatus.nameToIdMap.vrfNameToId[ vrfName ]
         fd = sys.stdout.fileno()
         fmt = mode.session_.outputFormat()
         unprogrammedRoutePrinter.printRoutes( vrfId, True, fd, fmt )

BasicCli.addShowCommandClass( Ipv6RouteKernelUnprogrammedCmd )

#--------------------------------------------------------------------------------
# show tech-support kernelFib [ detail ]
#--------------------------------------------------------------------------------
class TechSupportKernelfibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support kernelFib [ detail ]'
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'kernelFib' : matcherKernelfib,
      'detail' : 'Show verbose state',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      detail = 'detail' in args
      command = 'DUMP_DETAILED_STATE' if detail else 'DUMP_STATE'
      timeout = os.environ.get( 'KERNELFIB_BTEST', 120 )
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            KernelFibAgent.name(),
                                            'tech-support', command,
                                            timeout=timeout )

BasicCli.addShowCommandClass( TechSupportKernelfibCmd )

TechSupportCli.registerShowTechSupportCmdCallback( 
                  '2014-04-07 04:42:20',
                  lambda: [ 'show ip route kernel unprogrammed',
                            'show ipv6 route kernel unprogrammed' ] )

matcherScheduler = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'scheduler',
                                      helpdesc='KernelFib scheduler statistics' ) )

#--------------------------------------------------------------------------------
# Deprecate 'show kernelFib scheduler reset'.  It was a
# lazy implementation and doesn't fit the new one.
# --------------------------------------------------------------------------------
class BgpMonitoringInternalSchedulerResetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernelFib scheduler reset'
   data = {
      'kernelFib' : matcherKernelfib,
      'scheduler' : matcherScheduler,
      'reset': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'reset',
         helpdesc='Reset scheduler statistics' ),
         deprecatedByCmd='clear agent kernelFib task scheduler' ),
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      # Since it's deprecated, this should never get called, but
      # if it is, let's make sure that the user knows that it didn't
      # work.
      mode.addError( 'Use "clear agent kernelFib task scheduler"' )

BasicCli.addShowCommandClass( BgpMonitoringInternalSchedulerResetCmd )

#--------------------------------------------------------------------------------
# show kernelFib scheduler [ OPTIONS ]
#--------------------------------------------------------------------------------
class KernelfibSchedulerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show kernelFib scheduler [ OPTIONS ]'
   data = {
      'kernelFib' : matcherKernelfib,
      'scheduler' : matcherScheduler,
      'OPTIONS' : CliCommand.SetEnumMatcher( {
         'verbose' : 'Include more information',
         'internal' : 'Display memory addresses of tasks',
         'history' : 'Display scheduler event history',
      } ),
   }
   privileged = True
   cliModel = ShowTaskSchedulerModel.Overall

   @staticmethod
   def handler( mode, args ):
      # Translate options to those used in AgentCli
      translation = { 'verbose': 'detail',
                      'internal': 'debug',
                      'history': 'history' }
      agentArgs = { 'AGENT': 'KernelFib',
                    'SCHEDULER_OPTS': [ translation[ i ] for i in translation
                                        if i in args.get( 'OPTIONS', [] ) ],
      }
      return AgentLibShowCommands.showTaskSchedulerHandler( mode, agentArgs )

BasicCli.addShowCommandClass( KernelfibSchedulerCmd )

def kernelUnprogrammedRoute( vrfName=DEFAULT_VRF ):
   if vrfName == DEFAULT_VRF:
      return SmashLazyMount.mount( myEntManager,
                              'routing/kernel/unprogrammedRoute',
                              'Smash::KernelUnprogrammedRoute::UnprogrammedRoute',
                              SmashLazyMount.mountInfo( 'reader' ) )
   else:
      return SmashLazyMount.mount( myEntManager,
            'routing/vrf/kernel/unprogrammedRoute',
            'Smash::KernelUnprogrammedRoute::UnprogrammedRoute',
            SmashLazyMount.mountInfo( 'reader' ) ) 

def Plugin( entityManager ):
   global myEntManager
   global routingHardwareStatus
   global kernelfibConfig, kernelfibConfig6
   global vrfNameStatus
   global unprogrammedRoutePrinter

   myEntManager = entityManager
   routingHardwareStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
                                            'Routing::Hardware::Status', 'r' )

   kernelfibConfig = ConfigMount.mount( entityManager, 'routing/kernel/config',
                                        'KernelFib::KernelRouteConfigDir', 'w' )
   kernelfibConfig6 = ConfigMount.mount( entityManager, 'routing6/kernel/config',
                                         'KernelFib::KernelRouteConfigDir', 'w' )

   vrfNameStatus = LazyMount.mount( entityManager, 
                                       Cell.path( 'vrf/vrfNameStatus' ), 
                                       'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

   unprogrammedRoutePrinter = Tac.newInstance( 
         'KernelFib::UnprogrammedRoutePrinter',
         kernelUnprogrammedRoute( DEFAULT_VRF ),
         kernelUnprogrammedRoute( 'all' ) )


