#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
from ArnetModel import IpGenericAddress
from CliModel import Bool
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
import datetime
import TableOutput

fl = TableOutput.Format( justify='left' )
fl.padLimitIs( True )

class ItsSession( Model ):
   class SessionCounters( Model ):
      inBytes = Int( help="Number of bytes received from the telemetry session" )
      inPkts = Int( help="Number of packets received from the telemetry session" )
      inPktsDrop = Int( help="Number of packets dropped after being received "
                             "from the telemetry session" )
      outBytes = Int( help="Number of bytes sent out of the telemetry session" )
      outPkts = Int( help="Number of packets sent out of the telemetry session" )
      outPktsDrop = Int( help="Number of packets dropped before being sent "
                              "out of the telemetry session" )

   class SessionCharacts( Model ):
      active = Bool( help="The telemetry session state is active" )
      seconds = Int( help="Number of seconds the telemetry"
                        "session has been in this state" )
      latency = Float( help="One way delay in milliseconds of the "
                       "telemetry session" )
      jitter = Float( help="Jitter in milliseconds of the telemetry session" )
      throughput = Float( help="Throughput in Mbps of the telemetry session" )
      lossRate = Float( help="Loss rate in percentage of the telemetry session" )

   counters = Submodel( valueType=SessionCounters,
                        help="Counters for the telemetry session",
                        optional=True )
   characteristics = Submodel( valueType=SessionCharacts,
                               help="Characteristics of the telemetry session",
                               optional=True )

class ItsPath( Model ):
   srcIp = IpGenericAddress( help='Source IP address of the path' )
   dstIp = IpGenericAddress( help='Destination IP address of the path' )
   # key is tc (0-7)
   sessions = Dict( keyType=int,
                    help="A mapping of traffic class to telemetry session",
                    valueType=ItsSession )

class ItsGroup( Model ):
   paths = Dict( keyType=str,
                 help="A mapping of path name to path state",
                 valueType=ItsPath )

class ItsPeer( Model ):
   peerName = Str( help='Peer name' )
   # key is path name, exp "path12"
   groups = Dict( keyType=str,
                  help="A mapping of path group name to its paths",
                  valueType=ItsGroup )

class ItsPaths( Model ):
   # key is peer ip
   peers = Dict( keyType=str,
                 help="A mapping of peer IP address to peer path groups",
                 valueType=ItsPeer )
   _charact = Bool( help='Show path characteristics or counters' )
   _detail = Bool( help='Show detail information' )

   def render( self ):
      table = None
      if not self._detail:
         if self._charact:
            header = [ "Peer", "Path Group", "Path Name", "TC",
                       "Tx State", "Latency(ms)", "Jitter(ms)",
                       "Throughput(Mbps)", "Loss Rate(%)" ]
         else:
            header = [ "Peer", "Path Group", "Path Name", "TC",
                       "In Bytes", "In Packets", "In Drops",
                       "Out Bytes", "Out Packets", "Out Drops" ]
         table = TableOutput.createTable( header, tableWidth=160 )
         table.formatColumns( *( fl for _ in header ) )

      for peerIp in sorted( self.peers,
                            key=lambda peerIp: IpGenAddr( peerIp ).sortKey ):
         peer = self.peers[ peerIp ]
         if self._detail:
            print 'Peer: %s, Name: %s' % ( peerIp, peer.peerName )
         for groupName, group in sorted( peer.groups.items() ):
            if self._detail:
               print '  Path group: %s' % groupName
            for pathName, path in sorted( group.paths.items() ):
               if self._detail:
                  print '    Path name: %s' % pathName
                  print '    Source: %s, Destination: %s' % \
                     ( path.srcIp, path.dstIp )
               for tc, sess in sorted( path.sessions.items() ):
                  if self._detail:
                     print '      Traffic class: %u' % tc
                  if self._charact:
                     printSessCharacteristics( peerIp, peer, groupName, pathName,
                                               tc, sess, table )
                  else:
                     printSessCounters( peerIp, peer, groupName, pathName,
                                        tc, sess, table )
      if table:
         print table.output()

def printSessCharacteristics( peerIp, peer, groupName, pathName,
                              tc, sess, table ):
   if table:
      # show ITS session characteristics
      peerName = peerIp + ( ' (' + peer.peerName + ')' \
               if ( peer.peerName and peer.peerName != '' ) else '' )
      sessState = 'inactive'
      if sess.characteristics.active:
         sessState = 'active (' + str( datetime.timedelta( \
                        seconds=sess.characteristics.seconds ) ) + ')'
      table.newRow( peerName, groupName, pathName, tc,
                    sessState,
                    sess.characteristics.latency,
                    sess.characteristics.jitter,
                    sess.characteristics.throughput,
                    sess.characteristics.lossRate )
   else:
      # show ITS session characteristics detail
      rate = '        {0:s}: {1:.2f}%'.format( 'Loss rate',
                                               sess.characteristics.lossRate )
      print '        Tx state: %s' % 'active' if sess.characteristics.active \
                                              else 'inactive'
      print '        Active time: %s' % str( datetime.timedelta( \
                                  seconds=sess.characteristics.seconds ) )
      print '        Latency: %.3f ms' % \
         sess.characteristics.latency
      print '        Jitter: %.3f ms' % sess.characteristics.jitter
      print '        Throughput: %.2f Mbps' % \
         sess.characteristics.throughput
      print rate

def printSessCounters( peerIp, peer, groupName, pathName,
                       tc, sess, table ):
   if table:
      # show ITS session counters
      peerName = peerIp + ( ' (' + peer.peerName + ')' \
                           if ( peer.peerName and peer.peerName != '' ) else '' )
      table.newRow( peerName, groupName, pathName, tc,
                    sess.counters.inBytes, sess.counters.inPkts,
                    sess.counters.inPktsDrop,
                    sess.counters.outBytes,
                    sess.counters.outPkts,
                    sess.counters.outPktsDrop )
   else:
      # show ITS session counters detail
      print '        In bytes: %u' % sess.counters.inBytes
      print '        In packets: %u' % sess.counters.inPkts
      print '        In drops: %u' % sess.counters.inPktsDrop
      print '        Out bytes: %u' % sess.counters.outBytes
      print '        Out packets: %u' % sess.counters.outPkts
      print '        Out drops: %u' % sess.counters.outPktsDrop

