#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from datetime import datetime
from collections import OrderedDict
from operator import itemgetter, attrgetter
from time import strftime, localtime
import struct

from CliModel import Int, Bool, Str, Enum, List, Model, Submodel
from CliModel import Float, GeneratorList, GeneratorDict, Dict

from CliPlugin.VrfCli import generateVrfCliModel
import Tac
from ArnetModel import Ip4Address, Ip6Address, IpGenericAddress
from IntfModel import Interface
from DeviceNameLib import kernelIntfFilter
from TunnelModels import ( MplsVia,
                           tunnelTypeEnumValues,
                           TunnelInfo )
import TableOutput
import PyRibAmiClient
import IsisCliHelper
import IsisCliModelCommon

isTypeHelp = '''Type of IS( Intermediate System ) 
unknown : unknown  IS Type
level1 : IS type is level-1
level2 : IS type is level-2
level1-2 : IS type is level-1-2
   '''
addressFamilyHelp = '''Address Family
none : Address Family disabled
ipv4 : Routes ipv4 only 
ipv6 : Routes ipv6 only
both : Routes both ipv4 and ipv6
   '''
authModeHelp = '''
none : No Authentication
md5 : Authentication mode is MD5
text : Authentication mode is TEXT
    '''
authModeHelpL1 = '''Authentication mode Level1''' + authModeHelp

authModeHelpL2 = '''Authentication mode Level2''' + authModeHelp

bfdStateHelp = '''ISIS neighbor BFD state.
  adminDown - Session taken down for administrative purposes 
  init - Waiting for session establishment
  up - Session established successfully
  down - Session is down or has just been created'''

ngbHeaderFormatStr = "%-9s %-8s %-16s %-4s %-18s %-17s %-5s %-11s %-20s"
ngbEntryFormatStr = "%-9.9s %-8.8s %-16s %-4s %-18s %-17s %-5s %-11s %-20s"
spfLogHeaderFormatStr = "    %-30s %-10s %-7s %-7s %-17s"
spfLogMtHeaderFormatStr = "    %-30s %-10s %-9s %-9s %-7s %-17s"
spfLogEntryFormatStr = "   %s%-30s %-14.3f %-7d %-7d %-17s"
spfLogMtEntryFormatStr = "   %s%-30s %-14.3f %-9d %-9d %-7d %-17s"
grNgbHeaderFormatStr = "  %-18s %-7s %-18s %-17s %-10s"

ISIS_GR_RCVD_RR = 0x1
ISIS_GR_RCVD_SA = 0x2
ISIS_GR_SENT_RA = 0x4
ISIS_GR_SENT_CSNP = 0x8

LINE_MAX = 78

class IsisInstanceSummarySpfModel( Model ):
   spfMaxWaitInterval = Int( help='Spf Interval:Max wait(s)' )
   spfInitialWaitInterval = Int( help='Spf Interval:Initial wait(ms)' )
   spfHoldInterval = Int( help='Spf Interval:Hold Interval(ms)' )
   spfCurrentHoldIntervalL1 = Int( help='Current SPF hold interval(ms) for Level1', 
                                   optional=True )
   spfCurrentHoldIntervalL2 = Int( help='Current SPF hold interval(ms) for Level2', 
                                   optional=True )
   spfLastRunL1 = Float( help='Last Level 1 SPF ran time', 
                         optional=True )
   spfLastRunL2 = Float( help='Last Level 2 SPF ran time', 
                         optional=True )
   
class IsisInstanceSummaryLspGenInterval( Model ):
   lspGenMaxWaitInterval = Int( help='Lsp Generation Interval:Max wait(s)' )
   lspGenInitialWaitInterval = Int( help='Lsp Generation Interval:Initial wait(ms)' )
   lspGenHoldInterval = Int( help='Lsp Generation Interval:Hold Interval(s)' )

class IsisInstanceSummaryPreferenceModel( Model ):
   internalV4PreferenceL1 = Int( help='internal preference of level1', 
                               optional=True )
   internalV4PreferenceL2 = Int( help='internal preference of level2', 
                               optional=True )

   internalV6PreferenceL1 = Int( help='internal preference of level1', 
                               optional=True )
   internalV6PreferenceL2 = Int( help='internal preference of level2', 
                               optional=True )

class IsisInstanceSummaryModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   instanceId = Int( help='Instance ID' )
   hostName = Str( help="Hostname", optional=True )
   systemId = Str( help='System Id' )
   enabled = Bool( default=False, help='Instance Enabled or not' )
   attached = Bool( default=False, help='IPv4 Attached or not' )
   multiTopologyEnabled = Bool( default=False, 
                                help='multiTopology Enabled or not' )
   multiTopologyAttached = Bool( default=False, help='IPv6 Attached or not' )
   isType = Enum( values=( 'unknown', 'level1', 'level2', 'level1-2' ), 
                  default='unknown', help=isTypeHelp ) 
   numActiveInterfaces = Int( help='Number of active interfaces' )
   addressFamily = Enum( values=( 'none', 'ipv4', 'ipv6', 'both' ),  
                         default='none', help=addressFamilyHelp )
   authModeL1 = Enum( values=( 'none', 'md5', 'text', 'sha' ),
                      optional=True, help=authModeHelpL1 )
   authModeKeyIdL1 = Int( help='Authentication mode key id in Level1',
                          optional=True )
   authModeL2 = Enum( values=( 'none', 'md5', 'text', 'sha' ),
                      optional=True, help=authModeHelpL2 )
   authModeKeyIdL2 = Int( help='Authentication mode key id in Level2',
                          optional=True )
   authRxDisabledL1 = Bool( help='Received LSP authentication check is disabled for '
                            'level-1', optional=True )
   authRxDisabledL2 = Bool( help='Received LSP authentication check is disabled for '
                            'level-2', optional=True )
   preferenceInfo = Submodel( valueType=IsisInstanceSummaryPreferenceModel, 
                              help='Preference Information' )
   spfInfo = Submodel( valueType=IsisInstanceSummarySpfModel, 
                       help='Spf Information' )
   shortcutSpfForLdpTunneling = Bool( help='Shortcut SPF enabled for LDP tunneling',
                                      optional=True )
   shortcutSpfForIgp = Bool( help='Shortcut SPF enabled for IGP', optional=True )
   lspGenInfo = Submodel( valueType=IsisInstanceSummaryLspGenInterval,
                          help="Lsp Gen Information" )
   areaIds = List( valueType=str, help='List of area-ids in this instance' )
   numDisInterfacesL1 = Int( help='Number of DIS interfaces for leve1', 
                             optional=True )
   lsdbSizeL1 = Int( help='LSDB size (number of link state database records)' +  
                     'for level1', optional=True )
   numDisInterfacesL2 = Int( help='Number of DIS interfaces for level2',
                             optional=True )
   lsdbSizeL2 = Int( help='LSDB size (number of link state database records)' +  
                     'for level2', optional=True )
   gracefulRestart = Bool( default=False, help='GR Enabled or not' )
   gracefulRestartHelper = Bool( default=False, help='GR helper Enabled or not' )
   routerIdV4 = Ip4Address( help='ISIS router ID in IP format', optional=True )
   areaLeaderL1 = Str( help='Area leader for Level 1', optional=True )
   areaLeaderL2 = Str( help='Area leader for Level 2', optional=True )
   dynamicFloodingL1 = Bool( default=False,
                             help='Dynamic Flooding is enabled for Level 1' )
   dynamicFloodingL2 = Bool( default=False,
                             help='Dynamic Flooding is enabled for Level 2' )
   lspSizeMaxL1 = Int( help='LSP size maximum for Level 1' )
   lspSizeMaxL2 = Int( help='LSP size maximum for Level 2' )
   
   def processData( self, data ):
      from RoutingIsisCli import getHostName
      # importing in function as to avoid circular dependency issue
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
      self.instanceId = data.pop( 'instance-id' )
      self.systemId = data.pop( 'system-id' )
      self.hostName = getHostName( getRouterId( self.systemId ),
                                                     vrfName=self._vrf )
      if 'router-id-v4' in data:
         self.routerIdV4 = data.pop( 'router-id-v4' )
      else:
         self.routerIdV4 = '0.0.0.0'
      self.attached = ord( data.pop( 'attached' ) ) != 0 
      self.enabled = ord( data.pop( 'enabled' ) ) != 0
      self.multiTopologyEnabled = ord( data.pop( 'mt' ) ) != 0
      self.multiTopologyAttached = ord( data.pop( 'mt-id2-attached' ) ) != 0
      if data[ 'level' ] in IsisCliHelper.LEVEL_MAP:
         self.isType = IsisCliHelper.LEVEL_MAP[ data[ 'level' ] ]
      del data['level']
      if data[ 'addr-family' ] in IsisCliHelper.ADDRESS_FAMILY_MAP:
         self.addressFamily = IsisCliHelper.ADDRESS_FAMILY_MAP[ \
               data[ 'addr-family' ] ]
      del data[ 'addr-family' ]
      self.preferenceInfo = IsisInstanceSummaryPreferenceModel()
      self.numActiveInterfaces = data.pop( 'num-circuits' )
      self.spfInfo = IsisInstanceSummarySpfModel()
      self.spfInfo.spfHoldInterval = data.pop( 'spf-hold-int' )
      self.spfInfo.spfMaxWaitInterval = data.pop( 'spf-max-int' )
      self.spfInfo.spfInitialWaitInterval = data.pop( 'spf-start-int' )

      # These are guarded by toggles (gated IsisIgpShortcut and RsvpLib LdpOverRsvp),
      # but to avoid depending on both here, instead the attribute optional and only
      # set when gated sends true. This is sane because gated will only send true
      # when the appropriate toggles are on. When the toggles are removed, this can
      # change to be non-optional and send it no matter what. This should be a
      # "compatible" change, thus not require a CAPI version bump.
      if ord( data.pop( 'ldp-shortcut-spf' ) ) != 0:
         self.shortcutSpfForLdpTunneling = True
      if ord( data.pop( 'igp-shortcut-spf' ) ) != 0:
         self.shortcutSpfForIgp = True

      self.lspGenInfo = IsisInstanceSummaryLspGenInterval()
      self.lspGenInfo.lspGenMaxWaitInterval = data.pop( 'lsp-gen-max-int' )
      self.lspGenInfo.lspGenInitialWaitInterval = data.pop( 'lsp-gen-start-int' )
      self.lspGenInfo.lspGenHoldInterval = data.pop( 'lsp-gen-hold-int' )
      if 'areaId1' in data:
         self.areaIds.append( data.pop( 'areaId1' ) )
      if 'lsdb-size-l1' in data:
         self.lsdbSizeL1 = data.pop( 'lsdb-size-l1' )
         self.numDisInterfacesL1 = data.pop( 'ndis-l1' )
      if 'lsdb-size-l2' in data:
         self.lsdbSizeL2 = data.pop( 'lsdb-size-l2' )
         self.numDisInterfacesL2 = data.pop( 'ndis-l2' )
      if 'L1 authentication mode' in data:
         mode = data.pop( 'L1 authentication mode' )
         if mode == 1:
            self.authModeL1 = "md5" 
         if mode == 2:
            self.authModeL1 = "text"
         if mode == 3:
            self.authModeL1 = "sha"
            self.authModeKeyIdL1 = data.pop( 'L1-authentication-mode-id' )
      else:
         self.authModeL1 = 'none'
      self.spfInfo.spfCurrentHoldIntervalL1 = \
               data.pop( 'spf-l1-cur-int' )

      if self.isType == 'level1' or self.isType == 'level1-2':
         if 'l1-spf-lastran' in data:
            self.spfInfo.spfLastRunL1 = ( Tac.utcNow() ) - \
            IsisCliHelper.ribdToSecTime( data.pop( 'l1-spf-lastran' ) ) 
      self.preferenceInfo.internalV4PreferenceL1 = \
                  data.pop( 'l1-int-pref-v4' ) 
      self.preferenceInfo.internalV6PreferenceL1 = \
                  data.pop( 'l1-int-pref-v6' ) 

      if 'L2 authentication mode' in data:      
         mode = data.pop( 'L2 authentication mode' )
         if mode == 1:
            self.authModeL2 = "md5" 
         if mode == 2:
            self.authModeL2 = "text"
         if mode == 3:
            self.authModeL2 = "sha"
            self.authModeKeyIdL2 = data.pop( 'L2-authentication-mode-id' )
      else:
         self.authModeL2 = 'none'
      if 'auth-rx-disabled-l1' in data:
         self.authRxDisabledL1 = ord( data[ 'auth-rx-disabled-l1' ] ) != 0
      if 'auth-rx-disabled-l2' in data:
         self.authRxDisabledL2 = ord( data[ 'auth-rx-disabled-l2' ] ) != 0

      self.spfInfo.spfCurrentHoldIntervalL2 = \
                              data.pop( 'spf-l2-cur-int' )
      if self.isType == 'level2' or self.isType == 'level1-2':
         if 'l2-spf-lastran' in data:
            self.spfInfo.spfLastRunL2 = ( Tac.utcNow() ) - \
            IsisCliHelper.ribdToSecTime( data.pop( 'l2-spf-lastran' ) )
            
      self.preferenceInfo.internalV4PreferenceL2 = \
                  data.pop( 'l2-int-pref-v4' ) 
      self.preferenceInfo.internalV6PreferenceL2 = \
                  data.pop( 'l2-int-pref-v6' ) 
      self.gracefulRestart = ord( data.pop( 'graceful restart' ) ) != 0
      self.gracefulRestartHelper = ord( data.pop( 'graceful restart helper' ) ) != 0
      self.areaLeaderL1 = data.pop( 'area-leader-l1', "None" )
      self.areaLeaderL2 = data.pop( 'area-leader-l2', "None" )
      self.dynamicFloodingL1 = ord( data.pop( 'dynamic-flooding-l1', '0' ) ) != 0
      self.dynamicFloodingL2 = ord( data.pop( 'dynamic-flooding-l2', '0' ) ) != 0
      self.lspSizeMaxL1 = data.pop( 'lsp-size-max-l1' )
      self.lspSizeMaxL2 = data.pop( 'lsp-size-max-l2' )
      return data

   def render( self ):
      if self._instanceName is None:
         return
      print  " "
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      print "  Instance ID: %d" % ( self.instanceId )
      print "  System ID: %s, administratively %s" % ( self.systemId,  
            "enabled" if self.enabled else "disabled" )
      print "  Router ID: IPv4: %s" % self.routerIdV4
      if self.hostName:
         print "  Hostname: %s" % ( self.hostName )
      if self.multiTopologyEnabled:
         mt_str1 = "enabled"
         mt_str2 = "IPv4 " if self.attached else "IPv4 not "
         mt_str3 = ", IPv6 attached" if self.multiTopologyAttached \
                   else ", IPv6 not attached"
      else:
         mt_str1 = "disabled"
         mt_str2 = "" if self.attached else "not "
         mt_str3 = ""
      print "  Multi Topology %s, %sattached%s" % ( mt_str1, mt_str2, mt_str3 )
      if self.preferenceInfo:
         prefStr = "  IPv4 Preference:"
         if self.preferenceInfo.internalV4PreferenceL1:
            prefStr += " Level 1: %3d" % \
                       ( self.preferenceInfo.internalV4PreferenceL1 )
            if self.preferenceInfo.internalV4PreferenceL2:
               prefStr += ","
         if self.preferenceInfo.internalV4PreferenceL2:
            prefStr += " Level 2: %3d" % \
                       ( self.preferenceInfo.internalV4PreferenceL2 )
         print prefStr
         prefStr = "  IPv6 Preference:"
         if self.preferenceInfo.internalV6PreferenceL1:
            prefStr += " Level 1: %3d" % \
                       ( self.preferenceInfo.internalV6PreferenceL1 )
            if self.preferenceInfo.internalV6PreferenceL2:
               prefStr += ","
         if self.preferenceInfo.internalV6PreferenceL2:
            prefStr += " Level 2: %3d" % \
                       ( self.preferenceInfo.internalV6PreferenceL2 )
         print prefStr
      levelStr = IsisCliHelper.IS_TYPE_MAP[ self.isType ]
      print "  IS-Type: %s, Number active interfaces: %d" % ( levelStr,  
            self.numActiveInterfaces )
      routeStr = IsisCliHelper.ROUTE_MAP[ self.addressFamily ]
      print routeStr

      lspSizeStr = "  LSP size maximum:"
      lspSizeStr += " Level 1: %4d," % ( self.lspSizeMaxL1 )
      lspSizeStr += " Level 2: %4d" % ( self.lspSizeMaxL2 )
      print lspSizeStr
      
      if self.spfInfo:
         print " " * 27 + " Max wait(s) Initial wait(ms) Hold interval(ms)"
         print "  LSP Generation Interval:%6d %15d %16d" % \
         ( self.lspGenInfo.lspGenMaxWaitInterval,  
           self.lspGenInfo.lspGenInitialWaitInterval,
           self.lspGenInfo.lspGenHoldInterval )
         
         print "  SPF Interval:           %6d %15d %16d" % \
            ( self.spfInfo.spfMaxWaitInterval,  
              self.spfInfo.spfInitialWaitInterval, self.spfInfo.spfHoldInterval )

         currStr = "  Current SPF hold interval(ms):"
         currStr += " Level 1: %d" % ( self.spfInfo.spfCurrentHoldIntervalL1 )
         currStr += ","
         currStr += " Level 2: %d" % ( self.spfInfo.spfCurrentHoldIntervalL2 )
         print currStr
         if self.spfInfo.spfLastRunL1 or self.spfInfo.spfLastRunL2:
            spfLastRun = []
            if self.spfInfo.spfLastRunL1:
               spfLastRunL1TimeSec = int ( Tac.utcNow() -
                                           ( self.spfInfo.spfLastRunL1 ) )
               spfLastRunL1Time = IsisCliHelper.secToRibdTime( spfLastRunL1TimeSec )
               spfLastRunL1Time = spfLastRunL1Time if ( spfLastRunL1Time ) else "0"
               spfLastRun.append( ( 1, spfLastRunL1Time ) )

            if self.spfInfo.spfLastRunL2:
               spfLastRunL2TimeSec = int( Tac.utcNow() -
                                          ( self.spfInfo.spfLastRunL2 ) )
               spfLastRunL2Time = IsisCliHelper.secToRibdTime( spfLastRunL2TimeSec )
               spfLastRunL2Time = spfLastRunL2Time if ( spfLastRunL2Time ) else "0"
               spfLastRun.append( ( 2, spfLastRunL2Time ) )
            for ( level, val ) in spfLastRun:
               if len( val ) > 5:
                  suffix = "hours ago"
               elif len( val ) > 2:
                  suffix = "minutes ago"
               else: 
                  suffix = "seconds ago"
               
               print "  Last Level %s SPF run %s %s" % ( level, val, suffix )
      if self.dynamicFloodingL1 and self.dynamicFloodingL2:
         currStr = "Level 1 and 2"
      elif self.dynamicFloodingL1:
         currStr = "Level 1"
      elif self.dynamicFloodingL2:
         currStr = "Level 2"
      else:
         currStr = "Disabled"
      print "  Dynamic Flooding: %s" % currStr

      if self.shortcutSpfForLdpTunneling:
         print( "  Shortcut SPF for LDP tunneling: Enabled" )
      if self.shortcutSpfForIgp:
         print( "  Shortcut SPF for IGP: Enabled" )

      authStr = "  Authentication mode:"
      if self.authModeL1:
         authStr += " Level 1: %s" % ( IsisCliHelper.AUTHENTICATION_MAP[ \
               self.authModeL1 ] )
         if self.authModeL1 == "sha":
            authStr += " Key id: %d" % self.authModeKeyIdL1
         if self.authModeL2:
            authStr += ","
      if self.authModeL2:
         authStr += " Level 2: %s" % ( IsisCliHelper.AUTHENTICATION_MAP[ \
               self.authModeL2 ] )
         if self.authModeL2 == "sha":
            authStr += " Key id: %d" % self.authModeKeyIdL2
      print authStr

      rxDisabledLevels = []
      if self.authRxDisabledL1:
         rxDisabledLevels.append( 'Level 1: disabled' )
      if self.authRxDisabledL2:
         rxDisabledLevels.append( 'Level 2: disabled' )
      if rxDisabledLevels:
         print "  Received LSP authentication check:", ", ".join( rxDisabledLevels )
         
      if self.gracefulRestart == True:
         grStr = "  Graceful Restart: Enabled,"
      else:
         grStr = "  Graceful Restart: Disabled,"
      if self.gracefulRestartHelper == True:
         grStr += " Graceful Restart Helper: Enabled"
      else:
         grStr += " Graceful Restart Helper: Disabled"
      print grStr   
       
      print "  Area Addresses:"
      for area in self.areaIds:
         print "    %s" % area
      if self.numDisInterfacesL1 is not None and self.lsdbSizeL1 is not None:
         print "  level 1: number DIS interfaces: %d, LSDB size: %d" % (
               self.numDisInterfacesL1, self.lsdbSizeL1 )
         print "    Area Leader:", self.areaLeaderL1
      if self.numDisInterfacesL2 is not None and self.lsdbSizeL2 is not None:
         print "  level 2: number DIS interfaces: %d, LSDB size: %d" % ( 
               self.numDisInterfacesL2, self.lsdbSizeL2 )
         print "    Area Leader:", self.areaLeaderL2
      return

isisSummaryModel = IsisCliModelCommon. \
                   generateIsisDictCliModel( IsisInstanceSummaryModel )
isisSummaryVRFsModel = generateVrfCliModel( isisSummaryModel,
                                    "ISIS instance information for all VRFs" )

LSP_GEN_EVENT_MAP = [
   "backoff started, hold value", 
   "backoff continue, remaining wait time", 
   "backoff restarted, hold value", 
   "out-delay applied, delayed by"
]

class IsisLspLogEntryModel( Model ):
   logTime = Int( help='Log time stamp' )
   logId = Int( help='Log entry identifier' )
   lspLevel = Int( help='Level of the LSP' )
   event = Enum( values=LSP_GEN_EVENT_MAP,
                 help='Event associated with Log' )
   holdInterval = Int( help='Hold interval for the LSP in ms' )
   lspId = Str( help='IS-IS LSPID' )
   
   def render( self ):
      print ( "%s: %d lsp %s level-%d %s %d msec" ) % (
         strftime( "%Y-%m-%d %H:%M:%S", localtime( self.logTime ) ), 
         self.logId,
         self.lspId, self.lspLevel,
         self.event, self.holdInterval )

   def processData( self, data ):
      self.logTime = data.pop( 'time' )
      self.logId = data.pop( 'id' )
      self.lspLevel = ord( data.pop( 'level' ) )
      self.event = LSP_GEN_EVENT_MAP[ ord( data.pop( 'event' ) ) ]
      self.holdInterval = data.pop( 'hold-interval' )
      self.lspId = data.pop( 'lspid' )

class IsisLspLogTableModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   lspLogs = GeneratorList( valueType=IsisLspLogEntryModel,
                              help='List of lsplog Entries' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
   
   def render( self ):
      if self._instanceName is None:
         return
      print  ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      print ""
      for entry in self.lspLogs:
         entry.render()

lspLogTableModel = IsisCliModelCommon. \
                   generateIsisDictCliModel( IsisLspLogTableModel )
lspLogTableVRFsModel = generateVrfCliModel( lspLogTableModel,
                                    "IS-IS instance information for all VRFs" )

class LspFlags( Model ):
   partitionSupport = Bool( default=False,
                            help='Support to Partition Repair' )
   errorAttached = Bool( default=False,
                          help='Attached to other areas by the Error Metric' )
   expenseAttached = Bool( default=False,
                         help='Attached to other areas by the Expense Metric' )
   delayAttached = Bool( default=False,
                          help='Attached to other areas by the Delay Metric' )
   defaultAttached = Bool( default=False,
                            help='Attached to other areas by the Default Metric' )
   dbOverload = Bool( default=False,
                      help='LSP Database Overload' )

def getFlagsString( flags ):
   flagStr = ''
   for flag in IsisCliHelper.ISIS_LSP_FLAGS_MAP:
      if getattr( flags, flag ):
         flagStr += IsisCliHelper.ISIS_LSP_FLAGS_MAP[ flag ] + ' '
   return flagStr.strip()

class LspMtModel( Model ):
   topologyId = Int( help="Topology Id" )
   mtOverloaded = Bool( default=False, help="MultiTopology Overload Flag" )
   mtAttached = Bool( default=False, help="MultiTopology Attached Flag" )

   def processData( self, data ):
      self.topologyId = data.pop( 'topo_id', None ) 
      self.mtOverloaded = ord ( data.pop( 'mt_overload' ) ) != 0
      self.mtAttached = ord ( data.pop( 'mt_attach' ) ) != 0
      return data
   
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_MTID
   
   def render( self ):
      overloadString = ", overloaded" if self.mtOverloaded else ""
      attachString = ", attached" if self.mtAttached else ""
      if self.topologyId == 0:
         print "      Topology: %s (IPv4)%s%s" % ( self.topologyId,
                                                   overloadString,
                                                   attachString )
      elif self.topologyId == 2:
         print "      Topology: %s (IPv6)%s%s" % ( self.topologyId,
                                                   overloadString,
                                                   attachString )  
      else:
         print "      Topology: %s (Unsupported)%s%s" % ( self.topologyId,
                                                           overloadString,
                                                           attachString )
         
class LspAuthModel( Model ):
   authMode = Enum( values=IsisCliHelper.AUTH_MODE_MAP.values(),
                    help="Authentication Mode" )
   authLength = Int( help="Authentication Key Length" )
   authKeyId = Int( help="Authentication Key Id", optional=True )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AUTH

   def processData( self, data ):
      self.authMode = IsisCliHelper.AUTH_MODE_MAP[ data.pop( 'Authentication mode' \
            , None ) ]
      self.authLength = data.pop( 'Authentication length', None )
      if self.authMode == "SHA":
         self.authKeyId = data.pop( 'Authentication-key-id' )
      return data
   
   def render( self ):
      if self.authMode is None or self.authLength is None :
         return
      else:
         keyIdStr = ""
         if self.authKeyId is not None:
            keyIdStr = " Key id: %d" % self.authKeyId
         print "      Authentication mode: %s%s Length: %s" % ( self.authMode,
                                                                keyIdStr,
                                                                self.authLength )
class LspAreaAddress( Model ):
   address = Str( help="Area Address for lsp" )
   
   def processData( self, data ):
      self.address = data.pop( 'area-id' )
      return data
   
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA
   
   def render( self ):
      print "      Area address: %s" % self.address
      
class LspHostname( Model ):
   name = Str( help="LSP Hostname" )
   
   def processData( self, data ):
      self.name = data.pop( 'hostname' )
      return data
   
   def render( self ):
      print "      Hostname: %s" % ( self.name )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_LSDB_HOSTNAME
   
class InterfaceAddresses( Model ):
   ipv4Address = Ip4Address( help="IPv4 address", optional=True )
   ipv6Address = Ip6Address( help="IPv6 address", optional=True )

   def processData( self, data ):
      if 'ipv4-ifaddr' in data:
         self.ipv4Address = data.pop( 'ipv4-ifaddr' )
      if 'ipv6-ifaddr' in data:   
         self.ipv6Address = data.pop( 'ipv6-ifaddr' )
   def render( self ):
      outputStr = "      Interface address: %s"
      address = self.ipv4Address if self.ipv4Address else self.ipv6Address
      print outputStr % address
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ADDRESS

class LspAdjInterfaceAddress( Model ):
   adjInterfaceAddress = Ip4Address( help="Interface Address" )
   
   def processData( self, data ):
      self.adjInterfaceAddress = data.pop( 'intf-addr' )
      
   def render( self ):
      print "        IPv4 Interface Address: %s" % ( self.adjInterfaceAddress )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ADJ_INTF_ADDR

def setAdjFlagsRenderStr( flagSubmodel, flag ):
   flagSubmodel.adjSet = True if ( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_SET' ] ) else False
   flagSubmodel.adjLocal = True if ( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_LOCAL' ] ) else False
   flagSubmodel.adjValue = True if ( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_VALUE' ] ) else False
   flagSubmodel.adjBackup = True if ( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_BACKUP' ] ) else False
   flagSubmodel.adjAf = True if ( flag &
         IsisCliHelper.ADJ_SID_SUBTLV_FLAGS[ 'ISIS_SR_ADJ_AF' ] ) else False

def getAdjFlagsRenderList( flags ):
   adjSet = "S " if ( flags.adjSet )  else ""
   adjLocal = "L " if ( flags.adjLocal ) else ""
   adjValue = "V " if ( flags.adjValue ) else ""
   adjBackup = "B " if ( flags.adjBackup ) else ""
   adjAf = "F " if ( flags.adjAf ) else ""
   return [ adjSet, adjLocal, adjValue, adjBackup,
            adjAf ]

class AdjFlags( Model ):
   adjSet = Bool( help="Sr Adj Set Flag" )
   adjLocal = Bool( help="Sr Adj Local Flag" )
   adjValue = Bool( help="Sr Adj Value Flag" )
   adjBackup = Bool( help="Sr Adj Backup Flag" )
   adjAf = Bool( help="Sr Adj AddressFamily Flag" )
   
class LspAdjLanSid( Model ):
   adjLanSid = Int( help="Adjacency LAN SID" )
   adjLanSystemId = Str( help="Adjacency SystemId" )
   adjLanFlags = Submodel( valueType=AdjFlags, help="Adjacency LAN Flags" )
   adjLanWeight = Int( help="Adjacency LAN Weight" )
   
   def processData( self, data ):
      self.adjLanSid = data.pop( 'adj-lan-sid' )   
      self.adjLanSystemId = data.pop( 'adj-lan-system-id' )
      flags = ord( data.pop( 'adj-lan-flags' ) )
      self.adjLanFlags = AdjFlags()
      setAdjFlagsRenderStr( self.adjLanFlags, flags )
      self.adjLanWeight = ord( data.pop( 'adj-lan-weight' ) )
      
   def render( self ):
      adjFlags = getAdjFlagsRenderList( self.adjLanFlags )
      outFmt = "        LAN-Adj-sid: %s flags: [ %s%s%s%s%s] weight: "
      outFmt += "%u system ID: %s"
      print outFmt % ( self.adjLanSid, adjFlags[ 0 ], adjFlags[ 1 ], adjFlags[ 2 ],
                       adjFlags[ 3 ], adjFlags[ 4 ], self.adjLanWeight,
                       self.adjLanSystemId )

   def getMioAttrId( self ):
      return  PyRibAmiClient.MIO_DGET_ISIS_ADJ_LAN_SID

class LspAdjSid( Model ):
   adjSid = Int( help="Adjacency SID" )
   adjFlags = Submodel( valueType=AdjFlags, help="Adjacency Flags" )
   adjWeight = Int( help="Adjacency Weight" )
   
   def processData( self, data ):
      self.adjSid = data.pop( 'adj-sid' )
      flags = ord( data.pop( 'adj-flags' ) )
      self.adjFlags = AdjFlags()
      setAdjFlagsRenderStr( self.adjFlags, flags )
      self.adjWeight = ord( data.pop( 'adj-weight' ) )
      
   def render( self ):
      adjFlags = getAdjFlagsRenderList( self.adjFlags )
      print "        Adj-sid: %s flags: [ %s%s%s%s%s] weight: 0x%x" % (
         self.adjSid, adjFlags[ 0 ], adjFlags[ 1 ], adjFlags[ 2 ],
                       adjFlags[ 3 ], adjFlags[ 4 ], self.adjWeight )
      
   def getMioAttrId( self ):
      return  PyRibAmiClient.MIO_DGET_ISIS_ADJ_SID
   
ADJ_MTID_MAP = { 'MTID_IPV6' : 2, 
}

def bw_best_value_units( bw ):
   # bw is in bps, convert it to mpbs for renderding
   bw = ( bw * 1.0 ) / ( 10 **6 )
   value = bw / 1000 if ( bw >= 1000 ) else bw
   unit = "Gbps" if bw >= 1000 else "Mbps"
   return ( value, unit )

class TeReservablePriorityBandwidth( Model ):
   priority0 = Int( help="Reservable Bandwidth (bps) for priority level 0" )
   priority1 = Int( help="Reservable Bandwidth (bps) for priority level 1" )
   priority2 = Int( help="Reservable Bandwidth (bps) for priority level 2" )
   priority3 = Int( help="Reservable Bandwidth (bps) for priority level 3" )
   priority4 = Int( help="Reservable Bandwidth (bps) for priority level 4" )
   priority5 = Int( help="Reservable Bandwidth (bps) for priority level 5" )
   priority6 = Int( help="Reservable Bandwidth (bps) for priority level 6" )
   priority7 = Int( help="Reservable Bandwidth (bps) for priority level 7" )
   
   def render( self ):
      print "        Unreserved BW:"
      NUM_BW_CLASSES = 8
      PriorityBwHeader1 = "            TE class %s: %0.2f %s\tTE class %s: %0.2f %s"
      PriorityBwHeader2 = "\tTE class %s: %0.2f %s"
      linesWithThreeClasses = int( NUM_BW_CLASSES / 3 )
      for i in range( linesWithThreeClasses ):
         bw0 = getattr( self, "priority" + str( 3 *i ) )
         bw1 = getattr( self, "priority" + str( 3 *i + 1 ) )
         bw2 = getattr( self, "priority" + str( 3 *i + 2 ) )
         best_value_units_bw0 = bw_best_value_units( bw0 )
         best_value_units_bw1 = bw_best_value_units( bw1 )
         best_value_units_bw2 = bw_best_value_units( bw2 )
         print ( PriorityBwHeader1 + PriorityBwHeader2 ) % ( 3*i,
                                    best_value_units_bw0[ 0 ],
                                    best_value_units_bw0[ 1 ],
                                    3*i + 1,
                                    best_value_units_bw1[ 0 ],
                                    best_value_units_bw1[ 1 ],
                                    3 *i + 2,
                                    best_value_units_bw2[ 0 ],
                                    best_value_units_bw2[ 1 ] )

      bw0 = getattr( self, "priority" + str( 3 *linesWithThreeClasses ) )
      bw1 = getattr( self, "priority" + str( 3 *linesWithThreeClasses + 1 ) )
      best_value_units_bw0 = bw_best_value_units( bw0 )
      best_value_units_bw1 = bw_best_value_units( bw1 )
      print PriorityBwHeader1 % ( 3 *linesWithThreeClasses,
                                  best_value_units_bw0[ 0 ],
                                  best_value_units_bw0[ 1 ],
                                  3 *linesWithThreeClasses + 1,
                                  best_value_units_bw1[ 0 ],
                                  best_value_units_bw1[ 1 ] )


class LinkTeInfo( Model ):
   administrativeGroup = Int( help="Administrative Group of the Link",
                              optional=True )
   metric = Int( help="TE Link cost (if configured on link)", optional=True )
   maxLinkBw = Int( help="Maximum bandwidth (bps) that can be used " \
                      "on Directed Link",
                      optional=True )
   maxReservableBw = Int( help="Maximum bandwidth (bps) that can be reserved "
                            "on Directed Link",
                            optional=True )
   maxReservablePriorityBw = Submodel( valueType=TeReservablePriorityBandwidth,
                                        help="Maximum bandwidth "
                                        "reservable for a priority",
                                        optional=True )
   
   def processData( self, data ):
      self.administrativeGroup = data.pop( 'color', None )
      self.metric = data.pop( 'metric', None )
      maxLinkBw = data.pop( 'max_link_bw', None )
      self.maxLinkBw = int( maxLinkBw * ( 10 **6 ) ) if maxLinkBw is not None \
                       else None
      maxReservableBw = data.pop( 'max_resv_bw', None )
      self.maxReservableBw = int( maxReservableBw * ( 10 **6 ) ) if maxReservableBw \
                             is not None else None

      maxReservablePriorityBw = TeReservablePriorityBandwidth()
      # pylint: disable=W0612
      priority0 = data.pop( 'max_unresv_bw0', None )
      priority1 = data.pop( 'max_unresv_bw1', None )
      priority2 = data.pop( 'max_unresv_bw2', None )
      priority3 = data.pop( 'max_unresv_bw3', None )
      priority4 = data.pop( 'max_unresv_bw4', None )
      priority5 = data.pop( 'max_unresv_bw5', None )
      priority6 = data.pop( 'max_unresv_bw6', None )
      priority7 = data.pop( 'max_unresv_bw7', None )

      NUM_BW_CLASSES = 8
      for i in range( NUM_BW_CLASSES ):
         val = locals()[ "priority" + str( i ) ]
         if val is not None:
            setattr( maxReservablePriorityBw, "priority" + str( i ),
                     int( val * ( 10 **6 ) ) )
         else:
            setattr( maxReservablePriorityBw, "priority" + str( i ), None )
      if( maxReservablePriorityBw.priority0 is not None ):
         self.maxReservablePriorityBw = maxReservablePriorityBw

   def render( self ):
      maxLinkBwHeader = "        Maximum link BW: %0.2f %s"
      maxReservableBwHeader = "        Maximum reservable link BW: %0.2f %s"
      adminColorHeader = "        Administrative group (Color): 0x%x"
      if self.administrativeGroup is not None:
         print adminColorHeader % ( self.administrativeGroup )
      if self.metric is not None:   
         print "        TE default metric: %s" % ( self.metric )
      if self.maxLinkBw is not None:
         bestValueUnit = bw_best_value_units( self.maxLinkBw )
         print maxLinkBwHeader % ( bestValueUnit[ 0 ],
                                   bestValueUnit[ 1 ] )
      if self.maxReservableBw is not None:
         bestValueUnit = bw_best_value_units( self.maxReservableBw )
         print maxReservableBwHeader % ( bestValueUnit[ 0 ], bestValueUnit[ 1 ] )
      if self.maxReservablePriorityBw:
         self.maxReservablePriorityBw.render()
         
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_LINK_TE


class TeRouterId( Model ):
   v4RouterId = Ip4Address( help="TE IPv4 Router-Id" )
   v6RouterId = Ip6Address( help="TE IPv6 Router-Id", optional=True )

   def render( self ):
      if self.v4RouterId is not None:
         print "      TE IPv4 router ID: %s" % ( self.v4RouterId )
      if self.v6RouterId is not None:
         print "      TE IPv6 router ID: %s" % ( self.v6RouterId )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_RTRID

class LspNeighbors( Model ):
   systemId = Str( help="IS-IS System Id" )
   metric = Int( help="Link Metric" )
   neighborAddr = Ip4Address( help="Neighbor Interface address" )
   adjTopoId = Int( help="Neighbor topology Id" )
   adjNarrowMetricsEnabled = Bool( help="Neighbor narrow metric" )
   adjInterfaceAddresses = GeneratorList( valueType=LspAdjInterfaceAddress,
                                 help="List of interface addresses" )
   adjSids = GeneratorList( valueType=LspAdjSid, help="List of Adj SID" )
   adjLanSids = GeneratorList( valueType=LspAdjLanSid, help="List of LAN  Adj SID" )
   ip6NeighborAddress = Ip6Address( help="Neighbor IPv6 global address",
                                    optional=True )
   ip6GlobalInterfaceAddress = Ip6Address( help="IPv6 global interface address",
                                           optional=True )
   TEInfo = Submodel( valueType=LinkTeInfo,
                              help="TE Information for"
                              "the Link" )
   def processData( self, data ):
      self.systemId = data.pop( 'system-id', None )
      self.metric = data.pop( 'metric', None )
      self.neighborAddr = data.pop( 'ngb-addr', None )
      self.adjTopoId = data.pop( 'adj-topo-id', None )
      self.adjNarrowMetricsEnabled = 'adj-narrow-metrics' in data
      self.ip6NeighborAddress = data.pop( 'ipv6-ngb-addr', None )
      self.ip6GlobalInterfaceAddress = data.pop( 'ipv6-intf-addr', None )
      return data

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ADJACENCY

   def render( self ):
      if self.adjTopoId:
         if self.adjTopoId == ADJ_MTID_MAP[ 'MTID_IPV6' ]:
            print "      IS Neighbor (MT-IPv6): %-19s Metric: %s" % ( self.systemId,
                                                                      self.metric )
         else:
            print "      IS Neighbor (MT %s)  : %-19s Metric: %s" % ( self.adjTopoId,
                                                                      self.systemId,
                                                                      self.metric )
      elif self.adjNarrowMetricsEnabled:
         fmt = "      IS Neighbor (Narrow metrics, unsupported): %-19s Metric: %s"
         print fmt % ( self.systemId, self.metric )
      else:
         print "      IS Neighbor          : %-19s Metric: %s" % ( self.systemId,
                                                             self.metric )
         if self.neighborAddr:
            print "        IPv4 Neighbor Address: %s" % ( self.neighborAddr )

      for adjInterfaceAddress in self.adjInterfaceAddresses:
         adjInterfaceAddress.render()

      if self.ip6NeighborAddress:
         print "        IPv6 Neighbor Address: %s" % ( self.ip6NeighborAddress )
      if self.ip6GlobalInterfaceAddress:
         print "        Global IPv6 Interface Address: %s" % \
                                              ( self.ip6GlobalInterfaceAddress )

      for adjSid in self.adjSids :
         adjSid.render()
      for adjLanSid in self.adjLanSids :
         adjLanSid.render()
      if self.TEInfo:
         self.TEInfo.render()

class SrPrefixOptions( Model ):
   readvert = Bool( help="SR readvertisement Flag" )
   nodeSID = Bool( help="SR Node SID Flag" )
   noPenultimate = Bool( help="SR No Penultimate Hop Flag" )       
   explicitNull = Bool( help="SR Explicit Null Flag" )
   value = Bool( help="Prefix-SID carries a value instead of an index" )
   local = Bool( help="Value/Index carried by the" +
                     " Prefix-SID has local significance" )
   
def setPrefixOptions( subModel, prefixOptions ):
   subModel.readvert = True if ( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_READVERT' ] ) else False
   subModel.nodeSID = True if ( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_NODE' ] ) else False
   subModel.noPenultimate = True if ( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_NOPHP' ] ) else False
   subModel.explicitNull = True if ( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_EXPLICIT_NULL' ] ) \
         else False
   subModel.value = True if ( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_VALUE' ] ) else False
   subModel.local = True if ( prefixOptions &
         IsisCliHelper.PREFIX_SEGMENT_FLAGS[ 'ISIS_SR_PFX_LOCAL' ] ) else False
   
def getPrefixOptionsFlagsList( prefixOptions ):
   readvert = "R " if ( prefixOptions.readvert ) else ""
   nodeSID = "N " if ( prefixOptions.nodeSID ) else ""
   noPenultimate = "P " if ( prefixOptions.noPenultimate ) else ""
   explicitNull = "E " if ( prefixOptions.explicitNull ) else ""
   value = "V " if ( prefixOptions.value )  else ""
   local = "L " if ( prefixOptions.local ) else "" 
   return [ readvert, nodeSID, noPenultimate, explicitNull,
            value, local ]

class SrPrefixReachability( Model ):
   algo = Enum( values=[ IsisCliHelper.SR_ALGO_SPF, IsisCliHelper.SR_ALGO_SSPF,
                         IsisCliHelper.SR_ALGO_UNDEFINED ],
                help="SR Algo Used" )
   algoNum = Int( help="SR Algorithm" )
   sid = Int( help="SR Prefix SID" )
   options = Submodel( valueType=SrPrefixOptions,
                       help="SR Flags for Prefix options" )

   def processData( self, data ):
      self.algoNum = ord( data.pop( 'prefix-algo' ) )
      self.algo = IsisCliHelper.SR_ALGO_MAP.get( self.algoNum,
                                                 IsisCliHelper.SR_ALGO_UNDEFINED )
      self.sid = data.pop( 'prefix-sid' )
      options = ord( data.pop( 'prefix-option' ) )
      self.options = SrPrefixOptions()
      setPrefixOptions( self.options, options )
      
   def render( self ):
      flagsStr = getPrefixOptionsFlagsList( self.options )
     
      print "        SR Prefix-SID: %s Flags: [ %s%s%s%s%s%s] Algorithm: %d" % (
         self.sid, flagsStr[ 0 ], flagsStr[ 1 ], flagsStr[ 2 ], flagsStr[ 3 ],
         flagsStr[ 4 ], flagsStr[ 5 ], self.algoNum )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_PFX_SID

class RouteTag( Model ):
   tag = Int( help="Administrative Tag for IP Prefix" )
   def processData( self, data ):
      self.tag = data.pop( 'route-tag', None )

   def render( self ):
      if self.tag:
         print "        Route Tag: %d" % self.tag

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ROUTE_TAG
  
class UnsupportedTlv( Model ):
   tlvType = Int( help="TLV type" )
   tlvLength = Int( help="TLV length" )

   def processData( self, data ):
      self.tlvType = data.pop( 'tlv-type', None )
      self.tlvLength = data.pop( 'tlv-length', None )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_UNSUPPORTED_TLV

   def render( self ):
      if self.tlvType is not None and self.tlvLength is not None:
         print "      Unsupported TLV: Type: {} Length: {}".format( 
                                                               self.tlvType,
                                                               self.tlvLength )
   
class LspReachabilities( Model ):
   reachabilityV4Addr = Ip4Address( help="Reachability IPv4 address",
                                    optional=True )
   reachabilityV6Addr = Ip6Address( help="Reachability IPv6 address",
                                    optional=True )
   maskLength = Int( help="Mask Length" )
   metric = Int( help="Cost of the Adjacency" )
   metricType = Enum( values=IsisCliHelper.METRIC_TYPE_MAP.values(), \
                      help="Metric Type " )
   reachabilityTopoId = Int( help="Reachable Topology Id" )
   reachabilityNarrowMetrics = Bool( help="Unsupported Reachability Narrow Metrics" )
   reachabilityUpDown = Bool( help="Reachability Status" )
   routeTags = GeneratorList( valueType=RouteTag, help="List of Route Tags",
                            optional=True )
   srPrefixReachabilities = GeneratorList( valueType=SrPrefixReachability,
                                  help="List of SR Prefix Reachability" )
   
   def processData( self, data ):
      self.reachabilityV4Addr = data.pop( 'ipv4-ifaddr', None )
      self.reachabilityV6Addr = data.pop( 'ipv6-ifaddr', None )
      self.maskLength = data.pop( 'masklen', None )
      self.metric = data.pop( 'metric', None )
      self.metricType = IsisCliHelper.METRIC_TYPE_MAP[ int( data.pop( \
            'metric-type' ) ) ]
      self.reachabilityTopoId = data.pop( 'reach-topo-id', None )
      self.reachabilityNarrowMetrics = 'reach-narrow-metrics' in data
      self.reachabilityUpDown =  'reach-updown' in data
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_REACHABILITY
   
   def render( self ):
      reachUpDown = "Down" if self.reachabilityUpDown else "Up"
      if self.reachabilityV4Addr is not None:
         if self.reachabilityNarrowMetrics:
            fmt = "      Reachability (Narrow metrics, unsupported): %s/%s Metric:"
            fmt += " %s Type: %s"
            print fmt % ( self.reachabilityV4Addr, self.maskLength,
                 self.metric, IsisCliHelper.METRIC_TYPE_MAP.reverse()[ \
                 self.metricType ] )
         else:
            fmt = "      Reachability         : %s/%s Metric: %s Type: %s %s"
            print fmt % ( self.reachabilityV4Addr, self.maskLength,
                          self.metric, IsisCliHelper.METRIC_TYPE_MAP.reverse()[ \
                          self.metricType ], reachUpDown )

      if self.reachabilityV6Addr is not None:
         if self.reachabilityTopoId:
            if self.reachabilityTopoId == ADJ_MTID_MAP[ 'MTID_IPV6' ]:
               print "      Reachability (MT-IPv6): %s/%s Metric: %s Type: %s %s" % (
                  self.reachabilityV6Addr, self.maskLength, self.metric,
                  IsisCliHelper.METRIC_TYPE_MAP.reverse()[ self.metricType ],
                  reachUpDown )
            else:
               print "      Reachability (MT %s)  : %s/%s Metric: %s Type: %s %s" % \
                  ( self.reachabilityTopoId, self.reachabilityV6Addr,
                    self.maskLength, self.metric,
                  IsisCliHelper.METRIC_TYPE_MAP.reverse()[ self.metricType ],
                  reachUpDown )
         else:
            print "      Reachability          : %s/%s Metric: %s Type: %s %s" % \
               ( self.reachabilityV6Addr, self.maskLength,
                 self.metric, IsisCliHelper.METRIC_TYPE_MAP.reverse()[ \
                 self.metricType ], reachUpDown )

      for routeTag in self.routeTags:
         routeTag.render()

      for srPrefixRechable in self.srPrefixReachabilities:
         srPrefixRechable.render()

class SrAlgo( Model ):
   srAlgo = Enum( values=[ IsisCliHelper.SR_ALGO_SPF, IsisCliHelper.SR_ALGO_SSPF,
                           IsisCliHelper.SR_ALGO_UNDEFINED ],
                  help="SR Algo Used" )
   algoNum = Int( help="SR Algorithm" )

   def processData( self, data ):
      self.algoNum = ord( data.pop( 'sr algo' ) )
      self.srAlgo = IsisCliHelper.SR_ALGO_MAP.get( self.algoNum,
                                                   IsisCliHelper.SR_ALGO_UNDEFINED )

   def render( self ):
      print "        Algorithm: %d" % ( self.algoNum )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_ALGO

class SrCapabilitySrgb( Model ):
   srgbBase = Int( help="Srgb base" )
   srgbRange = Int( help="Srgb range" )

   def processData( self, data ):
      self.srgbBase = data.pop( 'srgb-base', None )
      self.srgbRange = data.pop( 'srgb-range', None )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_CAP_SRGB

   def render( self ):
      out = ""
      if self.srgbBase:
         out = "          SRGB Base: %s" % ( self.srgbBase )
      if self.srgbRange:   
         out += " Range: %s" % ( self.srgbRange )
      if out:
         print out
     
class SrCapabilityFlags( Model ):
   mplsV4 = Bool( help="Sr Capability mpls V4 Flag" )
   mplsV6 = Bool( help="Sr capability mpls V6 Flag" )
   srcapV6 = Bool( help="Sr Capability V6 Flag" )

def setSrCapabilityFlags( subModel, flags ):
   subModel.mplsV4 = True if ( flags &
         IsisCliHelper.SRCAPABILITY_MAP[ 'ISIS_SR_CAP_MPLS_IPV4' ] ) else False
   subModel.mplsV6 = True if ( flags &
         IsisCliHelper.SRCAPABILITY_MAP[ 'ISIS_SR_CAP_MPLS_IPV6' ] ) else False
   subModel.srcapV6 = True if ( flags &
         IsisCliHelper.SRCAPABILITY_MAP[ 'ISIS_SR_CAP_IPV6' ] ) else False
def getSrCapabilityFlagsList( subModel ):
   mplsV4 = "I " if ( subModel.mplsV4 ) else ""
   mplsV6 = "V " if ( subModel.mplsV6 ) else ""
   srcapV6 = "H " if ( subModel.srcapV6 ) else ""
   return [ mplsV4, mplsV6, srcapV6 ]

class SrCapabilityModel( Model ):
   flags = Submodel( valueType=SrCapabilityFlags, help="SR Capability Flags" )
   srCapabilitySrgb = GeneratorList( valueType=SrCapabilitySrgb,
                            help="Segment Routing Global Block Capability Info" )
   
   def processData( self, data ):
      flags = ord( data.pop( 'flags' ) )
      self.flags = SrCapabilityFlags()
      setSrCapabilityFlags( self.flags, flags )
      
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_CAP

   def render( self ):
      flags = getSrCapabilityFlagsList( self.flags )
      print "        SR Capability: Flags: [ %s%s%s]" % ( flags[ 0 ],
                                                         flags[ 1 ], flags[ 2 ] )
      for srgbCap in self.srCapabilitySrgb:
         srgbCap.render()

class SrlbRange( Model ):
   srlbBase = Int( help="Minimum label in range" )
   srlbRange = Int( help="Length of label range" )

   def processData( self, data ):
      self.srlbBase = data.pop( "srlb-base", None )
      self.srlbRange = data.pop( "srlb-range", None )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_LB_RANGE

   def render( self ):
      print( ( " " * 10 ) + "SRLB Base: {} Range: {}".format( self.srlbBase,
                                                              self.srlbRange ) )

class SrlbModel( Model ):
   # no flags defined as of draft-ietf-isis-segment-routing-extensions-15
   srlbRanges = List( valueType=SrlbRange,
                      help="Ranges of labels making up SR Local Block" )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_LB

   def render( self ):
      print( "        SR Local Block:" )
      for srlbRange in self.srlbRanges:
         srlbRange.render()

class AreaLeaderModel( Model ):
   prio = Int( help="Area leader priority" )
   algo = Int( help="Area leader algorithm" )

   def processData( self, data ):
      self.prio = ord( data.pop( "prio", chr( 0 ) ) )
      self.algo = ord( data.pop( "algo", chr( 0 ) ) )

   def render( self ):
      print "        Area leader priority:", self.prio, "algorithm:", self.algo

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA_LEADER

class DynamicFloodingModel( Model ):
   enabled = Bool( default=False, help="Dynamic flooding enabled" )

   def processData( self, data ):
      self.enabled = ord( data.pop( 'dynamic-flooding', '0' ) ) != 0

   def render( self ):
      if self.enabled:
         print "        Dynamic Flooding: Enabled"

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_DYNAMIC_FLOODING

class RouterCapabilityFlags( Model ):
   rtrCapS = Bool( help="Router Capability S Flag" )
   rtrCapD = Bool( help="Router Capability D Flag" )
   
def setRouterCapabilityFlags( subModel, flags ):
   subModel.rtrCapS = True if \
         ( IsisCliHelper.RTR_CAPABILITY_MAP[ 'ISIS_RTR_CAP_FLAG_S' ] & flags ) \
         else False
   subModel.rtrCapD = True if \
         ( IsisCliHelper.RTR_CAPABILITY_MAP[ 'ISIS_RTR_CAP_FLAG_D' ] & flags ) \
         else False
   
def getRouterCapabilityFlagsList( subModel ):
   rtrCapS = "S " if ( subModel.rtrCapS ) else ""
   rtrCapD = "D " if ( subModel.rtrCapD ) else ""
   return [ rtrCapS, rtrCapD ]

class RouterCapability( Model ):
   routerId = Str( help="Router Id" )
   flags = Submodel( valueType=RouterCapabilityFlags,
                     help="Router Capability Flags" )
   srAlgos = GeneratorList( valueType=SrAlgo, help="Algorithm used" )
   srCapabilities = GeneratorList( valueType=SrCapabilityModel,
                                   help="SR Capabilities" )
   # beware, RibCapiLib populates submodels right after populating the
   # parent model, see also isis_dget_lsdb_caps_write
   srlb = Submodel( valueType=SrlbModel, optional=True,
                    help="SR Local Block Information" )
   areaLeader = Submodel( valueType=AreaLeaderModel, optional=True,
                          help="Area Leader" )
   dynamicFlooding = Submodel( valueType=DynamicFloodingModel, optional=True,
                               help="Dynamic Flooding" )

   def processData( self, data ):
      self.routerId = data.pop( 'router-id' )
      flags = ord( data.pop( 'flags' ) )
      self.flags = RouterCapabilityFlags()
      setRouterCapabilityFlags( self.flags, flags )
      
   def render( self ):
      flags = getRouterCapabilityFlagsList( self.flags )
      print "      Router Capabilities: Router Id: %s Flags: [ %s%s]" % (
         self.routerId, flags[ 0 ], flags[ 1 ] )
      for attr in ( 'srlb', 'areaLeader', 'dynamicFlooding' ):
         obj = getattr( self, attr )
         if obj:
            obj.render()
      for attr in ( 'srCapabilities', 'srAlgos' ):
         for item in getattr( self, attr ):
            item.render()

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_RTR_CAPABILITY


def setSrBindingFlag( subModel, flag ):
   subModel.attach = True if ( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_ATTACH' ] ) else False
   subModel.leak = True if ( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_LEAK' ] ) else False
   subModel.flood = True if ( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_FLOOD' ] ) else False
   subModel.mirror = True if ( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_MIRROR' ] ) else False
   subModel.addrFamily = True if ( flag &
         IsisCliHelper.SR_BINIDING_MAP[ 'ISIS_SR_BINDING_AF' ] ) else False

def getSrBindingFlagList( flags ):
   attach = "A " if ( flags.attach ) else ""
   leak = "D " if ( flags.leak ) else ""
   flood = "S " if ( flags.flood ) else ""
   mirror = "M " if ( flags.mirror ) else ""
   addrFamily = "F " if ( flags.addrFamily ) else ""
   return [ attach, leak, flood, mirror, addrFamily ]

class SrBindingFlags( Model ):
   attach = Bool( help="Sr Binding Attach Flags" )
   leak = Bool( help="Sr Binding leak Flag" )
   flood = Bool( help="Sr Binding flood Flag" )
   mirror = Bool( help="Sr Binding Mirror Flag" )
   addrFamily = Bool( help="Sr Binding Address Family Flag" )
   
class SrBindingModel( Model ):
   flags = Submodel( valueType=SrBindingFlags, help="SR Binding Flags" )
   label = Int( help="SR Binding Label" )
   weight = Int( help="SR Binding Weight of the path" )
   prefix = IpGenericAddress( help="SR Binding Ip Prefix" )
   maskLength = Int( help="SR Binding Mask Length" )
   rangeValue = Int( help="SR Binding Range" )
   srBindingTopologyId = Int( help="Sr Binding Topology Id" )
   srPrefixReachabilities = GeneratorList( valueType=SrPrefixReachability,
                                  help="List of SR Reachable Prefixes" )
   
   def processData( self, data ):
      flags = ord( data.pop( 'flag' ) )
      self.flags = SrBindingFlags()
      setSrBindingFlag( self.flags, flags )
      self.weight = ord( data.pop( 'weight' ) ) 
      self.prefix = data.pop( 'prefix' )
      self.maskLength = data.pop( 'mask' )
      self.rangeValue = data.pop( 'range' )
      self.srBindingTopologyId = data.pop( 'mtid', None )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_SR_BINDING
  
   def render( self ):
      mtstr = ""
      if self.srBindingTopologyId is not None:
         if self.srBindingTopologyId == ADJ_MTID_MAP[ 'MTID_IPV6' ]:
            mtstr = " (MT-IPv6)"
         else:
            mtstr = " (MT %s)" % self.srBindingTopologyId
         
      flagsStr = getSrBindingFlagList( self.flags )
      outputFmt = "      Segment Binding%s: Flags: [ %s%s%s%s%s] Weight: %d"
      outputFmt += " Range: %s Pfx %s/%s"
      print outputFmt % ( mtstr, flagsStr[ 0 ], flagsStr[ 1 ], flagsStr[ 2 ],
                          flagsStr[ 3 ], flagsStr[ 4], self.weight, self.rangeValue,
                          self.prefix, self.maskLength )

      for srPrefixReachable in self.srPrefixReachabilities:
         srPrefixReachable.render()

class SrlgIds( Model ):
   gid = Int( help="Shared Risk Link Group Identifier" )

   def processData( self, data ):
      self.gid = data.pop( 'srlg-gid' )
      return data

   def render( self ):
      print "        Group ID: %s" % self.gid

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_SRLG_GID

class TeSrlgBase( Model ):
   ngbSystemId = Str( help="System identifier of the neighbor" )
   srlgIds = GeneratorList( valueType=SrlgIds,
                            help="List of Shared Risk Link Group IDs" )

   def processData( self, data ):
      self.ngbSystemId = data.pop( "system-id" )
      return data

   def render( self ):
      for srlgid in self.srlgIds:
         srlgid.render()

class TeSrlg( TeSrlgBase ):
   flags = Enum( values=( 'Numbered', 'Unnumbered' ), help="SRLG flags" )
   intfAddr = IpGenericAddress( help="Interface address" )
   ngbAddr = IpGenericAddress( help="Neighbor address" )

   def processData( self, data ):
      data = TeSrlgBase.processData( self, data )
      self.flags = IsisCliHelper.INTF_SRLG_FLAG_MAP[ ord( data.pop( "flags" ) ) ]
      self.intfAddr = data.pop( "intf-addr" )
      self.ngbAddr = data.pop( "nbr-addr" )

   def render( self ):
      if self.ngbSystemId:
         srlgHeader = "      Shared Risk Link Group:  Neighbor %s (%s)"
         print srlgHeader % ( self.ngbSystemId, self.flags )
         print "        Interface Address: %s" % self.intfAddr
         print "        Neighbor Address: %s" % self.ngbAddr
         TeSrlgBase.render( self )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_IPV4_SRLG

class v6SrlgFlags( Model ):
   neighborAddressIncluded = Bool( default=False,
                                   help='IPv6 neighbor address is '
                                   'included (NA bit)' )

   def processData( self, data ):
      self.neighborAddressIncluded = ord( data.pop( 'flags' ) ) != 0

class TeIpv6Srlg( TeSrlgBase ):
   flags = Submodel( valueType=v6SrlgFlags, help='IPv6 SRLG flags' )
   v6InterfaceAddress = Ip6Address( help='IPv6 interface address' )
   v6NeighborAddress = Ip6Address( optional=True, help='IPv6 neighbor address' )

   def processData( self, data ):
      data = TeSrlgBase.processData( self, data )
      flags = v6SrlgFlags()
      flags.processData( data )
      self.flags = flags
      self.v6InterfaceAddress = data.pop( 'intf-addr' )
      self.v6NeighborAddress = data.pop( 'nbr-addr', None )

   def render( self ):
      if self.ngbSystemId:
         srlgHeader = "      IPv6 Shared Risk Link Group:  Neighbor %s"
         print srlgHeader % ( self.ngbSystemId )
         print "        Interface Address: %s" % self.v6InterfaceAddress
         if self.v6NeighborAddress is not None:
            print "        Neighbor Address: %s" % self.v6NeighborAddress
         TeSrlgBase.render( self )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_TE_IPV6_SRLG

class LspDynFloodNodeId( Model ):
   startIndex = Int( help="Start index of enclosed systems" )
   endIndex = Int( help="End index of enclosed systems" )

   def processData( self, data ):
      self.startIndex = data.pop( 'start-index' )
      self.endIndex = data.pop( 'end-index' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_DYNFLOOD_NODEID_BRIEF

class LspDynFloodPath( Model ):
   path = Str( help="Path of node indices" )

   def processData( self, data ):
      string = data.pop( 'path' )
      result = ""
      start = 0
      while start < len( string ):
         # Extract the next index
         index = int( string[ start:start + 2 ].encode( 'hex' ), 16 )
         start = start + 2
         if result:
            result = result + " "
         result = result + "%d" % index
      self.path = result

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_DYNFLOOD_PATH_TLV

   def render( self ):
      if self.path:
         output = "      Dynamic Flooding Path: " + self.path
         while len( output ) > LINE_MAX:
            index = output.rfind( " ", 0, LINE_MAX )
            print output[ 0 : index ]
            output = ' ' * 8 + output[ index + 1 : ]
         if len ( output ) > 8:
            print output

class AreaSegmentFlags( Model ):
   areaSegAf = Bool( help="Area segment address family flag" )
   areaSegValue = Bool( help="Area segment value flag" )
   areaSegLocal = Bool( help="Area segment local flag" )

   def set( self, charVal ):
      self.areaSegAf = False
      self.areaSegValue = False
      self.areaSegLocal = False
      if charVal:
         intVal = ord( charVal )
         asf = IsisCliHelper.AREA_SEGMENT_FLAGS
         self.areaSegAf = ( asf[ 'ISIS_SR_AREA_SEGMENT_AF' ] & intVal ) != 0
         self.areaSegValue = ( asf[ 'ISIS_SR_AREA_SEGMENT_VALUE' ] & intVal ) != 0
         self.areaSegLocal = ( asf[ 'ISIS_SR_AREA_SEGMENT_LOCAL' ] & intVal ) != 0

   def render( self ):
      return '%s%s%s' % (
         'F ' if self.areaSegAf else '',
         'V ' if self.areaSegValue else '',
         'L ' if self.areaSegLocal else '' )

class AreaProxySubTlv( Model ):
   areaProxySystemId = Str( help="Area proxy system ID", optional=True )
   flags = Submodel( valueType=AreaSegmentFlags, help="Area segment flags",
                     optional=True )
   index = Int( help="Area segment index/label", optional=True )
   prefix = IpGenericAddress( help="Area segment IP Prefix", optional=True )
   maskLength = Int( help="Area segment IP Prefix Mask Length", optional=True )

   def processData( self, data ):
      self.areaProxySystemId = data.pop( 'area-proxy-sysid', None )
      if 'area-segment-flags' in data:
         self.flags = AreaSegmentFlags()
         self.flags.set( data.pop( 'area-segment-flags' ) )
      self.index = data.pop( 'area-segment-index', None )
      self.prefix = data.pop( 'area-segment-addr', None )
      if 'area-segment-mask-len' in data:
         self.maskLength = ord( data.pop( 'area-segment-mask-len' ) )

   def render( self ):
      if self.areaProxySystemId:
         print "        System ID: %s" % self.areaProxySystemId
      if self.index:
         print "        Area Segment ID: %d flags: [ %s] prefix: %s/%s" % (
            self.index, self.flags.render(), self.prefix, self.maskLength )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA_PROXY_SUBTLV

class LspBaseModel( Model ):
   expiryTime = Float( help="UTC time for Future Expiry time of LSP" )
   _lspid = Str( help="LSPID for this LSP" )
   sequence = Int( help="Sequence number for this LSP" )
   checksum = Int( help="Checksum value" )
   intermediateSystemType = Enum( values=( 'none', 'L1', 'L2' ),
                  optional=True, help="IS-type for LSP" )
   flags = Submodel( valueType=LspFlags, optional=True,
                     help="IS-IS Lsp flags" )

   def processData( self, data ):
      self.expiryTime = data.pop( 'lifetime' ) + Tac.utcNow()
      self._lspid = data.pop( 'lsid' )
      self.sequence = data.pop( 'sequence' )
      self.checksum = data.pop( 'checksum' )
      intermediateSystemType = data.pop( 'is-type' )
      self.intermediateSystemType = 'none'
      if intermediateSystemType == 1:
         self.intermediateSystemType = 'L1'
      if intermediateSystemType == 2:
         self.intermediateSystemType = 'L2'
      flags = LspFlags()
      reversedFlagsDict = IsisCliHelper.ISIS_LSP_FLAGS_MAP.reverse()
      if 'flags' in data and data[ 'flags' ]:
         flagList = data.pop( 'flags' ).split()
         for flag in flagList:
            setattr( flags, reversedFlagsDict[ flag ], True )
      self.flags = flags
      return data
   
   def renderData( self, vrf=None ):
      from RoutingIsisCli import getHostName
      showFormat = "    %-21s %-9s %-6s %-5s %s <%s>"
      hostname = getHostName( getRouterId( self._lspid[ 0:14 ] ),
                                                     vrfName=vrf )
      lspidRender = self._lspid
      if hostname:
         lspidRender = hostname + self._lspid[ -6: ]
      lspRemainingLife = int( self.expiryTime - Tac.utcNow() )
      if lspRemainingLife < 0:
         lspRemainingLife = 0
      print showFormat % ( lspidRender, 
                           self.sequence, self.checksum,
                           lspRemainingLife,
                           self.intermediateSystemType,
                           getFlagsString( self.flags ) )
      
   def getKey( self, data ):
      return data[ 'lsid' ]
   
class LspDetailModel( LspBaseModel ):
   addrFamily = Str( help="Address Family", optional=True )
   refreshTime = Float( help="Future refresh time of LSP in UTC", 
                        optional=True )
   lspGenerationTime = Float( help="Next LSP generation time in UTC",
                      optional=True )
   rcvdLifetime = Int( help="Received remaining lifetime of an LSP ( in seconds )",
                        optional=True )
   modifiedLifetime = Int( help="Modified remaining lifetime of an LSP "\
                           "( in seconds )", optional=True )
   hostname = Submodel( valueType=LspHostname, help="IS-IS hostname",
                        optional=True )
   teRouterID = Submodel( valueType=TeRouterId, help="IS-IS TE Router-ID",
                          optional=True )
   auth = Submodel( valueType=LspAuthModel, help="Auth details for the LSP",
                       optional=True )

   areaAddresses = GeneratorList( valueType=LspAreaAddress,
                                  help="List of Area Addresses" )

   multiTopologies = GeneratorList( valueType=LspMtModel,
                           help="Multi Topology details for the LSP",
                           optional=True )

   interfaceAddresses = GeneratorList( valueType=InterfaceAddresses,
                              help="List of Interface Addresses",
                              optional=True )
   neighbors = GeneratorList( valueType=LspNeighbors,
                              help="List of LSP neighbors Details", optional=True )
   reachabilities = GeneratorList( valueType=LspReachabilities ,
                        help="Reachabilities Information of LSP",
                        optional=True )
   areaProxySubTlvs = GeneratorList( valueType=AreaProxySubTlv,
                                     help="Area proxy information", optional=True )
   routerCapabilities = GeneratorList( valueType=RouterCapability,
                            help="Router Capability Information",
                            optional=True )
   srBindings = GeneratorList( valueType=SrBindingModel,
                         help="Segment Routing Binding Information",
                         optional=True )
   srlgInfos = GeneratorList( valueType=TeSrlg,
                              help="Traffic Engineering Shared Risk "
                              "Link Group information", optional=True )
   ip6SrlgInfos = GeneratorList( valueType=TeIpv6Srlg, optional=True,
                        help="Traffic Engineering IPv6 Shared "
                             "Risk Link Group information" )
   _generationTimersSet = Bool( default=False ,
                                help="Generation Timers Set" + 
                                "for Self Originated Lsp" )
   dynFloodNodeId = GeneratorList( valueType=LspDynFloodNodeId,
                                   help="Dynamic Flooding Area Node IDs",
                                   optional=True )
   dynFloodPath = GeneratorList( valueType=LspDynFloodPath,
                                 help="Dynamic Flooding Paths",
                                 optional=True )
   purgeOriginator = Str( help="Purge originator", optional=True )
   purgeUpstreamNgb = Str( help="Purge originator upstream neighbor", optional=True )
   unsupportedTlvs = GeneratorList( valueType=UnsupportedTlv,
                                    help="Unsupported IS-IS TLV Information",
                                    optional=True )

   def processData( self, data ):
      LspBaseModel.processData( self, data )
      remLspGenerationWaitTime = data.pop( 'rem-wait-time', None )
      if remLspGenerationWaitTime is not None:
         self._generationTimersSet = True
         if remLspGenerationWaitTime > 0:
            self.lspGenerationTime = Tac.utcNow() + float(
               remLspGenerationWaitTime ) / 1000 
      refreshTime = data.pop( 'refresh-time', None )
      if refreshTime is not None:
         self.refreshTime = refreshTime + Tac.utcNow()
      self.rcvdLifetime = data.pop( 'rcvd-lifetime', None )
      self.modifiedLifetime = data.pop( 'modified-lifetime', None )
      addrFamily = data.pop( 'addr-family', None )
      if addrFamily:
         self.addrFamily = ""
         for nlpId in IsisCliHelper.NLPID_MAP:
            if nlpId[ 1 ] & addrFamily:
               self.addrFamily += nlpId[ 0 ]
      self.purgeOriginator = data.pop( 'purge-originator', None )
      self.purgeUpstreamNgb = data.pop( 'purge-upstream-ngb', None )
      return data
   
   def getKey( self, data ):
      return LspBaseModel.getKey( self, data )
      
   def renderData( self, vrf=None ):
      LspBaseModel.renderData( self, vrf=vrf )
      lspGenRender = "      LSP generation remaining wait time: %s ms"
      if self.lspGenerationTime != None:
         generationTime = max( int( ( self.lspGenerationTime - Tac.utcNow() ) \
                                    * 1000 ), 0 )
         print lspGenRender % ( generationTime )
      else:
         if self._generationTimersSet is True:
            print lspGenRender % ( 0 )
      lspRefreshRender = "      Time remaining until refresh: %s s"
      if self.refreshTime is not None:
         lspRemainingRefresh = max( int( self.refreshTime - Tac.utcNow() ), 0 )
         print lspRefreshRender % lspRemainingRefresh
      if self.rcvdLifetime is not None and self.modifiedLifetime is not None:
         lifetimeRender = "      Remaining lifetime received: %s s Modified to: %s s"
         print lifetimeRender % ( self.rcvdLifetime, self.modifiedLifetime )
         
      if self.addrFamily:
         print "      NLPID:%s" % ( self.addrFamily )
      if self.hostname:
         self.hostname.render()
      if self.auth:
         self.auth.render()

      if self.purgeOriginator:
         print "      Purge originator: %s" % ( self.purgeOriginator )
      if self.purgeUpstreamNgb:
         print "      Purge originator upstream neighbor: %s" %\
                      ( self.purgeUpstreamNgb )
      if self.teRouterID:
         self.teRouterID.render()
      for address in self.areaAddresses:
         address.render()
      for mtDetail in self.multiTopologies:
         mtDetail.render()
      for interface in self.interfaceAddresses:
         interface.render()
      for neighbor in self.neighbors:
         neighbor.render()
      for reachable in self.reachabilities:
         reachable.render()

      first = True
      for info in self.areaProxySubTlvs:
         if first:
            print "      Area Proxy:"
            first = False
         info.render()

      for caps in self.routerCapabilities:
         caps.render()
      for binding in self.srBindings:
         binding.render()
      for srlginfo in self.srlgInfos:
         srlginfo.render()
      for ip6Srlginfo in self.ip6SrlgInfos:
         ip6Srlginfo.render()
      # To save space, format the output of dynamic flooding system IDs here.
      # It shows as
      #       Dynamic Flooding Area Node IDs: n1-n2, n3-n4,
      #         n5-n6, n7-n8, ...
      # One range of indices for each TLV.
      bufPrev = None
      bufPrevLen = 0
      header = ' ' * 6 + 'Dynamic Flooding Area Node IDs:'
      if self.dynFloodNodeId:
         bufPrev = header
         bufPrevLen = len( bufPrev )

      for nodeId in sorted( self.dynFloodNodeId, key=attrgetter( 'startIndex' ) ):
         buf = ' %d-%d,' % ( nodeId.startIndex, nodeId.endIndex )
         bufLen = len( buf )
         if bufLen + bufPrevLen <= LINE_MAX:
            bufPrev += buf
            bufPrevLen += bufLen
         else:
            # If the current node id output does not fit, print out the bufPrev
            # and start a new line
            print bufPrev
            bufPrev = ' ' * 8 + buf
            bufPrevLen = len( bufPrev )

      if bufPrev != header:
         # get rid of the "," in the end
         print bufPrev[ : -1 ]

      for path in self.dynFloodPath:
         path.render()

      # unsupported TLVs always rendered at the end.
      for unsupportedTlv in self.unsupportedTlvs:
         unsupportedTlv.render()

class LevelBaseModel( Model ):
   _level = Int( help='Lsdb Level' )
   
   def getKey( self, data ):
      return int( data[ 'level-num' ] )
   
   def processData( self, data ):
      self._level = data.pop( 'level-num' )
      return data
   
   def renderData( self, vrf=None ):
      print "  IS-IS Level %d Link State Database" % ( self._level )
      print ( "    %-21s %-9s %-6s %-5s %-2s %-5s" %
              ( "LSPID", "Seq Num", "Cksum", "Life", "IS", "Flags" ) )
      for _,lsp in self.lsps:
         lsp.renderData( vrf=vrf )
         
class LevelDetailModel( LevelBaseModel ):
   lsps = GeneratorDict( keyType=str, valueType=LspDetailModel,
                help='A mapping of systemId to LSP in LSDB' )
      
   def processData( self, data ):
      LevelBaseModel.processData( self, data )
      
   def renderData( self, vrf=None ):
      LevelBaseModel.renderData( self, vrf=vrf )
   def getKey( self, data ):
      return LevelBaseModel.getKey( self, data )

class LevelSummaryModel( LevelBaseModel ):
   lsps = GeneratorDict( keyType=str, valueType=LspBaseModel,
                help='A mapping of systemId to LSP in LSDB' )
      
   def processData( self, data ):
      LevelBaseModel.processData( self, data )
         
   def renderData( self, vrf=None ):
      LevelBaseModel.renderData( self, vrf=vrf )
         
   def getKey( self, data ):
      return LevelBaseModel.getKey( self, data )
   
class IsisLsdbBaseModel( Model ):
   _instanceName = Str( help='ISIS Instance name' )
   _vrf = Str( help='VRF Name' )
   
   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )
      
   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for _,levelEntry in self.level:
         levelEntry.renderData( vrf=self._vrf )

class IsisLsdbDetailModel( IsisLsdbBaseModel ):
   level = GeneratorDict( keyType=int, valueType=LevelDetailModel,
                                 help="A mapping of isis level to LSDB" )
   def processData( self, data ):
      IsisLsdbBaseModel.processData( self, data )
      
   def render( self ):
      IsisLsdbBaseModel.render( self )
      
class IsisLsdbSummaryModel( IsisLsdbBaseModel ):
   level = GeneratorDict( keyType=int, valueType=LevelSummaryModel,
                                 help="A mapping of isis level to LSDB" )
   def processData( self, data ):
      IsisLsdbBaseModel.processData( self, data )
      
   def render( self ):
      IsisLsdbBaseModel.render( self )
         
lspDbModel = IsisCliModelCommon. \
             generateIsisDictCliModel( IsisLsdbDetailModel )
lspDbVRFModel = generateVrfCliModel( lspDbModel,
                                     "IS-IS instance information for all VRFs" ) 
lspDbSummaryModel = IsisCliModelCommon. \
             generateIsisDictCliModel( IsisLsdbSummaryModel )
lspDbSummaryVRFModel = generateVrfCliModel( lspDbSummaryModel,
                                     "IS-IS instance information for all VRFs" ) 

ISIS_SR_GB_MAX = 65536

ISIS_GLOBAL_HELLO_PADDING_ON = 1

ISIS_HELLO_PADDING_DEFAULT = 2
ISIS_HELLO_PADDING_ON = 1

ISIS_INTERFACE_AREA_INDENT = "        "
ISIS_MIO_LEVEL_1 = 1
ISIS_MIO_LEVEL_2 = 2
ISIS_MIO_LEVEL_1_2 = 3
CIRC_FLAG_SENT_RR = 1
CIRC_FLAG_SENT_SA = 2
CIRC_FLAG_RCVD_RA = 4
CIRC_FLAG_RCVD_CSNP = 8

class IsisIntfAreaModel( Model ):
   areaId = Str( help='Area ID' )

   def processData(self, data):
      self.areaId = data.pop('area-id')

   def render( self ):
      print( "%s%s" % (ISIS_INTERFACE_AREA_INDENT, self.areaId) )

class IsisLevelAdjModel( Model ):
   systemId = Str( help='System ID of neighbor' )
   hostname = Str( help='Hostname', optional=True )
   level = Enum( values=IsisCliHelper.INTERFACE_LEVEL_RENDER_MAP.keys(), \
                 help = 'Level' )
   state = Enum( values=IsisCliHelper.INTERFACE_ADJ_STATE_MAP.keys(), \
                 help = 'Adjacency State' )
   adjType = Enum( values=IsisCliHelper.INTERFACE_ADJ_TYPE_MAP.keys(), \
                   help = 'Adjacency Type' )
   addrFamily = Enum( values=IsisCliHelper.ADDR_TOPO_RENDER_MAP.keys(), 
                      help='Address Family' )
   holdTime = Int( help='Hold Time in seconds' )
   supportedTopologies = Enum( values=IsisCliHelper.ADDR_TOPO_RENDER_MAP.keys(), 
                      help='Supported topologies', 
                      optional=True )
   snpa = Str( help='Subnetwork Point of Attachment of neighbor', 
               optional=True )
   priority = Int( help='Priority', optional=True )
   ipv4IntfAddr = Ip4Address( help='IPv4 Interface Address', optional=True )
   ipv6IntfAddr = Ip6Address( help='IPv6 Interface Address', optional=True )
   intfAreas = GeneratorList( valueType=IsisIntfAreaModel, 
                                 help='List of interface Areas' )

   def processData( self, data ):
      self.systemId = data.pop('system-id-raw')
      self.hostname = data.pop('system-id')
      if self.systemId == self.hostname:
         self.hostname = None
      levelNum = data.pop('level')
      if levelNum in IsisCliHelper.LEVEL_MAP:
         self.level = IsisCliHelper.LEVEL_MAP[ levelNum ]
      else:
         self.level = "unknown"

      self.state = IsisCliHelper.INTERFACE_ADJ_STATE_MAP.reverse()[ \
            data.pop('state') ]
      self.adjType = IsisCliHelper.INTERFACE_ADJ_TYPE_MAP.reverse()[ \
            data.pop('type') ]

      addrFamilyNum = data.pop('address-family')
      if addrFamilyNum in IsisCliHelper.ADDRESS_FAMILY_MAP:
         self.addrFamily = IsisCliHelper.ADDRESS_FAMILY_MAP[ addrFamilyNum ]
      else:
         self.addrFamily = "unknown"

      self.holdTime = data.pop('hold-time')
      if 'mt' in data and ord(data.pop('mt')) != 0:
         if data[ 'supported-topo' ] in IsisCliHelper.INTERFACE_ADDR_MAP:
            self.supportedTopologies = IsisCliHelper.INTERFACE_ADDR_MAP[ 
                                          data.pop('supported-topo') ]
         else:
            self.supportedTopologies = "none"

      if 'snpa' in data:
         self.snpa = data.pop('snpa')
         self.priority = data.pop('priority')
      if 'ipv4-ifaddr' in data:
         self.ipv4IntfAddr = data.pop('ipv4-ifaddr')
      if 'ipv6-ifaddr' in data:
         self.ipv6IntfAddr = data.pop('ipv6-ifaddr')

      return data

   def render( self ):
      print( "    Adjacency %s:" 
             % ( self.hostname if self.hostname else self.systemId ) )
      print( "      State: %s, Level: %s Type: %s" 
            % ( IsisCliHelper.INTERFACE_ADJ_STATE_MAP[ self.state ], 
                IsisCliHelper.INTERFACE_LEVEL_RENDER_MAP[ self.level ], 
                IsisCliHelper.INTERFACE_ADJ_TYPE_MAP[ self.adjType ]) )
      print( "      Advertised Hold Time: %s" % (self.holdTime) )
      print( "      Supported Protocols: %s" 
            % (IsisCliHelper.ADDR_TOPO_RENDER_MAP[ self.addrFamily ]) )
      if self.supportedTopologies is not None:
         if self.supportedTopologies != "none":
            print("      Supported Topologies: %s" 
                  % (IsisCliHelper.ADDR_TOPO_RENDER_MAP[ self.supportedTopologies ]))
         else:
            print("      Supported Topologies:")
      if self.snpa is not None:
         print( "      SNPA: %s, Priority: %d" % (self.snpa, self.priority) )
      if self.ipv4IntfAddr is not None:
         print( "      IPv4 Interface Address: %s" % (self.ipv4IntfAddr) )
      if self.ipv6IntfAddr is not None:
         print( "      IPv6 Interface Address: %s" % (self.ipv6IntfAddr) )
      print("      Areas:")

      for entry in self.intfAreas:
         entry.render()

class GracefulRestartStatus( Model ):
   rrSent = Bool( help='Restart Request Sent' )
   saSent = Bool( help='Suppress Adjacency Sent' )
   raRcvd = Bool( help='Restart Acknowledgement received' )
   csnpRcvd = Bool( help='CSNP received' )

class BroadcastCircuitInfoModel( Model ):
   lanId = Str( help='LAN ID for broadcast circuit' )
   lanIdWithHostname = Str( help='Lan ID with Hostname for broadcast circuit', 
                              optional=True )
   priority = Int( help='Priority for broadcast circuit' )
   dis = Str( help='System ID of DIS for broadcast circuit' )
   disHostname = Str( help='Hostname of DIS for broadcast circuit', optional=True )
   disPriority = Int( help='DIS Priority for broadcast circuit' )
   
class IsisInterfaceLevelModel( Model ):
   _level = Int(help = 'IS-IS interface level')
   ipv6Metric = Int( help='IPv6 metric', optional=True )
   ipv4Metric = Int( help='IPv4 metric' )
   numAdjacencies = Int( 
                     help='Number of Adjacencies (except for loopback interfaces)',
                     optional=True )
   broadcastCircuitInfo = Submodel( valueType=BroadcastCircuitInfoModel,
                  help='Broadcast Circuit Information', optional=True )
   linkId = Str( help='Link ID for P2P interface', optional=True )
   authenticationMode = Enum(values=IsisCliHelper.AUTH_MODE_MAP.values(), 
                              help='Authentication Mode', optional=True)
   authenticationModeKeyId = Int( help='Authentication mode key ID',
                                 optional=True )
   authRxDisabled = Bool( help='Received Hello authentication check is disabled',
                          optional=True )
   grStatus = Submodel( valueType=GracefulRestartStatus,
         help='Status of graceful restart', optional=True )
   isisAdjacencies = GeneratorList( valueType=IsisLevelAdjModel, 
         help='List of ISIS Adjacency entries' )
   passive = Bool( default=False, help='Passive state' )
   v4Protection = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv4' )
   v6Protection = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv6' )
   v4SrlgProtection = Enum( values=IsisCliHelper.SRLG_PROTECTION_MAP.values(),
                            optional=True,
                            help='TI-LFA SRLG protection mode for IPv4' )
   v6SrlgProtection = Enum( values=IsisCliHelper.SRLG_PROTECTION_MAP.values(),
                            optional=True,
                            help='TI-LFA SRLG protection mode for IPv6' )

   def getKey( self, data ):
      assert data[ 'level-num' ]
      if data[ 'level-num' ] == 1 or data[ 'level-num' ] == 2:
         return str( data[ 'level-num' ] )
      else:
         return

   def processData( self, data ):
      self._level = data.pop('level-num')
      metricType = ('mt' in data and ord(data.pop('mt')) != 0)
      self.ipv4Metric = data.pop('metric')
      
      if metricType:
         self.ipv6Metric = data.pop('mt-metric')

      if 'num-adj' in data:
         self.numAdjacencies = int(data.pop('num-adj'))
      
      if 'lan-id' in data:
         self.broadcastCircuitInfo = BroadcastCircuitInfoModel()
         if data[ 'lan-id-raw' ] == data[ 'lan-id' ]:
            self.broadcastCircuitInfo.lanId = data.pop('lan-id')
            data.pop('lan-id-raw')
         else:
            self.broadcastCircuitInfo.lanId = data.pop('lan-id-raw')
            self.broadcastCircuitInfo.lanIdWithHostname = data.pop('lan-id')
         self.broadcastCircuitInfo.priority = data.pop('priority')
         if data[ 'dis-raw' ] == data[ 'dis' ]:
            self.broadcastCircuitInfo.dis = data.pop('dis')
            data.pop('dis-raw')
         else:
            self.broadcastCircuitInfo.dis = data.pop('dis-raw')
            self.broadcastCircuitInfo.disHostname = data.pop('dis')
         self.broadcastCircuitInfo.disPriority = data.pop('dis-priority')
      elif 'link-id' in data:
         self.linkId = data.pop('link-id')
      
      if 'Circuit Auth' in data:
         self.authenticationMode = IsisCliHelper.AUTH_MODE_MAP[ 
               int(data.pop('Circuit Auth')) ]
      if self.authenticationMode == "SHA":
         self.authenticationModeKeyId = data.pop( 'Circuit-Auth-mode-id' )

      if 'auth-rx-disabled' in data:
         self.authRxDisabled = ord( data[ 'auth-rx-disabled' ] ) != 0

      if 'gr-flags' in data and ord( data[ 'gr-flags' ] ) != 0:
         grFlags = ord( data.pop( 'gr-flags' ) )
         self.grStatus = GracefulRestartStatus()
         self.grStatus.rrSent = bool(grFlags & CIRC_FLAG_SENT_RR)
         self.grStatus.saSent = bool(grFlags & CIRC_FLAG_SENT_SA)
         self.grStatus.raRcvd = bool(grFlags & CIRC_FLAG_RCVD_RA)
         self.grStatus.csnpRcvd = bool(grFlags & CIRC_FLAG_RCVD_CSNP)
      self.passive = ord( data.pop( 'passive' ) ) != 0
      self.v4Protection = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v4' ) ) ]
      self.v6Protection = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v6' ) ) ]
      if 'srlg-protection-v4' in data:
         self.v4SrlgProtection = IsisCliHelper.SRLG_PROTECTION_MAP[
            ord( data.pop( 'srlg-protection-v4' ) ) ]
      if 'srlg-protection-v6' in data:
         self.v6SrlgProtection = IsisCliHelper.SRLG_PROTECTION_MAP[
            ord( data.pop( 'srlg-protection-v6' ) ) ]

   def renderEntry( self, levelName ):
      print( "    Level %d:" % (self._level) )
      passive = ' (Passive Interface)' if self.passive else ''
      adjStr = str( self.numAdjacencies ) + passive
      if self.numAdjacencies is not None:
         if self.ipv6Metric is not None:
            print( 
                  "      IPv4 Metric: %d, IPv6 Metric: %d, Number of adjacencies: %s"
                  % (self.ipv4Metric, self.ipv6Metric, adjStr) 
                  )
         else:
            print( "      Metric: %d, Number of adjacencies: %s"
                  % (self.ipv4Metric, adjStr) )
      else:
         if self.ipv6Metric is not None:
            print( "      IPv4 Metric: %d, IPv6 Metric: %d%s" 
                  % (self.ipv4Metric, self.ipv6Metric, passive) )
         else:
            print( "      Metric: %d%s" % (self.ipv4Metric, passive) )
      
      if self.broadcastCircuitInfo is not None:
         renderLanId = self.broadcastCircuitInfo.lanIdWithHostname \
                           if self.broadcastCircuitInfo.lanIdWithHostname \
                           else self.broadcastCircuitInfo.lanId
         print( "      LAN-ID: %s, Priority: %d" % (renderLanId, 
                                 self.broadcastCircuitInfo.priority) )
         renderDis = ( self.broadcastCircuitInfo.disHostname 
                        if self.broadcastCircuitInfo.disHostname 
                        else self.broadcastCircuitInfo.dis )
         print( "      DIS: %s, DIS Priority: %d" % (renderDis, 
                                       self.broadcastCircuitInfo.disPriority) )
      elif self.linkId is not None:
         print( "      Link-ID: %s" % (self.linkId) )

      modeIdStr = ""
      if self.authenticationMode == "SHA":
         modeIdStr = " Key id: %d" % self.authenticationModeKeyId
      print( "      Authentication mode: %s%s" % ( self.authenticationMode
                   if self.authenticationMode else 'None', modeIdStr ) )

      if self.authRxDisabled:
         print( "      Received Hello authentication check: disabled" )

      if self.grStatus is not None:
         print( "      Graceful Restart Status: %s%s%s%s" 
               % ("RR sent " if self.grStatus.rrSent else "", 
                  "SA sent " if self.grStatus.saSent else "", 
                  "RA rcvd " if self.grStatus.raRcvd else "", 
                  "CSNP rcvd" if self.grStatus.csnpRcvd else "") )

      protectionStr = "      TI-LFA {} protection {}is enabled for the following " \
                      "IPv{} segments: node segments, adjacency segments"
      disabledStr = "      TI-LFA protection is disabled for IPv{}"

      srlgProtectionFormat = "with SRLG {} protection "
      v4SrlgProtectionStr = ""
      if self.v4SrlgProtection is not None:
         v4SrlgProtectionStr = srlgProtectionFormat.format(
            self.v4SrlgProtection )

      v6SrlgProtectionStr = ""
      if self.v6SrlgProtection is not None:
         v6SrlgProtectionStr = srlgProtectionFormat.format(
            self.v6SrlgProtection )

      if self.v4Protection == 'disabled':
         print disabledStr.format( 4 )
      else:
         print protectionStr.format( self.v4Protection, v4SrlgProtectionStr,
                                     4 )

      if self.v6Protection == 'disabled':
         print disabledStr.format( 6 )
      else:
         print protectionStr.format( self.v6Protection, v6SrlgProtectionStr,
                                     6 )

      for entry in self.isisAdjacencies:
         if self._level is not None:
            entry.render()


class DisabledReason( Model ):
   message = Str( help='Reason for interface being disabled')
   circuitType = Str( help='Configured circuit type on interface' )
   routerCircType = Str( help='Circuit type of router instance' )

class IsisInterface( Model ):
   _interfaceName = Str( help='Interface name' )
   enabled = Bool( help='Interface enabled or not' )
   disabledReason = Submodel( valueType=DisabledReason,
                              help='Reason for interface being disabled', 
                              optional=True )
   index = Int( help='Interface index', optional=True )
   snpa = Str( help='Subnetwork Point of Attachment', optional=True )
   mtu = Int( help='MTU', optional=True )
   interfaceType = Enum(values=IsisCliHelper.INTERFACE_TYPES_MAP.values(), 
         help="Interface type", optional=True)
   srIndexV4 = Int( help='Node Segment Index for IPv4', optional=True )
   srIndexV6 = Int( help='Node Segment Index for IPv6', optional=True ) 
   topology = Enum( values=IsisCliHelper.INTERFACE_ADDR_MAP.values(), 
                    help='Type of topologies enabled', 
                    optional=True )
   bfdIpv4Enabled = Bool( help="Whether IPv4 bfd is enabled on the interface", 
                           optional=True )
   bfdIpv6Enabled = Bool( help="Whether IPv6 bfd is enabled on the interface", 
                           optional=True )
   helloPaddingEnabled = Bool( 
                           help="Whether Hello Padding is enabled on the interface",
                           optional=True )
   intfLevels = GeneratorDict( valueType=IsisInterfaceLevelModel, 
                               help="""Dictionary of Levels of the interface 
                               keyed by level name""", 
                               optional=True )
   ipv4MetricProfile = Str( help='IPv4 metric profile', optional=True )
   ipv6MetricProfile = Str( help='IPv6 metric profile', optional=True )
   interfaceSpeed = Int( help='Interface speed in mbps', optional=True )

   ipv4RouteTag = Int( help='IPv4 route tag', optional=True )
   ipv6RouteTag = Int( help='IPv6 route tag', optional=True )
   areaProxyBoundary = Bool( help='Interface enabled for area proxy boundary',
                             optional=True )
   def getKey( self, data ):
      assert data[ 'circ-id' ]
      ifName = kernelIntfFilter( data[ 'circ-id' ] )
      return ifName
   
   def fetchExtraDataFromSysDb( self, intfStatus ):
      if intfStatus.metricProfile:
         self.ipv4MetricProfile = intfStatus.metricProfile
      if intfStatus.metricProfileV6:
         self.ipv6MetricProfile = intfStatus.metricProfileV6
      if intfStatus.interfaceSpeed:
         self.interfaceSpeed = intfStatus.interfaceSpeed

   def processData( self, data ):
      if 'circ-id' not in data:
         return

      self._interfaceName = data.pop('circ-id', None)
      self.enabled = True
      self.index = data.pop('circ-index')

      self.interfaceType = IsisCliHelper.INTERFACE_TYPES_MAP[ data.pop('type') ] \
            if data['type'] in IsisCliHelper.INTERFACE_TYPES_MAP.keys() else None   
      
      if 'sr-index-v4' in data:
         indexV4 = data.pop('sr-index-v4')
         if indexV4 != ISIS_SR_GB_MAX:
            self.srIndexV4 = indexV4
      
      if 'sr-index-v6' in data:
         indexV6 = data.pop('sr-index-v6')
         if indexV6 != ISIS_SR_GB_MAX:
            self.srIndexV6 = indexV6

      if 'mt' in data and ord(data.pop('mt')) != 0:
         self.topology = IsisCliHelper.INTERFACE_ADDR_MAP[ \
               data.pop('protos-enabled') ]

      bfdConfig = data.pop('bfd-config')
      if bfdConfig == 0:
         bfdGlobal = data.pop('bfd-global')
         if ord(bfdGlobal) == 0 or bfdGlobal == '':
            self.bfdIpv4Enabled = False 
         else:
            self.bfdIpv4Enabled = True
      elif bfdConfig == 1:
         self.bfdIpv4Enabled = True
      else:
         self.bfdIpv4Enabled = False

      bfdConfig6 = data.pop('bfd-config-v6')
      if bfdConfig6 == 0:
         bfdGlobal6 = data.pop('bfd-global-v6')
         if ord(bfdGlobal6) == 0 or bfdGlobal6 == '':
            self.bfdIpv6Enabled = False
         else:
            self.bfdIpv6Enabled = True
      elif bfdConfig6 == 1:
         self.bfdIpv6Enabled = True
      else:
         self.bfdIpv6Enabled = False

      self.ipv4RouteTag = data.pop( 'route-tag-v4', None )
      self.ipv6RouteTag = data.pop( 'route-tag-v6', None )

      helloPaddingConfig = ord(data.pop('hello-padding-intf'))
      if helloPaddingConfig == ISIS_HELLO_PADDING_DEFAULT \
            and 'hello-padding-inst' in data \
            and ord(data.pop('hello-padding-inst')) == ISIS_GLOBAL_HELLO_PADDING_ON:
         self.helloPaddingEnabled = True
      elif helloPaddingConfig == ISIS_HELLO_PADDING_ON:
         self.helloPaddingEnabled = True
      else:
         self.helloPaddingEnabled = False

      areaProxyBoundaryData = ord( data.pop( 'area-proxy-boundary', chr( 0 ) ) )
      self.areaProxyBoundary = ( 0 != areaProxyBoundaryData )

      return data

   def renderEntry( self, intfFullName ):
      if self._interfaceName is None:
         return
      print( "  Interface %s:" % (intfFullName) )
      if not self.enabled:
         print("    %s" % (self.disabledReason.message) )
         return
      print( "    Index: %s SNPA: %s" % (self.index, self.snpa) )
      if self.interfaceType is not None:
         print( "    MTU: %s Type: %s" % ( self.mtu, self.interfaceType ) )
      if self.areaProxyBoundary:
         print( "    Area Proxy Boundary is Enabled" )
      else:
         print( "    Area Proxy Boundary is Disabled" )
      if self.ipv4RouteTag and self.ipv6RouteTag:
         print( "    Route Tag: IPv4: %d, IPv6: %d" % ( self.ipv4RouteTag,
                                                        self.ipv6RouteTag ) )
      elif self.ipv4RouteTag:
         print( "    Route Tag: IPv4: %d" % ( self.ipv4RouteTag ) )
      elif self.ipv6RouteTag:
         print( "    Route Tag: IPv6: %d" % ( self.ipv6RouteTag ) )

      if self.ipv4MetricProfile:
         print( "    Metric profile for IPv4: '%s'" % self.ipv4MetricProfile )
      if self.ipv6MetricProfile:
         print( "    Metric profile for IPv6: '%s'" % self.ipv6MetricProfile )
      if self.interfaceSpeed:
         print( "    Speed: %d mbps" % self.interfaceSpeed )
      if self.srIndexV4 is not None or self.srIndexV6 is not None:
         if self.srIndexV4 is not None and self.srIndexV6 is None:
            print( "    Node segment Index IPv4: %s" % (self.srIndexV4) )
         elif self.srIndexV4 is None and self.srIndexV6 is not None:
            print( "    Node segment Index IPv6: %s" % (self.srIndexV6) )
         else:
            print( "    Node segment Index IPv4: %s IPv6: %s" 
                  % (self.srIndexV4, self.srIndexV6) )
      
      if self.topology is not None:
         print( "    Enabled Topologies: %s" 
               % (IsisCliHelper.ADDR_TOPO_RENDER_MAP[ self.topology ]) )

      print( "    %s" 
            % ("BFD IPv4 is Enabled" if self.
               bfdIpv4Enabled else "BFD IPv4 is Disabled") )
      print( "    %s" 
            % ("BFD IPv6 is Enabled" if self. 
               bfdIpv6Enabled else "BFD IPv6 is Disabled") )
      print( "    %s" 
            % ("Hello Padding is Enabled" if self.
               helloPaddingEnabled else "Hello Padding is Disabled") )

      for levelName, level in self.intfLevels:
         level.renderEntry(levelName)
        
class IsisInterfaceModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   interfaces = GeneratorDict( keyType=Interface, valueType=IsisInterface, 
                               help="""Dictionary of IS-IS interfaces keyed 
                                       by interface name""" )

   def processData( self, data ):
      self._vrf = data.pop('vrf-name')
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      if self._instanceName is None:
         return
      print( "" )
      print( "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf ) )
      print( "" )
      for intfName, intf in self.interfaces:
         intf.renderEntry( intfName )

interfaceModel = IsisCliModelCommon.generateIsisDictCliModel( IsisInterfaceModel )
interfaceVRFsModel = generateVrfCliModel( interfaceModel,  
                                          "IS-IS interface information for all VRFs"
                                        )

def ngbGrStateString( state ):
   statusString = ''
   rrSa = False
   if state & ISIS_GR_RCVD_SA:
      statusString += "Status: Starting (SA rcvd"
      rrSa = True
   elif state & ISIS_GR_RCVD_RR:
      statusString += "Status: Restarting (RR rcvd"
      rrSa = True

   if rrSa:
      if state & ISIS_GR_SENT_RA:
         statusString += " RA sent"
      if ( state & ISIS_GR_RCVD_RR ) and ( state & ISIS_GR_RCVD_SA ):
         statusString += " RR+SA rcvd"
      if state & ISIS_GR_SENT_CSNP:
         statusString += " CSNP sent"
      statusString += ")"

   return statusString

def getRouterId( systemId ):
   RouterId = Tac.Type( "Mpls::RouterId" )
   rId = RouterId()
   rId.stringValue = systemId
   return rId


class SrInfoDetail( Model ):
   srLabelV4 = Int( help='Adjacency Label for V4', optional=True )
   srLabelV6 = Int( help='Adjacency Label for V6', optional=True )
   srRouterId = Ip4Address( help='SR Router Id' )
   srGbBase = Int( help='SR Base Value' )
   srGbRange = Int( help='SR Range' )

class IsisNeighborDetails( Model ):
   priority = Int( help='ISIS neighbor priority', optional=True )
   advertisedHoldTime = Int( help='Negotiated hold time \
                           (in seconds) between neighbors' )
   stateChanged = Int( help='UTC Time at which the state changed' )
   ip4Address = Ip4Address( help='ISIS Neighbor IPv4 address', optional=True )
   ip6Address = Ip6Address( help='ISIS Neighbor IPv6 address', optional=True )
   ip6GlobalAddress = Ip6Address( help='ISIS Neighbor Global IPv6 address',
                                  optional=True )
   areaIds = List( valueType=str, help='ISIS Neighbor area(s)' )
   bfdIpv4State = Enum( values=( 'up', 'down', 'init', 'adminDown', 'NA' ), 
                        help=bfdStateHelp, optional=True )
   bfdIpv6State = Enum( values=( 'up', 'down', 'init', 'adminDown', 'NA' ), 
                        help=bfdStateHelp, optional=True )
   srEnabled = Bool( help='Shows whether SR enabled on dut' )
   srInfoDetail = Submodel( valueType=SrInfoDetail,
                      help='Information about segment routing', optional=True )
   grSupported = Str( help='Shows whether GR is Supported or Not' )
   grState = Str( help='Shows the state of GR', optional=True )

class IsisNeighborEntry( Model ):
   _instanceName = Str( help='ISIS Instance name' )
   _vrf = Str( help='VRF Name' )
   hostname = Str( help='Hostname', optional=True )
   circuitId = Str( help='Circuit Identifier' )
   interfaceName = Interface( help='ISIS Interface Name' )
   #No 'failed' state
   state = Enum( values=( 'up', 'down', 'init' ), help='ISIS Adjacency state' )
   lastHelloTime = Int( help='UTC timestamp when the last hello was received' )
   snpa = Str( help='Sub Network Point Of Attachment Address', optional=True )
   level = Enum( values=( 'level-1', 'level-2', 'level-1-2' ),
                     help='Level information of the neighbor' )
   details = Submodel( valueType=IsisNeighborDetails, 
                       help='ISIS neighbor details', optional=True )
   routerIdV4 = Ip4Address( help='ISIS neighbor router ID in IP format',
                            optional=True )

   def processData( self, data ):
      from RoutingIsisCli import getHostName
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )
      self.hostname = getHostName( getRouterId( data[ 'system-id' ] ),
                                                     vrfName=self._vrf )
      self.circuitId = data.pop( 'circuit-id' )
      self.level = IsisCliHelper.LEVEL_TYPE_MAP[ data.pop( 'type' ) ]
      self.state = IsisCliHelper.STATE_MAP[ data.pop( 'state' ) ]
      self.interfaceName = kernelIntfFilter( data.pop( 'interface-name' ) )
      self.lastHelloTime = data.pop( 'iih-recv-time' )
      self.snpa = data.pop( 'snpa' )
      if 'router-id-v4' in data:
         self.routerIdV4 = data.pop( 'router-id-v4' )
      else:
         self.routerIdV4 = '0.0.0.0'
      details = IsisNeighborDetails()
      setattr( details, 'priority', data.pop( 'priority', None ) )
      setattr( details, 'ip4Address', data.pop( 'ipv4-ifaddr', None ) )
      setattr( details, 'ip6Address', data.pop( 'ipv6-ifaddr', None ) )
      setattr( details, 'ip6GlobalAddress', data.pop( 'ipv6-global-ifaddr', None ) )
      #Convert to list
      if 'area-ids' in data:
         details.areaIds = data.pop( 'area-ids' ).split()
      else:
         details.areaIds = [ '*NONE*' ]
      if data[ 'bfd-neighbor' ] != 4:
         details.bfdIpv4State = IsisCliHelper.BFD_STATE_MAP[ data.pop( \
               'bfd-neighbor' ) ]
      if data[ 'bfd-neighbor-v6' ] != 4:
         details.bfdIpv6State = IsisCliHelper.BFD_STATE_MAP[ data.pop( \
               'bfd-neighbor-v6' ) ]
      details.srEnabled = struct.unpack( '1B', data['sr-enabled'] )[0] != 0
      srInfoDetail = SrInfoDetail()
      if details.srEnabled:
         srInfoDetail.srGbRange = data.pop( 'srgb-range' )
         srInfoDetail.srGbBase = data.pop( 'srgb-base' )
         if 'sr-label-v4' in data:
            srInfoDetail.srLabelV4 = data.pop( 'sr-label-v4' )
         if 'sr-label-v6' in data:
            srInfoDetail.srLabelV6 = data.pop( 'sr-label-v6' )
         details.srInfoDetail = srInfoDetail
      details.advertisedHoldTime = data.pop( 'hold-time' )
      details.stateChanged = data.pop( 'state-change' )
      grSupported = "Supported" if struct.unpack( '1B',
                              data.pop( 'gr-capable' ) )[0] else "Not Supported" 
      details.grSupported = grSupported
      if grSupported == "Supported":
         grState = struct.unpack( '1B', data.pop( 'gr-state' ) )[0]
         details.grState = ngbGrStateString( grState )
      self.details = details

   def renderDetails( self, ifName ):
      details_indent = "  "
      #Accumulating all areas without any separator
      #In future separator can be used
      areaIds = ''
      for area in self.details.areaIds:
         areaIds += area
      if not areaIds:
         areaIds = "*NONE*"
      print "%s%s: %s" % ( details_indent, "Area Address(es)", areaIds )
      print "%s%s: %s" % ( details_indent, "SNPA", self.snpa )
      print "%s%s: %s" % ( details_indent, "Router ID", self.routerIdV4 )

      print "%s%s: %s" % ( details_indent, "Advertised Hold Time",
                           self.details.advertisedHoldTime )
      print "%s%s: %s" % ( details_indent, "State Changed",
                           IsisCliHelper.elapsedTime( self.details.stateChanged ) + \
                           " ago" )
      #print priority only when LAN neighbor
      if self.details.priority:
         print "%s%s: %s" % ( details_indent, "LAN Priority", 
                              self.details.priority )
      ip4Address = 'none'
      if self.details.ip4Address:
         ip4Address = self.details.ip4Address
      print "%s%s: %s" % ( details_indent, "IPv4 Interface Address", ip4Address )
      ip6Address = 'none'
      if self.details.ip6Address:
         ip6Address = self.details.ip6Address
      print "%s%s: %s" % ( details_indent, "IPv6 Interface Address", ip6Address )
      if self.details.ip6GlobalAddress:
         print "%s%s: %s" % ( details_indent, "Global IPv6 Interface Address",
                              self.details.ip6GlobalAddress )
      print "%s%s: %s" % ( details_indent, "Interface name", ifName )
      print "%s%s: %s %s" % ( details_indent, "Graceful Restart",
                              self.details.grSupported,
                              self.details.grState if self.details.grState else '' )
      #print BFD only when BFD state is UP
      if ( self.details.bfdIpv4State == 'up' ):
         print "%s%s" % ( details_indent, "BFD IPv4 state is Up" )
      if self.details.bfdIpv6State == 'up':
         print "%s%s" % ( details_indent, "BFD IPv6 state is Up" )
      if self.details.srEnabled:
         print "%sSegment Routing Enabled" % ( details_indent )
         print "%s  SRGB Base: %s Range: %s" % ( details_indent,
                                 self.details.srInfoDetail.srGbBase,
                                       self.details.srInfoDetail.srGbRange )
         srV4Lbl = self.details.srInfoDetail.srLabelV4
         srV6Lbl = self.details.srInfoDetail.srLabelV6

         if srV4Lbl or srV6Lbl:
            adjacencyLabel = "%s  Adjacency Label" % ( details_indent )
            if srV4Lbl:
               adjacencyLabel += " IPv4: %s" % ( srV4Lbl )
               adjacencyLabel += "," if srV6Lbl else ""
            if srV6Lbl:
               adjacencyLabel += " IPv6: %s" % ( srV6Lbl ) 
            print adjacencyLabel

   def renderEntry( self, _displayHeader, _detailsPresent, systemId ):
      if _displayHeader:
         print " "
         print ngbHeaderFormatStr % ( "Instance", "VRF", "System Id", "Type", 
                                       "Interface", "SNPA", "State", 
                                       "Hold time", "Circuit Id" )
      level = IsisCliHelper.LEVEL_SHORT_MAP[ self.level ]
      ifName = self.interfaceName.stringValue # pylint: disable-msg=E1101
      state = self.state.upper()
      remHoldTime = self.details.advertisedHoldTime - int( Tac.utcNow() - \
                                                           self.lastHelloTime )
      # BUG243034 - remaining hold time can be negative due to Cli responding 
      # slow in case of high cpu utilization or due to hello packet peeking logic
      # where we keep adjacency up if hello pkts are pending in queue for
      # processing. Display 0 value in those cases as holdTime can't have -ve value 
      remHoldTime = remHoldTime if (remHoldTime > 0) else 0
      if self.hostname:
         print ngbEntryFormatStr % ( self._instanceName, self._vrf, self.hostname, 
                                  level, ifName, self.snpa, state,
                                  remHoldTime, self.circuitId )
      else:
         print ngbEntryFormatStr % ( self._instanceName, self._vrf, systemId, 
                                     level, ifName, self.snpa, state,
                                     remHoldTime, self.circuitId )

      if not  _detailsPresent:
         return
      self.renderDetails( ifName )

class IsisSystemIdNeighbors( Model ):
   adjacencies = List( valueType=IsisNeighborEntry,
                         help="List of neighbors for SystemID" )
   _systemId = Str( help="System Identifier" )

   def getKey( self, data ):
      return data['system-id']

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.adjacencies:
         self._systemId = data[ 'system-id' ]
         neighborData = IsisNeighborEntry()
         neighborData.processData( data )
         self.adjacencies.append( neighborData )
         return ( neighborData, readNext )
      elif self._systemId == self.getKey( data ):
         neighborData = IsisNeighborEntry()
         neighborData.processData( data )
         self.adjacencies.append( neighborData )
         return ( neighborData, readNext )
      else:
         return ( data, False )

   def renderEntry( self, _displayHeader, _detailsPresent, systemId ):
      for neighborData in self.adjacencies :
         neighborData.renderEntry( _displayHeader, _detailsPresent,
                                   systemId )
         _displayHeader = False

class IsisInstanceNeighbors( Model ):
   neighbors = GeneratorDict( valueType=IsisSystemIdNeighbors, keyType=str,
                           help='Dictionary of ISIS neighbors keyed by SystemID' )
   _detailsPresent = Bool( default=False, help='Private attribute to indicate'
                           ' if detailed output was requested' )

   def render( self ):
      #private variable to print header for every instance
      _displayHeader = True
      for systemId, entry in self.neighbors:
         entry.renderEntry( _displayHeader, self._detailsPresent, systemId )
         _displayHeader = False

showNeighborTableModel = IsisCliModelCommon. \
                         generateIsisDictCliModel( IsisInstanceNeighbors )
showNeighborTableVRFsModel = generateVrfCliModel( showNeighborTableModel,
                                    "ISIS instance information for all VRFs" )

class IsisInterfaceDropReasonsMalformedPdus( Model ):
   shortHeaders = Int( help='Header length of the IS-IS PDU received is short',
                       optional=True )
   wrongVersion = Int( help='Version of the PDU header is incompatible with the'
                       ' version we support', optional=True )
   idFieldLengthMismatches = Int( help='System Id Length in the PDU is different '
                                  'from the System Id length supported by the sytem'
                                  ' on which the PDU was received. ', optional=True )

   def processData( self, data ):
      self.shortHeaders = data.pop( 'short_hdrs' )
      self.wrongVersion = data.pop( 'wrong_version' )
      self.idFieldLengthMismatches = data.pop( 'id_field_len_mismatch' )

   def getMioAttrId( self ):
      return \
         PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_MALFORMED_PDUS

class IsisInterfaceDropReasons( Model ):
   badLength = Int( help='Length indicated in the packet header is different from '
                    'the actual length of the packet', optional=True )
   malformedTlv = Int( help='one of the optional TLVs in the PDU could not be'
                       ' validated', optional=True )
   authFailure = Int( help='The PDU could not be authenticated', optional=True )

class IsisInterfaceDropReasonsHellos( IsisInterfaceDropReasons ):
   circuitTypeMismatches = Int( help='The Circuit type indicated in the PDU does not'
                                ' match with that of the interface on which it was '
                                'received For instance, a p2p Hello received on a '
                                'IS-IS Broadcast interface is dropped with this '
                                'reason', optional=True )
   circuitLevelMismatch = Int( help='The circuit level indicated in the PDU does '
                               ' not match with the level on which the IS-IS '
                               'interface receiving this PDU is configured on. ',
                               optional=True )   
   ownSystemIdPdu = Int( help='The System Id on the PDU is similar to that of the '
                         'system on which it is received indicating the system '
                         'received the PDU it has sent (or) some other system is '
                         'using the same system ID. The PDU is dropped.',
                         optional=True )
   unknownCircuitType = Int( help='The circuit type indicated in the PDU is '
                             'unknown. This Happens when a IS-IS Hello has a'
                             ' circuit type other than 1,2 or 3 indicating Level-1'
                             ',Level-2,Level-1-2 respectively', optional=True )
   lengthGreaterThanMTU = Int( help='Length of the PDU is greater than the MTU'
                               ' configured on the interface on which it is '
                               'received.', optional=True )
   badOptionLength = Int( help='Length of one or more of the optional TLVs in the '
                          'PDU is different from that indicated in the Length '
                          'section of that TLV', optional=True )
   noSupportedProtocols = Int( help='Hellos are dropped with this reason. The Hello'
                               ' received does not have NLPID TLV indicating that no'
                               ' protocols are supported by the system sending the'
                               ' Hello', optional=True )
   topoSuppProtocolsMismatch = Int( help='Hellos are dropped with this reaosn. This '
                                    'happens when the MTIDs and NLPIDs in the PDU'
                                    ' mismatch, indicating that there is an '
                                    'inconsistency between the topologies and'
                                    ' protocols supported by the system sending this'
                                    ' Hello.', optional=True )
   protocolAddressesMismatch = Int( help='Hellos are dropped with this reason.'
                                    'The Interface addresses received in the '
                                    'Hello are inconsistent with the NLPIDs received'
                                    ' indicating that there is an inconsistency '
                                    'between the protocols supported and interface '
                                    'addresses configured on the system sending this'
                                    ' Hello', optional=True )
   supportedProtocolsMismatch = Int( help='Hellos are dropped with this reason.'
                                     'The NLPIDs received in the PDU do not match '
                                     'the protocols supported by the system '
                                     'receiving this PDU', optional=True )
   incompatibleAdjAttrs = Int( help='Hellos are dropped with this reason. we drop '
                               'with this reason when there are no usable addresses '
                               'configured on the interface receiving this Hello OR '
                               'the protocols supported by our syatem have changed '
                               'and have become inconsistent with those supported '
                               'by the system sending this Hello.', optional=True )
   badAreaOption = Int( help='Hellos are dropped with this reason. The Area address'
                        ' sub-option in the Areas option is malformed. The areas '
                        ' indicated in the PDU can hence not be used. ',
                        optional=True )
   maxAreaAddressMismatch = Int( help='Hellos are dropped with this reason. The max '
                                 'number of areas indicated in the Hello mismatch '
                                 'with the maximum number of areas supported by the '
                                 'system receiving this hello', optional=True )
   noMatchingAreas = Int( help='Hellos are dropped with this reason. The IS-IS'
                          ' interface on which the Hello was received is in level-1'
                          ' but none of the areas indicated in the Hello match with'
                          ' the areas supported by the system. ', optional=True )
   noRemTimeInRestartTlvWithRA = Int( help='Hellos are dropped with this reason. '
                                      'Remaining time is not indicated in restart '
                                      'TLV with RA.', optional=True )

class IsisInterfaceDropReasonsP2pHellos( IsisInterfaceDropReasonsHellos ):
   noCommonTopology = Int( help='Hellos are dropped with this reason. This happens '
                           'when the system receiving the Hello is in Multi-Topology'
                           ' mode but no MTIDs in the Hello and system are common.',
                           optional=True )
   badAdjStateOption = Int( help='P2P Hellos are dropped with this reason. The '
                            'option length or the Adjacency state indicated in the '
                            'option is incompatible with the state of adjacency in '
                            'which the system receiving this hello currently in. ',
                            optional=True )
   nonMatchingThreeWayHandshake = Int( help='P2P Hellos are dropped with this reason'
                                       '. The neighbour system Id and the Neighbour '
                                       'extended circuit Id indicated in the '
                                       'adjacency state option do not match the '
                                       'system Id and the circuit Id of the system '
                                       'receiving the Hello.', optional=True )

   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitTypeMismatches = data.pop( 'ckt_type_mismatch' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.ownSystemIdPdu = data.pop( 'own_sysid_pdu' )
      self.unknownCircuitType = data.pop( 'unknown_ckt_type' )
      self.lengthGreaterThanMTU = data.pop( 'len_greater_than_mtu' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.noSupportedProtocols = data.pop( 'no_supp_protocols' )
      self.topoSuppProtocolsMismatch = data.pop( 'topo_supp_protocols_mismatch' )
      self.protocolAddressesMismatch = data.pop( 'proto_addr_mismatch' )
      self.supportedProtocolsMismatch = data.pop( 'supp_proto_mismatch' )
      self.incompatibleAdjAttrs = data.pop( 'incomp_adj_attr' )
      self.badAreaOption = data.pop( 'bad_area_opt' )
      self.maxAreaAddressMismatch = data.pop( 'max_area_addr_mismatch' )
      self.noMatchingAreas = data.pop( 'no_matching_areas' )
      self.noRemTimeInRestartTlvWithRA = data.pop( 'no_rem_time_in_rest_tlv_with_'
                                                   'ra' )
      self.noCommonTopology = data.pop( 'no_comm_topology' )
      self.badAdjStateOption = data.pop( 'bad_adj_state_opt' )
      self.nonMatchingThreeWayHandshake = data.pop( 'non_matching_three_way_'
                                                    'handshake' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_IIH_P2P

class IsisInterfaceDropReasonsLanHellos( IsisInterfaceDropReasonsHellos ):
   noSourceMac = Int( help='LAN Hellos are dropped with this reason. The Hello is '
                      'received without any source MAC address. ', optional=True )
   neighbourSysIdChanged = Int( help='LAN Hellos are Dropped with this reason. '
                                'The system Id of the neighbour sending this Hello'
                                'has changed. We hence drop the Hello and delete the'
                                ' adjacency. ', optional=True )
   dupSysIdDetected = Int( help='We already have an adjacency with the same '
                           'system Id', optional=True )
   missingAreas = Int( help='Hellos are dropped with this reason. The IS-IS '
                       'interface receiving this Hello is on Level-1 but there is '
                       'no area TLV in the Hello. We hence drop the Hello. ',
                       optional=True )
   ngbSysIdMismatchInRestartTLVWithRA = Int( help='Hellos are dropped with this '
                                             'reason. The neighbour system Id '
                                             'indicated in restart TLV with RA '
                                             'does not match our own System Id.',
                                             optional=True )
   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitTypeMismatches = data.pop( 'ckt_type_mismatch' )
      self.ownSystemIdPdu = data.pop( 'own_sysid_pdu' )
      self.unknownCircuitType = data.pop( 'unknown_ckt_type' )
      self.lengthGreaterThanMTU = data.pop( 'len_greater_than_mtu' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.noSupportedProtocols = data.pop( 'no_supp_protocols' )
      self.topoSuppProtocolsMismatch = data.pop( 'topo_supp_protocols_mismatch' )
      self.protocolAddressesMismatch = data.pop( 'proto_addr_mismatch' )
      self.supportedProtocolsMismatch = data.pop( 'supp_proto_mismatch' )
      self.incompatibleAdjAttrs = data.pop( 'incomp_adj_attr' )
      self.badAreaOption = data.pop( 'bad_area_opt' )
      self.maxAreaAddressMismatch = data.pop( 'max_area_addr_mismatch' )
      self.noMatchingAreas = data.pop( 'no_matching_areas' )
      self.noRemTimeInRestartTlvWithRA = data.pop( 'no_rem_time_in_rest_tlv_'
                                                   'with_ra' )
      self.noSourceMac = data.pop( 'no_src_mac' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.neighbourSysIdChanged = data.pop( 'ngb_sysid_changed' )
      self.dupSysIdDetected = data.pop( 'dup_sysid_detected' )
      self.missingAreas = data.pop( 'missing_areas' )
      self.ngbSysIdMismatchInRestartTLVWithRA = data.pop( 'ngb_sysid_mismatch_'
                                                          'in_ra' )
class IsisInterfaceDropReasonsL1LanHellos( IsisInterfaceDropReasonsLanHellos ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_IIH_LAN_L1

class IsisInterfaceDropReasonsL2LanHellos( IsisInterfaceDropReasonsLanHellos ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_IIH_LAN_L2

class IsisInterfaceDropReasonsLsps( IsisInterfaceDropReasons ):
   circuitLevelMismatch = Int( help='The circuit level indicated in the PDU does '
                               ' not match with the level on which the IS-IS '
                               'interface receiving this PDU is configured on. ',
                               optional=True )
   nonAdjSource = Int( help='LSPs, CSNPs or PSNPs are dropped with this reason. '
                       'The system receiving this PDU does not have an adjacency '
                       'with the system sending this PDU. ', optional=True )
   parsingFailure = Int( help='LSPs are dropped with this reasons. The options in'
                         ' the LSP could not be parsed properly, indicating that '
                         'the received LSP is corrupted. ', optional=True )
   invalidPurge = Int( help='LSPs are dropped with this reason. The Purge LSP must '
                       ' contain only the Header (or) the Header and Auth option. If'
                       ' it contains anything more or less, it is dropped. ',
                       optional=True )
   badLspLength = Int( help='Length of the LSP is greater than the length of LSP '
                       'buffer of the receiving system. ', optional=True )
   badChecksum = Int( help='The checksum indicated in the LSP is wrong indicating '
                      'that the Lsp is corrupted. ', optional=True )
   ownLspPurge = Int( help='Self-originated LSP of the system is received with a '
                      'zero-life but the remaining life of the LSP as indicated in '
                      'the LSP database is non-zero. The system drops this LSP and '
                      'regenerates the LSP with a newer sequence number. ',
                      optional=True )
   olderPurgeLsp = Int( help=' Purge LSP with a sequence number older than that in'
                          ' the LSP database of the system is received. The LSP is '
                          ' dropped and the latest LSP is flooded. ' )
   ownExpiredLsp = Int( help='LSP is received with the system Id in the LSPID '
                        'matching the system Id of the system receiving it, but the '
                        'LSP is no longer present in the LSP database of the system.'
                        'This LSP might be from the previous incarnation of the '
                        'system. The LSP is dropped and is force expired. ',
                        optional=True )
   differentChecksumLsp = Int( help='LSP received has samilar sequence number as '
                               'that in the LSP database of the system receiving the'
                               ' LSP, but their checksums are different. The LSP is'
                               ' dropped. If the LSP is self generated then it is '
                               'regenerated with a newer sequence number. If it is'
                               ' not self generated then it is force expired. ',
                               optional=True )
   ownNewerSequenceLsp = Int( help='Self generated LSP with a newer sequence number'
                              ' has been received. It is dropped and a new LSP is '
                              'regenerated and flooded', optional=True )
   olderLsp = Int( help='LSP with older sequence number than that of '
                   'the corresponding LSP in the database of the system is received'
                   '. The LSP is dropped and the latest LSP is flooded into the '
                   'circuit on which the LSP was received ', optional=True )

   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.nonAdjSource = data.pop( 'non_adj_src' )
      self.parsingFailure = data.pop( 'parsing_failure' )
      self.invalidPurge = data.pop( 'invalid_purge' )
      self.badLspLength = data.pop( 'bad_lsp_length' )
      self.badChecksum = data.pop( 'bad_cksum' )
      self.ownLspPurge = data.pop( 'own_lsp_purge' )
      self.olderPurgeLsp = data.pop( 'older_purge_lsp' )
      self.ownExpiredLsp = data.pop( 'own_exp_lsp' )
      self.differentChecksumLsp = data.pop( 'diff_cksum_lsp' )
      self.ownNewerSequenceLsp = data.pop( 'newer_seq_lsp' )
      self.olderLsp = data.pop( 'older_lsp' )

class IsisInterfaceDropReasonsL1Lsps( IsisInterfaceDropReasonsLsps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_LSP_L1

class IsisInterfaceDropReasonsL2Lsps( IsisInterfaceDropReasonsLsps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_LSP_L2

class IsisInterfaceDropReasonsSnps( IsisInterfaceDropReasons ):
   circuitLevelMismatch = Int( help='The circuit level indicated in the PDU does '
                               ' not match with the level on which the IS-IS '
                               'interface receiving this PDU is configured on. ',
                               optional=True )
   badOptionLength = Int( help='Length of one or more of the optional TLVs in the '
                          'PDU is different from that indicated in the Length '
                          'section of that TLV', optional=True )
   invalidLspEntryOption = Int( help='CSNPs and PSNPs are dropped with this reason.'
                                ' one or more of the  LSP entry sub-TLVs in the PDU'
                                ' has an invalid length. ', optional=True )
   nonAdjSource = Int( help='LSPs, CSNPs or PSNPs are dropped with this reason. '
                       'The system receiving this PDU does not have an adjacency '
                       'with the system sending this PDU. ', optional=True )

class IsisInterfaceDropReasonsCsnps( IsisInterfaceDropReasonsSnps ):
   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.nonAdjSource = data.pop( 'non_adj_src' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.invalidLspEntryOption = data.pop( 'inv_lsp_entry_opt' )

class IsisInterfaceDropReasonsL1Csnps( IsisInterfaceDropReasonsCsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_CSNP_L1

class IsisInterfaceDropReasonsL2Csnps( IsisInterfaceDropReasonsCsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_CSNP_L2

class IsisInterfaceDropReasonsPsnps( IsisInterfaceDropReasonsSnps ):
   psnpOnNonDisSystem = Int( help='PSNP is received on a circuit of the system of '
                             'type broadcast but system is not a DIS on that '
                             'circuit', optional=True )
   def processData( self, data ):
      self.badLength = data.pop( 'bad_len' )
      self.malformedTlv = data.pop( 'malformed_tlv' )
      self.authFailure = data.pop( 'auth_failure' )
      self.circuitLevelMismatch = data.pop( 'ckt_lev_mismatch' )
      self.badOptionLength = data.pop( 'bad_opt_len' )
      self.invalidLspEntryOption = data.pop( 'inv_lsp_entry_opt' )
      self.psnpOnNonDisSystem = data.pop( 'psnp_on_non_dis_system' )
      self.nonAdjSource = data.pop( 'non_adj_src' )

class IsisInterfaceDropReasonsL1Psnps( IsisInterfaceDropReasonsPsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_PSNP_L1

class IsisInterfaceDropReasonsL2Psnps( IsisInterfaceDropReasonsPsnps ):
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_REASONS_PSNP_L2

class IsisInterfaceDropCountersDetails( Model ):
   malformedPdusDropReasons = Submodel( valueType=
                                        IsisInterfaceDropReasonsMalformedPdus,
                                        help='Reasons which led to malformed PDUs'
                                        'getting dropped and their respective '
                                        'counts', optional=True )
   iihP2pDropReasons = Submodel( valueType=IsisInterfaceDropReasonsP2pHellos,
                                 help='Reasons which led IS-IS P2P Hellos'
                                 'getting dropped and their respective '
                                 'counts', optional=True )
   iihLanL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1LanHellos,
                                 help='Reasons which led to Level-1 IS-IS LAN Hellos'
                                 'getting dropped and their respective '
                                   'counts', optional=True )
   iihLanL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2LanHellos,
                                 help='Reasons which led to Level-2 IS-IS LAN Hellos'
                                 'getting dropped and their respective '
                                   'counts', optional=True )
   lspL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1Lsps,
                              help='Reasons which led to IS-IS Level-1 Link State '
                                'PDUs getting dropped and their respective counts',
                                optional=True )
   lspL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2Lsps,
                              help='Reasons which led to IS-IS Level-2 Link State '
                                'PDUs getting dropped and their respective counts',
                                optional=True )
   csnpL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1Csnps,
                              help='Reasons which led to Level-1 IS-IS Complete '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )
   csnpL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2Csnps,
                              help='Reasons which led to Level-2 IS-IS Complete '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )
   psnpL1DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL1Psnps,
                              help='Reasons which led to IS-IS Level-1 Partial '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )
   psnpL2DropReasons = Submodel( valueType=IsisInterfaceDropReasonsL2Psnps,
                              help='Reasons which led to IS-IS Level-2 Partial '
                                 'Sequence number packets getting dropped and their '
                                 'respective counts',
                                 optional=True )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS_DETAILS

class IsisInterfaceDropCounters( Model ):
   malformedPdus = Int( help='Number of IS-IS PDUs received malformed. The received '
                        'PDU had either a wrong header length, wrong version, or the'
                        ' System Id length in the PDU is not supported by the system'
                        ' received' )
   iihP2p = Int( help='Number of IS-IS P2P Hello Packets Dropped' )
   iihLanL1 = Int( help='Number of Level-1 IS-IS Lan Hello Packets Dropped' )
   iihLanL2 = Int( help='Number of Level-2 IS-IS Lan Hello Packets Dropped' )
   csnpL1 = Int( help='Number of Level-1 IS-IS Complete Sequence Number Packets '
                 'dropped' )
   csnpL2 = Int( help='Number of Level-2 IS-IS Complete Sequence Number Packets '
                 'dropped' )
   psnpL1 = Int( help='Number of Level-1 IS-IS Partial Sequence Number Packets '
                 'dropped' )
   psnpL2 = Int( help='Number of Level-2 IS-IS Partial Sequence Number Packets '
                 'dropped' )
   lspL1 = Int( help='Number of Level-1 IS-IS Link State PDUs '
                 'dropped' )
   lspL2 = Int( help='Number of Level-2 IS-IS Link State PDUs '
                 'dropped' )
   details = Submodel( valueType=IsisInterfaceDropCountersDetails, help='Reasons of'
                       'drop for each PDU with their respective counts',
                       optional=True )

   def processData( self, data ):
      self.malformedPdus = data.pop( 'malformed_pdus' )
      self.iihP2p = data.pop( 'iih_p2p' )
      self.iihLanL1 = data.pop( 'iih_lan_l1' )
      self.iihLanL2 = data.pop( 'iih_lan_l2' )
      self.csnpL1 = data.pop( 'csn_l1' )
      self.csnpL2 = data.pop( 'csn_l2' )
      self.psnpL1 = data.pop( 'psn_l1' )
      self.psnpL2 = data.pop( 'psn_l2' )
      self.lspL1 = data.pop( 'lsp_l1' )
      self.lspL2 = data.pop( 'lsp_l2' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_DROP_COUNTERS

class IsisInterfacePacketCountersData( Model ):
   iihL1 = Int( help='Number of L1 LAN IS-IS Hello Packets' )
   iihL2 = Int( help='Nunber of L2 LAN IS-IS Hello Packets' )
   iihP2p = Int( help='Number of p2p Hello Packets' )
   csnpL1 = Int( help='Number of L1 CSN Packets' )
   csnpL2 = Int( help='Number of L2 CSN Packets' )
   psnpL1 = Int( help='Number of L1 PSN Packets' )
   psnpL2 = Int( help='Number of L2 PSN Packets' )
   lspL1 = Int( help='Number of L1 LSPs' )
   lspL2 = Int( help='Number of L2 LSPs' )

class IsisInterfacePacketCounters( Model ):
   transmitted = Submodel( valueType=IsisInterfacePacketCountersData,
                           help='Transmitted packet counters' )
   received = Submodel( valueType=IsisInterfacePacketCountersData,
                        help='Received packet counters' )

   def processData( self, data ):
      transmittedData = IsisInterfacePacketCountersData()
      transmittedData.iihL1 = data.pop( 'iih-L1-tx' )
      transmittedData.iihL2 = data.pop( 'iih-L2-tx' )
      transmittedData.iihP2p = data.pop( 'iih-p2p-tx' )
      transmittedData.csnpL1 = data.pop( 'csn-L1-tx' )
      transmittedData.csnpL2 = data.pop( 'csn-L2-tx' )
      transmittedData.psnpL1 = data.pop( 'psn-L1-tx' )
      transmittedData.psnpL2 = data.pop( 'psn-L2-tx' )
      transmittedData.lspL1 = data.pop( 'lsp-L1-tx' )
      transmittedData.lspL2 = data.pop( 'lsp-L2-tx' )
      self.transmitted = transmittedData

      receivedData = IsisInterfacePacketCountersData()
      receivedData.iihL1 = data.pop( 'iih-L1-rx' )
      receivedData.iihL2 = data.pop( 'iih-L2-rx' )
      receivedData.iihP2p = data.pop( 'iih-p2p-rx' )
      receivedData.csnpL1 = data.pop( 'csn-L1-rx' )
      receivedData.csnpL2 = data.pop( 'csn-L2-rx' )
      receivedData.psnpL1 = data.pop( 'psn-L1-rx' )
      receivedData.psnpL2 = data.pop( 'psn-L2-rx' )
      receivedData.lspL1 = data.pop( 'lsp-L1-rx' )
      receivedData.lspL2 = data.pop( 'lsp-L2-rx' )
      self.received = receivedData

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_INTERFACE_PKT_COUNTERS
   
class IsisInterfaceCounters( Model ):
   _interfaceName = Interface( help='The IS-IS interface name' )
   packetCounters = Submodel( valueType=IsisInterfacePacketCounters,
                              help='Data pertaining to the packet counters',
                              optional=True )
   dropCounters = Submodel( valueType=IsisInterfaceDropCounters,
                            help='Data pertaining to IS-IS drop counters',
                            optional=True )
   lastResetTime = Int( optional=True, help='UTC timestamp when the counters are' +
                                             ' reset last' )

   def processData( self, data ):
      self._interfaceName = kernelIntfFilter( data.pop( 'interface-name' ) )
      if 'last-reset-time' in data:
         self.lastResetTime = data.pop( 'last-reset-time' )

   def getKey( self, data ):
      return kernelIntfFilter( data['interface-name'] )


class IsisInstanceCounters( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   interfaceCounters = GeneratorDict( valueType=IsisInterfaceCounters,
                                      keyType=Interface,
                                      help='Dictionary of Isis Interface counters'
                                      'keyed by interface name' )
   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      if self._instanceName is None:
         return
      def getReasonsTable( reasonsObj ):
         reasonsDict = vars( reasonsObj )
         reasonsTable = None
         for key, count in reasonsDict.items():
            if not isinstance( getattr( reasonsObj, key ), long ):
               continue
            if count == 0:
               continue
            if not reasonsTable:
               reasonsTable = TableOutput.createTable( dropReasonsHeading )
               f4 = TableOutput.Format( justify='left' )
               f5 = TableOutput.Format( justify='right' )
               reasonsTable.formatColumns( f4, f5 )
            reasonsTable.newRow( IsisCliHelper.PKT_DROP_REASON_MAP[ key ], count )
         return reasonsTable

      packetCountersHeadings = ( "Interface", "IIH L1", "IIH L2", "IIH P2P",
                                 "CSN L1", "CSN L2", "PSN L1", "PSN L2", "LSP L1",
                                 "LSP L2", "Last Reset Time" )
      dropCountersHeadings = ( "Interface", "Malformed Pdus", "IIH L1", "IIH L2",
                               "IIH P2P", "CSN L1", "CSN L2", "PSN L1", "PSN L2",
                               "LSP L1", "LSP L2", "Last Reset Time" )
      dropReasonsHeading = ( "Drop Reason", "Drop Count" )
      transmittedTable = TableOutput.createTable( packetCountersHeadings )
      receivedTable = TableOutput.createTable( packetCountersHeadings )
      dropCountersTable = TableOutput.createTable ( dropCountersHeadings )
      f1 = TableOutput.Format( maxWidth=18, justify='left', wrap=True )
      f2 = TableOutput.Format( maxWidth=10, wrap=True )
      f3 = TableOutput.Format( maxWidth=5, wrap=True )
      dropCountersTable.formatColumns( f1, f2, f3, f3, f3, f3, f3, f3, f3, f3, f3 )
      pktCountersPresent = False
      dropCountersPresent = False
      dropReasonsTablePerIntf = OrderedDict()
      interfaceCounters = dict( self.interfaceCounters )
      for intf in sorted( interfaceCounters ):
         counters = interfaceCounters[ intf ]
         packetCounters = counters.packetCounters
         dropCounters = counters.dropCounters
         lastReset = "Never"
         if counters.lastResetTime is not None:
            lastReset = datetime.fromtimestamp( counters.lastResetTime ).strftime(
                                      '%Y-%m-%d %H:%M:%S' )
         if packetCounters != None:
            pktCountersPresent = True
            transmitted = packetCounters.transmitted
            transmittedTable.newRow( intf, transmitted.iihL1, transmitted.iihL2,
                           transmitted.iihP2p, transmitted.csnpL1,
                           transmitted.csnpL2, transmitted.psnpL1,
                           transmitted.psnpL2,
                           transmitted.lspL1, transmitted.lspL2, lastReset )
            received = packetCounters.received
            receivedTable.newRow( intf, received.iihL1, received.iihL2,
                                  received.iihP2p, received.csnpL1, received.csnpL2,
                                  received.psnpL1, received.psnpL2, received.lspL1,
                                  received.lspL2, lastReset )
         if dropCounters != None:
            dropCountersPresent = True
            dropCountersTable.newRow( intf, dropCounters.malformedPdus,
                                      dropCounters.iihLanL1, dropCounters.iihLanL2,
                                      dropCounters.iihP2p, dropCounters.csnpL1,
                                      dropCounters.csnpL2, dropCounters.psnpL1,
                                      dropCounters.psnpL2, dropCounters.lspL1,
                                      dropCounters.lspL2, lastReset )
            details = dropCounters.details
            if details != None:
               dropReasonTableDict = OrderedDict()
               for dropType in IsisCliHelper.isisDropPduTypes:
                  dropDetailType = IsisCliHelper.PKT_DROP_TYPE_MAP[ dropType ]
                  reasonsTable = getReasonsTable( getattr( details,
                                                           dropDetailType ) )
                  if reasonsTable:
                     dropReasonTableDict[ dropType ] = reasonsTable
               if dropReasonTableDict:
                  dropReasonsTablePerIntf[ intf ] = dropReasonTableDict
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      if pktCountersPresent:
         print "Packet Counters"
         print "----------------"
         print "Transmitted"
         print transmittedTable.output()
         print "Received"
         print receivedTable.output()
      if dropCountersPresent:
         print "Drop Counters"
         print "-------------"
         print dropCountersTable.output()
      if dropReasonsTablePerIntf:
         print "Drop Details"
         print "------------"
         for intf, dropDetailDict in dropReasonsTablePerIntf.items():
            print intf
            print "-----------"
            for key, table in dropDetailDict.items() :
               print key + ':'
               print "----------"
               print table.output()

showCountersInstModel = IsisCliModelCommon. \
                         generateIsisDictCliModel( IsisInstanceCounters )
showCountersVRFsModel = generateVrfCliModel( showCountersInstModel,
                                    "ISIS instance information for all VRFs" )

class IsisSpfLogEntry( Model ):
   startTime = Float( help='SPF start time in UTC' )
   timeTaken = Float( optional=True, help='SPF run duration(ms)' )
   scheduled = Bool( optional=True, help='Scheduled SPF run' ) 
   ipv4Nodes = Int( optional=True,
                    help='Number of IPv4 routers that make up the topology' )
   ipv6Nodes = Int( optional=True,
                    help='Number of IPv6 routers that make up the topology' )
   triggers = Int( help='Number of SPF triggers consumed in this SPF run' )
   reason = Str( help='The first trigger that scheduled the SPF' )
   partial = Bool( optional=True, help='Partial SPF optimization' ) 

   def processData( self, data ):
      if 'duration' in data:
         self.timeTaken = data.pop( ( 'duration' ) ) / 1000.0
         self.ipv4Nodes = data.pop( ( 'nnodes-v4' ) )
         if 'nnodes-v6' in data:
            self.ipv6Nodes = data.pop( ( 'nnodes-v6' ) )
      else:
         self.scheduled = True
      self.startTime = float( data.pop( 'when' ) )
      self.triggers = data.pop( ( 'count' ) )
      self.reason = kernelIntfFilter( data.pop( ( 'reason' ) ) )
      self.partial = ( 0 != ord( data.pop( 'partial', chr( 0 ) ) ) )

   def renderData( self, _mtEnabled ):
      partialStr = "*" if self.partial else " "
      if self.timeTaken:
         startTimeStamp = datetime.fromtimestamp( self.startTime ).strftime(
                                                            '%Y-%m-%d %H:%M:%S.%f')
         if _mtEnabled:
            print spfLogMtEntryFormatStr % (
               partialStr, startTimeStamp, self.timeTaken,
               self.ipv4Nodes, self.ipv6Nodes, self.triggers, self.reason )
         else:
            print spfLogEntryFormatStr % (
               partialStr, startTimeStamp, self.timeTaken,
               self.ipv4Nodes, self.triggers, self.reason )
      else:
         if _mtEnabled:
            print "   %s%-65s %-7d %-17s" % (
               partialStr, "(scheduled)", self.triggers, self.reason )
         else:
            print "   %s%-53s %-7d %-17s" % (
               partialStr, "(scheduled)", self.triggers, self.reason )


class IsisLevelSpfLogModel( Model ):
   _mtEnabled = Bool( default=False, help='multi-topology enabled or not' )
   spfLogs = GeneratorList( valueType=IsisSpfLogEntry,
             help = 'ISIS SPF log entries' )

   def getKey( self, data ):
      return int( data[ 'level-num' ] )

   def processData( self, data ):
      self._mtEnabled = ord( data.pop( 'mt' ) ) != 0

   def renderData( self, level ):
      print "  IS-IS SPF Log level-%s" % ( level )
      print "  * - Partial SPF"
      if self._mtEnabled:
         print spfLogMtHeaderFormatStr % ( "Start time", "Duration(msec)",
               "Nodes(v4)", "Nodes(v6)", "Count", "Reason" )
      else:
         print spfLogHeaderFormatStr % ( "Start time", "Duration(msec)",
               "Nodes", "Count", "Reason" )
      for logEntry in self.spfLogs:
         logEntry.renderData( self._mtEnabled )


class IsisSpfLogTableModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='Vrf Name' )
   levels = GeneratorDict( keyType=int, valueType=IsisLevelSpfLogModel,
           help='Dictionary of SPF logs keyed by level' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      if self._instanceName is None:
         return
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for level, entry in self.levels:
         entry.renderData( level )  

spfLogTableModel = IsisCliModelCommon. \
                   generateIsisDictCliModel( IsisSpfLogTableModel )
spfLogTableVRFsModel = generateVrfCliModel( spfLogTableModel,
                       "IS-IS instance information for all VRFs" )

# Models for 'show isis network topology'
class IsisTopologyEntryListModel( Model ):
   metric = Int( help='IS-IS metric value' )
   nexthopSystemId = Str( help='Next Hop' )
   interface = Interface( help='IS-IS interface name', default='' )
   snpa = Str( help='Subnetwork Point of Attachment of Neighbor' )
   nexthopHostname = Str( help='Hostname', optional=True )
   iaMetric = Int( help='IS-IS inter-area metric value' )
   igpShortcutTunnel = Submodel( valueType=TunnelInfo,
                                 help="IGP Shortcut tunnel", optional=True )

   def processData( self, data ):
      self.metric = data.get( 'metric' )
      self.nexthopSystemId = data.pop( 'next-hop' )
      ifName = data.pop( 'interface-name' )
      if ifName != 'N.A.':
         self.interface = kernelIntfFilter( ifName )
      self.snpa = data.pop( 'snpa' )
      self.nexthopHostname = data.pop( 'nh-hostname' )
      if self.nexthopHostname == self.nexthopSystemId:
         self.nexthopHostname = None
      self.iaMetric = data.get( 'intra-area-metric' )
      tunnelId = data.get( 'tunnelid' )
      if tunnelId:
         tacTunnelId = Tac.Value( "Tunnel::TunnelTable::TunnelId", int( tunnelId ) )
         tunInfo = TunnelInfo( tunnelIndex=tacTunnelId.tunnelIndex(),
                               tunnelType=tacTunnelId.typeCliStr(),
                               tunnelVias=None )
         self.igpShortcutTunnel = tunInfo

class IsisTopologyNextHopsEntryModel( Model ):
   vias = List( valueType=IsisTopologyEntryListModel,
                            help='Isis nexthops list' )
   destinationHostname = Str( help='Hostname', optional=True )
   _systemId = Str( help='SystemID' )
   _metric = Int( help='Isis Metric Value' )
   _iaMetric = Int( help='Isis Intra-Area Metric Value' )

   def getKey( self, data ):
      return data.get( 'system-id', self._systemId )
   
   def overrideHierarchy( self, data ):
      readNext = True
      if not self.vias:
         self._systemId = data[ 'system-id' ]
         self.destinationHostname = data[ 'sysid-hostname' ]
         if self.destinationHostname == self._systemId:
            self.destinationHostname = None
         self._metric = data[ 'metric' ]
         self._iaMetric = data[ 'intra-area-metric' ]
         nexthopData = IsisTopologyEntryListModel()
         nexthopData.processData( data )
         self.vias.append( nexthopData )
         return (nexthopData, readNext)
      elif self._systemId == self.getKey( data ):
         data[ 'metric' ] = self._metric
         data[ 'intra-area-metric' ] = self._iaMetric
         nexthopData = IsisTopologyEntryListModel()
         nexthopData.processData( data )
         self.vias.append( nexthopData )
         return ( nexthopData, readNext )
      else:
         return ( data, False )

   def renderEntry( self, systemId ):
      for model in self.vias:
         metricValue = " " if systemId == " " else str( model.metric )
         iaMetricValue = " " if systemId == " " else str( model.iaMetric )
         if model.nexthopHostname:
            model.nexthopSystemId = model.nexthopHostname
         if systemId != " " and self.destinationHostname:
            systemId = self.destinationHostname
         if model.igpShortcutTunnel:
            tunInfo = model.igpShortcutTunnel
            ifName = "%s tunnel index %d" % ( tunInfo.tunnelType,
                                              tunInfo.tunnelIndex )
         else:
            ifName = model.interface.stringValue
         if ifName == '':
            ifName = 'N.A.'
         print ( "    %-16s %-8s %-9s %-16s %-24s %-17s"
                 % ( systemId, metricValue, iaMetricValue, model.nexthopSystemId,
                     ifName, model.snpa ) )
         systemId = " "


class IsisTopologyNextHopModel( Model ):
   nexthops = GeneratorDict( keyType=str, 
                                valueType=IsisTopologyNextHopsEntryModel,
                                help='Dict of next-hops keyed by system-id' )
   _addrFamily = Int( help='Address Family' )
   _mt = Bool( default=False, help='MultiTopology Enabled or not' )
   _level = Int( help='level entry of ipv4/ipv6' )

   def processData( self, data ):
      self._addrFamily = data[ 'address-family' ]
      self._mt = ord( data[ 'mt' ] ) != 0
      self._level = int( data[ 'level' ] )
   
   def getKey( self, data ):
      # https://tools.ietf.org/html/rfc5120#section-7.5
      # we will return topology-id as key for address-family
      # In case of Non-Mt topology-id returned will be "0"
      # In case of Mt 
      #         if address-family is 'ipv6' return "2"
      #         else if address-family is 'ipv4' return "0:
      if ord(data[ 'mt' ]) != 0 and data[ 'address-family' ] == 2:
         return "2"
      return "0"

   def renderEntry( self ):
      ipstr = ""
      if self._mt:
         ipstr = 'IPv4 ' if self._addrFamily == 1 else 'IPv6 '

      print "  IS-IS " + ipstr + "paths to level-" + str(self._level) + " routers"
      print "    %-16s %-8s %-9s %-16s %-24s %-17s" % ( "System Id", "Metric",
                                                        "IA Metric", "Next-Hop",
                                                        "Interface", "SNPA" )
      for systemId, model in self.nexthops:
         model.renderEntry( systemId )

class IsisTopologyEntryModel( Model ):
   topologies = GeneratorDict( keyType=str, valueType=IsisTopologyNextHopModel,
                              help='Dict of address-families keyed by topology-id' )
   
   def getKey( self, data ):
      return int( data[ 'level' ] )

   def renderEntry( self ):
      for _, model in self.topologies:
         model.renderEntry()

class IsisTopologyModel( Model ):
   _vrf = Str( help='Vrf Name' )
   _instanceName = Str( help='Instance Name' )
   levels = GeneratorDict( keyType=int, valueType=IsisTopologyEntryModel,
                          help='A dict of isis Topology keyed by isis level' )
   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for _, model in self.levels:
         model.renderEntry()

isisTopologyModel = IsisCliModelCommon . \
                    generateIsisDictCliModel( IsisTopologyModel )
isisTopologyVRFsModel = generateVrfCliModel( isisTopologyModel,
                        "Topology information for all VRFs" )

class IsisHostnameEntryModel( Model ):
   hostname = Str( help='Hostname' )
   level = Enum( values=IsisCliHelper.LEVEL_SHORT_MAP.values(),
                 help='IS type' )

   def renderEntry( self, systemId ):
      print "%-6s %-19s %-s" % ( self.level, systemId, self.hostname )
      
   def getKey( self, data ):
      return str( data[ 'system-id' ] )

class IsisHostnameInstanceModel( Model ):
   _vrf = Str( help='Vrf Name' )
   _instanceName = Str( help='Instance Name' )
   systemIds = GeneratorDict( keyType=str, valueType=IsisHostnameEntryModel, 
                             help='Dictionary of IS-IS hostnames keyed by SystemID' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      print "Level  System ID           Hostname"
      for systemId, model in self.systemIds:
         model.renderEntry( systemId )


isisHostnameModel = IsisCliModelCommon. \
                     generateIsisDictCliModel( IsisHostnameInstanceModel )
isisHostnameVRFModel = generateVrfCliModel( isisHostnameModel,
                     "Hostname information for all VRFs" )

class IsisMplsLdpSyncInterfaceModel( Model ):
   _instanceName = Str( help='Instance Name' )
   ldpEnabled = Bool( help='LDP Enabled' )
   syncRequired = Bool( help='LDP sync required' )
   syncAchieved = Bool( help='LDP sync achieved', optional=True )
   igpNotifyDelay = Int( help='IGP notify delay in seconds. Immediate if zero',
                         optional=True )
   holddownTime = Int( help='Holddown time in seconds. '
                            'Hold until established if zero',
                       optional=True )

   def toSecondsStr( self, seconds, delay=False ):
      if seconds == 0:
         return ( "immediate" if delay else "infinite" )
      return "%d second%s" % ( seconds, "" if seconds == 1 else "s" )

   def renderEntry( self, instanceName, interface ):
      print "Interface: " + interface + "; IS-IS instance: " + instanceName
      ldpEnabled = "No"
      required = "No"
      achieved = "No"
      if self.ldpEnabled:
         ldpEnabled = "Yes"
      if self.syncRequired:
         required = "Yes"
      if self.syncAchieved is None:
         achieved = "Not applicable"
      elif self.syncAchieved:
         achieved = "Yes"

      print " Ldp Enabled: " + ldpEnabled
      print " SYNC information:"
      print "   Required: %s, Achieved: %s" % ( required, achieved )
      if self.igpNotifyDelay is not None:
         print "   IGP Notify Delay: %s" % \
            self.toSecondsStr( self.igpNotifyDelay, delay=True ) 
      if self.holddownTime is not None:
         print "   Holddown time: %s" % self.toSecondsStr( self.holddownTime )
      print ""

class IsisMplsLdpSyncModel( Model ):
   _vrf = Str( 'Vrf Name' )
   _instanceName = Str( 'Instance Name' )
   interfaces = Dict( keyType=Interface, valueType=IsisMplsLdpSyncInterfaceModel,
                      help='MPLS LDP sync information keyed by interface name' )

   def render( self ):
      for interfaceName, interface in self.interfaces.items():
         interface.renderEntry( self._instanceName, interfaceName )

isisMplsLdpSyncModel = IsisCliModelCommon.\
                           generateIsisDictCliModel( IsisMplsLdpSyncModel )
isisMplsLdpSyncVRFsModel = generateVrfCliModel( isisMplsLdpSyncModel,
                                        "MPLS LDP sync information for all vrfs")
# CAPI model for 'show isis lsp purges'

PURGE_GEN_REASON_MAP = {
   "lspLifetimeExpired" : "LSP lifetime expired",
   "receivedOldIncarnation" : "received old incarnation of own LSP",
   "differentChecksum" : "LSP has a different checksum",
   "disResigned" : "resigned as DIS",
   "noContents" : "no contents",
   "userClearedLsp" : "LSP cleared by user",
}

class IsisLspPurgesEntryModel( Model ):
   logTime = Float( help='Log time stamp' )
   lspId = Str( help='IS-IS LSPID of purge' )
   _vrf = Str( help='VRF Name' )
   originatorSystemId = Str( help='Purge Originator or receiver SystemID',
                             optional=True )
   originatorHostname = Str( help='Purge Originator or receiver Dynamic Hostname',
                             optional=True )
   upstreamNgbSystemId = Str( help='Neighbour SystemID who send purge',
                              optional=True )
   upstreamNgbHostname = Str( help='Neighbour Hostname who send purge', 
                              optional=True )
   reason = Enum( values=PURGE_GEN_REASON_MAP.keys(),
                  help='Reason for purge generation', optional=True )
   lspIdWithHostname = Str( help='LSP generator hostname', optional=True )
   
   def processData( self, data ):
      from RoutingIsisCli import getHostName
      self.logTime = float( data.pop( 'time' ) )
      self._vrf = data.pop( 'vrf-name' )
      self.lspId = data.pop( 'lspid' )
      self.lspIdWithHostname = getHostName( getRouterId( self.lspId[ :14 ] ), 
                                            self._vrf )
      if 'originatorSystemId' in data:
         self.originatorSystemId = data.pop( 'originatorSystemId' )
         if self.originatorSystemId != "self":
            self.originatorHostname = getHostName( 
                              getRouterId( self.originatorSystemId ), self._vrf ) 
      if 'upstreamNgbSystemId' in data:
         self.upstreamNgbSystemId = data.pop( 'upstreamNgbSystemId' )
         self.upstreamNgbHostname = getHostName( 
                              getRouterId( self.upstreamNgbSystemId ), self._vrf )
      if 'reason' in data:
         self.reason = data.pop( 'reason' )
             
   def renderData( self ):
      logTimestampValue = datetime.fromtimestamp( self.logTime ).strftime( 
                                                      '%Y-%m-%d %H:%M:%S.%f' )
      originator = self.originatorSystemId if self.originatorHostname is None \
            else self.originatorHostname
      upstreamNgb = self.upstreamNgbSystemId if self.upstreamNgbHostname is None \
            else self.upstreamNgbHostname
      helperLspId = ( self.lspIdWithHostname + '.00-00' ) if self.lspIdWithHostname \
            is not None else self.lspId
      output = "%s Purge of %s received"
      if self.originatorSystemId != 'self' and upstreamNgb is None:
         print ( output + ", originated by %s" ) % ( logTimestampValue ,helperLspId,
                                                     originator )
      elif self.originatorSystemId != 'self' and \
            upstreamNgb is not None:
         print ( output + ", flooded by %s towards %s" ) % ( logTimestampValue,
                                             helperLspId, upstreamNgb, originator )
      elif self.originatorSystemId == 'self' and upstreamNgb is not None:
         print ( output + ", flooded by %s towards self" ) % (
                  logTimestampValue, helperLspId, upstreamNgb )
      else:
         print "%s Purge of %s generated by self, %s" % (
            logTimestampValue, helperLspId, 
            PURGE_GEN_REASON_MAP.get( self.reason, "unknown" ) )

class IsisLevelLspPurgesModel( Model ):
   purgeLogs = GeneratorList( valueType=IsisLspPurgesEntryModel, 
               help='List of IS-IS purge entries' )
   
   def getKey( self, data ):
      return int( data[ 'level-num' ] )

   def renderData( self, level ):
      print "IS-IS Level-%s" % ( level )
      for purgeEntry in self.purgeLogs:
         purgeEntry.renderData()

class IsisLspPurgesModel( Model ):
   _instanceName = Str( help='IS-IS Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int, valueType=IsisLevelLspPurgesModel, 
                           help='Dictionary of purges keyed by level' )
   
   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )

   def render( self ):
      if self._instanceName is None:
         return
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for level, entry in self.levels:
         entry.renderData( level )
      
isisLspPurgesModel = IsisCliModelCommon.generateIsisDictCliModel(
                  IsisLspPurgesModel )
isisLspPurgesVRFsModel = generateVrfCliModel( isisLspPurgesModel,
                           "IS-IS instance information for all VRFs" )

# CAPI model for 'show isis graceful-restart'
class IsisGrNgbListModel( Model ):
   level = Enum( values=IsisCliHelper.LEVEL_SHORT_MAP.values(), help='IS-IS level' )
   interface = Interface( help='IS-IS interface name' )
   gracefulRestartCapable = Bool( help='Graceful-Restart Capable' )
   status = Enum( values=IsisCliHelper.GRACEFUL_RESTART_STATUS_MAP.keys(),
                           help='Graceful-Restart State' )

   def ngbGrState( self, status, gr_capable ):
      if not gr_capable:
         return "na"
      if (status) & ( ISIS_GR_RCVD_SA ):
         return "starting"
      elif (status) & ( ISIS_GR_RCVD_RR ):
         return "reStarting"
      else:
         return "running"

   def getLevel( self, levelType ):
      if levelType == "Level 1 IS":
         return "L1"
      elif levelType == "Level 2 IS":
         return "L2"
      elif levelType == "Level 1 and 2 IS":
         return "L1L2"
      return levelType

   def processData( self, data ):
      self.level = self.getLevel( data.pop( 'type' ) )
      self.interface = kernelIntfFilter( data.pop( 'interface-name' ) )
      grC = struct.unpack( '1B', data.pop( 'gr-capable' ) )[ 0 ]
      grS = struct.unpack( '1B', data.pop( 'gr-state' ) )[ 0 ]

      self.gracefulRestartCapable = bool( grC )
      self.status = self.ngbGrState( grS, grC )

class IsisGracefulRestartNgbEntryModel( Model ):
   hostname = Str( 'Hostname of Neighbor', optional=True )
   adjacencies = List( valueType=IsisGrNgbListModel,
                       help='List of adjacency data for a neighbor' )
   _systemId = Str( 'SystemId' )


   def getKey( self, data ):
      return data[ 'system-id' ]

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.adjacencies:
         self._systemId = data[ 'system-id' ]
         self.hostname = data.get( 'hostname', None )
         entryData = IsisGrNgbListModel()
         entryData.processData( data )
         self.adjacencies.append( entryData )
         return ( entryData, readNext )
      elif self._systemId == self.getKey( data ):
         entryData = IsisGrNgbListModel()
         entryData.processData( data )
         self.adjacencies.append( entryData )
         return ( entryData, readNext )
      else:
         return ( data, False )

   def renderEntry( self, systemId ):
      if self.hostname is not None:
         systemId = self.hostname
      for model in self.adjacencies:
         grC = IsisCliHelper.YES_NO_MAP[ bool( model.gracefulRestartCapable ) ] 
         status = IsisCliHelper.GRACEFUL_RESTART_STATUS_MAP[ model.status ]
         print grNgbHeaderFormatStr % ( systemId, model.level,
                                        model.interface.stringValue,
                                        grC, status )

class IsisGracefulRestartModel( Model ):
   _vrf = Str( help='vrf-name' )
   _instanceName = Str( help='IS-IS instance name' )
   systemId = Str( help='SystemId' )
   gracefulRestart = Enum( values=( IsisCliHelper.ENABLED_DISABLED_MAP.keys() ), 
                           help='Graceful Restart' )
   gracefulRestartHelper = Enum( values=( IsisCliHelper.ENABLED_DISABLED_MAP.keys()),
                                 help='Graceful Restart Helper' )
   state = Enum( values=( IsisCliHelper.GRACEFUL_RESTART_STATE_MAP.values() ),
                          help='Graceful Restart State' )
   t1Timer = Int( help='T1 timer - Time after which an unacknowledged start or ' +
                       'restart attempt will be repeated (in seconds)' )
   t3Timer = Int( help='T3 timer - Time after which the router will declare that ' +
                       'it has failed to achieve database synchronization during ' +
                       'restart (in seconds)',
                       optional=True)
   t2ConfigLevel1 = Int( help='T2 timer - Maximum time to wait for LSP database ' +
                              'synchronization for level-1 (in seconds) ' )
   t2Level1 = Int( help='Remaining T2 for level-1 (in seconds)', optional=True )
   t2ConfigLevel2 = Int( help='T2 timer - Maximum time to wait for LSP database ' +
                              'synchronization for level-2 (in seconds) ' )
   t2Level2 = Int( help='Remaining T2 for level-2 (in seconds)', optional=True )

   grNeighbors = GeneratorDict( keyType=str,
                                 valueType=IsisGracefulRestartNgbEntryModel,
                                 help='A Dict of GracefulRestart Neighbors ' +  
                                      'entries keyed by systemID' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
      self.systemId = data.pop( 'system-id' )
      self.gracefulRestart = "enabled" if struct.unpack( '1B',
                               data.pop( 'graceful-restart' ) )[ 0 ] else "disabled"
      self.gracefulRestartHelper = "enabled" if struct.unpack( '1B',
                                      data.pop( 'gr-helper') )[ 0 ] else "disabled"
      self.state = data.pop( 'gr-state' )
      self.t1Timer = 3

      self.t2ConfigLevel1 = data.pop( 'gr-cfg-t2-l1' )
      if 'gr-t2-l1' in data:
         self.t2Level1 = data.pop( 'gr-t2-l1' )

      self.t2ConfigLevel2 = data.pop( 'gr-cfg-t2-l2' )
      if 'gr-t2-l2' in data:
         self.t2Level2 = data.pop( 'gr-t2-l2' )

      if 'gr-t3' in data:
         self.t3Timer = data.pop( 'gr-t3' )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      print "  System ID: %s" % self.systemId
      print "  Graceful Restart: %s, Graceful Restart Helper: %s" % \
               ( IsisCliHelper.ENABLED_DISABLED_MAP[ self.gracefulRestart ], 
                 IsisCliHelper.ENABLED_DISABLED_MAP[ self.gracefulRestartHelper ] )
      print "  State: %s" % self.state
      print "  T1 : 3s"
      t2l1 = str( self.t2ConfigLevel1 ) + "s/"
      if self.t2Level1 is not None:
         t2l1 += str( self.t2Level1 ) + "s remaining"
      else:
         t2l1 += "not running"
      t2l2 = str( self.t2ConfigLevel2 ) + "s/"
      if self.t2Level2 is not None:
         t2l2 += str( self.t2Level2 ) + "s remaining"
      else:
         t2l2 += "not running"
      t3 = "not running"
      if self.t3Timer is not None:
         t3 = str( self.t3Timer ) + "s"
      print "  T2 (level-1) : %s" % t2l1
      print "  T2 (level-2) : %s" % t2l2
      print "  T3 : %s" % t3
      print ""
      print grNgbHeaderFormatStr % ( "System ID", "Type", "Interface",
                                     "Restart Capable", "Status" )

      for systemId, model in self.grNeighbors:
         model.renderEntry( systemId )

isisGracefulRestartModel = IsisCliModelCommon. \
                           generateIsisDictCliModel( IsisGracefulRestartModel )
isisGracefulRestartVRFsModel = generateVrfCliModel( isisGracefulRestartModel,
                              "Graceful Restart information for all VRFs" )

class IsisUloopDelayedV4RouteModel( Model ):
   v4DestAddr = Ip4Address( help="IPv4 destination address of delayed route" )
   maskLength = Int( help="Mask Length" )

   def processData( self, data ):
      self.v4DestAddr = data.pop( 'ipv4-dest' )
      self.maskLength = data.pop( 'masklen' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ULOOP_DELAYED_V4_ROUTES
   
   def render( self ):
      print "        %s/%s" % ( self.v4DestAddr, self.maskLength )

class IsisUloopDelayedV6RouteModel( Model ):
   v6DestAddr = Ip6Address( help="IPv6 destination address of delayed route" )
   maskLength = Int( help="Mask Length" )

   def processData( self, data ):
      self.v6DestAddr = data.pop( 'ipv6-dest' )
      self.maskLength = data.pop( 'masklen' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_ULOOP_DELAYED_V6_ROUTES
   
   def render( self ):
      print "        %s/%s" % ( self.v6DestAddr, self.maskLength )

class IsisUloopSmModel( Model ):
   level = Int( help='Level' )
   lastAttempt = Bool( default=False, help='Last attempt' )
   state = Enum( values=IsisCliHelper.ULOOP_SM_STATE_RENDER_MAP.keys(),
                 help='State of Micro-loop prevention attempt' )
   cause = Enum( values=IsisCliHelper.ULOOP_SM_CAUSE_RENDER_MAP.keys(),
                 help='Reason for Micro-loop prevention attempt' )
   result = Enum( values=IsisCliHelper.ULOOP_SM_RESULT_RENDER_MAP.keys(),
                  help='Result of Micro-loop prevention attempt' )
   intfName = Interface( help='Protected interface' )
   intfIndex = Int( help='Interface index' )
   protectionV4 = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv4' )
   protectionV6 = Enum( values=IsisCliHelper.TILFA_PROTECTION_MAP.values(),
                        help='TI-LFA protection mode for IPv6' )
   delayV4 = Int( help='IPv4 local convergence delay in msecs' )
   delayV6 = Int( help='IPv6 local convergence delay in msecs' )
   routesDelayedV4 = Int( help='Number of IPv4 routes delayed' )
   routesDelayedV6 = Int( help='Number of IPv6 routes delayed' )
   delayTimerStartedV4 = Int( help='Time delay timer started for IPv4 routes' )
   delayTimerStoppedV4 = Int( help='Time delay timer stopped for IPv4 routes' )
   delayTimeRemainingV4 = Int( help='Delay time remaining for IPv4 routes' )
   delayTimerStartedV6 = Int( help='Time delay timer started for IPv6 routes' )
   delayTimerStoppedV6 = Int( help='Time delay timer stopped for IPv6 routes' )
   delayTimeRemainingV6 = Int( help='Delay time remaining for IPv6 routes' )
   adjacencyDownTimestamp = Int( help='Link or BFD failure timestamp' )

   delayedRoutesV4 = GeneratorList( valueType=IsisUloopDelayedV4RouteModel,
                                    help='List of IPv4 routes delayed',
                                    optional=True )
   delayedRoutesV6 = GeneratorList( valueType=IsisUloopDelayedV6RouteModel,
                                    help='List of IPv6 routes delayed',
                                    optional=True )

   def processData( self, data ):
      self.level = data.pop( 'level' )
      self.lastAttempt = ord( data.pop( 'last-attempt' ) ) != 0
      self.state = data.pop( 'state' )
      if self.state == "delayNotRunning":
         return
      self.cause = data.pop( 'cause' )
      self.result = data.pop( 'result' )
      self.intfName = kernelIntfFilter( data.pop( 'ifl-name' ) )
      self.intfIndex = data.pop( 'ifl-index' )
      self.protectionV4 = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v4' ) ) ]
      self.protectionV6 = IsisCliHelper.TILFA_PROTECTION_MAP[
         ord( data.pop( 'protection-mode-v6' ) ) ]
      self.delayV4 =  data.pop( 'delay-v4' )
      self.delayV6 =  data.pop( 'delay-v6' )
      self.routesDelayedV4 = data.pop( 'routes-delayed-v4' )
      self.routesDelayedV6 = data.pop( 'routes-delayed-v6' )
      self.delayTimerStartedV4 = data.pop( 'delay-timer-started-v4' )
      self.delayTimerStoppedV4 = data.pop( 'delay-timer-stopped-v4' )
      self.delayTimeRemainingV4 = data.pop( 'delay-time-remaining-v4' )
      self.delayTimerStartedV6 = data.pop( 'delay-timer-started-v6' )
      self.adjacencyDownTimestamp = data.pop( 'adjacency-down-timestamp' )
      self.delayTimerStoppedV6 = data.pop( 'delay-timer-stopped-v6' )
      self.delayTimeRemainingV6 = data.pop( 'delay-time-remaining-v6' )

   def renderEntry( self, detail ):
      if self.state == "delayNotRunning":
         return
      stateStr = IsisCliHelper.ULOOP_SM_STATE_RENDER_MAP.get( self.state, "unknown" )
      print ""
      print "  Level %d" % self.level,
      if self.lastAttempt:
         print "last attempt due to %s on %s," % (
               IsisCliHelper.ULOOP_SM_CAUSE_RENDER_MAP.get( self.cause, "unknown" ),
               self.intfName.stringValue ),
         if self.result == "delaySucceeded":
            print IsisCliHelper.ULOOP_SM_RESULT_RENDER_MAP[ self.result ]
         else:
            # all abort results reported simply as "Failed"
            print "Failed"
      else:
         print "in progress due to %s on %s" % (
               IsisCliHelper.ULOOP_SM_CAUSE_RENDER_MAP.get( self.cause, "unknown" ),
               self.intfName.stringValue )

      print "    Adjacency failure at %s" % datetime.fromtimestamp( 
               self.adjacencyDownTimestamp ).strftime( '%Y-%m-%d %H:%M:%S' )

      if self.protectionV4 == "disabled":
         print "    TI-LFA protection is disabled for IPv4"
      else:
         print "    TI-LFA %s protection is enabled for IPv4"  % self.protectionV4
         if not self.lastAttempt and self.state == 'delaySpfScheduled':
            print "    %s" % stateStr
         elif self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ]:
            print "    IPv4 Routes delayed: %d"  % self.routesDelayedV4
         if self.delayTimerStartedV4:
            timeStamp = datetime.fromtimestamp( self.delayTimerStartedV4 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print "      Delay timer started at:", timeStamp
         if self.delayTimeRemainingV4:
            print "      Delay timer expires in %d secs" % self.delayTimeRemainingV4
         if self.delayTimerStoppedV4:
            timeStamp = datetime.fromtimestamp( self.delayTimerStoppedV4 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print "      Delay timer stopped at:", timeStamp
         if ( detail and
              self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ] and 
              self.routesDelayedV4 > 0 ):
            print "      Delayed routes:"
            for route in self.delayedRoutesV4:
               route.render()

      if self.protectionV6 == "disabled":
         print "    TI-LFA protection is disabled for IPv6"
      else:
         print "    TI-LFA %s protection is enabled for IPv6"  % self.protectionV6
         if not self.lastAttempt and self.state == 'delaySpfScheduled':
            print "    %s" % stateStr
         elif self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ]:
            print "    IPv6 Routes delayed: %d"  % self.routesDelayedV6
         if self.delayTimerStartedV6:
            timeStamp = datetime.fromtimestamp( self.delayTimerStartedV6 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print "      Delay timer started at:", timeStamp
         if self.delayTimeRemainingV6:
            print "      Delay timer expires in %d secs" % self.delayTimeRemainingV6
         if self.delayTimerStoppedV6:
            timeStamp = datetime.fromtimestamp( self.delayTimerStoppedV6 ).strftime(
                                                   '%Y-%m-%d %H:%M:%S')
            print "      Delay timer stopped at:", timeStamp
         if ( detail and
              self.state in [ 'delayRoutes', 'delayRoutesNewSpf' ] and 
              self.routesDelayedV6 > 0 ):
            print "      Delayed routes:"
            for route in self.delayedRoutesV6:
               route.render()

class IsisUloopModel( Model ):
   _vrf = Str( help='vrf-name' )
   _instanceName = Str( help='IS-IS instance name' )
   _printDetails = Bool( default=False, help='Display details' )
   systemId = Str( help='System ID' )
   cfgDelayV4 = Int( help='IPv4 local convergence delay' )
   cfgDelayV6 = Int( help='IPv6 local convergence delay' )
   delayAttemptsL1 = Int( help='Number of attempts at level-1' )
   delayAbortsL1 = Int( help='Number of failures at level-1' )
   delayAttemptsL2 = Int( help='Number of attempts at level-2' )
   delayAbortsL2 = Int( help='Number of failures at level-2' )

   uloopSms = GeneratorList( valueType=IsisUloopSmModel,
                             help='List of Micro-loop SMs' )

   def processData( self, data ):
      self._vrf = data.pop( 'vrf-name' )
      self._instanceName = data.pop( 'instance-name', None )
      self.systemId = data.pop( 'system-id' )
      self.cfgDelayV4 = data.pop( 'cfg-delay-v4' )
      self.cfgDelayV6 = data.pop( 'cfg-delay-v6' )
      self.delayAttemptsL1 = data.pop( 'delay-attempts-l1' )
      self.delayAbortsL1 = data.pop( 'delay-aborts-l1' )
      self.delayAttemptsL2 = data.pop( 'delay-attempts-l2' )
      self.delayAbortsL2 = data.pop( 'delay-aborts-l2' )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      print "  System ID: %s" % self.systemId
      if self.cfgDelayV4 == 0 and self.cfgDelayV6 == 0:
         print "  Micro-loop local convergence delay is not configured"
      else:
         if self.cfgDelayV4:
            print "  IPv4 local convergence delay configured, %d msecs" \
                  % self.cfgDelayV4
         if self.cfgDelayV6:
            print "  IPv6 local convergence delay configured, %d msecs" \
                  % self.cfgDelayV6
         print "  Level 1 attempts %d, failures %d" % (
               self.delayAttemptsL1, self.delayAbortsL1 )
         print "  Level 2 attempts %d, failures %d" % (
               self.delayAttemptsL2, self.delayAbortsL2 ) 

      for entry in self.uloopSms:
         entry.renderEntry( self._printDetails )

isisUloopModel = IsisCliModelCommon.generateIsisDictCliModel( IsisUloopModel )
isisUloopVRFsModel = generateVrfCliModel( isisUloopModel,
                              "Micro-loop information for all VRFs" )

class TilfaTunnelTableEntry( Model ):
   tunnelType = Enum( values=tunnelTypeEnumValues, help="Tunnel type" )
   tunnelIndex = Int( help="Tunnel index within tunnel type" )
   vias = List( valueType=MplsVia, help="List of tunnel vias" )
   backupVias = List( valueType=MplsVia,
                      help="List of tunnel backup vias" )
   tunnelId = Int( help="Tunnel identifier" )

   def render( self ):
      print "Tunnel Index %d" % self.tunnelIndex
      for via in self.vias:
         via.render()
      for via in self.backupVias:
         via.render()

class TilfaTunnelTable( Model ):
   entries = Dict( keyType=long, valueType=TilfaTunnelTableEntry,
                   help="Tunnel table entries keyed by tunnel index" )

   def render( self ):
      for _, tunnelTableTilfaEntry in sorted( self.entries.iteritems() ):
         tunnelTableTilfaEntry.render()

class IsisDynFloodNodeEntryModel( Model ):
   nodeId = Str( 'Node ID' )

   def processData( self, data ):
      self.nodeId = data.pop( 'node' )

   def getKey( self, data ):
      return int( data.get( 'index' ) )

class IsisDynFloodNodeLevelModel( Model ):
   nodes = GeneratorDict( keyType=int,
                          valueType=IsisDynFloodNodeEntryModel,
                          help='A dict of IS-IS nodes keyed by index' )

   def getKey( self, data ):
      return ord( data.get( 'level' ) )

   def render( self ):
      indexNodeFormat = "%-15s %-20s"
      print " " * 4, indexNodeFormat % ( "Index", "Node ID" )
      # print the node in order of the index
      for index, node in sorted( self.nodes, key=itemgetter( 0 ) ):
         print " " * 4, indexNodeFormat % ( index, node.nodeId )

class IsisDynFloodNodeModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodNodeLevelModel,
                           help="A dict of IS-IS nodes keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for level, levelModel in self.levels:
         print " " * 2, "Level %s Nodes:" % level
         levelModel.render()

isisDynFloodNodeModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodNodeModel )
isisDynFloodNodeVRFsModel = generateVrfCliModel(
   isisDynFloodNodeModel, "Dynamic flooding nodes for all VRFs" )

class IsisDynFloodPathsEntryModel( Model ):
   path = Str( help='Path' )

   def processData( self, data ):
      string = data.pop( 'path' )
      result = ""
      start = 0
      while start < len( string ):
         # Extract the next index
         index = int( string[ start:start + 2 ].encode( 'hex' ), 16 )
         start = start + 2
         if result:
            result = result + " "
         result = result + "%d" % index
      self.path = result

   def render( self ):
      if self.path:
         output = "    Path: " + self.path
         while len( output ) > LINE_MAX:
            index = output.rfind( " ", 0, LINE_MAX )
            print output[ 0 : index ]
            output = ' ' * 6 + output[ index + 1 : ]
         if len ( output ) > 8:
            print output

class IsisDynFloodPathsLevelModel( Model ):
   paths = GeneratorList( valueType=IsisDynFloodPathsEntryModel,
                          help='A list of IS-IS paths' )

   def getKey( self, data ):
      return ord( data.get( 'level' ) )

   def render( self ):
      for path in self.paths:
         path.render()

class IsisDynFloodPathsModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodPathsLevelModel,
                           help="A dict of lists of IS-IS paths keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for level, levelModel in self.levels:
         print "  Level %s:" % level
         levelModel.render()

isisDynFloodPathsModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodPathsModel )
isisDynFloodPathsVRFsModel = generateVrfCliModel(
   isisDynFloodPathsModel, "Dynamic flooding paths for all VRFs" )

class IsisDynFloodTopologyEntryModel( Model ):
   name = Str( help='A node in a path in the flooding topology' )

   def processData( self, data ):
      self.name = data.pop( 'name', None )

class IsisDynFloodTopologyPathModel( Model ):
   index = Int( help="Path index" )
   path = GeneratorList( valueType=IsisDynFloodTopologyEntryModel,
                         help='A path in the flooding topology' )

   def processData( self, data ):
      self.index = data.pop( 'path', 0 )

   def format( self, string ):
      output = "    Path: " + string
      while len( output ) > LINE_MAX:
         index = output.rfind( " ", 0, LINE_MAX )
         print output[ 0 : index ]
         output = ' ' * 4 + output[ index + 1 : ]
      if len ( output ) > 8:
         print output

   def render( self ):
      string = ""
      for node in self.path:
         if string:
            string += " "
         string += node.name
      if string:
         self.format( string )

class IsisDynFloodTopologyLevelModel( Model ):
   topology = GeneratorList( valueType=IsisDynFloodTopologyPathModel,
                             help='A list of IS-IS flooding paths' )

   def getKey( self, data ):
      return ord( data[ 'level' ] )

   def render( self ):
      for path in self.topology:
         path.render()

class IsisDynFloodTopologyModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodTopologyLevelModel,
                           help="A dict of lists of IS-IS flooding " +
                           "topology keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for level, levelModel in self.levels:
         print "  Level %s:" % level
         levelModel.render()

isisDynFloodTopologyModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodTopologyModel )
isisDynFloodTopologyVRFsModel = generateVrfCliModel(
   isisDynFloodTopologyModel, "Dynamic flooding topology for all VRFs" )

class IsisDynFloodInterfacesEntryModel( Model ):
   intf = Interface( help='Interface name' )
   tempIntf = Bool( help='Temporary interface addition to the flooding topology' )

   def processData( self, data ):
      self.intf = kernelIntfFilter( data.pop( 'name' ) )
      self.tempIntf = ord( data.pop( 'temp' ) ) != 0

   def render( self ):
      print '%s%s' % ( self.intf.stringValue, ' (temporary)'
                       if self.tempIntf else '' )

class IsisDynFloodInterfacesLevelModel( Model ):
   interfaces = GeneratorList( valueType=IsisDynFloodInterfacesEntryModel,
                          help='A list of IS-IS flooding paths' )

   def getKey( self, data ):
      return ord( data[ 'level' ] )

   def render( self ):
      for item in self.interfaces:
         print '   ',
         item.render()

class IsisDynFloodInterfacesModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   levels = GeneratorDict( keyType=int,
                           valueType=IsisDynFloodInterfacesLevelModel,
                           help="A dict of lists of IS-IS flooding " +
                           "interfaces keyed by level" )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      for level, levelModel in self.levels:
         print "  Level %s:" % level
         levelModel.render()

isisDynFloodInterfacesModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisDynFloodInterfacesModel )
isisDynFloodInterfacesVRFsModel = generateVrfCliModel(
   isisDynFloodInterfacesModel, "Dynamic flooding interfaces for all VRFs" )

#
# Area Proxy Models
#
class IsisShowAreaProxyEntryModel( Model ):
   proxySysId = Str( help='Area proxy system identifier' )
   proxyHostname = Str( help='Area proxy hostname' )
   status = Str( help='Area proxy status' )

   def processData( self, data ):
      self.proxySysId = data.get( 'proxy-sysid', '' )
      self.proxyHostname = data.get( 'proxy-hostname', '' )
      self.status = data.get( 'status', '' )

   def render( self ):
      print "  System Id: %s" % self.proxySysId
      print "  Hostname: %s" % self.proxyHostname
      print "  %s" % self.status

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_ISIS_AREA_PROXY_ENTRY

class IsisShowAreaProxyModel( Model ):
   _instanceName = Str( help='Instance Name' )
   _vrf = Str( help='VRF Name' )
   entry = Submodel( valueType=IsisShowAreaProxyEntryModel,
                     help='Show area proxy information' )

   def processData( self, data ):
      self._instanceName = data.pop( 'instance-name', None )
      self._vrf = data.pop( 'vrf-name' )

   def render( self ):
      if self._instanceName is None:
         return
      print ""
      print "IS-IS Instance: %s VRF: %s" % ( self._instanceName, self._vrf )
      if self.entry:
         self.entry.render()

isisShowAreaProxyModel = IsisCliModelCommon.generateIsisDictCliModel(
   IsisShowAreaProxyModel )
isisShowAreaProxyVRFsModel = generateVrfCliModel(
   isisShowAreaProxyModel, "Area proxy for all VRFs" )
