# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
      Bool,
      Dict,
      List,
      Model,
      Str,
      Enum,
      Submodel,
      Int
)
from TableOutput import (
      createTable,
      Format
)
import IntfModel
import textwrap
import re

class ProtocolInfo( Model ):
   supported = Bool( help="Indicates whether the protocol is supported" )
   protocolState = Enum( values=( "up", "down" ),
         help="Protocol state is down when the vrf is being created, state "
         "is up otherwise" )
   routingState = Enum( values=(
         "up", "down", "initializingUp", "initializingDown" ),
         help="Indicates the various states of routing for the Vrf."
         "'up' indicates routing is enabled for this vrf. "
         "'down' indicates routing is disabled for this vrf. "
         "'initializing up/down' indicates routing state is being configured" )
   multicastState = Enum( values=( "up", "initializing" ),
         help="Indicates the state of the multicast routing after it is"
         " configured", optional=True )

   def formatStateStr( self ):
      if self.protocolState == "down":
         return "incomplete"
      elif self.routingState == "up":
         return "routing"
      elif self.routingState == "initializingUp":
         return "pending_routing"
      elif self.routingState == "initializingDown":
         return "pending_no routing"
      elif self.routingState == "down":
         return "no routing"
      else:
         return ""

   def formatMulticastStr( self ):
      if self.multicastState == "up":
         return "multicast"
      elif self.multicastState == "initializing":
         return "pending_multicast"
      return ""

class IpProtocolDesc( Model ):
   ipv4 = Submodel( valueType=ProtocolInfo,
         help="ipv4 protocol information for a vrf" )
   ipv6 = Submodel( valueType=ProtocolInfo,
         help="ipv6 protocol information for a vrf" )

   def formatProtocolStr( self ):
      protoStr = []
      if self.ipv4.supported == True:
         protoStr.append( "ipv4" )
      if self.ipv6.supported == True:
         protoStr.append( "ipv6" )
      return ','.join( protoStr )

   def formatStateStr( self, stateStr, protoObj ):
      stateStr += protoObj.formatStateStr()
      mcast = protoObj.formatMulticastStr()
      if mcast:
         stateStr += "; "
         stateStr += mcast
      return stateStr

class VrfEntry( Model ):
   routeDistinguisher = Str( help="Route Distinguisher associated with this vrf" )
   protocols = Submodel( valueType=IpProtocolDesc,
         help="Information on ipv4 and ipv6 support for this vrf" )
   vrfState = Enum( values=( "creating", "up", "deleting" ),
         help="Various states of vrf. 'creating' indicates vrf is being"
         " created, 'up' indicates vrf is created successfully and 'deleting'"
         " indicates vrf is being deleted" )
   interfaces = List( valueType=IntfModel.Interface,
         help="List of interfaces which are assigned to this vrf" )

   def formatStateStr( self ):
      if self.vrfState == "up":
         strStateList = []
         strStateList.append( self.protocols.formatStateStr(
            "v4:", self.protocols.ipv4 ) )
         strStateList.append( self.protocols.formatStateStr(
            "v6:", self.protocols.ipv6 ) )
         strState = ',\n'.join( strStateList )

         # break on commas if more than 40 wide
         pat = re.compile( "(.{,40})($|, )" )
         strState = pat.sub( "\\1,\n", strState )
         # remove trailing comma
         pat = re.compile( "(.*)(,\n$)" )
         strState = pat.sub( "\\1\n", strState )
         return strState

      elif self.vrfState == "creating":
         return "pending create"
      elif self.vrfState == "deleting":
         return "pending delete"
      else:
         return ""

class VrfSummary( Model ):
   vrfCount = Int( help="Number of VRFs" )
   vrfUpCount = Int( help="Number of up VRFs" )
   vrfV4RoutingCount = Int( help="Number of VRFs supporting IPv4 routing" )
   vrfV6RoutingCount = Int( help="Number of VRFs supporting IPv6 routing" )

   def render( self ):
      print "VRF count: %d" % self.vrfCount
      print "VRF up count: %d" % self.vrfUpCount
      print "VRF IPv4 routing count: %d" % self.vrfV4RoutingCount
      print "VRF IPv6 routing count: %d" % self.vrfV6RoutingCount

#--------------------------------------------------------
# show vrf Model
# Contains a dictionary of vrf entries based on vrf name
# and the output is rendered in the form of a table
#---------------------------------------------------------
class Vrf( Model ):
   ''' render all the goodies in the Vrf '''
   vrfs = Dict( keyType=str, valueType=VrfEntry,
         help="Mapping of Vrf name to Vrf information" )

   def render( self ):
      t = createTable( ( 'VRF', 'RD', 'Protocols', 'State', 'Interfaces' ) )
      f1 = Format( justify='left' )
      f2 = Format( justify='left' )
      f3 = Format( justify='left' )
      ifWidth = 30  # If more then 1 interface in table, we don't like it to wrap
      f4 = Format( justify='left', maxWidth=40 )
      f5 = Format( justify='left', maxWidth=ifWidth )
      t.formatColumns( f1, f2, f3, f4, f5 )

      for vrf, vrfEntry in sorted( self.vrfs.iteritems() ):
         intfList = [ intf.stringValue for intf in vrfEntry.interfaces ]
         t.newRow( vrf, "<not set>" if vrfEntry.routeDistinguisher == ""
               else vrfEntry.routeDistinguisher,
               vrfEntry.protocols.formatProtocolStr(),
               vrfEntry.formatStateStr(),
               textwrap.fill( ', '.join( intfList ),
                  ifWidth ) )

      print t.output()

class ShowCliVrfCtxModel( Model ):
   vrfRoutingContext = Str( help="Name of VRF for the current routing-context." )

   def render( self ):
      print "Current VRF routing-context is %s" % self.vrfRoutingContext
