# Copyright (c) 2011-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# VRF-related CLI

'''Configuration commands for VRF'''

import os
from collections import defaultdict

import BasicCli
import Tac
import CliMatcher
import CliCommand
import CliToken.Cli
import ShowCommand
import CliParser
import LazyMount
import ConfigMount
import exceptions
import RouteDistinguisher
import Arnet
import Tracing
import Cell
from IpLibConsts import (
      DEFAULT_VRF,
      DEFAULT_VRF_OLD,
      VRFNAMES_RESERVED,
)
from IraIpIntfCli import canSetVrfHook
from CliMode.Ira import VrfDefinitionBaseMode
import BasicCliSession
import CmdExtension, CliExtensions
from Arnet.NsLib import DEFAULT_NS
from CliPlugin.IraVrfModel import (
      Vrf,
      VrfSummary,
      VrfEntry,
      IpProtocolDesc,
      ProtocolInfo,
      ShowCliVrfCtxModel
)
import CliPlugin.VrfCli as VrfCli
import IraRouteCommon

allVrfConfig = None
rdAutoInputDir = None
rdConfigInputDir = None
multiAgentVrfProtoConfig = None
allVrfStatusLocal = None
allVrfStatusGlobal = None
routingVrfConfigDir = None
routingVrfRouteConfigDir = None
routingVrfInfoDir = None
routingConfig = None
routing6VrfConfigDir = None
routing6VrfRouteConfigDir = None
routing6VrfInfoDir = None
routing6Config = None
runnabilityConfig = None
arpInputConfigCli = None
routingHwStatus = None
ip = None
ip6 = None
ipConfig = None
ip6Config = None
l3ConfigDir = None

getAllPlusReservedVrfNames = VrfCli.getAllPlusReservedVrfNames
ipRedirect = IraRouteCommon.Ip4()
ip6Redirect = IraRouteCommon.Ip6()
routing = IraRouteCommon.routing( ipRedirect )
routing6 = IraRouteCommon.routing( ip6Redirect )
t0 = Tracing.trace0

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )

vrfHelpDesc = 'Configure VRFs'
vrfInstHelpDesc = 'Enter VRF definition sub-mode'
rdHelpDesc = 'BGP route distinguisher'

# The CLI hook for VRFs is used to allow individual features to notify/block
# deletion of global VRF config which is otherwise supported. This allows those
# those features to remove any VRF specific configuration. Ira calls all hooks
# and ANDs the results, if any are False, the config is blocked. This function
# is called as:
#
# (accept, message) = hook( vrfName )
#
# vrfName is the name of relevant vrf.
#
# accept must be a boolean, message is a text string giving the specific
# reason why the deletion is blocked. The message is required.

canDeleteVrfHook = CliExtensions.CliHook()

deletedVrfHook = CliExtensions.CliHook()

######################################################################
# vrf instance sub-mode
#
# legacy:
# vrf definition <vrfName>
#
# Note that this is the new protocol-independent VRF definition mode.
# We are not supporting the old per-protocol 'ip vrf' and 'ipv6 vrf' CLI.
######################################################################

class VrfDefinitionMode( VrfDefinitionBaseMode, BasicCli.ConfigModeBase ):
   name = 'VRF Definition'
   vrfName = None
   vrfConfig = None
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      VrfDefinitionBaseMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.vrfName = vrfName
      self.maybeMakeVrf()
      self.redirectDefault()

   def maybeMakeVrf( self ):
      vrfConfig = allVrfConfig.vrf.get( self.vrfName )
      if not vrfConfig:
         # Initialize default address family configuration to
         # True. The VrfSm will consider the configuration in the
         # context of the vrfCapability to determine the actual
         # enablement status of each address family.
         vrfConfig = allVrfConfig.vrf.newMember( self.vrfName )
         vrfConfig.addressFamily[ 'ipv4' ] = True
         vrfConfig.addressFamily[ 'ipv6' ] = True

      if self.vrfName not in routingVrfConfigDir.vrfConfig:
         vrfConfig = routingVrfConfigDir.newVrfConfig( self.vrfName )
         vrfRouteConfig = routingVrfRouteConfigDir.newVrfConfig( self.vrfName )

      if self.vrfName not in routing6VrfConfigDir.vrfConfig:
         vrfConfig = routing6VrfConfigDir.newVrfConfig( self.vrfName )
         vrfRouteConfig = routing6VrfRouteConfigDir.newVrfConfig( self.vrfName )
         assert vrfConfig is not None
         assert vrfRouteConfig is not None

      if self.vrfName not in arpInputConfigCli.vrf:
         c = arpInputConfigCli.newVrf( self.vrfName )
         assert c is not None

   def redirectDefault( self ):
      r4 = routing.config( DEFAULT_VRF )
      routingVrfConfigDir.vrfConfig[ self.vrfName ].sendRedirects = r4.sendRedirects
      r6 = routing6.config( DEFAULT_VRF )
      routing6VrfConfigDir.vrfConfig[ self.vrfName ].sendRedirects = r6.sendRedirects

   def noAddressFamily( self, addressFamily ):
      allVrfConfig.vrf[ self.vrfName ].addressFamily[ addressFamily ] = False

def gotoVrfDefMode( mode, vrfName, deprecatedCmd=False ):
   vrfCapability = routingHwStatus.vrfCapability
   def vrfCreationAllowed():
      if vrfCapability.maxVrfs == -1:
         # this means that we haven't initialized the maxVrfs field
         # yet.  This happens when parsing the system startup config,
         # and since we don't know how many are supported we assume
         # that the config file was written correctly and accept
         # everything into Config
         return True
      return len( allVrfConfig.vrf ) < vrfCapability.maxVrfs

   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "The vrf name '%s' is reserved." % vrfName )
      return

   try:
      if vrfName not in allVrfConfig.vrf and not vrfCreationAllowed():
         mode.addError( "Can't create VRF %s (maximum number of VRFs supported"
                        " is %d)" % ( vrfName, vrfCapability.maxVrfs ) )
         return
   except exceptions.IndexError:
      mode.addError( "'%s' too long: must be no more than %d characters"
                     % ( vrfName, Tac.Type( "L3::VrfName" ).maxLength ) )
      return

   childMode = mode.childMode( VrfDefinitionMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteVrf( mode, vrfName ):
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "The vrf name '%s' is reserved." % vrfName )
      return
   intfModifiedCount = 0

   # first check to see if anyone is preventing us from deleting the VRF
   for hook in canDeleteVrfHook.extensions():
      (accept, hookMsg) = hook( vrfName )
      if hookMsg:
         # pylint: disable-msg=W0106
         mode.addWarning( hookMsg ) if accept else mode.addError( hookMsg )
      if not accept:
         return

   # first check to see if anyone is preventing us from deleting the VRF
   for ipIntfName in ipConfig.ipIntfConfig:
      iic = ipConfig.ipIntfConfig.get( ipIntfName )
      if iic != None and iic.vrf == vrfName:
         for hook in canSetVrfHook.extensions():
            (accept, hookMsg) = hook( ipIntfName, oldVrf=vrfName, newVrf=DEFAULT_VRF,
                                      vrfDelete=True )
            if hookMsg:
               if accept:
                  mode.addWarning( hookMsg )
               else:
                  mode.addError( hookMsg )
            if not accept:
               return

   # first check to see if anyone is preventing us from deleting the VRF
   for ipIntfName in ip6Config.intf:
      iic = ip6Config.intf.get( ipIntfName )
      if iic != None and iic.vrf == vrfName:
         for hook in canSetVrfHook.extensions():
            (accept, hookMsg) = hook( ipIntfName, oldVrf=vrfName, newVrf=DEFAULT_VRF,
                                      vrfDelete=True )
            if hookMsg:
               if accept:
                  mode.addWarning( hookMsg )
               else:
                  mode.addError( hookMsg )
            if not accept:
               return

   # clear out the address(es) on all of the interfaces in the VRF &
   # set the VRF to null
   for ipIntfName in ipConfig.ipIntfConfig:
      iic = ipConfig.ipIntfConfig.get( ipIntfName )
      if iic is None:
         continue
      if iic.vrf == vrfName:
         # FIXFIXFIX really ought to find a way to call the noIpAddr()
         # function in IraIpIntfCli.py rather than replicating the
         # logic here.  Ick.
         intfModifiedCount = intfModifiedCount + 1
         for a in iic.secondaryWithMask:
            t0( 'clearing address %s from intf %s due to deletion of vrf %s' %
                ( a, ipIntfName, vrfName ) )
            del iic.secondaryWithMask[ a ]

         for a in iic.virtualSecondaryWithMask:
            t0( 'clearing virtual secondary address %s from \
                 intf %s due to deletion of vrf %s' % ( a, ipIntfName, vrfName ) )
            del iic.virtualSecondaryWithMask[ a ]
         # Disable dhcp on the interface
         # This would ensure that the interface doesn't get an IP address
         # via dhcp in the default vrf when this vrf is deleted
         iic.addrSource = 'manual'
         zeroAddr = Arnet.AddrWithMask( "0.0.0.0/0" )
         if iic.addrWithMask != zeroAddr:
            t0( 'clearing address %s from intf %s due to deletion of vrf %s' %
                ( iic.addrWithMask, ipIntfName, vrfName ) )
            iic.addrWithMask = zeroAddr

         if iic.virtualAddrWithMask != zeroAddr:
            t0( 'clearing virtual address %s from \
                 intf %s due to deletion of vrf %s' %
                 ( iic.virtualAddrWithMask, ipIntfName, vrfName ) )
            iic.virtualAddrWithMask = zeroAddr

         # now v6
         if ip6Config.intf.get( ipIntfName ):
            i6ic = ip6Config.intf[ ipIntfName ]
            i6ic.addrSource = 'manual'
            for a in i6ic.addr:
               t0( 'clearing address %s from intf %s due to deletion of vrf %s' %
                   ( a, ipIntfName, vrfName ) )
               del i6ic.addr[ a ]

         # now clear out the VRF
         l3ConfigDir.intfConfig[ ipIntfName ].vrf = DEFAULT_VRF

   for ipIntfName in ip6Config.intf:
      iic = ip6Config.intf.get( ipIntfName )
      if iic is None:
         continue
      if iic.vrf == vrfName:
         intfModifiedCount = intfModifiedCount + 1

         iic.addrSource = 'manual'
         if ip6Config.intf.get( ipIntfName ):
            i6ic = ip6Config.intf[ ipIntfName ]
            for a in i6ic.addr:
               t0( 'clearing address %s from intf %s due to deletion of vrf %s' %
                   ( a, ipIntfName, vrfName ) )
               del i6ic.addr[ a ]

         # now clear out the VRF
         l3ConfigDir.intfConfig[ ipIntfName ].vrf = DEFAULT_VRF

   if intfModifiedCount > 0:
      mode.addWarning(
         'IP addresses from all interfaces in VRF %s have been removed' % vrfName )

   if allVrfConfig.vrf.get( vrfName ):
      del allVrfConfig.vrf[ vrfName ]

   vrfDefinitionRouteDistinguisherInput = getVrfDefinitionRouteDistinguisherInput()
   if vrfName in vrfDefinitionRouteDistinguisherInput.routeDistinguisher:
      del vrfDefinitionRouteDistinguisherInput.routeDistinguisher[ vrfName ]

   if vrfName in routingVrfConfigDir.vrfConfig:
      del routingVrfConfigDir.vrfConfig[ vrfName ]
      del routingVrfRouteConfigDir.vrfConfig[ vrfName ]

   if vrfName in routing6VrfConfigDir.vrfConfig:
      del routing6VrfConfigDir.vrfConfig[ vrfName ]
      del routing6VrfRouteConfigDir.vrfConfig[ vrfName ]

   if vrfName in runnabilityConfig.vrf:
      del runnabilityConfig.vrf[ vrfName ]

   if vrfName in arpInputConfigCli.vrf:
      del arpInputConfigCli.vrf[ vrfName ]

   # Notify interested parties that the vrf has been deleted
   for hook in deletedVrfHook.extensions():
      hook( vrfName )

   # Reset the CLI session to VRF mapping, if required.
   cliSessVrf = VrfCli.vrfMap.getCliSessVrf( mode.session )
   if ( cliSessVrf == vrfName ):
      VrfCli.vrfMap.setCliVrf( mode.session )


class VrfAddressFamilyMode( BasicCli.ConfigModeBase ):
   name = 'VRF AF Mode'
   vrfName = None
   vrfConfig = None
   addressFamily = None
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName, addressFamily ):
      self.modeKey = 'vrf-af'
      self.longModeKey = 'vrf-%s-af-%s' % ( vrfName, addressFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.vrfName = vrfName
      self.addressFamily = addressFamily
      self.vrfConfig = allVrfConfig.vrf.get( self.vrfName )
      assert( self.vrfConfig )
      self.vrfConfig.addressFamily[ addressFamily ] = True

   # route-target config commands go here when we add them

   def exitAddressFamily( self ):
      self.session_.gotoParentMode()

def gotoVrfAfMode( mode, addressFamily ):
   childMode = mode.childMode( VrfAddressFamilyMode,
                               vrfName=mode.vrfName,
                               addressFamily=addressFamily )
   mode.session_.gotoChildMode( childMode )

def getVrfNameFromIntf( _ipConfig, _ipStatus, _intfName, af=AddressFamily.ipv4 ):
   intfStatusAttr = 'ipIntfStatus' if af == AddressFamily.ipv4 else 'intf'
   ipIntfStatus = getattr( _ipStatus, intfStatusAttr ).get( _intfName )
   if ipIntfStatus:
      return ipIntfStatus.vrf
   intfConfigAttr = 'ipIntfConfig' if af == AddressFamily.ipv4 else 'intf'
   ipIntfConfig = getattr( _ipConfig, intfConfigAttr ).get( _intfName )
   if ipIntfConfig:
      return ipIntfConfig.vrf

   return DEFAULT_VRF

########## enter VRF mode
# 'definition' is actually a matcher wrapped by a Node, because only the node
# supports deprecatedBy.
# 'definition' is deliberately not hidden. I do not know why, but it was also not
# hidden before conversion to new parser.
vrfDefDeprecatedMatcher = CliMatcher.KeywordMatcher(
   'definition', 'Enter VRF definition sub-mode' )
vrfDefDeprecatedNode = CliCommand.Node( vrfDefDeprecatedMatcher,
                                        deprecatedByCmd='vrf instance [VRF_ID]' )

vrfNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: allVrfConfig.vrf,
                                                'VRF name' )
allVrfNameMatcher = CliMatcher.DynamicNameMatcher( VrfCli.getAllVrfNames,
                                                   helpdesc='VRF name' )

allPlusReservedVrfNameMatcher = CliMatcher.DynamicNameMatcher(
      getAllPlusReservedVrfNames, 'VRF name' )

class EnterVrfDefMode( CliCommand.CliCommandClass ):
   syntax = '''vrf ( instance | definition ) VRF_NAME'''
   noOrDefaultSyntax = syntax

   data = { "vrf": vrfHelpDesc,
            "instance": vrfInstHelpDesc,
            "definition": vrfDefDeprecatedNode,
            "VRF_NAME": vrfNameMatcher }

   @staticmethod
   def handler( mode, args ):
      gotoVrfDefMode( mode, args[ "VRF_NAME" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      deleteVrf( mode, args[ "VRF_NAME" ] )

BasicCli.GlobalConfigMode.addCommandClass( EnterVrfDefMode )

########## set the RD

routeDistinguisherMatcher = RouteDistinguisher.RdDistinguisherMatcher(
                            helpdesc='BGP route distinguisher' )

class VrfRdCmd( CliCommand.CliCommandClass ):
   syntax = '''rd <rd>'''
   noOrDefaultSyntax = '''rd ...'''

   data = { "rd": rdHelpDesc,
            "<rd>": routeDistinguisherMatcher }

   @staticmethod
   def handler( mode, args ):
      checker = Tac.Value( 'Ira::RdAssignmentChecker',
                           LazyMount.force( rdAutoInputDir ),
                           LazyMount.force( rdConfigInputDir ),
                           mode.vrfName, args[ "<rd>" ] )
      if not checker.acceptable:
         mode.addError( checker.errorMessage )
         return

      mode.addMessage( "Since the RD is required for BGP operation, "
                       "please configure the RD for VRF %s "
                       "under the 'router bgp vrf' submode. "
                       "The configuration of the RD "
                       "under the VRF definition submode "
                       "is deprecated and no longer required."
                       % mode.vrfName )

      tacRd = Tac.Value( 'Arnet::RouteDistinguisher' )
      try:
         tacRd.stringValue = args[ "<rd>" ]
      except IndexError:
         mode.addError( 'Malformed Route Distinguisher: %s' % args[ "<rd>" ] )
         return

      vrfDefinitionRouteDistinguisherInput = \
         getVrfDefinitionRouteDistinguisherInput()
      vrfDefinitionRouteDistinguisherInput.routeDistinguisher[ mode.vrfName ] = tacRd

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfDefinitionRouteDistinguisherInput = \
            getVrfDefinitionRouteDistinguisherInput()
      if mode.vrfName in vrfDefinitionRouteDistinguisherInput.routeDistinguisher:
         del vrfDefinitionRouteDistinguisherInput.routeDistinguisher[ mode.vrfName ]

VrfDefinitionMode.addCommandClass( VrfRdCmd )

######### set description command

class VrfDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = '''description <desc>'''
   noOrDefaultSyntax = '''description ...'''

   data = { "description": 'VRF description',
            "<desc>": CliMatcher.StringMatcher( helpname='LINE',
                              helpdesc='Up to 240 characters describing this VRF' )
          }

   @staticmethod
   def handler( mode, args ):
      allVrfConfig.vrf[ mode.vrfName ].description = args[ "<desc>" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      allVrfConfig.vrf[ mode.vrfName ].description = ''

VrfDefinitionMode.addCommandClass( VrfDescriptionCmd )

########## AF mode entry/exit

#
# This is currently disabled.  For the first release we are only
# supporting IPv4.  Rather than force the user to type a bunch of
# useless commands to enable IPv4, we're going with the following
# semantics:
#
# IPv4 is enabled by default (and cannot be disabled in 4.10.0)
# IPv6 is disabled by default (and cannot be enabled in 4.10.0)
#
# to disable IPv4 in future releases we will do:
# ! address-family ipv4
# !   shutdown
# ! exit-address-family
#

#tokenAfInVrf = CliParser.KeywordRule( 'address-family',
#                                      helpdesc='Configure address family for VRF' )
#
#afOrRule = CliParser.OrRule( name='addressFamily' )
#afOrRule |= CliParser.KeywordRule( 'ipv4', helpdesc='IPv4 related' )
#afOrRule |= CliParser.KeywordRule( 'ipv6', helpdesc='IPv6 related' )
#
#VrfDefinitionMode.addCommand( ( tokenAfInVrf, afOrRule, gotoVrfAfMode ) )
#VrfDefinitionMode.addCommand( ( BasicCli.noOrDefault, tokenAfInVrf, afOrRule,
#                                VrfDefinitionMode.noAddressFamily ) )
#
#tokenExitAf = CliParser.KeywordRule( 'exit-address-family',
#                                     helpdesc='Exit from address-family mode' )
#
#VrfAddressFamilyMode.addCommand( ( tokenExitAf,
#                                   VrfAddressFamilyMode.exitAddressFamily ) )


######################################################################
# show vrf.
######################################################################

showMCastHook = CliExtensions.CliHook()
# showMCastHook allows Mroute to provide extra
# information about enabled/disabled multicast routing.

def intfsByVrf( ipIntfs, ip6Intfs ):
   '''
   return a list of interface assignment by VRFs
   '''
   intfsByVrfDict = defaultdict( set )
   for intfs in [ ipIntfs, ip6Intfs ]:
      for intfName, intf in intfs.items():
         intfsByVrfDict[ intf.vrf ].add( intfName )
   return intfsByVrfDict

def populateProtocolInfoModel( protocolInfoModel, af, vrfState, vrfKey,
                               isDefaultVrf, vrfConfigOrStatus, mode ):
   proto = af.strip( 'ip' )
   if af == 'ipv4':
      afEnabled = routingHwStatus.vrfCapability.ipv4EnabledDefault
   else:
      afEnabled = routingHwStatus.vrfCapability.ipv6EnabledDefault
   if isDefaultVrf or ( vrfConfigOrStatus is not None and
                        af in vrfConfigOrStatus.addressFamily and
                        vrfConfigOrStatus.addressFamily[ af ] and
                        afEnabled ):
      protocolInfoModel.supported = True

   if vrfState != "up":
      return

   vrfConfigDir = routingVrfConfigDir if af == 'ipv4' else routing6VrfConfigDir
   configDir = routingConfig if af == 'ipv4' else routing6Config
   vrfInfoDir = routingVrfInfoDir if af == 'ipv4' else routing6VrfInfoDir

   rc = configDir if isDefaultVrf else vrfConfigDir.vrfConfig.get( vrfKey )
   rs = vrfInfoDir.get( vrfKey )
   if not ( rc and rs ):
      protocolInfoModel.protocolState = "down"
      protocolInfoModel.routingState = "down"
   else:
      if rc.routing:
         protocolInfoModel.routingState = "up"
         for hook in showMCastHook.extensions():
            if( hook( mode, vrfKey, proto ) ):
               if rc.routing != rs.routing:
                  protocolInfoModel.multicastState = "initializing"
               else:
                  protocolInfoModel.multicastState = "up"

         if rc.routing != rs.routing:
            protocolInfoModel.routingState = "initializingUp"
      else:
         protocolInfoModel.routingState = "down"
         if rc.routing != rs.routing:
            protocolInfoModel.routingState = "initializingDown"

def getVrfState( vrfS, vrfC, isDefaultVrf ):
   vrfState = "up"
   if vrfS is None and not isDefaultVrf:
      vrfState = "creating"
   if vrfC is None and not isDefaultVrf:
      vrfState = "deleting"
   return vrfState

def getVrfEntryModel( mode, vrfKey ):
   isDefaultVrf = vrfKey in ( DEFAULT_VRF, DEFAULT_VRF_OLD )
   iByV = intfsByVrf( ip.ipIntfStatus, ip6.intf )

   vrfS = allVrfStatusGlobal.vrf.get( vrfKey )
   vrfC = allVrfConfig.vrf.get( vrfKey )
   if not vrfS and not vrfC and not isDefaultVrf:
      return None
   vrfEntryModel = VrfEntry()
   vrfEntryModel.protocols = IpProtocolDesc()
   vrfEntryModel.interfaces = []
   vrfEntryModel.routeDistinguisher = ""
   vrfEntryModel.protocols.ipv4 = ProtocolInfo(
         supported=False, protocolState="up",
         routingState="up" )
   vrfEntryModel.protocols.ipv6 = ProtocolInfo(
         supported=False, protocolState="up",
         routingState="up" )
   protoDict = { 'v4': vrfEntryModel.protocols.ipv4,
                 'v6': vrfEntryModel.protocols.ipv6 }

   vrfConfigOrStatus = vrfS
   if vrfS is None and not isDefaultVrf:
      vrfConfigOrStatus = vrfC

   vrfEntryModel.vrfState = getVrfState( vrfS, vrfC, isDefaultVrf )

   for af in ( 'ipv4', 'ipv6' ):
      proto = af.strip( 'ip' )
      populateProtocolInfoModel(
            protoDict[ proto ], af, vrfEntryModel.vrfState, vrfKey,
            isDefaultVrf, vrfConfigOrStatus, mode )

   internalIntf = Tac.Type( 'Arnet::InternalIntfId' )
   intfs = iByV.get( vrfKey )
   if intfs:
      for intfName in sorted( intfs ):
         intfId = Arnet.IntfId( intfName )
         if not internalIntf.isInternalIntfId( intfId ):
            vrfEntryModel.interfaces.append( intfId )

   # For default vrf, we can only retrieve value
   # from BGP routeDistinguisherInputDir
   if isDefaultVrf:
      assert 'bgp' in rdConfigInputDir
      bgpRdInput = rdConfigInputDir.get( "bgp" ).routeDistinguisher
      rd = bgpRdInput.get( vrfKey )
      if rd and rd != 'INVALID':
         vrfEntryModel.routeDistinguisher = rd
      else:
         autoRdInput = rdAutoInputDir.get( "bgpAuto" )
         if autoRdInput:
            rd = autoRdInput.routeDistinguisher.get( vrfKey )
            if rd and rd != 'INVALID':
               vrfEntryModel.routeDistinguisher = rd
   else:
      if vrfS and vrfS.routeDistinguisher != 'INVALID':
         vrfEntryModel.routeDistinguisher = vrfS.routeDistinguisher
   return vrfEntryModel

def showVrfSummary( mode ):
   vrfCount = 0
   vrfUpCount = 0
   vrfV4RoutingCount = 0
   vrfV6RoutingCount = 0

   allVrfs = set( allVrfStatusGlobal.vrf ).\
      union( set( allVrfConfig.vrf ) )
   allVrfs.add( DEFAULT_VRF )

   for vrfKey in allVrfs:
      isDefaultVrf = vrfKey in ( DEFAULT_VRF, DEFAULT_VRF_OLD )
      vrfS = allVrfStatusGlobal.vrf.get( vrfKey )
      vrfC = allVrfConfig.vrf.get( vrfKey )
      if not vrfS and not vrfC and not isDefaultVrf:
         continue
      vrfConfigOrStatus = vrfS
      if vrfS is None and not isDefaultVrf:
         vrfConfigOrStatus = vrfC
      vrfCount += 1
      v4ProtocolInfo = ProtocolInfo( supported=False, protocolState="up",
                                     routingState="up" )
      v6ProtocolInfo = ProtocolInfo( supported=False, protocolState="up",
                                     routingState="up" )
      vrfState = getVrfState( vrfS, vrfC, isDefaultVrf )
      if vrfState == "up":
         vrfUpCount += 1
      else:
         # if the vrfState is not up we don't care about its protocols
         continue

      populateProtocolInfoModel(
         v4ProtocolInfo, 'ipv4', vrfState, vrfKey, isDefaultVrf, vrfConfigOrStatus,
         mode )
      populateProtocolInfoModel(
         v6ProtocolInfo, 'ipv6', vrfState, vrfKey, isDefaultVrf, vrfConfigOrStatus,
         mode )

      if v4ProtocolInfo.routingState == "up":
         vrfV4RoutingCount += 1
      if v6ProtocolInfo.routingState == "up":
         vrfV6RoutingCount += 1

   return VrfSummary( vrfCount=vrfCount, vrfUpCount=vrfUpCount,
                      vrfV4RoutingCount=vrfV4RoutingCount,
                      vrfV6RoutingCount=vrfV6RoutingCount )

def showVrf( mode, vrfName=None ):
   vrfModel = Vrf()
   assert( allVrfStatusGlobal is not None and
           allVrfConfig is not None )
   if not vrfName:
      vrfMax = routingHwStatus.vrfCapability.maxVrfs
      if vrfMax > 0:
         print "Maximum number of vrfs allowed: %d" % vrfMax

      allVrfs = set( allVrfStatusGlobal.vrf.keys() ).\
         union( set( allVrfConfig.vrf.keys() ) )

      # get vrfEntryModel for all vrfs in
      # allVrfStatusGlobal/allVrfConfig collection
      for vrfKey in allVrfs:
         vrfEntryModel = getVrfEntryModel( mode, vrfKey )
         if vrfEntryModel is None:
            continue
         vrfModel.vrfs[ vrfKey ] = vrfEntryModel

      # special case: default vrf
      # default vrf is not in allVrfStatusGlobal/allVrfConfig collections
      vrfEntryModel = getVrfEntryModel( mode, DEFAULT_VRF )
      if vrfEntryModel:
         vrfModel.vrfs[ DEFAULT_VRF ] = vrfEntryModel
   else:
      vrfEntryModel = getVrfEntryModel( mode, vrfName )
      if vrfEntryModel:
         vrfModel.vrfs[ vrfName ] = vrfEntryModel

   return vrfModel

vrfKwForShow = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Display VRF state' )

class ShowVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vrf [ <vrfName> ]'''
   data = { 'vrf' : vrfKwForShow,
            '<vrfName>': allVrfNameMatcher
          }
   cliModel = Vrf

   @staticmethod
   def handler( mode, args ):
      return showVrf( mode, vrfName=args.get( "<vrfName>" ) )

BasicCli.addShowCommandClass( ShowVrfCmd )

class ShowVrfSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vrf summary'
   data = {
      'vrf': vrfKwForShow,
      'summary': 'VRF summary'
   }
   cliModel = VrfSummary

   @staticmethod
   def handler( mode, args ):
      return showVrfSummary( mode )

BasicCli.addShowCommandClass( ShowVrfSummaryCmd )

######################################################################
# cli vrf <vrfName>
#
# legacy:
# routing-context vrf <vrfName>
######################################################################

def chVrf( mode, vrfName ):
   assert vrfName != ''
   t0( 'changing vrf context to %s' % vrfName )
   if vrfName in ( DEFAULT_VRF, DEFAULT_VRF_OLD ):
      nsName = DEFAULT_NS
   else:
      vrfS = allVrfStatusLocal.vrf.get( vrfName )
      if vrfS is None:
         mode.addError( "VRF %s does not exist" % vrfName )
         return
      elif vrfS.state == 'active':
         nsName = vrfS.networkNamespace
         assert nsName != ''
      else:
         mode.addError( "VRF %s is not active" % vrfName )
         return

   VrfCli.vrfMap.setCliVrf( mode.session, vrfName, nsName )

routeCtxKwDeprecated = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                             'routing-context',
                                             helpdesc='Enter VRF context' ),
                                        deprecatedByCmd='cli vrf [VRF_ID]' )

class VrfForRtCtxCmd( CliCommand.CliCommandClass ):
   syntax = "( cli | routing-context ) vrf ( <vrfName> | %s | %s )" % (
            DEFAULT_VRF, DEFAULT_VRF_OLD )

   data = { "cli": CliToken.Cli.cliForExecMatcher,
            "routing-context": routeCtxKwDeprecated,
            "vrf": 'Enter VRF context',
            "<vrfName>": vrfNameMatcher,
            DEFAULT_VRF: CliCommand.Node(
                  CliMatcher.KeywordMatcher( DEFAULT_VRF,
                        helpdesc='Default virtual routing and forwarding instance',
                                             priority=CliParser.PRIO_HIGH ) ),
            DEFAULT_VRF_OLD: CliCommand.Node(
                  CliMatcher.KeywordMatcher( DEFAULT_VRF_OLD,
                   helpdesc='Default virtual routing and forwarding instance (old)',
                                             priority=CliParser.PRIO_HIGH ),
                  hidden=True )
          }

   @staticmethod
   def handler( mode, args ):
      if DEFAULT_VRF in args:
         vrfName = DEFAULT_VRF
      elif DEFAULT_VRF_OLD in args:
         vrfName = DEFAULT_VRF_OLD
      else:
         vrfName = args.get( "<vrfName>" )
      chVrf( mode, vrfName )

BasicCli.EnableMode.addCommandClass( VrfForRtCtxCmd )

######################################################################
# show routing-context vrf
# show cli vrf
######################################################################

def showCliVrfCtx( mode, args ):
   vrfName = VrfCli.vrfMap.getCliSessVrf( mode.session )
   vrfContext = ShowCliVrfCtxModel()
   if vrfName is not None:
      vrfContext.vrfRoutingContext = vrfName
      if vrfName not in ( DEFAULT_VRF, DEFAULT_VRF_OLD ) and\
            vrfName not in allVrfConfig.vrf.members():
         mode.addWarning( "Warning: VRF %s does not exist" % vrfName )
   else:
      vrfContext.vrfRoutingContext = "Unknown"
   #TBD: Do we care if vrfName is None. Why is this an error? What if
   # we assume that if no vrf is associated with a session, then it is
   # the default vrf and execute commands in the default vrf?
   return vrfContext

cliKwForShow = CliToken.Cli.cliForShowMatcher
routeCtxKw = CliMatcher.KeywordMatcher( 'routing-context', helpdesc='VRF context' )
vrfKwForShowRtCtx = CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF context' )

class ShowCliVrfCtxCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ( cli | routing-context ) vrf'''
   data = { 'cli': cliKwForShow,
            'routing-context': routeCtxKw,
            'vrf': vrfKwForShowRtCtx }
   handler = showCliVrfCtx
   cliModel = ShowCliVrfCtxModel

BasicCli.addShowCommandClass( ShowCliVrfCtxCmd )

#-------------------------------------------------------------
# Register a format specifier for a vrf Name in the Cli prompt
#-------------------------------------------------------------
def getVrfCliPrompt( session ):
   vrfName = VrfCli.vrfMap.getCliSessVrf( session )
   if VrfCli.vrfMap.isDefaultVrf( vrfName ):
      return ""
   else:
      return "(vrf:%s)" % vrfName

#-------------------------------------------------------------
# Add vrfname to 'routing/multiAgentVrfProtocolConfig'
# when router <ospf/isis> <instance> vrf <vrfname>
# is run for the first time.
#-------------------------------------------------------------
def findOrCreateMultiAgentVrfProtoEntry( vrfName ):
   multiAgentVrfProtocolEntry = \
         multiAgentVrfProtoConfig.vrfConfig.newMember( vrfName )
   return multiAgentVrfProtocolEntry

#-------------------------------------------------------------
# Remove vrfName from 'routing/multiAgentVrfProtocolConfig'
# when no router <ospf/isis> <instance> vrf <vrfname>
# is run, and all agents have been unconfigured
#-------------------------------------------------------------
def maybeDelMultiAgentVrfProtoEntry( vrfName ):
   if len( multiAgentVrfProtoConfig.vrfConfig.get( vrfName ).agent ) == 0:
      del multiAgentVrfProtoConfig.vrfConfig[ vrfName ]

def addAgentVrfEntry( vrfName, agentName ):
   if vrfName == '' or vrfName is None:
      vrfName = DEFAULT_VRF
   multiAgentVrfProtocolEntry = \
         findOrCreateMultiAgentVrfProtoEntry( vrfName )
   multiAgentVrfProtocolEntry.agent.newMember( agentName, "", "" )

def removeAgentVrfEntry( vrfName, agentName ):
   if vrfName == '' or vrfName is None:
      vrfName = DEFAULT_VRF
   if vrfName in multiAgentVrfProtoConfig.vrfConfig:
      del multiAgentVrfProtoConfig.vrfConfig.get( vrfName ).agent[ agentName ]
      maybeDelMultiAgentVrfProtoEntry( vrfName )

def getVrfDefinitionRouteDistinguisherInput():
   vrfDefinitionRouteDistinguisherInput = rdConfigInputDir.get( 'vrfDefinition' )
   assert vrfDefinitionRouteDistinguisherInput
   return vrfDefinitionRouteDistinguisherInput

######################################################################
# boilerplate
######################################################################

def Plugin( entityManager ):
   routing.plugin( entityManager )
   routing6.plugin( entityManager )

   global allVrfConfig
   global rdAutoInputDir
   global rdConfigInputDir
   global allVrfStatusLocal
   global allVrfStatusGlobal
   global ip
   global ip6
   global routingVrfConfigDir
   global routingVrfRouteConfigDir
   global routingConfig
   global routingVrfInfoDir
   global routing6VrfConfigDir
   global routing6VrfRouteConfigDir
   global routing6VrfInfoDir
   global routing6Config
   global runnabilityConfig
   global arpInputConfigCli
   global ipConfig
   global ip6Config
   global l3ConfigDir
   global routingHwStatus
   global multiAgentVrfProtoConfig

   allVrfConfig = ConfigMount.mount( entityManager, "ip/vrf/config",
                                     "Ip::AllVrfConfig", "w" )
   rdAutoInputDir = \
      LazyMount.mount( entityManager, 'ip/vrf/routeDistinguisherInputDir/auto',
                         'Tac::Dir', 'ri' )
   rdConfigInputDir = \
      ConfigMount.mount( entityManager, 'ip/vrf/routeDistinguisherInputDir/config',
                         'Tac::Dir', 'w' )

   multiAgentVrfProtoConfig = ConfigMount.mount( entityManager,
         'routing/multiAgentVrfProtocolConfig',
         'VrfTypes::MultiAgentVrfProtoConfig', 'w' )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   allVrfStatusGlobal = LazyMount.mount( entityManager, "ip/vrf/status/global",
                                         "Ip::AllVrfStatusGlobal", "r" )
   routingHwStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
   ip = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ip6 = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   routingVrfConfigDir = ConfigMount.mount( entityManager, "routing/vrf/config",
                                            "Routing::VrfConfigDir", "wi" )
   routingVrfRouteConfigDir = ConfigMount.mount( entityManager,
                                            "routing/vrf/routelistconfig",
                                            "Routing::VrfRouteListConfigDir", "wi" )
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                          "routing/vrf/routingInfo/status",
                                          "Tac::Dir", "ri" )
   routingConfig = LazyMount.mount( entityManager,
                                           "routing/config",
                                           "Routing::Config", "ri" )
   routing6VrfConfigDir = ConfigMount.mount( entityManager, "routing6/vrf/config",
                                            "Routing6::VrfConfigDir", "wi" )
   routing6VrfRouteConfigDir = ConfigMount.mount( entityManager,
                                            "routing6/vrf/routelistconfig",
                                            "Routing6::VrfRouteListConfigDir", "wi" )
   routing6VrfInfoDir = LazyMount.mount( entityManager,
                                         "routing6/vrf/routingInfo/status",
                                         "Tac::Dir", "ri" )
   routing6Config = LazyMount.mount( entityManager,
                                           "routing6/config",
                                           "Routing6::Config", "ri" )
   runnabilityConfig = ConfigMount.mount( entityManager,
                                          "routing6/runnability/config",
                                          "Routing6::Runnability::Config", "w" )
   arpInputConfigCli = ConfigMount.mount( entityManager,
                                       "arp/input/config/cli",
                                       "Arp::ArpInputConfig", "wS" )

   ipConfig = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   ip6Config = ConfigMount.mount( entityManager, "ip6/config", "Ip6::Config", "w" )
   l3ConfigDir = ConfigMount.mount( entityManager, "l3/intf/config",
                                    "L3::Intf::ConfigDir", "w" )
   VrfCli.vrfMap = VrfCli.CliVrfMapper( allVrfStatusLocal, os.getuid(), os.getgid() )
   VrfCli.vrfCmdExt = VrfCli.CliVrfCmdExtension( VrfCli.vrfMap )
   CmdExtension.addCmdExtender( VrfCli.vrfCmdExt )
   BasicCliSession.addFormatSpecifier( 'v', getVrfCliPrompt )
