#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliCommand
import CliMatcher
import CliParser
import LazyMount
import IraCommonCli
import Tac, Tracing
from TypeFuture import TacLazyType
from RoutingUtils import getUrpfGlobalHwStatus

routingHardwareStatus = None
routing6HardwareStatus = None
allIntfStatusDir = None
urpfHwStatusDir = None
urpf6HwStatusDir = None
aclConfig = None

urpfStatus = TacLazyType( 'Urpf::UrpfState' )

def getUrpfStatus( routingHwStatus, statusValue ):
   urpfStatusToCliString = {
         urpfStatus.urpfDisabled: 'disabled',
         urpfStatus.urpfEnabled: 'enabled',
         urpfStatus.urpfErrorDisabled: 'errorDisabled',
      }
   if routingHwStatus.urpfSupported:
      return urpfStatusToCliString[ statusValue ]
   return None

def getV4UrpfStatus():
   return getUrpfStatus( routingHardwareStatus,
                         getUrpfGlobalHwStatus( urpfHwStatusDir ) )

def getV6UrpfStatus():
   return getUrpfStatus( routing6HardwareStatus,
                         getUrpfGlobalHwStatus( urpf6HwStatusDir ) )

def urpf4SupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.urpfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform


def urpf6SupportedGuard( mode, token ):
   rhs = routing6HardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing6' ][ 'hardware' ][ 'status' ]
   if rhs.urpfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def urpfStrictModeSupported( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.urpfStrictModeSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def urpfLooseModeSupported( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.urpfLooseModeSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def urpf4ExceptionSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.urpfExceptionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def urpf6ExceptionSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing6' ][ 'hardware' ][ 'status' ]
   if rhs.urpfExceptionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def urpfExceptionSupportedGuard( mode, token ):
   urpf4Support = urpf4ExceptionSupportedGuard( mode, token )
   urpf6Support = urpf6ExceptionSupportedGuard( mode, token )
   if urpf4Support is None or urpf6Support is None:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getUserAclNames( types ):
   # Return generator object for names of all user acl of given aclType(s).
   # types : list of aclTypes : [ 'ip', 'ipv6', 'mac' ]
   # name : generator object
   for aclType in types:
      for name, value in aclConfig.config[ aclType ].acl.iteritems():
         if not value.readonly:
            yield name

urpfSourceMatcher = CliMatcher.KeywordMatcher( 'source',
                                          helpdesc='RPF mode' )
urpfReachMatcher = CliMatcher.KeywordMatcher( 'reachable-via',
                                          helpdesc='RPF mode' )
urpfMode =  Tac.Type( "Urpf::UrpfMode" )
urpfStrictMatcher = CliMatcher.KeywordMatcher( 'rx',
      helpdesc='Set RPF check to strict-mode' )
urpfLooseMatcher = CliCommand.guardedKeyword( 'any',
      helpdesc='Set RPF check to loose-mode',
      guard=urpfLooseModeSupported )
urpfAllowDefaultMatcher = CliMatcher.KeywordMatcher( 'allow-default',
      helpdesc='Do uRPF check against default route too' )

def doUrpf( intf, no=False, rpfMode=False, allowDefault=False, aclName=None ):
   cfg = intf.config()
   urpfEnum = Tac.Type( 'Urpf::UrpfMode' )
   mode = urpfEnum.disable
   if aclName is None:
      aclName = ""

   if no == 'default':
      mode = urpfEnum.disable
      aclName = ""
   else:
      if no:
         mode = urpfEnum.disable
         aclName = ""
      elif rpfMode == 'loose':
         assert not allowDefault, 'allow-default not allowed in loose-mode'
         mode = urpfEnum.loose
      elif rpfMode == 'strict':
         if allowDefault:
            mode = urpfEnum.strictDefault
         else:
            mode = urpfEnum.strict
      if not no:
         rhs = routingHardwareStatus
         if not rhs.urpfSupportedOnAllFaps:
            intf.mode.addWarning(
               'Some linecards do not support uRPF on this switch' )

   cfg.urpf = Tac.Value( 'Urpf::UrpfConfig', mode=mode, acl=aclName )
   Tracing.trace0( intf.name, " uRPF mode: ", mode, " uRPF acl: ", aclName )

   IraCommonCli.printWarningIfNotRoutedPort( intf.mode,
                                             intf.name, configType='uRPF' )

def showUrpfStatus( v6 ):
   # Display URPF Status information
   if ( not v6 and routingHardwareStatus.urpfSupported ) or \
      ( v6 and routing6HardwareStatus.urpfSupported ):

      def getUrpfStatusString( status ):
         if status == urpfStatus.urpfDisabled:
            return "Disabled"
         elif status == urpfStatus.urpfEnabled:
            return "Enabled"
         elif status == urpfStatus.urpfErrorDisabled:
            return "Error Disabled"
         else:
            assert True

      if not v6:
         urpfStatusString = \
         getUrpfStatusString( getUrpfGlobalHwStatus( urpfHwStatusDir ) )
         print "IP uRPF : %s" % ( urpfStatusString, )
      else:
         urpfStatusString = \
         getUrpfStatusString( getUrpfGlobalHwStatus( urpf6HwStatusDir ) )
         print "IPv6 uRPF : %s" % ( urpfStatusString, )

def showV4UrpfStatus():
   showUrpfStatus( False )

def showV6UrpfStatus () :
   showUrpfStatus( True )

def Plugin( entityManager ):
   global routingHardwareStatus, routing6HardwareStatus, urpfHwStatusDir, \
       urpf6HwStatusDir, allIntfStatusDir, aclConfig
   # Force mount routingHardwareStatus as it is needed by doUrpf
   routingHardwareStatus = entityManager.mount( "routing/hardware/status",
                                                "Routing::Hardware::Status", "r" )
   routing6HardwareStatus = LazyMount.mount( entityManager,
                                            "routing6/hardware/status",
                                            "Routing6::Hardware::Status", "r" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   urpfHwStatusDir = LazyMount.mount( entityManager, "routing/hardware/urpfStatus",
                                      "Tac::Dir", "ri" )
   urpf6HwStatusDir = LazyMount.mount( entityManager, "routing6/hardware/urpfStatus",
                                       "Tac::Dir", "ri" )
   aclConfig = LazyMount.mount( entityManager, "acl/config/cli",
                                "Acl::Input::Config", "r" )
