# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements ipv4 and ipv6 routing commands
#-------------------------------------------------------------------------------
'''Routing configuration commands support for IPv4 and IPv6'''

import CliParser, IntfCli, Arnet, LazyMount, Tac, socket, IpUtils, Cell
import TacSigint
import ConfigMount
import SmashLazyMount
import SharedMem
import Smash
from CliModel import (
      Dict,
      Int,
      Model,
      Str,
      List,
      Bool,
      Enum,
)
from IntfModel import Interface
from ArnetModel import IpGenericAddress
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
from RoutingConsts import (
      defaultStaticRouteName,
      defaultStaticRoutePreference,
      defaultStaticRouteTag,
      defaultStaticRouteMetric
)
from SrTePolicyCommonLib import srTePolicyStatusPath, cbfFecOverrideMapPath
import sys
import TableOutput
import Tracing
from TypeFuture import TacLazyType
from CliPlugin.TunnelCli import readMountTunnelTable
from CliPlugin.TunnelModels import tunnelTypeStrDict
from CliPlugin.VrfCli import vrfExists
from CliPrint import CliPrint
from itertools import chain

printer = CliPrint().lib

t0 = Tracing.trace0

# Modified Ira Cli plugins to support Smash routing/status and forwarding/status.
# Supports only Ip4 with default Vrfs.
# Also Supports Ip4 with non-defaults Vrfs
# non-default rStatus() and fStatus() Smash entity is being
# dereferenced directly when iterating the route table for non-default VRFs
# need to be careful that we don't dereference a NULL rStatus() and fStatus()
# BUG112217

DynTunnelIntfId = Tac.Type( "Arnet::DynamicTunnelIntfId" )
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )
TunnelId = Tac.Type( "Tunnel::TunnelTable::TunnelId" )
FecId = Tac.Type( 'Smash::Fib::FecId' )
AdjType = Tac.Type( 'Smash::Fib::AdjType' )
orderedNexthopsFlag = Tac.enumValue( "Smash::Fib::FecFlags", 'orderedNexthops' )

def ipDottedToLong( ipAddress ):
   ipaddr = ipAddress.split( '.' )
   binaryipOuter = int( ipaddr[ 0 ] )
   for i in range( 1, 4 ):
      binaryipOuter = binaryipOuter << 8 | int( ipaddr[ i ] )
   return binaryipOuter

class IpVslReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vsl, master ):
      self.vsl_ = vsl
      self.master_ = master
      Tac.Notifiee.__init__( self, vsl )

   @Tac.handler( 'state' )
   def handleState( self ):
      self.master_.handleVrfState( self.vsl_ )

class Ip4( object ):
   def __init__ ( self ):
      self.status = None
      self.config = None
      self.rtConfig = None
      self.routeConfig = None
      self.rtInputConfigDir = None
      self.rtHwStatus = None
      self.rtHwRouteStatus = None
      self.arpStaticNeighConfig = None
      self.neighborConfig = None
      self.neighborStatus = None
      self.intfAddrRoutesSupported = True
      self.routingVrfConfigDir = None
      self.vrfRouteConfigDir = None
      self.arpInputConfigCli = None
      self.kernelUnprogrammedRoute = None
      self.allNDVrfKernelUnprogrammedRoute = None
      self.defaultRoute = Arnet.Prefix( '0.0.0.0/0' )
      self.shmemEm = None
      self.fibRoutingStatus = None
      self.fibForwardingStatus = None
      self.fibForwarding6Status = None
      self.fibUnprogrammedRouteStatus = None
      self.policyForwardingStatus = None
      self.tunnelFib = None
      self.nexthopEntryStatus = None
      self.allIntfConfigDir = None
      self.forwardingVrfStatusDir = None
      self.allVrfStatusLocal = None
      self.l3ConfigDir = None
      self.l3Config = None
      self.evpnVxlanTunnelTable = None
      self.routeCacheUnprogrammedRoute = None
      self.routeCacheRouteConfig = None
      self.routeCacheViaConfig = None
      self.ldpRouteStatus = None

      self.vrfsMounted = dict()
      self.vrfNameToId = dict()
      self.islReactor = None
      self.bfdStatus = None
      self.cbfFecOverrideMap = None
      self.srTePolicyStatus = None
      self.ipStr = 'IP'
      self.addVrfDict = True

   @staticmethod
   def rtkey( prefix, preference ):
      return Tac.Value( 'Routing::RouteKey', 
                        prefix=prefix, preference=preference )

   @staticmethod
   def subnet( prefix ):
      return Arnet.Subnet( prefix ).__str__()

   @staticmethod
   def containsSubnet(  outer_, inner_ ):
      '''checks if outer_ contains inner_
      '''
      outerPfx = outer_.split( '/' )
      innerPfx = inner_.split( '/' )
      outerPfxLen = int( outerPfx[ 1 ] )
      innerPfxLen = int( innerPfx[ 1 ] )
      if innerPfxLen < outerPfxLen:
         return False 

      ipAddrOuter = ipDottedToLong( outerPfx[ 0 ] )
      ipAddrInner = ipDottedToLong( innerPfx[ 0 ] )

      allOnesMask = 0xffffffff
      outerNetMask = allOnesMask << ( 32 - int( outerPfxLen ) )

      subnetOuter = ipAddrOuter & outerNetMask
      subnetInner = ipAddrInner & outerNetMask
      
      if subnetInner == subnetOuter:
         return True

      return False

   @staticmethod
   def compareAddress( a, b ):
      return IpUtils.compareIpAddress( a, b )

   def mount( self, entityManager ):
      # TODO prakash make this vrf specific.
      self.tunnelFib = SmashLazyMount.mount(
         entityManager, 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
         SmashLazyMount.mountInfo( 'reader' ) )
      self.shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      self.status = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
      self.config = ConfigMount.mount( entityManager, "ip/config", "Ip::Config",
                                       "w" )
      self.rtConfig = ConfigMount.mount(
         entityManager, "routing/config", "Routing::Config", "wS" )
      self.routeConfig = ConfigMount.mount(
         entityManager, "routing/routelistconfig", "Routing::RouteListConfig", "wS" )
      self.rtHwRouteStatus = LazyMount.mount(
                                    entityManager, "routing/hardware/route/status", 
                                              "Routing::Hardware::RouteStatus", "r" )
      self.routingVrfConfigDir = ConfigMount.mount( entityManager,
                                                    "routing/vrf/config",
                                                    "Routing::VrfConfigDir", "wS" )
      self.vrfRouteConfigDir = ConfigMount.mount( entityManager,
                                             "routing/vrf/routelistconfig",
                                             "Routing::VrfRouteListConfigDir", "wS" )

      self.allIntfConfigDir = LazyMount.mount( entityManager,
                                            "interface/config/all",
                                            "Interface::AllIntfConfigDir",
                                            "r" )
      self.allIntfStatusDir = LazyMount.mount( entityManager,
                                            "interface/status/all",
                                            "Interface::AllIntfStatusDir",
                                            "r" )
      self.bfdStatus = LazyMount.mount( entityManager, "routing/rib/staticBfdStatus",
                                        "Tac::Dir", "ri" )
      self.attachRouteStatus = SmashLazyMount.mount( entityManager,
                                               "routing/attachedRouteStatus",
                                               "Routing::AttachedRouteStatus",
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.vrfIdMap = SmashLazyMount.mount( entityManager, "vrf/vrfIdMapStatus",
                                            "Vrf::VrfIdMap::Status",
                                            SmashLazyMount.mountInfo( 'reader' ) )
      self.fibRoutingStatus = SmashLazyMount.mount( entityManager,
                                               'routing/status',
                                               'Smash::Fib::RouteStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.fibForwardingStatus = SmashLazyMount.mount( entityManager,
                                               'forwarding/status',
                                               'Smash::Fib::ForwardingStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.fibForwarding6Status = SmashLazyMount.mount( entityManager,
                                               'forwarding6/status',
                                               'Smash::Fib6::ForwardingStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.unifiedForwardingStatus = SmashLazyMount.mount( entityManager,
                                             "forwarding/unifiedStatus",
                                             "Smash::Fib::ForwardingStatus",
                                             SmashLazyMount.mountInfo( 'reader' ) )
      self.unifiedForwarding6Status = SmashLazyMount.mount( entityManager,
                                             "forwarding6/unifiedStatus",
                                             "Smash::Fib6::ForwardingStatus",
                                             SmashLazyMount.mountInfo( 'reader' ) )
      self.fibUnprogrammedRouteStatus = SmashLazyMount.mount( entityManager,
                                      'routing/unprogrammedRouteStatus',
                                      'Smash::Fib::UnprogrammedRouteStatus',
                                      SmashLazyMount.mountInfo( 'reader' ) )
      self.routeCacheRouteConfig = SmashLazyMount.mount( entityManager,
                               'routing/rib/routeConfig/ipv4/staticRouteCacheConfig',
                               'Routing::Rib::RouteConfig',
                                SmashLazyMount.mountInfo( 'reader' ) )
      self.routeCacheUnprogrammedRoute = SmashLazyMount.mount( entityManager,
                  'routing/rib/routeConfig/ipv4/unprogrammedStaticRouteCacheConfig',
                  'Routing::Rib::RouteConfig', SmashLazyMount.mountInfo( 'reader' ) )
      self.routeCacheViaConfig = SmashLazyMount.mount( entityManager,
                               'routing/rib/viaConfig/evpn/staticRouteCacheConfig',
                               'Routing::Rib::RouteCache::Evpn::ViaConfig',
                                SmashLazyMount.mountInfo( 'reader' ) )
      self.policyForwardingStatus = SmashLazyMount.mount( entityManager,
                                               'forwarding/srte/status',
                                               'Smash::Fib::ForwardingStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.evpnVxlanTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.evpnVxlanTunnelTable, entityManager, lazy=False )
      self.srTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.srTunnelTable, entityManager, lazy=False )
      self.staticTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.staticTunnelTable, entityManager, lazy=False )
      self.ldpTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.ldpTunnelTable, entityManager, lazy=False )
      self.bgpLuTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.bgpLuTunnelTable, entityManager, lazy=False )
      self.nexthopGroupTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.nexthopGroupTunnelTable, entityManager, lazy=False )
      self.rsvpLerTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.rsvpLerTunnelTable, entityManager, lazy=False )
      self.nexthopEntryStatus = SmashLazyMount.mount( entityManager,
                                               'routing/nexthopgroup/entrystatus',
                                               'NexthopGroup::EntryStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.kernelUnprogrammedRoute = SmashLazyMount.mount( entityManager,
                              "routing/kernel/unprogrammedRoute",
                              "Smash::KernelUnprogrammedRoute::UnprogrammedRoute",
                              SmashLazyMount.mountInfo( 'reader' ) )
      self.allNDVrfKernelUnprogrammedRoute = self.shmemEm.doMount(
            "routing/vrf/kernel/unprogrammedRoute",
            "Smash::KernelUnprogrammedRoute::UnprogrammedRoute",
            Smash.mountInfo( 'reader' ) )
      self.l3ConfigDir = ConfigMount.mount( entityManager, 'l3/intf/config',
                                            'L3::Intf::ConfigDir', 'wi' )
      self.l3Config = LazyMount.mount( entityManager,
                                       "l3/config", "L3::Config", "ri" )
      normalMg = entityManager.mountGroup()
      # routing/hardware/status is used by IraIpRouteCliHelper manageRoute function
      # that can be called from inside the activity loop, this means we cannot
      # use a LazyMount for it.
      self.rtHwStatus = normalMg.mount( "routing/hardware/status",
            "Routing::Hardware::Status", "r" )
      self.allVrfStatusLocal = normalMg.mount( Cell.path( 'ip/vrf/status/local' ),
                                               'Ip::AllVrfStatusLocal', 'r' )
      self.rtInputConfigDir = normalMg.mount( 'routing/vrf/input', 
                                              'Tac::Dir', 'ri' )
      self.cbfFecOverrideMap = LazyMount.mount( entityManager,
            cbfFecOverrideMapPath(), 'SrTePolicy::CbfFecOverrideMap', 'ri' )
      self.srTePolicyStatus = SmashLazyMount.mount( entityManager,
                                         srTePolicyStatusPath(),
                                         "SrTePolicy::PolicyStatus",
                                         SmashLazyMount.mountInfo( 'reader' ) )
      self.ldpRoutingStatus = SmashLazyMount.mount( entityManager,
                                               'ldp-over-rsvp/vrf/default/v4/status',
                                               'Smash::Fib::RouteStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )

      def _done():
         self.islReactor = Tac.collectionChangeReactor(
            self.allVrfStatusLocal.vrf, IpVslReactor, reactorArgs = ( self, ) )

      normalMg.close( _done )
                                                  

   def smFibRoutingStatus( self, vrfName ):
      vrfFibRoutingStatus = self.shmemEm.doMount( 'routing/vrf/status/%s' % vrfName,
                                                  'Smash::Fib::RouteStatus',
                                                  Smash.mountInfo( 'reader' ) )
      self.vrfsMounted[ vrfName ] = True
      return vrfFibRoutingStatus

   def smFibForwardingStatus( self, vrfName ):
      vrfFibForwardingStatus = self.shmemEm.doMount(
         'forwarding/vrf/status/%s' % vrfName, 'Smash::Fib::ForwardingStatus',
         Smash.mountInfo( 'reader' ) )
      self.vrfsMounted[ vrfName ] = True
      return vrfFibForwardingStatus

   def smFibForwarding6Status( self, vrfName ):
      forwardingPath = 'forwarding6/vrf/status/%s' % vrfName
      vrfFibForwarding6Status = self.shmemEm.doMount(
         forwardingPath, 'Smash::Fib6::ForwardingStatus',
         Smash.mountInfo( 'reader' ) )
      self.vrfsMounted[ vrfName ] = True
      return vrfFibForwarding6Status

   def smFibUnprogrammedRouteStatus( self, vrfName ):
      return self.fibUnprogrammedRouteStatus

   def smKernelUnprogrammedRoute( self, vrfName ):
      return self.allNDVrfKernelUnprogrammedRoute

   def handleVrfState( self, vsl ):
      assert vsl
      if vsl.state == 'active':
         return

      vrfName = vsl.vrfName

      if vrfName in self.vrfNameToId:
         del self.vrfNameToId[ vrfName ]

      if vrfName in self.vrfsMounted:
         del self.vrfsMounted[ vrfName ]
         self.shmemEm.doUnmount( 'routing/vrf/status/%s' % vrfName )
         self.shmemEm.doUnmount( 'forwarding/vrf/status/%s' % vrfName )
         self.shmemEm.doUnmount( 'forwarding6/vrf/status/%s' % vrfName )

   def routingConfig( self, vrfName ):
      assert vrfName and vrfName != ''
      if vrfName == DEFAULT_VRF:
         return self.rtConfig
      else:
         return self.routingVrfConfigDir.vrfConfig.get( vrfName )

   def getRouteConfig( self, vrfName ):
      if vrfName == DEFAULT_VRF:
         return self.routeConfig
      else:
         return self.vrfRouteConfigDir.vrfConfig.get( vrfName )

   def routingInputConfig( self, vrfName ):
      routeConfList = []
      if not self.rtInputConfigDir.has_key( vrfName ):
         return routeConfList
      rtInputDir = self.rtInputConfigDir.entity[ vrfName ]
      subDirList = [ s for s in rtInputDir.keys() if "EosSdk" in s ]
      for subDir in subDirList:
         routeConfList.append( rtInputDir.entity[ subDir ] )
      return routeConfList

   def routingHardwareConfig( self, vrfName ):
      return None

   def routingHardwareStatus( self ):
      return self.rtHwStatus

   def staticBfdStatus( self, vrfName ):
      assert vrfName and vrfName != ''
      return self.bfdStatus.get( vrfName )

   def routingHardwareRouteStatus( self ):
      return self.rtHwRouteStatus

   def rtList( self ):
      rtl = [ self.routeConfig ]
      for rt in self.vrfRouteConfigDir.vrfConfig.itervalues():
         rtl.append( rt )
      return rtl

class Ip6( object ):
   def __init__ ( self ):
      self.status = None
      self.config = None
      self.rtConfig = None
      self.routeConfig = None
      self.rtInputConfigDir = None
      self.rtHwStatus = None
      self.intfAddrRoutesSupported = True
      self.kernelUnprogrammedRoute = None
      self.allNDVrfKernelUnprogrammedRoute = None
      self.arpStaticNeighConfig = None
      self.arpStaticNeighStatus = None
      self.arpInputConfigCli = None
      self.defaultRoute = Arnet.Ip6Prefix( '::/0' )
      self.routingVrfConfigDir = None
      self.vrfRouteConfigDir = None
      self.fibRoutingStatus = None
      self.fibRouting4Status = None
      self.fibForwardingStatus = None
      self.fibForwarding6Status = None
      self.fibUnprogrammedRouteStatus = None
      self.policyForwardingStatus = None
      self.tunnelFib = None
      self.nexthopEntryStatus = None
      self.allIntfConfigDir = None
      self.l3ConfigDir = None
      self.l3Config = None
      self.evpnVxlanTunnelTable = None
      self.routeCacheRouteConfig = None
      self.routeCacheViaConfig = None
      self.routeCacheUnprogrammedRoute = None
      self.shmemEm = None
      self.fwdInfo = None
      self.forwardingVrfStatusDir = None
      self.allVrfStatusLocal = None

      self.vrfsMounted = dict()
      self.vrfNameToId = dict()
      self.islReactor = None
      self.bfdStatus = None
      self.cbfFecOverrideMap = None
      self.srTePolicyStatus = None
      self.ipStr = 'IPv6'
      self.addVrfDict = True

   @staticmethod
   def rtkey( prefix, preference ):
      return Tac.Value( 'Routing::RouteKey',
                 prefix=prefix, preference=preference )

   @staticmethod
   def subnet( prefix ):
      return Arnet.Ip6AddrWithMask( prefix.stringValue ).subnet.stringValue

   @staticmethod
   def containsSubnet( outer, inner ):
      outer = Arnet.Subnet( outer, addrFamily=socket.AF_INET6 )
      inner  = Arnet.Subnet( inner, addrFamily=socket.AF_INET6 )

      return outer.containsAddr( inner.address_ ) and \
          outer.mask_.masklen_ <= inner.mask_.masklen_

   @staticmethod
   def compareAddress( a, b ):
      # BUG19710 We don't have a standard function to compare two IPv6 addresses;
      # until we do, just use the default string ordering.
      return cmp( a, b )

   def mount( self, entityManager ):
      # TODO prakash make this vrf specific.
      self.tunnelFib = SmashLazyMount.mount(
         entityManager, 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
         SmashLazyMount.mountInfo( 'reader' ) )
      self.status = LazyMount.mount(
         entityManager, "ip6/status", "Ip6::Status", "r" )
      self.config = ConfigMount.mount(
         entityManager, "ip6/config", "Ip6::Config", "w" )
      self.rtConfig = ConfigMount.mount(
         entityManager, "routing6/config", "Routing6::Config", "w" )
      self.routeConfig = ConfigMount.mount(
         entityManager, "routing6/routelistconfig",
         "Routing6::RouteListConfig", "w" )
      self.routingVrfConfigDir = ConfigMount.mount( entityManager,
                                                    "routing6/vrf/config",
                                                    "Routing6::VrfConfigDir", "wi" )
      self.vrfRouteConfigDir = ConfigMount.mount( entityManager,
                                           "routing6/vrf/routelistconfig",
                                           "Routing6::VrfRouteListConfigDir", "wi" )
      self.rtHwStatus = LazyMount.mount(
         entityManager, "routing6/hardware/status", 
         "Routing6::Hardware::Status", "r" )
      self.rtHwConfig = ConfigMount.mount(
         entityManager, "routing6/hardware/config", 
         "Routing6::Hardware::Config", "w" )
      self.allIntfConfigDir = LazyMount.mount( entityManager,
                                            "interface/config/all",
                                            "Interface::AllIntfConfigDir",
                                            "r" )
      self.allIntfStatusDir = LazyMount.mount( entityManager,
                                            "interface/status/all",
                                            "Interface::AllIntfStatusDir",
                                            "r" )
      self.bfdStatus = LazyMount.mount( entityManager, "routing/rib/staticBfdStatus",
                                        "Tac::Dir", "ri" )
      self.shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      self.attachRouteStatus = SmashLazyMount.mount( entityManager,
                                               "routing6/attachedRouteStatus",
                                               "Routing::AttachedRouteStatus",
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.vrfIdMap = SmashLazyMount.mount( entityManager, "vrf/vrfIdMapStatus",
                                            "Vrf::VrfIdMap::Status",
                                            SmashLazyMount.mountInfo( 'reader' ) )
      self.fibRoutingStatus = SmashLazyMount.mount( entityManager,
                                               'routing6/status',
                                               'Smash::Fib6::RouteStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.fibRouting4Status = SmashLazyMount.mount( entityManager,
                                               'routing/status',
                                               'Smash::Fib::RouteStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.fibForwardingStatus = SmashLazyMount.mount( entityManager,
                                               'forwarding/status',
                                               'Smash::Fib::ForwardingStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.fibForwarding6Status = SmashLazyMount.mount( entityManager,
                                               'forwarding6/status',
                                               'Smash::Fib6::ForwardingStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.unifiedForwardingStatus = SmashLazyMount.mount( entityManager,
                                             "forwarding/unifiedStatus",
                                             "Smash::Fib::ForwardingStatus",
                                             SmashLazyMount.mountInfo( 'reader' ) )
      self.unifiedForwarding6Status = SmashLazyMount.mount( entityManager,
                                             "forwarding6/unifiedStatus",
                                             "Smash::Fib6::ForwardingStatus",
                                             SmashLazyMount.mountInfo( 'reader' ) )
      self.fibUnprogrammedRouteStatus = SmashLazyMount.mount( entityManager,
                                      'routing6/unprogrammedRouteStatus',
                                      'Smash::Fib6::UnprogrammedRouteStatus',
                                      SmashLazyMount.mountInfo( 'reader' ) )
      self.policyForwardingStatus = SmashLazyMount.mount( entityManager,
                                               'forwarding/srte/status',
                                               'Smash::Fib::ForwardingStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )
      self.routeCacheRouteConfig = SmashLazyMount.mount( entityManager,
                               'routing/rib/routeConfig/ipv6/staticRouteCacheConfig',
                               'Routing::Rib::RouteConfig',
                                SmashLazyMount.mountInfo( 'reader' ) )
      self.routeCacheViaConfig = SmashLazyMount.mount( entityManager,
                               'routing/rib/viaConfig/evpn/staticRouteCacheConfig',
                               'Routing::Rib::RouteCache::Evpn::ViaConfig',
                                SmashLazyMount.mountInfo( 'reader' ) )
      self.routeCacheUnprogrammedRoute = SmashLazyMount.mount( entityManager,
                  'routing/rib/routeConfig/ipv6/unprogrammedStaticRouteCacheConfig',
                  'Routing::Rib::RouteConfig', SmashLazyMount.mountInfo( 'reader' ) )
      self.evpnVxlanTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.evpnVxlanTunnelTable, entityManager )
      self.srTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.srTunnelTable, entityManager )
      self.staticTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.staticTunnelTable, entityManager )
      self.ldpTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.ldpTunnelTable, entityManager )
      self.bgpLuTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.bgpLuTunnelTable, entityManager )
      self.nexthopGroupTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.nexthopGroupTunnelTable, entityManager )
      self.rsvpLerTunnelTable = readMountTunnelTable(
         TunnelTableIdentifier.rsvpLerTunnelTable, entityManager )
      self.nexthopEntryStatus = SmashLazyMount.mount( entityManager,
                                               'routing/nexthopgroup/entrystatus',
                                               'NexthopGroup::EntryStatus',
                                               SmashLazyMount.mountInfo( 'reader' ) )

      self.kernelUnprogrammedRoute = SmashLazyMount.mount( entityManager,
                           "routing/kernel/unprogrammedRoute",
                           "Smash::KernelUnprogrammedRoute::UnprogrammedRoute",
                           SmashLazyMount.mountInfo( 'reader' ) )
      self.allNDVrfKernelUnprogrammedRoute = self.shmemEm.doMount(
            "routing/vrf/kernel/unprogrammedRoute",
            "Smash::KernelUnprogrammedRoute::UnprogrammedRoute",
            Smash.mountInfo( 'reader' ) )

      self.rtHwRouteStatus = LazyMount.mount(
         entityManager, "routing6/hardware/route/status", 
         "Routing6::Hardware::RouteStatus", "r" )

      self.l3ConfigDir = ConfigMount.mount( entityManager, 'l3/intf/config',
                                            'L3::Intf::ConfigDir', 'wi' )
      self.l3Config = LazyMount.mount( entityManager,
                                       "l3/config", "L3::Config", "ri" )

      self.arpInputConfigCli = \
          ConfigMount.mount( entityManager, "arp/input/config/cli",
                             "Arp::ArpInputConfig", "wS" )

      normalMg = entityManager.mountGroup()
      self.allVrfStatusLocal = normalMg.mount( Cell.path( 'ip/vrf/status/local' ),
                                               'Ip::AllVrfStatusLocal', 'r' )
      self.rtInputConfigDir = normalMg.mount( 'routing6/vrf/input', 
                                              'Tac::Dir', 'ri' )
      self.cbfFecOverrideMap = LazyMount.mount( entityManager,
            cbfFecOverrideMapPath(), 'SrTePolicy::CbfFecOverrideMap', 'ri' )
      self.srTePolicyStatus = SmashLazyMount.mount( entityManager,
                                         srTePolicyStatusPath(),
                                         "SrTePolicy::PolicyStatus",
                                         SmashLazyMount.mountInfo( 'reader' ) )
      def _done():
         self.islReactor = Tac.collectionChangeReactor(
            self.allVrfStatusLocal.vrf, IpVslReactor, reactorArgs = ( self, ) )

      normalMg.close( _done )

   def smFibRoutingStatus( self, vrfName ):
      vrfFibRoutingStatus = self.shmemEm.doMount( 'routing6/vrf/status/%s' % vrfName,
                                                  'Smash::Fib6::RouteStatus',
                                                  Smash.mountInfo( 'reader' ) )
      self.vrfsMounted[ vrfName ] = True
      return vrfFibRoutingStatus

   def smFibForwardingStatus( self, vrfName ):
      vrfFibForwardingStatus = self.shmemEm.doMount(
         'forwarding/vrf/status/%s' % vrfName, 'Smash::Fib::ForwardingStatus',
         Smash.mountInfo( 'reader' ) )
      self.vrfsMounted[ vrfName ] = True
      return vrfFibForwardingStatus

   def smFibForwarding6Status( self, vrfName ):
      vrfFibForwardingStatus = self.shmemEm.doMount(
         'forwarding6/vrf/status/%s' % vrfName, 'Smash::Fib6::ForwardingStatus',
         Smash.mountInfo( 'reader' ) )
      self.vrfsMounted[ vrfName ] = True
      return vrfFibForwardingStatus
   
   def smFibUnprogrammedRouteStatus( self, vrfName ):
      return self.fibUnprogrammedRouteStatus

   def smKernelUnprogrammedRoute( self, vrfName ):
      return self.allNDVrfKernelUnprogrammedRoute


   def handleVrfState( self, vsl ):
      assert vsl
      if vsl.state == 'active':
         return

      vrfName = vsl.vrfName

      if vrfName in self.vrfNameToId:
         del self.vrfNameToId[ vrfName ]

      if vrfName in self.vrfsMounted:
         del self.vrfsMounted[ vrfName ]
         self.shmemEm.doUnmount( 'routing/vrf/status/%s' % vrfName )
         self.shmemEm.doUnmount( 'routing6/vrf/status/%s' % vrfName )
         self.shmemEm.doUnmount( 'forwarding6/vrf/status/%s' % vrfName )
   
   @staticmethod
   def printHwAggRouteHeader():
      print "Codes: S - Software Forwarded"

   def arpStaticNeighborConfig( self, vrfName ):
      assert vrfName and vrfName != ''
      return self.arpInputConfigCli.vrf[ vrfName ]

   def routingConfig( self, vrfName ):
      if vrfName == DEFAULT_VRF:
         return self.rtConfig
      else:
         return self.routingVrfConfigDir.vrfConfig.get( vrfName )

   def getRouteConfig( self, vrfName ):
      if vrfName == DEFAULT_VRF:
         return self.routeConfig
      else:
         return self.vrfRouteConfigDir.vrfConfig.get( vrfName )

   def routingInputConfig( self, vrfName ):
      routeConfList = []
      if not self.rtInputConfigDir.has_key( vrfName ):
         return routeConfList
      rtInputDir = self.rtInputConfigDir.entity[ vrfName ]
      subDirList = [ s for s in rtInputDir.keys() if "EosSdk" in s ]
      for subDir in subDirList:
         routeConfList.append( rtInputDir.entity[ subDir ] )
      return routeConfList

   def smFibRouting4Status( self, vrfName ):
      vrfFibRouting4Status = self.shmemEm.doMount( 'routing/vrf/status/%s' % vrfName,
                                                   'Smash::Fib::RouteStatus',
                                                   Smash.mountInfo( 'reader' ) )
      self.vrfsMounted[ vrfName ] = True
      return vrfFibRouting4Status

   def routingHardwareStatus( self ):
      return self.rtHwStatus

   def staticBfdStatus( self, vrfName ):
      assert vrfName and vrfName != ''
      return self.bfdStatus.get( vrfName )

   def routingHardwareConfig( self, vrfName ):
      assert vrfName and vrfName != ''
      if vrfName == DEFAULT_VRF:
         return self.rtHwConfig
      else:
         # VRFVRFVRF this needs to actually work when IPv6 supports VRFs
         return None

   def rtList( self ):
      rtl = [ self.routeConfig ]
      for rt in self.vrfRouteConfigDir.vrfConfig.itervalues():
         rtl.append( rt )
      return rtl

#----------------------------------------------------------------------------
# Destroys the route and via objects associated with an interface
#----------------------------------------------------------------------------
routing4ConfType = type( Tac.newInstance( "Routing::RouteListConfig", "route4" ) )
routing6ConfType = type( Tac.newInstance( "Routing6::RouteListConfig", "route6" ) )
clearHelper = Tac.Value( "Ira::ClearIntfRoutesAndViasHelper" )
def clearIntfRoutesAndVias( routingConfigList, intfName ):
   for routeConf in routingConfigList:
      if not( isinstance( routeConf, routing4ConfType ) or
              isinstance( routeConf, routing6ConfType ) ):
         routeConf = ConfigMount.force( routeConf )

      # check came from before ConfigGen existed, likely still good to
      # have for extra safety
      if ( isinstance( routeConf, routing4ConfType ) or
           isinstance( routeConf, routing6ConfType ) ):
         clearHelper.clearIntfRoutesAndViasHelper( routeConf, intfName )
      else:
         raise TypeError( "Route Configuration type unknown" )

class routeReaper( IntfCli.IntfDependentBase ):
   def __init__( self, intf, sysdbRoot, rtList ):
      self.rtList = rtList
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   def setDefault( self ):
      clearIntfRoutesAndVias( self.rtList, self.intf_.name )

class showRoutes( object ):
   # pylint: disable-msg=W0621
   def __init__( self, routing, prefix, intf, protocol, detail, longerPrefixes, 
                 vrfName, host, degradeNhgModel, flattenTunnelOverTunnel,
                 routeFilter=None ):
      self.ip = routing.ip
      self.routing = routing
      self.prefix = prefix
      self.intf = intf
      self.boolIntf = bool (self.intf)
      self.protocol = protocol
      self.detail = detail
      self.longerPrefixes = longerPrefixes
      assert vrfName and vrfName != ''
      self.vrfName = vrfName
      self.host = host
      self.rt = routing.rStatus( vrfName )
      self.fw = routing.fStatus( vrfName )
      self.hwConfig = routing.hwConfig( vrfName )
      self.allIntfConfigDir = self.ip.allIntfConfigDir
      self.degradeNhgModel = degradeNhgModel
      self.routeFilter = routeFilter
      self.flattenTunnelOverTunnel = flattenTunnelOverTunnel

   def tacRouteSorter( self, quantify=0, originalMask=True ):
      if self.prefix is not None:
         pfx = Arnet.Prefix( self.prefix )
      else:
         pfx = Tac.Value( 'Arnet::Prefix' ).nullPrefix
      proto = ''
      if self.protocol:
         proto = self.protocol
      if self.routeFilter is None:
         self.routeFilter = Tac.newInstance( "Ira::RouteFilterColl", False )

      # Force mount all the entities that need to be
      # passed to RouteSorter and were lazy-mounted 
      # earlier.
      for entity in [ self.routing.config( self.vrfName ), 
                      self.routing.routeConfig( self.vrfName ),
                      self.allIntfConfigDir,
                      self.ip.status,
                      self.routing.hwStatus(), 
                      self.routing.tunnelFib( self.vrfName ),
                      self.ip.nexthopEntryStatus,
                      self.ip.evpnVxlanTunnelTable,
                      self.ip.srTunnelTable,
                      self.ip.staticTunnelTable,
                      self.ip.ldpTunnelTable,
                      self.ip.bgpLuTunnelTable,
                      self.ip.nexthopGroupTunnelTable,
                      self.ip.rsvpLerTunnelTable,
                      self.ip.cbfFecOverrideMap,
                      self.ip.srTePolicyStatus,
                      self.ip.l3Config,
                      self.ip.routeCacheRouteConfig,
                      self.ip.routeCacheViaConfig,
                      self.ip.routeCacheUnprogrammedRoute,
                      self.ip.ldpRouteStatus ]:
         if ( isinstance( entity, LazyMount._Proxy ) ): # pylint: disable-msg=W0212
            LazyMount.force( entity )
         elif isinstance( entity, SmashLazyMount.SmashProxy ):
            entity.mount_()

      routeSorter = Tac.newInstance( "Ira::RouteSorter",
                                     self.routing.config( self.vrfName ), 
                                     self.routing.routeConfig( self.vrfName ),
                                     self.routing.rStatus( self.vrfName ), 
                                     self.routing.fStatus( self.vrfName ),
                                     self.routing.f6Status( self.vrfName ),
                                     self.routing.tunnelFib( self.vrfName ),
                                     self.routing.attachRouteStatus( self.vrfName ),
                                     self.routing.vrfIdMap(),
                                     self.allIntfConfigDir,
                                     self.ip.status,
                                     self.routing.hwStatus(), 
                                     self.ip.nexthopEntryStatus,
                                     self.routing.kernelUnprogrammedRoute(
                                        self.vrfName ),
                                     self.ip.evpnVxlanTunnelTable,
                                     self.ip.srTunnelTable,
                                     self.ip.staticTunnelTable,
                                     self.ip.ldpTunnelTable,
                                     self.ip.bgpLuTunnelTable,
                                     self.ip.nexthopGroupTunnelTable,
                                     self.ip.rsvpLerTunnelTable,
                                     pfx,
                                     self.routing.intfStatus( self.intf ),
                                     proto, 
                                     self.detail, 
                                     self.longerPrefixes, 
                                     self.vrfName,
                                     self.routing.vrfId( self.vrfName ),
                                     self.host,
                                     sys.stdout.fileno(),
                                     quantify,
                                     originalMask,
                                     self.ip.cbfFecOverrideMap,
                                     self.ip.srTePolicyStatus,
                                     self.ip.policyForwardingStatus,
                                     self.degradeNhgModel,
                                     self.flattenTunnelOverTunnel,
                                     self.routing.unprogrammedRouteStatus(),
                                     self.routeFilter,
                                     self.ip.l3Config,
                                     self.ip.routeCacheRouteConfig,
                                     self.ip.routeCacheViaConfig,
                                     self.ip.routeCacheUnprogrammedRoute,
                                     self.routing.usingLdpRouteStatus )
      return routeSorter

   def tacRoute6Sorter( self, quantify=0, originalMask=True ):
      if self.prefix is not None:
         pfx = Arnet.Ip6Prefix( self.prefix )
      else:
         pfx = Tac.Value( 'Arnet::Ip6Prefix' ).nullPrefix
      proto = ''
      if self.protocol:
         proto = self.protocol
      if self.routeFilter is None:
         self.routeFilter = Tac.newInstance( "Ira::RouteFilterColl", False )

      # Force mount all the entities that need to be
      # passed to RouteSorter and were lazy-mounted 
      # earlier.
      for entity in [ self.routing.config( self.vrfName ), 
                      self.routing.routeConfig( self.vrfName ),
                      self.allIntfConfigDir,
                      self.ip.status,
                      self.routing.hwStatus(),
                      self.routing.tunnelFib( self.vrfName ),
                      self.ip.nexthopEntryStatus,
                      self.ip.evpnVxlanTunnelTable,
                      self.ip.srTunnelTable,
                      self.ip.staticTunnelTable,
                      self.ip.ldpTunnelTable,
                      self.ip.bgpLuTunnelTable,
                      self.ip.nexthopGroupTunnelTable,
                      self.ip.rsvpLerTunnelTable,
                      self.ip.cbfFecOverrideMap,
                      self.ip.srTePolicyStatus,
                      self.ip.l3Config,
                      self.ip.routeCacheRouteConfig,
                      self.ip.routeCacheViaConfig,
                      self.ip.routeCacheUnprogrammedRoute ]:
         # pylint: disable-msg=W0212
         if ( isinstance( entity, LazyMount._Proxy ) ):
            LazyMount.force( entity )
         elif isinstance( entity, SmashLazyMount.SmashProxy ):
            entity.mount_()

      routeSorter = Tac.newInstance( "Ira::Route6Sorter", 
                                     self.routing.config( self.vrfName ), 
                                     self.routing.routeConfig( self.vrfName ),
                                     self.routing.rStatus( self.vrfName ), 
                                     self.routing.fStatus( self.vrfName ),
                                     self.routing.f6Status( self.vrfName ),
                                     self.routing.tunnelFib( self.vrfName ),
                                     self.routing.attachRouteStatus( self.vrfName ),
                                     self.routing.vrfIdMap(),
                                     self.allIntfConfigDir,
                                     self.ip.status,
                                     self.routing.hwStatus(),
                                     self.ip.nexthopEntryStatus,
                                     self.routing.kernelUnprogrammedRoute(
                                        self.vrfName ),
                                     self.ip.evpnVxlanTunnelTable,
                                     self.ip.srTunnelTable,
                                     self.ip.staticTunnelTable,
                                     self.ip.ldpTunnelTable,
                                     self.ip.bgpLuTunnelTable,
                                     self.ip.nexthopGroupTunnelTable,
                                     self.ip.rsvpLerTunnelTable,
                                     pfx,
                                     self.routing.intfStatus( self.intf ),
                                     proto, 
                                     self.detail, 
                                     self.longerPrefixes, 
                                     self.vrfName,
                                     self.routing.vrfId( self.vrfName ),
                                     self.host,
                                     sys.stdout.fileno(),
                                     quantify,
                                     originalMask,
                                     self.boolIntf,
                                     self.ip.cbfFecOverrideMap,
                                     self.ip.srTePolicyStatus,
                                     self.ip.policyForwardingStatus,
                                     self.degradeNhgModel,
                                     self.flattenTunnelOverTunnel,
                                     self.routing.unprogrammedRouteStatus(),
                                     self.routeFilter,
                                     self.ip.l3Config,
                                     self.ip.routeCacheRouteConfig,
                                     self.ip.routeCacheViaConfig,
                                     self.ip.routeCacheUnprogrammedRoute )
      return routeSorter

   def tacPrint( self, routeSorter, fmt ):
      fd = sys.stdout.fileno()   
      with TacSigint.immediateMode():
         # this allows SIGINTs to go into the handler in CliPrint instead of being
         # hidden by CliServer
         return routeSorter.printRoutes( fd, fmt, self.routing.ip.addVrfDict )

   def routes( self ):
      if self.rt != None:
         if isinstance( self.ip, Ip4 ):
            routeSorter = self.tacRouteSorter()
         else:
            routeSorter = self.tacRoute6Sorter() 
      return routeSorter.sortedRoute

   def show( self, quantify=False, fmt=None, originalMask=True ):
      before = Tac.now() if quantify else 0
      if self.rt != None:
         if isinstance( self.ip, Ip4 ):
            routeSorter = self.tacRouteSorter( before, originalMask )
         else:
            routeSorter = self.tacRoute6Sorter( before, originalMask )
         return self.tacPrint( routeSorter, fmt )
      else:
         # go on to print next VRF anyway
         return True

   def ribOutput( self, fd, fmt, tag ):
      tacType = 'Ira::StaticRoutesRibCommon'
      routeConf = self.routing.inputConfig( self.vrfName )
      for entity in routeConf + [ self.allIntfConfigDir ]:
         if isinstance( entity, LazyMount._Proxy ): #pylint: disable=W0212
            LazyMount.force( entity )
         elif isinstance( entity, SmashLazyMount.SmashProxy ):
            entity.mount_()
      ribOutput = Tac.Value( tacType, self.ip.allIntfConfigDir )
      ribOutput.dynamic = True
      ribOutput.fd = fd
      ribOutput.fmt = fmt
      ribOutput.tag = tag
      # routeConf is a collection of RouteInput dirs created by EosSdk agents.
      # Create routeInputConf inside ribOutput using all the routeConfs.
      for conf in routeConf:
         ribOutput.routeInputConf[ conf ] = True
      return ribOutput

   def showDynamic( self, fmt, mlib, tag ):
      if self.rt is not None:
         iraConsts = Tac.Value( 'Ira::RoutingConsts',
                                defaultStaticRoutePreference,
                                defaultStaticRouteTag,
                                defaultStaticRouteName,
                                defaultStaticRouteMetric )
         # Initialize the printer with the specified format and point it to
         # stdout.  'fmt' is an integer value, either JSON or TEXT, and will
         # determine what is printed by saveStaticRoutes in CliSupport.tin
         fd = sys.stdout.fileno()
         p = mlib.initPrinter( fd, fmt, True )
         mlib.start( p )
         mlib.startDict( p, 'vrfs' )
         mlib.startDict( p, self.vrfName )

         mlib.addFrills( p, '\nVRF name: %s\n', 1, self.vrfName )

         ribOutput = self.ribOutput( fd=fd, fmt=fmt, tag=tag )
         ribOutput.dump( iraConsts )

         mlib.endDict( p, self.vrfName )
         mlib.endDict( p, 'vrfs' )
         mlib.end( p )
         mlib.deinitPrinter( p )

class showHwAggregateRoutes( object ):
   def __init__( self, _routing, ip, prefix ):
      self.hwConfig = _routing.hwConfig( DEFAULT_VRF )
      self.ip = ip
      self.prefix = prefix

   def printHwAggRoute( self, r ):
      routeString = ""
      
      if self.prefix:
         routePrefix = self.ip.subnet( r )
         outer, inner = ( routePrefix, self.prefix )
         if( not self.ip.containsSubnet( outer, inner ) ):
            return

      routeString = "S  %s" % ( r.stringValue )

      print routeString

   def printHwAggRouteHeader( self ):
      print "Codes: S - Software Forwarded"

   def sortHwAggRoutes( self ):
      return sorted( self.hwConfig.aggregateRoute.members() )

   def show( self ):
      self.printHwAggRouteHeader()
      sortedRoutes = self.sortHwAggRoutes()
      for r in sortedRoutes:
         self.printHwAggRoute( r )
      print ""
      
class FecNextHopModel( Model ):
   groupName = Str( help="The name of the next hop group", optional=True )
   groupID = Int( help="The ID of the next hop group", optional=True )
   isDynamicTunnel = Bool( help="This route is a dynamic tunnel connection",
                           optional=True )
   tunnelID = Int( help="The ID of the tunnel connection", optional=True )
   intfID = Interface( help="The Interface ID", optional=True )
   hop = IpGenericAddress( help="The hop prefix", optional=True )
   directlyConnected = Bool( help="This is a direct connection",
                             optional=True )
   internalConnection = Bool( help="The connection is internal",
                              optional=True )
   viaNull = Bool( help="There are no next hops", optional=True )
   tunnelTypeStr = Enum( help="The type of tunnel connection", optional=True,
                         values=tunnelTypeStrDict.keys() )

class FecRouteModel( Model ):
   ipv4PrefixCount = Int( help="Number of IPv4 prefixes using this FEC" )
   ipv6PrefixCount = Int( help="Number of IPv6 prefixes using this FEC" )
   nextHops = List( valueType=FecNextHopModel,
                    help="Next hops for this FEC" )
   orderedFec = Bool( help="This is an ordered FEC", optional=True )

class FecRoutesModel( Model ):
   fecRoutes = Dict( keyType=str, valueType=FecRouteModel,
                     help="A mapping of FEC ID to its routes" )

class VrfFecRoutesModel( Model ):
   vrfs = Dict( keyType=str, valueType=FecRoutesModel,
                help="Mapping of VRF name to its FECs" )
   _showVrfs = Bool( help="Display all VRF names before corresponding FEC routes" )

   def setShowVrfs( self ):
      self._showVrfs = True

   def render( self ):
      for vrf in self.vrfs:
         fecRoutes = self.vrfs[ vrf ]
         if vrf and self._showVrfs and fecRoutes.fecRoutes:
            print "VRF: " + vrf
         for ( fecId, route ) in fecRoutes.fecRoutes.items():
            print "FEC ID %s, used by %d IPv4 prefixes and %d IPv6 prefixes" % \
                  ( fecId, route.ipv4PrefixCount, route.ipv6PrefixCount )
            if route.nextHops:
               orderedStr = ""
               if route.orderedFec:
                  orderedStr = " (ordered)"
               print "Next hops" + orderedStr + ": "
               for nextHop in route.nextHops:
                  isTunnel = nextHop.isDynamicTunnel
                  if nextHop.viaNull:
                     print "    via Null0"
                  elif nextHop.groupName:
                     print "    via %s, Nexthop Group ID %s" % ( nextHop.groupName,
                                                                 nextHop.groupID )
                  elif ( nextHop.directlyConnected ) \
                        and not isTunnel and nextHop.intfID:
                     print ( "    via %s, directly connected" %
                             str( nextHop.intfID ).strip( '\'' ) )
                  elif ( nextHop.directlyConnected ) \
                        and not isTunnel and not nextHop.intfID:
                     print "    directly connected, <Internal>"
                  elif isTunnel:
                     tunnelType = tunnelTypeStrDict.get( nextHop.tunnelTypeStr )
                     print "    via %s tunnel, ID : %d" % ( tunnelType,
                                                            nextHop.tunnelID )
                  else:
                     if nextHop.intfID:
                        nextHopIntfId = str( nextHop.intfID ).strip( '\'' )
                     else:
                        nextHopIntfId = ""
                     print "    via %s, %s" % ( nextHop.hop, nextHopIntfId )
            print ""

class FecBreakdown( Model ):
   fecSize = Int( help="FEC Size" )
   numFecs = Int( help="Total number of FECs" )
   numPrefixes = Int( help="Total number of prefixes" )
   numHierarchicalRefs = Int( help="Total number of hierarchical references" )

   def tableRowVals( self ):
      return ( self.fecSize, self.numFecs, self.numPrefixes,
               self.numHierarchicalRefs )

class FecSummaryModel( Model ):
   fecBreakdowns = Dict(
      keyType=int, valueType=FecBreakdown,
      help="Breakdown of FECs, Prefixes"
           " and Hierarchical References per FEC Size" )

   def render( self ):
      headings = ( "FEC Size", "Total FECs", "Total Prefixes",
                   "Total Hierarchical References" )
      fecBreakdowns = self.fecBreakdowns

      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatRight, formatRight )

      totalFecs = 0
      if 0 in fecBreakdowns:
         totalFecs += fecBreakdowns[ 0 ].numFecs
      for key in sorted( fecBreakdowns ):
         fecBrkdw = fecBreakdowns[ key ]
         table.newRow( *fecBrkdw.tableRowVals() )
         totalFecs += fecBrkdw.fecSize * fecBrkdw.numFecs
      print table.output()
      print "Total FEC entries: {}".format( totalFecs )


class routing( object ):
   def __init__ ( self, ip ):
      self.ip = ip
      self.protoCliTable = { 'connected':'C', 'static':'S', 'kernel':'K',
                             'bgp':'B', 'rip':'R', 'aggregate':'A', 'isis':'I',
                             'vxlan-control-service': 'V', 'dhcp':'DH',
                             'martian':'M', 'dynamic-policy':'DP', 'cached':'RC' }
      self.cachedNexthopGroupIdToName = None

      # 'show ipv6 route ospf' returns OSPFv3 routes with protocol code O3
      # while 'show ip route ospf' uses protocol code 'O'. We set the right
      # code here based on the IP version
      if isinstance( ip, Ip6 ):
         self.protoCliTable[ 'ospf' ] = 'O3'
      else:
         self.protoCliTable[ 'ospfv3' ] = 'O3'
         self.protoCliTable[ 'ospf' ] = 'O'

      self.fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
      self.fecModeSm = None
      self.usingLdpRouteStatus = False

   def vrfFromIntfConfig( self, intfName ):
      try:
         ipIntfConfig = self.ip.config.ipIntfConfig.get( intfName )
      except AttributeError:
         ipIntfConfig = self.ip.config.intf.get( intfName )

      if ipIntfConfig:
         return ipIntfConfig.vrf
      return None

   # The mounts don't happen until later, so config, status, and hwStatus
   # must be functions

   def config( self, vrfName, intfName=None ):
      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
         #when adding a v6 address to an interface, we call routing4.config
         #to check if the v6 address is being used as a nexthop for an interface
         #with a v4 address. Since intfName is a v6 interface, we may not
         #find a vrfName when we call vrfFromIntfConfig especially if
         #the interface does not have a v4 address
         if vrfName == None:
            return None
      return self.ip.routingConfig( vrfName )

   def routeConfig( self, vrfName, intfName=None ):
      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
         # when adding a v6 address to an interface, we call routing4.config
         # to check if the v6 address is being used as a nexthop for an interface
         # with a v4 address. Since intfName is a v6 interface, we may not
         # find a vrfName when we call vrfFromIntfConfig especially if
         # the interface does not have a v4 address
         if vrfName == None:
            return None
      return self.ip.getRouteConfig( vrfName )

   def inputConfig( self, vrfName ):
      return self.ip.routingInputConfig( vrfName )

   def rStatus( self, vrfName, intfName=None ):
      assert vrfName and vrfName != ''
      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
      if vrfName == DEFAULT_VRF:
         return self.ip.fibRoutingStatus
      else:
         return self.ip.smFibRoutingStatus( vrfName )

   def unprogrammedRouteStatus( self ):
      return self.ip.fibUnprogrammedRouteStatus

   def kernelUnprogrammedRoute( self, vrfName, intfName=None ):
      assert vrfName and vrfName != ''
      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
      if vrfName == DEFAULT_VRF:
         return self.ip.kernelUnprogrammedRoute
      else:
         return self.ip.smKernelUnprogrammedRoute( vrfName )

   def r4Status( self, vrfName, intfName=None ):
      assert vrfName and vrfName != ''
      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
      if vrfName == DEFAULT_VRF:
         return self.ip.fibRouting4Status
      else:
         return self.ip.smFibRouting4Status( vrfName )

   def isFecModeUnified( self ):
      if self.fecModeSm is None:
         # fecModeSm is instantiated to generate the correct fecMode in fecModeStatus
         assert self.ip.l3Config is not None
         self.fecModeSm = Tac.newInstance( 'Ira::FecModeSm',
                                           self.ip.l3Config,
                                           self.fecModeStatus )

      FecMode = Tac.Type( 'Smash::Fib::FecMode' )
      return self.fecModeStatus.fecMode == FecMode.fecModeUnified

   def fStatus( self, vrfName, intfName=None ):
      assert vrfName and vrfName != ''
      if self.isFecModeUnified():
         return self.ip.unifiedForwardingStatus

      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
      if vrfName == DEFAULT_VRF:
         return self.ip.fibForwardingStatus
      else:
         return self.ip.smFibForwardingStatus( vrfName )

   def f6Status( self, vrfName, intfName=None ):
      assert vrfName and vrfName != ''
      if self.isFecModeUnified():
         return self.ip.unifiedForwarding6Status

      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
      if vrfName == DEFAULT_VRF:
         return self.ip.fibForwarding6Status
      else:
         return self.ip.smFibForwarding6Status( vrfName )

   def tunnelFib( self, vrfName=DEFAULT_VRF ):
      return self.ip.tunnelFib

   def policyForwardingStatus( self ):
      return self.ip.policyForwardingStatus

   def attachRouteStatus( self, vrfName ):
      return self.ip.attachRouteStatus

   def vrfIdMap( self ):
      return self.ip.vrfIdMap

   def vrfId( self, vrfName ):
      # vrfNameToIdMap is the cached reverse mapping
      # When the reverse mapping the inconsistent with vrfIdMap we try to update our
      # cached copy

      # Case where cache hits
      if vrfName in self.ip.vrfNameToId:
         vrfId = self.ip.vrfNameToId[ vrfName ]
         # ascertain that the vrfId is present in the vrfId -> VrfName mapping exists
         if vrfId in self.ip.vrfIdMap.vrfIdToName and \
             vrfName == self.ip.vrfIdMap.vrfIdToName[ vrfId ].vrfName:
            return vrfId
         else:
            del self.ip.vrfNameToId[ vrfName ]

      # When cache misses, loop through vrfIdMap
      for vrfId, entry in self.ip.vrfIdMap.vrfIdToName.iteritems():
         if vrfName == entry.vrfName:
            self.ip.vrfNameToId[ vrfName ] = vrfId
            return vrfId

      return Tac.Value( 'Vrf::VrfIdMap::VrfId' )

   def hwConfig( self, vrfName, intfName=None ):
      if intfName:
         assert vrfName == None
         vrfName = self.vrfFromIntfConfig( intfName )
      return self.ip.routingHardwareConfig( vrfName )

   def hwStatus( self ):
      return self.ip.routingHardwareStatus()

   def intfStatus( self, cliIntf ):
      if cliIntf:
         intfId = Tac.Value( 'Arnet::IntfId', cliIntf.name_ )
         return self.ip.allIntfStatusDir.get( intfId )
      return None
  
   def staticBfdStatus( self, vrfName ):
      return self.ip.staticBfdStatus( vrfName )

   def hwSupportedGuard( self, mode, token ):
      # VRFVRFVRF need to add VRF here once we add hardware VRF support
      if self.hwStatus().routingSupported: 
         return None
      return CliParser.guardNotThisPlatform

   def hwRoutingEcmpNotSupportedGuard( self, mode, token ):
      # VRFVRFVRF need to add VRF here once we add hardware VRF support
      if self.hwStatus().ecmpSupported == False:
         return None
      return CliParser.guardNotThisPlatform

   def reaper( self, intf, sysdbRoot ):
      return routeReaper( intf, sysdbRoot, self.ip.rtList() )

   def buildNexthopGroupIdToName( self ):
      if self.cachedNexthopGroupIdToName:
         return
      self.cachedNexthopGroupIdToName = dict()
      nexthopGroupEntry = self.ip.nexthopEntryStatus.nexthopGroupEntry
      for entry in nexthopGroupEntry:
         self.cachedNexthopGroupIdToName[ nexthopGroupEntry[ entry ].nhgId ] \
               = entry.nhgName()

   def showRoute( self, prefix, intf, protocol, detail, vrfName,
                  longerPrefixes=False, host=False, quantify=False, fmt=None,
                  originalMask=True, degradeNhgModel=False,
                  flattenTunnelOverTunnel=False, routeFilter=None ):
      if prefix:
         prefix = self.ip.subnet( prefix )
         
      if not protocol:
         protocol = ""
      protocol = self.protoCliTable.get( protocol.lower() )

      sr = showRoutes( self, prefix, intf, protocol, detail, longerPrefixes, vrfName,
                       host, degradeNhgModel, flattenTunnelOverTunnel, routeFilter )
      return sr.show( quantify, fmt, originalMask )

   def showUserRoutes( self, vrfName, fmt, mlib, tag ):
      sr = showRoutes( routing=self, prefix=None, intf=None, protocol=None,
                       detail=False, vrfName=vrfName, longerPrefixes=False,
                       host=False, degradeNhgModel=False,
                       flattenTunnelOverTunnel=False, routeFilter=None )
      sr.showDynamic( fmt, mlib, tag )

   def route( self, prefix, intf=None, vrfName=None, longerPrefixes=False ):
      assert prefix
      prefix = self.ip.subnet( prefix )
      sortedRoutes = showRoutes( self, prefix, intf, None, False, False, vrfName, 
                                 False, False, False, None ).routes()
      routeEntry = sortedRoutes.values()[ 0 ] if \
          len( sortedRoutes.values() ) > 0 else None
      return routeEntry

   def showHwAggregateRoute( self, mode, prefix ):
      if prefix:
         prefix = self.ip.subnet( prefix )
      showHwAggregateRoutes( self, self.ip, prefix ).show() 

   def showFecs( self, mode, prefix=None, vrfName=DEFAULT_VRF, summary=False ):
      """Displays information about FECs (e.g. how many prefixes point to each).

      prefix: When specified, only display fec information for all matching prefixes.
      vrfName: When specified, displays fec information for a particular vrf.
      summary: When set, displays a table output of aggregate fec information.

      Note: only the summary command will take into account internal hfecs (policy
            and tunnel fecs inside the hfec graph).
      """
      rStatus = self.rStatus( vrfName )
      fStatus = self.fStatus( vrfName )      
      f6Status = self.f6Status( vrfName )
      policyStatus = self.policyForwardingStatus()
      tunnelFib = self.tunnelFib()
      if not fStatus and not f6Status:
         return

      def countFecs( status, count ):
         for r in status.route.values():
            if hex( r.fecId ) in count:
               count[ hex( r.fecId ) ] += 1
            else:
               count[ hex( r.fecId ) ] = 1

      def getFec( fecId ):
         '''Returns a FIB Fec, policy FIB fec, or mpls tunnel entry.
         Returns None if not found.'''
         adjType = FecId( fecId ).adjType()
         if adjType == AdjType.srTePolicyAdj:
            f = policyStatus.fec.get( fecId )
         elif adjType == AdjType.tunnelFibAdj:
            tunnelId = FecId.fecIdToTunnelId( fecId )
            f = tunnelFib.entry.get( tunnelId )
         else:
            f = getFibFec( fecId )
         return f

      def getViaIterator( fecId ):
         '''Returns a FIB Fec, policy FIB fec, or mpls tunnel entry's via iterator.
         Returns None if fec is not found.'''
         fec = getFec( fecId )
         if not fec:
            return None

         adjType = FecId( fecId ).adjType()
         if adjType == AdjType.tunnelFibAdj:
            return fec.tunnelVia.itervalues()
         else:
            return fec.via.itervalues()

      def getFibFec( fecId ):
         adjType = FecId( fecId ).adjType()
         if adjType == AdjType.fibV6Adj:
            f = f6Status.fec.get( fecId )
         else:
            f = fStatus.fec.get( fecId )
         return f

      def getSumOfViaWeights( fecId ):
         '''Returns the sum of a fec or a tunnel's vias' weights. Returns None
         if fec is not found.'''
         fec = getFec( fecId )
         if not fec:
            return None

         adjType = FecId( fecId ).adjType()
         if adjType == AdjType.tunnelFibAdj:
            # BUG224067 Note that each of a tunnel's via is assumed to have
            # weight 1. This might no longer be true once a tunnel can point
            # to other Fecs and vias will have a weight of more than 1.
            return len( fec.tunnelVia )
         else:
            weight = 0
            for i in xrange( len( fec.via ) ):
               weight += fec.via[ i ].weight
            return weight

      def getFecSummaryStatistics( fecId ):
         f = getFibFec( fecId )
         if f is None:
            return ( None, None )
         cnt4 = fec4RouteCount.get( hex( fecId ), 0 )
         cnt6 = fec6RouteCount.get( hex( fecId ), 0 )
         weight = sum( f.via[ i ].weight for i in f.via )
         cnt = cnt4 if isinstance( self.ip, Ip4 ) else cnt6
         return( weight, cnt )

      def populateFecRouteModel( fecId ):
         f = getFibFec( fecId )
         if f is None:
            return
         fecRoute = FecRouteModel()
         fecRoute.ipv4PrefixCount = fec4RouteCount.get( hex( fecId ), 0 )
         fecRoute.ipv6PrefixCount = fec6RouteCount.get( hex( fecId ), 0 )
         weight = 0
         if len( f.via ):
            if f.fecFlags and orderedNexthopsFlag != 0:
               fecRoute.orderedFec = True
         for via in f.via.values():
            nextHop = FecNextHopModel()
            nextHop.isDynamicTunnel = False
            weight += via.weight
            if via.intfId.startswith( 'DynamicTunnel' ):
               nextHop.isDynamicTunnel = True
            if via.intfId == "" and via.hop == "":
               nextHop.viaNull = True
            elif via.intfId and \
                  Tac.Type( "Arnet::NexthopGroupIntfId" ) \
                  .isNexthopGroupIntfId( via.intfId ):
               self.buildNexthopGroupIdToName()
               nhgId = via.nexthopGroupId
               try:
                  nhgName = self.cachedNexthopGroupIdToName[ nhgId ]
               except KeyError:
                  nhgName = "NexthopGroupUnknown"
               nextHop.groupName = nhgName
               nextHop.groupID = nhgId
            elif not nextHop.isDynamicTunnel and \
                  ( via.hop == Arnet.Ip6Addr( "::" ) or
                    via.hop == "0.0.0.0" ):
               nextHop.directlyConnected = True
               if via.intfId:
                  nextHop.intfID = via.intfId
               else:
                  nextHop.internalConnection = True
            elif nextHop.isDynamicTunnel:
               nextHop.tunnelID = DynTunnelIntfId.tunnelId( via.intfId )
               nextHop.tunnelTypeStr = TunnelId( nextHop.tunnelID ).tunnelType()
            else:
               nextHop.hop = str( via.hop )
               nextHop.intfID = via.intfId if via.intfId else ""
            fecRoute.nextHops.append( nextHop )
         return fecRoute

      def visitVia( via, fecSummary, fecsVisited ):
         '''Processes the via as a possible Hfec. Returns true if it should be
         in the next frontier of the BFS traversal: i.e. is a fec AND has not
         been visited before.

         This is meant as a helper function for the breath-first traversal for
         populating table data for 'show ip[v6] route fec summary'.
         '''
         fecId = FecId.hierarchicalIntfIdToFecId( via.intfId )
         if not FecId( fecId ).isValid():
            return False
         fecSize = getSumOfViaWeights( fecId )
         if fecSize is None:
            return False

         # Since in the BFS traversal each edge (via) is visited once and only
         # once, count a hierarchical reference in the graph for each visit
         # (via) which points to a fec to get the correct overall count.
         fecBrkdw = fecSummary.fecBreakdowns.setdefault(
            fecSize, FecBreakdown( fecSize=fecSize, numFecs=0,
                                   numPrefixes=0, numHierarchicalRefs=0 ) )
         fecBrkdw.numHierarchicalRefs += 1

         # Here is the canonical part of BFS which visits unvisited nodes (fecs).
         if fecId in fecsVisited:
            return False
         fecBrkdw.numFecs += 1
         # There is no need to check for this fec's prefixes. Since all
         # possibly prefix-referenced fecs (ones in FIB) are visited prior to
         # the BFS (which is the only time this function is called), and
         # added to fecsVisited, all that is left is to count the tunnel and
         # policy fecs and their hierarchical references.
         fecsVisited.add( fecId )
         return True

      fec4RouteCount = {}
      fec6RouteCount = {}
      if isinstance( self.ip, Ip4 ):
         countFecs( rStatus, fec4RouteCount )
      else:
         countFecs( rStatus, fec6RouteCount )
         r4Status = self.r4Status( vrfName )
         countFecs( r4Status, fec4RouteCount )

      if prefix is not None:
         fecRoutes = FecRoutesModel()
         # First look for an exact match
         rt = rStatus.route.get( prefix )
         if rt:
            fecRoute = populateFecRouteModel( rt.fecId )
            if fecRoute:
               fecRoutes.fecRoutes[ str( rt.fecId ) ] = fecRoute
         else:
            prefix = self.ip.subnet( prefix )
            for p in rStatus.route:
               if self.ip.containsSubnet( str(p), prefix ):
                  route = rStatus.route.get( p )
                  if route is not None:
                     fecId = route.fecId
                     fecRoute = populateFecRouteModel( fecId )
                     if fecRoute:
                        fecRoutes.fecRoutes[ str( fecId ) ] = fecRoute
                     break
         return fecRoutes
      else:
         # Prepare table values for FECs in fStatus (FIB fecs).
         if isinstance( self.ip, Ip4 ):
            fecStatus = fStatus
         else:
            fecStatus = f6Status
         # Note: only the summary command will take into account internal hfecs
         # (policy and tunnel fecs inside the hfec graph).
         if summary:
            fecModel = FecSummaryModel()
            for fecId in fecStatus.fec:
               ( viaCnt, prefixes ) = getFecSummaryStatistics( fecId )
               if viaCnt is None or prefixes is None:
                  continue
               fecBrkdw = fecModel.fecBreakdowns.setdefault(
                                                            viaCnt,
                                                            FecBreakdown(
                                                               fecSize=viaCnt,
                                                               numFecs=0,
                                                               numPrefixes=0,
                                                               numHierarchicalRefs=0
                                                               )
                                                            )
               fecBrkdw.numFecs += 1
               fecBrkdw.numPrefixes += prefixes

            # Prepare table values for internal FECs using BFS traversal.
            frontier = fecStatus.fec.keys()
            fecsVisited = set( fecStatus.fec.keys() )
            while frontier:
               next_frontier = list()
               for fecId in frontier:
                  vias = getViaIterator( fecId )
                  if vias: # This is here to guard against transient state.
                     for via in vias:
                        if visitVia( via, fecModel, fecsVisited ):
                           fecId = FecId.hierarchicalIntfIdToFecId( via.intfId )
                           next_frontier.append( fecId )
               frontier = next_frontier
            return fecModel

         else:
            fecModel = FecRoutesModel()
            for fecId in fecStatus.fec:
               fecRoute = populateFecRouteModel( fecId )
               if fecRoute:
                  fecModel.fecRoutes[ str( fecId ) ] = fecRoute
            return fecModel

   def plugin( self, entityManager ):
      self.ip.mount( entityManager )
      # Setup reverse mapping from vrfName to vrfId
      self.ip.vrfNameToId = { entry.vrfName: vrfId for vrfId, entry in
            self.ip.vrfIdMap.vrfIdToName.iteritems() }
      # Cause "no interface" to remove vias and routes
      IntfCli.Intf.registerDependentClass( self.reaper, priority=11 )

class LdpRibRouting( routing ):
   '''
   helper class to print routes from special Ldp RouteStatus that has computed ISIS
   prefixes over rsvp tunnels
   '''
   def __init__( self, ip ):
      routing.__init__( self, ip )
      self.usingLdpRouteStatus = True

   def rStatus( self, vrfName, intfName=None ):
      assert vrfName == DEFAULT_VRF
      return self.ip.ldpRoutingStatus

   def showRoute( self, prefix, intf, protocol, detail, vrfName,
                  longerPrefixes=False, host=False, quantify=False, fmt=None,
                  originalMask=True, degradeNhgModel=False,
                  flattenTunnelOverTunnel=False, routeFilter=None ):
      if prefix:
         prefix = self.ip.subnet( prefix )

      assert protocol is None # Protocol filter is not supported yet
      sr = showRoutes( self, prefix, intf, protocol, detail, longerPrefixes, vrfName,
                       host, degradeNhgModel, flattenTunnelOverTunnel )
      return sr.show( fmt=fmt )

# Routine to convert URPF mode to user friendly string
def urpfModeStrings( mode ):
   if mode == 'strict':
      return ( 'Strict', 'No' )
   elif mode == 'loose':
      return ( 'Loose', 'No' )
   elif mode == 'strictDefault':
      return ( 'Strict', 'Yes' )
   else:
      return ( 'Disable', 'No' )


#------------------------------------------------------------------------------
# Routine to display a masklength histogram for "show ip[v6] route summary
#------------------------------------------------------------------------------
def showRouteSummaryMaskLengths(maskLengths):
   WIDTH = 80
   COLUMNS = 5

   if not maskLengths:
      return

   print "Number of routes per mask-length:"
   idx = 0
   total = len(maskLengths)

   print "  ",
   for maskLength, num in maskLengths:
      entry =  ("/%d: %d" % (maskLength, num))
      strlen = len(entry)
      pad = (WIDTH/COLUMNS)-strlen-3
      if pad < 1: 
         pad = 1 # make sure that there is at least a space gap
      print entry + " "*pad, 
      idx += 1
      if (idx == total): 
         print
         break
      #end if
      if (idx % COLUMNS) == 0: 
         print "\n  ",
   print 

#-------------------------------------------------------------------------------
# Determine if assigning the given address/mask to some interface in Ira
# should be allowed
# return a pair of a boolean with a string, where the string is an error or
# warning message
#-------------------------------------------------------------------------------
def canSetIntfIpHelper( config, intfName, ipAddrWithMask, af, allowOtherAf ):
   message = ""
   for route in config.route.itervalues():
      for via in route.via:
         if allowOtherAf:
            # nexthops might not be the same AF as "af"
            addrMatches = via.hop.af == af
         else:
            assert via.hop.af == af or via.hop.af == 'ipunknown'
            addrMatches = True
         if addrMatches:
            viaAddr = via.hop.v4Addr if af == 'ipv4' else via.hop.v6Addr
            if ipAddrWithMask.address == viaAddr:
               formatStr = "Warning: %(ip)s is used as the next hop of route " \
                           "%(prefix)s via %(hop)s" 
               outputValues = { "ip" : str( ipAddrWithMask.address ), 
                                "prefix" : route.prefix.stringValue,
                                "hop" : via.hop.stringValue }
               message = formatStr % outputValues
               return [ True, message ]
   return [ True, message ]

def warnIfRoutingDisabledPrinter( p, mode, vrfName=DEFAULT_VRF, routingAf=None,
                                  allVrfConfig=None ):
   # OSPF (and other protocols in future) allow creating instances even before
   # corresponding VRF is defined. In such cases, vrf/config/<vrfname> may not
   # exist yet.
   if vrfName and ( not vrfExists( vrfName ) or
                    not routingAf.config( vrfName ).routing ):
      printer.addFrills( p, "! %s routing not enabled\n" % routingAf.ip.ipStr, 0 )

def showRouteHostForVrf( p, vrf, quantify, fmt, mode, routingAf,
                         allVrfConfig ):
   if routingAf.ip.addVrfDict:
      printer.startDict( p, vrf )
   keepGoing = routingAf.showRoute( None, None, None, False, vrf, False,
                                    host=True, quantify=quantify, fmt=fmt )
   warnIfRoutingDisabledPrinter( p, mode, vrf, routingAf, allVrfConfig )
   if routingAf.ip.addVrfDict:
      printer.endDict( p, vrf )
   return keepGoing

def showRouteHostCommon( mode, vrfName=None, quantify=False,
                         allVrfStatusLocal=None, routingAf=None,
                         allVrfConfig=None ):
   requestedRev = mode.session_.requestedModelRevision()
   vrfAllSupportedRev = 2
   vrfAllSupported = True
   model = None
   if ( mode.session_.outputFormat_ == 'json' and
        requestedRev < vrfAllSupportedRev ):
      if vrfName == ALL_VRF_NAME:
         mode.addError( "'vrf all' option not supported in revision 1" )
         return model
      # requestRev is not lower than vrfAllSupportedRev
      vrfAllSupported = False
   routingAf.ip.addVrfDict = vrfAllSupported
   # based on revision and addr family, pick the model to return
   if routingAf.ip.ipStr == 'Ipv6':
      # lazy import to avoid the circular dep btw IraRouteCommon and IraIp6Model
      from CliPlugin.IraIp6Model import Ip6RoutesHost, VrfIp6RoutesHost
      if vrfAllSupported:
         model = VrfIp6RoutesHost
      else:
         model = Ip6RoutesHost
   else:
      # lazy import to avoid the circular dep btw IraRouteCommon and IraIpModel
      from CliPlugin.IraIpModel import IpRoutesHost, VrfIpRoutesHost
      if vrfAllSupported:
         model = VrfIpRoutesHost
      else:
         model = IpRoutesHost

   # create the outputFormat here, since we will not have access to mode later
   # down the stack, in routing.showRoute
   fmt = mode.session_.outputFormat()
   outputFormat = printer.stringToOutputFormat( mode.session_.outputFormat_ )
   fd = sys.stdout.fileno()
   p = printer.initPrinter( fd, outputFormat, True )
   printer.start( p )
   if routingAf.ip.addVrfDict:
      printer.startDict( p, "vrfs" )
   if vrfName == ALL_VRF_NAME:
      for vrf in chain( [ 'default' ], sorted( allVrfStatusLocal.vrf ) ):
         # make sure we don't attempt to write if SIGINT happened outside the
         # immediateMode section (since RouteSorter handles those internally)
         #
         # Additionally, "VRF: ..." header is printed inside RouterSorter C++ code
         # for safety when pager exits, see BUG284545
         TacSigint.check()
         keepGoing = showRouteHostForVrf( p, vrf, quantify, fmt, mode, routingAf,
                                          allVrfConfig )
         if not keepGoing:
            break
   else:
      keepGoing = showRouteHostForVrf( p, vrfName, quantify, fmt, mode, routingAf,
                                       allVrfConfig )
   if routingAf.ip.addVrfDict:
      printer.endDict( p, "vrfs" )
   printer.end( p )
   printer.deinitPrinter( p )
   return model
