#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import Model
from CliModel import Submodel
from CliModel import DeferredModel
from CliModel import Dict
from CliModel import Int
from CliModel import Str
from CliModel import Enum
from CliModel import List
from CliModel import Bool
from IntfModel import Interface
from ArnetModel import IpGenericAddress, MacAddress, IpGenericPrefix
from TunnelModels import TunnelTableEntry
from TableOutput import createTable, Format

from NexthopGroupConsts import (
      ipTunnelNexthopGroupTypes,
      NexthopGroupType,
      EntryCounterType,
      GreKeyType,
)

IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )

class L2Via( Model ):
   vlanId = Int( help="Vlan Id" )
   l2Intf = Interface( help="Egress L2 interface" )
   macAddr = MacAddress( help="Destination MAC address" )


class Via( Model ):
   entryIndex = Int( help='Nexthop group entry index' )
   nexthopGroupType = Enum( optional=True, help='Nexthop group type',
                            values=NexthopGroupType.attributes )
   unresolvedNexthopAction = Enum( values=( 'drop', 'notProgrammed' ),
                                   help='Nexthop programming action if not resolved',
                                   optional=True )
   nexthopAddr = IpGenericAddress( help='Nexthop address' )
   route = IpGenericPrefix( help='Route used to resolve destination' )
   l3Intf = Interface( help="Egress L3 interface" )
   destinationAddr = IpGenericAddress( help='Destination Ip address' )
   mplsLabelStack = List( optional=True,
                          valueType=int,
                          help="""MPLS label stack to push on to the packet.
                                  Left most label is the top of the stack.""" )
   mplsLabelOperation = Enum( optional = True, help='Label Stack Operation',
                                   values=['push', 'pop', 'swap', 'unknown'] )
   l2Via = Submodel( valueType=L2Via, help="L2 via" )
   packets = Int( help='Number of packets matched', optional=True )
   bytes = Int( help='Number of bytes matched', optional=True )
   sharedTunnel = Bool( help='True for shared tunnel', optional=True )
   isTunnelVia = Bool( help='True if destination is a tunnel' )
   nextLevelFecIndex = Int( help='Next level FIB FEC if the via is hierarchical',
                            optional=True )
   platformNextLevelFecIndex = Int( help='Platform-specific version of the next '
                                         'level FIB FEC', optional=True )

nhgInfoCliStr = '''\
%s
  Id         %s
  Type       %s
  Size       %d
  TTL        %d
  Source IP  %s'''

entry = '  Entries (left most label is the top of the stack)'
entryList = '    %d  %s' #eg: '    0  12.20.0.1'

def nhgSummary( name, nhgType, size, ngid=None, ttl=None,
                srcIp=None, srcIntf=None,
                entryCounterType=None ):

   if not ngid:
      ngid = r'\d+'
   ngid = str( ngid )
   if nhgType == NexthopGroupType.ip:
      ttl = 0
      srcIp = ''
      srcIntf = ''
   nhgSummaryStr = nhgInfoCliStr % ( name, ngid, nhgType, size, ttl, srcIp,
                                     entryCounterType )
   if srcIntf:
      nhgSummaryStr += ' (%s)' % srcIntf
   if nhgType not in ipTunnelNexthopGroupTypes:
      # Strip off TTL and source IP fields for non ip-tunnel types
      nhgSummaryStr = nhgSummaryStr[ : nhgSummaryStr.rindex( 'TTL' ) ].rstrip()
   if entryCounterType:
      nhgSummaryStr += '\n  Counters   %s' % entryCounterType
   return nhgSummaryStr

def showNhgRegex( name, nhgType, size, ngid=None, ttl=None, srcIp=None,
                  dstIpList=None, entryCounterType=None ):
   nhgSummaryStr = nhgSummary( name, nhgType, size, ngid, ttl, srcIp,
                               entryCounterType )
   nhgRegex = nhgSummaryStr.replace( "\n", '' )
   nhgRegex += entry
   if dstIpList:
      assert len( dstIpList ) == size
      for i in range( size ):
         nhgRegex += entryList % ( i, dstIpList[ i ] )
         nhgRegex += '.*'
   return nhgRegex


class NexthopGroup( Model ):
   nexthopGroupName = Str( help='Nexthop group name' )
   nexthopGroupId = Int( help='Nexthop group Id' )
   nexthopGroupType = Enum( help='Nexthop group type',
                            values=NexthopGroupType.attributes )
   size = Int( help='Nexthop group entry size' )
   numProgrammedTunnels = Int( help='Number of tunnels programmed in hardware',
                               optional=True)
   autoSize = Bool( help='Nexthop group auto resizing', optional=True )
   ttl = Int( help='Time to live value' )
   # Source intf takes precedence over source IP configuration
   # if source intf doesn't exist, source IP is taken from config
   sourceIp = IpGenericAddress( help='Source Ip address' )
   sourceIntf = Interface( help='Source interface', optional=True )
   via = Dict( keyType=int, valueType=Via, 
               help="Mapping between entry index, and via" )
   greKeyType = Enum( help='GRE key type', 
                      values=GreKeyType.attributes,
                      optional=True )
   programmed = Bool( help="Whether the group has been programmed in hardware",
                      optional=True )
   destinationIps = Dict( keyType=int, valueType=str,
                          help='Destination Ip address list' )
   entryCounterType = Enum( help='Entry counter type',
                            values=EntryCounterType.attributes,
                            optional=True )
   hierarchicalFecsEnabled = Bool( help='True if hierarchical fecs are enabled '
                                        'for this nexthop group', optional=True )

class NexthopGroups( DeferredModel ):
   nexthopGroups = Dict( keyType=str, valueType=NexthopGroup,
                   help='Mapping between nexthop group name and status' )

class NexthopGroupSummary( DeferredModel ):
   numNexthopGroups = Int( help='Number of nexthop groups configured' )
   numUnprogrammedNexthopGroups = Int( help='Number of unprogrammed nexthop groups' )
   numNexthopGroupsPerType = Dict( keyType=str, valueType=int,
                                  help='Number of nexthop groups configured '
                                  'indexed by nexthop group type' )
   numNexthopGroupsPerSize = Dict( keyType=str, valueType=int,
                                   help='Number of nexthop groups configured '
                                   'indexed by number of entries' )

class NexthopGroupTunnelTableEntry( TunnelTableEntry ):
   nexthopGroupName = Str( help='Nexthop group name' )
   igpMetric = Int( help="IGP Metric for tunnel entry" )
   igpPref = Int( help="IGP Preference for tunnel entry" )

   def renderNexthopGroupTunnelTableEntry( self, table, tunnelIndex ):
      table.newRow( tunnelIndex, str( self.endpoint ), str( self.nexthopGroupName ),
                    str( self.igpMetric ), str( self.igpPref ) )

class NexthopGroupTunnelTable( Model ):
   indexes = Dict( keyType=long, valueType=NexthopGroupTunnelTableEntry,
                   help="Nexthop Group tunnel entries keyed by tunnel index" )

   def render( self ):
      headings = ( "Index", "Endpoint", "Nexthop Group",
                   "IGP Metric", "IGP Preference" )
      fl = Format( justify='left' )
      fl.noPadLeftIs( True )
      fr = Format( justify='right' )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      table = createTable( headings )
      table.formatColumns( fl, fl, fl, fr, fr )
      for tunnelIndex, nhgTunnelTableEntry in sorted( self.indexes.iteritems() ):
         nhgTunnelTableEntry.renderNexthopGroupTunnelTableEntry( table, tunnelIndex )

      print table.output()
