# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

import Assert
import Arnet
import Arnet.MplsLib
import BasicCli
import CliCommand
from CliCommon import InvalidInputError
import CliExtensions
import CliMatcher
from CliMode.Ira import NexthopGroupConfigBase
import CliParser
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.Ip6LlAddr as Ip6LlAddr
from CliPlugin.IraIp6IntfCli import isLinkLocalAddress
from CliPlugin.TunnelCliLib import getTunnelIdFromIndex
from CliPlugin.TunnelCli import (
   getTunnelIndexFromId,
   readMountTunnelTable,
   tokenTunnelMatcher,
   tunnelIndexMatcher,
   tunnelIgpPreferenceMatcher,
   tunnelIgpPreferenceRangeMatcher,
   tunnelIgpMetricMatcher,
   tunnelIgpMetricRangeMatcher,
   TunnelTableIdentifier,
)
import CliPlugin.VrfCli as VrfCli
import CliToken.Ip
import ConfigMount
import IraRouteCommon
import IraCommonCli
from IraNexthopGroupModel import (
      NexthopGroups,
      NexthopGroupSummary,
      NexthopGroupTunnelTableEntry,
      NexthopGroupTunnelTable,
)
import LazyMount
from NexthopGroupConsts import (
      NhgSizeConstants,
      NexthopGroupType,
      EntryCounterType,
      ipTunnelNexthopGroupTypes,
      GreKeyType,
)
from NexthopGroupUtils import nexthopGroupCliString, nexthopGroupCliStringToTacType
import ShowCommand
import SmashLazyMount
import Tac
import Toggles.NexthopGroupToggleLib
import Toggles.IraToggleLib
from TypeFuture import TacLazyType

IpGenAddrTac = TacLazyType( 'Arnet::IpGenAddr' )
IntfIdTac = TacLazyType( 'Arnet::IntfId' )
DestIpIntfTac = TacLazyType( 'Routing::NexthopGroup::DestIpIntf' )
AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
NexthopGroupCounterHelper = TacLazyType( 'Ira::NexthopGroupCounterHelper' )
TunnelConfigEntry = TacLazyType(
      'Tunnel::NexthopGroup::NexthopGroupTunnelConfigEntry' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )

routingHardwareStatus = None
routingHwNhgStatus = None
routingHardwareNexthopGroupStatus = None
cliNexthopGroupConfig = None
cliNexthopGroupConfigReadOnly = None
nexthopGroupMergedConfig = None
nexthopGroupConfigDir = None
nexthopGroupTunnelConfig = None
nexthopGroupTunnelTable = None
smashNhgStatus = None
counterHelper = None
myEntityManager = None
nexthopGroupConfigMergeSm = None

ip4 = IraRouteCommon.Ip4()
ip6 = IraRouteCommon.Ip6()
routing4 = IraRouteCommon.routing( ip4 )
routing6 = IraRouteCommon.routing( ip6 )

def nexthopGroupSupportedGuard( mode, token ):
   if os.environ.get( 'SIMULATION_STRATA' ):
      return None
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.nexthopGroupSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def nexthopGroupVrfSupported( mode ):
   if mode.session_.startupConfig():
      return True
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   return rhs.nexthopGroupVrfSupported

def dlbEcmpSupportedGuard( mode, token ):
   if os.environ.get( 'SIMULATION_STRATA' ):
      return None
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.dlbEcmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def nexthopGroupIpv6TypeSupportedGuard( mode, token ):
   ipV6ToggleGreEnabled = Toggles.IraToggleLib.toggleIpv6GreSupportToggleEnabled()
   ngType = nexthopGroupMember( mode.nexthopGroupName_ ).type
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if ngType in rhs.nexthopGroupIpv6TypeSupported:
      return None
   else:
      if ngType == NexthopGroupType.gre and ipV6ToggleGreEnabled:
         return None
   return CliParser.guardNotThisPlatform

def nexthopGroupCounterSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.nexthopGroupCounterSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def hierarchicalFecSupported( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.hierarchicalFecSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def ipv6NhgLinkLocalGuard( mode, token ):
   if Toggles.IraToggleLib.toggleIpv6NhgLinkLocalToggleEnabled():
      return None
   return CliParser.guardNotThisPlatform

def getNexthopGroupNames( mode ):
   return sorted( cliNexthopGroupConfig.nexthopGroup.members() )

#-------------------------------------------------------------------------------
# The '[no] nexthop-group' command.
#
#-------------------------------------------------------------------------------
def nexthopGroupSizeMin():
   return 1

def nexthopGroupSizeMax( mode ):
   '''
   During startup, the platform might not be ready to give the max size, so allow
   the maximum possible size. Ira NexthopGroupSm will skip publishing to status.
   Once the size is published, then Ira sm will prune the config and publish it
   to status. Any size exceeding platfrom size will be reset to default size.
   '''
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   assert rhs
   if rhs.maxNexthopGroupEcmp == 0:
      return int( NhgSizeConstants.maxSize )
   return rhs.maxNexthopGroupEcmp

def nexthopGroupSizeRange( mode ):
   return ( nexthopGroupSizeMin(), nexthopGroupSizeMax( mode ) )

def nexthopGroupEntryRange( mode ):
   nexthopGroupName = mode.nexthopGroupName_
   ent = nexthopGroupMember( nexthopGroupName )
   if not ent:
      return None
   if ent.size == 0:
      return ( 0, nexthopGroupSizeMax( mode )-1 )
   return ( 0, ent.size-1 )

class NexthopGroupConfigMode( NexthopGroupConfigBase, BasicCli.ConfigModeBase ):
   name = 'Nexthop group configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, nexthopGroupName, nexthopGroupType ):
      NexthopGroupConfigBase.__init__( self, nexthopGroupName, nexthopGroupType )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      return str( self.nexthopGroupType )

def nexthopGroupMember( nexthopGroupName, tunnelType=None, add=False ):
   assert nexthopGroupName is not None
   eg = cliNexthopGroupConfig
   assert eg is not None
   ent = eg.nexthopGroup.get( nexthopGroupName )
   if not ent and add:
      assert tunnelType is not None
      ent = eg.nexthopGroup.newMember( nexthopGroupName, tunnelType )
      ent.size = ngTypeToDefaultSize( ent.type )
      ent.ttl = ngTypeToDefaultTtl( ent.type )
      ent.sourceIp = IpGenAddrTac( '0.0.0.0' )
      ent.dlbEcmpEnable = False
      ent.version = 0
   return ent

def manageSrcGen( mode, args ):
   # tunnel source can be either interface or IP address
   nexthopGroupName = mode.nexthopGroupName_
   ent = nexthopGroupMember( nexthopGroupName )
   if 'INTF' in args:
      nhi = args[ 'INTF' ]
      if not nhi.config():
         mode.addError( 'Interface %s does not exist' % nhi.name )
         return
      if not nhi.routingSupported():
         mode.addError( 'Interface %s is not routable' % nhi.name )
         return

      # Mixing of Ipv4 and IPv6 entries is not allowed.
      # Check the tunnel dests, add IPv4 srcIP from intf if it is IPv4 NHG,
      # add ipv6 srcIP from intf if it is IPv6 NHG, add ipv4 srcIP by default,
      # lastly, if intf is not configured with compatible af, dont add any srcIP

      af = ent.destinationIpFamily
      if af == 'ipunknown':
         # Cache may be invalid. Find the actual current family
         af = calculateAddressFamily( ent )
      if af == 'ipv6':
         ipIntf = ip6.config.intf.get( nhi.name )
         if not ipIntf:
            l3Config = ip6.l3ConfigDir.newIntfConfig( nhi.name )
            ipIntf = ip6.config.newIntf( nhi.name )
            ipIntf.l3Config = l3Config
         ent.intfId = ipIntf.intfId
         # reset sourceIp address in case intf is configured
         ent.sourceIp = IpGenAddrTac( '' )
      else: # af is 'ipv4' or 'ipunknown'
         ipIntf = ip4.config.ipIntfConfig.get( nhi.name )
         if not ipIntf:
            l3Config = ip4.l3ConfigDir.newIntfConfig( nhi.name )
            ipIntf = ip4.config.newIpIntfConfig( nhi.name )
            ipIntf.l3Config = l3Config
         ent.intfId = ipIntf.intfId
         # reset sourceIp address in case intf is configured
         ent.sourceIp = IpGenAddrTac( '0.0.0.0' )
   else:
      if 'V4_ADDR' in args:
         addr = IpGenAddrTac( args[ 'V4_ADDR' ] )
      else:
         addr = IpGenAddrTac( str( args[ 'V6_ADDR' ] ) )
      if addr.af == 'ipv4':
         if ( IpAddrMatcher.validateMulticastIpAddr( addr.v4Addr ) is None or
                addr.v4Addr == '255.255.255.255' ):
            mode.addError( 'Source address must be unicast' )
            return
      if addr.af == 'ipv6' and isLinkLocalAddress( mode, addr.v6Addr ):
         mode.addError( 'Source address must not be link-local' )
         return
      if not verifySanityForNhg( ent, addr ):
         mode.addError(
            'Mixing V4 and V6 entries in the same nexthopgroup not allowed' )
         return
      ent.sourceIp = addr
      # reset intf in case sourceIp address configured
      ent.intfId = ''
      ent.destinationIpFamily = addr.af

   ent.version += 1

def noManageSrcGen( mode, args ):
   ent = nexthopGroupMember( mode.nexthopGroupName_ )
   if not ent:
      return

   if 'INTF' in args:
      nhi = args[ 'INTF' ]
      ipIntf = ip4.config.ipIntfConfig.get( nhi.name )
      ip6Intf = ip6.config.intf.get( nhi.name )
      if not ipIntf and not ip6Intf:
         return
      if ipIntf and ipIntf.intfId != ent.intfId or \
             ip6Intf and ip6Intf.intfId != ent.intfId:
         return
      ent.intfId = ''
   else:
      if 'V4_ADDR' in args:
         addr = IpGenAddrTac( args[ 'V4_ADDR' ] )
      else:
         addr = IpGenAddrTac( str( args[ 'V6_ADDR' ] ) )
      sourceIp = ent.sourceIp.v4Addr if isinstance( addr, str ) else ent.sourceIp
      if sourceIp == addr:
         if ent.destinationIpFamily == 'ipv6':
            ent.sourceIp = IpGenAddrTac( '' )
         else:
            ent.sourceIp = IpGenAddrTac( '0.0.0.0' )
         ent.destinationIpFamily = 'ipunknown'

   ent.version += 1

# Verify that the new address is the same family as the existing entries
def verifySanityForNhg( ent, newAddr ):
   af = ent.destinationIpFamily
   if af == 'ipunknown':
      # Cache may be invalid. Find the actual current family
      af = calculateAddressFamily( ent )

   if af == 'ipunknown':
      return True
   elif af == newAddr.af:
      return True
   else:
      return False

def calculateAddressFamily( ent ):
   nhgAf = 'ipunknown'
   size = max( ent.destinationIpIntf.keys() ) + 1 if \
               ent.destinationIpIntf.keys() else 0
   for i in range( ent.size if ent.size != 0 else size ):
      entryAf = ent.destinationIp( i ).af
      if entryAf != 'ipunknown':
         nhgAf = entryAf
         break
   # check sourceIp also to find af
   if nhgAf == 'ipunknown' and ent.sourceIp.af != 'ipunknown' and \
          not ent.sourceIp.isAddrZero:
      nhgAf = ent.sourceIp.af
   return nhgAf

def manageDA4( mode, idx, destAddr, labelStack=None ):
   Assert.assertIsInstance( destAddr, str )
   return manageDAGen( mode, idx, IpGenAddrTac( destAddr ), labelStack )

def manageDA4WithLabelStack( mode, idx, labelStack, destAddr ):
   Assert.assertIsInstance( destAddr, str )
   return manageDAGen( mode, idx, IpGenAddrTac( destAddr ), labelStack )

def noManageDA4WithLabelStack( mode, idx, labelStack=None, destAddr=None ):
   if labelStack is None and destAddr is not None:
      # We need to support different forms of 'no entry' on mplsOverGre groups:
      # 1. no entry <index>
      # 2. no entry <index> push label-stack <label(s)>
      # 3. no entry <index> push label-stack <label(s)> tunnel-destination <ip-addr>
      #
      # For variations 2 and 3 to work properly, we had to define two independent
      # optional rules (push... and tunnel-dest...). Because they're independent and
      # optional, the following command is valid from the CLI parser POV:
      #
      #   no entry <index> tunnel-destination <ip-addr>
      #
      # We instead generate the invalid input error right here.
      raise InvalidInputError()

   if labelStack is not None:
      labelStack = getMplsLabelStack( labelStack )

   if destAddr is not None:
      Assert.assertIsInstance( destAddr, str )
      destAddr = IpGenAddrTac( destAddr )

   nexthopGroupName = mode.nexthopGroupName_
   ent = nexthopGroupMember( nexthopGroupName )
   if not ent:
      return

   if labelStack is None or labelStack == ent.mplsLabelStack[ idx ]:
      if destAddr is None or ent.destinationIp( idx ) == destAddr:
         ent.destIpIntfDel( idx )
         ent.mplsLabelStack[ idx ] = getMplsLabelStack()
         ent.version += 1

def manageDAGen( mode, idx, destAddr, labelStack=None, intfId=None ):
   if not IraCommonCli.validNextHopGen( mode, destAddr, VrfCli.DEFAULT_VRF,
                                        intfId=intfId ):
      mode.addWarning( 'Invalid next hop %s' % destAddr )
      return
   nexthopGroupName = mode.nexthopGroupName_
   ent = nexthopGroupMember( nexthopGroupName )
   # Check new entry being added is of the same af
   # as existing entries
   if not verifySanityForNhg( ent, destAddr ):
      mode.addError(
            'Mixing V4 and V6 entries in the same nexthopgroup not allowed' )
      return
   ent.destinationIpFamily = destAddr.af
   if ent.type in ( NexthopGroupType.mpls, NexthopGroupType.mplsOverGre,
         NexthopGroupType.mplsOverUdp, NexthopGroupType.ip ):
      if labelStack is not None:
         # Check the entry being added is consistent
         # with the explicit NULL label (0 or 2)
         if ( str( MplsLabel.explicitNullIpv4 ) == labelStack[ 0 ] and
              destAddr.af == AddressFamily.ipv6 ):
            mode.addError( 'Label 0 can only be configured with IPv4 nexthop' )
            return
         if ( str( MplsLabel.explicitNullIpv6 ) == labelStack[ 0 ] and
              destAddr.af == AddressFamily.ipv4 ):
            mode.addError( 'Label 2 can only be configured with IPv6 nexthop' )
            return
         ent.mplsLabelStack[ idx ] = getMplsLabelStack( labelStack )
      else:
         assert ent.type in ( NexthopGroupType.mpls,
                              NexthopGroupType.ip )
         ent.mplsLabelStack[ idx ] = getMplsLabelStack( labelStack )
   else:
      assert labelStack is None

   if intfId is None:
      intfId = ''
   ent.destinationIpIntf[ idx ] = DestIpIntfTac( destAddr, intfId )
   ent.version += 1

def noManageDA4( mode, idx, destAddr=None ):
   destAddrGen = destAddr
   if destAddrGen:
      Assert.assertIsInstance( destAddr, str )
      destAddrGen = IpGenAddrTac( destAddr )

   return noManageDAGen( mode, idx, destAddrGen )

def noManageDAGen( mode, idx, destAddr=None ):
   nexthopGroupName = mode.nexthopGroupName_
   ent = nexthopGroupMember( nexthopGroupName )
   if not ent or ( idx >= ent.size and ent.size != 0 ):
      return
   if destAddr is None or ent.destinationIp( idx ) == destAddr:
      ent.destIpIntfDel( idx )
      # There may or may not be valid entries left. Invalidate the cached value
      ent.destinationIpFamily = 'ipunknown'
      ent.mplsLabelStack[ idx ] = getMplsLabelStack()
      ent.version += 1

def getMaxMplsLabels():
   maxLabels = Tac.Type( 'Arnet::MplsStackEntryIndex' ).max + 1
   if routingHardwareStatus:
      platformMaxLabels = routingHardwareStatus.nexthopGroupMplsLabelStackSize
      if platformMaxLabels > 0 and platformMaxLabels < maxLabels:
         maxLabels = platformMaxLabels
   return maxLabels

def getMaxMplsOverGreLabels():
   maxLabels = Tac.Type( 'Arnet::MplsStackEntryIndex' ).max + 1
   if routingHardwareStatus:
      platformMaxLabels = routingHardwareStatus.nexthopGroupMplsOverGreLabelStackSize
      if platformMaxLabels > 0 and platformMaxLabels < maxLabels:
         maxLabels = platformMaxLabels
   return maxLabels

def getMaxMplsOverUdpLabels():
   maxLabels = Tac.Type( 'Arnet::MplsStackEntryIndex' ).max + 1
   if routingHardwareStatus:
      platformMaxLabels = routingHardwareStatus.nexthopGroupMplsOverUdpLabelStackSize
      if platformMaxLabels > 0 and platformMaxLabels < maxLabels:
         maxLabels = platformMaxLabels
   return maxLabels

def getMplsLabelStack( labelStack=None ):
   stack = Tac.newInstance( 'Arnet::MplsLabelOperation' )
   if labelStack:
      stack.stackSize = len( labelStack )
      stack.operation = Tac.Type( 'Arnet::MplsLabelAction' ).push
      for labelVal, index in \
      zip( reversed( labelStack ), range( len( labelStack ) ) ):
         labelVal = int( labelVal )
         stack.labelStackIs( index, labelVal )
   return stack

def tunnelDATokendict( mode ):
   if mode.nexthopGroupType != 'ip' and mode.nexthopGroupType != 'mpls' :
      return { 'tunnel-destination': 'IP address for the destination' }
   else:
      return { 'nexthop': 'IP address for the destination' }

def tunnelDASupportedGuard( mode, token ):
   ngType = nexthopGroupMember( mode.nexthopGroupName_ ).type
   if ngType in ( NexthopGroupType.mplsOverGre,
                  NexthopGroupType.mplsOverUdp ):
      return CliParser.guardNotThisPlatform
   else:
      return None

def supportedNgTypeMatcher( mode ):
   rhs = routingHardwareStatus
   assert rhs
   # dict to store nexthop-group types supported by platfrom
   # The key is the nexthop-group type, and the value is the helpdesc for the
   # type.
   supportedNgType = {}
   helpStrings = {
      NexthopGroupType.ipInIp: 'IP-in-IP tunnel type',
      NexthopGroupType.gre: 'GRE tunnel type',
      NexthopGroupType.mpls: 'MPLS tunnel type',
      NexthopGroupType.ip: 'ECMP next hop',
      NexthopGroupType.mplsOverGre: 'MPLS-over-GRE tunnel type',
      NexthopGroupType.mplsOverUdp: 'MPLS-over-UDP tunnel type',
      NexthopGroupType.vxlan: 'VXLAN tunnel type',
   }

   for ngType, supported in rhs.nexthopGroupTypeSupported.items():
      if supported:
         helpStr = helpStrings[ ngType ]
         supportedNgType[ nexthopGroupCliString( ngType ) ] = helpStr

   # For ip duts, which do not support any nexthop group types (so the above
   # iterable is empty), we force one so we can use the IpEth config session
   # tamper.
   if os.environ.get( 'ALLOW_NHG_CONFIGURATION' ) and not supportedNgType:
      ngType = NexthopGroupType.ipInIp
      helpStr = helpStrings[ ngType ]
      supportedNgType[ nexthopGroupCliString( ngType ) ] = helpStr

   return supportedNgType

#------------------------------------------------------------
# common matchers
#------------------------------------------------------------
matcherTunnelEntry = CliMatcher.KeywordMatcher(
      'entry',
      helpdesc='Nexthop Group Entry Config' )
matcherTunnelIndex = CliMatcher.DynamicIntegerMatcher(
      rangeFn=nexthopGroupEntryRange,
      helpname=None,
      helpdesc='Index value for the entry' )
matcherPush = CliMatcher.KeywordMatcher( 'push',
      helpdesc='Push a MPLS label stack on to packets using this route' )
matcherLabelStack = CliMatcher.KeywordMatcher( 'label-stack',
      helpdesc=( 'MPLS label(s) to push on to the packets. '
         'Top of Stack (Outermost Label) first.' ) )
matcherTunnelDestination = CliMatcher.KeywordMatcher( 'tunnel-destination',
      helpdesc='IP address for the destination' )
matcherDstAddr = IpAddrMatcher.IpAddrMatcher( 'IP address for the destination' )

class MplsNextHopGroupConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      ngType = nexthopGroupMember( mode.nexthopGroupName_ ).type
      return ngType == NexthopGroupType.mpls

class ManageDaGenCmd( CliCommand.CliCommandClass ):
   syntax = ( 'entry IDX push label-stack { LABEL } nexthop '
              '( DEST_ADDR | LL_DEST_ADDR )' )
   noOrDefaultSyntax = ( 'entry IDX push label-stack { LABEL } '
                         '[ nexthop DEST_ADDR ]' )
   data = {
      'entry': matcherTunnelEntry,
      'IDX': matcherTunnelIndex,
      'push': matcherPush,
      'label-stack': matcherLabelStack,
      'LABEL': Arnet.MplsLib.LabelValWithExpNullExprFactory(),
      'nexthop': 'IP Address of nexthop',
      'DEST_ADDR': IpGenAddrMatcher.IpGenAddrMatcher( 'Nexthop IP Address' ),
      'LL_DEST_ADDR': CliCommand.Node(
         matcher=Ip6LlAddr.Ip6LlAddrMatcher(
            helpdesc='Nexthop Ipv6 Link-local Address' ),
         guard=ipv6NhgLinkLocalGuard )
   }

   @staticmethod
   def handler( mode, args ):
      labelStack = args[ 'LABEL' ]
      maxMplsLabels = getMaxMplsLabels()
      if maxMplsLabels and len( labelStack ) > getMaxMplsLabels():
         mode.addError( 'Too many MPLS Labels. Max is %d' % maxMplsLabels )
         return None

      if 'LL_DEST_ADDR' in args:
         # destAddr's format is fe80::A:B:C:D%<intfName>
         splitAddr = args[ 'LL_DEST_ADDR' ].split( '%' )
         return manageDAGen( mode, args[ 'IDX' ], IpGenAddrTac( splitAddr[ 0 ] ),
                             labelStack, IntfIdTac( splitAddr[ 1 ] ) )
      return manageDAGen( mode, args[ 'IDX' ], args[ 'DEST_ADDR' ],
            labelStack=labelStack )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return noManageDAGen( mode, args[ 'IDX' ], destAddr=args.get( 'DEST_ADDR' ) )

MplsNextHopGroupConfigModelet.addCommandClass( ManageDaGenCmd )

class MplsOverGreNextHopGroupConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      ngType = nexthopGroupMember( mode.nexthopGroupName_ ).type
      return ngType == NexthopGroupType.mplsOverGre

class MogPushLabelAndDest4Cmd( CliCommand.CliCommandClass ):
   syntax = 'entry IDX push label-stack { LABEL } tunnel-destination DEST_ADDR'
   noOrDefaultSyntax = ( 'entry IDX push label-stack { LABEL } '
                         '[ tunnel-destination DEST_ADDR ]' )
   data = {
      'entry': matcherTunnelEntry,
      'IDX': matcherTunnelIndex,
      'push': matcherPush,
      'label-stack': matcherLabelStack,
      'LABEL': Arnet.MplsLib.LabelValWithExpNullExprFactory(),
      'tunnel-destination': matcherTunnelDestination,
      'DEST_ADDR': matcherDstAddr,
   }

   @staticmethod
   def handler( mode, args ):
      labelStack = args[ 'LABEL' ]
      maxMplsLabels = getMaxMplsLabels()
      if maxMplsLabels and len( labelStack ) > getMaxMplsOverGreLabels():
         mode.addError( 'Too many MPLS Labels. Max is %d' % maxMplsLabels )
         return None
      return manageDA4WithLabelStack( mode, args[ 'IDX' ], labelStack,
            args[ 'DEST_ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return noManageDA4WithLabelStack( mode, args[ 'IDX' ], args.get( 'LABEL' ),
            args.get( 'DEST_ADDR' ) )

MplsOverGreNextHopGroupConfigModelet.addCommandClass( MogPushLabelAndDest4Cmd )

class MplsOverUdpNextHopGroupConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      ngType = nexthopGroupMember( mode.nexthopGroupName_ ).type
      return ngType == NexthopGroupType.mplsOverUdp

class MouPushLabelAndDest4Cmd( CliCommand.CliCommandClass ):
   syntax = 'entry IDX push label-stack { LABEL } tunnel-destination DEST_ADDR'
   noOrDefaultSyntax = ( 'entry IDX push label-stack { LABEL } '
                         '[ tunnel-destination DEST_ADDR ]' )
   data = {
      'entry': matcherTunnelEntry,
      'IDX': matcherTunnelIndex,
      'push': matcherPush,
      'label-stack': matcherLabelStack,
      'LABEL': Arnet.MplsLib.LabelValWithExpNullExprFactory(),
      'tunnel-destination': matcherTunnelDestination,
      'DEST_ADDR': matcherDstAddr,
   }

   @staticmethod
   def handler( mode, args ):
      labelStack = args[ 'LABEL' ]
      maxMplsLabels = getMaxMplsLabels()
      if maxMplsLabels and len( labelStack ) > getMaxMplsOverUdpLabels():
         mode.addError( 'Too many MPLS Labels. Max is %d' % maxMplsLabels )
         return None
      return manageDA4WithLabelStack( mode, args[ 'IDX' ], labelStack,
            args[ 'DEST_ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return noManageDA4WithLabelStack( mode, args[ 'IDX' ], args.get( 'LABEL' ),
            args.get( 'DEST_ADDR' ) )

MplsOverUdpNextHopGroupConfigModelet.addCommandClass( MouPushLabelAndDest4Cmd )

class GreNextHopGroupConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      ngType = nexthopGroupMember( mode.nexthopGroupName_ ).type
      return ngType == NexthopGroupType.gre

class GreKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel-key ingress-interface'
   noOrDefaultSyntax = 'tunnel-key [ ingress-interface ]'
   data = {
      'tunnel-key': 'GRE key',
      'ingress-interface': 'Ingress interface information'
   }

   @staticmethod
   def handler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName, add=True )
      ent.greKeyType = GreKeyType.ingressIntf
      ent.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName, add=True )
      if ent:
         ent.greKeyType = GreKeyType.noGreKey
         ent.version += 1

GreNextHopGroupConfigModelet.addCommandClass( GreKeyCmd )

class IpTunnelNextHopGroupConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      ngType = nexthopGroupMember( mode.nexthopGroupName_ ).type
      return ngType in ipTunnelNexthopGroupTypes

class TtlCmd( CliCommand.CliCommandClass ):
   syntax = 'ttl TTL'
   noOrDefaultSyntax = syntax
   data = {
      'ttl': 'Tunnel encapsulation TTL value',
      'TTL': CliMatcher.IntegerMatcher( 1, 64,
         helpdesc='Tunnel encapsulation TTL value' )
   }

   @staticmethod
   def handler( mode, args ):
      ent = nexthopGroupMember( mode.nexthopGroupName_ )
      ent.ttl = args[ 'TTL' ]
      ent.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ent = nexthopGroupMember( mode.nexthopGroupName_ )
      if not ent:
         return
      if args[ 'TTL' ] == ent.ttl:
         ent.ttl = 64
         ent.version += 1

IpTunnelNextHopGroupConfigModelet.addCommandClass( TtlCmd )

class TunnelSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel-source ( ( intf INTF ) | ( V4_ADDR | V6_ADDR ) )'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel-source': 'Source Interface or Address',
      'intf': 'Source Interface',
      'INTF': IntfCli.Intf.matcher,
      'V4_ADDR': IpAddrMatcher.IpAddrMatcher( 'IP address of the source' ),
      'V6_ADDR': CliCommand.Node(
         matcher=Ip6AddrMatcher.Ip6AddrMatcher( 'IPv6 address of the source' ),
         guard=nexthopGroupIpv6TypeSupportedGuard )
   }

   handler = manageSrcGen
   noOrDefaultHandler = noManageSrcGen

IpTunnelNextHopGroupConfigModelet.addCommandClass( TunnelSourceCmd )

for cls in ( IpTunnelNextHopGroupConfigModelet, GreNextHopGroupConfigModelet,
      MplsNextHopGroupConfigModelet, MplsOverGreNextHopGroupConfigModelet,
      MplsOverUdpNextHopGroupConfigModelet ):
   NexthopGroupConfigMode.addModelet( cls )

#------------------------------------------------------------
# dynamic-load-balancing
#------------------------------------------------------------
class DynamicLoadBalancingCmd( CliCommand.CliCommandClass ):
   syntax = 'dynamic-load-balancing'
   noOrDefaultSyntax = syntax
   data = {
         'dynamic-load-balancing': CliCommand.guardedKeyword(
            'dynamic-load-balancing',
            helpdesc='Enable/Disable Dynamic Load Balancing',
            guard=dlbEcmpSupportedGuard )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      ent.dlbEcmpEnable = True
      ent.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      ent.dlbEcmpEnable = False
      ent.version += 1

NexthopGroupConfigMode.addCommandClass( DynamicLoadBalancingCmd )

#------------------------------------------------------------
# fec hierarchical
#------------------------------------------------------------
class FecHierarchicalCmd( CliCommand.CliCommandClass ):
   syntax = 'fec hierarchical'
   noOrDefaultSyntax = 'fec [ hierarchical ] [ ... ]'
   data = {
         'fec': CliCommand.guardedKeyword( 'fec',
            helpdesc='FEC configuration',
            guard=hierarchicalFecSupported ),
         'hierarchical': 'Enable hierarchical FECs'
   }

   @staticmethod
   def handler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      ent.hierarchicalFecsEnabled = True
      ent.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      ent.hierarchicalFecsEnabled = False
      ent.version += 1

NexthopGroupConfigMode.addCommandClass( FecHierarchicalCmd )

#------------------------------------------------------------
# size <size>
#------------------------------------------------------------
class SizeCmd( CliCommand.CliCommandClass ):
   syntax = 'size SIZE'
   noOrDefaultSyntax = 'size [ SIZE ]'
   data = {
         'size': 'Nexthop Group Entry Size',
         'SIZE': CliMatcher.DynamicIntegerMatcher( rangeFn=nexthopGroupSizeRange,
            helpname=None, helpdesc='Entry Size' )
   }

   @staticmethod
   def handler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      ent.size = args[ 'SIZE' ]
      ent.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      size = args.get( 'SIZE', None )
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      if not ent:
         return
      if size is None or size == ent.size:
         ent.size = ngTypeToDefaultSize( ent.type )
         ent.version += 1

NexthopGroupConfigMode.addCommandClass( SizeCmd )

#------------------------------------------------------------
# entry counters unshared
#------------------------------------------------------------
class UnsharedCmd( CliCommand.CliCommandClass ):
   syntax = 'entry counters unshared'
   noOrDefaultSyntax = 'entry counters'
   data = {
         'entry': matcherTunnelEntry,
         'counters': CliCommand.guardedKeyword( 'counters',
            helpdesc='Entry counters type',
            guard=nexthopGroupCounterSupportedGuard ),
         'unshared': 'Enable unshared entry counters'
   }

   @staticmethod
   def handler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      if ent and ent.entryCounterType != EntryCounterType.unshared:
         ent.entryCounterType = EntryCounterType.unshared
         ent.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nexthopGroupName = mode.nexthopGroupName_
      ent = nexthopGroupMember( nexthopGroupName )
      if ent and ent.entryCounterType != EntryCounterType.shared:
         ent.entryCounterType = EntryCounterType.shared
         ent.version += 1

NexthopGroupConfigMode.addCommandClass( UnsharedCmd )

#------------------------------------------------------------
# entry <idx> [ nexthop | tunnel-destination ] <ip_addr>
#------------------------------------------------------------
class EntryNextHopIpCmd( CliCommand.CliCommandClass ):
   syntax = 'entry IDX DESTINATION ( V4 | V6 )'
   noOrDefaultSyntax = 'entry IDX [ DESTINATION ] [ ( V4 | V6 ) ]'
   data = {
         'entry': matcherTunnelEntry,
         'IDX': matcherTunnelIndex,
         'DESTINATION': CliCommand.Node(
            matcher=CliMatcher.DynamicKeywordMatcher( tunnelDATokendict ),
            guard=tunnelDASupportedGuard ),
         'V4': matcherDstAddr,
         'V6': CliCommand.Node(
            matcher=Ip6AddrMatcher.Ip6AddrMatcher(
               'IP address for the destination' ),
            guard=nexthopGroupIpv6TypeSupportedGuard )
      }

   @staticmethod
   def handler( mode, args ):
      idx = args[ 'IDX' ]
      if 'V4' in args:
         ipAddr = args[ 'V4' ]
         manageDA4( mode, idx, ipAddr )
      elif 'V6' in args:
         ipAddr = args[ 'V6' ]
         manageDAGen( mode, idx, IpGenAddrTac( str( ipAddr ) ) )
      else:
         assert False, 'Invalid IP'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      idx = args[ 'IDX' ]
      if 'V4' in args:
         ipAddr = args[ 'V4' ]
         noManageDA4( mode, idx, ipAddr )
      elif 'V6' in args:
         ipAddr = args[ 'V6' ]
         noManageDAGen( mode, idx, IpGenAddrTac( str( ipAddr ) ) )
      else:
         noManageDAGen( mode, idx )

NexthopGroupConfigMode.addCommandClass( EntryNextHopIpCmd )

def ngTypeToEnum( ngType ):
   return nexthopGroupCliStringToTacType( ngType )

def ngTypeToDefaultSize( ngType ):
   return NhgSizeConstants.defaultSize

def ngTypeToDefaultTtl( ngType ):
   ttlVal = 0
   if ngType in ipTunnelNexthopGroupTypes:
      ttlVal = 64
   return ttlVal

def _gotoNexthopGroupConfigMode( mode, nexthopGroupName, ngType ):

   if ( mode.session_.guardsEnabled() and
        nexthopGroupSupportedGuard( mode, None ) is not None ):
      mode.addError( 'Nexthop-Group not supported on this hardware platform' )
      return
   assert nexthopGroupName is not None
   eg = cliNexthopGroupConfig
   assert eg is not None

   ent = eg.nexthopGroup.get( nexthopGroupName )
   if not ent:
      ent = eg.nexthopGroup.newMember( nexthopGroupName, ngTypeToEnum( ngType ) )
      ent.ttl = ngTypeToDefaultTtl( ent.type )
      ent.size = ngTypeToDefaultSize( ent.type )
      ent.sourceIp = IpGenAddrTac( '0.0.0.0' )
      ent.dlbEcmpEnable = False
      ent.version = 0
   else:
      if ent.type != ngTypeToEnum( ngType ):
         mode.addError( 'Changing the type of existing next hop group not allowed' )
         return
   childMode = mode.childMode( NexthopGroupConfigMode,
                               nexthopGroupName=nexthopGroupName,
                               nexthopGroupType=nexthopGroupCliString( ent.type ) )
   mode.session_.gotoChildMode( childMode )

nexthopGroupCfgNode = CliCommand.guardedKeyword( 'nexthop-group',
      helpdesc='Specify nexthop group name',
      guard=nexthopGroupSupportedGuard )
nexthopGroupNamePattern = '[:a-zA-Z0-9_-][.:a-zA-Z0-9_-]*'
nexthopGroupCfgName = CliMatcher.PatternMatcher(
      pattern=nexthopGroupNamePattern,
      helpdesc='Name of nexthop group',
      helpname='WORD' )
ngTypeNode = CliCommand.Node(
      CliMatcher.DynamicKeywordMatcher(
         supportedNgTypeMatcher,
         alwaysMatchInStartupConfig=True ) )

class NextHopGroupCmdHandlerCmd( CliCommand.CliCommandClass ):
   syntax = 'nexthop-group NAME type NG_TYPE'
   noOrDefaultSyntax = 'nexthop-group NAME [ type NG_TYPE ]'
   data = {
         'nexthop-group': nexthopGroupCfgNode,
         'NAME': nexthopGroupCfgName,
         'type': 'nexthop group type',
         'NG_TYPE': ngTypeNode
      }

   @staticmethod
   def handler( mode, args ):
      nexthopGroupName = args[ 'NAME' ]
      ngType = args[ 'NG_TYPE' ]
      _gotoNexthopGroupConfigMode( mode, nexthopGroupName, ngType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      eg = cliNexthopGroupConfig
      if eg is None:
         return
      nexthopGroupName = args[ 'NAME' ]
      if nexthopGroupName is not None:
         del eg.nexthopGroup[ nexthopGroupName ]
         return
      names = getNexthopGroupNames( mode )
      for name in names:
         del eg.nexthopGroup[ name ]

BasicCli.GlobalConfigMode.addCommandClass( NextHopGroupCmdHandlerCmd )

#-------------------------------------------------------------------------------
#
# show nexthop-group <name>
#
#-------------------------------------------------------------------------------

# showNexthopGroupCounterHook allows another package to provide counters for
# nexthop groups. The extension function takes mode, outLif, interface name as
# arguments and returns packet, byte counter. Ex: hook( mode, outLif, intfName )
showNexthopGroupCounterHook = CliExtensions.CliHook()

def setupCounterHelper():
   global counterHelper

   # Instantiate a counterHelper if it does not exist yet.
   # Mount the necessary smash tables for fetching counters.
   counterHelper = NexthopGroupCounterHelper()
   counterHelper.mountCounterSmashes( myEntityManager.cEntityManager() )
   counterHelper.counterEnabled = False

nexthopGroupKw = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'nexthop-group',
         helpdesc='Specify nexthop group name' ), guard=nexthopGroupSupportedGuard )

nexthopGroupNameMatcher = CliMatcher.DynamicNameMatcher( getNexthopGroupNames,
                                                   priority=CliParser.PRIO_NORMAL,
                                                   helpdesc='Nexthop group name' )

vrf = CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF Name' )
vrfNameMatcher = CliMatcher.DynamicNameMatcher( VrfCli.getAllVrfNames,
                                                helpdesc='VRF Name' )

def showNexthopGroups( mode, nexthopGroupName=None, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not VrfCli.vrfExists( vrfName ):
      mode.addError( 'VRF %s doesn\'t exist' % vrfName )
      return None

   if not nexthopGroupName:
      nexthopGroupName = ''

   LazyMount.force( routingHardwareNexthopGroupStatus )

   counterEnabled = False
   if routingHardwareStatus.nexthopGroupCounterSupported:
      # The extensions are expected to returns a tac instance of a
      # NexthopGroupCliCounterInterface dervied class or None
      for hook in showNexthopGroupCounterHook.extensions():
         counterEnabled = hook( mode )
   counterHelper.counterEnabled = counterEnabled

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   # passing nexthopGroupName as empty string will print all nexthop groups
   helper = Tac.newInstance( 'Ira::NexthopGroupHelper', nexthopGroupMergedConfig,
                             smashNhgStatus, routingHardwareNexthopGroupStatus,
                             routingHardwareStatus, counterHelper,
                             nexthopGroupName )
   helper.render( fd, fmt )
   return NexthopGroups

class ShowNexthopGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show nexthop-group [ ( GROUP | ( vrf VRF ) ) ]'
   data = {
      'nexthop-group': nexthopGroupKw,
      'GROUP': nexthopGroupNameMatcher,
      'vrf': vrf,
      'VRF': vrfNameMatcher,
   }
   cliModel = NexthopGroups

   @staticmethod
   def handler( mode, args ):
      if 'GROUP' in args:
         return showNexthopGroups( mode, args[ 'GROUP' ] )
      elif 'VRF' in args:
         return showNexthopGroups( mode, vrfName=args[ 'VRF' ] )
      else:
         return showNexthopGroups( mode )

BasicCli.addShowCommandClass( ShowNexthopGroupCmd )

#-------------------------------------------------------------------------------
#
# show nexthop-group summary
#
#-------------------------------------------------------------------------------
def showNexthopGroupSummary( mode, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not VrfCli.vrfExists( vrfName ):
      mode.addError( 'Vrf %s doesn\'t exist' % vrfName )
      return None

   LazyMount.force( routingHardwareNexthopGroupStatus )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   helper = Tac.newInstance( 'Ira::NexthopGroupSummaryHelper',
                              nexthopGroupMergedConfig, smashNhgStatus,
                              routingHardwareNexthopGroupStatus,
                              routingHardwareStatus )
   helper.render( fd, fmt )
   return NexthopGroupSummary

class ShowNexthopGroupSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show nexthop-group summary'''
   data = { 'nexthop-group' : nexthopGroupKw,
            'summary' : CliMatcher.KeywordMatcher( 'summary',
                                       helpdesc='Summarize all nexthop groups',
                                       priority=CliParser.PRIO_HIGH )
          }

   cliModel=NexthopGroupSummary

   @staticmethod
   def handler( mode, args ):
      return showNexthopGroupSummary( mode )

BasicCli.addShowCommandClass( ShowNexthopGroupSummaryCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip tunnel PREFIX nexthop-group GROUP [ preference <pref> |
# metric <metric> ]
#--------------------------------------------------------------------------------
def nhgTunnelConfigEntry( tep, nhgName, preference=None, metric=None ):
   tep = Arnet.IpGenPrefix( str( tep ) )
   entry = TunnelConfigEntry( tep )
   entry.nhgName = nhgName
   if preference:
      entry.igpPref = preference
   if metric:
      entry.igpMetric = metric
   return entry

class NhgTunnelCmd( CliCommand.CliCommandClass ):
   syntax = 'ip tunnel PREFIX nexthop-group GROUP ' \
            '[ igp-cost { ( preference PREFERENCE ) | ( metric METRIC ) } ]'
   noOrDefaultSyntax = 'ip tunnel PREFIX nexthop-group ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'tunnel': 'Tunnel configuration',
      'PREFIX': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
         ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
         ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'nexthop-group': nexthopGroupKw,
      'GROUP': nexthopGroupNameMatcher,
      'igp-cost': ( 'Configure tunnel parameters to influence IGP cost '
                    'of next-hops resolving over nexthop-group tunnels' ),
      'preference': CliCommand.Node( tunnelIgpPreferenceMatcher,
                                     maxMatches=1 ),
      'PREFERENCE': CliCommand.Node( tunnelIgpPreferenceRangeMatcher,
                                     maxMatches=1 ),
      'metric': CliCommand.Node( tunnelIgpMetricMatcher,
                                 maxMatches=1 ),
      'METRIC': CliCommand.Node( tunnelIgpMetricRangeMatcher,
                                 maxMatches=1 ),
   }

   @staticmethod
   def handler( mode, args ):
      configEntry = nhgTunnelConfigEntry( args[ 'PREFIX' ], args[ 'GROUP' ],
                                          args.get( 'PREFERENCE' ),
                                          args.get( 'METRIC' ) )
      nexthopGroupTunnelConfig.entry.addMember( configEntry )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      endpoint = args[ 'PREFIX' ]
      endpoint = Arnet.IpGenPrefix( str( endpoint ) )
      del nexthopGroupTunnelConfig.entry[ endpoint ]

BasicCli.GlobalConfigMode.addCommandClass( NhgTunnelCmd )

def getNhgTunnelTableEntryModel( cachedNHG, tunnelId, endpoint=None ):
   nhgTunnelTableEntry = nexthopGroupTunnelTable.entry.get( tunnelId )
   if nhgTunnelTableEntry:
      if endpoint is not None and nhgTunnelTableEntry.tep != endpoint:
         return None
      nhgId = nhgTunnelTableEntry.nhgId
      nhgName = cachedNHG.get( nhgId, "NexthopGroupUnknown" )
      igpMetric = nhgTunnelTableEntry.igpMetric
      igpPref = nhgTunnelTableEntry.igpPref
      return NexthopGroupTunnelTableEntry(
                endpoint=nhgTunnelTableEntry.tep,
                nexthopGroupName=nhgName,
                igpMetric=igpMetric, igpPref=igpPref )
   return None 

def showNexthopGroupTunnel( mode, args ):
   nhgTunnelEntries = {}
   cachedNHG = {}
   for key, entry in smashNhgStatus.nexthopGroupEntry.iteritems():
      cachedNHG[ entry.nhgId ] = key.nhgName()
   tunnelIndex = args.get( "TUNNEL_INDEX" )
   endpoint = args.get( "END_POINT" )
   if tunnelIndex is None:
      if endpoint is not None:
         endpoint = Arnet.IpGenPrefix( str( endpoint ) )
      for tunnelId in nexthopGroupTunnelTable.entry:
         nhgTunnelEntryModel = getNhgTunnelTableEntryModel( cachedNHG, tunnelId,
                                                            endpoint=endpoint )
         if nhgTunnelEntryModel:
            nhgTunnelEntries[ getTunnelIndexFromId( tunnelId ) ] = \
               nhgTunnelEntryModel
   else:
      tunnelId = getTunnelIdFromIndex( 'nexthopGroupTunnel', tunnelIndex )
      nhgTunnelEntryModel = getNhgTunnelTableEntryModel( cachedNHG,
                                                         tunnelId )
      if nhgTunnelEntryModel:
         nhgTunnelEntries[ getTunnelIndexFromId( tunnelId ) ] = nhgTunnelEntryModel

   return NexthopGroupTunnelTable( indexes=nhgTunnelEntries )

class ShowNhgTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show nexthop-group tunnel [ TUNNEL_INDEX | END_POINT ]'''
   data = { 'nexthop-group' : nexthopGroupKw,
            'tunnel' : tokenTunnelMatcher,
            'TUNNEL_INDEX' : tunnelIndexMatcher,
            'END_POINT': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
               ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
               ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
          }
   cliModel = NexthopGroupTunnelTable

   handler = showNexthopGroupTunnel

BasicCli.addShowCommandClass( ShowNhgTunnelCmd )

def startNexthopGroupConfigMergeSm():
   global nexthopGroupMergedConfig
   global nexthopGroupConfigMergeSm
   nexthopGroupMergedConfig = Tac.newInstance( 'Routing::NexthopGroup::Config' )
   if Toggles.NexthopGroupToggleLib.toggleNhgCliDynamicMergeSmEnabled():
      control = Tac.Value( 'Arx::SmControl' )
      # Make the state machine totally synchronous, until we add support
      # within Arex to react to output pipeline being ready(), at which point
      # we can set ...MergedConfig.configReady = True.
      control.runToCompletion = True
      SmTypeName = 'NexthopGroup::NexthopGroupCliDynamicConfigMergeSm'
      nexthopGroupConfigMergeSm = Tac.newInstance( SmTypeName,
                                                   nexthopGroupMergedConfig,
                                                   cliNexthopGroupConfigReadOnly,
                                                   nexthopGroupConfigDir,
                                                   control )
      nexthopGroupMergedConfig.configReady = True
   else:
      SmTypeName = 'Ira::NexthopGroupConfigMergeSm'
      nexthopGroupConfigMergeSm = Tac.newInstance( SmTypeName,
                                                   nexthopGroupMergedConfig,
                                                   cliNexthopGroupConfigReadOnly,
                                                   nexthopGroupConfigDir )

def pluginInitNexthopGroupConfigMergeSm( entityManager ):
   '''
   Set up a single instance of the NexthopGroupConfigMergeSm.  It needs to to have
   mount the CLI config and the config-input directories as input.  The aggregated
   view of Routing::NexthopGroup::Config is instantiated in
   startNexthopGroupConfigMergeSm().
   '''
   global cliNexthopGroupConfigReadOnly
   global nexthopGroupConfigDir
   mg = entityManager.mountGroup()
   # cliNexthopGroupConfigReadOnly is used to directly access the
   # cli-config entity for use by the NexthopGroupConfigMergeSm.  The
   # cliNexthopGroupConfig is a ConfigMount Proxy that the plugin can write to,
   # and cliNexthopGroupConfigReadOnly can only be used as an input to the merge SM.
   # Any other uses in this plugin will break CLI sessions.
   cliNexthopGroupConfigReadOnly = mg.mount( 'routing/nexthopgroup/input/cli',
                                             'Routing::NexthopGroup::ConfigInput',
                                             'wi' )
   nexthopGroupConfigDir = mg.mount( 'routing/nexthopgroup/input/config',
                                     'Tac::Dir', 'ri' )
   mg.close( startNexthopGroupConfigMergeSm )

def Plugin( entityManager ):
   routing4.plugin( entityManager )
   routing6.plugin( entityManager )

   global routingHardwareStatus
   global routingHardwareNexthopGroupStatus
   global cliNexthopGroupConfig
   global nexthopGroupTunnelConfig
   global smashNhgStatus
   global nexthopGroupTunnelTable
   global myEntityManager

   myEntityManager = entityManager

   routingHardwareStatus = LazyMount.mount( entityManager,
                                            'routing/hardware/status',
                                            'Routing::Hardware::Status', 'r' )
   routingHardwareNexthopGroupStatus = LazyMount.mount( entityManager,
                                      'routing/hardware/nexthopgroup/status',
                                      'Routing::Hardware::NexthopGroupStatus', 'r' )
   cliNexthopGroupConfig = ConfigMount.mount( entityManager,
                                              'routing/nexthopgroup/input/cli',
                                              'Routing::NexthopGroup::ConfigInput',
                                              'wi' )
   nexthopGroupTunnelConfig = ConfigMount.mount( entityManager,
                                                 'tunnel/nexthopgroup/input/cli',
                                                 'Tunnel::NexthopGroup::ConfigInput',
                                                 'w' )
   smashNhgStatus = SmashLazyMount.mount( entityManager,
                                          'routing/nexthopgroup/entrystatus',
                                          'NexthopGroup::EntryStatus',
                                          SmashLazyMount.mountInfo( 'reader' ) )
   nexthopGroupTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.nexthopGroupTunnelTable, entityManager )

   setupCounterHelper()
   pluginInitNexthopGroupConfigMergeSm( entityManager )
