# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from CliModel import (
   Dict,
   Int,
   List,
   Model,
   Str,
   Bool,
   Enum,
   Submodel
)
from ArnetModel import (
   MacAddress,
   Ip4Address,
   Ip6Address,
   IpGenericAddress
)

class IpKernelModel( Model ):
   def addVrf( self, vrfDetails ):
      self._lines.extend( vrfDetails )
      self._lines.append( '' )

   def render( self ):
      for i in self._lines:
         print( i )

class ShowKernelIpCountersIcmp( IpKernelModel ):
   details = Dict( keyType=str, valueType=long, help="A mapping of ICMP(6) details "
                   "to ICMP(6) values" )
   inputHistogram = Dict( keyType=str, valueType=long, help="A mapping of ICMP(6) "
                          "input histogram details to input histogram values" )
   outputHistogram = Dict( keyType=str, valueType=long, help="A mapping of ICMP(6) "
                           "output histogram details to output histogram values" )

class ShowKernelIpCountersVrfDetails( IpKernelModel ):
   arista = Dict( keyType=str, valueType=long, help="A mapping of Arista details "
                  "to Arista values" )
   icmp = Submodel( valueType=ShowKernelIpCountersIcmp, help="VRF ICMP infomation",
                    optional=True )
   icmp6 = Submodel( valueType=ShowKernelIpCountersIcmp,
                     help="VRF ICMP6 information", optional=True )
   icmpMsg = Dict( keyType=str, valueType=long, help="A mapping of ICMPMsg "
                   "details to IMCPMsg values" )
   ip = Dict( keyType=str, valueType=long, help="A mapping of IP details to "
              "IP values", optional=True )
   ip6 = Dict( keyType=str, valueType=long, help="A mapping IP6 details to "
               "IP6 values", optional=True )
   ipExt = Dict( keyType=str, valueType=long, help="A mapping of IP extension "
                 "details to IP extension values" )
   tcp = Dict( keyType=str, valueType=long, help="A mapping of TCP details to "
               "TCP values" )
   tcpExt = Dict( keyType=str, valueType=long, help="A mapping of TCP extension "
                  "details to TCP extension values" )
   udp = Dict( keyType=str, valueType=long, help="A mapping of UDP details to "
               "UDP values", optional=True )
   udp6 = Dict( keyType=str, valueType=long, help="A mapping of UDP6 details to "
                "UDP6 values", optional=True )
   udpLite = Dict( keyType=str, valueType=long, help="A mapping of UDPLite details "
                   "to UDPLite values", optional=True )
   udpLite6 = Dict( keyType=str, valueType=long, help="A mapping of UDPLite6 "
                    "details to UDPLite6 values", optional=True )
   sctp = Dict( keyType=str, valueType=long, help="A mapping of SCTP details "
                "to SCTP values", optional=True )

   _lines = List( valueType=str, help="Show kernel ip(6) counters CLI ouput" )

class ShowKernelInterfaceCountersInet6( IpKernelModel ):
   address = Ip6Address( help="IPv6 address" )
   prefixlen = Int( help="Prefix length", optional=True )
   scopeid = Int( help="Scope ID", optional=True )

class ShowKernelInterfaceCountersRxTxDetails( IpKernelModel ):
   numPackets = Int( help="Number of packets" )
   numBytes = Int( help="Number of bytes" )
   numErrors = Int( help="Number of errors" )
   dropped = Int( help="Number of frames dropped" )
   overruns = Int( help="Number of overruns" )
   misaligned = Int( help="Number of misaligned frames", optional=True )
   carrier = Int( help="Number of carriers", optional=True )
   collisions = Int( help="Number of collisions", optional=True )

class ShowKernelInterfaceCountersVrfDetails( IpKernelModel ):
   flags = Str( help="Flag information" )
   mtu = Int( help="Maximum transmission unit" )
   ether = MacAddress( help="Ethernet(MAC) address", optional=True )
   txqueuelen = Int( help="Transaction queue length" )
   inet = Ip4Address( help="IPv4 internet networking", optional=True )
   inet6 = Dict( valueType=ShowKernelInterfaceCountersInet6,
                 help="A mapping of IPv6 internet networking to its details",
                 optional=True )
   netmask = Ip4Address( help="Netmask mask", optional=True )
   broadcast = Ip4Address( help="Broadcast address", optional=True )
   deviceInterrupt = Int( help="Device interrupt number", optional=True )
   loop = Str( help="Type of loop", optional=True )
   unspec = Str( help="Unspecified address", optional=True )
   rx = Submodel( valueType=ShowKernelInterfaceCountersRxTxDetails,
                  help="Packets received details" )
   tx = Submodel( valueType=ShowKernelInterfaceCountersRxTxDetails,
                  help="Packets transmitted details" )

class ShowKernelInterfaceCountersVrf( IpKernelModel ):
   interfaces = Dict( keyType=str, valueType=ShowKernelInterfaceCountersVrfDetails,
                      help="A mapping of interfaces to interface details" )

   _lines = List( valueType=str, help="Show kernel interface counters CLI ouput" )

class ShowKernelIpRouteDetails( IpKernelModel ):
   dev = Str( help="Device name", optional=True )
   proto = Str( help="Protocol type", optional=True )
   scope = Str( help="Route scope", optional=True )
   src = IpGenericAddress( help="Source address", optional=True )
   metric = Int( help="Metric value", optional=True )
   via = IpGenericAddress( help="Next hop address", optional=True )
   pref = Enum( values=( "low", "medium", "high" ),
                help="Preference value", optional=True )
   errorNum = Int( help="Error number", optional=True )
   routeType = Str( help="Route type", optional=True )
   linkdown = Bool( help="Route has linkdown flag", optional=True )
   dead = Bool( help="Route has dead flag", optional=True )
   onlink = Bool( help="Route has onlink flag", optional=True )
   pervasive = Bool( help="Route has pervasive flag", optional=True )
   offload = Bool( help="Route has offload flag", optional=True )
   notify = Bool( help="Route has notify flag", optional=True )
   unresolved = Bool( help="Route has unresolved flag", optional=True )

class ShowKernelIpRouteVrf( IpKernelModel ):
   routes = Dict( keyType=IpGenericAddress, valueType=ShowKernelIpRouteDetails,
                  help="A mapping of IP routes to route details" )

   _lines = List( valueType=str, help="Show kernel ip(v6) route CLI ouput" )

class ShowKernelInterfaceAddrInet( IpKernelModel ):
   validLft = Str( help="Valid lifetime" )
   preferredLft = Str( help="Preferred lifetime" )
   addr = IpGenericAddress( help="Inet(6) address", optional=True )
   brd = IpGenericAddress( help="Broadcast address", optional=True )

class ShowKernelInterfaceAddrDetails( IpKernelModel ):
   flags = Str( help="Interface flags" )
   mtu = Int( help="Maximum transmission unit" )
   qdisc = Str( help="Queuing discipline", optional=True )
   state = Enum( values=( "unknown", "notPresent", "down", "lowerLayerDown",
                          "testing", "dormant", "up" ), help="Interface state",
                          optional=True )
   group = Str( help="Group associated with interface", optional=True )
   qlen = Int( help="Queue length", optional=True )
   # Changed these two to Str values as addresses can be
   # IpGenericAddress or MacAddress values
   link = Str( help="Link address", optional=True )
   brd = Str( help="Broadcast address", optional=True )
   linkNetnsid = Int( help="Link network namespace id", optional=True )
   extraDetails = Dict( keyType=str, valueType=str, help="A mapping of extra "
                        "kernel details to their values", optional=True )
   inet = Dict( keyType=str, valueType=ShowKernelInterfaceAddrInet,
                help="A mapping of IPv4 internet networking scope to its details" )
   inet6 = Dict( keyType=str, valueType=ShowKernelInterfaceAddrInet,
                 help="A mapping of IPv6 internet networking scope to its details" )

class ShowKernelInterfaceAddrVrf( IpKernelModel ):
   interfaces = Dict( keyType=str, valueType=ShowKernelInterfaceAddrDetails,
                      help="A mapping of interface to its details" )

   _lines = List( valueType=str, help="Show kernel (ip) interface addr CLI ouput" )

class AclDetailsInvertedString( IpKernelModel ):
   value = Str( help="ACL details value" )
   inverted = Bool( help="Attribute is inverted" )

class AclDetailsInvertedIpAddress( IpKernelModel ):
   value = IpGenericAddress( help="ACL details value" )
   inverted = Bool( help="Attribute is inverted" )

class ShowKernelIpAclDetails( IpKernelModel ):
   numPackets = Int( help="Number of packets" )
   numBytes = Int( help="Number of bytes" )
   target = Str( help="Target value" )
   proto = Submodel( valueType=AclDetailsInvertedString, help="IP protocol" )
   opt = Submodel( valueType=AclDetailsInvertedString, help="Interface option" )
   inbound = Submodel( valueType=AclDetailsInvertedString, help="Inbound rule" )
   outbound = Submodel( valueType=AclDetailsInvertedString,
                        help="Outbound rule" )
   source = Submodel( valueType=AclDetailsInvertedIpAddress, help="Source address" )
   destination = Submodel( valueType=AclDetailsInvertedIpAddress,
                           help="Destination address" )
   aclExtraDetails = Str( help="ACL extra details", optional=True )

class ShowKernelIpAclChain( IpKernelModel ):
   policy = Str( help="Policy type", optional=True )
   chainPackets = Int( help="Number of packets in ACL", optional=True )
   chainBytes = Int( help="Number of bytes in ACL", optional=True )
   references = Int( help="Number of references", optional=True )
   details = List( valueType=ShowKernelIpAclDetails,
                   help="A list of the ACLs details" )

class ShowKernelIpAclVrf( IpKernelModel ):
   chainAcls = Dict( keyType=str, valueType=ShowKernelIpAclChain,
                     help="A mapping of ACLs to their details" )

   _lines = List( valueType=str, help="Show kernel IP(v6) ACL CLI ouput" )

class ShowKernelIpArpDetails( IpKernelModel ):
   dev = Str( help="ARP device name" )
   proxy = Bool( help="This neighbor is being proxied", optional=True )
   router = Bool( help="This neighbor is a router", optional=True )
   externallyLearned = Bool( help="This entry was externally learned",
                             optional=True )
   lladdr = MacAddress( help="Link layer address", optional=True )
   state = Enum( values=( "permanent", "noArp", "reachable", "stale", "none",
                          "incomplete", "delay", "probe", "failed" ),
                 help="State of neighbour entry" )

class ShowKernelIpArpVrf( IpKernelModel ):
   arpAddresses = Dict( keyType=IpGenericAddress, valueType=ShowKernelIpArpDetails,
                        help="A mapping of ARP addresses to their details" )

   _lines = List( valueType=str,
                  help="Show kernel (ip arp|ipv6 neighbors) CLI ouput" )
