# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliMatcher
import Ethernet
import IpAddrMatcher
import IraCommonCli
import IraRouteCommon
import Tac
from IpLibConsts import DEFAULT_VRF
from IraNexthopGroupCli import nexthopGroupSupportedGuard, nexthopGroupVrfSupported
from RoutingConsts import (
      defaultStaticRouteName,
      defaultStaticRoutePreference,
      defaultStaticRouteTag,
      defaultStaticRouteMetric
)

routeMatcherForConfig = CliMatcher.KeywordMatcher( 'route',
                                                   helpdesc='Manage static routes' )

prefixMatcher = IpAddrMatcher.ipPrefixExpr( 'Destination prefix',
                                            'Destination prefix mask',
                                            'Destination address with prefix',
                                            partial=True )

preferenceRangeMatcher = CliMatcher.IntegerMatcher( 1, 255,
      helpdesc='Administrative distance for this route' )

tagMatcherForConfig = CliMatcher.KeywordMatcher( 'tag', helpdesc='Route tag' )
tagNumberMatcher = CliMatcher.IntegerMatcher( 0, 4294967295,
                                              helpdesc='Tag number' )

metricMatcherForConfig = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='Metric for this route' )
metricValueMatcher = CliMatcher.IntegerMatcher( 0, 4294967295,
      helpdesc='Value of the route metric' )

ip = IraRouteCommon.Ip4()
routing = IraRouteCommon.routing( ip )

IntfId = Tac.Type( 'Arnet::IntfId' )
noRouteTableForVrfMsg = "IP Routing table for VRF %s does not exist."

def staticRoutingTable( vrfName ):
   return routing.routeConfig( vrfName )

@Tac.memoize
def iraIpRouteCliHelper():
   return Tac.newInstance( "Ira::IpRouteCliPluginHelper",
         ip.routingHardwareStatus() )

def isValidPrefix( prefix ):
   """Returns whether or not a prefix is a valid Arnet::Prefix; that is, that whether
   the least-significant (32-len) bits of the address are zero."""
   return iraIpRouteCliHelper().isValidPrefix( prefix, False )

def isValidPrefixWithError( mode, prefix, ucastOnly=False ):
   """Returns True if a prefix is valid.  If invalid, it adds a CLI error
   and returns False."""
   if not iraIpRouteCliHelper().isValidPrefix( prefix, ucastOnly ):
      mode.addError( iraIpRouteCliHelper().errorMsg )
      return False
   return True

def manageRoute( mode, prefix, nexthop, routeOptions=None, vrfName=DEFAULT_VRF,
                 leakToVrf=None, egressVrf=None ):
   if not isValidPrefixWithError( mode, prefix, ucastOnly=True ):
      return

   if not mode.session_.startupConfig() and 'nexthopGroupName' in nexthop and \
         nexthopGroupSupportedGuard( mode, None ) is not None:
      mode.addError( "Nexthop-Group not supported on this hardware platform" )
      return

   routeOptionsDict = dict( routeOptions if routeOptions else [] )
   preference = routeOptionsDict.get( 'preference' )
   tag = routeOptionsDict.get( 'tag' )
   nextHopName = routeOptionsDict.get( 'nextHopName' )
   trackingProto = routeOptionsDict.get( 'trackingProto' )
   metric = routeOptionsDict.get( 'metric' )

   if preference is None:
      preference = defaultStaticRoutePreference
   if tag is None:
      tag = defaultStaticRouteTag
   if nextHopName is None:
      nextHopName = defaultStaticRouteName
   if metric is None:
      metric = defaultStaticRouteMetric

   label = False
   labelVal = 0
   vtepAddr = ''
   vni = False
   vniVal = 0
   routerMac = ''
   vxlanIntf = IntfId( '' )
   nexthopGroupName = ''
   ipIntfName = ''
   hop = ''

   # nexthop is actually either a rule with mpls, evpn, nexthop or an intf,
   # and it is a dict, produced by a Cli OrRule.
   routeType = 'forward'
   if 'mpls' in nexthop:
      hop = nexthop[ 'mpls' ][ 'nexthop' ]
      label = True
      labelVal = int( nexthop[ 'mpls' ][ 'label' ] )
      # need to allow mcast address as nexthop here - we use a mcast addr as nexthop
      # and configure static arp for that address, to handle the case where nexthop
      # does not have a ipv4 address
      if not IraCommonCli.validNextHop4( mode, hop, vrfName, allowMcast=True ):
         return
   elif 'evpn' in nexthop:
      hop = '0.0.0.0'
      vtepAddr = nexthop[ 'evpn' ][ 'vtepAddr' ]
      if hasattr( vtepAddr, 'stringValue' ):
         vtepAddr = vtepAddr.stringValue
      vni = True
      vniVal = int( nexthop[ 'evpn' ][ 'vni' ] )
      routerMac = nexthop[ 'evpn' ][ 'routerMac' ]
      vxlanIntf = nexthop[ 'evpn' ][ 'vxlanIntf' ]
   elif 'nexthop' in nexthop:
      hop = nexthop[ 'nexthop' ]
      if not IraCommonCli.validNextHop4( mode, hop, vrfName ):
         return
   elif 'nexthop6' in nexthop:
      hop = nexthop[ 'nexthop6' ]
      if not IraCommonCli.validNextHop6( mode, hop, vrfName ):
         return
      hop = hop.stringValue
   elif 'Null0' in nexthop or 'Null 0' in nexthop:
      routeType = 'drop'
   elif 'routeCacheConnected' in nexthop:
      routeType = 'routeCacheConnected'
   elif 'nexthopGroupName' in nexthop:
      routeType = 'nexthopGroup'
      nexthopGroupName = nexthop[ 'nexthopGroupName' ]
   else:
      nhi = nexthop[ 'intf' ][ 'intf' ]
      hop = ip.intfAddrRoutesSupported and nexthop[ 'intf' ][ 'intfnexthop' ]
      if hop:
         if not IraCommonCli.validNextHop4( mode, hop, vrfName ):
            mode.addError( "Invalid next hop %s" % hop )
            return
      else:
         hop = '0.0.0.0'

      if not nhi.config():
         mode.addError( "Interface %s does not exist" % nhi.name )
         return

      if not nhi.routingSupported():
         mode.addError( "Interface %s is not routable" % nhi.name )
         return

      ipIntf = ip.config.ipIntfConfig.get( nhi.name )
      if not ipIntf:
         l3Config = ip.l3ConfigDir.newIntfConfig( nhi.name )
         ipIntf = ip.config.newIpIntfConfig( nhi.name )
         ipIntf.l3Config = l3Config
      ipIntfName = ipIntf.intfId

   if routeType == 'nexthopGroup' and vrfName != DEFAULT_VRF and \
         not nexthopGroupVrfSupported( mode ):
      mode.addError( "Nexthop group not supported in non-default vrf" )
      return

   bfdTracked = ( trackingProto == 'bfd' )
   if not leakToVrf:
      leakToVrf = ''
   if not egressVrf:
      egressVrf = ''
   rt = staticRoutingTable( vrfName )
   assert rt is not None

   if not iraIpRouteCliHelper().manageRoute( rt, vrfName, prefix, preference,
         routeType, leakToVrf, ipIntfName, label, labelVal, vni, vniVal, routerMac,
         vtepAddr, False, False, tag, egressVrf, hop, nextHopName, bfdTracked,
         nexthopGroupName, vxlanIntf, metric ):
      mode.addError( iraIpRouteCliHelper().errorMsg )

def noRoute( mode, prefix, nexthop, preference, vrfName, egressVrf=None ):
   if not isValidPrefixWithError( mode, prefix ):
      return

   # If egress VRF is the same as the VRF route is pointing to, there is no
   # meaning setting egress VRF in via
   if egressVrf is None or egressVrf == vrfName:
      egressVrf = ""

   rt = staticRoutingTable( vrfName )
   prefix = Arnet.IpGenPrefix( prefix )

   # optimization
   if preference:
      key = Tac.Value( "Routing::RouteKey",
                       prefix=prefix,
                       preference=preference )
      if key in rt.route:
         keys = [ key ]
      else:
         keys = []
   elif len( rt.route ) > 255:
      # If the routing table is big in this VRF test all possible
      # values of preference, it can be a lot faster than iterating
      # over all the routes.
      prefType = Tac.Value( "Routing::RoutePreference" )
      keys = []
      for pref in range( prefType.min, prefType.max + 1 ):
         key = Tac.Value( "Routing::RouteKey",
                          prefix=prefix,
                          preference=pref )
         if key in rt.route:
            keys.append( key )
   else:
      keys = rt.route.keys()

   for key in keys:
      if key.prefix != prefix:
         continue

      if preference and ( preference != key.preference ):
         mode.addError( " pref %s %s " % ( preference, key.preference ) )
         continue

      if not nexthop:
         # no ip route <prefix>
         # no nexthop or intf specified. Delete every route with this prefix.
         del rt.route[ key ]
         continue

      if 'Null0' in nexthop or 'Null 0' in nexthop:
         if rt.route[ key ].routeType == 'drop':
            del rt.route[ key ]
         continue
      elif 'routeCacheConnected' in nexthop:
         if rt.route[ key ].routeType == 'routeCacheConnected':
            del rt.route[ key ]
         continue

      def delRoutes( matching ):
         r = rt.route.get( key )
         rvias = r.via.keys()
         for v in rvias:
            if matching( v ):
               if len( rvias ) == 1:
                  del rt.route[ key ]
               else:
                  del r.via[ v ]
                  del r.viaNextHopName[ v ]
                  del r.bfdTracked[ v ]

      nullLabel = Tac.Type( 'Arnet::MplsLabel' ).null
      if 'intf' in nexthop:
         # no ip route <prefix> <intf> [hop]
         intf = nexthop[ 'intf' ][ 'intf' ]
         hop = ip.intfAddrRoutesSupported and nexthop[ 'intf' ][ 'intfnexthop' ]
         if not hop:
            hop = '0.0.0.0'
         hop = Arnet.IpGenAddr( hop )
         delRoutes( lambda via: via.intfId == intf.name and via.hop == hop
                    and via.mplsLabel == nullLabel )
      elif 'nexthopGroupName' in nexthop:
         # no ip route <prefix> nexthop-group <name>
         r = rt.route.get( key )
         if ( r.routeType == 'nexthopGroup' and
              r.nexthopGroupName == nexthop[ 'nexthopGroupName' ] ):
            del rt.route[ key ]
      elif 'mpls' in nexthop:
         # no ip route <prefix> label <label>
         hop = Arnet.IpGenAddr( nexthop[ 'mpls' ][ 'nexthop' ] )
         label = nexthop[ 'mpls' ][ 'label' ]
         delRoutes( lambda via: via.hop == hop and via.mplsLabel == label
                    and via.intfId == '' )
      elif 'nexthop6' in nexthop:
         hop = Arnet.IpGenAddr( nexthop[ 'nexthop6' ].stringValue )
         delRoutes( lambda via: via.hop == hop and via.intfId == ''
                    and via.egressVrf == egressVrf )
      elif 'evpn' in nexthop:
         vtepAddr = nexthop[ 'evpn' ][ 'vtepAddr' ]
         if hasattr( vtepAddr, 'stringValue' ):
            vtepAddr = vtepAddr.stringValue
         vtepAddr = Tac.Value( 'Arnet::IpGenAddr', vtepAddr )
         vni = nexthop[ 'evpn' ][ 'vni' ]
         vni = int( vni )
         routerMac = nexthop[ 'evpn' ][ 'routerMac' ]
         routerMac = Ethernet.convertMacAddrToCanonical( routerMac )
         delRoutes( lambda via: via.intfId == '' and
                    via.vni == vni and
                    via.vtepAddr.stringValue == vtepAddr.stringValue and
                    via.routerMac == routerMac )
      else:
         # no ip route <prefix> <hop>
         # no ip route <prefix> egress-vrf <egressVrf> <hop>
         hop = Arnet.IpGenAddr( nexthop[ 'nexthop' ] )
         delRoutes( lambda via: via.hop == hop and via.intfId == ''
                    and via.mplsLabel == nullLabel and via.egressVrf == egressVrf )
