# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for IP'''

import sys

import Arnet
import Arnet.MplsLib
from Arnet.Verify import verify
import BasicCli
import Cell
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Ip
import CliToken.Icmp
import ConfigMount
import Intf.IntfRange
import IntfCli
import Ip6AddrMatcher
import IpAddrMatcher
import IraIp6Cli
import IraCommonCli
import IraUrpfCli
import IraIpIntfCli
from IraIpIntfCli import zeroIpAddress
import IraRouteCommon
import LazyMount
import MultiRangeRule
import ShowCommand
import SmashLazyMount
import Tac
import TacSigint
from CliModel import UnknownEntityError
from CliPlugin.IraCommonModel import (
      ServiceRoutingProtoModelStatus,
      VrfNotInstalledBfd,
)
from CliPlugin.IraIpModel import (
      InterfaceAddress,
      IpAddress,
      IpRouteSummary,
      IpRouteSummaryForVrf,
      IpModel,
      IpMulticastRouting,
      IpStatus,
      IpStatuses,
      InterfaceAddressBrief,
      IpStatusBrief,
      IpStatusesBrief,
      VrfIpRibRoutes,
      VrfIpRoutes,
      VrfIpRoutesHost,
      bgpTotals,
      isisTotals,
      ospfTotals,
      ospfv3Totals,
      UrpfInterface,
      UrpfInterfaces,
)
from CliPlugin.IraCommonCli import NexthopCliExpression
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import (
      ALL_VRF_NAME,
      DEFAULT_VRF,
      VrfExprFactory,
      getAllVrfNames,
      vrfExists,
)
from CliPrint import CliPrint
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from IraIpRouteCliLib import (
      ip,
      manageRoute,
      noRoute,
      noRouteTableForVrfMsg,
      preferenceRangeMatcher,
      prefixMatcher,
      routeMatcherForConfig,
      routing,
      staticRoutingTable,
      tagMatcherForConfig,
      tagNumberMatcher,
      metricMatcherForConfig,
      metricValueMatcher,
)
from IraNexthopGroupCli import (
      nexthopGroupSupportedGuard,
      getNexthopGroupNames,
      nexthopGroupNamePattern,
)
from VirtualIntfRule import VirtualIntfMatcher

printer = CliPrint().lib

OptimizeProfileType = Tac.Type( 'Routing::Hardware::OptimizeProfileType' )
TriStateBoolEnum = Tac.Type( "Ip::TristateBool" )

# pylint: disable-msg=E0602
# Disabling E0602 filewide to prevent getting errors for using log messages

routingHardwareConfig = None
routingHardwareStatus = None
routingHardwareStatus6 = None
routingHardwareRouteStatus = None
ipStatus = None
ipConfig = None
l3Config = None
allVrfConfig = None
allVrfStatusLocal = None

noArpTableForVrfMsg = "ARP table for VRF %s does not exist."
notUnicastPrefix = "The specified prefix %s is not a unicast prefix."
VERIFY_MAX_ROUTES = 100

ipMatcherForConfig = CliToken.Ip.ipMatcherForConfig
ipMatcherForShow = CliToken.Ip.ipMatcherForShow
icmpMatcherForConfig = CliToken.Icmp.icmpMatcherForConfig
rateLimitUnreachableMatcher = CliToken.Icmp.rateLimitUnreachableMatcherForConfig

def vrfRoutingSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.vrfRoutingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsPushSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.mplsPushSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def routeCacheSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.routeCacheSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

# XXX If the following guard is modified, please update ArosTest plugin
# v4RouteWithV6NhSupported() as well.
def v4RouteWithV6NhSupportedGuard( mode, token ):
   # strictly speaking it's a bit of a hack to base this off the V6
   # ECMP supported capability, but that's really the deciding factor
   # in gated that causes us to not support it on Petra, so...
   assert routingHardwareStatus6
   if routingHardwareStatus6.ecmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

#-------------------------------------------------------------------------------
# The "ip routing [ipv6 interfaces] [vrf]" and
#     "no|default ip routing [keep-static-routes | delete-static-routes]
#       [addressless] [vrf]"
#     commands.
#-------------------------------------------------------------------------------

def setIpRouting( mode, no, vrfName, interfaces=False, delStaticRoutes=False ):
   assert vrfName != ''
   if not vrfName:
      vrfName = DEFAULT_VRF
   if not vrfExists( vrfName ):
      mode.addError( "No such VRF %s." % vrfName )
      return

   r = routing.config( vrfName )

   if no == 'default':
      if not interfaces:
         r.routing = False
      else:
         r.addresslessForwarding = TriStateBoolEnum.isInvalid
   elif no:
      if not interfaces:
         r.routing = False
      else:
         r.addresslessForwarding = TriStateBoolEnum.isFalse
   else:
      r.routing = True
      if interfaces:
         r.addresslessForwarding = TriStateBoolEnum.isTrue

   # only warn if not startup and there are static routes
   if ( no and
        delStaticRoutes is None and
        not mode.session_.startupConfig() and
        staticRoutingTable( vrfName ).route ):
      mode.addWarning( "Preserving static routes.  Use "
                       "'no ip routing delete-static-routes' to clear them." )
   if no and delStaticRoutes:
      # Delete all of the static routes if requested.
      with Tac.ActivityLockHolder():
         rt = staticRoutingTable( vrfName )
         for k in rt.route.keys():
            del rt.route[ k ]

vrfExprFactory = VrfExprFactory( helpdesc='Configure routing in a VRF',
                                 guard=vrfRoutingSupportedGuard )

class IpRoutingCmd( CliCommand.CliCommandClass ):
   syntax = '''ip routing [ VRF ]'''

   data = { "ip": ipMatcherForConfig,
            "routing": CliCommand.Node( matcher=CliMatcher.KeywordMatcher( "routing",
                                 helpdesc="Routing for IP packets" ),
                                 guard=routing.hwSupportedGuard ),
            "VRF": vrfExprFactory,
          }

   @staticmethod
   def handler( mode, args ):
      setIpRouting( mode, no=False, vrfName=args.get( "VRF" ) )

configMode.addCommandClass( IpRoutingCmd )

# NoIpRoutingCmd is defined separately from IpRoutingCmd because the help description
# for keyword routing is different in these two commands.
class NoIpRoutingCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = "ip routing " \
                       "[ ( keep-static-routes | delete-static-routes ) ] " \
                       "[ VRF ]"

   data = { "ip": ipMatcherForConfig,
            "routing": CliMatcher.KeywordMatcher( "routing",
                              helpdesc="Disable IP routing" ),
            "keep-static-routes": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                        "keep-static-routes",
                        helpdesc='Disable IP forwarding, ' \
                                 'but do not remove the configured static routes' ),
                  hidden=True ),
            "delete-static-routes": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                        "delete-static-routes",
                        helpdesc='Disable IP forwarding, ' \
                                 'and remove the configured static routes' ) ),
            "VRF": vrfExprFactory,
          }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delStaticRoutes = None
      if "delete-static-routes" in args:
         delStaticRoutes = True
      if "keep-static-routes" in args:
         delStaticRoutes = False
      setIpRouting( mode, no=True, vrfName=args.get( "VRF" ),
                    delStaticRoutes=delStaticRoutes )

configMode.addCommandClass( NoIpRoutingCmd )

class IpRoutingV6IntfCmd( CliCommand.CliCommandClass ):
   syntax = '''ip routing ipv6 interfaces [ VRF ]'''

   data = { "ip": ipMatcherForConfig,
            "routing": CliCommand.Node( matcher=CliMatcher.KeywordMatcher( "routing",
                                             helpdesc="Routing for IP packets" ),
                                        guard=routing.hwSupportedGuard ),
            "ipv6": "Global V4 Forwarding for V6 Interfaces",
            "interfaces": "Global V4 Forwarding for V6 Interfaces",
            "VRF": vrfExprFactory,
          }

   @staticmethod
   def handler( mode, args ):
      setIpRouting( mode, no=False,
                    vrfName=args.get( "VRF" ), interfaces=True )

configMode.addCommandClass( IpRoutingV6IntfCmd )

class NoIpRoutingV6IntfCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = '''ip routing ipv6 interfaces [ VRF ]'''

   data = { "ip": ipMatcherForConfig,
            "routing": CliMatcher.KeywordMatcher( "routing",
                                    helpdesc="Disable IP routing" ),
            "ipv6": "Global V4 Forwarding for V6 Interfaces",
            "interfaces": "Global V4 Forwarding for V6 Interfaces",
            "VRF": vrfExprFactory,
          }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setIpRouting( mode, no=True,
                    vrfName=args.get( "VRF" ), interfaces=True )

configMode.addCommandClass( NoIpRoutingV6IntfCmd )

#-------------------------------------------------------------------------------
# The "(no|default) ip icmp rate-limit-unreachable 0" command.
#-------------------------------------------------------------------------------
class IcmpUnreachableRateCmd( CliCommand.CliCommandClass ):
   syntax = '''ip icmp rate-limit-unreachable RATE'''

   noOrDefaultSyntax = '''ip icmp rate-limit-unreachable ...'''

   data = { "ip": ipMatcherForConfig,
            "icmp": icmpMatcherForConfig,
            "rate-limit-unreachable": rateLimitUnreachableMatcher,
            "RATE": CliMatcher.IntegerMatcher( 0, 0, helpdesc='Rate' ),
          }

   @staticmethod
   def handler( mode, args ):
      # Setting rate to 0 makes ICMP _not_ unreachable.
      routingHardwareConfig.icmpUnreachable = 'RATE' not in args

   noOrDefaultHandler = handler

configMode.addCommandClass( IcmpUnreachableRateCmd )

#-------------------------------------------------------------------------------
# The "[no] ip route" command.
#-------------------------------------------------------------------------------
nexthopAddrMatcher = IpAddrMatcher.IpAddrMatcher(
      'Address of the nexthop router' )
nexthop6AddrNodeHidden = CliCommand.Node(
      Ip6AddrMatcher.Ip6AddrMatcher( 'Address of the nexthop router' ),
      guard=v4RouteWithV6NhSupportedGuard,
      hidden=True )
intfAddrMatcher = IpAddrMatcher.IpAddrMatcher(
      "Forwarding router's address on destination interface" )
nullIntfMatcher = VirtualIntfMatcher( 'Null', 0, 0,
      helpdesc='Interface that drops all traffic' )
nexthopGroupNodeForConfig = CliCommand.Node(
                                  CliMatcher.KeywordMatcher( 'nexthop-group',
                                     helpdesc='Specify nexthop group name'),
                                  guard=nexthopGroupSupportedGuard )

nexthopGroupNameMatcher = CliMatcher.DynamicNameMatcher( getNexthopGroupNames,
                              'Nexthop group name', pattern=nexthopGroupNamePattern )

trackingProtocols = {}
trackingProtocolsHidden = {}

# call addTrackingProto before TrackingProtocolExpr
IraIp6Cli.addTrackingProto( trackingProtocols, trackingProtocolsHidden,
                            'bfd', 'Track this route using BFD' )

class TrackingProtocolExpr( CliCommand.CliExpression ):
   expression = 'TRACKING_PROTOCOLS | HIDDEN_TRACKING_PROTOCOLS'
   data = { 'TRACKING_PROTOCOLS': CliCommand.Node(
                                     CliMatcher.DynamicKeywordMatcher(
                                        lambda mode: trackingProtocols ),
                                     alias='trackingProto' ),
            'HIDDEN_TRACKING_PROTOCOLS': CliCommand.Node(
                                            CliMatcher.DynamicKeywordMatcher(
                                              lambda mode: trackingProtocolsHidden ),
                                            alias='trackingProto',
                                            hidden=True )
          }

vrfExprFactoryForConfig = VrfExprFactory( helpdesc='Configure routes in a VRF' )
labelMatcherForConfig = CliCommand.Node( CliMatcher.KeywordMatcher( 'label',
                                            helpdesc="Push an MPLS label" ),
                                         guard=mplsPushSupportedGuard )

def _intfValueFunc( mode, **kwargs ):
   kwargs.setdefault( 'intfnexthop', None )
   return kwargs

def intfAddrRoutesSupportedGuard( mode, token ):
   if ip.intfAddrRoutesSupported:
      return None
   return CliParser.guardNotThisPlatform

def manageIpRoute( mode, prefix, nexthop, routeOptions=None, vrfName=DEFAULT_VRF,
                   leakToVrf=None, egressVrf=None ):
   assert vrfName != ''
   if not vrfName:
      vrfName = DEFAULT_VRF

   if not vrfExists( vrfName ):
      mode.addError( "%s Create first." % noRouteTableForVrfMsg % vrfName )
      return

   manageRoute( mode, prefix, nexthop, routeOptions, vrfName, leakToVrf, egressVrf )

def noIpRoute( mode, prefix, nexthop, preference, vrfName, egressVrf ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   assert vrfName != ''

   if not vrfExists( vrfName ):
      mode.addError( "%s Create first." % noRouteTableForVrfMsg % vrfName )
      return

   noRoute( mode, prefix, nexthop, preference, vrfName, egressVrf )

class NexthopAddrExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP | NEXTHOP6'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'NEXTHOP6': nexthop6AddrNodeHidden
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'NEXTHOP6' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop6' ] = args.pop( 'NEXTHOP6' )

class NexthopOrIntfNexthopExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP | NEXTHOP6 | NULL0_EXPR | ( COMMON_INTF INTF_ADDR )'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'NEXTHOP6': nexthop6AddrNodeHidden,
            'NULL0_EXPR': IraCommonCli.Null0Expr,
            'COMMON_INTF': CliCommand.Node( CliMatcher.WrapperMatcher(
                                               IntfCli.Intf.matcher,
                                               priority=CliParser.PRIO_HIGH ),
                                            guard=intfAddrRoutesSupportedGuard ),
            'INTF_ADDR': intfAddrMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'NEXTHOP6' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop6' ] = args.pop( 'NEXTHOP6' )
      elif 'Null0' in args:
         del args[ 'Null0' ]
         args[ 'NEXTHOP_INTF' ][ 'Null0' ] = 'Null0'
      elif 'Null 0' in args:
         args[ 'NEXTHOP_INTF' ][ 'Null 0' ] = args.pop( 'Null 0' )
      elif 'COMMON_INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ]= args.pop( 'COMMON_INTF' )
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = args.pop( 'INTF_ADDR' )

class MplsLabelOrNexthopGroupOrIntfExpr( CliCommand.CliExpression ):
   expression = '( NH label LABEL_VALUE ) | ( nexthop-group NHG_NAME ) | ' \
                'INTF'
   data = { 'NH': IpAddrMatcher.IpAddrMatcher( 'Address of the nexthop router',
                                               priority=CliParser.PRIO_HIGH ),
            'label': labelMatcherForConfig,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher,
            'nexthop-group': nexthopGroupNodeForConfig,
            'NHG_NAME': nexthopGroupNameMatcher,
            'INTF': IntfCli.Intf.matcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'label' in args:
         args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NH' )
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )
      elif 'nexthop-group' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )
      elif 'INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

class NhMplsLabelExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( NH label LABEL_VALUE )'
   data = { 'NH': IpAddrMatcher.IpAddrMatcher( 'Address of the nexthop router',
                                               priority=CliParser.PRIO_HIGH ),
            'label': labelMatcherForConfig,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NH' )
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )

class NexthopGroupExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( nexthop-group NHG_NAME )'
   data = { 'nexthop-group': nexthopGroupNodeForConfig,
            'NHG_NAME': nexthopGroupNameMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )

class IpRouteNexthopOrComboCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-only routes, combo routes(nexthop+intf)'''
   syntax = 'ip route [ VRF ] PREFIX NEXTHOP_ADDR_OR_INTF ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) | ' \
            '( track TRACKING_PROTO ) | ( metric METRICVALUE ) } ] [ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                       maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'track': CliCommand.Node( IraCommonCli.trackMatcherForConfig,
                                      maxMatches=1 ),
            'TRACKING_PROTO': TrackingProtocolExpr,
            'metric': CliCommand.Node( metricMatcherForConfig,
                                       maxMatches=1 ),
            'METRICVALUE': CliCommand.Node( metricValueMatcher,
                                            maxMatches=1 ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )
      if 'track' in args:
         args[ 'OPTION' ].add( ( 'trackingProto',
                                 args.pop( 'trackingProto' )[ 0 ] ) )
      if 'metric' in args:
         args[ 'OPTION' ].add( ( 'metric', args.pop( 'METRICVALUE' ) ) )

      manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args[ 'OPTION' ],
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( IpRouteNexthopOrComboCmd )

class IpRouteMplsLabelCmd( CliCommand.CliCommandClass ):
   '''This class covers mpls route.'''
   syntax = 'ip route [ VRF ] PREFIX NEXTHOP_LABEL ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) } ] ' \
            '[ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'NEXTHOP_LABEL': NhMplsLabelExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                       maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

      manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args[ 'OPTION' ],
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( IpRouteMplsLabelCmd )

class IpRouteNexthopGroupCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-group route'''
   syntax = 'ip route [ VRF ] PREFIX NEXTHOP_GROUP ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) | ' \
            '( metric METRICVALUE ) } ] [ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'NEXTHOP_GROUP': NexthopGroupExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                       maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'metric': CliCommand.Node( metricMatcherForConfig,
                                       maxMatches=1 ),
            'METRICVALUE': CliCommand.Node( metricValueMatcher,
                                            maxMatches=1 ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )
      if 'metric' in args:
         args[ 'OPTION' ].add( ( 'metric', args.pop( 'METRICVALUE' ) ) )

      manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args[ 'OPTION' ],
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( IpRouteNexthopGroupCmd )

class IpRouteInterfaceCmd( CliCommand.CliCommandClass ):
   '''This class  covers interface routes'''
   syntax = 'ip route [ VRF ] PREFIX INTF ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) | ' \
            '( metric METRICVALUE ) } ] [ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'INTF': IntfCli.Intf.matcher,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                       maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'metric': CliCommand.Node( metricMatcherForConfig,
                                       maxMatches=1 ),
            'METRICVALUE': CliCommand.Node( metricValueMatcher,
                                            maxMatches=1 ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )
      if 'metric' in args:
         args[ 'OPTION' ].add( ( 'metric', args.pop( 'METRICVALUE' ) ) )

      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

      manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args[ 'OPTION' ],
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( IpRouteInterfaceCmd )

class IpRouteEgressVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip route [ VRF ] PREFIX EGRESS_VRF NEXTHOP_ADDR ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) } ] ' \
            '[ LEAK_VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'EGRESS_VRF': IraIp6Cli.EgressVrfCliExpr,
            'NEXTHOP_ADDR': NexthopAddrExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                       maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'LEAK_VRF': IraIp6Cli.LeakVrfCliExpr
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

   @staticmethod
   def handler( mode, args ):
      manageIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args[ 'OPTION' ],
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ),
                     egressVrf=args.get( 'EGRESS_VRF' ) )

configMode.addCommandClass( IpRouteEgressVrfCmd )

class NoIpRouteCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip route [ VRF ] PREFIX ' \
                       '[ ( EGRESS_VRF EGRESS_NEXTHOP ) |' \
                       '  NEXTHOP_ADDR_OR_INTF |' \
                       '  NON_NEXTHOP_ADDR_OR_INTF ] ' \
                       '[ PREFERENCE ] ...'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'EGRESS_VRF': IraIp6Cli.EgressVrfCliExpr,
            'EGRESS_NEXTHOP': NexthopAddrExpr,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'NON_NEXTHOP_ADDR_OR_INTF': MplsLabelOrNexthopGroupOrIntfExpr,
            'PREFERENCE': preferenceRangeMatcher
          }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noIpRoute( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                 args.get( 'PREFERENCE' ), args.get( 'VRF' ),
                 args.get( 'EGRESS_VRF' ) )

configMode.addCommandClass( NoIpRouteCmd )

cachedMatcherForConfig = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'cached', helpdesc='Route cache' ),
      guard=routeCacheSupportedGuard )

class IpRouteCachedCmd( CliCommand.CliCommandClass ):
   syntax = 'ip route [ VRF ] PREFIX cached'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'cached': cachedMatcherForConfig
            }

   @staticmethod
   def handler( mode, args ):
      if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
         mode.addWarning( "Routing protocols model multi-agent must be "
                          "configured for route cache configuration" )
      manageIpRoute( mode, args[ 'PREFIX' ], 'routeCacheConnected',
                     vrfName=args.get( 'VRF' ) )

configMode.addCommandClass( IpRouteCachedCmd )

class NoIpRouteCachedCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip route [ VRF ] PREFIX cached'
   data = { 'ip': ipMatcherForConfig,
            'route': routeMatcherForConfig,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': prefixMatcher,
            'cached': cachedMatcherForConfig
            }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noIpRoute( mode, args[ 'PREFIX' ], 'routeCacheConnected',
                 preference=None, vrfName=args.get( 'VRF' ), egressVrf=None )

configMode.addCommandClass( NoIpRouteCachedCmd )

#-------------------------------------------------------------------------------
# The "show ip" command.
#-------------------------------------------------------------------------------
mfibVrfConfig = None
mfib6VrfConfig = None

# Get the number of configured Vrrp Interfaces
fhrpVrrpIntfsHook = CliExtensions.CliHook()

# Check whether IPv4/IPv6 multicast routing is enabled
mCastRoutingHook = CliExtensions.CliHook()

def generateIpModel( mode, args ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
   assert vrfName
   r = routing.config( vrfName )
   ipModel = IpModel()
   ipModel.v4RoutingEnabled = r.routing

   # FHRP VRRP Interfaces Support information
   for hook in fhrpVrrpIntfsHook.extensions():
      # Only one fhrpVrrpIntfs hook is expected
      assert len( fhrpVrrpIntfsHook.extensions() ) == 1
      ipModel.vrrpIntfs = hook()

   # Supported Multicast routing
   for hook in mCastRoutingHook.extensions():
      # Only one mCastRouting hook is expected
      assert len( mCastRoutingHook.extensions() ) == 1
      ipModel.multicastRouting = IpMulticastRouting()
      ipModel.multicastRouting.ipMulticastEnabled = hook().get( 'v4' )
      ipModel.multicastRouting.ip6MulticastEnabled = hook().get( 'v6' )

   # Addressless Forwarding information
   if r.addresslessForwarding == 'isTrue':
      forwardingState = True
   elif r.addresslessForwarding == 'isFalse':
      forwardingState = False
   else:
      forwardingState = None
   ipModel.v6IntfForwarding = forwardingState

   # Populate CAPI model with the IPv4 URPF state
   ipModel.v4uRpfState = IraUrpfCli.getV4UrpfStatus()

   # Populate CAPI model with the IPv6 Ecmp state
   ( ipModel.v6RoutingEnabled, ipModel.v6EcmpInfo ) = IraIp6Cli.\
         generateIp6EcmpModel( mode, vrfName )

   # Populate CAPI model with IPv6 URPF state
   ipModel.v6uRpfState = IraUrpfCli.getV6UrpfStatus()
   return ipModel

class ShowIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip'''
   data = { 'ip': ipMatcherForShow }

   cliModel = IpModel
   handler = generateIpModel

BasicCli.addShowCommandClass( ShowIpCmd )

def intfAddressBrief( ipIntfConfig, ipIntfStatus ):
   # We first check ip/status to see the active ip addr
   # If we dont find a valid ip addr in status, we check to see
   # if one has been configured by cli.
   addr = ipIntfStatus.activeAddrWithMask
   if addr == zeroIpAddress:
      addr = ipIntfConfig.addrWithMask
   if addr == zeroIpAddress:
      addr = ipIntfConfig.virtualAddrWithMask

   intfAddrBrief = InterfaceAddressBrief(
      ipAddr=IpAddress( address=addr.address, maskLen=addr.len ) )
   if ipIntfStatus.unnumberedIntfId:
      intfAddrBrief.unnumberedIntf = ipIntfStatus.unnumberedIntfId
   return intfAddrBrief

ipv4Routable240ClassEHook = CliExtensions.CliHook()

#-------------------------------------------------------------------------------
# The "show ip interface [ intf | ipaddr | vrf ]" command.
#-------------------------------------------------------------------------------

def printIpIntfFull( mode, intfs=None ):
   results = IpStatuses()
   if intfs is None:
      return results
   for intf in intfs:
      if not intf.lookup():
         raise UnknownEntityError( "Interface %s does not exist" % intf.name )

      config = intf.config() or intf.dynConfig()
      result = IpStatus( description=intf.description(),
                         enabled=config.enabled,
                         mtu=intf.status().mtu )

      result.name = intf.name
      ( result.lineProtocolStatus, result.interfaceStatus ) = intf.getStatus()

      ipIntf = IraIpIntfCli.IpIntf( intf, mode.sysdbRoot,
                                createIfMissing=False )
      config = ipIntf.config()
      intfStatus = ipIntf.getStatus()
      interfaceAddress = InterfaceAddress()
      if config.addrSource == 'dhcp':
         interfaceAddress.dhcp = True

      interfaceAddress.primaryIp = IpAddress(
                                   address=intfStatus.activeAddrWithMask.address,
                                   maskLen=intfStatus.activeAddrWithMask.len )
      # We first look at ip/status, if there's no ip address then we look at
      # ip/config ( written to by the cli ).
      if intfStatus.activeAddrWithMask == zeroIpAddress:
         interfaceAddress.primaryIp = IpAddress(
                                      address=config.addrWithMask.address,
                                      maskLen=config.addrWithMask.len )
      if config.unnumberedIntfId:
         interfaceAddress.unnumberedIntf = config.unnumberedIntfId
      if intfStatus.useVirtualAddr:
         interfaceAddress.primaryIp = IpAddress( address='0.0.0.0', maskLen=0 )

      for sa in config.secondaryWithMask:
         interfaceAddress.secondaryIpsOrderedList.append( IpAddress(
            address=sa.address, maskLen=sa.len ) )

      interfaceAddress.virtualIp = \
          IpAddress( address=config.virtualAddrWithMask.address,
                     maskLen=config.virtualAddrWithMask.len )

      for sa in config.virtualSecondaryWithMask:
         interfaceAddress.virtualSecondaryIpsOrderedList.append( IpAddress(
            address=sa.address, maskLen=sa.len ) )

      ipIntfConfig = ipIntf.config()
      ipIntfStatus = ipIntf.getStatus()
      result.interfaceAddressBrief = intfAddressBrief( ipIntfConfig, ipIntfStatus )
      ipv4Routable240ClassE = True
      for hook in ipv4Routable240ClassEHook.extensions():
         if not hook():
            ipv4Routable240ClassE = False
      result.ipv4Routable240 = ipv4Routable240ClassE
      # We don't let user configure broadcast address yet, so it's hard-coded here.
      interfaceAddress.broadcastAddress = "255.255.255.255"

      result.interfaceAddress = interfaceAddress
      result.proxyArp = config.proxyArpEnabled
      result.localProxyArp = config.localProxyArpEnabled
      result.gratuitousArp = config.gratuitousArpAccepted
      ipIntfStatus = ipStatus.ipIntfStatus.get( intf.name )
      if ipIntfStatus !=  None:
         result.addresslessForwarding = ipIntfStatus.addresslessForwarding
      else:
         result.addresslessForwarding = "isInvalid"
      if config.vrf:
         result.vrf = config.vrf

      # URPF information
      result.urpf = 'disable'
      if intf.name in ip.config.ipIntfConfig:
         result.urpf = ip.config.ipIntfConfig[ intf.name ].urpf.mode

      result.directedBroadcastEnabled = False
      if intf.name in ip.config.ipIntfConfig:
         enabled = ip.config.ipIntfConfig[ intf.name ].directedBroadcastEnabled
         result.directedBroadcastEnabled = enabled

      result.maxMssIngress = 0
      result.maxMssEgress = 0
      if intf.name in ip.config.ipIntfConfig:
         maxMssIngress = ip.config.ipIntfConfig[ intf.name ].maxMssIngress
         maxMssEgress = ip.config.ipIntfConfig[ intf.name ].maxMssEgress
         result.maxMssIngress = maxMssIngress
         result.maxMssEgress = maxMssEgress

      results.interfaces[ intf.name ] = result

   return results

#-------------------------------------------------------------------------------
# The "show ip interface [ intf | ipaddr | vrf ] brief" command.
#-------------------------------------------------------------------------------

def printIpIntfBrief( mode, intfs=None ):
   briefModel = IpStatusesBrief()
   if intfs is not None:
      for intf in intfs:
         if not intf.lookup():
            raise UnknownEntityError( "Interface %s does not exist" % intf.name )

         ( lps, intfStatus ) = intf.getStatus()
         briefStatus = IpStatusBrief( name=intf.name,
                                      lineProtocolStatus=lps,
                                      interfaceStatus=intfStatus,
                                      mtu=intf.status().mtu )

         ipIntf = IraIpIntfCli.IpIntf( intf, mode.sysdbRoot, createIfMissing=False )
         ipIntfConfig = ipIntf.config()
         ipIntfStatus = ipIntf.getStatus()
         briefStatus.interfaceAddress = \
               intfAddressBrief( ipIntfConfig, ipIntfStatus )
         ipv4Routable240ClassE = True
         for hook in ipv4Routable240ClassEHook.extensions():
            if not hook():
               ipv4Routable240ClassE = False
         briefStatus.ipv4Routable240 = ipv4Routable240ClassE
         briefModel.interfaces[ intf.name ] = briefStatus

   return briefModel

def isPrimaryIpAddr( ipIntfStatus, intfAddr ):
   return ipIntfStatus.activeAddrWithMask.address == intfAddr

def isSecondIpAddr( ipIntfStatus, intfAddr ):
   for secondAddr in ipIntfStatus.activeSecondaryWithMask:
      if secondAddr.address == intfAddr:
         return True
   return False

def isPrimaryOrSecondIpAddr( ipIntfStatus, intfAddr ):
   return ( isPrimaryIpAddr( ipIntfStatus, intfAddr )
            or isSecondIpAddr( ipIntfStatus, intfAddr ) )

def filterIntfByRouteable( mode, intfs, ifname=None ):
   filtered = []

   for intf in intfs:
      if intf.routingCurrentlySupported():
         filtered.append( intf )

   if not filtered:
      msg = "No IP capable interfaces"
      if ifname is not None:
         names = list( ifname.intfNames() )
         if len( names ) == 1:
            msg = "%s does not support IP" % names[ 0 ]
         else:
            msg = "No IP capable interfaces in range"

      raise UnknownEntityError( msg )

   return filtered

def filterIntfByVrf( mode, intfs, vrf ):
   if vrf is None or not vrfExists( vrf ):
      raise UnknownEntityError( noRouteTableForVrfMsg % vrf )

   return [ intf for intf in intfs if IraIpIntfCli.IpIntf( intf, mode,
                                       createIfMissing=False ).config().vrf == vrf ]

def filterIntfByIntf( mode, ifname ):
   intfs = IntfCli.Intf.getAll( mode, ifname )
   if not intfs:
      raise UnknownEntityError( "Interface does not exist" )

   return intfs

def filterIntfByAddr( mode, intfs, addr ):
   filtered = []
   for ipIntfStatus in ipStatus.ipIntfStatus.itervalues():
      if isPrimaryOrSecondIpAddr( ipIntfStatus, addr ):
         intfType, _ = Intf.IntfRange.intfTypeFromName( ipIntfStatus.intfId )
         intf = intfType.getCliIntf( mode, ipIntfStatus.intfId )
         filtered.append( intf )

   return filtered

def filterIpIntf( mode, dispFcn, intfOrAddr=None, vrfName=None ):
   if intfOrAddr and isinstance( intfOrAddr, MultiRangeRule.IntfList ):
      # Filter by interface name.
      intfs = filterIntfByIntf( mode, intfOrAddr )
      intfs = filterIntfByRouteable( mode, intfs, intfOrAddr )
   elif intfOrAddr and not isinstance( intfOrAddr, MultiRangeRule.IntfList ):
      # Filter by ip address.
      intfs = IntfCli.Intf.getAll( mode )
      intfs = filterIntfByRouteable( mode, intfs )
      intfs = filterIntfByAddr( mode, intfs, intfOrAddr )
   else:
      # Get all routeale intefaces.
      intfs = IntfCli.Intf.getAll( mode )
      intfs = filterIntfByRouteable( mode, intfs )

   if vrfName:
      # Filter by vrf.
      intfs = filterIntfByVrf( mode, intfs, vrfName )

   return dispFcn( mode, intfs )

def showIpIntfFull( mode, args ):
   intfOrAddr = args.get( 'ADDR' ) or args.get( 'INTFS' )
   vrfName = args.get( 'VRF' )
   return filterIpIntf( mode, printIpIntfFull, intfOrAddr, vrfName )

def showIpIntfBrief( mode, args ):
   intfOrAddr = args.get( 'ADDR' ) or args.get( 'INTFS' )
   vrfName = args.get( 'VRF' )
   return filterIpIntf( mode, printIpIntfBrief, intfOrAddr, vrfName )

vrfKwMatcher = CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF instance' )
vrfNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: allVrfConfig.vrf,
                                                'VRF name' )

interfaceAfterShowMatcher = CliMatcher.KeywordMatcher(
   'interface',
   helpdesc='Details related to IPv4 interfaces' )

class ShowIpIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ip interface
                [ INTFS | ADDR ]
                [ vrf VRF ]
            """
   data = { 'ip': ipMatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'ADDR': IpAddrMatcher.IpAddrMatcher( 'Match this IP Address' ),
            'vrf': vrfKwMatcher,
            'VRF': vrfNameMatcher
          }
   cliModel = IpStatuses

   handler = showIpIntfFull

BasicCli.addShowCommandClass( ShowIpIntfCmd )

briefKwMatcher = CliMatcher.KeywordMatcher( 'brief', helpdesc='Condensed output' )

class ShowIpIntfBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ip interface
                [ INTFS | ADDR ]
                [ vrf VRF ]
                brief
            """
   data = { 'ip': ipMatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'ADDR': IpAddrMatcher.IpAddrMatcher( 'Match this IP Address' ),
            'vrf': vrfKwMatcher,
            'VRF': vrfNameMatcher,
            'brief': briefKwMatcher
          }
   cliModel = IpStatusesBrief

   handler = showIpIntfBrief

BasicCli.addShowCommandClass( ShowIpIntfBriefCmd )

#-------------------------------------------------------------------------------
# The "show ip route gateway" command.
#-------------------------------------------------------------------------------
def warnIfRoutingDisabled( mode, vrfName=DEFAULT_VRF ):
   # OSPF (and other protocols in future) allow creating instances even before
   # corresponding VRF is defined. In such cases, vrf/config/<vrfname> may not
   # exist yet.
   if vrfName and not vrfExists( vrfName ):
      mode.addWarning( "IP routing not enabled" )
   elif not routing.config( vrfName ).routing:
      mode.addWarning( "IP routing not enabled" )

#-------------------------------------------------------------------------------
# The "show ip route host" command.
#-------------------------------------------------------------------------------
def showRouteHost( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName and not vrfExists( vrfName ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( "IP routing table %s does not exist." % vrfName )
         return None
   model = IraRouteCommon.showRouteHostCommon( mode, vrfName, quantify,
                        allVrfStatusLocal, routing, allVrfConfig )
   return model

#-------------------------------------------------------------------------------
# The "show ip route [ vrf <vrfName>|all ] [ <prefix> ] fec" command.
#-------------------------------------------------------------------------------
def showRouteFec( mode, vrfName=None, prefix=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfFecRoutesModel = IraRouteCommon.VrfFecRoutesModel()
   if isinstance( prefix, str ):
      prefix = Arnet.Prefix( prefix )
   if vrfName == ALL_VRF_NAME:
      vrfFecRoutesModel.setShowVrfs()
      vrf = DEFAULT_VRF
      vrfFecRoutesModel.vrfs[ vrf ] = routing.showFecs( mode,
                                                        prefix=prefix,
                                                        vrfName=vrf )
      vrfs = allVrfStatusLocal.vrf
   else:
      vrfs = [ vrfName ]

   for vrf in vrfs:
      vrfFecRoutesModel.vrfs[ vrf ] = routing.showFecs( mode,
                                                        prefix=prefix,
                                                        vrfName=vrf )
   return vrfFecRoutesModel

#-------------------------------------------------------------------------------
# The "show ip route fec summary" command.
#-------------------------------------------------------------------------------
def showIpFecSummary( mode, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   return routing.showFecs( mode, vrfName=vrfName, summary=True )

#-------------------------------------------------------------------------------
# The "show ip route [vrf <vrfName>|all] ..." command.
#-------------------------------------------------------------------------------
routeMatcherAfterShowIp = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Entries used for routing packets' )

ipPrefixExpr = IpAddrMatcher.ipPrefixExpr(
   'Match this IP address', 'Match this subnet mask',
   'Match this IP prefix', partial=True,
   overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )

ipAddrExpr = IpAddrMatcher.IpAddrMatcher( helpdesc='Match this IP address' )

detailKwAfterShowIpRoute = CliMatcher.KeywordMatcher( 'detail',
                                                      helpdesc='All routes' )

sharedCountObj = object()

def _protocolMatcher( protocolName, helpdesc ):
   return CliCommand.Node( CliMatcher.KeywordMatcher(
                              protocolName, helpdesc=helpdesc ),
                           sharedMatchObj=sharedCountObj, maxMatches=1,
                           alias='PROTOCOL' )

class ProtocolCliExpression( CliCommand.CliExpression ):
   expression = "( bgp | connected | kernel | martian | dynamic-policy | " \
                "ospf | ospfv3 | isis | rip | static | vxlan-control-service | " \
                "aggregate | cached )"
   data = { 'bgp': _protocolMatcher( 'bgp', 'Show only BGP routes' ),
            'connected': _protocolMatcher( 'connected',
                                           'Show only connected routes' ),
            'kernel': _protocolMatcher( 'kernel', 'Show only kernel routes' ),
            'martian': _protocolMatcher( 'martian', 'Show only kernel routes' ),
            'dynamic-policy': _protocolMatcher( 'dynamic-policy',
                                                'Show only Dynamic Policy routes' ),
            'ospf': _protocolMatcher( 'ospf', 'Show only OSPF routes' ),
            'ospfv3': _protocolMatcher( 'ospfv3', 'Show only OSPFv3 routes' ),
            'isis': _protocolMatcher( 'isis', 'Show only IS-IS routes' ),
            'rip': _protocolMatcher( 'rip', 'Show only RIP routes' ),
            'static': _protocolMatcher( 'static', 'Show only static routes' ),
            'vxlan-control-service': _protocolMatcher( 'vxlan-control-service',
                                          'Show only VXLAN Control Service routes' ),
            'aggregate': _protocolMatcher( 'aggregate',
                                           'Show only aggregate routes' ),
            'cached': _protocolMatcher( 'cached', 'Show only Route Cache routes' )
          }

def showRoute( mode, prefix=None, optionsSet=None, vrfName=None,
               routeFilter=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   outputFormat = printer.stringToOutputFormat( mode.session_.outputFormat_ )
   fd = sys.stdout.fileno()

   # Return with a warning if a multicast prefix is specified
   if prefix is not None:
      savedPrefix = prefix
      if isinstance( prefix, str ):
         prefix = Arnet.Prefix( prefix )
      if not prefix.isUnicast:
         mode.addWarning( notUnicastPrefix % savedPrefix )
         return VrfIpRoutes

   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( noRouteTableForVrfMsg % vrfName )
         return None

   p = printer.initPrinter( fd, outputFormat, True )
   printer.start( p )
   printer.startDict( p, "vrfs" )
   if vrfName == ALL_VRF_NAME:
      vrfs = sorted( allVrfStatusLocal.vrf )
      vrfs.insert( 0, "default" )
   else:
      vrfs = [ vrfName ]

   for vrf in vrfs:
      # make sure we don't attempt to write if SIGINT happened outside the
      # immediateMode section (since RouteSorter handles those internally)
      #
      # Additionally, "VRF: ..." header is printed inside RouterSorter C++ code
      # for safety when pager exits, see BUG284545
      TacSigint.check()

      printer.startDict( p, vrf )
      keepGoing = showRoutePerVrf( p, mode, prefix, optionsSet, vrf, routeFilter )
      printer.endDict( p, vrf )
      if not keepGoing:
         break

   printer.endDict( p, "vrfs" )
   printer.end( p )
   printer.deinitPrinter( p )
   return VrfIpRoutes

def showRoutePerVrf( p, mode, prefix=None, optionsSet=None, vrfName="default",
                     routeFilter=None ):
   optionsDict = dict( optionsSet or [] )
   detail = 'detail' in optionsDict
   longerPrefixes = 'longerPrefixes' in optionsDict
   quantify = 'quantify' in optionsDict.itervalues()
   protocol = optionsDict.get( 'protocol' )
   if isinstance( prefix, str ):
      prefix = Arnet.Prefix( prefix )
   # create the outputFormat here, since we will not have access to mode later
   # down the stack, in routing.showRoute
   fmt = mode.session_.outputFormat()
   requestedRevision = mode.session_.requestedModelRevision()
   # In revision 2 of the VrfIpRoutes CAPI model (IraIpModel.py) we moved
   # the Nexthop-Group submodel from the per-route level to the per-via level.
   # If revision 1 is requested, we "degrade" the JSON output by picking
   # the first Nexthop-Group that a route points to and putting it in the per-route
   # level Nexthop-Group submodel.
   nhgModelRevision = 2

   # revision 3 prints the nested tunnel representation for routes resolving over
   # tunnels. If a lower revision is requested, it will flatten the tunnel entries,
   # and print the final nexthop, interface and combined label stack
   flatTunnelRevision = 3

   json = mode.session_.outputFormatIsJson()
   degradeNhgModel = json and requestedRevision < nhgModelRevision
   flattenTunnelOverTunnel = json and requestedRevision < flatTunnelRevision

   keepGoing = routing.showRoute( prefix, None, protocol, detail, vrfName,
                                  longerPrefixes, quantify=quantify, fmt=fmt,
                                  degradeNhgModel=degradeNhgModel,
                                  flattenTunnelOverTunnel=flattenTunnelOverTunnel,
                                  routeFilter=routeFilter )
   IraRouteCommon.warnIfRoutingDisabledPrinter( p, mode, vrfName, routing,
                                                allVrfConfig )
   return keepGoing


vrfExprFactoryForShow = VrfExprFactory( helpdesc='Show routes in a VRF',
                                        inclDefaultVrf=True )

def showIpRouteAdapter( mode, args, argsList ):
   ipPrefix = args.pop( 'PREFIX', None )
   if ipPrefix is not None:
      args[ 'PREFIX' ] = ipPrefix
   elif 'ADDR' in args:
      args[ 'PREFIX' ] = args.pop( 'ADDR' )

   args[ 'optionSet' ] = set()
   if 'detail' in args:
      del args[ 'detail' ]
      args[ 'optionSet' ].add( ( 'detail', 'detail' ) )
   if 'PROTOCOL' in args:
      args[ 'optionSet' ].add( ( 'protocol', args.pop( 'PROTOCOL' ) ) )
   if 'quantify' in args:
      del args[ 'quantify' ]
      args[ 'optionSet' ].add( ( 'quantify', 'quantify' ) )

class ShowIpRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ip route
                [ VRF ]
                [ PREFIX | ADDR ]
                [ { detail | PROTOCOL | quantify } ]
                [ NEXTHOP ]
            """
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'PREFIX': ipPrefixExpr,
            'ADDR': ipAddrExpr,
            'detail': CliCommand.Node( detailKwAfterShowIpRoute,
                                       maxMatches=1 ),
            'PROTOCOL': ProtocolCliExpression,
            'quantify': CliCommand.Node( CliMatcher.KeywordMatcher( 'quantify',
                                         helpdesc='Print the time taken' ),
                                         hidden=True, maxMatches=1 ),
            'NEXTHOP': NexthopCliExpression,
            }
   cliModel = VrfIpRoutes

   adapter = showIpRouteAdapter

   @staticmethod
   def handler( mode, args ):
      return showRoute( mode, prefix=args.get( 'PREFIX' ),
               optionsSet=args[ 'optionSet' ], vrfName=args.get( 'VRF' ),
               routeFilter=args.get( 'nexthop-matchers' ) )

BasicCli.addShowCommandClass( ShowIpRouteCmd )

#-------------------------------------------------------------------------------
# The "show ip route static not-installed bfd" command
#-------------------------------------------------------------------------------

def showV4NotInstalledBfd( mode, vrfName=None, optionsSet=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   if not ( vrfName is None or vrfName == ALL_VRF_NAME or vrfExists( vrfName ) ):
      mode.addError( "IP routing table %s does not exist." % vrfName )
      return None

   vrfs = []
   if vrfName == ALL_VRF_NAME:
      for vrf in sorted( allVrfStatusLocal.vrf ):
         vrfs.append(vrf)
      vrfs.insert( 0, DEFAULT_VRF )
   elif vrfName is not None:
      vrfs.append( vrfName )
   else:
      vrfs.append( DEFAULT_VRF )

   return IraCommonCli.showCommonNotInstalledBfd( vrfs=vrfs )

class ShowNotInstalledBfdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] static not-installed bfd'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'static': 'Show only static routes',
            'not-installed': 'Show not installed routes',
            'bfd': 'Not installed routes tracked by Bfd'
          }
   cliModel = VrfNotInstalledBfd

   @staticmethod
   def handler( mode, args ):
      return showV4NotInstalledBfd( mode, vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowNotInstalledBfdCmd )

def vrfSummaryCommon( ipRouteSummaryModel, vrfs ):
   routeCountByTypeDict = { 'connected':0,
                            'static':0,
                            'staticPersistent':0,
                            'staticNonPersistent':0,
                            'vcs':0,
                            'staticNexthopGroup':0,
                            'rip':0,
                            'internal':0,
                            'attached':0,
                            'dynamicPolicy':0,
                            'aggregate':0 }

   ospfBgpIsisSpecific = { 'ospfTotal':0,
                           'ospfIntraArea':0,
                           'ospfInterArea':0,
                           'ospfExternal1':0,
                           'ospfExternal2':0,
                           'nssaExternal1':0,
                           'nssaExternal2':0,
                           'ospfv3IntraArea':0,
                           'ospfv3InterArea':0,
                           'ospfv3External1':0,
                           'ospfv3External2':0,
                           'ospfv3NssaExternal1':0,
                           'ospfv3NssaExternal2':0,
                           'ospfv3Total':0,
                           'bgpTotal':0,
                           'bgpInternal':0,
                           'bgpExternal':0,
                           'bgpLocal': 0,
                           'isisTotal':0,
                           'isisLevel1':0,
                           'isisLevel2':0 }

   for vrfName in vrfs:
      # Force mount routing.config as it needs to be
      # passed to RouteSummary and was lazy-mounted earlier.
      # Tac.newInstance does not like unmounted proxies
      # pylint: disable=protected-access
      if isinstance( routing.config( vrfName ), LazyMount._Proxy ):
         LazyMount.force( routing.config( vrfName ))
      if isinstance( routing.routeConfig( vrfName ), LazyMount._Proxy ):
         LazyMount.force( routing.routeConfig( vrfName ) )
      routeSummary = Tac.newInstance( "Ira::RouteSummary",
                                   routing.config( vrfName ),
                                   routing.routeConfig( vrfName ),
                                   routing.rStatus( vrfName ),
                                   routing.attachRouteStatus( vrfName ),
                                   routing.vrfIdMap(),
                                   routing.ip.routeCacheRouteConfig,
                                   vrfName )
      routeSummary.loadInternalStructures()
      routeSummary.loadAttachedRoutes()
      routeSummary.loadRouteCacheRoutes()

      vrfMaskLen = dict( routeSummary.routeCountByMask )
      for key, value in vrfMaskLen.iteritems():
         if key not in ipRouteSummaryModel.maskLen:
            ipRouteSummaryModel.maskLen[key] = value
         else:
            ipRouteSummaryModel.maskLen[key] += value

      fibSummaryTable = routeSummary.routeCountByType
      for source in routeCountByTypeDict:
         if source in fibSummaryTable:
            routeCountByTypeDict[ source ] += fibSummaryTable[ source ]

      for source in ospfBgpIsisSpecific:
         if source in fibSummaryTable:
            ospfBgpIsisSpecific[ source ] += fibSummaryTable[ source ]

      ipRouteSummaryModel.totalRoutes += routeSummary.totalNumRoutes

   ipRouteSummaryModel.connected = routeCountByTypeDict[ 'connected' ]
   ipRouteSummaryModel.static = routeCountByTypeDict[ 'static' ]
   ipRouteSummaryModel.staticPersistent = routeCountByTypeDict[ 'staticPersistent' ]
   ipRouteSummaryModel.staticNonPersistent = \
       routeCountByTypeDict[ 'staticNonPersistent' ]
   ipRouteSummaryModel.vcs = routeCountByTypeDict[ 'vcs' ]
   ipRouteSummaryModel.staticNexthopGroup = \
       routeCountByTypeDict[ 'staticNexthopGroup' ]
   ipRouteSummaryModel.rip = routeCountByTypeDict[ 'rip' ]
   ipRouteSummaryModel.internal = routeCountByTypeDict[ 'internal' ]
   ipRouteSummaryModel.attached = routeCountByTypeDict[ 'attached' ]
   ipRouteSummaryModel.aggregate = routeCountByTypeDict[ 'aggregate' ]
   ipRouteSummaryModel.dynamicPolicy = routeCountByTypeDict[ 'dynamicPolicy' ]


   ospfTotalsModel = ospfTotals()
   ospfTotalsModel.ospfTotal = ospfBgpIsisSpecific[ 'ospfTotal' ]
   ospfTotalsModel.ospfIntraArea = ospfBgpIsisSpecific[ 'ospfIntraArea' ]
   ospfTotalsModel.ospfInterArea = ospfBgpIsisSpecific[ 'ospfInterArea' ]
   ospfTotalsModel.ospfExternal1 = ospfBgpIsisSpecific[ 'ospfExternal1' ]
   ospfTotalsModel.ospfExternal2 = ospfBgpIsisSpecific[ 'ospfExternal2' ]
   ospfTotalsModel.nssaExternal1 = ospfBgpIsisSpecific[ 'nssaExternal1' ]
   ospfTotalsModel.nssaExternal2 = ospfBgpIsisSpecific[ 'nssaExternal2' ]
   ipRouteSummaryModel.ospfCounts = ospfTotalsModel

   ospfv3TotalsModel = ospfv3Totals()
   ospfv3TotalsModel.ospfv3Total = ospfBgpIsisSpecific[ 'ospfv3Total' ]
   ospfv3TotalsModel.ospfv3IntraArea = ospfBgpIsisSpecific[ 'ospfv3IntraArea' ]
   ospfv3TotalsModel.ospfv3InterArea = ospfBgpIsisSpecific[ 'ospfv3InterArea' ]
   ospfv3TotalsModel.ospfv3External1 = ospfBgpIsisSpecific[ 'ospfv3External1' ]
   ospfv3TotalsModel.ospfv3External2 = ospfBgpIsisSpecific[ 'ospfv3External2' ]
   ospfv3TotalsModel.ospfv3NssaExternal1 = \
       ospfBgpIsisSpecific[ 'ospfv3NssaExternal1' ]
   ospfv3TotalsModel.ospfv3NssaExternal2 = \
       ospfBgpIsisSpecific[ 'ospfv3NssaExternal2' ]
   ipRouteSummaryModel.ospfv3Counts = ospfv3TotalsModel

   bgpTotalsModel = bgpTotals()
   bgpTotalsModel.bgpTotal = ospfBgpIsisSpecific[ 'bgpTotal' ]
   bgpTotalsModel.bgpExternal = ospfBgpIsisSpecific[ 'bgpExternal' ]
   bgpTotalsModel.bgpInternal = ospfBgpIsisSpecific[ 'bgpInternal' ]
   bgpTotalsModel.bgpLocal = ospfBgpIsisSpecific[ 'bgpLocal' ]
   ipRouteSummaryModel.bgpCounts = bgpTotalsModel

   isisTotalsModel = isisTotals()
   isisTotalsModel.isisTotal = ospfBgpIsisSpecific[ 'isisTotal' ]
   isisTotalsModel.isisLevel1 = ospfBgpIsisSpecific[ 'isisLevel1' ]
   isisTotalsModel.isisLevel2 = ospfBgpIsisSpecific[ 'isisLevel2' ]
   ipRouteSummaryModel.isisCounts = isisTotalsModel

   return ipRouteSummaryModel

#-------------------------------------------------------------------------------
# The "show ip route [vrf <vrfName>] summary" command.
#-------------------------------------------------------------------------------
def showRouteSummary( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfsToShow = []

   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( "Ip routing table %s does not exist." % vrfName )
         return None

   if vrfName == ALL_VRF_NAME:
      vrfsToShow = [ "default" ]
      vrfsToShow.extend( sorted( allVrfStatusLocal.vrf ) )
   else:
      vrfsToShow = [ vrfName ]

   svcRoutingProtoModelStatus = ServiceRoutingProtoModelStatus(
      operatingProtoModel = l3Config.protocolAgentModel,
      configuredProtoModel = l3Config.configuredProtocolAgentModel )
   ipSummary = IpRouteSummary()

   ipSummary.protoModelStatus = svcRoutingProtoModelStatus
   for vrf in vrfsToShow:
      result = showRouteSummaryForVrf( mode, vrfName=vrf,
                                       quantify=quantify )
      if result:
         ipSummary.vrfs[ vrf ] = result

   return ipSummary



def showRouteSummaryForVrf( mode, vrfName=None, quantify=False ):
   beforeSum = Tac.now()
   IpRouteSummaryModel = IpRouteSummaryForVrf()
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ( vrfName is None or vrfExists( vrfName ) ):
      mode.addError( "IP routing table %s does not exist." % vrfName )
      return None

   if not routing.rStatus( vrfName ):
      return None

   vrfs = [vrfName]
   IpRouteSummaryModel.totalRoutes = 0
   IpRouteSummaryModel = vrfSummaryCommon( IpRouteSummaryModel, \
               vrfs )
   # pylint: disable=protected-access
   if quantify:
      afterSum = Tac.now()
      IpRouteSummaryModel._quantify = afterSum-beforeSum
   return IpRouteSummaryModel

summaryAfterShowIpRouteMatcher = CliMatcher.KeywordMatcher( 'summary',
                                     helpdesc='IP routing table summary' )
quantifyKwMatcher = CliCommand.Node( CliMatcher.KeywordMatcher( 'quantify',
                                       helpdesc='Print the time taken' ),
                                     hidden=True )

class ShowIpRouteSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] summary [ quantify ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'summary': summaryAfterShowIpRouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = IpRouteSummary

   @staticmethod
   def handler( mode, args ):
      quantify = True if 'quantify' in args else False
      return showRouteSummary( mode, vrfName=args.get( 'VRF' ),
                               quantify=quantify )

BasicCli.addShowCommandClass( ShowIpRouteSummaryCmd )

hostAfterShowIpRouteMatcher = CliMatcher.KeywordMatcher(
   'host',
   helpdesc='Host routes' )

fecMatcher = CliMatcher.KeywordMatcher( 'fec', helpdesc="show route adjacencies" )

summaryMatcherAfterShowIp = CliMatcher.KeywordMatcher( 'summary',
                                    helpdesc='show adjacency size and distribution' )

class ShowIpRouteHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] host [ quantify ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'host': hostAfterShowIpRouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = VrfIpRoutesHost

   @staticmethod
   def handler( mode, args ):
      quantify = True if 'quantify' in args else False
      return showRouteHost( mode, args.get( 'VRF' ), quantify=quantify )

BasicCli.addShowCommandClass( ShowIpRouteHostCmd )

class ShowIpRouteFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] [ PREFIX | ADDR ] fec'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'PREFIX': ipPrefixExpr,
            'ADDR': ipAddrExpr,
            'fec': fecMatcher
          }

   cliModel = IraRouteCommon.VrfFecRoutesModel

   @staticmethod
   def adapter( mode, args, argsList ):
      ipPrefix = args.pop( 'PREFIX', None )
      if ipPrefix is not None:
         args[ 'PREFIX' ] = ipPrefix
      elif 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   @staticmethod
   def handler( mode, args ):
      return showRouteFec( mode, prefix=args.get( 'PREFIX' ),
                           vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIpRouteFecCmd )

class ShowIpFecSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] fec summary'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'fec': fecMatcher,
            'summary': summaryMatcherAfterShowIp
            }
   cliModel = IraRouteCommon.FecSummaryModel

   @staticmethod
   def handler( mode, args ):
      return showIpFecSummary( mode, vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIpFecSummaryCmd )

#-------------------------------------------------------------------------------
# The "show ip route [vrf vrfName] (prefix) longer-prefixes" command.
#-------------------------------------------------------------------------------

longerPrefixesMatcherAfterShowIpRoute = CliMatcher.KeywordMatcher(
   'longer-prefixes', helpdesc='Match longer prefixes' )

def showRouteLongerPrefixes( mode, prefix=None, optionsSet=None, vrfName=None ):
   # pylint check fails if appended directly to set
   optionsDict = dict( optionsSet or [] )
   optionsDict[ 'longerPrefixes' ] = 'longer-prefixes'
   optionsSet = optionsDict.items()
   return showRoute(mode, prefix, optionsSet, vrfName)

class ShowIpRouteLongerPrefixesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route [ VRF ] ( PREFIX | ADDR ) ' \
            'longer-prefixes [ { detail | PROTOCOL | quantify } ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'VRF': vrfExprFactoryForShow,
            'PREFIX': ipPrefixExpr,
            'ADDR': ipAddrExpr,
            'longer-prefixes': longerPrefixesMatcherAfterShowIpRoute,
            'detail': CliCommand.Node( detailKwAfterShowIpRoute,
                                       maxMatches=1 ),
            'PROTOCOL': ProtocolCliExpression,
            'quantify': CliCommand.Node( CliMatcher.KeywordMatcher( 'quantify',
                                            helpdesc='Print the time taken' ),
                                         hidden=True, maxMatches=1 )
          }
   cliModel = VrfIpRoutes

   adapter = showIpRouteAdapter

   @staticmethod
   def handler( mode, args ):
      return showRouteLongerPrefixes( mode, prefix=args.get( 'PREFIX' ),
                                      optionsSet=args[ 'optionSet' ],
                                      vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIpRouteLongerPrefixesCmd )

#-------------------------------------------------------------------------------
# The "show ip route vrf all summary brief" command.
#-------------------------------------------------------------------------------

def showIpRouteVrfAllSumBrief( mode, quantify=False ):
   beforeSum = Tac.now()
   vrfs = []

   for vrfName in sorted(allVrfStatusLocal.vrf):
      vrfs.append(vrfName)
   vrfs.insert( 0, DEFAULT_VRF )

   ipRouteSummaryBriefModel = IpRouteSummaryForVrf()
   ipRouteSummaryBriefModel.maskLen = {}
   ipRouteSummaryBriefModel.totalRoutes = 0
   ipRouteSummaryBriefModel.vrfCount = len( vrfs )
   ipRouteSummaryBriefModel = vrfSummaryCommon( ipRouteSummaryBriefModel, \
         vrfs )

   # pylint: disable=protected-access
   if quantify:
      afterSum = Tac.now()
      ipRouteSummaryBriefModel._quantify = afterSum-beforeSum
   return ipRouteSummaryBriefModel

allKwMatcher = CliMatcher.KeywordMatcher( ALL_VRF_NAME,
            helpdesc='Show summary of all vrfs' )
vrfForShowMatcher = CliMatcher.KeywordMatcher( 'vrf',
            helpdesc='Specify the vrf' )

class ShowIpRouteSummaryBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route vrf all summary brief [ quantify ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'vrf': vrfForShowMatcher,
            'all': allKwMatcher,
            'summary': summaryAfterShowIpRouteMatcher,
            'brief': briefKwMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = IpRouteSummaryForVrf

   @staticmethod
   def handler( mode, args ):
      quantify = 'quantify' in args
      return showIpRouteVrfAllSumBrief( mode, quantify=quantify )

BasicCli.addShowCommandClass( ShowIpRouteSummaryBriefCmd )

#-------------------------------------------------------------------------------
# Determine if assigning the given address/mask to some interface in Ira
# should be allowed

# return a pair of a boolean with a string, where the string is an error or
# warning message
#-------------------------------------------------------------------------------
def canSetIntfIp( intfName, ipAddrWithMask, secondary=False, mode=None ):
   config = routing.routeConfig( vrfName=None, intfName=intfName )
   return IraRouteCommon.canSetIntfIpHelper( config, intfName, ipAddrWithMask,
                                             'ipv4', True )

#-------------------------------------------------------------------------------
# The Resilient Ecmp command
# The syntax of the command is:
#  ip hardware fib ecmp <prefix> capacity <C> redundancy <R>
#  no/default ip hardware fib ecmp <prefix>
#  no/default ip hardware fib ecmp
#-------------------------------------------------------------------------------

def maxResilientEcmpCapacityRangeFn( mode ):
   if routingHardwareStatus is None or routingHardwareStatus.maxEcmp == 0:
      return( 1, sys.maxint )
   else :
      return( 2, routingHardwareStatus.maxEcmp )

def maxResilientEcmpRedundancyRangeFn( mode ):
   if routingHardwareStatus is None or routingHardwareStatus.maxResilientEcmp == 0:
      return( 1, sys.maxint )
   else:
      return( 1, routingHardwareStatus.maxResilientEcmp / 2 )

def resilientEcmpGuard( mode, token ):
   if routingHardwareStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

def resilientEcmpPrefixIs( mode, prefix, capacity, redundancy ) :

   # if C * R > maxEcmp value, throw max value error
   if ( routingHardwareStatus.maxResilientEcmp and
        ( capacity * redundancy ) > routingHardwareStatus.maxResilientEcmp ):
      mode.addError( "Maximum value supported for capacity * redundancy is %d"
          % ( routingHardwareStatus.maxResilientEcmp ) )
      return
   ecmpConfig = Tac.Value( "Routing::Hardware::ResilientEcmpConfig", \
                                 capacity, redundancy )
   # IpAddr.IpPrefixRule makes string (unlike Ip6Addr.Ip6PRefixRule) so prefix is a
   # IpAddrWithFullMaskTest
   genPrefix = Arnet.IpGenPrefix( prefix )
   routingHardwareConfig.resilientEcmpPrefix[ genPrefix ] = ecmpConfig

def noResilientEcmp( mode , prefix=None ):

   # if there is a prefix, delete prefix, otherwise delete all
   if prefix is None:
      routingHardwareConfig.resilientEcmpPrefix.clear()
      return
   # IpAddr.IpPrefixRule makes string (unlike Ip6Addr.Ip6PRefixRule) so prefix is a
   # IpAddrWithFullMask
   genPrefix = Arnet.IpGenPrefix( prefix )
   del routingHardwareConfig.resilientEcmpPrefix[ genPrefix ]

hardwareHelpDesc = "Configure hardware-specific parameters"
fibHelpDesc = "Routing table"
ecmpHelpDesc = "Configure ECMP routes"
resilienceHelpDesc = "Configure Resilience in ECMP routes"

class ResilientEcmpCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib ecmp resilience PREFIX " \
            "capacity CAPACITY redundancy REDUNDANCY"
   noOrDefaultSyntax = "ip hardware fib ecmp resilience [ PREFIX ] ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "ecmp": CliCommand.Node( matcher=CliMatcher.KeywordMatcher( "ecmp",
                                                helpdesc=ecmpHelpDesc ),
                                     guard=resilientEcmpGuard ),
            "resilience": resilienceHelpDesc,
            "PREFIX": IpAddrMatcher.ipPrefixExpr(
                         "Network address", "Network mask", "Prefix",
                         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                         maskKeyword=False ),
            "capacity": "specify capacity value",
            "CAPACITY": CliMatcher.DynamicIntegerMatcher(
                                       maxResilientEcmpCapacityRangeFn,
                                       helpdesc="capacity value" ),
            "redundancy": "specify redundancy value",
            "REDUNDANCY": CliMatcher.DynamicIntegerMatcher(
                                       maxResilientEcmpRedundancyRangeFn,
                                       helpdesc="redundancy value" )
            }

   @staticmethod
   def handler( mode, args ):
      resilientEcmpPrefixIs( mode, args[ "PREFIX" ], args[ "CAPACITY" ],
                             args[ "REDUNDANCY" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noResilientEcmp( mode, prefix=args.get( "PREFIX" ) )

configMode.addCommandClass( ResilientEcmpCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib exact-match prefix-length <length>" command
#  This command is made hidden starting from ga.evans release
#-------------------------------------------------------------------------------

def routesInExactMatchGuard( mode, token ):
   if routingHardwareStatus.prefixLenSupportedInExactMatch:
      return None
   return CliParser.guardNotThisPlatform

def prefixLengthGuard( mode, token ):
   if int( token ) not in routingHardwareStatus.prefixLenSupportedInExactMatch:
      return CliParser.guardNotThisPlatform
   return None

lenData = { str( i ): 'Prefix length %d' % i for i in range( 0, 33 ) }

def expandLenGuard( mode, token ):
   if int( token ) not in range( 8, 32 ):
      return CliParser.guardNotThisPlatform
   return None

def compressLenGuard( mode, token ):
   # We do not support prefix compression feature for prefix length part of
   # prefixLenNotSupportedInCompression
   if int( token ) in routingHardwareStatus.prefixLenNotSupportedInCompression:
      return CliParser.guardNotThisPlatform
   return None

restartL3AgentWarnStr = "Please restart layer 3 forwarding agent to ensure IPv4 " \
                        "routes are "

def setRoutesInExactMatch( mode, prefixLenSet=None, expandLenSet=None,
                           compressLenSet=None, urpf=None ):
   configuredLengths = map( int, prefixLenSet or [] )
   expandLengths = map( int, expandLenSet ) if expandLenSet else []
   compressLengths = map( int, compressLenSet ) if compressLenSet else []
   supportedLengths = routingHardwareStatus.prefixLenSupportedInExactMatch.keys()
   startupConfig = mode.session_.startupConfig()

   if not startupConfig and ( len( configuredLengths ) >
          routingHardwareStatus.numberOfPrefixLengthsInExactMatch ):
      mode.addError( "This platform supports %d lookups in the exact match table" %
                     routingHardwareStatus.numberOfPrefixLengthsInExactMatch )
      return
   numberOfNonNibblePrefixLengths = 0
   for configuredLength in configuredLengths:
      if configuredLength % 4:
         numberOfNonNibblePrefixLengths += 1
   if not startupConfig and ( numberOfNonNibblePrefixLengths >
         routingHardwareStatus.numberOfNonNibblePrefixLengthsInExactMatch ):
      mode.addError( "This platform supports %d non-nibble aligned lookups in the "
                     "exact match table" % routingHardwareStatus.\
                           numberOfNonNibblePrefixLengthsInExactMatch )
      return
   if not startupConfig and ( len( compressLengths ) >
         routingHardwareStatus.numberOfCompressedPrefixLengthsInExactMatch ):
      mode.addError( "This platform supports %d compressed prefix lengths in the "
                     "exact match table" % routingHardwareStatus.\
                           numberOfCompressedPrefixLengthsInExactMatch )
      return

   if urpf:
      if not startupConfig and ( len( configuredLengths ) >
           routingHardwareStatus.numberOfPrefixLengthInExactMatchWithUrpf ):
         mode.addError( "This platform supports %d lookups in the exact match table"
                        " along with Unicast RPF" %
                     routingHardwareStatus.numberOfPrefixLengthInExactMatchWithUrpf )
         return
      if numberOfNonNibblePrefixLengths:
         mode.addError( "This platform supports %d non-nibble aligned lookups in the"
               " exact match table along with Unicast RPF" % routingHardwareStatus.
                        numberOfNonNibblePrefixLengthsInExactMatchWithUrpf )
         return

   # Check value of the max prefix-expansion range in routingHwStatus
   # allow prefix-lengths for expansion in the range of
   # < REM-prefix-length - val> and <REM-prefix-length - 1 >
   val = routingHardwareStatus.maxExpandPrefixLenRange + 1
   for length in expandLengths:
      prefixLenAllowed = False
      for i in range( 1, val ):
         if length + i in configuredLengths:
            prefixLenAllowed = True
            break
      if not prefixLenAllowed:
         mode.addError( "Cannot expand prefix-length %d. No prefix-length in "
                        "range %d to %d is optimized." %
                        ( length, length + 1, length + val - 1 ) )
         return
      if length in configuredLengths:
         # We do not allow a prefix-length in the expanded list if it is already
         # present in the prefix list.
         # For e.g., /23 cannot be expanded at the same time it is optimized:
         # ip hardware fib optimize prefix-length 23 24 expand 22 23
         mode.addError( "Cannot expand prefix-length %d. It is already optimized."
                        % length )
         return
   val = routingHardwareStatus.maxCompressionPefixLenRange + 1
   for length in compressLengths:
      prefixLenAllowed = False
      for i in range( 1, val ):
         if length - i in configuredLengths:
            prefixLenAllowed = True
            break
      if not prefixLenAllowed:
         mode.addError( "Cannot compress prefix-length %d. Prefix-length %d is "
                        "not optimized." %
                        ( length, length - val + 1 ) )
         return
      if length in configuredLengths or length in expandLengths:
         # We do not allow a prefix-length in the compressed list if it is already
         # present in the prefix list or expanded list.
         # For e.g., /23 cannot be compressed at the same time it is optimized:
         # ip hardware fib optimize prefix-length 20 23 compress 23
         # For e.g., /21 cannot be compressed at the same time it is expanded:
         # ip hardware fib optimize prefix-length 20 23 expand 21 22 compress 21
         mode.addError( "Cannot compress prefix-length %d. It is already"
                        " optimized." % length )
         return

   # Keep track of configuration changes that will require an agent restart.
   # If remConfigChanged, we dont really care about the expandConfigChanged.
   # In future, we can eliminate the restart requirement if only expansion
   # configuration changes.
   remConfigChanged = False
   prefixProfileConfigChanged = False
   processingDone = False
   urpfConfigChanged = False

   # Disable optimization with profiles if enabled
   if routingHardwareConfig.optimizeProfileName:
      routingHardwareConfig.optimizeProfileName = ''
      prefixProfileConfigChanged = True

   # Add all the prefix-lengths in current REM configuration, if valid
   # If this was not enabled in previous config, remConfigChanged = True
   for length in configuredLengths:
      if not startupConfig and length not in supportedLengths:
         continue
      if length not in routingHardwareConfig.prefixLenInExactMatch:
         remConfigChanged = True
      else:
         prevConfig = routingHardwareConfig.prefixLenInExactMatch[ length ] == length
         if not prevConfig:
            remConfigChanged = True
      routingHardwareConfig.prefixLenInExactMatch[ length ] = length

   if urpf:
      if not routingHardwareConfig.urpfExactMatchEnabled:
         routingHardwareConfig.urpfExactMatchEnabled = True
         urpfConfigChanged = True
   else:
      if routingHardwareConfig.urpfExactMatchEnabled:
         routingHardwareConfig.urpfExactMatchEnabled = False
         urpfConfigChanged = True

   # Check all the previously configured REM prefix-lengths. If not
   # set in current configuration, delete from config and
   # remConfigChanged = true
   for length in routingHardwareConfig.prefixLenInExactMatch:
      if routingHardwareConfig.prefixLenInExactMatch[ length ] == length:
         if length not in configuredLengths:
            del routingHardwareConfig.prefixLenInExactMatch[ length ]
            remConfigChanged = True

   # If any REM configuration changed, clear all the existing expansion
   # configuration, otherwise check and delete the values not configured now.
   if remConfigChanged:
      for length in routingHardwareConfig.prefixLenInExactMatch:
         if routingHardwareConfig.prefixLenInExactMatch[ length ] != length:
            del routingHardwareConfig.prefixLenInExactMatch[ length ]
   else:
      for length, remLength in  \
            routingHardwareConfig.prefixLenInExactMatch.iteritems():
         if length not in expandLengths and \
               length not in configuredLengths:
            del routingHardwareConfig.prefixLenInExactMatch[ length ]
            remConfigChanged = True
   # Add all the prefix-lengths in current expand configuration. If this was not
   # enabled in previous config, expandConfigChanged = True
   for length in expandLengths:
      for remLength in sorted( configuredLengths ):
         if remLength < length:
            continue
         if length not in routingHardwareConfig.prefixLenInExactMatch:
            remConfigChanged = True
         routingHardwareConfig.prefixLenInExactMatch[ length ] = remLength
         break
   # Add all the prefix-lengths in current compress configuration. If this was not
   # enabled in previous config, remConfigChanged = True
   for length in compressLengths:
      for remLength in configuredLengths:
         if remLength >= length:
            continue
         for i in range( 1, val ):
            if length - i == remLength:
               if length not in routingHardwareConfig.prefixLenInExactMatch:
                  remConfigChanged = True
               routingHardwareConfig.prefixLenInExactMatch[ length ] = remLength
               processingDone = True
               break
         if processingDone:
            break
   warn = remConfigChanged or prefixProfileConfigChanged or urpfConfigChanged
   if warn:
      mode.addWarning( restartL3AgentWarnStr + "optimized" )

def unsetRoutesInExactMatch( mode ):
   # Keep track of configuration changes that will require an agent restart.
   # In future, we can eliminate the restart requirement if only expansion
   # configuration changes.
   remConfigChanged = False
   prefixProfileConfigChanged = False
   urpfConfigChanged = False

   # Disable optimization with profiles if enabled
   if routingHardwareConfig.optimizeProfileName:
      routingHardwareConfig.optimizeProfileName = ''
      prefixProfileConfigChanged = True

   if routingHardwareConfig.urpfExactMatchEnabled:
      routingHardwareConfig.urpfExactMatchEnabled = False
      urpfConfigChanged = True

   # 'no ip hardware fib optimize prefix-length ...' will delete all existing
   # REM, expansion and compression configs. Set the remConfigChanged if there
   # were previously enabled REM prefixes.
   if routingHardwareConfig.prefixLenInExactMatch:
      remConfigChanged = True
   routingHardwareConfig.prefixLenInExactMatch.clear()
   warn = remConfigChanged or prefixProfileConfigChanged or urpfConfigChanged
   if warn:
      mode.addWarning( restartL3AgentWarnStr + "not optimized" )

def enableRoutesInExactMatch( mode, prefixLenSet=None ):
   setRoutesInExactMatch( mode, prefixLenSet=prefixLenSet or [] )

class ExactMatchPrefixLenCmd( CliCommand.CliCommandClass ):
   hidden = True
   syntax = "ip hardware fib exact-match prefix-length LENGTH"
   noOrDefaultSyntax = "ip hardware fib exact-match prefix-length ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "exact-match": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "exact-match",
                        helpdesc="Configure routes in exact-match table" ),
                  guard=routesInExactMatchGuard ),
            "prefix-length": "Length of the prefix in bits",
            "LENGTH": CliCommand.SetEnumMatcher( lenData, guard=prefixLengthGuard )
          }

   @staticmethod
   def handler( mode, args ):
      enableRoutesInExactMatch( mode, args[ "LENGTH" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unsetRoutesInExactMatch( mode )

configMode.addCommandClass( ExactMatchPrefixLenCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize prefix-length <length>
#               [ expand < prefix-lengths > ]
#               [ compress < prefix-lengths > ] [ urpf ]" command
#-------------------------------------------------------------------------------

def urpfExactMatchSupportedGuard( mode, token ):
   if not routingHardwareStatus.urpfExactMatchSupported:
      return CliParser.guardNotThisPlatform
   return None

class OptimizePrefixLenCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize prefix-length LENGTH " \
            "[ { ( expand EXPANDLEN ) | " \
            "( compress COMPRESSLEN ) } ] [ urpf ]"

   noOrDefaultSyntax = "ip hardware fib optimize prefix-length ..." \

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "optimize": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "optimize",
                        helpdesc="Optimize IPv4 routes" ),
                  guard=routesInExactMatchGuard ),
            "prefix-length": "Length of the prefix in bits",
            "LENGTH": CliCommand.SetEnumMatcher( lenData, guard=prefixLengthGuard ),
            "expand": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "expand",
                     helpdesc="Expand and optimize routes" ),
                  maxMatches=1 ),
            "EXPANDLEN": CliCommand.SetEnumMatcher( lenData, guard=expandLenGuard ),
            "compress": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "compress",
                     helpdesc="Compress and optimize routes" ),
                  maxMatches=1 ),
            "COMPRESSLEN": CliCommand.SetEnumMatcher( lenData,
                  guard=compressLenGuard ),
            "urpf": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "urpf",
                     helpdesc="Enable RPF lookup" ),
                  guard=urpfExactMatchSupportedGuard )
          }

   @staticmethod
   def handler( mode, args ):
      setRoutesInExactMatch( mode, args[ "LENGTH" ], args.get( "EXPANDLEN" ),
                             args.get( "COMPRESSLEN" ), args.get( "urpf" ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unsetRoutesInExactMatch( mode )

configMode.addCommandClass( OptimizePrefixLenCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize prefixes profile internet|urpf-internet"
# command
#-------------------------------------------------------------------------------

def enableOptimizePrefixes( mode, profileType='' ):
   # If the profileType is not set, then add
   if profileType and profileType != routingHardwareConfig.optimizeProfileName:
      # Delete any previously set expansion
      routingHardwareConfig.prefixLenInExactMatch.clear()
      if profileType == OptimizeProfileType.urpfInternet: # pylint: disable-msg=R1703
         routingHardwareConfig.urpfExactMatchEnabled = True
      else:
         routingHardwareConfig.urpfExactMatchEnabled = False
      # Set the profileType in optimizeProfileName
      routingHardwareConfig.optimizeProfileName = profileType
      mode.addWarning( restartL3AgentWarnStr + "optimized" )

def supportedProfileType( mode ):
   supportedProfiles = {}
   # Today, we have profiles just for Sand platforms
   # Add condition and path if this is supported on other platforms in future
   supportedProfiles[ OptimizeProfileType.internet ] = \
         'Optimize prefixes for Internet'
   if routingHardwareStatus.urpfExactMatchSupported:
      supportedProfiles[ "urpf-internet" ] = \
              'Optimize prefixes for Internet with urpf'
   return supportedProfiles

class OptimizePrefixesCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize prefixes profile PROFILE"

   noOrDefaultSyntax = "ip hardware fib optimize prefixes ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "optimize": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "optimize",
                        helpdesc="Optimize IPv4 routes" ),
                  guard=routesInExactMatchGuard ),
            "prefixes": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "prefixes",
                        helpdesc="Optimize prefixes" ),
                  guard=routesInExactMatchGuard ),
            "profile": CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( "profile",
                        helpdesc="Optimize based on profiles" ),
               guard=routesInExactMatchGuard ),
            "PROFILE": CliMatcher.DynamicKeywordMatcher( supportedProfileType )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if "PROFILE" in args and args[ "PROFILE" ] == "urpf-internet":
         args[ "PROFILE" ] = OptimizeProfileType.urpfInternet

   @staticmethod
   def handler( mode, args ):
      enableOptimizePrefixes( mode, profileType=args[ "PROFILE" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unsetRoutesInExactMatch( mode )

configMode.addCommandClass( OptimizePrefixesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib optimize [ disable-vrf <vrf-name> ]" command
#-------------------------------------------------------------------------------

restartL3AgentWarnStrDisableVrf = "Please restart layer 3 forwarding agent to " \
      "ensure that the disable-vrf option change takes effect"

def setDisableVrf( mode, disableVrfSet ):
   # If previously disabled VRFs disagree with the current configuration,
   # then disableVrfConfigChanged = True
   disableVrfConfigChanged = False

   # disableVrfSet should always be non-empty
   assert disableVrfSet
   if set( disableVrfSet ) != set( routingHardwareConfig.optimizeDisabledVrf ):
      disableVrfConfigChanged = True
      routingHardwareConfig.optimizeDisabledVrf.clear()
      for vrf in set( disableVrfSet ):
         routingHardwareConfig.optimizeDisabledVrf[ vrf ] = True

   if disableVrfConfigChanged:
      mode.addWarning( restartL3AgentWarnStrDisableVrf )

def unsetDisableVrf( mode ):
   if routingHardwareConfig.optimizeDisabledVrf:
      routingHardwareConfig.optimizeDisabledVrf.clear()
      mode.addWarning( restartL3AgentWarnStrDisableVrf )

class SetDisableVrfCmd( CliCommand.CliCommandClass ):
   syntax = "ip hardware fib optimize [ ( disable-vrf { VRF } ) ]"

   noOrDefaultSyntax = "ip hardware fib optimize [ disable-vrf ] ..."

   data = { "ip": ipMatcherForConfig,
            "hardware": hardwareHelpDesc,
            "fib": fibHelpDesc,
            "optimize": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "optimize",
                        helpdesc="Optimize IPv4 routes" ),
                  guard=routesInExactMatchGuard ),
            "disable-vrf": CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                    "disable-vrf",
                                    helpdesc="Disable fib optimizations in Vrf" ),
                                    guard=routesInExactMatchGuard ),
            "VRF": CliMatcher.DynamicNameMatcher(
                           lambda mode: allVrfConfig.vrf, "VRF name" )
          }

   @staticmethod
   def handler( mode, args ):
      if "VRF" in args:
         setDisableVrf( mode, args[ "VRF" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if "disable-vrf" not in args:
         unsetRoutesInExactMatch( mode )
      unsetDisableVrf( mode )

configMode.addCommandClass( SetDisableVrfCmd )

#-------------------------------------------------------------------------------
# The "show ip route rpf unicast [ vrf <vrf-name> ] [ prefix ]" command.
#-------------------------------------------------------------------------------

def populateUrpfModel( mode, intfs=None ):
   uRpfModel = UrpfInterfaces()
   if intfs is None:
      uRpfModel.interfaces = None
      return uRpfModel
   for intf in intfs:
      if not intf.lookup():
         raise UnknownEntityError( 'Interface %s does not exist' % intf.name )
      intf = IraIpIntfCli.IpIntf( intf, mode.sysdbRoot, createIfMissing=False )
      if intf.config().urpf.mode != 'disable':
         uRpfIntf = UrpfInterface()
         uRpfIntf.uRpfMode = intf.config().urpf.mode
         # Only strictDefault has allowDefault enabled
         uRpfIntf.allowDefault = uRpfIntf.uRpfMode == 'strictDefault'
         uRpfModel.interfaces[ intf.name ] = uRpfIntf
   return uRpfModel

def populateUrpfPrefixModel( mode, prefix, vrfName ):
   uRpfModel = UrpfInterfaces()
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ( vrfName is None or vrfExists( vrfName ) ):
      mode.addError( noRouteTableForVrfMsg % vrfName )
      uRpfModel.interfaces = None
      return uRpfModel

   prefix = Arnet.Prefix( prefix )
   routeEntry = routing.route( prefix, vrfName=vrfName )
   if routeEntry is None:
      uRpfModel.interfaces = None
      return uRpfModel

   uRpfModel.prefix = prefix.stringValue

   uRpfModel.viaDefaultRoute = \
      routeEntry.prefix == routing.ip.defaultRoute

   fec = routing.fStatus( vrfName ).fec.get( routeEntry.fecId )
   for via in fec.via.values():
      uRpfIntf = UrpfInterface()
      intf = via.intfId
      uRpfIntf.uRpfMode = ipConfig.ipIntfConfig[ intf ].urpf.mode
      uRpfIntf.allowDefault = uRpfIntf.uRpfMode == 'strictDefault'
      uRpfModel.interfaces[ intf ] = uRpfIntf
   return uRpfModel

def showUrpf( mode, vrfName=None, prefix=None ):

   # Get all routeable intefaces
   intfs = IntfCli.Intf.getAll( mode )
   intfs = filterIntfByRouteable( mode, intfs )

   # Filter interfaces by VRF
   if vrfName:
      intfs = filterIntfByVrf( mode, intfs, vrfName )

   if not prefix:
      return populateUrpfModel( mode, intfs )
   else:
      return populateUrpfPrefixModel( mode, prefix, vrfName )

class ShowUrpfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip route rpf unicast [ VRF ] [ PREFIX | ADDR ]'
   data = { 'ip': ipMatcherForShow,
            'route': routeMatcherAfterShowIp,
            'rpf': 'Reverse Path Forwarding information',
            'unicast': 'Unicast Reverse Path Forwarding information',
            'VRF': VrfExprFactory( helpdesc='Show Unicast Reverse Path '
                                            'Forwarding information in a VRF',
                                   inclDefaultVrf=True ),
            'PREFIX': IpAddrMatcher.ipPrefixExpr(
                         'Destination address',
                         'Destination subnet mask', 'Destination prefix',
                         partial=False,
                         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
            'ADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='Destination address' )
          }

   cliModel = UrpfInterfaces

   @staticmethod
   def adapter( mode, args, argsList ):
      ipPrefix = args.pop( 'PREFIX', None )
      if ipPrefix is not None:
         args[ 'PREFIX' ] = ipPrefix
      elif 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   @staticmethod
   def handler( mode, args ):
      return showUrpf( mode, vrfName=args.get( 'VRF' ), prefix=args.get( 'PREFIX' ) )

BasicCli.addShowCommandClass( ShowUrpfCmd )

#------------------------------------------------------------------------------------
#The "ip icmp redirect" and
#The "no|default ip icmp redirect"
#      commands.
#------------------------------------------------------------------------------------

def setIpRedirect( mode, no ):
   vrfNames = getAllVrfNames( mode )
   for vrf in vrfNames:
      r = routing.config( vrf )
      r.sendRedirects = not no

def noIpRedirect( mode ):
   setIpRedirect( mode, no=True )

def ipRedirect( mode):
   setIpRedirect( mode, no=False )

class IpIcmpRedirectCmd( CliCommand.CliCommandClass ):
   syntax = "ip icmp redirect"

   noSyntax = "ip icmp redirect"

   defaultSyntax = "ip icmp redirect"

   data = { "ip": ipMatcherForConfig,
            "icmp": icmpMatcherForConfig,
            "redirect": "Send ICMP redirects"
          }

   @staticmethod
   def handler( mode, args ):
      ipRedirect( mode )

   @staticmethod
   def noHandler( mode, args ):
      noIpRedirect( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      ipRedirect( mode )

configMode.addCommandClass( IpIcmpRedirectCmd )

#-------------------------------------------------------------------------------
# The "ip icmp source-interface" and  "no|default ip icmp source-interface"
# commands.
#-------------------------------------------------------------------------------
def setIcmpSrcIntf( mode, intf, vrfName=DEFAULT_VRF ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   r = routing.config( vrfName )
   if r:
      if intf:
         if intf.config():
            r.icmpSrcIntf = intf.config().intfId
         else:
            mode.addError( "Cannot find interface %s" % intf )
      else:
         r.icmpSrcIntf = ""

def noIcmpSrcIntf( mode, vrfName=DEFAULT_VRF ):
   setIcmpSrcIntf( mode, None, vrfName )

sourceInterfaceMatcherForConfig = CliMatcher.KeywordMatcher( 'source-interface',
      helpdesc="Which source interface's address to use" )

class SetIcmpSrcIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip icmp source-interface SRC_INTF [ VRF ]'
   noOrDefaultSyntax = 'ip icmp source-interface [ VRF ]'
   data = { 'ip': ipMatcherForConfig,
            'icmp': icmpMatcherForConfig,
            'source-interface': sourceInterfaceMatcherForConfig,
            'SRC_INTF': IntfCli.Intf.matcherWithIpSupport,
            'VRF': VrfExprFactory( helpdesc='Configure ICMP source '
                                            'interface in a VRF' ),
          }

   @staticmethod
   def handler( mode, args ):
      setIcmpSrcIntf( mode, args[ 'SRC_INTF' ],
                      vrfName=args.get( 'VRF', DEFAULT_VRF ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noIcmpSrcIntf( mode, vrfName=args.get( 'VRF', DEFAULT_VRF ) )

configMode.addCommandClass( SetIcmpSrcIntfCmd )

myEntManager = None

def fibRoutingStatus( self, vrfName=DEFAULT_VRF ):
   if vrfName == DEFAULT_VRF:
      return SmashLazyMount.mount( myEntManager,
                                   'routing/status',
                                   'Smash::Fib::RouteStatus',
                                   SmashLazyMount.mountInfo( 'reader' ) )
   else:
      return SmashLazyMount.mount( myEntManager,
                                   'routing/vrf/status/%s' % vrfName,
                                   'Smash::Fib::RouteStatus',
                                   SmashLazyMount.mountInfo( 'reader' ) )

class VerifyRouteCmd( CliCommand.CliCommandClass ):
   syntax = 'verify ip route ( all | ( PREFIX [ VRF ] ) )'
   data = { 'verify': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "verify",
                     helpdesc="Verify the state of an object in the system" ),
                  hidden=True ),
            'ip': 'Verify the state of an IP object',
            'route': 'Verify the state of a route',
            'all': 'Verify the state of all the routes in the system',
            'PREFIX':  IpAddrMatcher.ipPrefixExpr(
                          "Match this IP address",
                          "Match this subnet mask",
                          "Match this IP prefix",
                          partial=False ),
            'VRF': VrfExprFactory( helpdesc="Verify in a VRF",
                                   guard=vrfRoutingSupportedGuard ),
          }

   @staticmethod
   def handler( mode, args ):
      actors = [ 'KernelFib' ]
      routes = []
      if 'all' in args:
         for vrf in [ DEFAULT_VRF ] + allVrfConfig.vrf.keys():
            for route in fibRoutingStatus( vrf ).route:
               routes.append( ( route.stringValue, vrf ) )
      else:
         pfx = args[ 'PREFIX' ]
         vrf = args.get( 'VRF', DEFAULT_VRF )
         routes.append( ( pfx, vrf ) )
      # If total number of routes exceeds VERIFY_MAX_ROUTES limit, skip the
      # verification
      if len( routes ) > VERIFY_MAX_ROUTES:
         print "Skipping verification as total number of routes exceeds %d" \
               % VERIFY_MAX_ROUTES
      else:
         verify( 'Route', routes, actors, myEntManager )

BasicCli.EnableMode.addCommandClass( VerifyRouteCmd )

#-------------------------------------------------------------------------------
# The "ip fib compression redundant-specifics filter" and
# "[no|default] ip fib compression redundant-specifics filter"
# commands.
#-------------------------------------------------------------------------------
def fibSuppressionSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.fibSuppressionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def setFibFilter( mode, enable=True ):
   routingHardwareConfig.fibSuppression = enable

def noSetFibFilter( mode ):
   setFibFilter( mode, enable=False )

class FibCompressionCmd( CliCommand.CliCommandClass ):
   syntax = "ip fib compression redundant-specifics filter"

   noOrDefaultSyntax = "ip fib compression redundant-specifics filter"

   data = { "ip": ipMatcherForConfig,
            "fib": fibHelpDesc,
            "compression": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( "compression",
                              helpdesc="FIB compression" ),
                  guard=fibSuppressionSupportedGuard ),
            "redundant-specifics": "type of route filter to use",
            "filter": "filter command"
          }

   @staticmethod
   def handler( mode, args ):
      setFibFilter( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noSetFibFilter( mode )

configMode.addCommandClass( FibCompressionCmd )

#-------------------------------------------------------------------------------
# The "show ip user route" command
#-------------------------------------------------------------------------------
class ShowIpUserRouteCmd( ShowCommand.ShowCliCommandClass ):
   '''Display routes from the RIB that have the 'dynamic' attribute set to True
      These routes have been programmed via the EOS SDK with the 'persistent' flag
      set to False.'''
   syntax = 'show ip user route [ vrf VRF ] [ tag TAGNUM ]'
   data = {
         'ip': ipMatcherForShow,
         'user': 'Display non-persistent routes',
         'route': 'Static routes',
         'vrf': vrfForShowMatcher,
         'VRF': vrfNameMatcher,
         'tag': tagMatcherForConfig,
         'TAGNUM': tagNumberMatcher
         }
   cliModel = VrfIpRibRoutes

   @staticmethod
   def handler( mode, args ):
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, args.get( 'VRF' ) )
      if not vrfExists( vrfName ):
         mode.addError( noRouteTableForVrfMsg % vrfName )
         return None
      fmt = mode.session_.outputFormat()
      tag = args.get( 'TAGNUM', 0 )
      routing.showUserRoutes( vrfName=vrfName, tag=tag,
                              fmt=fmt, mlib=printer )
      return VrfIpRibRoutes

BasicCli.addShowCommandClass( ShowIpUserRouteCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )

   global routingHardwareConfig
   global routingHardwareStatus
   global routingHardwareStatus6
   global routingHardwareRouteStatus
   global ipStatus
   global allVrfConfig
   global ipConfig
   global allVrfStatusLocal
   global l3Config
   global myEntManager

   myEntManager = entityManager

   allVrfConfig = ConfigMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "w" )
   routingHardwareConfig = ConfigMount.mount( entityManager,
                                              "routing/hardware/config",
                                              "Routing::Hardware::Config", "w" )

   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routingHardwareStatus6 = LazyMount.mount( entityManager,
                                            "routing6/hardware/status",
                                            "Routing6::Hardware::Status", "r" )
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/route/status",
                                            "Routing::Hardware::RouteStatus", "r" )

   ConfigMount.mount( entityManager, "routing/vrf/config",
                      "Routing::VrfConfigDir", "wi" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", "r" )
   IraIpIntfCli.canSetIntfIpHook.addExtension( canSetIntfIp )
