# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliMatcher
import Ethernet
import Ip6AddrMatcher
import IraCommonCli
import IraRouteCommon
from IpLibConsts import DEFAULT_VRF
from IraNexthopGroupCli import nexthopGroupSupportedGuard
from RoutingConsts import (
      defaultStaticRouteName,
      defaultStaticRoutePreference,
      defaultStaticRouteTag,
      defaultStaticRouteMetric
)
import Tac

routeMatcherForIpv6Config = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Manage static IPv6 routes' )
ipv6PrefixMatcher = Ip6AddrMatcher.Ip6PrefixMatcher( "Destination prefix" )

ip6 = IraRouteCommon.Ip6()
routing6 = IraRouteCommon.routing( ip6 )

noIpv6RouteTableForVrfMsg = "IPv6 Routing table for VRF %s does not exist."

def staticIpv6RoutingTable( vrfName=DEFAULT_VRF ):
   return routing6.routeConfig( vrfName )

def isValidIpv6PrefixWithError( mode, prefix ):
   """Returns True if a prefix is valid.  If invalid, it addas a CLI error
   and returns False."""

   if not prefix.validAsPrefix:
      mode.addError( "Host part of destination prefix must be zero" )
      return False

   return True

def manageIpv6Route( mode, prefix, nexthop, routeOptions=None, vrfName=DEFAULT_VRF,
                     leakToVrf=None, egressVrf=None ):
   if not isValidIpv6PrefixWithError( mode, prefix ):
      return

   # TODO We're unnecessarily matching an Ip6AddrWithMask with the CLI
   # matcher, then manually converting it to an Ip6Prefix here. This
   # is wasteful, and is presumably done so that we can allow invalid
   # prefixes to be transparently converted (masking off the host
   # bits). When BUG35676 is fixed, this can be refactored
   prefix = Arnet.Ip6Prefix( prefix.stringValue )

   if prefix.address.isMulticast:
      mode.addError( "Destination prefix must be unicast" )
      return

   if not mode.session_.startupConfig() and 'nexthopGroupName' in nexthop and \
         nexthopGroupSupportedGuard( mode, None ) is not None:
      mode.addError( "Nexthop-Group not supported on this hardware platform" )
      return

   rt = staticIpv6RoutingTable( vrfName )

   routeOptionsDict = dict( routeOptions if routeOptions else [] )
   preference = routeOptionsDict.get( 'preference' )
   tag = routeOptionsDict.get( 'tag' )
   nexthopName = routeOptionsDict.get( 'nextHopName' )
   trackingProto = routeOptionsDict.get( 'trackingProto' )
   metric = routeOptionsDict.get( 'metric' )

   if preference is None:
      preference = defaultStaticRoutePreference
   if tag is None:
      tag = defaultStaticRouteTag
   if nexthopName is None:
      nexthopName = defaultStaticRouteName
   if metric is None:
      metric = defaultStaticRouteMetric

   label = None
   vtepAddr = None
   vni = None
   routerMac = None
   vxlanSrcIntf = None

   # nexthop is actually either a rule with mpls, evpn, nexthop or an intf,
   # and it is a dict, produced by a Cli OrRule.
   routeType = 'forward'
   if 'mpls' in nexthop:
      hop = nexthop[ 'mpls' ][ 'nexthop' ]
      label = nexthop[ 'mpls' ][ 'label' ]
      if not IraCommonCli.validNextHop6( mode, hop, vrfName ):
         return
      intf = None
   elif 'evpn' in nexthop:
      hop = Arnet.Ip6Addr( '::' )
      vtepAddr = nexthop[ 'evpn' ][ 'vtepAddr' ]
      vni = nexthop[ 'evpn' ][ 'vni' ]
      routerMac = nexthop[ 'evpn' ][ 'routerMac' ]
      vxlanSrcIntf = nexthop[ 'evpn' ][ 'vxlanSrcIntf' ]
      dynamic = nexthop[ 'evpn' ][ 'dynamic' ]
      ribBypass = nexthop[ 'evpn' ][ 'ribBypass' ]
      intf = None
   elif 'nexthop' in nexthop:
      hop = nexthop[ 'nexthop' ]
      if not IraCommonCli.validNextHop6( mode, hop, vrfName ):
         return
      intf = None
   elif 'Null0' in nexthop or 'Null 0' in nexthop:
      routeType = 'drop'
   elif 'nexthopGroupName' in nexthop:
      routeType = 'nexthopGroup'
      nexthopGroupName = nexthop[ 'nexthopGroupName' ]
   else:
      nhi = nexthop[ 'intf' ][ 'intf' ]
      hop = ip6.intfAddrRoutesSupported and nexthop[ 'intf' ][ 'intfnexthop' ]
      if hop:
         if not IraCommonCli.validNextHop6( mode, hop, vrfName, nhi ):
            return
      else:
         hop = Arnet.Ip6Addr( '::' )

      if not nhi.config():
         mode.addError( "Interface %s does not exist" % nhi.name )
         return

      if not nhi.routingSupported():
         mode.addError( "Interface %s is not routable" % nhi.name )
         return

      intf = ip6.config.intf.get( nhi.name )
      if not intf:
         intf = ip6.config.newIntf( nhi.name )
         intf.l3Config = ip6.l3ConfigDir.newIntfConfig( nhi.name )

   key = Tac.Value( "Routing::RouteKey",
                    prefix=Arnet.IpGenPrefix( prefix.stringValue ),
                    preference=preference )

   r = rt.route.get( key )
   if r:
      if ( ( r.routeType != 'nexthopGroup' and routeType == 'nexthopGroup' ) or
           ( routeType != 'nexthopGroup' and r.routeType == 'nexthopGroup' ) ):
         mode.addError( "Route %s already exists, cannot ECMP with nexthop group "
                        "and non-nexthop group routes " % prefix )
         return
      if( ( routeType == 'drop' or r.routeType == 'drop' ) and
          routeType != r.routeType ):
         mode.addError( "Cannot ECMP to Null0 interface." )
         return
   else:
      r = rt.route.newMember( key, routeType )
   r.tag = tag

   if leakToVrf:
      r.leakToVrf = leakToVrf
   else:
      r.leakToVrf = ''

   if routeType == 'forward':
      hop = Arnet.IpGenAddr( str( hop ) )
      if intf:
         via = Tac.Value( "Routing::Via", hop=hop, intfId=intf.intfId )
      else:
         via = Tac.Value( "Routing::Via", hop=hop, intfId='' )
         # If egress VRF is the same as the VRF route is pointing to, there is no
         # meaning setting egress VRF in via
         if egressVrf is not None and egressVrf != vrfName:
            via.egressVrf = egressVrf
      if label != None:
         via.mplsLabel = label

      if vtepAddr is not None:
         if vni is None or routerMac is None:
            mode.addError( "Static EVPN routes must specify VNI and "
                           "router-mac-address " )
            return

         vtepAddr = Arnet.IpGenAddr( vtepAddr )
         vni = int( vni )
         routerMac = Ethernet.convertMacAddrToCanonical( routerMac )
         via.vtepAddr = vtepAddr
         via.vni = vni
         via.routerMac = routerMac
         via.vxlanIntf = vxlanSrcIntf

         r.dynamic = dynamic
         r.ribBypass = ribBypass

      maxEcmp = None
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
      if rhs is not None and rhs.maxEcmp > 0:
         maxEcmp = rhs.maxEcmp

      if maxEcmp and len( r.via ) == maxEcmp and via not in r.via:
         mode.addError( "A maximum of %d routes are allowed "
                        "with the same prefix and preference" % maxEcmp )
         return

      r.via[ via ] = metric
      r.viaNextHopName[ via ] = nexthopName
      if trackingProto == 'bfd':
         r.bfdTracked[ via ] = True
      else:
         del r.bfdTracked[ via ]
   elif routeType == 'drop':
      r.dropNextHopName = nexthopName
   else:
      r.nexthopGroupName = nexthopGroupName
      r.nexthopGroupMetric = metric

def noIpv6Route( mode, prefix, nexthop, preference, vrfName, egressVrf=None ):
   if not isValidIpv6PrefixWithError( mode, prefix ):
      return

   # If egress VRF is the same as the VRF route is pointing to, there is no
   # meaning setting egress VRF in via
   if egressVrf is None or egressVrf == vrfName:
      egressVrf = ""

   prefix = Arnet.IpGenPrefix( prefix.stringValue )

   rt = staticIpv6RoutingTable( vrfName )

   # optimization
   if preference:
      key = Tac.Value( "Routing::RouteKey", prefix=prefix, preference=preference )
      if key in rt.route:
         keys = [ key ]
      else:
         keys = []
   else:
      keys = rt.route.keys()

   for key in keys:
      if key.prefix != prefix:
         continue

      if ( preference and preference != key.preference ):
         continue

      if not nexthop:
         # no nexthop or intf specified. Delete every route with this prefix.
         del rt.route[ key ]
         continue

      if 'Null0' in nexthop or 'Null 0' in nexthop:
         if rt.route[ key ].routeType == 'drop':
            del rt.route[ key ]
         continue

      def delRoutes( matching ):
         r = rt.route.get( key )
         rvias = r.via.keys()
         for v in rvias:
            if matching( v ):
               if len( rvias ) == 1:
                  del rt.route[ key ]
               else:
                  del r.via[ v ]
                  del r.viaNextHopName[ v ]
                  del r.bfdTracked[ v ]

      nullLabel = Tac.Type( 'Arnet::MplsLabel' ).null
      if 'intf' in nexthop:
         intf = nexthop[ 'intf' ][ 'intf' ]
         hop = ip6.intfAddrRoutesSupported and nexthop[ 'intf' ][ 'intfnexthop' ]
         if not hop:
            hop = Arnet.IpGenAddr( '::' )
         else:
            hop = Arnet.IpGenAddr( str( hop ) )
         delRoutes( lambda via: via.intfId == intf.name and via.hop == hop
                    and via.mplsLabel == nullLabel )
      elif 'nexthopGroupName' in nexthop:
         r = rt.route.get( key )
         if ( r.routeType == 'nexthopGroup' and
              r.nexthopGroupName == nexthop[ 'nexthopGroupName' ] ):
            del rt.route[ key ]
      elif 'mpls' in nexthop:
         hop = Arnet.IpGenAddr( str( nexthop[ 'mpls' ][ 'nexthop' ] ) )
         label = nexthop[ 'mpls' ][ 'label' ]
         delRoutes( lambda via: via.hop == hop and via.mplsLabel == label
                    and via.intfId == '' )
      elif 'evpn' in nexthop:
         vtepAddr = nexthop[ 'evpn' ][ 'vtepAddr' ]
         vtepAddr = Tac.Value( 'Arnet::IpGenAddr', vtepAddr )
         vni = nexthop[ 'evpn' ][ 'vni' ]
         vni = int( vni )
         routerMac = nexthop[ 'evpn' ][ 'routerMac' ]
         routerMac = Ethernet.convertMacAddrToCanonical( routerMac )
         delRoutes( lambda via: via.intfId == '' and
                    via.vni == vni and via.vtepAddr == vtepAddr and
                    via.routerMac == routerMac )
      else:
         hop = Arnet.IpGenAddr( str( nexthop[ 'nexthop' ] ) )
         delRoutes( lambda via: via.hop == hop and via.intfId == ''
                    and via.mplsLabel == nullLabel and via.egressVrf == egressVrf )
