# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import operator

import Arnet
from ArnetModel import (
      Ip6Address,
      IpGenericAddress,
      IpGenericPrefix,
)
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import Float
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliModel import DeferredModel
from IntfModels import Interface
from TableOutput import createTable, Format
from IpLibConsts import DEFAULT_VRF

from CliPlugin import IntfModel # pylint: disable-msg=F0401
from IraIpModel import IpStatusBase, IpRouteBase, IpRouteVia
import Tac
import IraRouteCommon
import IpLibTypes

from IraCommonModel import ServiceRoutingProtoModelStatus

# Common strings used for output rendering
DEFAULT_ADDR_AND_MASK_STR = "unassigned"
DEFAULT_ADDR_STATE_STR = ""

ADDR_STATE_INACTIVE = "inactive"
ADDR_STATE_DUP = "duplicate"
ADDR_STATE_ACTIVE = "up"

addrSource = IpLibTypes.addrSource( Tac.Type( 'Arnet::AddressFamily' ).ipv6 )

class IpAddress( Model ):
   address = Ip6Address( help="An IPv6 address (e.g. 'fc00::1234')" )
   subnet = Str( help="The subnet of this address (e.g. 'fc00::/64')" )
   active = Bool( help="Whether or not this IPv6 address is currently active" )
   dadfailed = Bool( help="Whether or not this IPv6 address is a duplicate" )

   def formatAddrAndMaskStr ( self ):
      if not self.address:
         return DEFAULT_ADDR_AND_MASK_STR
      netmask = int( self.subnet.split( "/" )[ 1 ] )
      return "%s/%d" % ( self.address, netmask)

   def formatStateStr( self ):
      # FIXME implement a more correct address status here?
      if self.dadfailed:
         return " [DUP]"
      else:
         return DEFAULT_ADDR_STATE_STR if self.active else " [INACTIVE]"

   def addrStateStr( self ):
      if self.dadfailed:
         return ADDR_STATE_DUP
      else:
         return ADDR_STATE_ACTIVE if self.active else ADDR_STATE_INACTIVE

class InterfaceAddressIp6( IntfModel.InterfaceAddressIp6Base ):
   linkLocalIp6 = Submodel( valueType=IpAddress,
                            help="Link-local IPv6 address",
                            optional=True )
   globalUnicastIp6s = List( valueType=IpAddress,
                             help="Global-unicast IPv6 addresses" )
   globalAddressesAreVirtual = Bool( help="Whether configured addresses are "
                                     "physical or virtual" )
   addrSource = Enum( values=( addrSource.manual, addrSource.slaac ),
                      help="How the address is configured on the interface",
                      optional=True )


   def render( self ):
      if self.linkLocalIp6:
         addrStr = self.linkLocalIp6.formatAddrAndMaskStr()
         stateStr = self.linkLocalIp6.formatStateStr()
      else:
         addrStr = DEFAULT_ADDR_AND_MASK_STR
         stateStr = DEFAULT_ADDR_STATE_STR

      print "  IPv6 link-local address is %s%s" % ( addrStr, stateStr)

      if self.addrSource == addrSource.slaac:
         if self.globalUnicastIp6s:
            print "  Address determined by SLAAC"
         else:
            print "  Address being determined by SLAAC"

      virtualStr = " (virtual)" if self.globalAddressesAreVirtual else ""
      if self.globalUnicastIp6s:
         print "  IPv6 global unicast address(es):"
         for ip6 in self.globalUnicastIp6s:
            print "    %s%s, subnet is %s%s" % \
                  ( ip6.address, virtualStr, ip6.subnet, ip6.formatStateStr() )
      else:
         print "  No IPv6 global unicast address is assigned"
         
class Ip6Status( IpStatusBase ):
   class NeighborDiscoverDetails( Model ):
      ndAdvertisedReachableTime = Int( help="Neighbor discovery reachable time"
                                       " being advertised (in milliseconds)" )
      ndRouterAdvSuppressed = Bool( help="Whether or not Neighbor Discovery"
                                    " Router Advertisement messages are enabled" )
      ndRouterAdvRetransmitInterval = Int( help="Neighbor discovery retransmit"
                                           " interval for Router Advertisement"
                                           " (in millisecond)" )
      ndRouterAdvInterval = Int( help="Interval in milliseconds between neighbor"
                                       " discovery route advertisements" )
      routerAdvLifetime = Int( help="Router lifetime in Router Advertisement "
                                    "messages" )
      routerPreference = Enum( values=( "High", "Low", "Medium" ),
                               help="Router preference to be used in Router "
                                   "Advertisements" )
      routerAdvHopLimit = Int( help="Current hop limit to use in Router "
                                    "Advertisements" )
      
   linkLocal = Submodel( valueType=IpAddress,
                         help="Link-local IPv6 address of this interface",
                         optional=True )
   state = Enum( values=( "disabled", "stalled", "enabled" ),
                 help="Whether or not the interface is enabled, or stalled in"
                 " case of a DAD failure on the link-local IP." )
   addresses = List( valueType=IpAddress,
                     help="Addresses configured on this interface" )
   globalAddressesAreVirtual = Bool( help="Whether configured addresses are "
                                     "physical or virtual" )
   multicastGroupAddresses = List( valueType=str,
                                   help="Addresses of multicast groups joined" )
   dadStatus = Enum( values=( "unsupported", "enabled", "unavailable", "disabled",
                              "strict" ),
                     help="Current status of Duplicate Address Detection",
                     optional=True )
   dadAttempts = Int( help="Number of DAD attempts", default=1 )
   ndReachableTime = Int( help="Neighbor discovery reachable time"
                               " (in milliseconds)", default=30000 )
   ndRetransmitInterval = Int( help="Neighbor discovery retransmit interval"
                                    " (in milliseconds)", default=1000 )
   enhancedDad = Bool(  help="RFC7527 enhanced duplicate address detection",
                        optional=True )
   autoConfigStatus = Enum( values=( "dhcp", "stateful", "stateless" ), 
                            help="IPv6 address assignment option", 
                            default="stateless"  )
   neighborDiscoverDetails = Submodel( valueType=NeighborDiscoverDetails, 
                                       optional=True, help="Neighbor discovery Rds" )
   urpf = Enum( values=( "disable", "loose", "strict", "strictDefault" ),
                help="Unicast Reverse Path Forwarding mode on the interface",
                optional=True )
   urpfV4V6Mismatch = Bool( help= \
                            "Whether V4 and V6 urpf conflict on the same interface",
                            optional=True )
   vrf = Str( help="Name of the VRF", optional=True )
   addrSource = Enum( values=( addrSource.manual, addrSource.slaac ),
                      help="How the address is configured on the interface",
                      optional=True )
   maxMssIngress = Int( help="TCP MSS ceiling in bytes enforced on packets"
                             " arriving from the network", optional=True )
   maxMssEgress = Int( help="TCP MSS ceiling in bytes enforced on packets"
                            " forwarded to the network", optional=True )

   def render( self ):
      super( Ip6Status, self ).render()

      if self.linkLocal:
         addrStr = self.linkLocal.formatAddrAndMaskStr()
         addrStateStr = self.linkLocal.formatStateStr()
      else:
         addrStr = DEFAULT_ADDR_AND_MASK_STR
         addrStateStr = DEFAULT_ADDR_STATE_STR

      print "  IPv6 is %s, link-local is %s%s" % \
            ( self.state, addrStr, addrStateStr )

      if self.addrSource == addrSource.slaac:
         if self.addresses:
            print "  Address determined by SLAAC"
         else:
            print "  Address being determined by SLAAC"

      virtualStr = " (virtual)" if self.globalAddressesAreVirtual else ""
      if self.addresses:
         print "  Global unicast address(es):"
         for addr in self.addresses:
            print "    %s%s, subnet is %s%s" % \
                  ( addr.address, virtualStr, addr.subnet, addr.formatStateStr() )
      else:
         print "  No global unicast address is assigned"

      if self.multicastGroupAddresses:
         print "  Joined group address(es):"
         print "\n".join( "    %s" % addr for addr in self.multicastGroupAddresses )
      else:
         print "  No joined group addresses"

      if self.urpfV4V6Mismatch:
         print "  IPv6 Unicast RPF disabled, conflicts with IPv4 Unicast RPF"
      elif self.urpf == 'strict':
         print "  IPv6 verify unicast source reachable-via RX"
      elif self.urpf == 'loose':
         print "  IPv6 verify unicast source reachable-via ANY"
      elif self.urpf == 'strictDefault':
         print "  IPv6 verify unicast source reachable-via RX, allow-default"

      if self.dadStatus == "unsupported":
         print "  ND DAD is not supported"
      elif self.dadStatus == "enabled":
         print "  ND DAD is enabled, number of DAD attempts: %(dadAttempts)d" % self
      elif self.dadStatus == "strict":
         print '  ND DAD is enabled and strict, number of DAD attempts: ' \
            '%(dadAttempts)d' % self
      elif self.dadStatus == "disabled":
         print "  ND DAD is disabled"
      else:
         print "  ND DAD status is unavailable"

      # dadstatus can be None while we're building and Ipv6NeighborDiscovery
      # hasn't been installed yet.  Then its hook into Ira will not kick in
      # and won't populate all the ND-related fields.
      if self.dadStatus is not None:
         print ( "  ND Reachable time is %(ndReachableTime)d milliseconds\n"
                 "  ND retransmit interval is %(ndRetransmitInterval)d milliseconds"
                 % self )
         print ( "  ND enhanced duplicate address detection %s" %
                 ( "enabled" if self.enhancedDad else "disabled" ) )

         if self.neighborDiscoverDetails:
            self._renderRouterAdv()

            msg = "  Hosts use %s addresses."
            if self.autoConfigStatus == "dhcp":
               if self.name.stringValue.startswith( "Loopback" ):
                  msg %= "DHCP to obtain routable"
               else:
                  msg %= "stateful autoconfig for"
            else:
               msg %= "stateless autoconfig for"
            print msg

            if self.neighborDiscoverDetails.ndRouterAdvSuppressed:
               print "  ND RAs are suppressed"

      if self.maxMssIngress and self.maxMssEgress:
         print "  IPv6 TCP MSS ceiling is %d bytes" % self.maxMssIngress
      elif self.maxMssIngress:
         print "  IPv6 TCP MSS ingress ceiling is %d bytes" % self.maxMssIngress
      elif self.maxMssEgress:
         print "  IPv6 TCP MSS egress ceiling is %d bytes" % self.maxMssEgress

      assert self.vrf and self.vrf != ''
      if self.vrf != DEFAULT_VRF:
         print '  VPN Routing/Forwarding "%(vrf)s"' % self

   def _renderRouterAdv( self ):
      ndDetails = self.neighborDiscoverDetails
      print ( "  ND advertised reachable time is %d milliseconds (using %d)" %
              ( ndDetails.ndAdvertisedReachableTime, 
                ndDetails.ndAdvertisedReachableTime ) )
      print ( "  ND advertised retransmit interval is %d milliseconds%s" %
              ( ndDetails.ndRouterAdvRetransmitInterval,
                "" if ndDetails.ndRouterAdvRetransmitInterval 
                else " (unspecified)" ) )
      if ndDetails.ndRouterAdvInterval % 1000 == 0:
         raIntervalInfo = ndDetails.ndRouterAdvInterval / 1000
         raIntervalUnit = ""
      else:
         raIntervalInfo = ndDetails.ndRouterAdvInterval
         raIntervalUnit = "milli"

      print ( "  ND router advertisements are sent every %d %sseconds" %
              ( raIntervalInfo, raIntervalUnit ) )

      print ( "  ND router advertisements live for %(routerAdvLifetime)d seconds\n"
              "  ND advertised default router preference is %(routerPreference)s\n"
              "  ND advertised maximum hop count limit is %(routerAdvHopLimit)s"
              % ndDetails )

class Ip6Statuses( Model ):
   interfaces = Dict( keyType=Interface, valueType=Ip6Status,
                      help="Maps an interface name to its IPv6 status" )

   def render( self ):
      for key in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ key ].render()

class Ip6StatusBrief( IpStatusBase ):
   linkLocal = Ip6Status.linkLocal
   addresses = Ip6Status.addresses
   addrSource = Ip6Status.addrSource

   def addRow( self, showTable ):
      # Handle the link-local address
      llStr = "<linklocal-unassigned>"
      llStateStr = ADDR_STATE_INACTIVE
      if self.linkLocal and self.linkLocal.address:
         llStr = self.linkLocal.formatAddrAndMaskStr()
         llStateStr = self.linkLocal.addrStateStr()
      showTable.newRow( self.name.shortName, IntfModel.formatShortIntfStatus( self ),
                        self.mtu, llStr, llStateStr, "link local" )

      # Handle global addresses
      sourceStr = "slaac" if self.addrSource == addrSource.slaac else "config"
      for addr in sorted( self.addresses, key=operator.attrgetter( 'address' ) ):
         showTable.newRow( "", "", "",
                           addr.formatAddrAndMaskStr(), addr.addrStateStr(),
                           sourceStr )

class Ip6StatusesBrief( Model ):
   interfaces = Dict( keyType=Interface, valueType=Ip6StatusBrief,
                      help="Maps an interface name to its abbreviated IPv6 status" )

   def render( self ):
      if not self.interfaces:
         return

      fl = Format( justify="left" )
      headerNames = [ "Interface", "Status", "MTU", "IPv6 Address",
                      "Addr State", "Addr Source" ]

      showTable = createTable( ( h, 'l' ) for h in headerNames )
      showTable.formatColumns( fl, fl, None, fl, fl, fl )

      for key in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ key ].addRow( showTable )

      print showTable.output()

#----------------------------------------
# show ipv6 route Model
#---------------------------------------
class Ip6Route( IpRouteBase ):
   routeType = Enum( values=( 'connected', 'static', 'kernel', 'ospf', 'bgp',
      'rip', 'bgpAggregate', 'ospfSummary', 'isis', 'isisL1', 'isisL2',
      'nexthopGroupStaticRoute', 'dropRoute', 'dhcp', 'dynamicPolicy', 'martian' ),
      help="The ipv6 route type" )
   vias = List( valueType=IpRouteVia, help="List of vias" )

class Ip6Routes( DeferredModel ):
   allRoutesProgrammedHardware = Bool( help="True if every route is programmed "
                                       "in hardware" )
   allRoutesProgrammedKernel = Bool( help="True if every route is programmed "
                                     "in kernel" )
   routingDisabled = Bool( help="True if routing is disabled" )
   defaultRouteState = Enum( values=( 'reachable', 'notReachable', 'notSet' ),
                                      help="The state of the default route" )
   routes = Dict( keyType=str, valueType=Ip6Route, help="List of ipv6 routes "
                  "keyed by prefix" )

class VrfIp6Routes( DeferredModel ):
   __revision__ = 3
   vrfs = Dict( keyType=str, valueType=Ip6Routes,
                  help="Route info of each VRF" )

#----------------------------------------
#model for 'show ipv6 route host'
#----------------------------------------
class IpRouteHostVia( Model ):
   toCpu = Bool( help="True if route goes to cpu" )
   nexthopAddr = IpGenericAddress( help="Nexthop address of the via", optional=True )
   interface = Interface( help="Name of the interface", optional=True )

class Ip6RouteHost( Model ):
   routeType = Enum( values=( 'receive', 'broadcast', 'FIB', 'attached' ),
                     help="The ipv6 route host type" )
   hardwareProgrammed = Bool( help="True if this route is "
         "programmed in hardware" )
   vias = List( valueType=IpRouteHostVia, help="List of host vias" )

class Ip6RoutesHost( DeferredModel ):
   routingDisabled = Bool( help="Routing is disabled" )
   allRoutesProgrammedHardware = Bool( help="Every route is programmed "
                                            "in hardware" )
   routes = Dict( keyType=str, valueType=Ip6RouteHost, help="IPv6 host "
                  "routes keyed by prefix" )

class VrfIp6RoutesHost ( DeferredModel ):
   '''
   __revision__ 2 supports printing all vrfs host routes.
   __revision__ 1 does not support printing all vrfs host routes.
   '''
   __revision__ = 2
   vrfs = Dict( keyType=str, valueType=Ip6RoutesHost,
                 help="A mapping of VRF to its IPv6 host routes" )

#------------------------------------------
#models for 'show ipv6 route summary' and 
#    'show ip route vrf all summary brief'
#------------------------------------------
class ospfTotals6( Model ):
   ospfTotal = Int( help="Total number of ospf routes" ) 
   ospfIntraArea = Int( help="Number of ospf Intra-area routes" )
   ospfInterArea = Int( help="Number of ospf Inter-area routes" )
   ospfExternal1 = Int( help="Number of ospf External-1 routes" )
   ospfExternal2 = Int( help="Number of ospf External-2 routes" )
   nssaExternal1 = Int( help="Number of NSSA External-1 routes" )
   nssaExternal2 = Int( help="Number of NSSA External-2 routes" )

class bgpTotals6( Model ):
   bgpTotal = Int( help="Total number of bgp routes" )
   bgpExternal = Int( help="Number of bgp External routes" )
   bgpInternal = Int( help="Number of bgp Internal routes" )
   bgpLocal = Int( help="Number of bgp Local routes" )

class isisTotals6( Model ):
   isisTotal = Int( help="Total number of isis routes" )
   isisLevel1 = Int( help="Number of isis Level-1 routes" )
   isisLevel2 = Int( help="Number of isis Level-2 routes" )

class Ip6RouteSummaryForVrf( Model ):
   connected = Int( help="Number of connected routes" )
   static = Int( help="Number of static routes" ) 
   staticPersistent = Int( help="Number of static routes configured to persist" )
   staticNonPersistent = Int( help="Number of static routes configured using the "
                                   "SDK with persistent=False" )
   staticNexthopGroup = Int( help="Number of static nexthop-group routes" )
   dynamicPolicy = Int( help="Number of dynamic policy routes" )
   vrfCount = Int( help="Total number of vrf instances", optional=True )
   vcs = Int( optional=True, help="Number of vxlan-control-service routes" )
   dhcp = Int( optional=True, help="Number of dhcp routes" )
   ospfCounts6 = Submodel( valueType=ospfTotals6, help="OSPF route count breakdown" )
   bgpCounts6 = Submodel( valueType=bgpTotals6, help="BGP route count breakdown" )
   isisCounts6 = Submodel( valueType=isisTotals6, help="IS-IS route count "
                           "breakdown" )
   # no rip support for ipv6.  rip = Int( help="Number of rip routes" ) 
   internal = Int( help="Number of internal routes" ) 
   attached = Int( help="Number of attached routes" ) 
   aggregate = Int( help="Number of aggregate routes" ) 
   maskLen = Dict( keyType=int, valueType=int, help="Counts for each mask length" )
   totalRoutes = Int( help="Total number of routes" )
   _quantify = Float( optional=True, help="Time (in seconds) taken to populate "
                     "and print the summary" )

   def render( self ):
      beforeRender = Tac.now()
      table = createTable( ( 'Route Source', 'Number Of Routes' ) )
      f1 = Format( justify='left' )
      f2 = Format( justify='right' )
      table.formatColumns( f1, f2 )
      table.newRow( "connected", self.connected )
      table.newRow( "static (persistent)", self.staticPersistent )
      table.newRow( "static (non-persistent)", self.staticNonPersistent )
      table.newRow( "static nexthop-group", self.staticNexthopGroup )
      table.newRow( "ospf", self.ospfCounts6.ospfTotal )
      #TODO: when ipv6 summary gets updated, uncomment the below
      #table.startRow() # format ospf 
      #table.newFormattedCell( '  Intra-area: %d Inter-area: ' \
      #      % ( self.ospfCounts.ospfIntraArea,
      #      self.ospfCounts.ospfInterArea ), 2, f1 )
      #table.startRow()
      #table.newFormattedCell( '  External-1: %d External-2: %d' \
      #      % ( self.ospfCounts.ospfExternal1, \
      #      self.ospfCounts.ospfExternal2 ), 2, f1 )
      #table.startRow()
      #table.newFormattedCell( '  NSSA External-1: %d NSSA External-2: %d' \
      #      % ( self.ospfCounts.nssaExternal1, \
      #      self.ospfCounts.nssaExternal2 ), 2, f1 )
      table.newRow( "bgp", self.bgpCounts6.bgpTotal )     
      #table.startRow() # format bgp
      #table.newFormattedCell( '  External: %d Internal: %d' \
      #      % ( self.bgpCounts.bgpExternal, \
      #      self.bgpCounts.bgpInternal ), 2, f1 )
      table.newRow( "isis", self.isisCounts6.isisTotal )
      table.startRow() # format isis
      table.newFormattedCell( '  Level-1: %d Level-2: %d' \
            % ( self.isisCounts6.isisLevel1, \
            self.isisCounts6.isisLevel2 ), 2, f1 )
      table.newRow( "internal", self.internal )
      table.newRow( "attached", self.attached )
      table.newRow( "dhcp", self.dhcp )
      table.newRow( "aggregate", self.aggregate )
      table.newRow( "dynamic policy", self.dynamicPolicy )
      table.newRow( '', '' )
      table.newRow( 'Total Routes', self.totalRoutes )
      print table.output()
      IraRouteCommon.showRouteSummaryMaskLengths( sorted( self.maskLen.items() ) )   
      if self.vrfCount is not None:
         print "Number of VRFs:", self.vrfCount
      if self._quantify:
         # we need to add the time spent in render with the time spent in populating
         # the model
         afterRender = Tac.now()
         print( "Time taken for the summary to be printed: %f seconds" % 
               ( ( afterRender-beforeRender)+self._quantify ) )
         
         
class Ip6RouteSummary( Model ):
   __revision__ = 3
   protoModelStatus = Submodel( valueType=ServiceRoutingProtoModelStatus,
                               help="Active and configured protocol models" )
   vrfs = Dict( keyType=str, valueType=Ip6RouteSummaryForVrf, 
         help="Route summary for each VRF" )
   
   def render( self ):
      self.protoModelStatus.render()
      for vrf, vrfSummary in self.vrfs.iteritems():
         print "\nVRF: %s" % vrf
         vrfSummary.render()

   def degrade( self, dictRepr, revision ):
      '''
      Revision 2:
      Two new attributes added:
       'staticNonPersistent'
       'staticPersistent'
      Revision 3:
         instead of returning {rev-2-output }, return
             { "vrfs": { "vrf1": { rev-2-output } } }
         which can now support "show ip route vrf all summary"
      Since the 'all' token was supported only form revision 3, the
       'show ipv6 route vrf all summary' | json for revisions 1 and 2 will
       return the dict representation of the default Vrf
      '''
      vrfCount = len( dictRepr[ 'vrfs' ] )
      if vrfCount > 1:
         # The output for 'vrf all', this will have multiple vrfs, the degrade
         # function will return the default vrf as this is always present in
         # the output dict
         vrfName = DEFAULT_VRF
      else:
         # The 'show ipv6 route vrf <vrfName> summary' command will always return
         # a single vrf
         vrfName = dictRepr[ 'vrfs' ].keys()[ 0 ]
      if revision == 1:
         del dictRepr[ 'vrfs' ][ vrfName ][ 'staticNonPersistent' ]
         del dictRepr[ 'vrfs' ][ vrfName ][ 'staticPersistent' ]
         dictRepr[ 'vrfs' ][ vrfName ][ 'isis' ] = \
               dictRepr[ 'vrfs' ][ vrfName ][ 'isisCounts6' ][ 'isisTotal' ]
         # Leave isisCounts6 in case somebody has been using it with rev=1
         #del dictRepr[ 'vrfs' ][ vrfName ][ 'isisCounts6' ]
      return dictRepr[ 'vrfs' ][ vrfName ]

#-------------------------------------------------------
# show ip route rpf unicast [ VRF ] [ PREFIX | ADDR ]
#-------------------------------------------------------

class Ip6UrpfInterface( Model ):
   uRpfMode = Enum( values=( "strict", "loose", "strictDefault", "disable" ),
                    help="URPF mode state" )
   allowDefault = Bool( help=( "Packets are permitted to"
                               " come in on the default interface" ) )

class Ip6UrpfInterfaces( Model ):
   interfaces = Dict( keyType=Interface, valueType=Ip6UrpfInterface,
                      help=( "Mapping of interface to the additional"
                             " interface state details" ),
                      optional=True )
   prefix = IpGenericPrefix( help="Destination Prefix",
                             optional=True )
   viaDefaultRoute = Bool( help="uRPF carried out via the default route",
                           optional=True )

   def formatUrpfMode( self, mode ):
      if mode == 'strict':
         return 'Strict'
      elif mode == 'loose':
         return 'Loose'
      elif mode == 'strictDefault':
         return 'Strict'
      else:
         return 'Disable'

   def render( self ):
      if self.interfaces:
         if self.prefix:
            defaultRouteStr = 'via default route' if self.viaDefaultRoute else ""
            print '\nRPF interfaces for %s %s\n' % ( self.prefix, defaultRouteStr )
         else:
            print '\nuRPF configured on interfaces:\n'

         print '%-22s %-18s %-10s' % ( 'Interface', 'uRPF mode', 'Allow Default' )
         for intfId in self.interfaces:
            mode = self.interfaces[ intfId ].uRpfMode
            allowDefault = 'Yes' if self.interfaces[ intfId ].allowDefault else "No"
            print '%-22s %-18s %-10s' % \
                  ( intfId,
                    self.formatUrpfMode( mode ),
                    allowDefault )
