# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


#-------------------------------------------------------------------------------
# This module implements interface-specific IPv6 configuration.
#-------------------------------------------------------------------------------
'''Commands supported in interface config mode'''

import CliParser
import IntfCli
import CliCommand
import CliMatcher
import LazyMount
import Tracing
import Ip6AddrMatcher
import ConfigMount
import Tac
import os
import CliToken.Ipv6, CliToken.Verify
import CliExtensions
import Arnet
import IpLibTypes
import IraUrpfCli
import IraCommonCli
from CliPlugin.IraIp6Model import IpAddress
from CliPlugin.IraIp6Model import Ip6Status
from CliPlugin.IraIp6Model import InterfaceAddressIp6

ip6StatusDir = None
ip6ConfigDir = None
dynIp6ConfigDir = None
allIntfStatusDir = None
ip4ConfigDir = None
routingHardwareStatus = None
l3ConfigDir = None

#Getting additional information from Ipv6RouterAdvt/CliPlugin/Ip6RaCli.py
#Interface name is the parameter that needs to be passed to this procedure
extraIp6NdRaInfoHook = lambda intf, result: None

addrSource = IpLibTypes.addrSource( Tac.Type( 'Arnet::AddressFamily' ).ipv6 )
acceptDad = Tac.Type( 'Ip6::AcceptDad' )

@Tac.memoize
def _mgmtPrefix():
   # MGMT_INTF_PREFIX allows a test to override the prefix that
   # identifies a management interface
   return os.environ.get( "MGMT_INTF_PREFIX", "Management" )

def isManagement( name ):
   return name.startswith( _mgmtPrefix() )

def checkLLAddrAndPrintWarning( mode, addrArgWithMask ):
   if not addrArgWithMask.address.isLinkLocal:
      mode.addWarning( " Address is not a link local address."
                       " Ignoring this command" )
      return False

   if addrArgWithMask.len != 64:
      mode.addWarning( " Address with only /64 mask allowed as link local."
                       " Ignoring this command" )
      return False

   return True

def isLinkLocalAddress( mode, address ):
   try:
      return Arnet.Ip6Addr( address ).isLinkLocal
   except IndexError:
      return False

#-------------------------------------------------------------------------------
# Adds IPv6-specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------

def ipv6SupportedOnIntfConfigMode( intfConfigMode ):
   # Don't configure ipv6 on ineligible interfaces
   if not intfConfigMode.intf.routingSupported():
      return False
   
   return True

class Ip6IntfConfigModelet( CliParser.Modelet ):

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ):
      CliParser.Modelet.__init__( self )
      self.intf = Ip6Intf( intfConfigMode.intf, intfConfigMode.sysdbRoot,
                           createIfMissing=True )
      self.mode = intfConfigMode

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ipv6SupportedOnIntfConfigMode( mode )

   def enableIpv6( self, args ):
      Tracing.trace0( "Enable Ipv6 on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                        self.intf.name, configType="IPv6" )
      self.intf.config().enable = True

   def noIpv6Enable( self, args ):
      Tracing.trace0( "Disabling IPv6 on", self.intf.name )
      self.intf.config().enable = False

   def setIp6Addr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      Tracing.trace0( "Adding IP6 address", ip6Prefix, "on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      if self.intf.config().addrSource == addrSource.slaac:
         self.intf.config().addr.clear()
         self.intf.config().addrSource = addrSource.manual
      self.intf.setAddr( ip6Prefix )
      if self.intf.getNsSrc( ) == ip6Prefix.address:
         self.intf.setUseNsSrc( False )

   def setIp6AddrAsNsSrc( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      addr = ip6Prefix.address
      Tracing.trace0( "Adding IP6 address", ip6Prefix, "on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      self.intf.setAddr( ip6Prefix )
      Tracing.trace0( "Setting Ip6 address", addr, "as the source for", \
                      "outgoing neighbour solicitations on this interface" )
      self.intf.setNsSrc( addr )
      self.intf.setUseNsSrc( True )

   def enableSlaac( self, args ):
      Tracing.trace0( "Enabling SLAAC on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      if not ( self.intf.config().addrSource == addrSource.slaac ):
         self.intf.config().addr.clear()
         self.intf.config().addrSource = addrSource.slaac

   def disableSlaac( self, args ):
      Tracing.trace0( "Disabling SLAAC on", self.intf.name )
      if self.intf.config().addrSource == addrSource.slaac:
         self.intf.config().addr.clear()
         self.intf.config().addrSource = addrSource.manual

   def noIp6Addr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      if self.intf.config().addrSource == addrSource.slaac:
         self.mode.addError( 'Address determined by SLAAC' )
         return
      self.intf.delAddr( ip6Prefix )
      Tracing.trace0( "Removing IPv6 address", ip6Prefix, "on", self.intf.name )

   def noIp6Addrs( self, args ):
      self.intf.config().addrSource = addrSource.manual
      self.intf.delAddrs()
      Tracing.trace0( "Removing Ipv6 addresses on ", self.intf.name )

   def enableAcceptRaDefRtr( self, args ):
      Tracing.trace0( "Accepting default router from RA on", self.intf.name )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                          self.intf.name, configType="IPv6" )
      self.intf.config().acceptRaDefRtr = True

   def disableAcceptRaDefRtr( self, args ):
      Tracing.trace0( "Not accepting default router from RA on", self.intf.name )
      self.intf.config().acceptRaDefRtr = False

   def ndNsRetransmitInterval( self, args ):
      ndNsRetransmitIntervalValue = args[ 'INTERVAL' ]
      Tracing.trace0( "Setting ns-interval on ", self.intf.name, " to ",
            ndNsRetransmitIntervalValue )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                       self.intf.name, configType="IPv6" )
      self.intf.ndNsRetransmitInterval( ndNsRetransmitIntervalValue )

   def noNdNsRetransmitInterval( self, args ):
      Tracing.trace0( "Default ns-interval of 0 on ", self.intf.name )
      self.intf.noNdNsRetransmitInterval()

   def ndDad( self, args ):
      ndDadValue = args.get( 'DAD_VALUE', 'enabled' )
      Tracing.trace0( "Setting accept-dad on ", self.intf.name, "to", ndDadValue )
      IraCommonCli.printWarningIfNotRoutedPort( self.mode,
                                       self.intf.name, configType="IPv6" )
      if ndDadValue == 'disabled':
         self.intf.ndDadDisabled()
      elif ndDadValue == 'strict':
         self.intf.ndDadStrict()
      else:
         self.intf.ndDad()

   def noNdDad( self, args ):
      Tracing.trace0( "No accept-dad on ", self.intf.name )
      self.intf.noNdDad()

   def defaultNdDad( self, args ):
      Tracing.trace0( "Default accept-dad on ", self.intf.name )
      self.intf.defaultNdDad()

   def setUrpf( self, args ):
      urpfMode = ( IraUrpfCli.urpfMode.loose if 'any' in args
                                             else IraUrpfCli.urpfMode.strict )
      allowDefault = 'allow-default' in args
      IraUrpfCli.doUrpf( self.intf, False, urpfMode,
                         allowDefault, args.get( 'ACL_NAME' ) )

   def noUrpf( self, args ):
      IraUrpfCli.doUrpf( self.intf, no=True )

   def setIp6LLaddr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.setLLAddr( ip6Prefix )

   def setIp6LLaddrNoMask( self, args ):
      ip6Addr = args[ 'IP6ADDR' ]
      ip6Prefix = Arnet.Ip6AddrWithMask( ip6Addr, mask=64 )
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.setLLAddr( ip6Prefix )

   def noIp6LLAddr( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.delLLAddr( ip6Prefix )

   def noIp6LLAddrNoMask( self, args ):
      ip6Addr = args[ 'IP6ADDR' ]
      ip6Prefix = Arnet.Ip6AddrWithMask (ip6Addr, mask=64)
      if checkLLAddrAndPrintWarning( self.mode, ip6Prefix ):
         self.intf.delLLAddr( ip6Prefix )

   def attachedRoutes( self, args ):
      self.intf.attachedRoutes()

   def defaultAttachedRoutes( self, args ):
      self.intf.defaultAttachedRoutes()

   def noAttachedRoutes( self, args ):
      self.intf.noAttachedRoutes()


#-------------------------------------------------------------------------------
# Associate the Ip6IntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( Ip6IntfConfigModelet )

#-------------------------------------------------------------------------------
# Adds IPv6-specific CLI commands to the "config-if" mode for routed ports.
#-------------------------------------------------------------------------------
class RoutingProtocolIntfConfigModelet( CliParser.Modelet ):

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ): # pylint: disable-msg=W0231
      CliParser.Modelet.__init__( self )
      self.mode = intfConfigMode

   @staticmethod
   def shouldAddModeletRule( mode ):
      # we want to exclude management interfaces and loopback interfaces
      return ( mode.intf.routingSupported() and
               not mode.intf.name.startswith( "Management" ) and
               not mode.intf.name.startswith( "Loopback" ) )

#-------------------------------------------------------------------------------
# Associate the RoutingProtocolIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( RoutingProtocolIntfConfigModelet )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 enable
#--------------------------------------------------------------------------------
class Ipv6EnableCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 enable'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'enable': 'Enable IPv6 on an interface',
   }

   handler = Ip6IntfConfigModelet.enableIpv6
   noOrDefaultHandler = Ip6IntfConfigModelet.noIpv6Enable

Ip6IntfConfigModelet.addCommandClass( Ipv6EnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6PREFIX ns source
#--------------------------------------------------------------------------------
matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Set the IPv6 address of an interface' )

class Ipv6AddressIp6PrefixNsSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address IP6PREFIX ns source'
   noOrDefaultSyntax = 'ipv6 address IP6PREFIX ns ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6PREFIX': Ip6AddrMatcher.ip6PrefixMatcher,
      'ns': 'ICMPv6 Neighbour Solicitation Options',
      'source': ( 'Make this IPv6 address the source of all outgoing '
                   'ICMPv6 neighbour solicitations' ),
   }

   handler = Ip6IntfConfigModelet.setIp6AddrAsNsSrc
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6Addr

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6PrefixNsSourceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6PREFIX
#--------------------------------------------------------------------------------
class Ipv6AddressIp6PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address IP6PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6PREFIX': Ip6AddrMatcher.ip6PrefixMatcher,
   }

   handler = Ip6IntfConfigModelet.setIp6Addr
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6Addr

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6PrefixCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ipv6 address
#--------------------------------------------------------------------------------
class Ipv6AddressCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 address'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
   }

   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6Addrs

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address auto-config
#--------------------------------------------------------------------------------
class Ipv6AddressAutoConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address auto-config'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'auto-config': 'Use SLAAC to automatically configure the IPv6 address',
   }

   handler = Ip6IntfConfigModelet.enableSlaac
   noOrDefaultHandler = Ip6IntfConfigModelet.disableSlaac

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressAutoConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6PREFIX link-local
#--------------------------------------------------------------------------------
class Ipv6AddressIp6PrefixLlCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address IP6PREFIX link-local'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6PREFIX': Ip6AddrMatcher.ip6PrefixMatcher,
      'link-local': 'Use link-local address',
   }

   handler = Ip6IntfConfigModelet.setIp6LLaddr
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6LLAddr

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6PrefixLlCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 address IP6ADDR link-local
#--------------------------------------------------------------------------------
class Ipv6AddressIp6AddrLinkLocalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 address IP6ADDR link-local'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'address': matcherAddress,
      'IP6ADDR': Ip6AddrMatcher.ip6AddrMatcher,
      'link-local': 'Use link-local address',
   }

   handler = Ip6IntfConfigModelet.setIp6LLaddrNoMask
   noOrDefaultHandler = Ip6IntfConfigModelet.noIp6LLAddrNoMask

Ip6IntfConfigModelet.addCommandClass( Ipv6AddressIp6AddrLinkLocalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra rx accept default-route
#--------------------------------------------------------------------------------
ndMatcher = CliMatcher.KeywordMatcher( 'nd',
      helpdesc='Neighbor Discovery / Router Advertisement' )

class Ipv6NdRaRxAcceptDefaultRouteCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra rx accept default-route'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'ra': 'Router Advertisement commands',
      'rx': 'Configure action on receiving RA',
      'accept': 'Accept information on received RA',
      'default-route': 'Accept default router information from RA',
   }

   handler = Ip6IntfConfigModelet.enableAcceptRaDefRtr
   noOrDefaultHandler = Ip6IntfConfigModelet.disableAcceptRaDefRtr

Ip6IntfConfigModelet.addCommandClass( Ipv6NdRaRxAcceptDefaultRouteCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ns-interval INTERVAL
#--------------------------------------------------------------------------------
class NdNsRetransmitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ns-interval INTERVAL'
   noOrDefaultSyntax = 'ipv6 nd ns-interval ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'ns-interval': 'Time between neighbor solicitation retransmissions',
      'INTERVAL': CliMatcher.IntegerMatcher( 1000, 0xffffffff,
         helpdesc='Milliseconds' ),
   }

   handler = Ip6IntfConfigModelet.ndNsRetransmitInterval
   noOrDefaultHandler = Ip6IntfConfigModelet.noNdNsRetransmitInterval

Ip6IntfConfigModelet.addCommandClass( NdNsRetransmitIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd dad [ DAD_VALUE ]
#--------------------------------------------------------------------------------
class Ipv6NdDadCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd dad [ DAD_VALUE ]'
   noOrDefaultSyntax = 'ipv6 nd dad ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'dad': 'Disable the IPv6 address on detecting a conflict',
      'DAD_VALUE': CliMatcher.EnumMatcher( {
         'disabled': 'Disable DAD',
         'strict': ( 'Disable the IPv6 interface on detecting a conflict of the '
                      'MAC address based autogenerated link-local address' ),
      } ),
   }

   handler = Ip6IntfConfigModelet.ndDad
   noHandler = Ip6IntfConfigModelet.noNdDad
   defaultHandler = Ip6IntfConfigModelet.defaultNdDad

Ip6IntfConfigModelet.addCommandClass( Ipv6NdDadCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 verify unicast source reachable-via (
#                            ( any [ exception-list ACL_NAME ] ) |
#                            ( rx [ allow-default ] [ exception-list ACL_NAME ] ) )
#--------------------------------------------------------------------------------
class UrpfCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ipv6 verify unicast source reachable-via ('
                        '( any [ exception-list ACL_NAME ] ) |'
                        '( rx [ allow-default ] [ exception-list ACL_NAME ] ) )' )
   noOrDefaultSyntax = 'ipv6 verify unicast ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'verify': CliToken.Verify.verifyMatcherForConfigIf,
      'unicast': CliCommand.guardedKeyword( 'unicast',
         helpdesc='Enable RPF on unicast traffic',
         guard=IraUrpfCli.urpf6SupportedGuard ),
      'source': IraUrpfCli.urpfSourceMatcher,
      'reachable-via': IraUrpfCli.urpfReachMatcher,
      'any': IraUrpfCli.urpfLooseMatcher,
      'rx': IraUrpfCli.urpfStrictMatcher,
      'allow-default': IraUrpfCli.urpfAllowDefaultMatcher,
      'exception-list': CliCommand.guardedKeyword( 'exception-list',
         helpdesc='Exception list',
         guard=IraUrpfCli.urpf6ExceptionSupportedGuard ),
      'ACL_NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: IraUrpfCli.getUserAclNames( [ 'ipv6' ] ),
         helpdesc='Exception Acl' )
   }

   handler = Ip6IntfConfigModelet.setUrpf
   noOrDefaultHandler = Ip6IntfConfigModelet.noUrpf

Ip6IntfConfigModelet.addCommandClass( UrpfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 attached-routes
#--------------------------------------------------------------------------------
class Ipv6AttachedRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 attached-routes'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'attached-routes': 'Install attached routes in FIB',
   }

   handler = Ip6IntfConfigModelet.attachedRoutes
   noHandler = Ip6IntfConfigModelet.noAttachedRoutes
   defaultHandler = Ip6IntfConfigModelet.defaultAttachedRoutes

Ip6IntfConfigModelet.addCommandClass( Ipv6AttachedRoutesCmd )

#-------------------------------------------------------------------------------
# Network-layer interface extension for IPv6.
#-------------------------------------------------------------------------------

# canSetIntfIpHook  allows other packages to check if a proposed ip
# address (with mask) is acceptable.  The hook takes an Arnet::Ip6AddrWithMask
# and an interface name as arguments and returns a pair [ b, str ] where b is
# True if the proposed address is acceptable, False otherwise, and str is an
# error or warning string suitable for display in the CLI.
canSetIntfIpHook = CliExtensions.CliHook()
canSetIntfLLAddrHook = CliExtensions.CliHook()

class Ip6Intf( IntfCli.IntfDependentBase ):
   emptyIntf = None
   emptyL3Intf = None
   def __init__( self, intf, sysdbRoot, createIfMissing=False ):
      self.intfStatuses = ip6StatusDir.intf
      self.intfStatus = None
      self.createIfMissing_ = createIfMissing
      self.mode = intf.mode_
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   name = property( lambda self: self.intf_.name ) # pylint: disable-msg=W1001

   #----------------------------------------------------------------------------
   # Returns whether the Ip6::IntfStatus object exists for this interface.
   #----------------------------------------------------------------------------
   def lookup( self ):
      # The Ip6Intf exists if an associated Ip6::IntfStatus object exists. Since
      # Ira is the one responsible for creating the status object, there is a
      # window in which the ip6 intf has been configured in the cli, but
      # 'show interfaces' does not display the ip6 intf status.
      self.intfStatus = self.intfStatuses.get( self.intf_.name )
      return self.intfStatus is not None

   #----------------------------------------------------------------------------
   # Returns the Ip6::IntfConfig object for this interface, creating it if it does
   # not already exist.  Will also create L3Config as side effect
   #----------------------------------------------------------------------------
   def config( self ):
      ( ip6Config, l3Config ) = self.bothConfigs()
      assert ip6Config and l3Config
      return ip6Config

   #----------------------------------------------------------------------------
   # Returns the L3Config object for this interface, creating it if it does
   # not already exist.  Will also create Ip6IntfConfig as a side effect
   #----------------------------------------------------------------------------
   def l3Config( self ):
      ( ip6Config, l3Config ) = self.bothConfigs()
      assert ip6Config and l3Config
      return l3Config

   def bothConfigs( self ):
      l3c = l3ConfigDir.intfConfig.get( self.intf_.name )
      if not l3c:
         if self.createIfMissing_:
            l3c = l3ConfigDir.intfConfig.newMember( self.intf_.name )
         else:
            if not Ip6Intf.emptyL3Intf:
               l3ConfigDirTmp = Tac.newInstance( "L3::Intf::ConfigDir", "" )
               Ip6Intf.emptyL3Intf = l3ConfigDirTmp.intfConfig.newMember( "" )
            l3c = Ip6Intf.emptyL3Intf

      i6ic = ip6ConfigDir.intf.get( self.intf_.name )
      if not i6ic:
         if self.createIfMissing_:
            i6ic = ip6ConfigDir.intf.newMember( self.intf_.name )
            i6ic.l3Config = l3c
         elif self.lookup():
            configSource = self.intfStatus.configSource
            if configSource in dynIp6ConfigDir.entityPtr and \
               configSource in dynIp6ConfigDir.entryState and \
               dynIp6ConfigDir[ configSource ]:
               i6ic = dynIp6ConfigDir[ configSource ].intf.get( self.intf_.name )
         if not i6ic:
            if not Ip6Intf.emptyIntf:
               configDir = Tac.newInstance( "Ip6::Config", "" )
               Ip6Intf.emptyIntf = configDir.intf.newMember( "" )
               Ip6Intf.emptyIntf.l3Config = l3c
            i6ic = Ip6Intf.emptyIntf
            
      return( i6ic, l3c )

   def setAddr( self, addressWithMask ):
      a1 = addressWithMask
      addrs = self.config().addr

      ( valid, errorString ) = self.intf_.isValidHostInet6Address( a1 )
      if not valid:
         self.mode.addError( errorString )
         return

      checker = Tac.Value( "Ira::Address6AssignmentChecker" )
      addr6QParams = Tac.newInstance( "Ira::Address6QueryParams", ip6ConfigDir,
                                      self.config().intfId, _mgmtPrefix(),
                                      a1 )
      ans = checker.check( addr6QParams )
      if ans.acceptable:
         if ans.msg:
            self.mode.addWarning( ans.msg )
      else:
         self.mode.addError( ans.msg )
         return False      

      # Check address against extension hooks, print warning or error message,
      # exit if the address is not acceptable.
      for hook in canSetIntfIpHook.extensions():
         [ accept, message ] = hook( self.name, addressWithMask, mode=self.mode )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return

      info = Tac.Value( "Ip6::AddressInfo", key=a1 )
      if addrs.has_key( info.key ):
         return
      addrs.addMember( info )

   def setNsSrc( self, addr ):
      self.config().nsSourceAddr = addr

   def getNsSrc( self ):
      return self.config().nsSourceAddr

   def setUseNsSrc( self, enable ):
      self.config().useNsSourceAddr = enable

      if not enable:
         # If we are disabling useNsSourceAddr, also
         # clear the nsSourceAddr
         self.setNsSrc( Arnet.Ip6Addr( '::' ) )

   def setLLAddr( self, addrArgWithMask ):
      # Check address against extension hooks, print warning or error messa
      # exit if the address is not acceptable.
      for hook in canSetIntfLLAddrHook.extensions():
         [ accept, message ] = hook( self.name, addrArgWithMask )
         if message:
            if accept:
               self.mode.addWarning( message )
            else:
               self.mode.addError( message )
         if not accept:
            return
      self.config().linkLocalAddrWithMask = addrArgWithMask

   def delLLAddr( self, addrArgWithMask ):
      llAddr = self.config().linkLocalAddrWithMask
      if not llAddr == addrArgWithMask:
         self.mode.addWarning( " Address not configured on interface "
                               " Ignoring this command " )
      else:
         nullLLAddr = Arnet.Ip6AddrWithMask( '::/0' )
         self.config().linkLocalAddrWithMask = nullLLAddr

   def delAddr( self, addressWithMask ):
      if addressWithMask.address.isLinkLocal:
         self.mode.addError( "Configuring link local addresses must use"
                             " 'link-local' keyword " )
         return
      a1 = addressWithMask
      addrs = self.config().addr
      
      if addrs.has_key( a1 ):
         if a1.address == self.config().nsSourceAddr:
            self.setUseNsSrc( False )
         del addrs[ a1 ]
      else:
         self.mode.addError( "Address not configured on interface" )
         
   def delAddrs( self ):
      addrs = self.config().addr
      for addr in addrs:
         if addrs.has_key( addr ):
            del addrs[ addr ]
      self.setUseNsSrc( False )
      self.delLLAddr( self.config().linkLocalAddrWithMask )


   def ndNsRetransmitInterval( self, ndNsRetransmitIntervalValue ):
      self.config().nsRetransmitInterval = ndNsRetransmitIntervalValue

   def noNdNsRetransmitInterval( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.nsRetransmitInterval = intfConfig.nsRetransmitIntervalDefault

   def ndDad( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadEnabled

   def ndDadStrict( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadStrict

   def ndDadDisabled( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadDisabled

   def noNdDad( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadGlobal

   def defaultNdDad( self ):
      intfConfig = self.config()
      if not intfConfig:
         return
      intfConfig.acceptDad = acceptDad.dadGlobal

   def attachedRoutes( self ):
      self.config().attachedRoutes = True

   def defaultAttachedRoutes( self ):
      self.config().attachedRoutes = True

   def noAttachedRoutes( self ):
      self.config().attachedRoutes = False

   #----------------------------------------------------------------------------
   # Destroys the IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      intf = self.intf_.name
      del ip6ConfigDir.intf[ intf ]
      if not intf in ip4ConfigDir.ipIntfConfig and intf in l3ConfigDir.intfConfig:
         del l3ConfigDir.intfConfig[ intf ]
         
   #----------------------------------------------------------------------------
   # Returns the IntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      return self.intfStatus

   def enabled( self ):
      nullLLAddr = Arnet.Ip6AddrWithMask( '::/0' )
      llConfigured = not ( self.config().linkLocalAddrWithMask == nullLLAddr )
      return self.config().enable or len( self.config().addr ) > 0 or llConfigured

   #----------------------------------------------------------------------------
   # Called by "show interface" to display addresses on the interface
   #----------------------------------------------------------------------------
   def showNetworkAddr( self ):
      return self.showAddr( lookupIpIntfStatus=True )

   #----------------------------------------------------------------------------
   # Returns an InterfaceAddressIp6 object for this interface.
   #
   # Pass lookupIpIntfStatus=True if you want to return an object only
   # if the interface is not in *enabled* state. Note that aggregate
   # interfaces (eg, port-channel, vlan) may require at least one
   # member before the aggregate can get into the enable state.
   #----------------------------------------------------------------------------
   def showAddr( self, lookupIpIntfStatus=False ):
      if lookupIpIntfStatus and not self.lookup():
         return None

      # Build the InterfaceAddressIp6 object from the interface state, since
      # the operational state may override the configuration state.

      operState = self.show()
      if operState is None:
         return None
      if not ( operState.linkLocal or operState.addresses ):
         return None

      result = InterfaceAddressIp6()
      result.linkLocalIp6 = operState.linkLocal
      result.globalUnicastIp6s = operState.addresses
      result.globalAddressesAreVirtual = operState.globalAddressesAreVirtual
      result.addrSource = operState.addrSource

      return result

   #----------------------------------------------------------------------------
   # Returns an Ip6Status object for this interface.
   #----------------------------------------------------------------------------
   def show( self ):
      self.lookup()
      status = self.status()
      result = Ip6Status()
      result.state = "disabled"
      if self.enabled():
         result.state = "stalled"  # DAD failure on the link local ip
         if self.intf_.name.startswith( "Loopback" ):
            # Loopback interfaces - state is disabled till the 1st global
            # addr is configured on the interface.
            if status and status.addr:
               result.state = "enabled"
               for value in status.addr.itervalues():
                  addr = value.key
                  if addr.address.isLinkLocal:
                     result.linkLocal = IpAddress(
                        address=addr.address.stringValue,
                        subnet=addr.subnet.stringValue,
                        active=True, dadfailed=False )
                     break
            else:
               result.state = "disabled"
         elif status:
            for addr in status.addr.keys():
               if addr.address.isLinkLocal:
                  # FIXME so far an address being configured and not
                  # showing up in status is a good sign that it is tentative
                  # but I may need something more definite in the future
                  result.state = "enabled"
                  result.linkLocal = IpAddress( address=addr.address.stringValue,
                                                subnet=addr.subnet.stringValue,
                                                active=True, dadfailed=False )
                  break
            # if link-local address is not in the the above collection,
            # look for it in the dadfailed collection
            else:
               for addr in status.dadfailed.keys():
                  if addr.address.isLinkLocal:
                     result.state = "stalled"
                     result.linkLocal = IpAddress( address=addr.address.stringValue,
                                                   subnet=addr.subnet.stringValue,
                                                   active=True, dadfailed=True )
                     break
            if self.config().addrSource == addrSource.slaac:
               intfStatus = allIntfStatusDir[ self.config().intfId ]
               for addr in status.addr.keys():
                  if addr.address.isLinkLocal:
                     continue
                  # add the address if it is a SLAAC configured address
                  if not IraCommonCli.isSlaacAddress( addr.address, intfStatus ):
                     continue
                  addr = IpAddress( address = addr.address.stringValue,
                                    subnet=addr.subnet.stringValue,
                                    active=True,
                                    dadfailed=False )
                  result.addresses.append( addr )
      config = self.config()
      result.addrSource = config.addrSource
      result.vrf = config.vrf
      result.globalAddressesAreVirtual = bool( config.virtualAddr )
      configAddr = config.virtualAddr if result.globalAddressesAreVirtual \
                   else config.addr
      for addr in configAddr:
         dadfailed = True if status and len( status.dadfailed ) and \
               addr in status.dadfailed else False
         addr = IpAddress( address=addr.address.stringValue,
                            subnet=addr.subnet.stringValue,
                            active=bool( status and addr in status.addr ),
                            dadfailed=dadfailed )
         result.addresses.append( addr )

      # Everything else is status so get out if ipv6 is disabled
      if not self.enabled():
         return result

      if status and status.mcaddr:
         for addr in status.mcaddr:
            result.multicastGroupAddresses.append( addr.stringValue )

      # URPF information
      # We need to display a warning message if V6 URPF is disabled due to
      # conflicts with V4 URPF
      result.urpfV4V6Mismatch = False
      result.urpf = config.urpf.mode
      if not routingHardwareStatus.urpfSupportedOnV4AndV6:
         if self.intf_.name in ip4ConfigDir.ipIntfConfig:
            v4Mode = ip4ConfigDir.ipIntfConfig[ self.intf_.name ].urpf.mode
            if config.urpf.mode != v4Mode and 'disable' not in \
                   [ config.urpf.mode, v4Mode ]:
               result.urpf = 'disable'
               result.urpfV4V6Mismatch = True

      result.maxMssIngress = config.maxMssIngress
      result.maxMssEgress = config.maxMssEgress
      # Check if Management interface is present in ip6/status to check if multiple
      # SUP have mgmt interface with only one being active at any point. TG series
      # has 2+ mgmt interface with only 1 being active hence kernel intf is missing
      # for the inactive intf.
      extraIp6NdRaInfoHook( self, result )
      return result

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ip6StatusDir, ip6ConfigDir, dynIp6ConfigDir, l3ConfigDir, ip4ConfigDir
   global routingHardwareStatus, allIntfStatusDir
   ip6StatusDir = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   ip6ConfigDir = ConfigMount.mount(
      entityManager, "ip6/config", "Ip6::Config", "w" )
   dynIp6ConfigDir = LazyMount.mount( entityManager, "ip6/input/dynIpConfig",
                                      "Tac::Dir", "ri" )
   l3ConfigDir = ConfigMount.mount(
      entityManager, "l3/intf/config", "L3::Intf::ConfigDir", "w" )
   ip4ConfigDir = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   # Cause no interface to destroy us....
   IntfCli.Intf.registerDependentClass( Ip6Intf, priority=10 )
