# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP6 configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for IP'''

import os.path
import sys

import Arnet
import Arnet.MplsLib
from Arnet.Verify import verify
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Hardware
import CliToken.Icmp
import CliToken.Ipv6
import ConfigMount
import Intf.IntfRange
import IntfCli
import Ip6AddrMatcher
import IpLibTypes
import IraCommonCli
from IraCommonCli import NexthopCliExpression
import IraIp6IntfCli
import IraIp6RouteCliLib
from IraIp6RouteCliLib import (
      ip6,
      routing6,
      noIpv6RouteTableForVrfMsg,
      staticIpv6RoutingTable,
      manageIpv6Route,
      noIpv6Route,
      routeMatcherForIpv6Config,
)
import IraRouteCommon
import LazyMount

import ShowCommand
import SmashLazyMount
import Tac
import TacSigint
from CliModel import UnknownEntityError
from CliPlugin.IraCommonModel import (
   ServiceRoutingProtoModelStatus,
   VrfNotInstalledBfd,
)
from CliPlugin.IraIp6Model import (
      Ip6Statuses,
      Ip6StatusBrief,
      Ip6StatusesBrief,
      Ip6Routes,
      Ip6RouteSummary,
      ospfTotals6,
      bgpTotals6,
      isisTotals6,
      VrfIp6Routes,
      VrfIp6RoutesHost,
      Ip6RouteSummaryForVrf,
      Ip6UrpfInterface,
      Ip6UrpfInterfaces,
)
from CliPlugin.IraIpModel import VrfIpRibRoutes, V6EcmpModel
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import (
      VrfExprFactory,
      getAllVrfNames,
      vrfExists,
)
from CliPrint import CliPrint
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
from IraIpRouteCliLib import (
      tagMatcherForConfig,
      tagNumberMatcher,
      preferenceRangeMatcher,
      metricMatcherForConfig,
      metricValueMatcher,
)
from IraNexthopGroupCli import (
      nexthopGroupNamePattern,
      nexthopGroupSupportedGuard,
      getNexthopGroupNames,
)
from TechSupportCli import registerShowTechSupportCmdCallback

printer = CliPrint().lib

routing4 = IraRouteCommon.routing( IraRouteCommon.Ip4() )
ip4Config = None
ip6Config = None
l3Config = None
routingHardwareStatus = None
routing6HardwareStatus = None
routing6HardwareConfig = None
allVrfConfig = None
allVrfStatusLocal = None
allIntfStatusDir = None
runnabilityConfig = None
VERIFY_MAX_ROUTES = 100

optimizeProfileType = Tac.Type( 'Routing6::Hardware::OptimizeProfileType' )
addrSource = IpLibTypes.addrSource( Tac.Type( 'Arnet::AddressFamily' ).ipv6 )

def mplsPushSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.mplsPushSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Matchers "ip6" for use in implementing "ipv6 xxx" and "show ipv6 xxx" commands.
#-------------------------------------------------------------------------------
ipv6MatcherForShow = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='Details related to IPv6' )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

def getVrfNames( mode ):
   return sorted( allVrfConfig.vrf.members() )

#-------------------------------------------------------------------------------
# The "ipv6 unicast-routing" and
#     "no ipv6 unicast-routing [keep-static-routes]" commands.
#-------------------------------------------------------------------------------
unicastRoutingMatcher = CliMatcher.KeywordMatcher(
   'unicast-routing',
   helpdesc='Enable IPv6 unicast forwarding' )

def noIpv6Routing( mode, vrfName=DEFAULT_VRF, delStaticRoutes=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      mode.addError( "No such VRF %s." % vrfName )
      return
   routing6.config( vrfName ).routing = False
   # only warn if not startup and there are static routes
   if ( not delStaticRoutes and
        not mode.session_.startupConfig() and
        staticIpv6RoutingTable( vrfName ).route ):
      mode.addWarning( "Preserving static routes.  Use "
                       "'no ipv6 unicast-routing delete-static-routes' "
                       "to clear them." )
   if delStaticRoutes == 'delete-static-routes':
      # Delete all of the static routes if requested.
      with Tac.ActivityLockHolder():
         rt = staticIpv6RoutingTable( vrfName )
         for k in rt.route.keys():
            del rt.route[ k ]
   if vrfName in runnabilityConfig.vrf:
      del runnabilityConfig.vrf[ vrfName ]

def ipv6Routing( mode , vrfName=DEFAULT_VRF ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      mode.addError( "No such VRF %s." % vrfName )
      return
   if not routing6HardwareStatus.routingSupported and \
         not mode.session_.startupConfig():
      mode.addWarning( "IPv6 Hardware forwarding is not supported on this "
                       "platform. This means that all IPv6 traffic will be routed "
                       "in software." )
   routing6.config( vrfName ).routing = True
   runnabilityConfig.vrf[ vrfName ] = True

def ipV6Vrf( mode, token ):
   if routingHardwareStatus.vrfCapability.ipv6EnabledDefault:
      return None
   return CliParser.guardNotThisPlatform

class Ipv6UnicastRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 unicast-routing [ VRF ]'
   noOrDefaultSyntax = 'ipv6 unicast-routing ' \
                       '[ keep-static-routes | delete-static-routes ] ' \
                       '[ VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'unicast-routing': unicastRoutingMatcher,
            'keep-static-routes': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'keep-static-routes',
                  helpdesc='Disable IPv6 forwarding, ' \
                           'but do not remove the configured static routes' ),
               hidden=True ),
            'delete-static-routes': CliMatcher.KeywordMatcher(
                                    'delete-static-routes',
                                    helpdesc='Disable IPv6 forwarding, ' \
                                    'and remove the configured static routes' ),
            'VRF': VrfExprFactory( helpdesc='Enable routing in a VRF',
                                   guard=ipV6Vrf ),
          }

   @staticmethod
   def handler( mode, args ):
      ipv6Routing( mode, vrfName=args.get( 'VRF', DEFAULT_VRF ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delStaticRoutes = None
      if 'keep-static-routes' in args:
         delStaticRoutes = 'keep-static-routes'
      elif 'delete-static-routes' in args:
         delStaticRoutes = 'delete-static-routes'
      noIpv6Routing( mode, vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     delStaticRoutes=delStaticRoutes )

configMode.addCommandClass( Ipv6UnicastRoutingCmd )

#-------------------------------------------------------------------------------
# The "[no] ipv6 icmp rate-limit-unreachable 0" command.
#-------------------------------------------------------------------------------
class Ipv6IcmpUnreachableRateCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 icmp rate-limit-unreachable RATE'
   noOrDefaultSyntax = 'ipv6 icmp rate-limit-unreachable ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'icmp': CliToken.Icmp.icmpMatcherForConfig,
            'rate-limit-unreachable':
                           CliToken.Icmp.rateLimitUnreachableMatcherForConfig,
            'RATE': CliMatcher.IntegerMatcher( 0, 0, helpdesc='Rate' )
          }

   @staticmethod
   def handler( mode, args ):
      # Setting rate to 0 makes ICMP _not_ unreachable.
      routing6HardwareConfig.icmpUnreachable = 'RATE' not in args

   noOrDefaultHandler = handler

configMode.addCommandClass( Ipv6IcmpUnreachableRateCmd )

#------------------------------------------------------------------------------------
#The "(no|default) ipv6 icmp redirect"
#      commands.
#------------------------------------------------------------------------------------
redirectMatcher = CliMatcher.KeywordMatcher( 'redirect',
                                             helpdesc='Send ICMP redirects' )

def setIpv6Redirect( mode, no ):
   vrfNames = getAllVrfNames( mode )
   for vrf in vrfNames:
      r = routing6.config( vrf )
      r.sendRedirects = not no

def noIpv6Redirect( mode ):
   setIpv6Redirect( mode, no=True )

def ipv6Redirect( mode):
   setIpv6Redirect( mode, no=False )

class Ipv6IcmpRedirectCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 icmp redirect'
   noOrDefaultSyntax = 'ipv6 icmp redirect'

   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'icmp': CliToken.Icmp.icmpMatcherForConfig,
            'redirect': redirectMatcher }

   @staticmethod
   def handler( mode, args ):
      ipv6Redirect( mode )

   @staticmethod
   def noHandler( mode, args ):
      noIpv6Redirect( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      ipv6Redirect( mode )

configMode.addCommandClass( Ipv6IcmpRedirectCmd )

#------------------------------------------------------------------------------------
#The "(no|default) ipv6 nd enhanced-dad default"
#      commands.
#------------------------------------------------------------------------------------
# Determine if kernel supports enhanced DAD
def enhancedDadSupported( mode, token ):
   if os.path.isfile( '/proc/sys/net/ipv6/conf/all/enhanced_dad' ):
      return None
   else:
      return CliParser.guardNotThisPlatform

ndEnhancedDadMatcher = CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'enhanced-dad',
                           helpdesc='RFC7527 enhanced duplicate address detection' ),
                        guard=enhancedDadSupported )

class EnhancedDadCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd enhanced-dad default'
   noOrDefaultSyntax = syntax
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'nd': IraIp6IntfCli.ndMatcher,
            'enhanced-dad': ndEnhancedDadMatcher,
            'default': 'Setting for all the interfaces' }

   @staticmethod
   def handler( mode, args ):
      ip6Config.enhancedDad = True

   @staticmethod
   def noHandler( mode, args ):
      ip6Config.enhancedDad = False

   defaultHandler = handler

configMode.addCommandClass( EnhancedDadCmd )

#-------------------------------------------------------------------------------
# The "[no] ipv6 route" command.
#-------------------------------------------------------------------------------
nexthopAddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher( 'Address of the nexthop router' )

intfAddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher(
      "Forwarding router's address on destination interface" )

def intfAddr6RoutesSupportedGuard( mode, token ):
   if ip6.intfAddrRoutesSupported:
      return None
   return CliParser.guardNotThisPlatform

nexthopGroupMatcherForConfig = CliCommand.Node(
                                  CliMatcher.KeywordMatcher( 'nexthop-group',
                                     helpdesc='Specify nexthop group name'),
                                  guard=nexthopGroupSupportedGuard )

nexthopGroupNameMatcher = CliMatcher.DynamicNameMatcher( getNexthopGroupNames,
                              'Nexthop group name', pattern=nexthopGroupNamePattern )

trackingProtocols = {}
trackingProtocolsHidden = {}

def addTrackingProto( _trackingProtocols, _trackingProtocolsHidden,
                      protocol, desc, hidden=False ):
   if hidden:
      _trackingProtocolsHidden[ protocol ] = desc
   else:
      _trackingProtocols[ protocol ] = desc

# call addTrackingProto before TrackingProtocolExpr
addTrackingProto( trackingProtocols, trackingProtocolsHidden,
                  'bfd', 'Track this route using BFD' )

class TrackingProtocolExpr( CliCommand.CliExpression ):
   expression = 'TRACKING_PROTOCOLS | HIDDEN_TRACKING_PROTOCOLS'
   data = { 'TRACKING_PROTOCOLS': CliCommand.Node(
                                     CliMatcher.DynamicKeywordMatcher(
                                        lambda mode: trackingProtocols ),
                                     alias='trackingProto' ),
            'HIDDEN_TRACKING_PROTOCOLS': CliCommand.Node(
                                            CliMatcher.DynamicKeywordMatcher(
                                              lambda mode: trackingProtocolsHidden ),
                                            alias='trackingProto',
                                            hidden=True )
          }

vrfExprFactoryForConfig = VrfExprFactory( helpdesc='Configure routes in a VRF' )

labelMatcherForConfig = CliCommand.Node( CliMatcher.KeywordMatcher( 'label',
                                            helpdesc="Push an MPLS label" ),
                                         guard=mplsPushSupportedGuard )

class NexthopAddrExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP'
   data = { 'NEXTHOP': nexthopAddrMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'label' not in args and 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )

class NexthopAddrOrIntfExpr( CliCommand.CliExpression ):
   expression = '( NEXTHOP [ label LABEL_VALUE ] ) | ( nexthop-group NHG_NAME ) ' \
                '| ( INTF [ INTF_ADDR ] ) | NULL0_EXPR'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'label': labelMatcherForConfig,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher,
            'nexthop-group': nexthopGroupMatcherForConfig,
            'NHG_NAME': nexthopGroupNameMatcher,
            'INTF': IntfCli.Intf.matcher,
            'INTF_ADDR': CliCommand.Node( intfAddrMatcher,
                                          guard=intfAddr6RoutesSupportedGuard ),
            'NULL0_EXPR': IraCommonCli.Null0Expr
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'label' in args:
         args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )
      elif 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'nexthop-group' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )
      elif 'INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ]= args.pop( 'INTF' )
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = args.pop( 'INTF_ADDR',
                                                                       None )
      elif 'Null0' in args:
         del args[ 'Null0' ]
         args[ 'NEXTHOP_INTF' ][ 'Null0' ] = 'Null0'
      elif 'Null 0' in args:
         args[ 'NEXTHOP_INTF' ][ 'Null 0' ] = args.pop( 'Null 0' )

class NhMplsLabelExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( NEXTHOP label LABEL_VALUE )'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'label': labelMatcherForConfig,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )

class NexthopGroupExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( nexthop-group NHG_NAME )'
   data = { 'nexthop-group': nexthopGroupMatcherForConfig,
            'NHG_NAME': nexthopGroupNameMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )

class NexthopOrIntfNexthopExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP | NULL0_EXPR | ( INTF INTF_ADDR )'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'NULL0_EXPR': IraCommonCli.Null0Expr,
            'INTF': CliCommand.Node( IntfCli.Intf.matcher,
                                     guard=intfAddr6RoutesSupportedGuard ),
            'INTF_ADDR': intfAddrMatcher
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'Null0' in args:
         del args[ 'Null0' ]
         args[ 'NEXTHOP_INTF' ][ 'Null0' ] = 'Null0'
      elif 'Null 0' in args:
         args[ 'NEXTHOP_INTF' ][ 'Null 0' ] = args.pop( 'Null 0' )
      elif 'INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ]= args.pop( 'INTF' )
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = args.pop( 'INTF_ADDR' )

def manageIp6Route( mode, prefix, nexthop, routeOptions=None, vrfName=DEFAULT_VRF,
                    leakToVrf=None, egressVrf=None ):
   assert vrfName != ''
   if not vrfName:
      vrfName = DEFAULT_VRF

   if not vrfExists( vrfName ):
      mode.addError( "%s Create first." % noIpv6RouteTableForVrfMsg % vrfName )
      return

   manageIpv6Route( mode, prefix, nexthop, routeOptions, vrfName, leakToVrf,
                    egressVrf )

def noIp6Route( mode, prefix, nexthop, preference, vrfName, egressVrf ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   if not vrfExists( vrfName ):
      mode.addError( "%s Create first." % noIpv6RouteTableForVrfMsg % vrfName )
      return

   noIpv6Route( mode, prefix, nexthop, preference, vrfName, egressVrf )

defaultVrfMatcher = CliMatcher.KeywordMatcher( DEFAULT_VRF,
      helpdesc='Default virtual routing and forwarding instance' )

class LeakVrfCliExpr( CliCommand.CliExpression ):
   expression = 'leak LEAK_VRF ( CONFIG_VRFNAME | DEFAULT_VRF )'
   data = { 'leak': IraCommonCli.leakMatcherForConfig,
            'LEAK_VRF': IraCommonCli.leakVrfMatcherForConfig,
            'CONFIG_VRFNAME': CliMatcher.DynamicNameMatcher( getVrfNames,
                                       'VRF name', priority=CliParser.PRIO_LOW ),
            'DEFAULT_VRF': defaultVrfMatcher,
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'LEAK_TO_VRF' ] = ( args.pop( 'CONFIG_VRFNAME', None ) or
                                args.pop( 'DEFAULT_VRF', None ) )

class EgressVrfCliExpr( CliCommand.CliExpression ):
   expression = 'egress-vrf ( NON_DEFAULT | DEFAULT )'
   data = { 'egress-vrf': IraCommonCli.egressVrfMatcher,
            'NON_DEFAULT': CliMatcher.DynamicNameMatcher( getVrfNames,
               'VRF name', priority=CliParser.PRIO_LOW ),
            'DEFAULT': defaultVrfMatcher,
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EGRESS_VRF' ] = ( args.pop( 'NON_DEFAULT', None ) or
                               args.pop( 'DEFAULT', None ) )

class Ipv6RouteMplsLabelCmd( CliCommand.CliCommandClass ):
   '''This class covers mpls route'''
   syntax = 'ipv6 route [ VRF ] PREFIX NEXTHOP_LABEL ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) } ] ' \
            '[ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'NEXTHOP_LABEL': NhMplsLabelExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                        maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

      manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                      routeOptions=args[ 'OPTION' ],
                      vrfName=args.get( 'VRF', DEFAULT_VRF ),
                      leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( Ipv6RouteMplsLabelCmd )

class Ipv6RouteNexthopGroupCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-group route'''
   syntax = 'ipv6 route [ VRF ] PREFIX NEXTHOP_GROUP ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) | ' \
            '( metric METRICVALUE ) } ] [ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'NEXTHOP_GROUP': NexthopGroupExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                        maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'metric': CliCommand.Node( metricMatcherForConfig,
                                       maxMatches=1 ),
            'METRICVALUE': CliCommand.Node( metricValueMatcher,
                                            maxMatches=1 ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )
      if 'metric' in args:
         args[ 'OPTION' ].add( ( 'metric', args.pop( 'METRICVALUE' ) ) )

      manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                      routeOptions=args[ 'OPTION' ],
                      vrfName=args.get( 'VRF', DEFAULT_VRF ),
                      leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( Ipv6RouteNexthopGroupCmd )

class Ipv6RouteNexthopOrComboCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-only routes, combo routes(nexthop+intf)'''
   syntax = 'ipv6 route [ VRF ] PREFIX NEXTHOP_ADDR_OR_INTF ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) | ' \
            '( track TRACKING_PROTO ) | ( metric METRICVALUE ) } ] [ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                        maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'track': CliCommand.Node( IraCommonCli.trackMatcherForConfig,
                                      maxMatches=1 ),
            'TRACKING_PROTO': TrackingProtocolExpr,
            'metric': CliCommand.Node( metricMatcherForConfig,
                                       maxMatches=1 ),
            'METRICVALUE': CliCommand.Node( metricValueMatcher,
                                            maxMatches=1 ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )
      if 'track' in args:
         args[ 'OPTION' ].add( ( 'trackingProto',
                                 args.pop( 'trackingProto' )[ 0 ] ) )
      if 'metric' in args:
         args[ 'OPTION' ].add( ( 'metric', args.pop( 'METRICVALUE' ) ) )

      manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                      routeOptions=args[ 'OPTION' ],
                      vrfName=args.get( 'VRF', DEFAULT_VRF ),
                      leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( Ipv6RouteNexthopOrComboCmd )

class Ipv6RouteInterfaceCmd( CliCommand.CliCommandClass ):
   '''This class covers interface routes'''
   syntax = 'ipv6 route [ VRF ] PREFIX INTF ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) | ' \
            '( metric METRICVALUE ) } ] [ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'INTF': IntfCli.Intf.matcher,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                        maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'metric': CliCommand.Node( metricMatcherForConfig,
                                       maxMatches=1 ),
            'METRICVALUE': CliCommand.Node( metricValueMatcher,
                                            maxMatches=1 ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   @staticmethod
   def handler( mode, args ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )
      if 'metric' in args:
         args[ 'OPTION' ].add( ( 'metric', args.pop( 'METRICVALUE' ) ) )

      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

      manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                      routeOptions=args[ 'OPTION' ],
                      vrfName=args.get( 'VRF', DEFAULT_VRF ),
                      leakToVrf=args.get( 'LEAK_TO_VRF' ) )

configMode.addCommandClass( Ipv6RouteInterfaceCmd )

class Ipv6RouteEgressVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 route [ VRF ] PREFIX EGRESS_VRF NEXTHOP_ADDR ' \
            '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) } ] ' \
            '[ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'EGRESS_VRF': EgressVrfCliExpr,
            'NEXTHOP_ADDR': NexthopAddrExpr,
            'PREFERENCE': CliCommand.Node( preferenceRangeMatcher,
                                           maxMatches=1 ),
            'tag': CliCommand.Node( tagMatcherForConfig,
                                    maxMatches=1 ),
            'TAGNUM': CliCommand.Node( tagNumberMatcher,
                                        maxMatches=1 ),
            'name': CliCommand.Node( IraCommonCli.nameMatcherForConfig,
                                     maxMatches=1 ),
            'NEXTHOP_NAME': CliCommand.Node( IraCommonCli.nexthopNameMatcher,
                                             maxMatches=1 ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'OPTION' ] = set()
      if 'PREFERENCE' in args:
         args[ 'OPTION' ].add( ( 'preference', args.pop( 'PREFERENCE' ) ) )
      if 'tag' in args:
         args[ 'OPTION' ].add( ( 'tag', args.pop( 'TAGNUM' ) ) )
      if 'name' in args:
         args[ 'OPTION' ].add( ( 'nextHopName', args.pop( 'NEXTHOP_NAME' ) ) )

   @staticmethod
   def handler( mode, args ):
      manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                      routeOptions=args[ 'OPTION' ],
                      vrfName=args.get( 'VRF', DEFAULT_VRF ),
                      leakToVrf=args.get( 'LEAK_TO_VRF' ),
                      egressVrf=args.get( 'EGRESS_VRF' ) )

configMode.addCommandClass( Ipv6RouteEgressVrfCmd )

class NoIpv6RouteCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 route [ VRF ] PREFIX ' \
                       '[ ( EGRESS_VRF NEXTHOP_ADDR ) | NEXTHOP_ADDR_OR_INTF ] ' \
                       '[ PREFERENCE ] ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'EGRESS_VRF': EgressVrfCliExpr,
            'NEXTHOP_ADDR': NexthopAddrExpr,
            'NEXTHOP_ADDR_OR_INTF': NexthopAddrOrIntfExpr,
            'PREFERENCE': preferenceRangeMatcher
          }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                  args.get( 'PREFERENCE' ),
                  args.get( 'VRF', DEFAULT_VRF ),
                  args.get( 'EGRESS_VRF' ) )

configMode.addCommandClass( NoIpv6RouteCmd )

#-------------------------------------------------------------------------------
# Ipv6 info for "show ip" command. This is called from IraIpCli.showIp
#-------------------------------------------------------------------------------

def generateIp6EcmpModel( mode, vrfName=DEFAULT_VRF ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   assert vrfName
   r = routing6.config( vrfName )
   v6RoutingEnabled = r.routing

   ip6Model = V6EcmpModel()
   if not routing6.hwStatus().ecmpSupported:
      hwConfig = routing6.hwConfig( vrfName )
      ip6Model.v6EcmpRouteSupport = False
      if hwConfig.ecmpNexthopIndex == hwConfig.defaultEcmpNexthopIndex:
         nexthopIndex = 0
         numPrefixBits = 0
      else:
         nexthopIndex = hwConfig.ecmpNexthopIndex
         numPrefixBits = hwConfig.numPrefixBitsForEcmpNexthopIndex

      ip6Model.nextHopIndex = nexthopIndex
      ip6Model.numPrefixBits = numPrefixBits
   else:
      ip6Model.v6EcmpRouteSupport = True

   return ( v6RoutingEnabled, ip6Model )

#-------------------------------------------------------------------------------
# The "show ipv6 interface [ intf | ipv6addr ]" command.
#-------------------------------------------------------------------------------
def ip6IntfShow( mode, intf ):
   if not intf.lookup():
      raise UnknownEntityError( "Interface %s does not exist" % intf.name )

   ip6Intf = IraIp6IntfCli.Ip6Intf( intf, mode.sysdbRoot,
                                createIfMissing=False )

   result = ip6Intf.show()
   result.name = intf.name
   result.mtu = intf.status().mtu
   ( result.lineProtocolStatus, result.interfaceStatus ) = intf.getStatus()
   return result

def ip6ConfiguredIntf( ifName ):
   if ifName in ip6.status.intf:
      return True
   ip6IntfConfig = ip6.config.intf.get( ifName )
   return ip6IntfConfig and ip6IntfConfig.ipv6Configured()

def filterIntfByVrf( mode, intfs, vrfName ):
   if vrfName is None or not vrfExists( vrfName ):
      raise UnknownEntityError( noIpv6RouteTableForVrfMsg % vrfName )

   return [ intf for intf in intfs if IraIp6IntfCli.Ip6Intf( intf, mode,
                 createIfMissing=False ).config().vrf == vrfName ]

def routedIntfsWithIp6Configured( mode, intf, vrfName=None ):
   """
   returns a list of lists "l3IntfInfoList" with two elements
   listoflist[0] includes L3 interfaces
   listoflist[1] includes L3 interfaces with ipv6 configured
   """
   intfs = IntfCli.Intf.getAll( mode, intf )
   if intfs is None:
      return None
   intfsL3 = [ i for i in intfs if i.routingCurrentlySupported() ]
   intfsIp6Configured = [ i for i in intfsL3 if ip6ConfiguredIntf( i.name ) ]
   if vrfName:
      intfsL3 = filterIntfByVrf( mode, intfsL3, vrfName )
      intfsIp6Configured = filterIntfByVrf( mode, intfsIp6Configured, vrfName )
   return [ intfsL3, intfsIp6Configured ]

#-------------------------------------------------------------------------------
# The "show ipv6 interface" command.
#-------------------------------------------------------------------------------
def showIp6InterfaceDetail( mode, intf, vrfName=None ):
   intfList = None
   l3IntfInfoList = routedIntfsWithIp6Configured( mode, intf, vrfName )
   result = Ip6Statuses()
   if l3IntfInfoList:
      intfList = l3IntfInfoList[ 0 ]
      for x in l3IntfInfoList[ 1 ]:
         result.interfaces[ x.name ] = ip6IntfShow( mode, x )


   # Print an error message if no interface got displayed.
   if not result.interfaces:
      if intfList:
         # There are ipv6 capable interfaces, but none has ipv6 configured.
         if intf:
            intfNames = list( intf.intfNames() )
            if len( intfNames ) == 1:
               msgStr = "IPv6 not configured on %s" % intfNames[ 0 ]
               if vrfName:
                  msgStr = " %s or %s doesn't exist in vrf %s" % \
                           ( msgStr, intfNames[ 0 ], vrfName )
            else:
               msgStr = "No IPv6 configured interfaces in range"
               if vrfName:
                  msgStr = " %s or the range doesn\'t exist in vrf %s" % \
                            ( msgStr, vrfName )
         else:
            msgStr = "No IPv6 configured interfaces"
            if vrfName:
               msgStr = " %s in vrf %s " % ( msgStr, vrfName )

      else:
         # no ipv6 capable interfaces
         if intf:
            intfNames = list( intf.intfNames() )
            if len( intfNames ) == 1:
               msgStr = "%s does not support IPv6" % intfNames[ 0 ]
               if vrfName:
                  msgStr = " %s or %s doesn\'t exist in vrf %s" % \
                           ( msgStr, intfNames[ 0 ], vrfName )
            else:
               msgStr = "No IPv6 capable interfaces in range"
               if vrfName:
                  msgStr = " %s or the range doesn\'t exist in vrf %s" % \
                           ( msgStr, vrfName )
         else:
            msgStr = "No IPv6 capable interfaces"
            if vrfName:
               msgStr = " %s in vrf %s " % ( msgStr, vrfName )
      mode.addError( msgStr )
   return result

#-------------------------------------------------------------------------------
# The "show ipv6 interface brief" command.
#-------------------------------------------------------------------------------
def showIp6InterfaceBrief( mode, args ):
   intf = args.get( 'INTFS' )
   vrfName = args.get( 'VRF' )
   briefModel = Ip6StatusesBrief()
   detailModel = showIp6InterfaceDetail( mode, intf, vrfName=vrfName )

   for intfDetail in detailModel.interfaces.itervalues():
      intfBrief = Ip6StatusBrief(
         name=intfDetail.name,
         mtu=intfDetail.mtu,
         interfaceStatus=intfDetail.interfaceStatus,
         lineProtocolStatus=intfDetail.lineProtocolStatus,
         linkLocal=intfDetail.linkLocal,
         addresses=intfDetail.addresses,
         addrSource=intfDetail.addrSource,
      )

      briefModel.interfaces[ intfBrief.name ] = intfBrief

   return briefModel

#-------------------------------------------------------------------------------
# Wrapper function for "show ipv6 interface [brief]"
#-------------------------------------------------------------------------------
def showIpv6IntfByAddr( mode, ipv6Addr=None, vrfName=None ):
   result = Ip6Statuses()
   if not ipv6Addr:
      return result
   for intfStatus in ip6.status.intf.itervalues():
      if vrfName != None and intfStatus.vrf != vrfName:
         continue
      for addrInfo in intfStatus.addr:
         if addrInfo.address == ipv6Addr:
            intfName = intfStatus.intfId
            intfType, _ = Intf.IntfRange.intfTypeFromName( intfName )
            intf = intfType.getCliIntf( mode, intfName )
            result.interfaces[ intfName ] = ip6IntfShow( mode, intf )
   return result

def showIpv6Intf( mode, args ):
   vrfName = args.get( 'VRF' )
   addr = args.get( 'ADDR' )
   if addr is not None:
      return showIpv6IntfByAddr( mode, ipv6Addr=addr, vrfName=vrfName )
   intfs = args.get( 'INTFS' )
   return showIp6InterfaceDetail( mode, intfs, vrfName=vrfName )

interfaceAfterShowMatcher = CliMatcher.KeywordMatcher(
   'interface',
   helpdesc='Details related to IPv6 interfaces' )

briefKwMatcher = CliMatcher.KeywordMatcher(
   'brief',
   helpdesc='Condensed output' )

vrfKwNode = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF instance' ),
   guard=ipV6Vrf )

vrfNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: allVrfConfig.vrf,
                                                'VRF name' )

class ShowIpv6IntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ipv6 interface
                [ INTFS | ADDR ]
                [ vrf VRF ]
            """
   data = { 'ipv6': ipv6MatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'Match this IPV6 Address' ),
            'vrf': vrfKwNode,
            'VRF': vrfNameMatcher
          }
   cliModel = Ip6Statuses

   handler = showIpv6Intf

BasicCli.addShowCommandClass( ShowIpv6IntfCmd )

class ShowIpv6IntfBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show ipv6 interface
               [ INTFS ]
               [ vrf VRF ]
               brief
            """
   data = { 'ipv6': ipv6MatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'vrf': vrfKwNode,
            'VRF': vrfNameMatcher,
            'brief': briefKwMatcher
          }
   cliModel = Ip6StatusesBrief

   handler = showIp6InterfaceBrief

BasicCli.addShowCommandClass( ShowIpv6IntfBriefCmd )

def warnIfRoutingDisabled( mode, vrfName=DEFAULT_VRF ):
   # OSPF and other protocols allow creating instances even before
   # corresponding VRF is defined. In such cases, vrf/config/<vrfname> may not
   # exist yet.
   if not ( vrfExists( vrfName ) and routing6.config( vrfName ).routing ):
      mode.addWarning( "IPv6 routing not enabled in VRF %s" % vrfName )

def warnIfRoutingDisabledGeneral( mode, vrfName=DEFAULT_VRF ):
   if mode.session_.outputFormat_ != 'json':
      if vrfName and not vrfExists( vrfName ):
         mode.addWarning( "IPv6 routing not enabled" )
      elif not routing6.config( vrfName ).routing:
         mode.addWarning( "IPv6 routing not enabled" )

#-------------------------------------------------------------------------------
# The "show ipv6 route host" command.
#-------------------------------------------------------------------------------
def showRouteHost( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName and not vrfExists( vrfName ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
         return None
   model = IraRouteCommon.showRouteHostCommon( mode, vrfName, quantify,
                        allVrfStatusLocal, routing6, allVrfConfig )
   return model

#-------------------------------------------------------------------------------
# The "show ipv6 route [ vrf <vrfName> ] [ <prefix> ] fec" command.
#-------------------------------------------------------------------------------
def showRouteFec( mode, prefix=None, vrfName=None ):

   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfFecRoutesModel = IraRouteCommon.VrfFecRoutesModel()
   if vrfName != ALL_VRF_NAME and not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      return vrfFecRoutesModel

   vrfs = []

   # If VRF name is given, we will validate it; 'all' is a special case
   # we will print info for the default VRF or all VRFs
   if vrfName == ALL_VRF_NAME:
      vrfs.append( DEFAULT_VRF )
      for vrf in sorted( allVrfStatusLocal.vrf ):
         vrfs.append( vrf )
   else:
      vrfs.append( vrfName )

   if prefix != None:
      # Ensure that I have an Ip6Prefix
      tn = prefix.tacType.fullTypeName
      if tn  in ( 'Arnet::Ip6Addr', 'Arnet::Ip6AddrWithMask' ):
         v = ( prefix.stringValue
               if tn == "Arnet::Ip6Addr"
               else prefix.subnet.stringValue )
         prefix = Arnet.Ip6Prefix( v )

   for vrf in vrfs:
      if vrfName == ALL_VRF_NAME:
         print "VRF: " + vrf
      vrfFecRoutesModel.vrfs[ vrf ] = routing6.showFecs( mode,
                                                         prefix=prefix,
                                                         vrfName=vrf )
      if not routing6.config( vrf ).routing:
         mode.addWarning( "IPv6 routing not enabled" )
   return vrfFecRoutesModel

#-------------------------------------------------------------------------------
# The "show ipv6 route fec summary" command.
#-------------------------------------------------------------------------------
def showIpFecSummary( mode, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   return routing6.showFecs( mode, vrfName=vrfName, summary=True )

#-------------------------------------------------------------------------------
# The "show ipv6 route" command.
#-------------------------------------------------------------------------------
routeAfterShowIpv6Matcher = CliMatcher.KeywordMatcher( 'route',
                                                 helpdesc='IPv6 routing table' )
ipv6PrefixMatcher = Ip6AddrMatcher.Ip6PrefixMatcher( 'Match this IPv6 prefix' )
ipv6AddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher( 'Match this IPv6 address' )
showIpv6StaticMatcher = CliMatcher.KeywordMatcher(
   'static',
   helpdesc='Show only static routes' )
detailAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'detail',
                                                             helpdesc='All routes')

sharedCountObj = object()

def _protocolMatcher( protocolName, helpdesc, alias='PROTOCOL' ):
   return CliCommand.Node( CliMatcher.KeywordMatcher(
                              protocolName, helpdesc=helpdesc ),
                           sharedMatchObj=sharedCountObj, maxMatches=1,
                           alias=alias )

class ProtocolCliExpression( CliCommand.CliExpression ):
   expression = "( bgp | connected | kernel | martian | " \
                "dynamic-policy | ospf | static | aggregate | isis | mpls | dhcp )"
   data = { 'bgp': _protocolMatcher( 'bgp', 'Show only BGP routes' ),
            'connected': _protocolMatcher( 'connected',
                                           'Show only connected routes' ),
            'kernel': _protocolMatcher( 'kernel', 'Show only kernel routes' ),
            'martian': _protocolMatcher( 'martian', 'Show only kernel routes' ),
            'dynamic-policy': _protocolMatcher( 'dynamic-policy',
                                                'Show only Dynamic Policy routes' ),
            'ospf': _protocolMatcher( 'ospf', 'Show only OSPF routes' ),
            'isis': _protocolMatcher( 'isis', 'Show only IS-IS routes' ),
            'static': _protocolMatcher( 'static', 'Show only static routes' ),
            'aggregate': _protocolMatcher( 'aggregate',
                                           'Show only aggregate routes' ),
            'mpls': _protocolMatcher( 'mpls', 'Show only static MPLS routes' ),
            'dhcp': _protocolMatcher( 'dhcp', 'Show only DHCP routes' )
          }

quantifyKwMatcher = CliMatcher.KeywordMatcher(
   'quantify',
   helpdesc='Print the time taken' )

class ShowIpv6RouteOptionalItemsCliExpr( CliCommand.CliExpression ):
   expression = '[ { detail | PROTOCOL | quantify } ]'
   data = { 'detail': CliCommand.Node(
               detailAfterShowIpv6RouteMatcher, maxMatches=1 ),
            'PROTOCOL': ProtocolCliExpression,
            'quantify': CliCommand.Node(
               quantifyKwMatcher, maxMatches=1, hidden=True )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      options = set()
      for token in ( 'detail', 'PROTOCOL', 'quantify' ):
         value = args.pop( token, None )
         if value is not None:
            options.add( ( token.lower(), value ) )
      args[ 'OPTIONS' ] = options

def showRoute( mode, prefix=None, optionsSet=None, vrfName=None,
               routeFilter=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfName == ALL_VRF_NAME and not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      return None

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()
   revision = mode.session_.requestedModelRevision()
   # Use old model only for revision 1 JSON output
   if mode.session_.outputFormat_ == 'json' and revision == 1:
      if vrfName == ALL_VRF_NAME:
         mode.addError( "'vrf all' option not supported in revision 1" )
         return None
      p = printer.initPrinter( fd, outputFormat, True )
      printer.start( p )
      showRoutePerVrf( p, mode, prefix, optionsSet, vrfName,
                       routeFilter=routeFilter )
      printer.end( p )
      printer.deinitPrinter( p )
      return Ip6Routes
   # For everything else, use revision 2 model
   else:
      p = printer.initPrinter( fd, outputFormat, True )
      printer.start( p )
      printer.startDict( p, "vrfs" )
      if vrfName == ALL_VRF_NAME:
         vrfs = sorted( v for v in allVrfStatusLocal.vrf )
         vrfs.insert( 0, 'default' )
      else:
         vrfs = [ vrfName ]
      for vrf in vrfs:
         # make sure we don't attempt to write if SIGINT happened outside the
         # immediateMode section (since RouteSorter handles those internally)
         #
         # Additionally, "VRF: ..." header is printed inside RouterSorter C++ code
         # for safety when pager exits, see BUG284545I
         TacSigint.check()
         printer.startDict( p, vrf )
         keepGoing = showRoutePerVrf( p, mode, prefix=prefix, optionsSet=optionsSet,
                                      vrfName=vrf, routeFilter=routeFilter )
         printer.endDict( p, vrf )
         if not keepGoing:
            break
      printer.endDict( p, "vrfs" )
      printer.end( p )
      printer.deinitPrinter( p )
      return VrfIp6Routes

def showRoutePerVrf( p, mode, prefix=None, optionsSet=None, vrfName=DEFAULT_VRF,
                     routeFilter=None ):
   optionsDict = dict( optionsSet or [] )
   detail = 'detail' in optionsDict
   protocol = optionsDict.get( 'protocol' )
   longerPrefixes = 'longerPrefixes' in optionsDict
   quantify = 'quantify' in optionsDict.itervalues()
   intf = None

   originalMask = True
   if prefix != None:
      # pass down to c++ file so it knows whether the user specified a mask,
      # or we appended a /128 for them.  Used to print "Routing entry for %s".
      # handle the printing in c++ with cliprint
      # Ensure I have an Ip6AddrWithMask and not an Ip6Addr
      if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
         originalMask = False
         prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   sys.stdout.flush()
   # create the outputFormat here, since we will not have access to mode later
   # down the stack, in routing.showRoute
   fmt = mode.session_.outputFormat()
   requestedRevision = mode.session_.requestedModelRevision()
   # In revision 3 of the VrfIp6Routes CAPI model (IraIp6Model.py) we moved
   # the Nexthop-Group submodel from the per-route level to the per-via level.
   # If revision 2 or below is requested, we "degrade" the JSON output by picking
   # the first Nexthop-Group that a route points to and putting it in the per-route
   # level Nexthop-Group submodel.
   nhgModelRevision = 3

   # revision 3 prints the nested tunnel representation for routes resolving over
   # tunnels. If a lower revision is requested, it will flatten the tunnel entries,
   # and print the final nexthop, interface and combined label stack
   flatTunnelRevision = 3

   json = mode.session_.outputFormatIsJson()
   degradeNhgModel = json and requestedRevision < nhgModelRevision
   flattenTunnelOverTunnel = json and requestedRevision < flatTunnelRevision

   keepGoing = routing6.showRoute(
      prefix, intf, protocol, detail, vrfName=vrfName,
      longerPrefixes=longerPrefixes, quantify=quantify, fmt=fmt,
      originalMask=originalMask, degradeNhgModel=degradeNhgModel,
      flattenTunnelOverTunnel=flattenTunnelOverTunnel, routeFilter=routeFilter )
   warnIfRoutingDisabledGeneral( mode, vrfName )
   sys.stdout.flush()
   return keepGoing

showRouteVrfNameExprFactory = VrfExprFactory( helpdesc='Show routes in a VRF',
                                              inclDefaultVrf=True )

class ShowIpv6RouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show ipv6 route [ VRF ] [ PREFIX | ADDR ]
               [ OPTIONS ] [ NEXTHOP ]"""

   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': ipv6AddrMatcher,
            'OPTIONS': ShowIpv6RouteOptionalItemsCliExpr,
            'NEXTHOP': NexthopCliExpression,
          }

   cliModel = VrfIp6Routes

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   @staticmethod
   def handler( mode, args ):
      return showRoute( mode, prefix=args.get( 'PREFIX' ),
                        optionsSet=args[ 'OPTIONS' ],
                        vrfName=args.get( 'VRF' ),
                        routeFilter=args.get( 'nexthop-matchers' ) )

BasicCli.addShowCommandClass( ShowIpv6RouteCmd )

hostAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'host',
                                     helpdesc='Show only Host routes' )

fecAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'fec',
                                       helpdesc='show route adjacencies' )

summaryAfterShowIpv6FecMatcher = CliMatcher.KeywordMatcher( 'summary',
                                   helpdesc='show adjacency size and distribution' )

class ShowIpv6RouteHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] host [ quantify ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'host': hostAfterShowIpv6RouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = VrfIp6RoutesHost

   @staticmethod
   def handler( mode, args ):
      quantify = True if 'quantify' in args else False
      return showRouteHost( mode, args.get( 'VRF' ), quantify=quantify )

BasicCli.addShowCommandClass( ShowIpv6RouteHostCmd )

class ShowIpv6RouteFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] [ PREFIX | ADDR ] fec'
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': ipv6AddrMatcher,
            'fec': fecAfterShowIpv6RouteMatcher
          }

   cliModel = IraRouteCommon.VrfFecRoutesModel

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   @staticmethod
   def handler( mode, args ):
      return showRouteFec( mode, prefix=args.get( 'PREFIX' ),
                           vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIpv6RouteFecCmd )

class ShowIpv6FecSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] fec summary'
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'fec': fecAfterShowIpv6RouteMatcher,
            'summary': summaryAfterShowIpv6FecMatcher
            }
   cliModel = IraRouteCommon.FecSummaryModel

   @staticmethod
   def handler( mode, args ):
      return showIpFecSummary( mode, vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIpv6FecSummaryCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route static not-installed bfd" command
#-------------------------------------------------------------------------------

def showV6NotInstalledBfd( mode, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   if not ( vrfName is None or vrfName == ALL_VRF_NAME or vrfExists( vrfName ) ):
      mode.addError( "IPv6 routing table %s does not exist." % vrfName )
      return None

   vrfs = []
   if vrfName == ALL_VRF_NAME:
      vrfs = [ DEFAULT_VRF ] + sorted( allVrfStatusLocal.vrf )
   elif vrfName is not None:
      vrfs.append( vrfName )
   else:
      vrfs.append( DEFAULT_VRF )

   return IraCommonCli.showCommonNotInstalledBfd( vrfs=vrfs, v4=False )

class ShowIpv6NotInstalledBfdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] static not-installed bfd'
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'static': showIpv6StaticMatcher,
            'not-installed': IraCommonCli.notInstalledMatcher,
            'bfd': IraCommonCli.bfdMatcher
          }
   cliModel = VrfNotInstalledBfd

   @staticmethod
   def handler( mode, args ):
      return showV6NotInstalledBfd( mode, vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIpv6NotInstalledBfdCmd )

def vrfSummaryCommon( ip6RouteSummaryModel, vrfs ):
   routeCountByTypeDict = { 'connected':0,
                            'static':0,
                            'staticPersistent':0,
                            'staticNonPersistent':0,
                            'vcs':0,
                            'staticNexthopGroup':0,
                            'internal':0,
                            'attached':0,
                            'dhcp':0,
                            'dynamicPolicy':0,
                            'aggregate':0 }
   ospfBgpIsisSpecific = { 'ospfTotal':0,
                           'ospfIntraArea':0,
                           'ospfInterArea':0,
                           'ospfExternal1':0,
                           'ospfExternal2':0,
                           'nssaExternal1':0,
                           'nssaExternal2':0,
                           'bgpTotal':0,
                           'bgpInternal':0,
                           'bgpExternal':0,
                           'bgpLocal': 0,
                           'isisTotal':0,
                           'isisLevel1':0,
                           'isisLevel2':0 }

   for vrfName in vrfs:
      # Force mount routing.config as it needs to be
      # passed to Route6Summary and was lazy-mounted earlier.
      # Tac.newInstance does not like unmounted proxies
      # pylint: disable=protected-access
      if isinstance( routing6.config( vrfName ), LazyMount._Proxy ):
         LazyMount.force( routing6.config( vrfName ) )
      if isinstance( routing6.routeConfig( vrfName ), LazyMount._Proxy ):
         LazyMount.force( routing6.routeConfig( vrfName ) )
      route6Summary = Tac.newInstance( "Ira::Route6Summary",
                                   routing6.config( vrfName ),
                                   routing6.routeConfig( vrfName ),
                                   routing6.rStatus( vrfName ),
                                   routing6.attachRouteStatus( vrfName ),
                                   routing6.vrfIdMap(),
                                   vrfName )

      route6Summary.loadInternalStructures()
      route6Summary.loadAttachedRoutes()

      vrfMaskLen = dict( route6Summary.routeCountByMask )
      for key, value in vrfMaskLen.iteritems():
         if key not in ip6RouteSummaryModel.maskLen:
            ip6RouteSummaryModel.maskLen[key] = value
         else:
            ip6RouteSummaryModel.maskLen[key] += value

      fibSummaryTable = route6Summary.routeCountByType
      for source in routeCountByTypeDict:
         if source in fibSummaryTable:
            routeCountByTypeDict[ source ] += fibSummaryTable[ source ]

      # This is done to reflect the special case of 'fe80::/10' route which
      # is listed as connnected route in 'show ipv6 route detail' but not
      # as receive route in 'show ipv6 route host' even though its
      # routeType is receive
      if Arnet.Ip6Prefix( 'fe80::/10' ) in \
            routing6.rStatus( vrfName=vrfName ).route.keys():
         routeCountByTypeDict[ 'connected' ] += 1
         routeCountByTypeDict[ 'internal' ] -= 1

      for source in ospfBgpIsisSpecific:
         if source in fibSummaryTable:
            ospfBgpIsisSpecific[ source ] += fibSummaryTable[ source ]

      ip6RouteSummaryModel.totalRoutes += route6Summary.totalNumRoutes

   ip6RouteSummaryModel.connected = routeCountByTypeDict[ 'connected' ]
   ip6RouteSummaryModel.static = routeCountByTypeDict[ 'static' ]
   ip6RouteSummaryModel.staticPersistent = routeCountByTypeDict[ 'staticPersistent' ]
   ip6RouteSummaryModel.staticNonPersistent = \
         routeCountByTypeDict[ 'staticNonPersistent' ]
   ip6RouteSummaryModel.vcs = routeCountByTypeDict[ 'vcs' ]
   ip6RouteSummaryModel.staticNexthopGroup = \
       routeCountByTypeDict[ 'staticNexthopGroup' ]
   ip6RouteSummaryModel.internal = routeCountByTypeDict[ 'internal' ]
   ip6RouteSummaryModel.attached = routeCountByTypeDict[ 'attached' ]
   ip6RouteSummaryModel.aggregate = routeCountByTypeDict[ 'aggregate' ]
   ip6RouteSummaryModel.dhcp = routeCountByTypeDict[ 'dhcp' ]
   ip6RouteSummaryModel.dynamicPolicy = routeCountByTypeDict[ 'dynamicPolicy' ]

   ospfTotalsModel = ospfTotals6()
   ospfTotalsModel.ospfTotal = ospfBgpIsisSpecific[ 'ospfTotal' ]
   ospfTotalsModel.ospfIntraArea = ospfBgpIsisSpecific[ 'ospfIntraArea' ]
   ospfTotalsModel.ospfInterArea = ospfBgpIsisSpecific[ 'ospfInterArea' ]
   ospfTotalsModel.ospfExternal1 = ospfBgpIsisSpecific[ 'ospfExternal1' ]
   ospfTotalsModel.ospfExternal2 = ospfBgpIsisSpecific[ 'ospfExternal2' ]
   ospfTotalsModel.nssaExternal1 = ospfBgpIsisSpecific[ 'nssaExternal1' ]
   ospfTotalsModel.nssaExternal2 = ospfBgpIsisSpecific[ 'nssaExternal2' ]
   ip6RouteSummaryModel.ospfCounts6 = ospfTotalsModel

   bgpTotalsModel = bgpTotals6()
   bgpTotalsModel.bgpTotal = ospfBgpIsisSpecific[ 'bgpTotal' ]
   bgpTotalsModel.bgpExternal = ospfBgpIsisSpecific[ 'bgpExternal' ]
   bgpTotalsModel.bgpInternal = ospfBgpIsisSpecific[ 'bgpInternal' ]
   bgpTotalsModel.bgpLocal = ospfBgpIsisSpecific[ 'bgpLocal' ]
   ip6RouteSummaryModel.bgpCounts6 = bgpTotalsModel

   isisTotalsModel = isisTotals6()
   isisTotalsModel.isisTotal = ospfBgpIsisSpecific[ 'isisTotal' ]
   isisTotalsModel.isisLevel1 = ospfBgpIsisSpecific[ 'isisLevel1' ]
   isisTotalsModel.isisLevel2 = ospfBgpIsisSpecific[ 'isisLevel2' ]
   ip6RouteSummaryModel.isisCounts6 = isisTotalsModel
   return ip6RouteSummaryModel

#-------------------------------------------------------------------------------
# The "show ipv6 route [vrf <vrfname>] summary" command.
#-------------------------------------------------------------------------------
def showRouteSummary( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfsToShow = []

   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
         return None

   if vrfName == ALL_VRF_NAME:
      vrfsToShow = [ "default" ]
      vrfsToShow.extend( sorted( allVrfStatusLocal.vrf ) )
   else:
      vrfsToShow = [ vrfName ]

   svcRoutingProtoModelStatus = ServiceRoutingProtoModelStatus(
      operatingProtoModel = l3Config.protocolAgentModel,
      configuredProtoModel = l3Config.configuredProtocolAgentModel )
   ip6Summary = Ip6RouteSummary()

   ip6Summary.protoModelStatus = svcRoutingProtoModelStatus
   for vrf in vrfsToShow:
      result = showRouteSummaryForVrf( mode, vrfName=vrf,
                                       quantify=quantify )
      if result:
         ip6Summary.vrfs[ vrf ] = result

   return ip6Summary

def showRouteSummaryForVrf( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      return None

   beforeSum = Tac.now()
   ip6RouteSummaryModel = Ip6RouteSummaryForVrf()
   # pylint: disable=protected-access
   vrfs = [ vrfName ]

   ip6RouteSummaryModel.totalRoutes = 0
   ip6RouteSummaryModel = vrfSummaryCommon( ip6RouteSummaryModel, vrfs )
   if quantify:
      afterSum = Tac.now()
      ip6RouteSummaryModel._quantify = afterSum-beforeSum
   return ip6RouteSummaryModel

summaryAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'summary',
                                     helpdesc='IPv6 routing table summary' )

class ShowIpv6RouteSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] summary [ quantify ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'summary': summaryAfterShowIpv6RouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = Ip6RouteSummary

   @staticmethod
   def handler( mode, args ):
      quantify = True if 'quantify' in args else False
      return showRouteSummary( mode, vrfName=args.get( 'VRF' ),
                               quantify=quantify )

BasicCli.addShowCommandClass( ShowIpv6RouteSummaryCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route (prefix) longer-prefixes" command.
#-------------------------------------------------------------------------------

def showRouteLongerPrefixes( mode, args ):
   # pylint check fails if appended directly to set
   prefix = args[ 'PREFIX' ]
   optionsSet = args[ 'OPTIONS' ]
   vrfName=args.get( 'VRF' )
   optionsDict = dict( optionsSet or [] )
   optionsDict[ 'longerPrefixes' ] = 'longer-prefixes'
   optionsSet = optionsDict.items()
   return showRoute( mode, prefix, optionsSet, vrfName )

class ShowIpv6RouteLongerPrefixesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show ipv6 route [ VRF ] ( PREFIX | ADDR )
               longer-prefixes [ OPTIONS ]
            """
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': CliCommand.Node( ipv6AddrMatcher, alias='PREFIX' ),
            'longer-prefixes': 'Match longer-prefixes',
            'OPTIONS': ShowIpv6RouteOptionalItemsCliExpr
          }
   cliModel = VrfIp6Routes

   handler = showRouteLongerPrefixes

BasicCli.addShowCommandClass( ShowIpv6RouteLongerPrefixesCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route vrf all summary brief " command.
#-------------------------------------------------------------------------------
def showRouteVrfAllSummaryBrief( mode, quantify=False ):
   beforeSum = Tac.now()
   vrfs = []

   for vrf in sorted( allVrfStatusLocal.vrf ):
      vrfs.append( vrf )
   vrfs.insert ( 0, DEFAULT_VRF )

   ip6RouteSummaryBriefModel = Ip6RouteSummaryForVrf()
   ip6RouteSummaryBriefModel.maskLen = {}
   ip6RouteSummaryBriefModel.totalRoutes = 0
   ip6RouteSummaryBriefModel.vrfCount = len( vrfs )
   ip6RouteSummaryBriefModel = vrfSummaryCommon( ip6RouteSummaryBriefModel, \
            vrfs )

   # pylint: disable=protected-access
   if quantify:
      afterSum = Tac.now()
      ip6RouteSummaryBriefModel._quantify = afterSum-beforeSum
   return ip6RouteSummaryBriefModel

allKwMatcher = CliMatcher.KeywordMatcher( ALL_VRF_NAME,
                                          helpdesc='Show summary of all vrfs' )

vrfForShowMatcher = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Specify the vrf' )

class ShowIpv6RouteSummaryBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route vrf all summary brief [ quantify ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'vrf': vrfForShowMatcher,
            'all': allKwMatcher,
            'summary': summaryAfterShowIpv6RouteMatcher,
            'brief': briefKwMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = Ip6RouteSummaryForVrf

   @staticmethod
   def handler( mode, args ):
      quantify = 'quantify' in args
      return showRouteVrfAllSummaryBrief( mode, quantify=quantify )

BasicCli.addShowCommandClass( ShowIpv6RouteSummaryBriefCmd )

#-------------------------------------------------------------------------------
#[no|default] ipv6 hardware fib aggregate-address <pfx> summary-only software-forward
#-------------------------------------------------------------------------------
def noipv6AggPrefix( mode, prefix ):
   if routing6.hwConfig( vrfName=DEFAULT_VRF ).aggregateRoute.has_key(
         Arnet.Ip6Prefix(prefix.stringValue) ):
      del routing6.hwConfig( vrfName=DEFAULT_VRF ).\
          aggregateRoute[ Arnet.Ip6Prefix(prefix.stringValue) ]

def doipv6AggPrefix( mode, prefix ):
   routing6.hwConfig( vrfName=DEFAULT_VRF ).aggregateRoute[
         Arnet.Ip6Prefix(prefix.stringValue) ] = True

summaryOnlyKwMatcher = CliMatcher.KeywordMatcher( 'summary-only',
      helpdesc='Filters all more-specific routes from hardware routing table' )
softwareForwardMatcherForConfig = CliMatcher.KeywordMatcher( 'software-forward',
      helpdesc='Software Forward the packets' )
aggPrefixForConfigMatcher = CliMatcher.KeywordMatcher( 'aggregate-address',
   helpdesc='Configure aggregate address' )

class Ipv6FibAggregateAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 hardware fib aggregate-address PREFIX ' \
            'summary-only software-forward'
   noOrDefaultSyntax = 'ipv6 hardware fib aggregate-address PREFIX ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'aggregate-address': aggPrefixForConfigMatcher,
            'PREFIX': ipv6PrefixMatcher,
            'summary-only': summaryOnlyKwMatcher,
            'software-forward': softwareForwardMatcherForConfig
          }

   @staticmethod
   def handler( mode, args ):
      doipv6AggPrefix( mode, args[ 'PREFIX' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noipv6AggPrefix( mode, args[ 'PREFIX' ] )

configMode.addCommandClass( Ipv6FibAggregateAddrCmd )

#-------------------------------------------------------------------------------
#show ipv6 hardware fib aggregate-address [<pfx>] [software-forward]
#-------------------------------------------------------------------------------
def showHardwareAggregateRoute( mode, prefix=None ):
   if prefix:
      # Ensure I have an Ip6AddrWithMask and not an Ip6Addr
      if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
         prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   routing6.showHwAggregateRoute( mode, prefix )

softwareForwardForShowMatcher = CliMatcher.KeywordMatcher( 'software-forward',
      helpdesc='Software Forwarded packets' )
aggPrefixForShowMatcher = CliMatcher.KeywordMatcher( 'aggregate-address',
   helpdesc='Display aggregate address' )

class ShowIpv6FibAggregateAddrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib aggregate-address ' \
            '[ PREFIX | ADDR ] [ software-forward ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'hardware': CliToken.Hardware.hardwareForShowMatcher,
            'fib': IraCommonCli.fibMatcher,
            'aggregate-address': aggPrefixForShowMatcher,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': ipv6AddrMatcher,
            'software-forward': softwareForwardForShowMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   @staticmethod
   def handler( mode, args ):
      showHardwareAggregateRoute( mode, prefix=args.get( 'PREFIX' ) )

BasicCli.addShowCommandClass( ShowIpv6FibAggregateAddrCmd )

#-------------------------------------------------------------------------------
# [no|default] ipv6 hardware fib optimize prefix-length <length>
#-------------------------------------------------------------------------------
def routesV6OptimizeGuard( mode, token ):
   if routing6HardwareStatus.v6RemSupportedPrefixLength:
      return None
   return CliParser.guardNotThisPlatform

optimizeMatcher = CliCommand.Node( CliMatcher.KeywordMatcher( 'optimize',
                                      helpdesc='Optimize IPv6 routes' ),
                                   guard=routesV6OptimizeGuard )

def prefixV6LengthGuard( mode, token ):
   if int( token ) not in routing6HardwareStatus.v6RemSupportedPrefixLength:
      return CliParser.guardNotThisPlatform
   return None

prefixLengthData = {}
for prefixLen in range( 0, 129 ):
   prefixLengthData[ str( prefixLen ) ] = 'Prefix length %d' % prefixLen

restartL3AgentWarnStr = "Please restart layer 3 forwarding agent to ensure IPv6 " \
                        "routes are "

def optimizeV6Routes( mode, prefixLenSet=None, profileType='', enable=True ):
   configuredLengths = map( int, prefixLenSet or [] )
   supportedLengths = routing6HardwareStatus.v6RemSupportedPrefixLength.keys()
   startupConfig = mode.session_.startupConfig()
   if not startupConfig and len( configuredLengths ) > \
          routing6HardwareStatus.v6RemSupportedPrefixCount:
      mode.addError( "This platform allows optimization of %d ipv6 prefix lengths" \
                        %  routing6HardwareStatus.v6RemSupportedPrefixCount )
      return

   warn = False

   # Disable optimization with profiles if enabled
   if routing6HardwareConfig.optimizeProfileName:
      routing6HardwareConfig.optimizeProfileName = ''
      warn = True

   if enable:
      for length in configuredLengths:
         if not startupConfig and length not in supportedLengths:
            continue
         if length not in routing6HardwareConfig.prefixLenV6InExactMatch:
            # prefix length not present in configured lengths collection
            warn = True
         elif not routing6HardwareConfig.prefixLenV6InExactMatch[ length ]:
            # prefix length is present in configured lengths collection
            # but is set to False
            warn = True
         routing6HardwareConfig.prefixLenV6InExactMatch[ length ] = True

      for length in routing6HardwareConfig.prefixLenV6InExactMatch:
         if routing6HardwareConfig.prefixLenV6InExactMatch[ length ] and \
                length not in configuredLengths:
            # remove previously configured prefix lengths
            del routing6HardwareConfig.prefixLenV6InExactMatch[ length ]
            warn = True
   else:
      for length in routing6HardwareConfig.prefixLenV6InExactMatch:
         if routing6HardwareConfig.prefixLenV6InExactMatch[ length ]:
            warn = True
         del routing6HardwareConfig.prefixLenV6InExactMatch[ length ]
   if warn:
      if enable:
         mode.addWarning( restartL3AgentWarnStr + "optimized" )
      else:
         mode.addWarning( restartL3AgentWarnStr + "not optimized" )

def enableV6RoutesOptimization( mode, prefixLenSet ):
   optimizeV6Routes( mode, prefixLenSet=prefixLenSet, enable=True )

def disableV6RoutesOptimization( mode, prefixLenSet=None ):
   optimizeV6Routes( mode, prefixLenSet=[], enable=False )

class OptimizeV6RoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 hardware fib optimize prefix-length LENGTH'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'optimize': optimizeMatcher,
            'prefix-length': IraCommonCli.prefixLengthMatcher,
            'LENGTH': CliCommand.SetEnumMatcher( prefixLengthData,
               guard=prefixV6LengthGuard )
            }

   @staticmethod
   def handler( mode, args ):
      enableV6RoutesOptimization( mode, prefixLenSet=args[ 'LENGTH' ] )

configMode.addCommandClass( OptimizeV6RoutesCmd )

# The noOrDefault command has been moved down

#-------------------------------------------------------------------------------
# [no|default] ipv6 hardware fib optimize prefixes profile internet
#-------------------------------------------------------------------------------

def setOptimizeV6Prefixes( mode, profileType='', enable=True ):
   if enable:
      # If the profileType is not set, then add
      if profileType and profileType != routing6HardwareConfig.optimizeProfileName:
         # Delete any previously set expansion
         routing6HardwareConfig.prefixLenV6InExactMatch.clear()
         # Set the profileType in optimizeProfileName
         routing6HardwareConfig.optimizeProfileName = profileType
         mode.addWarning( restartL3AgentWarnStr + "optimized" )

   else:
      disableV6RoutesOptimization( mode )

def enableOptimizeV6Prefixes( mode, profileType ):
   setOptimizeV6Prefixes( mode, profileType, enable=True )

def disableOptimizeV6Prefixes( mode ):
   setOptimizeV6Prefixes( mode, enable=False )

def supportedProfileType( mode ):
   supportedProfiles = {}
   # Today, we have profiles just for Sand platforms
   # Add condition and path if this is supported on other platforms in future
   supportedProfiles[ optimizeProfileType.internet ] = \
         'Optimize prefixes for Internet'
   return supportedProfiles

class Ipv6OptimizePrefixesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 hardware fib optimize prefixes profile PROFILE'

   noOrDefaultSyntax = 'ipv6 hardware fib optimize ...'

   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'optimize': optimizeMatcher,
            'prefixes': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'prefixes',
                        helpdesc='Optimize prefixes' ),
                  guard=routesV6OptimizeGuard ),
            'profile': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'profile',
                        helpdesc='Optimize based on profiles' ),
               guard=routesV6OptimizeGuard ),
            'PROFILE': CliMatcher.DynamicKeywordMatcher( supportedProfileType )
          }

   @staticmethod
   def handler( mode, args ):
      enableOptimizeV6Prefixes( mode, args[ 'PROFILE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      disableOptimizeV6Prefixes( mode )

configMode.addCommandClass( Ipv6OptimizePrefixesCmd )

#-------------------------------------------------------------------------------
# Determine if assigning the given address/mask to some interface in Ira
# should be allowed
# return a pair of a boolean with a string, where the string is an error or
# warning message
#-------------------------------------------------------------------------------
def canSetIntfIp( intfName, ipAddrWithMask, mode=None ):
   # skip IPv6 address sanity check only during config replace
   if mode and mode.session_.skipConfigCheck():
      return [True, ""]
   # This gets the right vrf route config based on the intf
   config = routing6.routeConfig( vrfName=None, intfName=intfName )
   result = IraRouteCommon.canSetIntfIpHelper( config, intfName, ipAddrWithMask,
                                               'ipv6', False )
   if result[ 1 ]:
      return result
   config = routing4.routeConfig( vrfName=None, intfName=intfName )
   if config is None:
      return result
   return IraRouteCommon.canSetIntfIpHelper( config, intfName, ipAddrWithMask,
                                             'ipv6', True )

#-------------------------------------------------------------------------------
# The Ipv6 Resilient Ecmp command
# The syntax of the command is:
#  ipv6 hardware fib ecmp <prefix> capacity <C> redundancy <R>
#  no/default ipv6 hardware fib ecmp <prefix>
#  no/default ipv6 hardware fib ecmp
#-------------------------------------------------------------------------------

def maxResilientEcmpIp6CapacityRangeFn( mode ):
   if routing6HardwareStatus is None or routing6HardwareStatus.maxEcmp == 0:
      return( 1, sys.maxint )
   else :
      return( 2, routing6HardwareStatus.maxEcmp )

def maxResilientEcmpIp6RedundancyRangeFn( mode ):
   if routing6HardwareStatus is None or routing6HardwareStatus.maxResilientEcmp == 0:
      return( 1, sys.maxint )
   else :
      return( 1, routing6HardwareStatus.maxResilientEcmp / 2 )

def resilientEcmpIp6Guard( mode, token ):
   if routing6HardwareStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

def resilientEcmpIp6PrefixIs( mode, prefix, capacity, redundancy ) :

   # if C * R > maxResilientEcmp value, throw max value error
   if ( routing6HardwareStatus.maxResilientEcmp and
        ( capacity * redundancy ) > routing6HardwareStatus.maxResilientEcmp ):
      mode.addError( "Maximum value supported for capacity * redundancy is %d"
          % ( routing6HardwareStatus.maxResilientEcmp ) )
      return

   ecmpConfig = Tac.Value(
         "Routing::Hardware::ResilientEcmpConfig", capacity, redundancy )
   genPrefix = Tac.Value( "Arnet::IpGenPrefix" )
   genPrefix.handleV6Prefix( prefix.subnet )
   routing6HardwareConfig.resilientEcmpPrefix[ genPrefix ] = ecmpConfig

def noResilientEcmpIp6( mode , prefix=None ):

   # if there is a prefix, delete prefix, otherwise delete all
   if prefix is None:
      routing6HardwareConfig.resilientEcmpPrefix.clear()
      return

   genPrefix = Tac.Value( "Arnet::IpGenPrefix" )
   genPrefix.handleV6Prefix( prefix.subnet )
   del routing6HardwareConfig.resilientEcmpPrefix[ genPrefix ]

class Ipv6ResilientEcmpCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 hardware fib ecmp resilience PREFIX capacity ' \
            'CAPACITY redundancy REDUNDANCY'
   noOrDefaultSyntax = 'ipv6 hardware fib ecmp resilience [ PREFIX ] ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'ecmp': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'ecmp',
                                       helpdesc='Configure ECMP routes'),
                                     guard=resilientEcmpIp6Guard ),
            'resilience': IraCommonCli.resilienceMatcher,
            'PREFIX': ipv6PrefixMatcher,
            'capacity': IraCommonCli.resilientEcmpCapacityMatcher,
            'CAPACITY': CliMatcher.DynamicIntegerMatcher(
                              rangeFn=maxResilientEcmpIp6CapacityRangeFn,
                              helpdesc='capacity value' ),
            'redundancy': IraCommonCli.resilientEcmpRedundancyMatcher,
            'REDUNDANCY': CliMatcher.DynamicIntegerMatcher(
                              rangeFn=maxResilientEcmpIp6RedundancyRangeFn,
                              helpdesc='redundancy value' )
          }

   @staticmethod
   def handler( mode, args ):
      resilientEcmpIp6PrefixIs( mode, args[ 'PREFIX' ], args[ 'CAPACITY' ],
                                args[ 'REDUNDANCY' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noResilientEcmpIp6( mode , prefix=args.get( 'PREFIX' ) )

configMode.addCommandClass( Ipv6ResilientEcmpCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route rpf unicast [ prefix ]" command.
#-------------------------------------------------------------------------------
def v6ResolvedUrpfStatus( intfId, v6Mode ):
   resolvedMode = v6Mode
   if not routingHardwareStatus.urpfSupportedOnV4AndV6:
      if intfId in ip4Config.ipIntfConfig:
         v4Mode = ip4Config.ipIntfConfig[ intfId ].urpf.mode
         if v4Mode != 'disable' and v6Mode != v4Mode:
            # v4 and v6 mode in conflict v6 to be disabled by platform
            resolvedMode = 'disable'
   return resolvedMode

def populateIpv6UrpfModel( mode, intfs=None ):
   uRpfModel = Ip6UrpfInterfaces()
   uRpfModel.viaDefaultRoute = False
   for intf in intfs:
      if not intf.name in ip6.config.intf:
         continue
      v6Mode = ip6.config.intf[ intf.name ].urpf.mode
      resolvedMode = v6ResolvedUrpfStatus( intf.name, v6Mode )
      if resolvedMode != 'disable':
         uRpfIntf = Ip6UrpfInterface()
         uRpfIntf.uRpfMode = resolvedMode
         uRpfIntf.allowDefault = uRpfIntf.uRpfMode == 'strictDefault'
         uRpfModel.interfaces[ intf.name ] = uRpfIntf
   return uRpfModel

def populateIpv6UrpfPrefixModel( mode, prefix, vrfName ):
   uRpfModel = Ip6UrpfInterfaces()
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      uRpfModel.interfaces = None
      return uRpfModel

   # Ensure I have an Ip6AddrWithMask and not an Ip6Addr
   if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
      prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   routeEntry = routing6.route( prefix, None, vrfName=vrfName )
   if not routeEntry:
      uRpfModel.interfaces = None
      return uRpfModel

   uRpfModel.prefix = prefix.stringValue
   uRpfModel.viaDefaultRoute = \
            routeEntry.prefix == routing6.ip.defaultRoute

   if vrfName == DEFAULT_VRF:
      fec = routing6.f6Status( 'default' ).fec.get( routeEntry.fecId )
   else:
      fec = routing6.f6Status( vrfName ).fec.get( routeEntry.fecId )

   for via in fec.via.values():
      intf = via.intfId
      if not intf in ip6.config.intf:
         v6Mode = 'disable'
      else:
         v6Mode = ip6.config.intf[ intf ].urpf.mode

      resolvedMode = v6ResolvedUrpfStatus( intf, v6Mode )
      uRpfIntf = Ip6UrpfInterface()
      uRpfIntf.uRpfMode = resolvedMode
      uRpfIntf.allowDefault = resolvedMode == 'strictDefault'
      uRpfModel.interfaces[ intf ] = uRpfIntf
   return uRpfModel

def showIpv6Urpf( mode, vrfName=None, prefix=None ):

   # Get all routeable intefaces
   intfs = routedIntfsWithIp6Configured( mode, None, vrfName )

   # Use the ipv6 one which is in intfs[ 1 ]
   if prefix is None:
      return populateIpv6UrpfModel( mode, intfs[ 1 ] )
   else:
      return populateIpv6UrpfPrefixModel( mode, prefix, vrfName )

class ShowIpv6UrpfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route rpf unicast [ VRF ] [ PREFIX | ADDR ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'rpf': 'Reverse Path Forwarding information',
            'unicast': 'Unicast Reverse Path Forwarding information',
            'VRF': VrfExprFactory( helpdesc='Show Unicast Reverse Path '
                                            'Forwarding information in a VRF',
                                   guard=ipV6Vrf,
                                   inclDefaultVrf=True ),
            'PREFIX': Ip6AddrMatcher.Ip6PrefixMatcher( 'Destination prefix' ),
            'ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'Destination address' )
          }

   cliModel = Ip6UrpfInterfaces

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   @staticmethod
   def handler( mode, args ):
      return showIpv6Urpf( mode, vrfName=args.get( 'VRF' ),
                           prefix=args.get( 'PREFIX' ) )

BasicCli.addShowCommandClass( ShowIpv6UrpfCmd )

myEntManager = None

def fibRoutingStatus( self, vrfName=DEFAULT_VRF ):
   if vrfName == DEFAULT_VRF:
      return SmashLazyMount.mount( myEntManager,
                                   'routing6/status',
                                   'Smash::Fib6::RouteStatus',
                                   SmashLazyMount.mountInfo( 'reader' ) )
   else:
      return SmashLazyMount.mount( myEntManager,
                                   'routing6/vrf/status/%s' % vrfName,
                                   'Smash::Fib6::RouteStatus',
                                   SmashLazyMount.mountInfo( 'reader' ) )

class VerifyRouteCmd( CliCommand.CliCommandClass ):
   syntax = 'verify ipv6 route ( all | ( PREFIX [ VRF ] ) )'
   data = {
         'verify': CliCommand.Node(
                     CliMatcher.KeywordMatcher( 'verify',
                           helpdesc='Verify the state of an object in the system' ),
                     hidden=True ),
         'ipv6': 'Verify the state of an IPv6 object',
         'route': 'Verify the state of a route',
         'all': 'Verify the state of all the routes in the system',
         'PREFIX': ipv6PrefixMatcher,
         'VRF': VrfExprFactory( helpdesc='Verify in a VRF' ),
         }

   @staticmethod
   def handler( mode, args ):
      actors = [ 'KernelFib' ]
      routes = []
      if 'all' in args:
         for vrf in [ DEFAULT_VRF ] + allVrfConfig.vrf.keys():
            for route in fibRoutingStatus( vrf ).route:
               routes.append( ( route.stringValue, vrf ) )
      else:
         pfx = str( args[ 'PREFIX' ] )
         vrf = args.get( 'VRF', DEFAULT_VRF )
         routes.append( ( pfx, vrf ) )
      # If total number of routes exceeds VERIFY_MAX_ROUTES limit, skip the
      # verification
      if len( routes ) > VERIFY_MAX_ROUTES:
         print "Skipping verification as total number of routes exceeds %d" \
               % VERIFY_MAX_ROUTES
      else:
         verify( 'Route', routes, actors, myEntManager )

BasicCli.EnableMode.addCommandClass( VerifyRouteCmd )

#-------------------------------------------------------------------------------
# The "ipv6 fib compression redundant-specifics filter" and
# "[no|default] ip fib compression redundant-specifics filter"
# commands.
#-------------------------------------------------------------------------------
def fibSuppressionSupportedGuard( mode, token ):
   rhs = routing6HardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing6' ][ 'hardware' ][ 'status' ]
   if rhs.fibSuppressionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

compressionMatcher = CliCommand.Node( CliMatcher.KeywordMatcher( 'compression',
                                          helpdesc='FIB compression' ),
                                      guard=fibSuppressionSupportedGuard )

def setFibFilter( mode, enable=True ):
   routing6HardwareConfig.fibSuppression = enable

def noSetFibFilter( mode ):
   setFibFilter( mode, enable=False )

class Ipv6FibFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 fib compression redundant-specifics filter'
   noOrDefaultSyntax = 'ipv6 fib compression redundant-specifics filter'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'fib': IraCommonCli.fibMatcher,
            'compression': compressionMatcher,
            'redundant-specifics': IraCommonCli.redundantSpecificMatcher,
            'filter': IraCommonCli.filterMatcher }

   @staticmethod
   def handler( mode, args ):
      setFibFilter( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noSetFibFilter( mode )

configMode.addCommandClass( Ipv6FibFilterCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 user route" command
#-------------------------------------------------------------------------------
class ShowIpv6UserRouteCmd( ShowCommand.ShowCliCommandClass ):
   '''Display routes from the RIB that have the 'dynamic' attribute set to True
      These routes have been programmed via the EOS SDK with the 'persistent' flag
      set to False.'''
   syntax = 'show ipv6 user route [ vrf VRF ] [ tag TAGNUM ]'
   data = {
         'ipv6': ipv6MatcherForShow,
         'user': 'Display non-persistent routes',
         'route': 'Static routes',
         'vrf': vrfForShowMatcher,
         'VRF': vrfNameMatcher,
         'tag': tagMatcherForConfig,
         'TAGNUM' : tagNumberMatcher
         }
   cliModel = VrfIpRibRoutes

   @staticmethod
   def handler( mode, args ):
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, args.get( 'VRF' ) )
      if not vrfExists( vrfName ):
         mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
         return None

      fmt = mode.session_.outputFormat()
      tag = args.get( 'TAGNUM', 0 )
      routing6.showUserRoutes( vrfName=vrfName, tag=tag,
                              fmt=fmt, mlib=printer )
      return VrfIpRibRoutes

BasicCli.addShowCommandClass( ShowIpv6UserRouteCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHardwareStatus, routing6HardwareStatus, routing6HardwareConfig, \
       ip4Config, ip6Config
   global allVrfConfig
   global allVrfStatusLocal
   global l3Config
   global myEntManager
   global allIntfStatusDir
   global runnabilityConfig

   myEntManager = entityManager

   routing4.plugin( entityManager )
   routing6.plugin( entityManager )
   allVrfConfig = ConfigMount.mount( entityManager, "ip/vrf/config",
                                     "Ip::AllVrfConfig", "w" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                       Cell.path( "ip/vrf/status/local" ),
                                       "Ip::AllVrfStatusLocal", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routing6HardwareStatus = LazyMount.mount( entityManager,
         "routing6/hardware/status", "Routing6::Hardware::Status", "r" )
   routing6HardwareConfig = ConfigMount.mount( entityManager,
         "routing6/hardware/config", "Routing6::Hardware::Config", "w" )
   ip4Config = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   ip6Config = ConfigMount.mount( entityManager, "ip6/config", "Ip6::Config", "w" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", "r" )
   runnabilityConfig = ConfigMount.mount( entityManager,
                                          "routing6/runnability/config",
                                          "Routing6::Runnability::Config", "w" )

   IraIp6IntfCli.canSetIntfIpHook.addExtension( canSetIntfIp )

   def _showTechCmds():
      commands = [ 'show ipv6 route vrf all detail',
                   'show ipv6 interface',
                   'show ipv6 route vrf all host',
                   'show ipv6 bgp peers' ]
      vrfCmds = [ 'show kernel ipv6 route vrf all',
                  'show kernel ipv6 interface addr vrf all',
                  'show kernel ipv6 acl vrf all' ]
      if routingHardwareStatus.vrfCapability.ipv6EnabledDefault:
         commands += vrfCmds
      return commands

   registerShowTechSupportCmdCallback( '2012-08-22 09:22:00',
                                       _showTechCmds )
   # Timestamps are made up to maintain historical order within show tech-support
   registerShowTechSupportCmdCallback( '2017-04-07 09:22:00',
         lambda: [ "show ipv6 route vrf all summary" ],
         summaryCmdCallback=lambda: [
            "show ipv6 interface brief",
            "show ipv6 route vrf all summary",
            ] )
