#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements global DLB configuration.
#-------------------------------------------------------------------------------
'''Configuration commands for global DLB'''

import Tac, CliParser, BasicCli
import CliToken.Ip
import CliCommand, ShowCommand, CliMatcher
import LazyMount
import ConfigMount
import IraCommonCli
import IraRouteCommon
from IraNexthopGroupCli import nexthopGroupSupportedGuard
from CliPlugin.IraDlbModel import GlobalDlb

routingHwDlbConfig = None
routingHardwareStatus = None
routingNhgConfig = None
myEntManager = None

ip = IraRouteCommon.Ip4()
routing = IraRouteCommon.routing( ip )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

#-------------------------------------------------------------------------------
# "[no/default] ip hardware fib load-balance distribution [ hash|dynamic]"
#  command
#-------------------------------------------------------------------------------
def disableGlobalDlb( mode ):
   routingHwDlbConfig.globalDlbEcmpEnable = False

def enableGlobalDlb( mode ):
   routingHwDlbConfig.globalDlbEcmpEnable = True

def dlbEcmpSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.dlbEcmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

loadBalanceKwMatcher = CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'load-balance',
            helpdesc='Configure load balancing on ECMP routes' ),
         guard=dlbEcmpSupportedGuard )

distributionKwMatcher = CliMatcher.KeywordMatcher(
   'distribution', helpdesc='Configure load balance distribution' )

hashKw = CliMatcher.KeywordMatcher( 'hash',
                                    helpdesc='Enable hash based load balancing' )

dynamicKw = CliMatcher.KeywordMatcher( 'dynamic',
                                       helpdesc='Enable dynamic load balancing' )

class FibLoadBalanceDistributionCmd( CliCommand.CliCommandClass ):
   syntax = '''ip hardware fib load-balance distribution ( hash | dynamic )'''
   noOrDefaultSyntax = '''ip hardware fib load-balance distribution ...'''
   
   data = { "ip": CliToken.Ip.ipMatcherForConfig,
            "hardware": CliToken.Hardware.hardwareForConfigMatcher,
            "fib": IraCommonCli.fibKw,
            "load-balance": loadBalanceKwMatcher,
            "distribution": distributionKwMatcher,
            "hash": hashKw,
            "dynamic": dynamicKw }

   @staticmethod
   def handler( mode, args ):
      if "hash" in args:
         disableGlobalDlb( mode )
      elif "dynamic" in args:
         enableGlobalDlb( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      disableGlobalDlb( mode )

configMode.addCommandClass( FibLoadBalanceDistributionCmd )

#-------------------------------------------------------------------------------
# "show ip hardware fib load-balance distribution" command
#-------------------------------------------------------------------------------

def showLoadBalanceDistribution( mode ):
   GlobalDlbModel = GlobalDlb()
   GlobalDlbModel.dlbEcmpNhg = []
   GlobalDlbModel.globalDlbEcmpEnabled = routingHwDlbConfig.globalDlbEcmpEnable
   if nexthopGroupSupportedGuard( mode, None ) is None:
      nhgs = routingNhgConfig.nexthopGroup.items()
      nhgs.sort()
      for nhgName, nhgInfo in nhgs:
         if nhgInfo.dlbEcmpEnable:
            GlobalDlbModel.dlbEcmpNhg.append( nhgName )
   return GlobalDlbModel

class ShowFibLoadBalanceDistributionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip hardware fib load-balance distribution'''
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'hardware': CliToken.Hardware.hardwareForShowMatcher,
            'fib': IraCommonCli.fibKw,
            'load-balance': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                              'load-balance',
                              helpdesc='Display load balancing on ECMP routes' ),
                                             guard=dlbEcmpSupportedGuard ),
            'distribution': 'Display load balance distribution'
          }

   cliModel=GlobalDlb

   @staticmethod
   def handler( mode, args ):
      return showLoadBalanceDistribution( mode )

BasicCli.addShowCommandClass( ShowFibLoadBalanceDistributionCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )

   global routingHwDlbConfig
   global routingHardwareStatus
   global routingNhgConfig
   global myEntManager

   myEntManager = entityManager

   routingHwDlbConfig = ConfigMount.mount( entityManager,
                                           "routing/hardware/dlb/config",
                                           "Routing::Hardware::DlbConfig", "w" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routingNhgConfig = LazyMount.mount( entityManager,
                                       "routing/nexthopgroup/input/cli",
                                       "Routing::NexthopGroup::ConfigInput", "r" )
