# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This file contains the Capi models which are common for both Ipv4 and IpV6 commands
from CliModel import (
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
)

from ArnetModel import IpGenericAddress
from IntfModels import Interface
from TableOutput import createTable, Format
import Tac
protocolAgentModelType = Tac.Type( "L3::ProtocolAgentModelType" )

#--------------------------------------------------------
# Model for "show ip/ipv6 route static not-installed bfd
# Model is shared for both v4/v6 show commands.
#--------------------------------------------------------

class IpRouteSimpleVia( Model ):
   nexthopAddr = IpGenericAddress( help="Nexthop address of the via", optional=True )
   interface = Interface( help="Name of the interface", optional=True )
   egressVrf = Str( help="Name of the nexthop's VRF if the route "
                         "is an inter-vrf route",
                    optional=True )

class ViaList( Model ):
   vias = List( valueType=IpRouteSimpleVia, help="List of vias" )
   preference = Int( help="Administrative distance; lower values have precedence",
                              optional=True )
   metric = Int( help="Used by routing protocol to determine whether one route "
                     "should be chosen over another.  Lower metrics have precedence",
                     optional=True )
class AllRoutes( Model ):
   viaGroups = List( valueType=ViaList, help="List of all paths" )

class NotInstalledBfd( Model ):
   routes = Dict( keyType=str, valueType=AllRoutes,
                  help="Bfd disabled static routes" )

class VrfNotInstalledBfd( Model ):
   vrfs = Dict( keyType=str, valueType = NotInstalledBfd,
                 help="Route info of each Vrf" )

   def printHeader( self ):
      print "The following static routes are not installed, because BFD"
      print "cannot verify connectivity to their nexthop addresses."
      print ""

   def getRouteStr( self, viaSet, route ):
      """ This function will return routeStr something like below - 
      10.10.0.0/16 [1/0] via 1.1.1.1
                         via 1.1.1.2, Ethernet1
                         via fe00::1
      """
      pref = viaSet.preference
      met = viaSet.metric
      newRouteFlag = True
      routeStr = ""
      for via in sorted( viaSet.vias ):
         viaStr = " via "
         if via.nexthopAddr:
            viaStr +=  str( via.nexthopAddr )
         if via.interface:
            viaStr += ", " + str( via.interface ).strip("'")
         if newRouteFlag:
            routeWithPref = ( route + " [" + str( pref ) + "/" +
                              str( met ) + "]" )
            length  = len( routeWithPref )
            routeStr = routeWithPref + viaStr
            newRouteFlag = False
         else:
            routeStr += "\n" +  length*" " + viaStr
      return routeStr

   def render( self ):
      """ Sample output
      (router>) show ip route vrf all static not-installed bfd
      The following static routes are not installed, because BFD 
      cannot verify connectivity to their nexthop addresses.

      VRF          Route
      ------------ -----------------------------
      default      10.10.0.0/16 [1/0] via 1.1.1.1
                                      via 1.1.1.2, Ethernet1
                                      via fe00::1
      default      2.3.4.0/24 [50/0] via 3.3.3.1
      default      10.10.0.0/16 [100/0] via 1.1.1.1
      xyz          2.3.4.0/24 [50/0] via 3.3.3.1
      """
      atleastOneRoute = False
      for vrf in sorted( self.vrfs.keys() ):
         if not atleastOneRoute:
            t = createTable( ( 'VRF', 'Route' ) )
            f1 = Format( justify='left' )
            f2 = Format( justify='left' )
            t.formatColumns( f1, f2 )
            atleastOneRoute = True

         for route in sorted( self.vrfs[ vrf ].routes.keys() ):
            allRoutes = self.vrfs[ vrf ].routes[ route ]
            for viaSet in sorted( allRoutes.viaGroups ):
               t.newRow( vrf, self.getRouteStr( viaSet, route ) )

      if atleastOneRoute:
         self.printHeader()
         print t.output()

protoModelStates = [ protocolAgentModelType.ribd, protocolAgentModelType.multiAgent ]

class ServiceRoutingProtoModelStatus( Model ): 
   """
   Sample output:
   
   Operating routing protocol model: ribd
   Configured routing protocol model: multi-agent (will apply after next reboot)
   """

   operatingProtoModel = Enum( values=protoModelStates,
                               help="Active service routing protocols model" )
   configuredProtoModel = Enum( values=protoModelStates,
                                help="Configured service routing protocols model" )

   def render( self ):
      protoModelStatusTemplate = [
         "\nOperating routing protocol model: %s",
         "Configured routing protocol model: %s%s"
      ]
      warningMsg = ""
      if self.operatingProtoModel != self.configuredProtoModel:
         warningMsg = " (will apply after next reboot)"
      print protoModelStatusTemplate[ 0 ] % self.operatingProtoModel
      print protoModelStatusTemplate[ 1 ] % ( self.configuredProtoModel, warningMsg )
         
