#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import Arnet
import CliCommand
import CliMatcher
import IpAddrMatcher
import Ip6AddrMatcher
import IntfCli
import IraRouteCommon
import CliExtensions
import LazyMount
from CliPlugin.IraCommonModel import VrfNotInstalledBfd, NotInstalledBfd, ViaList
from CliPlugin.IraCommonModel import IpRouteSimpleVia, AllRoutes

allIntfStatusDir = None

IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )
AddressFamily = Tac.Type( 'Arnet::AddressFamily' )

fibKw = CliMatcher.KeywordMatcher( 'fib', helpdesc='Routing table' )
fibMatcher = fibKw

resilienceMatcher = CliMatcher.KeywordMatcher( 'resilience' , \
                                   helpdesc='Configure Resilience in ECMP routes' )
resilientEcmpCapacityMatcher = CliMatcher.KeywordMatcher( 'capacity', \
                                  helpdesc='specify capacity value' )
resilientEcmpRedundancyMatcher = CliMatcher.KeywordMatcher( 'redundancy', \
                                  helpdesc='specify redundancy value ' )
prefixLengthMatcher = CliMatcher.KeywordMatcher( 'prefix-length',
                                          helpdesc='Length of the prefix in bits' )
redundantSpecificMatcher = CliMatcher.KeywordMatcher( 'redundant-specifics',
                              helpdesc="type of route filter to use" )
filterMatcher = CliMatcher.KeywordMatcher( 'filter',
                          helpdesc="filter command" )
trackMatcherForConfig = CliMatcher.KeywordMatcher( 'track',
                                                   helpdesc='Track this route' )
notInstalledMatcher = CliMatcher.KeywordMatcher( 'not-installed',
                                           helpdesc='Show not installed routes' )
bfdMatcher = CliMatcher.KeywordMatcher( 'bfd',
                                  helpdesc='Not installed routes tracked by Bfd' )

helpTokenNextHopName ='Next hop name'
nameMatcherForConfig = CliMatcher.KeywordMatcher( 'name',
                                                  helpdesc=helpTokenNextHopName )
nexthopNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9._-]+',
                                                helpname='WORD',
                                              helpdesc='String naming the next hop' )

nullIntfKwMatcher = CliMatcher.KeywordMatcher( 'Null0',
                           helpdesc='Interface that drops all traffic' )

class Null0Expr( CliCommand.CliExpression ):
   expression = 'Null0 | ( Null 0 )'
   data = { 'Null0': nullIntfKwMatcher,
            'Null': CliCommand.Node( CliMatcher.KeywordMatcher( 'Null',
                                        helpdesc='drops' ),
                                     hidden=True ),
            '0': CliCommand.Node( CliMatcher.KeywordMatcher( '0',
                                        helpdesc='drops' ),
                                     hidden=True ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'Null' in args and '0' in args:
         del args[ 'Null' ]
         del args[ '0' ]
         args[ 'Null 0' ] = 'Null 0'

leakMatcherForConfig = CliCommand.Node( CliMatcher.KeywordMatcher( 'leak',
                                 helpdesc='additional output VRF for this route' ),
                                 hidden=True )
leakVrfMatcherForConfig = CliMatcher.KeywordMatcher( 'vrf',
                                 helpdesc='additional output VRF for this route' )
egressVrfMatcher = CliMatcher.KeywordMatcher( 'egress-vrf',
                                 helpdesc='egress VRF for this route' )

def printWarningIfNotRoutedPort( mode, intf, configType ):
   intfStatus = allIntfStatusDir.intfStatus.get( intf )
   # Don't throw warning if operating inside config-session
   if intfStatus and intfStatus.forwardingModel != "intfForwardingModelRouted" \
      and not mode.session.inConfigSession():
      mode.addWarning( "%s configuration will be ignored while interface %s "
                       "is not a routed port." % ( configType, intf ) )

ip4 = IraRouteCommon.Ip4()
ip6 = IraRouteCommon.Ip6()
routing4 = IraRouteCommon.routing( ip4 )
routing6 = IraRouteCommon.routing( ip6 )

interfaceToken = CliMatcher.KeywordMatcher( 'interface',
                                 helpdesc='Show only via on this interface' )
nexthopToken = CliMatcher.KeywordMatcher( 'next-hop',
                                 helpdesc='Show only via resolved by this ip' )
ip6AddrExpr = Ip6AddrMatcher.Ip6AddrMatcher( helpdesc='Match this IPv6 address' )
ipAddrExpr = IpAddrMatcher.IpAddrMatcher( helpdesc='Match this IP address' )
intfExpr = IntfCli.Intf.matcher

class NexthopCliExpression( CliCommand.CliExpression ):
   """
   SubClass for matching filters on interface or nexthop address.
   This creates all the Tacc classes needed for the Tacc filter.
   """
   expression = """{ ( interface INTERFACE ) |
                     ( nexthop ( IPADDR | IP6ADDR ) ) } 
                   [ all-vias ]"""
   data = {
      "interface": CliCommand.Node( interfaceToken,
                                    maxMatches=1 ),
      "INTERFACE": intfExpr,
      "nexthop": CliCommand.Node( nexthopToken,
                                  maxMatches=1 ),
      "IPADDR": ipAddrExpr,
      "IP6ADDR": ip6AddrExpr,
      "all-vias": 'Show all the vias of a route if one those vias matches',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if "ip" in args:
         routing = routing4
      else:
         routing = routing6
      # if all-via is not given then the non matching vias will be discard
      routeFilterList = Tac.newInstance( "Ira::RouteFilterColl",
                                          "all-vias" not in args )
      if 'interface' in args:
         intfname = args.pop( "INTERFACE" )[ 0 ]
         intfStatus = routing.intfStatus( intfname )
         routeFilterIntf = Tac.newInstance( "Ira::RouteFilterIntf", intfStatus )
         routeFilterList.filterColl.add( routeFilterIntf )
      if "IPADDR" in args or "IP6ADDR" in args:
         addr = args.pop( "IPADDR" ) if "IPADDR" in args else args.pop( "IP6ADDR" )
         genAddr = Arnet.IpGenAddr( str( addr[ 0 ] ) )
         routeFilterIp6 = Tac.newInstance( "Ira::RouteFilterIp", genAddr )
         routeFilterList.filterColl.add( routeFilterIp6 )
      args[ "nexthop-matchers" ] = routeFilterList

# isValidNextHopHook allows another package to check if the next hop
# is one of the switch's own IP addresses. Takes the next hop and
# vrfname as the two arguments and returns a boolean: True if the next
# hop is valid, False if it is not.
isValidNextHopHook4 = CliExtensions.CliHook()
isValidNextHopHook6 = CliExtensions.CliHook()

def validNextHop4( mode, hop, vrfName, allowMcast=False ):
   if not allowMcast and IpAddrMatcher.validateMulticastIpAddr( hop ) == None or \
          hop == '255.255.255.255':
      mode.addError( "Next hop address must be unicast" )
      return False

   assert vrfName and vrfName != ''

   # Check if next hop is equal to one of the switch's IP addresses.
   checker = Tac.Value( "Ira::InvalidNextHopChecker" )
   invalidHop = checker.check( ip4.config.force(), hop, vrfName )

   if not invalidHop:
      for hook in isValidNextHopHook4.extensions():
         if not hook( hop, vrfName ):
            invalidHop = True
            break
   if invalidHop:
      mode.addWarning( "Invalid next hop address (it's this router)" )

   return True

def validNextHop6( mode, hop, vrfName, intfId=None ):
   if hop.isMulticast:
      mode.addError( "Next hop address must be unicast" )
      return False
   if hop.isLinkLocal and not intfId:
      mode.addError( "Link local next hop requires an interface" )
      return False
   if not hop.isGlobalUnicast and not hop.isLinkLocal:
      mode.addError( "Next hop must be routeable" )
      return False

   assert vrfName and vrfName != ''
   
   # Check if next hop is equal to one of the switch's IP addresses
   checker = Tac.Value( "Ira::InvalidNextHop6Checker" )
   invalidHop = checker.check( ip6.config.force(), hop, vrfName )

   for hook in isValidNextHopHook6.extensions():
      if not hook( hop, vrfName ):
         invalidHop = True
         break

   if invalidHop:
      mode.addWarning( "Invalid next hop address (it's this router)" )

   return True

def validNextHopGen( mode, hop, vrfName, intfId=None, allowMcast=False ):
   if isinstance( hop, str ) and IpGenAddr.isIpv4( hop ):
      return validNextHop4( mode, hop, vrfName, allowMcast )
   elif isinstance( hop, IpGenAddr ):
      if hop.af == AddressFamily.ipv4:
         return validNextHop4( mode, hop.v4Addr, vrfName, allowMcast )
      elif hop.af == AddressFamily.ipv6:
         return validNextHop6( mode, hop.v6Addr, vrfName, intfId )
   assert 0, 'Invalid nexthop'

def showCommonNotInstalledBfd( vrfs, v4=True ):
   """
   Common function which will be invoked from IraIpCli.py and IraIp6Cli.py. 
   """
   myVrfModel = VrfNotInstalledBfd()
   if v4:
      routing = routing4
   else:
      routing = routing6

   for vrfName in vrfs:
      # pylint: disable=protected-access
      if ( isinstance( routing.staticBfdStatus( vrfName ), LazyMount._Proxy ) ):
         LazyMount.force( routing.staticBfdStatus( vrfName ) )
      bfdStatus = routing.staticBfdStatus( vrfName )
      if not bfdStatus:
         continue

      for key in bfdStatus.route.keys():
         statusObj = bfdStatus.route.get( key )

         # If BFD is up then move to the next node as these routes are installed.
         if statusObj.status: 
            continue

         for routeKey in statusObj.dependantRoute.keys():
            routeVia = statusObj.dependantRoute.get( routeKey )
            prefix = routeVia.rKey.prefix
            pfx = str( prefix )

            # Check for Ipv4/Ipv6
            if ( not ( ( v4 and prefix.af == AddressFamily.ipv4 ) or 
                     ( not v4 and prefix.af == AddressFamily.ipv6 ) ) ):
               continue

            # Populate the vrfName in the model.
            if vrfName not in myVrfModel.vrfs.keys():
               myModel = NotInstalledBfd()
               myVrfModel.vrfs[ vrfName ] = myModel

            for via in routeVia.trackedVia.keys():
               # Create one via entry here
               viaModel = IpRouteSimpleVia()
               viaModel.nexthopAddr = via.hop
               if via.intfId:
                  viaModel.interface = via.intfId

               # Figure out where do we need to put the via entry created above.
               # The viaSet represents the list where we need to append via entry.
               viaSet = None
               if pfx not in myModel.routes.keys():
                  # This is the case where we are seeing this prefix for the first
                  # time in this vrf.
                  viaSet = ViaList()
                  myModel.routes[ pfx ] = AllRoutes()
                  myModel.routes[ pfx ].viaGroups.append( viaSet )
               else:
                  # This is the case where the prefix already exists for this vrf.
                  # This will again give us two cases.
                  for vSet in myModel.routes[ pfx ].viaGroups:
                     if vSet.preference == routeKey.preference:
                        # This is the case where the prefix which is present has the
                        # pref as the new one which we want to insert.
                        viaSet = vSet
                        break
                  else:
                     # This is the case where the pref of the prefix is different 
                     # than all the existing ones for the same prefix.
                     viaSet = ViaList()
                     myModel.routes[ pfx ].viaGroups.append( viaSet )

               # Since we now have a viaSet. Insert the via entry here.
               viaSet.vias.append( viaModel )
               viaSet.preference = routeKey.preference
               viaSet.metric = 0

   return myVrfModel

#---------------------------------------------------------------------------------
# Hook for show rib ready [ vrf vrfName ]
# This hook has been defined in Ira to avoid adding a dependency between gated
# and IpRib
#---------------------------------------------------------------------------------
ribReadyHook = CliExtensions.CliHook()

#---------------------------------------------------------------------------------
# Hook for show rib route ip[v6]
# This hook has been defined in Ira to avoid adding a dependency between gated
# and IpRib
#---------------------------------------------------------------------------------
ribRouteHook = CliExtensions.CliHook()
ribRouteModel = {}

#---------------------------------------------------------------------------------
# Hook for show rib next-hop resolution route [ipv4|ipv6] (vrf <vrfName>)
# This hook has been defined in Ira to avoid adding a dependency between gated
# and IpRib
#---------------------------------------------------------------------------------
ribResolutionRouteHook = CliExtensions.CliHook()

def Plugin( entityManager ):
   global allIntfStatusDir
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   routing4.plugin( entityManager )
   routing6.plugin( entityManager )

# As there is no exact identifier on an IPv6 address if it was generated by
# SLAAC, the latter 64 bits has to be checked against the EUI-64 of the
# relevant interface.
def isSlaacAddress( ip6Addr, intfStatus ):
   # Not all interfaces have routedAddr attribute,
   # which contains the needed MAC address
   if not hasattr( intfStatus, 'routedAddr' ):
      return False
   ethAddr = Tac.Value( "Arnet::EthAddr" )
   ethAddr.stringValue = intfStatus.routedAddr
   return ( ip6Addr.u16[ 4 ] == ethAddr.word0 ^ 0x200 ) and \
          ( ip6Addr.u16[ 5 ] == ethAddr.byte[ 2 ] << 8 | 0xff ) and \
          ( ip6Addr.u16[ 6 ] == ethAddr.byte[ 3 ] | 0xfe00 ) and \
          ( ip6Addr.u16[ 7 ] == ethAddr.word2 )
