# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher
import LazyMount
import BasicCli
import Tac
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IpsecModel as IpsecModel
import CliPlugin.VrfCli as VrfCli
import CliPlugin.TunnelIntfCli as TunnelIntfCli
import CliPlugin.TechSupportCli as TechSupportCli
import CliToken.Ip
import CliToken.Ipsec
import ShowCommand
import SmashLazyMount

#Assigning globals.
ipsecDaemonConfigDir = None
ipsecDaemonStatusDir = None
ipsecIkeConfigDir = None
ipsecIkeStatusDir = None
ipsecConnTableDir = None
tunnelIntfStatusDir = None
ipsecPathStatus = None
entMan = None
allVrfConfig = None
ipsecCounterTable = None

def haveValidIpsecLicense( func ):
   ''' Decorator for config commands. If a valid IPsec license is not 
   found, throw a warning.'''
   def newFunc( mode, *args, **kwargs ):
      if not ipsecIkeStatusDir.ipsecLicenseEnabled:
         mode.addWarning( 'No valid IPsec license found. '
                          'IPsec is disabled.' )
      return func( mode, *args, **kwargs )
   return newFunc

matcherSecurity = CliMatcher.KeywordMatcher( 'security',
      helpdesc='IP security information' )
matcherDaemon = CliMatcher.KeywordMatcher( 'daemon',
      helpdesc='Show IPsec Daemon information' )
profileMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
      helpdesc='name of the profile', helpname='WORD' )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF instance' )

#--------------------------------------------------------------------------------
# show ip security profile [ PROFILE_NAME ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecProfiles( mode, args ):
   ''' function to create list of profiles. '''

   pName = args.get( 'PROFILE_NAME' )
   def fillProfileInfo( profName, profilesModel, profiles ):
      profileModel = IpsecModel.IpsecProfile()
      profileModel.fillIpsecProfileInfo( profiles[profName] )
      profilesModel.ipsecProfiles[profName] = profileModel

   profilesModel = IpsecModel.IpsecProfiles()
   profiles = ipsecIkeConfigDir.ipsecProfile
   if pName is None:
      for pName in profiles:
         fillProfileInfo( pName, profilesModel, profiles )
   else:
      if profiles.has_key(pName):
         fillProfileInfo( pName, profilesModel, profiles )
      else:
         mode.addError( 'Profile %s has not been configured' % pName )

   return profilesModel

class IpSecurityProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security profile [ PROFILE_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'profile': 'Display Profile information',
      'PROFILE_NAME': profileMatcher,
   }
   handler = showIpsecProfiles
   cliModel = IpsecModel.IpsecProfiles

BasicCli.addShowCommandClass( IpSecurityProfileCmd )

#--------------------------------------------------------------------------------
# show ip security security-association [ TS_NAME ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecSecurityAssociations( mode, args ):
   ''' function to create list of transform-sets. '''
   secAssocName = args.get( 'TS_NAME' )
   def fillSecurityAssociationInfo( SA, secAssocsModel, securityAssociations ):
      secAssocModel = IpsecModel.IpsecSecurityAssociation()
      secAssocModel.fillIpsecSecurityAssociationInfo( securityAssociations[ SA ] )
      secAssocsModel.ipsecSecurityAssociations[ SA ] = secAssocModel

   secAssocsModel = IpsecModel.IpsecSecurityAssociations()
   ipsecSecurityAssociations = ipsecIkeConfigDir.securityAssoc
   if secAssocName is None:
      for secAssocName in ipsecSecurityAssociations:
         fillSecurityAssociationInfo( secAssocName, secAssocsModel, 
               ipsecSecurityAssociations )
   else:
      if ipsecSecurityAssociations.has_key(secAssocName):
         fillSecurityAssociationInfo( secAssocName, secAssocsModel, 
               ipsecSecurityAssociations )
      else:
         mode.addError( 'SA %s has not been configured' % secAssocName )

   return secAssocsModel

class IpSecuritySecurityAssociationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security security-association [ TS_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'security-association': 'Display Security Associations',
      'TS_NAME': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+', 
         helpdesc='name of the TS', helpname='WORD' ),
   }
   handler = showIpsecSecurityAssociations
   cliModel = IpsecModel.IpsecSecurityAssociations

BasicCli.addShowCommandClass( IpSecuritySecurityAssociationCmd )

#--------------------------------------------------------------------------------
# show ip security policy [ POLICY_NAME ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecPolicies( mode, args ):
   ''' function to create list of policies. '''

   polName = args.get( 'POLICY_NAME' )
   def fillPolicyInfo( polName, policiesModel, policies ):
      policyModel = IpsecModel.IpsecPolicy()
      policyModel.fillIpsecPolicyInfo( policies[polName] )
      policiesModel.ipsecPolicies[polName] = policyModel

   policiesModel = IpsecModel.IpsecPolicies()
   policies = ipsecIkeConfigDir.ikePolicy
   if polName is None:
      for polName in policies:
         fillPolicyInfo( polName, policiesModel, policies )
   else:
      if policies.has_key(polName):
         fillPolicyInfo( polName, policiesModel, policies )
      else:
         mode.addError( 'Policy %s has not been configured' % polName )

   return policiesModel

class IpSecurityPolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security policy [ POLICY_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'policy': 'Display Policy information',
      'POLICY_NAME': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
         helpdesc='name of the policy', helpname='WORD' ),
   }
   handler = showIpsecPolicies
   cliModel = IpsecModel.IpsecPolicies

BasicCli.addShowCommandClass( IpSecurityPolicyCmd )

#--------------------------------------------------------------------------------
# show ip security applied-profile [ vrf VRF_NAME ] [ PROFILE_NAME ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecAppliedProfiles( mode, args ):
   ''' function to create list of applied profiles. '''
   profName = args.get( 'PROFILE_NAME' )
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   def fillProfileInfo( profile, profilesModel, profiles, vrfName ):
      profileModel = IpsecModel.IpsecAppliedProfile()
      profileModel.fillIpsecAppliedProfileInfo( profiles[profile],
                                                vrfName,
                                                ipsecPathStatus,
                                                tunnelIntfStatusDir )
      profilesModel.ipsecAppliedProfiles[profile] = profileModel

   profilesModel = IpsecModel.IpsecAppliedProfiles()
   profiles = ipsecIkeStatusDir.appliedProfiles
   if profName is None:
      for profName in profiles:
         fillProfileInfo( profName, profilesModel, profiles, vrfName )
   else:
      if profiles.has_key(profName):
         fillProfileInfo( profName, profilesModel, profiles, vrfName )
      else:
         mode.addError( 'Ipsec Profile %s has not been applied' % profName )

   return profilesModel

class IpSecurityAppliedProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security applied-profile [ VRF ] [ PROFILE_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'applied-profile': 'Display Applied Profiles',
      'VRF': vrfExprFactory,
      'PROFILE_NAME': profileMatcher,
   }
   handler = showIpsecAppliedProfiles
   cliModel = IpsecModel.IpsecAppliedProfiles

BasicCli.addShowCommandClass( IpSecurityAppliedProfileCmd )

#--------------------------------------------------------------------------------
# show ip security daemon
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecDaemonState( mode, args ):
   ''' function to create list of profiles. '''
   ipsecDaemonStateModel = IpsecModel.IpsecDaemonState()
   ipsecDaemonStateModel.fillIpsecDaemonStateInfo( ipsecDaemonConfigDir, 
                                                   ipsecDaemonStatusDir )
   return ipsecDaemonStateModel

class IpSecurityDaemonCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security daemon'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'daemon': matcherDaemon,
   }
   handler = showIpsecDaemonState
   cliModel = IpsecModel.IpsecDaemonState

BasicCli.addShowCommandClass( IpSecurityDaemonCmd )

#--------------------------------------------------------------------------------
# show ip security daemon log
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecDaemonLog( mode, args ):
   model = IpsecModel.IpsecDaemonLog()
   cmd = "cat /var/log/charon.log"
   items = Tac.run( cmd.split(), stdout=Tac.CAPTURE,
                    stderr=Tac.CAPTURE, ignoreReturnCode=True, asRoot=True )
   model.daemonLog = items.splitlines()
   return model

class IpSecurityDaemonLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security daemon log'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'daemon': matcherDaemon,
      'log': 'Show IPsec daemon logs',
   }
   handler = showIpsecDaemonLog
   cliModel = IpsecModel.IpsecDaemonLog

BasicCli.addShowCommandClass( IpSecurityDaemonLogCmd )

#---------------------------------------------------------------------------------
# show ip security connection [ vrf VRF_NAME ]
#                             [ ( Tunnel | TUNNEL ) |
#                               ( path [ ( name PATH_NAME ) |
#                                        ( peer PEER_ADDR ) ] ) ]
#                             [ detail ]
#--------------------------------------------------------------------------------
@haveValidIpsecLicense
def showIpsecConnectionState( mode, args ):
   ''' function to create list of profiles. '''
   detailed = 'detail' in args
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   tunnelName = args[ 'TUNNEL' ].name if 'TUNNEL' in args else None
   tunnels = 'Tunnel' in args
   pathName = None
   pathPeerIp = None
   paths = False
   pathPeerFqdn = None

   if 'path' in args:
      if 'PATH_NAME' in args:
         pathName = args[ 'PATH_NAME' ]
      elif 'PEER_ADDR' in args:
         pathPeerIp = args[ 'PEER_ADDR' ]
      else:
         paths = True

   #if tunnelOrPath is not None:
      # following changes are needed when we add pathPeerFqdn
      #if type( tunnelOrPath ) == type( [] ):
      #   for keyValuePair in tunnelOrPath:
      #      key = keyValuePair[ 0 ]
      #      if key == 'pathPeerIp':
      #         pathPeerIp = keyValuePair[ 1 ]
      #      elif key == 'pathPeerFqdn':
      #         pathPeerFqdn = keyValuePair[ 1 ]
      #      break
      #else:

   IpsecConnectionStateModel = IpsecModel.IpsecConnectionState()
   IpsecConnectionStateModel.fillIpsecConnectionStateInfo(
                                       entMan, vrfName,
                                       ipsecIkeStatusDir,
                                       ipsecConnTableDir,
                                       tunnelIntfStatusDir, 
                                       ipsecCounterTable,
                                       ipsecPathStatus,
                                       tunnelName=tunnelName, 
                                       pathName=pathName, 
                                       pathPeerIp=pathPeerIp,
                                       pathPeerFqdn=pathPeerFqdn,
                                       tunnels=tunnels,
                                       paths=paths,
                                       detailed=detailed )
   return IpsecConnectionStateModel

class IpSecurityConnectionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip security connection [ VRF ] '
               '[ ( Tunnel | TUNNEL ) | '
               '  ( path [ ( name PATH_NAME ) | ( peer PEER_ADDR ) ] ) ]'
               '[ detail ]' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'connection': 'Show IPsec Connection information',
      'VRF': vrfExprFactory,
      'Tunnel': 'Tunnel interface',
      'TUNNEL': TunnelIntfCli.TunnelIntf.matcher,
      'path': 'Path information',
      'name': 'Name of the path',
      'PATH_NAME': CliMatcher.PatternMatcher( pattern=r'\S+',
         helpdesc='Name of the path', helpname='WORD' ),
      'peer': 'Peer IP address',
      'PEER_ADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='IP address' ),
      'detail': 'Detailed IKE information',
   }
   handler = showIpsecConnectionState
   cliModel = IpsecModel.IpsecConnectionState

BasicCli.addShowCommandClass( IpSecurityConnectionCmd )

@haveValidIpsecLicense
def showIpsecKeyControllerPeer( mode, args ):
   ''' function to fill the peer info model '''
   detailed = 'detail' in args
   peerIp = args.get( 'PEER_ADDR' )
   ipsecPeerModel = IpsecModel.IpsecPeerState()
   ipsecPeerModel.fillPeersInfo( ipsecIkeStatusDir, detailed, peerIpStr=peerIp )
   return ipsecPeerModel

matcherKey = CliMatcher.KeywordMatcher( 'key',
      helpdesc='IPsec key information' )
matcherController = CliMatcher.KeywordMatcher( 'controller',
      helpdesc='IPsec key controller information' )

class IpsecKeyControllerPeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security key controller peer [ PEER_ADDR ] [ detail ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'key': matcherKey,
      'controller': matcherController,
      'peer': 'Peer information of the key controller',
      'PEER_ADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='Peer IP address' ),
      'detail': 'Detailed peer information',
   }
   handler = showIpsecKeyControllerPeer
   cliModel = IpsecModel.IpsecPeerState

BasicCli.addShowCommandClass( IpsecKeyControllerPeerCmd )

@haveValidIpsecLicense
def showIpsecKeyControllerLocal( mode, args ):
   ''' function to fill the local key info model'''
   ipsecLocalKeyModel = IpsecModel.IpsecLocalKeyState()
   ipsecLocalKeyModel.fillLocalKeyState( ipsecIkeConfigDir, ipsecIkeStatusDir )
   return ipsecLocalKeyModel

class IpsecKeyControllerLocalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip security key controller local'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'security': matcherSecurity,
      'key': matcherKey,
      'controller': matcherController,
      'local': 'Local key information of the key controller',
   }
   handler = showIpsecKeyControllerLocal
   cliModel = IpsecModel.IpsecLocalKeyState

BasicCli.addShowCommandClass( IpsecKeyControllerLocalCmd )

def _ipsecShowTechCmds():
   return [ 'show ip security applied-profile',
            'show ip security connection',
            'show ip security connection detail',
            'show kernel ipsec',
            'show kernel ipsec datapath',
            'show ip security daemon log',
            'show agent Ipsec logs',
            'show agent Ipsec uptime',
          ]

TechSupportCli.registerShowTechSupportCmdCallback( '2017-03-17 16:14:46',
                                                   _ipsecShowTechCmds )

def Plugin( entityManager ):
   global entMan
   global allVrfConfig
   global ipsecIkeConfigDir, ipsecIkeStatusDir, ipsecConnTableDir
   global tunnelIntfStatusDir, ipsecDaemonConfigDir, ipsecDaemonStatusDir
   global ipsecCounterTable, ipsecPathStatus
   
   entMan = entityManager

   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                      'Ip::AllVrfConfig', 'r' )
   ipsecDaemonConfigDir = LazyMount.mount( entityManager, 'ipsec/daemonconfig',
                                           'Ipsec::DaemonConfig', 'r' )
   ipsecDaemonStatusDir = LazyMount.mount( entityManager, 'ipsec/daemonstatus',
                                           'Ipsec::DaemonStatus', 'r' )
   ipsecIkeConfigDir = LazyMount.mount( entityManager, 'ipsec/ike/config',
                                        'Ipsec::Ike::Config', 'r' )
   ipsecIkeStatusDir = LazyMount.mount( entityManager, 'ipsec/ike/status',
                                        'Ipsec::Ike::Status', 'r' )
   ipsecConnTableDir = LazyMount.mount( entityManager,
                                           'ipsec/ike/connectiontable',
                                           'Ipsec::Ike::ConnectionTable', 'r' )
   tunnelIntfStatusDir = LazyMount.mount( entityManager, 
                                           'interface/status/tunnel/intf',
                                           'Interface::TunnelIntfStatusDir', 'r' )
   ipsecCounterTableInfo = SmashLazyMount.mountInfo( 'reader' )
   ipsecCounterTable = SmashLazyMount.mount( entityManager, 'ipsec/countertable',
                                             'IpsecCounters::IpsecCounterTable',
                                             ipsecCounterTableInfo )
   ipsecPathStatus = LazyMount.mount( entityManager, 'ipsec/path/status',
                                      'Ipsec::Path::PathStatusDir', 'r' )

