# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from itertools import chain
import time
import collections
import socket
import Arnet, Tracing
from ArnetModel import IpGenericAddress
from CliModel import (
      Model,
      Submodel,
      Bool,
      Dict,
      Enum,
      Int,
      Str,
      List,
      Float
      )
from IntfModel import Interface
import Tac

__defaultTraceHandle__ = Tracing.Handle( "IpsecCliModel" )
t8 = Tracing.trace8
t9 = Tracing.trace9

IpsecHmacAlgorithmEnum = Tac.Type( "Ipsec::IpsecHmacAlgorithm" )
IpsecAuthTypeEnum = Tac.Type( "Ipsec::Ike::IpsecAuthType" )
IpsecEspAlgorithmEnum = Tac.Type( "Ipsec::IpsecEspAlgorithm" )
IpsecDhGroupEnum = Tac.Type( "Ipsec::IpsecDhGroup" )
IpsecPeerDownReasonEnum = Tac.Type( "Ipsec::PeerDownReason" )
IpsecPeerConnStatusEnum = Tac.Type( "Ipsec::PeerConnStatus" )

AF_INET = 2
AF_INET6 = 10
DIR_IN = 0
DIR_OUT = 1

IPSEC_DIRECTION_LIST = [ 'anyDir',
                         'inboundDir',
                         'outboundDir',
                         'fwdDir',
                         'unknownDir',]
IPSEC_DIRECTION_MAP = { 0 : 'Ipsec Any Policy',
                        1 : 'Ipsec Inbound Policy',
                        2 : 'Ipsec Outbound Policy',
                        3 : 'Ipsec Forward Policy',
                        -1 : 'Unknown Policy' }

IPSEC_PROTO_LIST = [ 'esp', 'ah', 'unknownProto', ]
IPSEC_PROTO_MAP = { 50 : 'Encapsulation Security Payload protocol',
                    51 : 'Authentication Header protocol',
                    -1 : 'Unknown Proto' }

IPSEC_MODE_LIST = [ 'transport',
                    'tunnel',
                    'routeOptimization',
                    'inboundTrigger',
                    'beet',
                    'unknownMode', ]
IPSEC_MODE_MAP = { 0 : 'Transport',
                   1 : 'Tunnel',
                   2 : 'Route optimization',
                   3 : 'Inbound trigger',
                   4 : 'Bound end-to-end tunnel',
                   -1 : 'Unknown Mode' }

IPSEC_INTEGRITY_MAP = { IpsecHmacAlgorithmEnum.sha1 : '160bit Hash',
                        IpsecHmacAlgorithmEnum.sha256 : '256bit Hash',
                        IpsecHmacAlgorithmEnum.sha384 : '384bit Hash',
                        IpsecHmacAlgorithmEnum.sha512 : '512bit Hash', 
                        IpsecHmacAlgorithmEnum.nullhash : 'Null Hash',
                        'unknownIntegrity' : 'Unknown', }

IPSEC_ENCRYPTION_MAP = { IpsecEspAlgorithmEnum.des : 'Triple DES', 
                         IpsecEspAlgorithmEnum.aes128 : '128-bit AES', 
                         IpsecEspAlgorithmEnum.aes256 : '256-bit AES', 
                         IpsecEspAlgorithmEnum.aes128gcm64 :
                             '128 bit AES-GCM',
                         IpsecEspAlgorithmEnum.aes128gcm128 :
                             '128 bit AES-GCM',
                         IpsecEspAlgorithmEnum.aes256gcm128 :
                             '256 bit AES-GCM',
                         IpsecEspAlgorithmEnum.nullesp : 'Null encryption', 
                         'unknownEncr' : 'Unknown', }

IPSEC_AUTH_MAP = { IpsecAuthTypeEnum.pre_share : 'Pre-shared', 
                   IpsecAuthTypeEnum.rsa_sig : 'RSA Signature', 
                   'unknownAuth' : 'Unknown', }

IPSEC_PRIORITY_LIST = [ 'passThrough',
                       'regular',
                       'trap',
                       'fallback',
                       'unknownPriority', ]
IPSEC_PRIORITY_MAP = { 0 : 'Passthrough policies',
                       1 : 'Regular IPsec policies',
                       2 : 'Trap policies',
                       3 : 'Fallback drop policies',
                       -1 : 'Unknown Priority', }

IPSEC_XFRMPOLICYFLAGS_LIST = [ 'unknownPolicy', 'allowOverride',
                               'includeIcmp',
                               'unknownPolicy', ]
IPSEC_XFRMPOLICYFLAGS_MAP = { 1 : 'Allow policy override',
                              2 : 'Include ICMP payloads',
                              -1 : 'Unknown', }

IPSEC_DHGROUP_MAP = {
      IpsecDhGroupEnum.modp768 : '768 bit',
      IpsecDhGroupEnum.modp1024 : '1024 bit',
      IpsecDhGroupEnum.modp1536 : '1536 bit',
      IpsecDhGroupEnum.modp2048 : '2k bit',
      IpsecDhGroupEnum.modp3072 : '3072 bit',
      IpsecDhGroupEnum.modp4096 : '4k bit',
      IpsecDhGroupEnum.modp6144 : '6144 bit',
      IpsecDhGroupEnum.ecp384 : '384 bit ecp',
      IpsecDhGroupEnum.ecp521 : '521 bit ecp',
      IpsecDhGroupEnum.modp2048s256 : '2k bit, 256 bit subgroup',
      }

IPSEC_KEY_CONTROLLER_PEER_STATUS = {
      IpsecPeerConnStatusEnum.peerUp: 'up',
      IpsecPeerDownReasonEnum.dhGroupMismatch : 'down (DH group mismtach)',
      IpsecPeerDownReasonEnum.dhLenMismatch : 'down (DH length mismtach)',
      IpsecPeerDownReasonEnum.nonceLenMismatch : 'down (nonce Length mismtach)',
      IpsecPeerDownReasonEnum.encryptAlgoMismatch : 
            'down (encrypt algorithm mismtach)',
      IpsecPeerDownReasonEnum.hashAlgoMismatch : 'down (hash algorithm mismtach)',
      IpsecPeerDownReasonEnum.cryptoKeyGenerationErr :
            'down (crypto key generation error)',
      'unknownPeerState': 'unknown',
      }

IPSEC_DHGROUP_NUMBER_MAP = { 
      IpsecDhGroupEnum.modp768 : '1', 
      IpsecDhGroupEnum.modp1024 : '2', 
      IpsecDhGroupEnum.modp1536 : '5', 
      IpsecDhGroupEnum.modp2048 : '14', 
      IpsecDhGroupEnum.modp3072 : '15', 
      IpsecDhGroupEnum.modp4096 : '16', 
      IpsecDhGroupEnum.modp6144 : '17', 
      IpsecDhGroupEnum.ecp384 : '20', 
      IpsecDhGroupEnum.ecp521 : '21', 
      IpsecDhGroupEnum.modp2048s256 : '24', 
      'unknownDhGroup' : 'unknown',
      }

appliedProfileFormatStr = '%-28s %s'
profileFormatStr = '%-28s %-28s %-28s'
connStateFormatStr = '%-10s %-16s %-16s %-12s %-10s %-16s %-16s %-12s'
ikePolicyFormatStr = '%-28s %-16s %-16s %-16s %-9s %-6s %-28s'
saFormatStr = '%-28s %-16s %-16s %-9s %-28s'
peerFormatStr = '%-19s %-11s %-32s %-9s'

# Get the Ipsec up/rekey time in secs/minutes/hours
def ipsecTimeString( elapsed, output, units ):
   if elapsed > 0 and units:
      localName, localUnits = units[ 0 ]
      if len( units ) == 1:
         # Last unit, skip the devision
         value = elapsed
      else:
         value = elapsed % localUnits
      if value > 0:
         if value == 1:
            localName = localName[ : -1 ]
         result = [ '%d %s' % ( value, localName ) ]
         if output and output != '0 seconds':
            result.append( output )
         output = ', '.join( result )
      return ipsecTimeString( elapsed // localUnits,
             output, units[ 1 : ] )
   else:
      return output

class IpsecProfile( Model ):
   ''' CLI Model for Ipsec's Profile information. '''
   policyName = Str( help="Unique identifier of IKE Policy" )
   securityAssociation = Str( help="Transformset associated with profile", 
                           optional=True )

   def fillIpsecProfileInfo( self, ipsecProfile ):
      if ipsecProfile.ikePolicy is not None:
         policies = ipsecProfile.ikePolicy
         self.policyName = ''
         for policy in policies: 
            if self.policyName:
               self.policyName = self.policyName + ', ' + policy
            else:
               self.policyName = policy
      if ipsecProfile.securityAssoc is not None:
         securityAssociations = ipsecProfile.securityAssoc
         self.securityAssociation = ''
         for sa in securityAssociations:
            if self.securityAssociation:
               self.securityAssociation = self.securityAssociation + ', '
            self.securityAssociation = self.securityAssociation + \
                                          securityAssociations[sa].saName

   def renderProfile( self, profileName ):
      if profileName is None:
         return

      if self.policyName is None:
         self.policyName = 'None'
      if self.securityAssociation is None:
         self.securityAssociation = 'None'
      print profileFormatStr % ( profileName, self.policyName, 
            self.securityAssociation )

class IpsecProfiles( Model ):
   ''' CLI Model for show ip security profile [ name ]. '''
   ipsecProfiles = Dict( help="A mapping between a IPSEC profile name "
                              "and an IPSEC profile.", keyType=str, 
                         valueType=IpsecProfile )

   def render( self ):
      if self.ipsecProfiles is None:
         return

      print profileFormatStr % ( 'Profile name', 
            'IKE Policy Name', 'SA')
      for ipsecProfileKey, ipsecProfile in sorted( self.ipsecProfiles.iteritems() ):
         ipsecProfile.renderProfile( ipsecProfileKey )

class ShowKernelIpsecConnections( Model ):
   summary = List( valueType=str, help="Details about a connection" )

class ShowKernelIpsecChildSa( Model ):
   summary = List( valueType=str, help="Details about a childSa" )

class ShowKernelIpsecIkeSa( Model ):
   summary = List( valueType=str, help="Details about an ikeSa" )
   childSa = Dict( valueType=ShowKernelIpsecChildSa,
                   help="ChildSa's relevant to this ikeSa" )

class ShowKernelIpsecSecurityAssociations( Model ):
   ikeSa = Dict( keyType=str, valueType=ShowKernelIpsecIkeSa,
                 help="Details about each security associations ikeSa's and "
                 "related childSa's", optional=True )

class ShowKernelIpsecWorkerThreads( Model ):
   queue = Str( help="Number of threads in the queue" )
   scheduled= Int( help="Number of scheduled threads" )
   idle = Int( help="Number of idle threads" )
   working = Str( help="Number of working threads" )
   total = Int( help="Total number of threads" )

class ShowKernelIpsecStrongSwan( Model ):
   version = Str( help="StrongSwan version number" )
   architecture = Str( help="Architecture running on" )
   linux = Str( help="Linux kernel version" )
   plugins = List( valueType=str, help="Plugins enabled", optional=True )
   malloc = Dict( valueType=int, help="Details of memory" )
   workerThreads = Submodel( valueType=ShowKernelIpsecWorkerThreads,
                             help="Details of all worker threads" )
   startTime = Float( help="Start time in UTC" )
   uptime = Int( help="Uptime in seconds" )

class ShowKernelIpsecModel( Model ):
   strongSwan = Submodel( valueType=ShowKernelIpsecStrongSwan,
                          help="Details about IKE charon daemon", optional=True )
   listeningIpAddresses = List( valueType=IpGenericAddress, help="IPsec Listening "
                                "IP Addresses", optional=True )
   connections = Dict( valueType=ShowKernelIpsecConnections,
                       help="IPsec Connections", optional=True )
   securityAssociations= Dict( valueType=ShowKernelIpsecSecurityAssociations,
                               help="IPsec Security Associations", optional=True )
   dataPath = Dict( keyType=str, valueType=int,
                    help="Details of /proc/net/xfrm_stat", optional=True )
   _showIpsec = List( valueType=str, help="Show Kernel IPsec command output to keep "
                      "CLI output the same", optional=True )
   _dataPath = Dict( keyType=str, valueType=str, help="Show Kernel IPsec Datapath "
                     "command output to keep CLI output the same", optional=True )
   _isDatapath = Bool( help="CLI command contained datapath" )

   def setDatapath( self, isDatapath ):
      self._isDatapath = isDatapath

   def appendShowIpsec( self, info ):
      self._showIpsec.append( info )

   def addToDataPath( self, key, value ):
      self._dataPath[ key ] = value

   def render( self ):
      if self._isDatapath:
         if self._showIpsec:
            # For datapath output, if this _showIpsec is not empty,
            # some error occurred. Print it out
            for k in self._showIpsec:
               print k
         else:
            for k in sorted( self._dataPath ):
               print "%s\t%s" % ( k,  self._dataPath[ k ] )
      else:
         for k in self._showIpsec:
            print k

class IpsecDaemonLog( Model ):
   daemonLog = List( valueType=str, help="Log in charon.log", optional=True )

   def render( self ):
      for k in self.daemonLog:
         print k

class IpsecSecurityAssociation( Model ):
   ''' CLI Model for Ipsec Security Association information '''
   espAes = Enum( values=IPSEC_ENCRYPTION_MAP,
                      help='ESP Advanced Encryption Standard Algorithm' )
   espSha = Enum( values=IPSEC_INTEGRITY_MAP,
                  help="ESP Secure Hash Algorithm" )
   pfsGroup  = Enum( values=IPSEC_DHGROUP_MAP,
                     help="PFS Group to force a new DH key exchange" )
   saLifetime = Int( help="Lifetime for the SA in hours" )
   replayWindowSize = Int( help="Anti-Replay Window size to detect replay attack" )
   packetLimit = Int( help="Packet count limit for the SA" )
   byteLimit = Int( help="Byte count limit for the SA" )
   
   def fillIpsecSecurityAssociationInfo( self, ipsecSecurityAssociation ):
      saParams = ipsecSecurityAssociation.saParams
      if ( IPSEC_INTEGRITY_MAP.has_key( saParams.espSha ) ):
         self.espSha = saParams.espSha
      else:
         self.espSha = 'unknownIntegrity'
      if ( IPSEC_ENCRYPTION_MAP.has_key( saParams.espAes ) ):
         self.espAes = saParams.espAes
      else:
         self.espAes = 'unknownEncr'
      if ( IPSEC_DHGROUP_MAP.has_key( saParams.pfsGroup ) ):
         self.pfsGroup = saParams.pfsGroup
      else:
         self.pfsGroup = IpsecDhGroupEnum.modp2048
      self.saLifetime = saParams.saLifetime
      self.replayWindowSize = saParams.replayWindowSize
      self.packetLimit = saParams.packetLimit
      self.byteLimit = saParams.byteLimit

   def renderSA( self, saKey ):
      saLifetime = "%d hours" % self.saLifetime
      espSha = IPSEC_INTEGRITY_MAP[self.espSha]
      espAes = IPSEC_ENCRYPTION_MAP[self.espAes]
      pfsGroup = IPSEC_DHGROUP_MAP[self.pfsGroup]
      print saFormatStr % ( saKey, espAes , 
            espSha, saLifetime, pfsGroup )

class IpsecSecurityAssociations( Model ):
   ''' CLI Model for show ip security security-association [ <saName> ] '''
   ipsecSecurityAssociations = Dict( help="A mapping between a security "
                                          "association and it's information",
                                     keyType=str,
                                     valueType=IpsecSecurityAssociation )
   def render( self ):
      if self.ipsecSecurityAssociations is None:
         return

      print saFormatStr % ( 'SA Name', 'ESP Encryption', 
            'ESP Integrity', 'Lifetime', 'PFS Group' )
      for ipsecSaKey, ipsecSa in sorted( 
            self.ipsecSecurityAssociations.iteritems() ):
         ipsecSa.renderSA( ipsecSaKey )

class IpsecPolicy( Model ):
   ''' CLI Model for Ipsec's ISAKMP Policy information. '''
   version = Str( help="Version of IKE Policy" )
   integrity = Enum( values=IPSEC_INTEGRITY_MAP,
                     help="Hash algorithm to verify integrity mechanism" )
   encryption = Enum( values=IPSEC_ENCRYPTION_MAP,
                      help='Encryption Algorithm used in IKE' )
   ipsecAuth = Enum( values=IPSEC_AUTH_MAP,
                     help='Peer Authentication method' )
   ikeLifetime = Int( help="Lifetime for the IKE SA in hours" )
   dhGroup  = Enum( values=( IPSEC_DHGROUP_MAP ),
                    help="Identifier for Diffie-Hellman (DH) group" )
   rekey = Bool( help="Rekey Enabled" )

   def fillIpsecPolicyInfo( self, ikePolicy ):
      ikeParams = ikePolicy.ikeParams
      self.version = ikeParams.version
      if ( IPSEC_INTEGRITY_MAP.has_key( ikeParams.integrity ) ):
         self.integrity = ikeParams.integrity
      else:
         self.integrity = 'unknownIntegrity'
      if ( IPSEC_ENCRYPTION_MAP.has_key( ikeParams.encryption.encryption ) ):
         self.encryption = ikeParams.encryption.encryption
      else:
         self.encryption = 'unknownEncr'
      if ( IPSEC_AUTH_MAP.has_key( ikeParams.auth ) ):
         self.ipsecAuth = ikeParams.auth
      else:
         self.ipsecAuth = 'unknownAuth'
      self.ikeLifetime = ikeParams.ikeLifetime
      if ( IPSEC_DHGROUP_MAP.has_key( ikeParams.dhGroup ) ):
         self.dhGroup = ikeParams.dhGroup
      else:
         self.dhGroup = IpsecDhGroupEnum.modp2048
      self.rekey = ikeParams.rekey

   def renderPolicy( self, policy ):
      ikeLifetime = "%d hours" % self.ikeLifetime
      if self.rekey:
         rekey = "True"
      else:
         rekey = "False"
      ipsecAuth = IPSEC_AUTH_MAP[self.ipsecAuth]
      encryption = IPSEC_ENCRYPTION_MAP[self.encryption]
      dhGroup = IPSEC_DHGROUP_MAP[self.dhGroup]
      integrity = IPSEC_INTEGRITY_MAP[self.integrity]
      print ikePolicyFormatStr % ( policy,
            ipsecAuth, encryption, integrity,
            ikeLifetime, rekey, dhGroup )

class IpsecPolicies( Model ):
   ''' CLI Model for show isakmp [ policies | policy ]. '''
   ipsecPolicies = Dict( help="A mapping between IPSEC policy name "
                              "and an IPSEC policy", 
                         keyType=str, 
                         valueType=IpsecPolicy )

   def render( self ):
      if self.ipsecPolicies is None:
         return

      print ikePolicyFormatStr % ( 'Policy Name', 
            'Authentication', 'Encryption', 'Integrity',
            'Lifetime', 'Rekey', 'DH Group' )
      for ipsecPolicyKey, ipsecPolicy in sorted( self.ipsecPolicies.iteritems() ):
         ipsecPolicy.renderPolicy( ipsecPolicyKey )

class IpsecDaemonState( Model ):
   ''' CLI Model for show ip security state daemon '''
   strongSwanEnabled = Bool( help="strongSwan enabled" )
   strongSwanRunning = Bool( help="strongSwan running" )

   def fillIpsecDaemonStateInfo( self, daemonConfig, daemonStatus ):
      self.strongSwanEnabled = False
      self.strongSwanRunning = False
      if daemonConfig is None or daemonStatus is None:
         return

      self.strongSwanEnabled = daemonConfig.enableSswan
      self.strongSwanRunning = daemonStatus.sswanRunning

   def render( self ):
      print "Ipsec daemon State :"
      if self.strongSwanEnabled is True:
         print "  Enabled : True"
      else:
         print "  Enabled : False"
      if self.strongSwanRunning is True:
         print "  state : Running"
      else:
         print "  state : Not Running"

class IpsecConnectionDetail ( Model ):
   ''' CLI Model for ipsec connection info of a tunnel'''
   connectionName = Str( help="Name of the IPsec connection" )
   srcAddr = IpGenericAddress( help="Source IP Address" )
   dstAddr = IpGenericAddress( help="Destination IP Address" )
   tunnelDict = Dict ( help="Tunnel interface dict, with SA info",
                       keyType=Interface,
                       valueType=str )
   pathDict = Dict( help="Ipsec path dict", keyType=str, valueType=str )
   tunnelNs = Str( help="Tunnel namespace for this profile" )

   def fillIpsecConnectionDetails( self, connName, srcAddr, dstAddr, vrfName, state, 
                                   ipsecType, tunId=None, pathName=None ):
      self.connectionName = connName
      self.srcAddr = srcAddr
      self.dstAddr = dstAddr
      self.tunnelNs = vrfName
      self.fillIpsecConnState( state, ipsecType, tunId, pathName )

   def fillIpsecConnState( self, state, ipsecType, tunId=None, pathName=None ):
      if state != "Shut":
         if ipsecType == "IpsecTunnel":
            self.tunnelDict[ tunId ] = state
         elif ipsecType == "IpsecPath":
            self.pathDict[ pathName ] = state

class IpsecAppliedProfile( Model ):
   ''' CLI Model for ipsec profiles that have been applied'''
   appliedIntf = List( help="Tunnel interface List for this profile",
                       valueType=Interface )
   appliedPath = List( help="Path list for this profile",
                       valueType=str )
   ipsecConnectionDetails = Dict( help="A mapping between Ipsec profile "
                                       "info and corresponding connection",
                                  keyType=str, 
                                  valueType=IpsecConnectionDetail )

   def fillIpsecAppliedProfileInfo( self, appliedProfile, 
                                    vrfName, ipsecPathStatus, tunnelIntfStatusDir ):
      for intfId in appliedProfile.appliedIntfId:
         tunnelIntfStatus = tunnelIntfStatusDir.intfStatus.get( intfId )
         ipsecTunnelInfo = appliedProfile.appliedIntfId.get( intfId )
         if not all( ( ipsecTunnelInfo, tunnelIntfStatus ) ):
            t9( "Key missing for intfId: ", intfId )
            continue
         if ( ipsecTunnelInfo.vrfName == vrfName ):
            self.appliedIntf.append( tunnelIntfStatus.intfId )
            t9( "self.appliedIntf.append ", intfId )
            connName = ipsecTunnelInfo.connectionName
            ipsecConnectionDetail = self.ipsecConnectionDetails.get( connName )
            if not ipsecConnectionDetail:
               connectionModel = IpsecConnectionDetail()
               connectionModel.fillIpsecConnectionDetails( 
                                                   connName,
                                                   ipsecTunnelInfo.srcAddr,
                                                   ipsecTunnelInfo.dstAddr,
                                                   ipsecTunnelInfo.vrfName,
                                                   ipsecTunnelInfo.connectionState,
                                                   "IpsecTunnel", tunId=intfId )
               self.ipsecConnectionDetails[connName] = connectionModel
            elif intfId not in ipsecConnectionDetail.tunnelDict:
               ipsecConnectionDetail.fillIpsecConnState( 
                                                ipsecTunnelInfo.connectionState,
                                                "IpsecTunnel", tunId=intfId )
         else:
            t9( "not self.appliedIntf.append ", intfId )
            continue
      for connectionKey in appliedProfile.appliedPath:
         pathStatus = ipsecPathStatus.ipsecPathStatus.get( connectionKey )
         if not pathStatus:
            continue
         if connectionKey.vrfName == vrfName:
            self.appliedPath.append( pathStatus.pathName )
            connName = pathStatus.connectionName
            pathName = pathStatus.pathName
            ipsecConnectionDetail = self.ipsecConnectionDetails.get( connName )
            if not ipsecConnectionDetail:
               connectionModel = IpsecConnectionDetail()
               connectionModel.fillIpsecConnectionDetails( 
                                                   connName,
                                                   connectionKey.srcAddr,
                                                   connectionKey.dstAddr,
                                                   connectionKey.vrfName,
                                                   pathStatus.connectionState,
                                                   "IpsecPath", pathName=pathName )
               self.ipsecConnectionDetails[connName] = connectionModel
            elif pathName not in ipsecConnectionDetail.pathDict:
               ipsecConnectionDetail.fillIpsecConnState( pathStatus.connectionState,
                                                         "IpsecPath", 
                                                         pathName=pathName )

   def renderProfile( self, profileName ):
      appliedConnList = ''
      for path in Arnet.sortIntf( self.appliedPath ):
         if appliedConnList:
            appliedConnList = appliedConnList + ",\n%-28s %s" % ( '', path )
         else:
            appliedConnList = "%s" % ( path )
      for intfId in Arnet.sortIntf( self.appliedIntf ):
         if appliedConnList:
            appliedConnList = appliedConnList + ",\n%-28s %s" % ( '', intfId )
         else:
            appliedConnList = "%s" % ( intfId )
      if appliedConnList != '':
         print appliedProfileFormatStr % ( profileName, appliedConnList )

class IpsecAppliedProfiles( Model ):
   ''' CLI Model for show ip security profile [ profileName ]. '''
   ipsecAppliedProfiles = Dict( help="A mapping between Ipsec profile "
                                     "name and profile applied to an"
                                     "interface", 
                                keyType=str, 
                                valueType=IpsecAppliedProfile )

   def render( self ):
      if self.ipsecAppliedProfiles is None:
         return

      print appliedProfileFormatStr % ( 'Profile Name', 'Interface' )
      for appProfileKey, appProfile in sorted( 
            self.ipsecAppliedProfiles.iteritems() ):
         appProfile.renderProfile( appProfileKey )

class LifetimeCfg( Model ):
   ''' CLI Model for Lifetime interval Config in bytes or time '''

   xfrmPolicyByteLimitWarning = Int( 
         help="Warning about impending expiry of byte transfer limit" )
   xfrmPolicyByteLimitExpiry = Int( help="Expiry of byte transfer limit" )
   xfrmPolicyPacketLimitWarning = Int(
         help="Warning about impending expiry of packet transfer limit" )
   xfrmPolicyPacketLimitExpiry = Int(
         help="Expiry of packet transfer limit" )
   xfrmPolicyTimeLimitWarning = Int(
         help="Warning about impending expiry of time limit (in secs)" )
   xfrmPolicyTimeLimitExpiry = Int(
         help="Expiry of time limit (in secs)" )
   xfrmPolicyUseTimeWarning = Int(
         help="Warning about impending expiry of use time" )
   xfrmPolicyUseTimeExpiry = Int( help="Expiry of use time" )

   def fillLifetimeCfgInfo( self, ikeSALifetimeCfgStatus=None ):
      if ikeSALifetimeCfgStatus is not None:
         self.xfrmPolicyByteLimitWarning = ikeSALifetimeCfgStatus.softByteLimit
         self.xfrmPolicyByteLimitExpiry = ikeSALifetimeCfgStatus.hardByteLimit
         self.xfrmPolicyPacketLimitWarning = ikeSALifetimeCfgStatus.softPacketLimit
         self.xfrmPolicyPacketLimitExpiry = ikeSALifetimeCfgStatus.hardPacketLimit
         self.xfrmPolicyTimeLimitWarning = ikeSALifetimeCfgStatus.softAddExpire
         self.xfrmPolicyTimeLimitExpiry = ikeSALifetimeCfgStatus.hardAddExpire
         self.xfrmPolicyUseTimeWarning = ikeSALifetimeCfgStatus.softUseExpire
         self.xfrmPolicyUseTimeExpiry = ikeSALifetimeCfgStatus.hardUseExpire
      else:
         return


   def render( self ):
      print "Lifetime interval Config for XFRM Policy:"
      print "  Xfrm Policy Byte Limit Warning: %d" % \
         self.xfrmPolicyByteLimitWarning
      print "  Xfrm Policy Byte Limit Expiry: %d" % \
         self.xfrmPolicyByteLimitExpiry
      print "  Xfrm Policy Packet Limit Warning: %d" % \
            self.xfrmPolicyPacketLimitWarning
      print "  Xfrm Policy Packet Limit Expiry: %d" % \
            self.xfrmPolicyPacketLimitExpiry
      print "  Xfrm Policy Time Limit Warning: %d" % \
            self.xfrmPolicyTimeLimitWarning
      print "  Xfrm Policy Time Limit Expiry: %d" % \
            self.xfrmPolicyTimeLimitExpiry
      print "  Xfrm Policy Use Time Warning: %d" % \
            self.xfrmPolicyUseTimeWarning
      print "  Xfrm Policy Use Time Expiry: %d" % \
            self.xfrmPolicyUseTimeExpiry

class LifetimeCur( Model ):
   ''' CLI Model for Current Status of Policy in context of Lifetime'''

   bytesProcessed = Int( help="Number of bytes processed in tx & rx" )
   packetsProcessed = Int( help="Number of pkts processed in tx & rx" )
   policyAddTime = Int( help="Timestamp when the policy was added" )
   policyLastAccessTime = Int( help="Timestamp when policy was last accessed" )

   def fillLifetimeCurInfo( self, entMan, ikeSALifetimeCurStatus=None ):
      if ikeSALifetimeCurStatus is None:
         return
      self.bytesProcessed = ikeSALifetimeCurStatus.bytes
      self.packetsProcessed = ikeSALifetimeCurStatus.pkts
      self.policyAddTime = ikeSALifetimeCurStatus.addTime
      self.policyLastAccessTime = ikeSALifetimeCurStatus.useTime
      if self.policyLastAccessTime:
         # If the value is not zero, convert the use time to an utc time by
         # adding the difference between Tac utcNow and now
         self.policyLastAccessTime = int( self.policyLastAccessTime +
                                          Tac.utcNow() - Tac.now() )

   def render( self ):
      print "Policy current status in context of Lifetime :"
      print "  Bytes: %d" % self.bytesProcessed
      print "  Packets: %d" % self.packetsProcessed
      print "  Add time: %d" % self.policyAddTime
      print "  Last used time: %d" % self.policyLastAccessTime

class SecurityAssociation ( Model ):
   ''' CLI Model for show ip security state ike sa '''
   saddr = IpGenericAddress( help="Source IP Address" )
   daddr = IpGenericAddress( help="Dest IP Address" )
   spi = Int( help="Security Policy Index for ESP/AH" )
   proto = Enum( values=( IPSEC_PROTO_LIST ), 
                 help="IP Transform Protocol (AH/ESP)" )
   mode = Enum( values=( IPSEC_MODE_LIST ),
                help="mode of SA (tunnel, transport, beet, pass etc)" )
   reqid = Int( help="Request Id for a given SA pair" )
   replayWindow = Int( help="Sequence number window for replay protection" )
   seq = Int( help="Sequence Number to help with replay protection" )
   family = Enum( values=( 'ipv4', 'ipv6', 'unknownFamily' ), 
                  help="v4 or v6 family" )
   packetsOutOfReplayWindow = Int( help="Number of packets outside Replay Window" )
   replayErrors = Int( help="Number of Replay Errors" )
   statsIntegrityFailed = Int( help="Number of Integrity check failures for "
                                    "incoming packets" )
   lifetimeCurrent = Submodel( valueType=LifetimeCur, 
                               help="Lifetime interval current status in "
                                    "bytes/time:" )
   lifetimeCfg  = Submodel( valueType=LifetimeCfg, 
                            help="Lifetime interval config in bytes/time:" )

   def fillSecurityAssociationInfo( self, entMan, ikeStatus, ipsecCounterTable,
                                    saKey ):
      saStatus = ikeStatus.sa.get( saKey )
      if not saStatus:
         saStatus = ikeStatus.controllerSa.get( saKey )
      if not saStatus:
         return
      counterKey = Tac.Value( "IpsecCounters::IpsecCounterKey",
                              saStatus.reqid, saKey.id.spi, saKey.id.daddr )
      saCounterEntry = ipsecCounterTable.ipsecSACounterEntry.get( counterKey )
      lifetimeCfg = ikeStatus.lifetimeCfg.get( saKey )
      if not all( ( saCounterEntry, lifetimeCfg ) ):
         return
      said = saKey.id
      self.spi = said.spi
      self.daddr = said.daddr
      if ( IPSEC_PROTO_MAP.has_key( said.proto ) ):
         self.proto = IPSEC_PROTO_LIST[said.proto-50]
      else:
         self.proto = 'unknownProto'
      self.saddr = saStatus.saddr
      if ( IPSEC_MODE_MAP.has_key( saStatus.mode ) ):
         self.mode = IPSEC_MODE_LIST[saStatus.mode]
      else:
         self.mode = 'unknownMode'
      self.reqid = saStatus.reqid
      self.replayWindow = saStatus.replay_window
      self.seq = saStatus.seq
      if ( saKey.family == AF_INET ):
         self.family = 'ipv4'
      elif ( saKey.family == AF_INET6 ):
         self.family = 'ipv6'
      else:
         self.family = 'unknownFamily'

      self.packetsOutOfReplayWindow = saCounterEntry.replayWindowErrors
      self.replayErrors = saCounterEntry.replayErrors
      self.statsIntegrityFailed = saCounterEntry.integrityErrors

      self.lifetimeCurrent = LifetimeCur()
      self.lifetimeCurrent.fillLifetimeCurInfo( entMan, saCounterEntry )

      self.lifetimeCfg = LifetimeCfg()
      self.lifetimeCfg.fillLifetimeCfgInfo( lifetimeCfg )

class PolicySelector( Model ):
   ''' Policy Selector for a Flow. '''

   daddr = IpGenericAddress( help="Dest IP Address for Policy" )
   saddr = IpGenericAddress( help="Source IP Address for Policy" )
   index = Int( help="Index of policy in the IPSec Engine" )
   direction = Enum( values=( IPSEC_DIRECTION_LIST ),
                     help="Direction- in/out/fwd" )
   family = Enum( values=( 'ipv4', 'ipv6', 'unknownFamily' ), 
                  help="v4 or v6 family" )
   proto = Enum( values=( IPSEC_PROTO_LIST ), 
                 help="IP Transform Protocol (AH/ESP)" )
 
   def fillPolicySelectorInfo( self, ikeSecurityPolicySelectorStatus=None ):
      if ikeSecurityPolicySelectorStatus is None:
         return
      self.daddr = ikeSecurityPolicySelectorStatus.daddr
      self.saddr = ikeSecurityPolicySelectorStatus.saddr
      if ( IPSEC_DIRECTION_MAP.has_key( ikeSecurityPolicySelectorStatus.dir ) ):
         self.direction = IPSEC_DIRECTION_LIST[ikeSecurityPolicySelectorStatus.dir]
      else: 
         self.direction = 'unknownDir'
      if ( ikeSecurityPolicySelectorStatus.family == AF_INET ):
         self.family = 'ipv4'
      elif ( ikeSecurityPolicySelectorStatus.family == AF_INET6 ):
         self.family = 'ipv6'
      else:
         self.family = 'unknownFamily'
      if ( IPSEC_PROTO_MAP.has_key( ikeSecurityPolicySelectorStatus.proto ) ):
         self.proto = IPSEC_PROTO_LIST[ikeSecurityPolicySelectorStatus.proto-50]
      else:
         self.proto = 'unknownProto'
      self.index = ikeSecurityPolicySelectorStatus.index

   def render( self ):
      print "  Policy Selector for the Flow."
      print "  Dest ip addr: %s" % self.daddr
      print "  Source ip addr: %s" % self.saddr
      proto = IPSEC_DIRECTION_LIST.index(self.direction)
      print "  Direction: %s" % IPSEC_DIRECTION_MAP[proto]
      print "  Family: %s" % self.family
      proto = IPSEC_PROTO_LIST.index(self.proto)
      print "  Proto: %s" % IPSEC_PROTO_MAP[50+proto]
      print "  Index: %d" % self.index

class SecurityPolicy ( Model ):
   ''' CLI Model for show ip security state ike policy '''
   selector = Submodel( valueType=PolicySelector, 
                        help="Policy Selector for a Flow" )
   priority = Enum( values=( IPSEC_PRIORITY_LIST ),
                    help="SA Priority for this Policy" )
   flags = Enum( values= ( IPSEC_XFRMPOLICYFLAGS_LIST ),
                 help="Override global policy vs allow matching ICMP payloads" )

   def fillSecurityPolicyInfo( self, ikeSecurityPolicyStatus=None ):
      if ikeSecurityPolicyStatus is not None:
         self.selector = PolicySelector()
         self.selector.fillPolicySelectorInfo( ikeSecurityPolicyStatus.selector )
         if ( IPSEC_PRIORITY_MAP.has_key( ikeSecurityPolicyStatus.priority ) ):
            self.priority = IPSEC_PRIORITY_LIST[ikeSecurityPolicyStatus.priority]
         else: 
            self.priority = 'unknownPriority'
         self.flags = ikeSecurityPolicyStatus.flags
         if ( IPSEC_XFRMPOLICYFLAGS_MAP.has_key( ikeSecurityPolicyStatus.flags ) ):
            self.flags = IPSEC_XFRMPOLICYFLAGS_LIST[ikeSecurityPolicyStatus.flags]
         else: 
            self.flags = 'unknownPolicy'
      else:
         return


   def renderSP( self, name ):
      print "Ipsec Security Policy State :"
      print "  Name: %s" % name
      flags = IPSEC_XFRMPOLICYFLAGS_LIST.index(self.flags)
      print "  Flags: %s" % IPSEC_XFRMPOLICYFLAGS_MAP[flags]
      priority = IPSEC_PRIORITY_LIST.index(self.priority)
      print "  Priority: %s" % IPSEC_PRIORITY_MAP[priority]
      if self.selector is not None:
         self.selector.render( )

def getIpsecConnList( ikeStatus=None, tunnelIntfStatusDir=None, 
                      ipsecPathStatus=None, vrfName=None, ):
   connList = []

   def fillConnectionList( profileName ):
      profile = profiles.get( profileName )
      if not profile:
         return
      profileModel = IpsecAppliedProfile()
      profileModel.fillIpsecAppliedProfileInfo( profile,
                                                vrfName, ipsecPathStatus,
                                                tunnelIntfStatusDir )
      for connName in profileModel.ipsecConnectionDetails.keys():
         t9( "fillConnectionList for connEntry: ", connName )
         connEntry = profileModel.ipsecConnectionDetails[connName]
         connList.append( connEntry )

   if ikeStatus is not None:
      profiles = ikeStatus.appliedProfiles
      for profileName in profiles:
         t9( "fillTunnelList for", profileName )
         fillConnectionList( profileName )

   return connList

class IpsecConnection( Model ):
   ''' CLI Model for Tunnel info scraped from strongswan statusall'''
   __revision__ = 2
   connName = Str( help="Name of the IPsec connection" )
   ipsecInputDataBytes = Int( help="Ipsec Input Data in bytes" )
   ipsecInputDataPkts = Int( help="Ipsec Input Data in packets" )
   ipsecOutputDataBytes = Int( help="Ipsec Output Data in bytes" )
   ipsecOutputDataPkts = Int( help="Ipsec Output Data in packets" )
   tunnelDict = Dict( help="Tunnel interface List for this connection",
                      keyType=Interface,
                      valueType=str, optional=True )
   pathDict = Dict( help="Path name for this connection", keyType=str,
                    valueType=str, optional=True )
   saddr = IpGenericAddress( help="Source IP Address" )
   daddr = IpGenericAddress( help="Destination IP Address" )
   tunnelNs = Str( help="Tunnel namespace for this profile" )
   inboundSA = Submodel( valueType=SecurityAssociation, 
                                   help="SA for an inbound Flow",
                                   optional=True )
   outboundSA = Submodel( valueType=SecurityAssociation, 
                                   help="SA for an outbound Flow",
                                   optional=True )
   upTimeStr = Str( help="Ipsec connection uptime" )
   rekeyTime = Int( help="Seconds remaining until next rekey" )

   def initConnection( self, connName ):
      self.connName = connName
      self.ipsecInputDataBytes = 0
      self.ipsecInputDataPkts = 0
      self.ipsecOutputDataBytes = 0
      self.ipsecOutputDataPkts = 0
      self.inboundSA = None
      self.outboundSA = None
      self.upTimeStr = "N/A"
      self.rekeyTime = 0

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'ipsecIkeState' ] = 'unknownState'
         dictRepr[ 'ikev1StateTime' ] = 0
         dictRepr[ 'ikev2StateTime' ] = 0
         dictRepr[ 'ikev1StateTimeUnits' ] = 'hours'
         dictRepr[ 'ikev2StateTimeUnits' ] = 'hours'
      return dictRepr
   
class IpsecConnectionState( Model ):
   ''' CLI Model for show ip security connection '''
   __revision__ = 2
   _detailed = Bool( help="Display detailed output" )
   connections = Dict( help="A mapping between tunnel name and"
                       "IPsec tunnel",
                       keyType=str,
                       valueType=IpsecConnection )

   def getIpsecSessionStats( self, ikeStatus, reqid, ipsecCounterTable, inSa=True ):
      numBytes = 0
      numPkts = 0
      ipsecUptime = 0

      sessionKey = Tac.Value( "IpsecCounters::IpsecSessionKey", reqid )
      ipsecInfo = ipsecCounterTable.ipsecSessionCounterEntry.get( sessionKey )
      if not ipsecInfo:
         return numBytes, numPkts, ipsecUptime

      if inSa:
         numBytes = ipsecInfo.bytesIn
         numPkts = ipsecInfo.pktsIn
      else:
         numBytes = ipsecInfo.bytesOut
         numPkts = ipsecInfo.pktsOut
      ipsecUptime = ipsecInfo.lastUpTime

      return numBytes, numPkts, ipsecUptime

   def getIpsecConnEntry( self, ipsecConnTable, vrfName, reqid ):
      connVrfStatus = ipsecConnTable.connVrfStatus.get( vrfName )
      if connVrfStatus is not None:
         return connVrfStatus.connEntries.get( reqid )
      return None

   def fillIpsecConnectionStateInfo( self, entMan, vrfName, ikeStatus=None,
                                     ipsecConnTable=None,
                                     tunnelIntfStatusDir=None, 
                                     ipsecCounterTable=None,
                                     ipsecPathStatus=None,
                                     tunnelName=None, 
                                     pathName=None,
                                     pathPeerIp=None,
                                     pathPeerFqdn=None,
                                     tunnels=False,
                                     paths=False,
                                     detailed=False ):
      # Nothing to do if ikeStatus is invalid
      if ikeStatus is None:
         return

      self._detailed = detailed
      connList = []
      vrfList = []
      connList = getIpsecConnList( ikeStatus, tunnelIntfStatusDir, 
                                   ipsecPathStatus, vrfName )
      if vrfName != 'default':
         vrfList.append( 'ns-' + vrfName )
      else:
         vrfList.append( vrfName )

      def fillConnDetails( connEntry, tunnelName=None, pathName=None,
                           tunnels=False, paths=False ):
         connection = IpsecConnection()
         connection.initConnection( connName )
         connection.saddr = connEntry.srcAddr
         connection.daddr = connEntry.dstAddr
         connection.tunnelNs = connEntry.tunnelNs

         # display info only for given tunnel, if specified.
         if tunnelName is not None:
            assert tunnelName in connEntry.tunnelDict
            connection.tunnelDict[ tunnelName ] = connEntry.tunnelDict[ tunnelName ]
         elif pathName is not None:
            assert pathName in connEntry.pathDict
            connection.pathDict[ pathName ] = connEntry.pathDict[ pathName ]
         elif tunnels:
            connection.tunnelDict = connEntry.tunnelDict
         elif paths:
            connection.pathDict = connEntry.pathDict
         else:
            connection.tunnelDict = connEntry.tunnelDict
            connection.pathDict = connEntry.pathDict

         t9( "Created Tunnel Entry for %s, src=%s dst=%s" %
               ( connName, connection.saddr, connection.daddr ) )
         return connection

      # Get IPsec connection details for all connections
      for connEntry in connList:
         connection = None
         connName = connEntry.connectionName
         t9( "filling conn details for conn: ", connName )
         if not tunnelName and not pathName and not pathPeerIp:
            connection = fillConnDetails( connEntry, tunnels=tunnels, paths=paths )
            self.connections[connName] = connection
            continue
         if tunnelName in connEntry.tunnelDict:
            # user specified tunnel name. get info for only this entry
            connection = fillConnDetails( connEntry, tunnelName=tunnelName )
            self.connections[connName] = connection
            break
         elif pathName in connEntry.pathDict:
            connection = fillConnDetails( connEntry, pathName=pathName )
            self.connections[connName] = connection
            break
         elif pathPeerIp:
            pathEntry = ipsecPathStatus.ipsecPathRemoteIp[ pathPeerIp ]
            for path in pathEntry.paths:
               if path in connEntry.pathDict:
                  connection = fillConnDetails( connEntry, pathName=path )
                  self.connections[connName] = connection
                  break
            continue

      t9( "vrfList is: ", vrfList )
      # pylint: disable-msg=R1702
      for vrf in vrfList:
         vrfStatus = ikeStatus.vrfStatus.get( vrf )
         if vrfStatus is None:
            t9( "vrfStatus has no vrf key named : ", vrf )
            continue
         for saKey in chain( vrfStatus.sa, vrfStatus.controllerSa ):
            sa = SecurityAssociation()
            sa.fillSecurityAssociationInfo( entMan, vrfStatus, ipsecCounterTable,
                                            saKey )
            if sa.reqid is None:
               # fillSecurityAssociationInfo did not fill the sa info ( encountered
               # an invalid case and returned before assigned the values )
               # skip finding the matching connection in this case
               continue
            srcAddr = "%s" % sa.saddr
            dstAddr = "%s" % sa.daddr
            lifetimeCur = sa.lifetimeCurrent
            ipsecConnEntry = self.getIpsecConnEntry( ipsecConnTable,
                                                     vrf, sa.reqid )
            t9( "Find Tunnel for SA: src=%s dst=%s" % (srcAddr, dstAddr) )
            for connectionName in self.connections:
               connection = self.connections[connectionName]
               tunnelSaddr = "%s" % connection.saddr
               tunnelDaddr = "%s" % connection.daddr
               t9( "tunnel is %s(s=%s, d=%s) sa=%s sd=%s" % ( connectionName,
                     tunnelSaddr, tunnelDaddr, srcAddr, dstAddr ) )
               if tunnelSaddr == srcAddr and tunnelDaddr == dstAddr:
                  if not connection.ipsecOutputDataBytes:
                     # enter this block only once. Assign the values in the session
                     # stats to counters and ipsecUptime
                     numBytes, numPkts, ipsecUptime = self.getIpsecSessionStats(
                                                      vrfStatus, sa.reqid, 
                                                      ipsecCounterTable, False )
                     connection.ipsecOutputDataBytes = numBytes
                     connection.ipsecOutputDataPkts = numPkts
                     connection.upTimeStr = ipsecTimeString( 
                                            int( Tac.utcNow() - ipsecUptime ),
                                            "0 seconds", [ ( "seconds", 60 ),
                                            ( "minutes", 60 ), ( "hours", 24 ),
                                            ( "days", 365 ) ] )
                  if lifetimeCur:
                     connection.ipsecOutputDataBytes += lifetimeCur.bytesProcessed
                     connection.ipsecOutputDataPkts += lifetimeCur.packetsProcessed
                  if ipsecConnEntry:
                     outSa = ipsecConnEntry.activeOutSa
                     if ( outSa and outSa.sakey == saKey ):
                        # activeOutSa in ipsecConnEntry should be the correct sa we
                        # should use for the connection.outboundSA.
                        connection.outboundSA = sa
                        if lifetimeCur and sa.lifetimeCfg:
                           rekeyTime = int( lifetimeCur.policyAddTime +
                                 sa.lifetimeCfg.xfrmPolicyTimeLimitWarning -
                                 Tac.utcNow() )
                           if ( not connection.rekeyTime or \
                                connection.rekeyTime > rekeyTime ):
                              # update the connection.rekeyTime if it has not been
                              # assigned, or the value assigned by the latest inSa
                              # is higher than the rekeyTime of this activeOutSa
                              connection.rekeyTime = rekeyTime
                  if not connection.outboundSA:
                     # guarantee that a connection has an outboundSA even if
                     # the ipsecConnEntry is not ready for the latest SA
                     connection.outboundSA = sa
                  break
               if tunnelSaddr == dstAddr and tunnelDaddr == srcAddr:
                  if not connection.ipsecInputDataBytes:
                     numBytes, numPkts, ipsecUptime = self.getIpsecSessionStats(
                                                      vrfStatus, sa.reqid, 
                                                      ipsecCounterTable, True )
                     connection.ipsecInputDataBytes = numBytes
                     connection.ipsecInputDataPkts = numPkts
                  if lifetimeCur:
                     connection.ipsecInputDataBytes += lifetimeCur.bytesProcessed
                     connection.ipsecInputDataPkts += lifetimeCur.packetsProcessed
                  if ipsecConnEntry:
                     inSas = ipsecConnEntry.inSa.values()
                     if ( inSas and inSas[ -1 ].sakey == saKey ):
                        # use the last ( newest ) sa in inSa queue as the
                        # connection.inboundSA
                        connection.inboundSA = sa
                        if lifetimeCur and sa.lifetimeCfg:
                           rekeyTime = int( lifetimeCur.policyAddTime +
                                 sa.lifetimeCfg.xfrmPolicyTimeLimitWarning -
                                 Tac.utcNow() )
                           if ( not connection.rekeyTime or \
                                connection.rekeyTime > rekeyTime ):
                              # update the connection.rekeyTime if this inSa 
                              # rekeyTime is lower than the rekeyTime assigned 
                              # by the active outSa. We want to use the lower 
                              # rekeyTime of the inSa and the outSa as the 
                              # connection.rekeyTime
                              connection.rekeyTime = rekeyTime
                  if not connection.inboundSA:
                     # guarantee that a connection has an inboundSA
                     connection.inboundSA = sa
                  break

      # pylint: disable-msg=W0612
      for connectionName, connInfo in self.connections.items():
         if len ( self.connections[connectionName]['tunnelDict'] ) == 0 and \
            len ( self.connections[connectionName]['pathDict'] ) == 0 :
            del self.connections[ connectionName ]

   def print_non_detailed_view( self, connection ):

      def getBriefTime( timeStr ):
         if timeStr == "0 seconds":
            return 'N/A'
         return timeStr.split( ', ', 1 )[ 0 ]

      upTime = getBriefTime( connection.upTimeStr )
      rekeyTimeStr = ipsecTimeString( connection.rekeyTime,
                                      "0 seconds", [ ( "seconds", 60 ),
                                      ( "minutes", 60 ), ( "hours", 24 ) ] )
      rekeyTime = getBriefTime( rekeyTimeStr )

      if connection.tunnelDict.keys():
         connNameDict = collections.OrderedDict( 
                                          sorted( connection.tunnelDict.items() ) )
      elif connection.pathDict.keys():
         connNameDict = collections.OrderedDict(
                                          sorted( connection.pathDict.items() ) )
      if len( connNameDict ) == 0:
         return

      if len( connNameDict ) > 1:
         print connStateFormatStr % ( connNameDict.keys()[0] + ',',
               connection.saddr, 
               connection.daddr, 
               connNameDict.values()[0],
               upTime, '%s bytes' % connection.ipsecInputDataBytes, 
               '%s bytes' % connection.ipsecOutputDataBytes, rekeyTime )
         if len( connNameDict ) == 2:
            # only two entries to print
            print connStateFormatStr % ( connNameDict.keys()[1],
                  "", "", "", "",
                  '%s pkts' % connection.ipsecInputDataPkts,
                  '%s pkts' % connection.ipsecOutputDataPkts, '')
            return
         print connStateFormatStr % ( connNameDict.keys()[1] + ',',
               "", "", "", "",
               '%s pkts' % connection.ipsecInputDataPkts,
               '%s pkts' % connection.ipsecOutputDataPkts, '' )
      else:
         print connStateFormatStr % ( connNameDict.keys()[0],
               connection.saddr,
               connection.daddr,
               connNameDict.values()[0],
               upTime, '%s bytes' % connection.ipsecInputDataBytes, 
               '%s bytes' % connection.ipsecOutputDataBytes, rekeyTime )
         print connStateFormatStr % ( "", "", "", "", "",
               '%s pkts' % connection.ipsecInputDataPkts,
               '%s pkts' % connection.ipsecOutputDataPkts, '' )
         return
         
      # Print the rest of the tunnel entries
      for i in range( 2, len( connNameDict ) ):
         if i == ( len( connNameDict ) - 1 ):
            print connNameDict.keys()[i]
         else:
            print connNameDict.keys()[i] + ',',

   def print_SA_info( self, sa, whichSA='Inbound' ):
      if sa:
         lcfg = sa.lifetimeCfg
         lcur = sa.lifetimeCurrent
         print "   %s SPI 0x%x:" % ( whichSA, socket.ntohl( sa.spi ) )
         print "      request id %d, mode %s replay-window %d, seq 0x%x" % \
               (sa.reqid, sa.mode, sa.replayWindow, sa.seq)
         print "      stats errors:"
         print "         replay-window %d, replay %d, integrity_failed %d" % \
               (sa.packetsOutOfReplayWindow, sa.replayErrors, 
                     sa.statsIntegrityFailed)
         print "      lifetime config:"
         print "         softlimit %d bytes, hardlimit %d bytes" % \
               (lcfg.xfrmPolicyByteLimitWarning, lcfg.xfrmPolicyByteLimitExpiry)
         print "         softlimit %d pkts, hardlimit %d pkts" % \
               (lcfg.xfrmPolicyPacketLimitWarning, lcfg.xfrmPolicyPacketLimitExpiry)
         print "         expire add soft %d secs, hard %d secs" % \
               (lcfg.xfrmPolicyTimeLimitWarning, lcfg.xfrmPolicyTimeLimitExpiry)
         print "      lifetime current:"
         print "         %d bytes, %d pkts" % \
               (lcur.bytesProcessed, lcur.packetsProcessed)
         if lcur.policyAddTime:
            aTime = "%s" % time.ctime(lcur.policyAddTime)
         else:
            aTime = '-'
         if lcur.policyLastAccessTime:
            uTime = "%s" % time.ctime(lcur.policyLastAccessTime)
         else:
            uTime = '-'
         print "         add time %s, use time %s" % (aTime, uTime)

   def print_detailed_view( self, connection ):
      if connection.tunnelDict.keys():
         connNameDict = collections.OrderedDict( 
                                          sorted( connection.tunnelDict.items() ) )
      elif connection.pathDict.keys():
         connNameDict = collections.OrderedDict( 
                                          sorted( connection.pathDict.items() ) )
      numConnections = len( connNameDict )
      for i in range( 0, numConnections ):
         if i == numConnections-1:
            print "%s:" % connNameDict.keys()[i]
         else:
            print "%s," % connection.tunnelDict.keys()[i],
      print "   source address %s, dest address %s" % ( connection.saddr,
                                                        connection.daddr )
      if connection.tunnelDict.keys():
         print "   state: %s" % ( connection.tunnelDict.values()[0] )
      else:
         print "   state: %s" % ( connection.pathDict.values()[0] )
      print "   uptime: %s" % ( connection.upTimeStr )
      self.print_SA_info( connection.inboundSA, whichSA='Inbound' )
      self.print_SA_info( connection.outboundSA, whichSA='Outbound' )

   def render( self ):
      # Get the tunnel interfaces that are in use from app profiles
      firstTime = True

      # Basically, try to print in sorted tunnel list order as much
      # as possible.
      connectionDict = dict()
      for connectionName in self.connections:
         connection = self.connections[connectionName]
         # get the first tunnelName from sorted list.
         if connection.tunnelDict.keys():
            tunnelName = sorted( i.title() for i in connection.tunnelDict )[0]
            connectionDict[ tunnelName ] = connection
         elif connection.pathDict.keys():
            pathName = sorted( i.title() for i in connection.pathDict )[0]
            connectionDict[ pathName ] = connection

      for connectionName in Arnet.sortIntf( connectionDict ):
         connection = connectionDict[connectionName]
         if self._detailed:
            self.print_detailed_view( connection )
         else:
            if firstTime is True:
               firstTime = False
               print connStateFormatStr % ( 'Tunnel', 'Source', 'Dest', 
                  'Status', 'Uptime', 'Input', 'Output', 'Rekey Time' )
            self.print_non_detailed_view( connection )

class IpsecPeer( Model ):
   ''' CLI Model for one Ipsec key controller peer'''
   peerIp = IpGenericAddress( help="Peer IP Address" )
   peerStatus = Enum( values=( IPSEC_KEY_CONTROLLER_PEER_STATUS ),
                      help="Status of the IPsec key controller peer" )
   dhGroup = Int( help="Diffie-Hellman (DH) group number" )
   encryptAlgo = Enum( values=IPSEC_ENCRYPTION_MAP,
                       help='Encryption Algorithm' )
   hashAlgo = Enum( values=IPSEC_INTEGRITY_MAP,
                    help="INTEGRITY Algorithm" )
   uptime = Float( help="Seconds since the peer came up" )
   lastKeyUptime = Int( help="Seconds since the last key material received" )
   initiator = Bool( help="The peer is the initiator" )

   def fillPeerInfo( self, peerConnState ):
      self.peerIp = peerConnState.remoteIp
      self.dhGroup = peerConnState.dhGroup

      if peerConnState.peerConnStatus != IpsecPeerConnStatusEnum.peerUp:
         self.uptime = 0.0
      else:
         self.uptime = Tac.utcNow() - peerConnState.upTime
      if not peerConnState.lastDhKeyAddTime:
         self.lastKeyUptime = 0
      else:
         self.lastKeyUptime = int( Tac.utcNow() - peerConnState.lastDhKeyAddTime )
      if peerConnState.encryptAlgo in IPSEC_ENCRYPTION_MAP:
         self.encryptAlgo = peerConnState.encryptAlgo
      else:
         self.encryptAlgo = 'unknownEncr'
      if peerConnState.hashAlgo in IPSEC_INTEGRITY_MAP:
         self.hashAlgo = peerConnState.hashAlgo
      else:
         self.hashAlgo = 'unknownIntegrity'
      if peerConnState.peerConnStatus == IpsecPeerConnStatusEnum.peerUp:
         self.peerStatus = IpsecPeerConnStatusEnum.peerUp
      elif peerConnState.peerDownReason in IPSEC_KEY_CONTROLLER_PEER_STATUS:
         self.peerStatus = peerConnState.peerDownReason
      else:
         self.peerStatus = 'unknownPeerState'
      self.initiator = peerConnState.initiator

def getCryptoSuiteStr( encryptAlgo, hashAlgo ):
   if encryptAlgo == IpsecEspAlgorithmEnum.aes128gcm64:
      hashAlgoStr = '64bit Hash'
   elif encryptAlgo == IpsecEspAlgorithmEnum.aes128gcm128 or \
        encryptAlgo == IpsecEspAlgorithmEnum.aes256gcm128:
      hashAlgoStr = '128bit Hash'
   else:
      hashAlgoStr = IPSEC_INTEGRITY_MAP[ hashAlgo ]
   return "%s , %s" % ( IPSEC_ENCRYPTION_MAP[ encryptAlgo ], hashAlgoStr )

class IpsecPeerState( Model ):
   ''' CLI Model for show ip security key controller peer '''
   _detailed = Bool( help="Display detailed output" )
   peers = Dict( help="A mapping between peer address and "
                      "IPsec peer info of the key controller",
                 keyType=IpGenericAddress,
                 valueType=IpsecPeer )

   def fillPeersInfo( self, status, detailed, peerIpStr=None ):
      self._detailed = detailed
      if peerIpStr is None:
         for peerIp in status.ipsecPeerConnectionState:
            ipsecPeer = IpsecPeer()
            ipsecPeer.fillPeerInfo( status.ipsecPeerConnectionState[ peerIp ] )
            self.peers[ peerIp ] = ipsecPeer
      else:
         peerIp = Tac.Value( "Arnet::IpGenAddr", stringValue=peerIpStr )
         if peerIp in status.ipsecPeerConnectionState:
            ipsecPeer = IpsecPeer()
            ipsecPeer.fillPeerInfo( status.ipsecPeerConnectionState[ peerIp ] )
            self.peers[ peerIp ] = ipsecPeer

   def render( self ):
      if self._detailed:
         peerDetailFormatStr = "Peer: %s\nStatus: %s\nDH Group: %s\n"\
                               "Crypto suite: %s\nUp time: %s\n"\
                               "Last key material received: %s\nInitiator: %s\n"
         for peerIp, peer in sorted( self.peers.iteritems() ):
            uptimeStr = "N/A"
            lastKeyUptimeStr = "N/A"
            if peer.uptime > 0:
               uptimeStr = ipsecTimeString( int( peer.uptime ),
                                            "0 seconds", [ ( "seconds", 60 ),
                                            ( "minutes", 60 ), ( "hours", 24 ),
                                            ( "days", 365 ) ] )
            if peer.lastKeyUptime > 0:
               lastKeyUptimeStr = \
                     ipsecTimeString( peer.lastKeyUptime,
                                      "0 seconds", [ ( "seconds", 60 ),
                                      ( "minutes", 60 ), ( "hours", 24 ),
                                      ( "days", 365 ) ] )
            peerCryptoSuiteStr = getCryptoSuiteStr( peer.encryptAlgo, peer.hashAlgo )
            print peerDetailFormatStr % ( peerIp,
                        IPSEC_KEY_CONTROLLER_PEER_STATUS[ peer.peerStatus ],
                        peer.dhGroup, peerCryptoSuiteStr,
                        uptimeStr, lastKeyUptimeStr, peer.initiator )
         
      else:
         print peerFormatStr % ( "Peer", "DH Group", "Crypto Suite", "Status" )
         print peerFormatStr % ( "----", "--------", "------------", "------" )
         for peerIp, peer in sorted( self.peers.iteritems() ):
            peerStatusStr = "down"
            if peer.peerStatus == IpsecPeerConnStatusEnum.peerUp:
               peerStatusStr = "up"
            peerCryptoSuiteStr = getCryptoSuiteStr( peer.encryptAlgo, peer.hashAlgo )
            print peerFormatStr % ( peerIp, peer.dhGroup, peerCryptoSuiteStr,
                                    peerStatusStr )

class IpsecLocalKey( Model ):
   ''' CLI Model for Ipsec key controller local key info '''
   dhGroup = Enum( values=IPSEC_DHGROUP_NUMBER_MAP,
                   help="Diffie-Hellman (DH) group number" )
   encryptAlgo = Enum( values=IPSEC_ENCRYPTION_MAP,
                       help='Encryption Algorithm' )
   hashAlgo = Enum( values=IPSEC_INTEGRITY_MAP,
                    help="Integrity Algorithm" )
   firstKeyAddTime = Float( help="Timestamp the initial DH key was created" )
   lastKeyUptime = Int( help="Seconds since the last key was created" )
   rekeyTime = Int( help="Seconds remaining until next rekey" )

   def fillLocalKeyInfo( self, config, status ):
      keyControllerCfg = config.keyController
      localKeyState = status.localKeyState
      if keyControllerCfg.dhGroup in IPSEC_DHGROUP_NUMBER_MAP:
         self.dhGroup = keyControllerCfg.dhGroup
      else:
         self.dhGroup = 'unknownDhGroup'
      self.firstKeyAddTime = localKeyState.firstDhKeyAddTime
      if not localKeyState.lastDhKeyAddTime:
         self.lastKeyUptime = 0
      else:
         self.lastKeyUptime = int( Tac.utcNow() - localKeyState.lastDhKeyAddTime )
      if not self.lastKeyUptime:
         self.rekeyTime = 0
      else:
         nextRekeyTime = int( keyControllerCfg.dhLifetime - self.lastKeyUptime )
         self.rekeyTime = nextRekeyTime if nextRekeyTime > 0 else 0

      self.encryptAlgo = 'unknownEncr'
      self.hashAlgo = 'unknownIntegrity'
      profileName = config.keyController.profileName
      profileCfg = config.ipsecProfile.get( profileName )
      if profileCfg:
         saCfgs = profileCfg.securityAssoc.values()
         for sa in saCfgs:
            saParams = sa.saParams
            if ( IPSEC_ENCRYPTION_MAP.has_key( saParams.espAes ) ):
               self.encryptAlgo = saParams.espAes
            if ( IPSEC_INTEGRITY_MAP.has_key( saParams.espSha ) ):
               self.hashAlgo = saParams.espSha
            # the profile should have only one SA
            break

   def render( self ):
      firstKeyAddTimeStr = "N/A"
      lastKeyUptimeStr = "N/A"
      rekeyTimeStr = "N/A"
      if self.firstKeyAddTime > 0:
         firstKeyAddTimeStr = time.asctime( time.gmtime( self.firstKeyAddTime ) )
      if self.lastKeyUptime > 0:
         lastKeyUptimeStr = ipsecTimeString( self.lastKeyUptime,
                                             "0 seconds", [ ( "seconds", 60 ),
                                             ( "minutes", 60 ), ( "hours", 24 ),
                                             ( "days", 365 ) ] )
      if self.rekeyTime > 0:
         rekeyTimeStr = ipsecTimeString( self.rekeyTime,
                                         "0 seconds", [ ( "seconds", 60 ),
                                         ( "minutes", 60 ), ( "hours", 24 ) ] )
      cryptoSuiteStr = getCryptoSuiteStr( self.encryptAlgo, self.hashAlgo )
      localKeyFormatStr = "DH group: %s\n" \
                          "Crypto suite: %s\n" \
                          "Initial DH key created: %s\n" \
                          "Last DH key created: %s\n" \
                          "DH rekey: %s"
      print localKeyFormatStr % ( IPSEC_DHGROUP_NUMBER_MAP[ self.dhGroup ],
                                  cryptoSuiteStr, firstKeyAddTimeStr,
                                  lastKeyUptimeStr, rekeyTimeStr )

class IpsecLocalKeyState( Model ):
   ''' CLI Model for configured Ipsec key controller local key '''
   keyController = Submodel( valueType=IpsecLocalKey, 
                             help="Configured IPsec Key Controller",
                             optional=True )
   def fillLocalKeyState( self, config, status ):
      self.keyController = None
      keyControllerCfg = config.keyController
      if keyControllerCfg.configured:
         self.keyController = IpsecLocalKey()
         self.keyController.fillLocalKeyInfo( config, status )

   def render( self ):
      if self.keyController is not None:
         self.keyController.render()

