# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import TunnelIntfCli
import BasicCli
import CliCommand
import CliMatcher
from CliMode.Ipsec import IkePolicyMode
from CliMode.Ipsec import IpsecProfileMode
from CliMode.Ipsec import IpSecurityMode
from CliMode.Ipsec import SecurityAssociationMode
from CliMode.Ipsec import SecurityAssociationVxlanMode
from CliMode.Ipsec import KeyControllerMode
import CliParser
import CliToken.Ip
import ConfigMount
import IpGenAddrMatcher
import LazyMount
import Tac
import Toggles.IpsecToggleLib
import ReversibleSecretCli

#Assigning globals.
ikeConfigDir = None
ikeStatusDir = None
allVrfStatusLocal = None
tunIntfConfigDir = None
tunIntfStatusDir = None
ikePreShareKey = None
hwCapabilities = None

dpdActionEnum = Tac.Type( 'Ipsec::Ike::IpsecDpdAction' )
IpsecHmacAlgorithmEnum = Tac.Type( 'Ipsec::IpsecHmacAlgorithm' )
IpsecAuthTypeEnum = Tac.Type( 'Ipsec::Ike::IpsecAuthType' )
IpsecEspAlgorithmEnum = Tac.Type( 'Ipsec::IpsecEspAlgorithm' )
IpsecDhGroupEnum = Tac.Type( 'Ipsec::IpsecDhGroup' )
IpsecEncapProtocolEnum = Tac.Type( 'Ipsec::Ike::IpsecEncapProtocol' )
IpsecSaTypeEnum = Tac.Type( 'Ipsec::IpsecSaType' )

#------------------------------------------------------------------------
# Hook for TunnelIntf to confirm if an Ipsec profile is present in config.
#-------------------------------------------------------------------------
def isIpsecProfilePresent( profileName ):
   errMessage = ''
   if not ikeConfig().ipsecProfile.has_key( profileName ):
      errMessage = 'Ipsec profile %s has not been created' % profileName
      return ( False, errMessage )
   return ( True, errMessage )

def ikeConfig():
   return ikeConfigDir

def ikeStatus():
   return ikeStatusDir

def createOrGetIpsecProfileConfig( profile ):
   profile = ikeConfig().ipsecProfile.get( profile )
   if profile is None:
      profile = ikeConfig().ipsecProfile.newMember ( profile )
   return profile

def createOrUpdateTransformSetConfig( tsKey, espAes, espSha ):
   ts = ikeConfig().securityAssoc.get( tsKey )
   if ts is None:
      ts = ikeConfig().securityAssoc.newMember( tsKey, espAes, espSha )
   else:
      ts.espAes = espAes
      ts.espSha = espSha

def getOrCreateAppliedPolicy( policy ):
   appliedPolicy = ikeStatus().appliedPolicy.get( policy )
   if appliedPolicy is None:
      appliedPolicy  = ikeStatus().appliedPolicy.newMember( policy )
   return appliedPolicy

def getOrCreateAppliedSa( sa ):
   appliedSa = ikeStatus().appliedSa.get( sa )
   if appliedSa is None:
      appliedSa  = ikeStatus().appliedSa.newMember( sa )
   return appliedSa

def deletePolicyMapping( policy, profile ):
   policyMap = getOrCreateAppliedPolicy( policy )
   del policyMap.appliedPolicy[ profile.profile ]
   if not policyMap.appliedPolicy:
      del ikeStatus().appliedPolicy[ policy ]

def deleteSaMapping( sa, profile ):
   applied = getOrCreateAppliedSa( sa )
   del applied.appliedSec[ profile.profile ]
   if not applied.appliedSec:
      del ikeStatus().appliedSa[sa]

def vxlanSecurityGuard( mode, token ):
   if hwCapabilities.vxlanSecuritySupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
class IpsecIkeConfigMode( IkePolicyMode, BasicCli.ConfigModeBase ):
   name = 'ISAKMP configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name ):
      self.ikeName = name
      self.ike = self.getOrCreateIkeConfig( ikeConfig() )
      self.ikeParams = Tac.nonConst( self.ike.ikeParams )
      IkePolicyMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateIkeConfig( self, config ):
      ikePolicyConfig = config.ikePolicy.get( self.ikeName )
      if ikePolicyConfig is None:
         ikePolicyConfig = config.ikePolicy.newMember( self.ikeName )
      return ikePolicyConfig

   def onExit( self ):
      if self.ike.ikeParams != self.ikeParams:
         self.ike.ikeParams = self.ikeParams

class IpsecVxlanSecurityAssocConfigMode( SecurityAssociationVxlanMode,
                                         BasicCli.ConfigModeBase ):
   name = 'IPsec Vxlan SA configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name ):
      self.saName = name
      self.secAssoc = self.getOrCreateSAConfig( ikeConfig() )
      self.saParams = Tac.nonConst( self.secAssoc.saParams )
      if self.saParams.saType != IpsecSaTypeEnum.IpsecVxlanSa:
         self.saParams.saType = IpsecSaTypeEnum.IpsecVxlanSa
      # Only 256 bit hash integrity currently supported
      if self.saParams.espSha != IpsecHmacAlgorithmEnum.sha256:
         self.saParams.espSha = IpsecHmacAlgorithmEnum.sha256
      # Only aes256-gcm encryption currently supported
      if self.saParams.espAes != IpsecEspAlgorithmEnum.aes256gcm128:
         self.saParams.espAes = IpsecEspAlgorithmEnum.aes256gcm128
      # No pfs is configured
      if self.saParams.pfsGroup != IpsecDhGroupEnum.IpsecDhGroupNone:
         self.saParams.pfsGroup = IpsecDhGroupEnum.IpsecDhGroupNone
      SecurityAssociationVxlanMode.__init__( self, self.saName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateSAConfig( self, config ):
      secAssocConfig = config.securityAssoc.get( self.saName )
      if secAssocConfig is None:
         secAssocConfig = config.securityAssoc.newMember( self.saName )
      return secAssocConfig

   def onExit( self ):
      if self.secAssoc.saParams != self.saParams:
         self.secAssoc.saParams = self.saParams

class IpsecSecurityAssocConfigMode( SecurityAssociationMode,
                                    BasicCli.ConfigModeBase ):
   name = 'IPsec SA configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name ):
      self.saName = name
      self.secAssoc = self.getOrCreateSAConfig( ikeConfig() )
      self.saParams = Tac.nonConst( self.secAssoc.saParams )
      if self.saParams.saType != IpsecSaTypeEnum.IpsecKeyMgmtSa:
         self.saParams.saType = IpsecSaTypeEnum.IpsecKeyMgmtSa
      SecurityAssociationMode.__init__( self, self.saName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateSAConfig( self, config ):
      secAssocConfig = config.securityAssoc.get( self.saName )
      if secAssocConfig is None:
         secAssocConfig = config.securityAssoc.newMember( self.saName )
      return secAssocConfig

   def onExit( self ):
      if self.secAssoc.saParams != self.saParams:
         self.secAssoc.saParams = self.saParams

class IpsecProfileConfigMode( IpsecProfileMode, BasicCli.ConfigModeBase ):
   name = 'IPsec Profile configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name ):
      self.profileName = name
      self.profile = self.getOrCreateProfileConfig( ikeConfig() )
      self.profileParams = Tac.nonConst( self.profile.profileParams )
      IpsecProfileMode.__init__( self, self.profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateProfileConfig( self, config ):
      ipsecProfileConfig = config.ipsecProfile.get( self.profileName )
      if ipsecProfileConfig is None:
         ipsecProfileConfig = config.ipsecProfile.newMember( self.profileName )
      return ipsecProfileConfig

   def onExit( self ):
      if self.profile.profileParams != self.profileParams:
         self.profile.profileParams = self.profileParams

class IpsecConfigMode( IpSecurityMode, BasicCli.ConfigModeBase ):
   name = 'IP Security Mode'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      IpSecurityMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      ikeConfig().ipsecModeExit = ikeConfig().ipsecModeExit + 1

class KeyControllerConfigMode( KeyControllerMode, BasicCli.ConfigModeBase ):
   name = 'IPsec Key Controller Config Mode'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.controllerCfg = Tac.nonConst( ikeConfig().keyController )
      KeyControllerMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if ikeConfig().keyController != self.controllerCfg:
         ikeConfig().keyController = self.controllerCfg

#-------------------------------------------------------------------------------
# 'ip security'
#-------------------------------------------------------------------------------
class IpSecurity( CliCommand.CliCommandClass ):
   syntax = 'ip security'
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'security': 'IP security' }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( IpsecConfigMode )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( IpSecurity )

#-------------------------------------------------------------------------------
# '[no] profile <name>'
#-------------------------------------------------------------------------------
class IpsecProfile( CliCommand.CliCommandClass ):
   syntax = 'profile <profileName>'
   noOrDefaultSyntax = 'profile <profileName>'
   data = { 'profile': 'Profile name',
            '<profileName>': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                             helpname='WORD',
                             helpdesc='name of the profile' )
          }

   @staticmethod
   def handler(  mode, args ):
      name = args.get( '<profileName>' )
      childMode = mode.childMode( IpsecProfileConfigMode, name=name )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateProfileConfig( ikeConfig() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( '<profileName>' )
      config = ikeConfig()
      if name and name in config.ipsecProfile:
         keyControllerCfg = config.keyController
         if keyControllerCfg.configured and keyControllerCfg.profileName == name:
            mode.addError( 'Cannot remove Ipsec profile %s. '
                           'It is applied to the key controller' % name )
            return
         #Check to see if this profile has been applied to a Tunnel/Path interface
         if name in ikeStatus().appliedProfiles:
            appliedProfile = ikeStatus().appliedProfiles[ name ]
            # It is possible that the profile has not been removed from the
            # applied profiles when we have removed it from the key controller
            # but has not exited from the ipsec mode. We want to allow
            # deleting the profile in this case
            if appliedProfile.appliedIntfId or appliedProfile.appliedPath:
               mode.addError( 'Cannot remove Ipsec profile %s. '
                              'It is applied to an interface' % name )
               return

         del config.ipsecProfile [ name ]

IpsecConfigMode.addCommandClass( IpsecProfile )

#------------------------------------------------------------------------------------
#sa policy <name>
#------------------------------------------------------------------------------------
class IpsecSecurityAssociation( CliCommand.CliCommandClass ):
   if Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
      syntax = 'sa policy [ vxlan security ] <saName>'
   else:
      syntax = 'sa policy <saName>'
   noOrDefaultSyntax = syntax
   data = { 'sa': 'Security Association',
            'policy': 'Policy to be applied on the data path',
            'vxlan': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'vxlan',
                  helpdesc='VXLANSec Security Association' ),
               guard=vxlanSecurityGuard ),
            'security': 'Security policy',
            '<saName>': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                                    helpname='WORD',
                                                    helpdesc='name of the SA' )
          }

   @staticmethod
   def handler( mode, args ):
      name = args.get( '<saName>' )
      secAssocConfig = ikeConfig().securityAssoc.get( name )
      # Verify the SA name configured doesn't conflict with another SA type
      if secAssocConfig is not None:
         saType = secAssocConfig.saParams.saType
         if ( 'vxlan' in args and saType != IpsecSaTypeEnum.IpsecVxlanSa ) or \
            ( 'vxlan' not in args and saType != IpsecSaTypeEnum.IpsecKeyMgmtSa ):
            mode.addError( "Conflicting SA name" )
            return
      if 'vxlan' in args:
         childMode = mode.childMode( IpsecVxlanSecurityAssocConfigMode, name=name )
      else:
         childMode = mode.childMode( IpsecSecurityAssocConfigMode, name=name )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateSAConfig( ikeConfig() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sa = args.get( '<saName>' )
      config = ikeConfig()
      if sa in config.securityAssoc:
         if ikeStatus().appliedSa.has_key( sa ):
            mode.addError( 'Cannot remove Security associsation %s. '
                           'It is applied to a profile' % sa )
            return
         else:
            del config.securityAssoc[ sa ]

IpsecConfigMode.addCommandClass( IpsecSecurityAssociation )

#--------------------------------------------------------------------------
# esp encryption [ aes128 | aes256 | aes128gcm64 | aes128gcm128 | null ]
#---------------------------------------------------------------------------
class IpsecSaEncryption( CliCommand.CliCommandClass ):
   syntax = 'esp encryption [ aes128 | aes256 | aes128gcm64 | \
         aes128gcm128 | aes256gcm128 | null ]'
   noOrDefaultSyntax = 'esp encryption ...'
   data = { 'esp': 'Encapsulation Security Payload',
            'encryption': 'Encryption type',
            'aes128': 'AES - 128-bit Advanced Encryption Standard',
            'aes256': 'AES - 256-bit Advanced Encryption Standard',
            'aes128gcm64': '128 bit AES-GCM with 64 bit ICV',
            'aes128gcm128': '128 bit AES-GCM with 128 bit ICV',
            'aes256gcm128': '256 bit AES-GCM with 128 bit ICV',
            'null': 'Null Encryption',
          }

   @staticmethod
   def handler( mode, args ):
      #aes128 is the default encryption algo
      encr = IpsecEspAlgorithmEnum.aes128
      if args.get( 'aes128' ):
         encr = IpsecEspAlgorithmEnum.aes128
      if args.get( 'aes256' ):
         encr = IpsecEspAlgorithmEnum.aes256
      if args.get( 'aes128gcm64' ):
         encr = IpsecEspAlgorithmEnum.aes128gcm64
      if args.get( 'aes128gcm128' ):
         encr = IpsecEspAlgorithmEnum.aes128gcm128
      if args.get( 'aes256gcm128' ):
         encr = IpsecEspAlgorithmEnum.aes256gcm128
      if args.get( 'null' ):
         encr = IpsecEspAlgorithmEnum.nullesp
      mode.saParams.espAes = encr

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.espAes = mode.saParams.espAesDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecSaEncryption )

#--------------------------------------------------------------------------------
# esp integrity [sha1 | sha256 | null ]
#---------------------------------------------------------------------------
class IpsecSaIntegrity( CliCommand.CliCommandClass ):
   syntax = 'esp integrity [ sha1 | sha256 | null ]'
   noOrDefaultSyntax = 'esp integrity ...'
   data = { 'esp': 'Encapsulation Security Payload',
            'integrity': 'Integrity type',
            'sha1': 'sha1 security profile',
            'sha256': 'sha256 security profile',
            'null': 'null security profile',
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'sha256' ):
         espSha = IpsecHmacAlgorithmEnum.sha256
      elif args.get( 'null' ):
         espSha = IpsecHmacAlgorithmEnum.nullhash
      else:
         espSha = IpsecHmacAlgorithmEnum.sha1
      mode.saParams.espSha = espSha

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.espSha = mode.saParams.espShaDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecSaIntegrity )

#-------------------------------------------------------------------------------
# ' ike policy <name>'
#-------------------------------------------------------------------------------
class IpsecIkePolicy( CliCommand.CliCommandClass ):
   '''Enter the Ike mode to enter the IKE profile, encryption method
      integrity, lifetime and Diffie-Hellman group number'''
   syntax = 'ike policy <policy>'
   noOrDefaultSyntax = 'ike policy <policy>'
   data = { 'ike': 'Internet Security Association and Key Mgmt Protocol',
            'policy': 'Assign a name',
            '<policy>': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                                   helpname='WORD',
                                                   helpdesc='name of the SA' )
          }

   @staticmethod
   def handler( mode, args ):
      ikeName = args.get( '<policy>' )
      childMode = mode.childMode( IpsecIkeConfigMode, name=ikeName )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateIkeConfig( ikeConfig() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policy = args.get( '<policy>' )
      config = ikeConfig()
      if policy in config.ikePolicy:
         if ikeStatus().appliedPolicy.has_key( policy ):
            mode.addError( 'Cannot remove Policy %s. It is currently applied '
                           'to a profile.' % policy )
            return
         else:
            del config.ikePolicy [ policy ]

IpsecConfigMode.addCommandClass( IpsecIkePolicy )

#-------------------------------------------------------------------------------
# 'version ( 1 | 2 )
#-------------------------------------------------------------------------------
class IpsecIkeVersion( CliCommand.CliCommandClass ):
   syntax = 'version ( 1 | 2 )'
   noOrDefaultSyntax = 'version ...'

   data = { 'version': 'IKE Version',
            '1': 'IKEv1',
            '2': 'IKEv2'
          }

   @staticmethod
   def handler ( mode, args ):
      if args.get( '1' ):
         version = '1'
      elif args.get( '2' ):
         version = '2'
      mode.ikeParams.version = version

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.version = mode.ikeParams.versionDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeVersion )

#-------------------------------------------------------------------------------
# 'authentication ( pre-share | rsa-sig )
#-------------------------------------------------------------------------------
class IpsecIkeAuthentication( CliCommand.CliCommandClass ):
   syntax = 'authentication ( pre-share | rsa-sig )'

   noOrDefaultSyntax = 'authentication ...'

   data = { 'authentication': 'Authentication type',
            'pre-share': 'Pre-Shared Key',
            'rsa-sig': 'Rivest-Shamir-Adleman Signature',
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'pre-share' ):
         auth = IpsecAuthTypeEnum.pre_share
      elif args.get( 'rsa-sig' ):
         auth = IpsecAuthTypeEnum.rsa_sig
      mode.ikeParams.auth = auth

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.ikeParams.auth = mode.ikeParams.authDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeAuthentication )

#-------------------------------------------------------------------------------
# 'encryption [ aes128 | aes256 | 3des ]
#-------------------------------------------------------------------------------
class IpsecIkeEncryption( CliCommand.CliCommandClass ):
   syntax = 'encryption [ aes128 | aes256 | 3des ]'
   noOrDefaultSyntax = 'encryption ...'
   data = { 'encryption': 'Encryption type',
            'aes128': 'AES - 128-bit Advanced Encryption Standard',
            'aes256': 'AES - 256-bit Advanced Encryption Standard',
            '3des': 'Three key triple DES',
          }

   @staticmethod
   def handler( mode, args ):
      encr = mode.ikeParams.encryptionDefault
      if args.get( 'aes128' ):
         encr = IpsecEspAlgorithmEnum.aes128
      if args.get( 'aes256' ):
         encr = IpsecEspAlgorithmEnum.aes256
      elif args.get( '3des' ):
         encr = IpsecEspAlgorithmEnum.des
      mode.ikeParams.encryption = Tac.Value( 'Ipsec::Ike::IpsecEncryption', encr, 0 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.encryption = mode.ikeParams.encryptionDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeEncryption )

#-------------------------------------------------------------------------------
# integrity ( sha1 | sha256 | sha384 | sha512 )
#-------------------------------------------------------------------------------
class IpsecIkeHash( CliCommand.CliCommandClass ):
   syntax = 'integrity ( sha1 | sha256 | sha384 | sha512 )'
   noOrDefaultSyntax = 'integrity ...'
   data = { 'integrity': 'Integrity algorithm',
            'sha1'  : 'Secure Hash Standard 1 (160 bit)',
            'sha256': 'Secure Hash Standard 2 (256 bit)',
            'sha384': 'Secure Hash Standard 2 (384 bit)',
            'sha512': 'Secure Hash Standard 2 (512 bit)',
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'sha1' ):
         integrity = IpsecHmacAlgorithmEnum.sha1
      elif args.get( 'sha256' ):
         integrity = IpsecHmacAlgorithmEnum.sha256
      elif args.get( 'sha384' ):
         integrity = IpsecHmacAlgorithmEnum.sha384
      elif args.get( 'sha512' ):
         integrity = IpsecHmacAlgorithmEnum.sha512
      mode.ikeParams.integrity = integrity

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.integrity = mode.ikeParams.integrityDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeHash )

matcherDHGroupNum = CliMatcher.EnumMatcher( {
                    '1': 'Diffie-Hellman group 1 (768 bit)',
                    '2': 'Diffie-Hellman group 2 (1024 bit)',
                    '5': 'Diffie-Hellman group 5 (1536 bit)',
                    '14': 'Diffie-Hellman group 14 (2048 bit)',
                    '15': 'Diffie-Hellman group 15 (3072 bit)',
                    '16': 'Diffie-Hellman group 16 (4096 bit)',
                    '17': 'Diffie-Hellman group 17 (6144 bit)',
                    '20': 'Diffie-Hellman group 20 (384 bit ecp)',
                    '21': 'Diffie-Hellman group 21 (521 bit ecp)',
                    '24': 'Diffie-Hellman group 24 (2048 bit, 256 bit subgroup)' } )

groupNumToIpsecDhGroupMap = {
      '1': IpsecDhGroupEnum.modp768,
      '2': IpsecDhGroupEnum.modp1024, 
      '5': IpsecDhGroupEnum.modp1536, 
      '14': IpsecDhGroupEnum.modp2048,
      '15': IpsecDhGroupEnum.modp3072,
      '16': IpsecDhGroupEnum.modp4096,
      '17': IpsecDhGroupEnum.modp6144,
      '20': IpsecDhGroupEnum.ecp384,
      '21': IpsecDhGroupEnum.ecp521,
      '24': IpsecDhGroupEnum.modp2048s256,
      }

def getDHGroupFromNum( dhGroupNum ):
   dhGroup = groupNumToIpsecDhGroupMap.get( dhGroupNum )
   return dhGroup

#-----------------------------------------------------------------------------------
# dh-group
#-----------------------------------------------------------------------------------
class IpsecIkeDHGroup( CliCommand.CliCommandClass ):
   syntax = 'dh-group GROUP_NUM'
   noOrDefaultSyntax = 'dh-group ...'
   data = { 'dh-group': 'Diffie-Hellman Group',
            'GROUP_NUM': matcherDHGroupNum 
          }

   @staticmethod
   def handler( mode, args ):
      dhGroupNum = args[ 'GROUP_NUM' ]
      mode.ikeParams.dhGroup = getDHGroupFromNum( dhGroupNum )

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.ikeParams.dhGroup = mode.ikeParams.dhGroupDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeDHGroup )

#-----------------------------------------------------------------------------------
# ike-lifetime
#-----------------------------------------------------------------------------------
timeMatcher = CliMatcher.IntegerMatcher( 1, 24, helpdesc='Lifetime (hours)' )
class IpsecIkeLifetime( CliCommand.CliCommandClass ):
   syntax = 'ike-lifetime <time>'
   noOrDefaultSyntax = 'ike-lifetime [ <time> ] '
   data = { 'ike-lifetime': 'Set ikeLifetime for ISAKMP security association',
            '<time>': timeMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      mode.ikeParams.ikeLifetime = Tac.newInstance( 'Ipsec::Ike::IpsecLifetime' ,
                                                  args.get( '<time>' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.ikeParams.ikeLifetime = mode.ikeParams.ikeLifetimeDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeLifetime )

#-----------------------------------------------------------------------------------
# local-id <ip>
#-----------------------------------------------------------------------------------
genIpAddrMatcher = IpGenAddrMatcher.ipGenAddrMatcher

class IpsecIkeLocalIdentity( CliCommand.CliCommandClass ):
   syntax = 'local-id <ip>'
   noOrDefaultSyntax = 'local-id [ <ip> ]'

   data = { 'local-id': 'Local IKE Identification',
            '<ip>': genIpAddrMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mode.ikeParams.localId = args.get( '<ip>' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.localId = mode.ikeParams.idDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeLocalIdentity )

#-----------------------------------------------------------------------------------
# remote-id <ip>
#-----------------------------------------------------------------------------------
class IpsecIkeRemoteIdentity( CliCommand.CliCommandClass ):
   syntax = 'remote-id <ip>'
   noOrDefaultSyntax = 'remote-id [ <ip> ]'

   data = { 'remote-id': 'Remote peer IKE Identification',
            '<ip>': genIpAddrMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mode.ikeParams.remoteId = args.get( '<ip>' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ikeParams.remoteId = mode.ikeParams.idDefault

IpsecIkeConfigMode.addCommandClass( IpsecIkeRemoteIdentity )

#-----------------------------------------------------------------------------------
# pfs group
#-----------------------------------------------------------------------------------
class IpsecPfsgroup( CliCommand.CliCommandClass ):
   syntax = 'pfs dh-group GROUP_NUM'
   noOrDefaultSyntax = 'pfs dh-group ...'
   data = { 'pfs': 'Perfect Forward Secrecy',
            'dh-group': 'Diffie-Hellman Group',
            'GROUP_NUM': matcherDHGroupNum 
          }

   @staticmethod
   def handler ( mode, args ):
      dhGroupNum = args[ 'GROUP_NUM' ]
      mode.saParams.pfsGroup = getDHGroupFromNum( dhGroupNum )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.pfsGroup = IpsecDhGroupEnum.IpsecDhGroupNone

IpsecSecurityAssocConfigMode.addCommandClass( IpsecPfsgroup )

#-----------------------------------------------------------------------------------
# sa lifetime <time> [ hours ]
# no sa lifetime [ <time> ]
# no sa lifetime <time> hours
#-----------------------------------------------------------------------------------
class IpsecSALifetime( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime <time> [ hours ]'
   noOrDefaultSyntax = 'sa lifetime [ <time> | [ <time> hours ] ]'
   data = { 'sa' : 'Security Association',
            'lifetime' : 'Lifetime for IPsec Security Association '\
                         '( default unit is hours )',
            '<time>': CliMatcher.IntegerMatcher( 1, 24,
                      helpdesc='Lifetime in hours ( default )' ),
            'hours' : 'Lifetime in hours ( default )',
          }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.saLifetime = Tac.newInstance( 'Ipsec::Ike::IpsecLifetime' ,
                                                  args.get( '<time>' ) )
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.saLifetime = mode.saParams.saLifetimeDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecSALifetime )

#-----------------------------------------------------------------------------------
# [no | default] anti-replay window <size>
#-----------------------------------------------------------------------------------
class IpsecAntiReplayWindow( CliCommand.CliCommandClass ):
   syntax = 'anti-replay window <size>'
   noOrDefaultSyntax = 'anti-replay window [ <size> ]'
   data = { 'anti-replay': 'IPsec duplicate IP datagram detection',
            'window': 'Anti-Replay Window',
            '<size>': CliMatcher.IntegerMatcher( 1024, 65535,
                                                 helpdesc='Set Window size' )
         }

   @staticmethod
   def handler( mode, args ):
      replayWindowSize = args.get( '<size>' )
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize', replayWindowSize )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.replayWindowSize = mode.saParams.replayWindowSizeDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecAntiReplayWindow )

#-----------------------------------------------------------------------------------
# [no | default] anti-replay detection
#
# Below command will set replay window to 0
# per cli-review feedback.
# no anti-replay
#
# Below commands will reset the replay window to default
# anti-replay detection
# default anti-replay [ detection ]
#-----------------------------------------------------------------------------------
class IpsecAntiReplayDetection( CliCommand.CliCommandClass ):
   syntax = 'anti-replay detection'
   noOrDefaultSyntax = 'anti-replay [ detection ]'
   data = { 'anti-replay': 'IPsec duplicate IP datagram detection',
            'detection': 'Disable Anti-Replay Window detection',
          }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize',
            mode.saParams.replayWindowSizeDefault )

   @staticmethod
   def noHandler( mode, args ):
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize', 0 )

   @staticmethod
   def defaultHandler( mode, args ):
      mode.saParams.replayWindowSize = Tac.newInstance(
            'Ipsec::Ike::ReplayWindowSize', mode.saParams.replayWindowSizeDefault )

IpsecSecurityAssocConfigMode.addCommandClass( IpsecAntiReplayDetection )

#-----------------------------------------------------------------------------------
# sa lifetime <packetLimit> thousand-packets
#-----------------------------------------------------------------------------------
class IpsecPacketLimit( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime <packetLimit> thousand-packets'
   noOrDefaultSyntax = 'sa lifetime <packetLimit> thousand-packets'

   data = {
         'sa': 'Security Association',
         'lifetime' : 'Lifetime for IPsec Security Association '\
                      '( default unit is hours )',
         '<packetLimit>' : CliMatcher.IntegerMatcher( 1, 4000000,
                           helpdesc="Packet limit in thousands" ),
         'thousand-packets' : 'Packet limit in thousands',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.packetLimit = args.get( '<packetLimit>' ) * 1000

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.packetLimit = mode.saParams.packetLimitDefault

IpsecSecurityAssocConfigMode.addCommandClass( IpsecPacketLimit )

#-----------------------------------------------------------------------------------
# sa lifetime <byteLimit> megabytes
#-----------------------------------------------------------------------------------
class IpsecMegaByteLimit( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime <byteLimit> megabytes'
   noOrDefaultSyntax = 'sa lifetime <byteLimit> megabytes'

   data = {
         'sa' : 'Security Association',
         'lifetime' : 'Lifetime for IPsec Security Association '\
                      '( default unit is hours )',
         '<byteLimit>' : CliMatcher.IntegerMatcher( 1, 6144000,
                         helpdesc="Byte limit in MB ( 1024 KB )" ),
         'megabytes' : 'Byte limit in MB ( 1024 KB )',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.byteLimit = args.get( '<byteLimit>' ) * ( 1024 ** 2 )
      mode.saParams.byteLimitUnit = 'MegaBytes'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.byteLimit = mode.saParams.byteLimitDefault
      mode.saParams.byteLimitUnit = 'GigaBytes'

IpsecSecurityAssocConfigMode.addCommandClass( IpsecMegaByteLimit )

#-----------------------------------------------------------------------------------
# sa lifetime <byteLimit> gigabytes
#-----------------------------------------------------------------------------------
class IpsecGigaByteLimit( CliCommand.CliCommandClass ):
   syntax = 'sa lifetime <byteLimit> gigabytes'
   noOrDefaultSyntax = 'sa lifetime <byteLimit> gigabytes'

   data = {
         'sa' : 'Security Association',
         'lifetime' : 'Lifetime for IPsec Security Association '\
                      '( default unit is hours )',
         '<byteLimit>' : CliMatcher.IntegerMatcher( 1, 6000,
                         helpdesc="Byte limit in GB ( 1024 MB )" ),
         'gigabytes' : 'Byte limit in GB ( 1024 MB )',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.byteLimit = args.get( '<byteLimit>' ) * ( 1024 ** 3 )
      mode.saParams.byteLimitUnit = 'GigaBytes'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.byteLimit = mode.saParams.byteLimitDefault
      mode.saParams.byteLimitUnit = 'GigaBytes'

IpsecSecurityAssocConfigMode.addCommandClass( IpsecGigaByteLimit )

#-----------------------------------------------------------------------------------
# encapsulation vxlan security
#-----------------------------------------------------------------------------------
class IpsecEncapsulationVxlan( CliCommand.CliCommandClass ):
   syntax = 'encapsulation vxlan security'
   noOrDefaultSyntax = syntax

   data = {
         'encapsulation': 'Encapsulation format',
         'vxlan' : 'VXLAN',
         'security' : 'VXLAN payload encryption',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.vxlanEncap = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.vxlanEncap = False

if Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
   IpsecVxlanSecurityAssocConfigMode.addCommandClass( IpsecEncapsulationVxlan )

#-----------------------------------------------------------------------------------
# traffic unprotected allow
#-----------------------------------------------------------------------------------
class IpsecVxlanUnprotectedTraffic( CliCommand.CliCommandClass ):
   syntax = 'traffic unprotected allow'
   noOrDefaultSyntax = syntax

   data = {
         'traffic': 'Traffic security policy',
         'unprotected' : 'Unprotected fallback mode without encryption',
         'allow' : 'Allow policy option',
   }

   @staticmethod
   def handler( mode, args ):
      mode.saParams.unprotectedTraffic = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.saParams.unprotectedTraffic = False

if Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
   IpsecVxlanSecurityAssocConfigMode.addCommandClass( IpsecVxlanUnprotectedTraffic )
   IpsecVxlanSecurityAssocConfigMode.addCommandClass( IpsecSALifetime )

#-----------------------------------------------------------------------------------
# connection [add | start | route ]
#-----------------------------------------------------------------------------------
class IpsecConnectionType( CliCommand.CliCommandClass ):
   syntax = 'connection [ add | start | route ]'
   noOrDefaultSyntax = 'connection ...'
   data = { 'connection': 'IPsec Connection (Initiator/Responder/Dynamic)',
            'add': 'Responder',
            'start': 'Initiator',
            'route': 'Dynamic'
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'add' ):
         mode.profileParams.connectionType = 'add'
      if args.get('start'):
         mode.profileParams.connectionType = 'start'
      if args.get('route'):
         mode.profileParams.connectionType = 'route'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.connectionType = mode.profileParams.connectionTypeDefault

IpsecProfileConfigMode.addCommandClass( IpsecConnectionType )

#-----------------------------------------------------------------------------------
# mode [ tunnel | transport ]
#-----------------------------------------------------------------------------------
class IpsecModeType( CliCommand.CliCommandClass ):
   syntax = 'mode [ tunnel | transport ]'
   noOrDefaultSyntax = 'mode ...'
   data = { 'mode': 'Ipsec mode type',
            'tunnel': 'Tunnel mode',
            'transport': 'Transport mode'
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'tunnel' ):
         mode.profileParams.mode = 'tunnel'
      if args.get( 'transport' ):
         mode.profileParams.mode = 'transport'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.mode = 'tunnel'

IpsecProfileConfigMode.addCommandClass( IpsecModeType )

#----------------------------------------------------------------------------------
# sa-policy <name>
#----------------------------------------------------------------------------------
class SetIpsecTransformSet( CliCommand.CliCommandClass ):
   syntax = 'sa-policy { <sa-policy> }'
   noOrDefaultSyntax = 'sa-policy { <sa-policy> }'
   data = { 'sa-policy': 'Security Assoc Name',
            '<sa-policy>': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                                      helpname='WORD',
                                             helpdesc='name of the transform-set' )
          }

   @staticmethod
   def handler( mode, args ):
      tsNames = args.get( '<sa-policy>' )
      isPolicyMatching = True
      if mode.profile:
         for name in tsNames:
            # If currently configured polices match already existing policies,
            # then prevent event handlers from being called unnecessarily
            if name in mode.profile.securityAssoc:
               continue
            else:
               isPolicyMatching = False
         if isPolicyMatching is True:
            return
         mode.profile.securityAssoc.clear()
      else:
         mode.addError( ' Ipsec profile cannot be created' )
         return

      for name in tsNames:
         if name not in ikeConfig().securityAssoc.keys():
            mode.addError( ' IPsec SA policy %s not created' % name )
            return
         else:
            #Add mapping and policy
            mode.profile.securityAssoc.addMember( ikeConfig().\
                  securityAssoc.get( name ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tsNames = args.get( '<sa-policy>' )
      if mode.profile:
         for name in tsNames:
            if name not in mode.profile.securityAssoc.keys():
               mode.addError( ' IPsec SA %s not applied' % name )
               return
            else:
               del mode.profile.securityAssoc[name]

IpsecProfileConfigMode.addCommandClass( SetIpsecTransformSet )

#----------------------------------------------------------------------------------
# ike-policy <policy>
#----------------------------------------------------------------------------------
class SetIsakmpPolicy( CliCommand.CliCommandClass ):
   syntax = 'ike-policy { <ike-policy> }'
   noOrDefaultSyntax = 'ike-policy { <ike-policy> }'
   data = { 'ike-policy': 'ISAKMP policy',
            '<ike-policy>': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                                       helpname='WORD',
                                                       helpdesc='name of the SA' )
          }

   @staticmethod
   def handler( mode, args ):
      policies = args.get( '<ike-policy>' )
      profile = mode.getOrCreateProfileConfig( ikeConfig() )
      isPolicyMatching = True
      #Remove the current policy and mappings
      if profile:
         for policy in policies:
            # If currently configured polices match already existing policies,
            # then prevent event handlers from being called unnecessarily
            if policy in profile.ikePolicy:
               continue
            else:
               isPolicyMatching = False
         if isPolicyMatching is True:
            return
         profile.ikePolicy.clear()

      #Iterate over the policies and apply them.
      for policy in policies:
         if policy not in ikeConfig().ikePolicy.keys():
            mode.addError( ' Ike policy %s not created' % policy )
            return
         else:
            #Add new ikepolicy and mapping
            profile.ikePolicy.addMember( ikeConfig().ikePolicy.
                                           get( policy ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policies = args.get( '<ike-policy>' )
      if mode.profile:
         for policy in policies:
            if policy not in mode.profile.ikePolicy.keys():
               mode.addError( ' Ike policy %s not created' % policy )
               return
            else:
               del mode.profile.ikePolicy[policy]

IpsecProfileConfigMode.addCommandClass( SetIsakmpPolicy )

#-----------------------------------------------------------------------------------
# "shared-key ( ( 0 TYPE0KEY ) | ( 7 TYPE7KEY ) )"
# "shared-key CLEARTEXT" is changed to a hidden command
#-----------------------------------------------------------------------------------
keyValueMatcher = CliMatcher.PatternMatcher( 
      pattern=r'^[-+`~&!@#$%^*(),.:;=/|{}<>_a-zA-Z0-9\\\[\]\'\"]+$',
      helpname='WORD', helpdesc='Clear text key value' )
nodeClearText = CliCommand.Node( keyValueMatcher, hidden=True )

class IpsecPreSharedKey( CliCommand.CliCommandClass ):
   syntax = 'shared-key ( CLEARTEXT | ( 0 TYPE0KEY ) | ( 7 TYPE7KEY ) )'
   noOrDefaultSyntax = 'shared-key ...'
   data = { 'shared-key': 'Shared key',
            'CLEARTEXT': nodeClearText,
            '0': ReversibleSecretCli.type0KwMatcher,
            'TYPE0KEY': keyValueMatcher,
            '7': ReversibleSecretCli.type7KwMatcher,
            'TYPE7KEY': ReversibleSecretCli.type7Node,
          }

   @staticmethod
   def handler( mode, args ):
      if mode.profile:
         if args.get( 'CLEARTEXT' ):
            sharedKey = Tac.Value( 'Ipsec::Ike::IkePreShareKey',
                                   args[ 'CLEARTEXT' ], '' )
            #mode.profileParams.ikeSharedKey = sharedKey
            # Will not print a obscured key in "show running-config"
            #mode.profileParams.obscuredSharedKey = False
            mode.addWarning( "This command saves the shared key in clear text. "
                             "Please use 'shared-key 0 <text>' to save the key "
                             "in obscured form." )

         elif args.get( 'TYPE0KEY' ):
            sharedKey = Tac.Value( 'Ipsec::Ike::IkePreShareKey',
                                   args[ 'TYPE0KEY' ],
                                   ReversibleSecretCli.encodeKey(
                                      args[ 'TYPE0KEY' ] ) )
         else:
            sharedKey = Tac.Value( 'Ipsec::Ike::IkePreShareKey',
                                   ReversibleSecretCli.decodeKey(
                                      args[ 'TYPE7KEY' ] ),
                                   args[ 'TYPE7KEY' ] )
         mode.profileParams.ikeSharedKey = sharedKey
         #mode.profileParams.obscuredSharedKey = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.ikeSharedKey = Tac.Value( 'Ipsec::Ike::IkePreShareKey',
                                                   '', '' )

IpsecProfileConfigMode.addCommandClass( IpsecPreSharedKey )

#-------------------------------------------------------------------------------
#  flow parallelization encapsulation [ udp ]
#-------------------------------------------------------------------------------
class IpsecFlow( CliCommand.CliCommandClass ):
   syntax = 'flow parallelization encapsulation udp'
   noOrDefaultSyntax = 'flow parallelization encapsulation ...'
   data = { 'flow': 'Flow',
            'parallelization': 'Parallelize flows',
            'encapsulation': 'Encapsulation protocol',
            'udp': 'UDP'
          }

   @staticmethod
   def handler( mode, args ):
      encap = IpsecEncapProtocolEnum.no_encap
      if args.get( 'udp' ):
         encap = IpsecEncapProtocolEnum.udp

      flow = Tac.Value( 'Ipsec::Ike::FlowParallelization' )
      flow.encapProtocol = encap
      mode.profileParams.flowParallelization = flow


   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.flowParallelization = \
         Tac.Value( 'Ipsec::Ike::FlowParallelization' )

IpsecProfileConfigMode.addCommandClass( IpsecFlow )

#-------------------------------------------------------------------------------
# dpd <interval> <timeout> [ clear | hold | restart ]
#-------------------------------------------------------------------------------
class IpsecDpd( CliCommand.CliCommandClass ):
   syntax = 'dpd <interval> <timeout> [ clear | hold | restart ]'
   noOrDefaultSyntax = 'dpd ...'
   data = { 'dpd': 'Dead Peer Detection',
            '<interval>': CliMatcher.IntegerMatcher( 2, 3600,
                       helpdesc='Interval(in seconds) between keep-alive messages' ),
            '<timeout>': CliMatcher.IntegerMatcher( 10, 3600,
                    helpdesc='Time(in seconds) after which the action is applied.' ),
            'clear': 'Delete all connections',
            'hold': 'Re-negotiate connection on demand',
            'restart': 'Restart connection immediately'
          }

   @staticmethod
   def handler( mode, args ):
      interval = args.get( '<interval>' )
      timeout = args.get( '<timeout>' )
      action = dpdActionEnum.none
      if args.get( 'clear' ):
         action = dpdActionEnum.clear
      elif args.get( 'hold' ):
         action = dpdActionEnum.hold
      elif args.get( 'restart' ):
         action = dpdActionEnum.restart

      dpd = Tac.Value( 'Ipsec::Ike::DeadPeerDetection' )
      dpd.interval = interval
      dpd.timeout = timeout
      dpd.action = action
      mode.profileParams.dpd = dpd

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profileParams.dpd = Tac.Value( 'Ipsec::Ike::DeadPeerDetection' )

IpsecProfileConfigMode.addCommandClass( IpsecDpd )

#-------------------------------------------------------------------------------
# "key controller"
#-------------------------------------------------------------------------------
def setKeyController( mode, args ):
   childMode = mode.childMode( KeyControllerConfigMode )
   mode.session_.gotoChildMode( childMode )
   if not childMode.controllerCfg.configured:
      childMode.controllerCfg.configured = True
      ikeConfig().keyController = childMode.controllerCfg

def noKeyController( mode, args ):
   controllerCfg = Tac.nonConst( ikeConfig().keyController )
   controllerCfg.profileName = ""
   controllerCfg.dhLifetime = \
         ikeConfig().keyController.dhLifetimeDefault
   controllerCfg.dhGroup = ikeConfig().keyController.dhGroupDefault
   controllerCfg.configured = False
   ikeConfig().keyController = controllerCfg

class IpsecKeyController( CliCommand.CliCommandClass ):
   '''Enter the key controller mode to apply the profile we want to use'''
   syntax = 'key controller'
   noOrDefaultSyntax = 'key controller'
   data = { 'key': 'Config IPsec key controller',
            'controller': 'Enter key controller config mode' }

   handler = setKeyController
   noOrDefaultHandler = noKeyController

IpsecConfigMode.addCommandClass( IpsecKeyController )

#-------------------------------------------------------------------------------
# "profile <profile-name>" in "key controller" mode
#-------------------------------------------------------------------------------
def setKeyControllerProfile( mode, args ):
   profileName = args[ '<profile-name>' ]

   [ val, message ] = isIpsecProfilePresent( profileName )
   if not val:
      mode.addError( message )
      return

   if mode.controllerCfg.profileName != profileName:
      mode.controllerCfg.profileName = profileName

def noSetKeyControllerProfile( mode, args ):
   mode.controllerCfg.profileName = ""

class IpsecKeyControllerProfile( CliCommand.CliCommandClass ):
   '''Apply profile to the key controller'''
   syntax = 'profile <profile-name>'
   noOrDefaultSyntax = 'profile ...'
   data = { 'profile': 'Apply IPsec profile',
            '<profile-name>': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                              helpdesc='Name of the profile', helpname='WORD' )
          }

   handler = setKeyControllerProfile
   noOrDefaultHandler = noSetKeyControllerProfile

KeyControllerConfigMode.addCommandClass( IpsecKeyControllerProfile )

#-----------------------------------------------------------------------------------
# 'lifetime <time> hours' for key controller
#-----------------------------------------------------------------------------------
class IpsecControllerDHLifetime( CliCommand.CliCommandClass ):
   syntax = 'lifetime <time> hours'
   noOrDefaultSyntax = 'lifetime ...'
   data = { 'lifetime': 'Set DH key lifetime',
            '<time>': timeMatcher,
            'hours' : 'Lifetime in hours',
          }

   @staticmethod
   def handler( mode, args ):
      mode.controllerCfg.dhLifetime = \
            Tac.newInstance( 'Ipsec::Ike::IpsecLifetime',
                             args.get( '<time>' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.controllerCfg.dhLifetime = mode.controllerCfg.dhLifetimeDefault

KeyControllerConfigMode.addCommandClass( IpsecControllerDHLifetime )

#-----------------------------------------------------------------------------------
# 'dh-group GROUP_NUM' for key controller
#-----------------------------------------------------------------------------------
class IpsecControllerDHGroup( CliCommand.CliCommandClass ):
   syntax = 'dh-group GROUP_NUM'
   noOrDefaultSyntax = 'dh-group ...'
   data = { 'dh-group': 'Set Diffie-Hellman group',
            'GROUP_NUM': matcherDHGroupNum 
          }

   @staticmethod
   def handler( mode, args ):
      dhGroupNum = args[ 'GROUP_NUM' ]
      mode.controllerCfg.dhGroup = getDHGroupFromNum( dhGroupNum )

   @staticmethod
   def noOrDefaultHandler( mode, args):
      mode.controllerCfg.dhGroup = mode.controllerCfg.dhGroupDefault

KeyControllerConfigMode.addCommandClass( IpsecControllerDHGroup )

#-------------------------------------------------------------------------------
# Plugin Func
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global tunIntfConfigDir, tunIntfStatusDir
   global ikeConfigDir, ikeStatusDir
   global hwCapabilities

   ikeConfigDir = ConfigMount.mount( entityManager, 'ipsec/ike/config',
                                        'Ipsec::Ike::Config', 'w' )
   ikeStatusDir = LazyMount.mount( entityManager, 'ipsec/ike/status',
                                      'Ipsec::Ike::Status', 'r' )
   tunIntfConfigDir = ConfigMount.mount( entityManager,
                                          'interface/config/tunnel/intf',
                                          'Interface::TunnelIntfConfigDir', 'w' )
   tunIntfStatusDir = LazyMount.mount( entityManager,
                                        'interface/status/tunnel/intf',
                                        'Interface::TunnelIntfStatusDir', 'r' )
   hwCapabilities = LazyMount.mount( entityManager,
                                     'bridging/hwcapabilities',
                                     'Bridging::HwCapabilities',
                                     'r' )

   TunnelIntfCli.ipsecHook.addExtension( isIpsecProfilePresent )

