# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Int, List, Model, Enum
from ArnetModel import Ip4Address, MacAddress
from IntfModel import Interface
import TableOutput
import Tac
import Arnet, Vlan
from IpSourceGuardMsg import VlanOpState, IntfOpState, BindingOpState

formatLeft = TableOutput.Format( justify="left" )
formatLeft.noPadLeftIs( True )

class IpsgInterfaceOperationalState( Model ):
   operationalState = Enum( values=( IntfOpState.enabled, IntfOpState.noResource,
                                     IntfOpState.disabled ),
                            help="Operational state of IP source guard on an"
                            "interface" )

class IpsgVlanOperationalState( Model ):
   operationalState = Enum( values=( VlanOpState.disabled, VlanOpState.noResource,
                                     VlanOpState.incomplete ),
                            help="Operational state of IP source guard on a VLAN" )

class IpsgBindingDetailState( Model ):
   macAddr = MacAddress( help="MAC address" )
   ipAddr = Ip4Address( help="IPv4 address" )
   vlanId = Int( help="VLAN ID of an IP source binding entry" )
   state = Enum( values=( BindingOpState.active, BindingOpState.noResource,
                          BindingOpState.disabled, 
                          BindingOpState.noPortChannelMember ),
                 help="Operational state of an IP source binding entry" )

class IpsgBindingDetailStateList( Model ):
   intfDetailList = List( valueType=IpsgBindingDetailState,
                          help="IP source binding entries and the states" )

class IpsgInterfaceState( Model ):
   interfaceStates = Dict( keyType=Interface,
                           valueType=IpsgInterfaceOperationalState,
                           help="A mapping from interfaces to IP source guard"
                           "operational states on these interfaces" )
   
   def render( self ):
      headings = ( "Interface", "Operational State" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatLeft )
      for key in Arnet.sortIntf( self.interfaceStates ):
         table.newRow( key, self.interfaceStates[key].operationalState )
      print table.output()

class IpsgVlanState( Model ):
   vlanStates = Dict( keyType=int,
                      valueType=IpsgVlanOperationalState,
                      help="A mapping from VLANs to IP source guard operational"
                      "states on these VLANs" )

   def render( self ):
      print 'IPSG disabled on VLANs:', \
            Vlan.vlanSetToCanonicalString( self.vlanStates.keys() ), '\n'
      headings = ( "VLAN", "Operational State" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatLeft )
      vlanStates = sorted( self.vlanStates.items(),
                           key=lambda configEntry: configEntry[0] )
      for entry in vlanStates:
         table.newRow( entry[0], entry[1].operationalState )
      print table.output()

class IpsgDetailState( Model ):
   detailStates = Dict( keyType=Interface,
                        valueType=IpsgBindingDetailStateList,
                        help="IP source binding entries and their state" )
   
   def render( self ):
      headings = ( "Interface", "IP Address", "MAC Address", "VLAN", "State" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatLeft, formatLeft, 
                           formatLeft, formatLeft )
      for key in Arnet.sortIntf( self.detailStates ):
         entryList = self.detailStates[ key ]
         for entry in entryList.intfDetailList:
            table.newRow( key, entry.ipAddr, entry.macAddr.displayString, 
                          entry.vlanId, entry.state )
      print table.output()

