# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount, LazyMount, Tac
import BasicCli, CliParser, CliToken.Ip, CliToken.Verify
import ShowCommand, CliCommand, CliMatcher
import IntfCli, VlanCli
from IpSourceGuardModels \
      import IpsgInterfaceOperationalState, IpsgVlanOperationalState,\
             IpsgBindingDetailState, IpsgBindingDetailStateList
from IpSourceGuardModels import IpsgInterfaceState, IpsgVlanState, IpsgDetailState
from IpSourceGuardMsg import VlanOpState, IntfOpState, BindingOpState

ipsgConfig = None
ipsgHwStatusDir = None
ipsgTable = None
lagConfigCli = None

def ipsgSupportedGuard( mode, token ):
   for hwStatus in ipsgHwStatusDir.itervalues():
      if hwStatus.ipsgSupported:
         return None
   return CliParser.guardNotThisPlatform

def ipsgIntfStatus( intf ):
   ''' Return 'enabled' if the interface is programmed into a drop entry;
   'disabled' if the interface is not supposed to be enabled (e.g., this interface
   is a lag member or routed port; 'pending' if the corresponding drop entry is not 
   programmed in the TCAM (e.g., exhaustion)'''
   enabled = False
   dropInstalled = True
   for hwStatus in ipsgHwStatusDir.itervalues():
      if hwStatus.enabledIntf.get( intf ):
         enabled = True
         dropInstalled = hwStatus.dropOnEnabledIntf if dropInstalled else False
   if not enabled:
      return 'disabled'
   if not dropInstalled:
      return 'pending'
   return 'enabled'

sourceConfigNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'source', 
                                           helpdesc='Enable IP source guard' ),
                                    guard=ipsgSupportedGuard )
sourceShowNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'source', 
                                         helpdesc='Show IP source guard' ),
                                  guard=ipsgSupportedGuard )

#-------------------------------------------------------------------------------
# Enable/disable Ip Source Guard for interfaces
# "[no|default] ip verify source [dhcp-snooping-vlan]"
#-------------------------------------------------------------------------------
class ConfigIpsgIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip verify source [ dhcp-snooping-vlan ]'
   noOrDefaultSyntax = 'ip verify source'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'verify': CliToken.Verify.verifyMatcherForConfigIf,
            'source': sourceConfigNode,
            'dhcp-snooping-vlan': "dhcp-snooping-vlan"
          }

   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      ipsgConfig.ipsgEnabledIntf[ intfName ] = True
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      del ipsgConfig.ipsgEnabledIntf[ intfName ]

VlanCli.SwitchportModelet.addCommandClass( ConfigIpsgIntfCmd )

#-------------------------------------------------------------------------------
# Enable/disable Ip Source Guard for vlans
# "[no|default] ip verify source vlan VLAN_SET
#-------------------------------------------------------------------------------

class ConfigIpsgVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'ip verify source vlan VLAN_SET'
   noOrDefaultSyntax = 'ip verify source vlan VLAN_SET'
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'verify': CliToken.Verify.verifyMatcherForConfig,
            'source': sourceConfigNode,
            'vlan': 'VLAN Keyword',
            'VLAN_SET': VlanCli.vlanSetMatcher
            }

   @staticmethod
   def handler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      if vlanSet:
         for vlanId in vlanSet.ids:
            if ipsgConfig.ipsgDisabledVlan.get( vlanId ):
               del ipsgConfig.ipsgDisabledVlan[ vlanId ]
      else:
         ipsgConfig.ipsgDisabledVlan.clear()

   @staticmethod
   def noHandler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      for vlanId in vlanSet.ids:
         ipsgConfig.ipsgDisabledVlan[ vlanId ] = True

   # By default, ipsg is enabled on a vlan
   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( ConfigIpsgVlanCmd )

#-------------------------------------------------------------------------------
# show Ip Source Guard config for interface
# "show ip verify source"
#-------------------------------------------------------------------------------
class ShowIpsgIntfConfigCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show ip verify source"
   data = {
            'ip': CliToken.Ip.ipMatcherForShow,
            'verify': CliToken.Verify.verifyMatcherForShow,
            'source': sourceShowNode
          }
   cliModel = IpsgInterfaceState

   @staticmethod
   def handler( mode, args ):
      intfEnabled = {}
      ipsgIntfConfig = ipsgConfig.ipsgEnabledIntf
      for intf in ipsgIntfConfig.iterkeys():
         # IPSG is considered enabled on an interface if this interface is added
         # to the drop entry on one of the chips
         intfStatus = ipsgIntfStatus( intf )
         if intfStatus == 'enabled':
            opState = IntfOpState.enabled
         elif intfStatus == 'disabled':
            opState = IntfOpState.disabled
         else:
            opState = IntfOpState.noResource
         intfEnabled[ intf ] = IpsgInterfaceOperationalState(
            operationalState=opState )
      return IpsgInterfaceState( interfaceStates=intfEnabled )

BasicCli.addShowCommandClass( ShowIpsgIntfConfigCmd )

#-------------------------------------------------------------------------------
# show Ip Source Guard config for VLAN
# "show ip verify source vlan"
#-------------------------------------------------------------------------------
class ShowIpsgVlanConfigCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show ip verify source vlan"
   data = {
            'ip': CliToken.Ip.ipMatcherForShow,
            'verify': CliToken.Verify.verifyMatcherForShow,
            'source': sourceShowNode,
            'vlan': 'VLAN Keyword'
          }
   cliModel = IpsgVlanState

   @staticmethod
   def handler( mode, args ):
      ipsgVlanConfig = ipsgConfig.ipsgDisabledVlan
      vlanDisabled = {}
      for vlan in ipsgVlanConfig.iterkeys():
         # IPSG is considered disabled on a vlan if the vlan permit entry
         # is installed and the vlan class is programmed on all chips.
         if len( ipsgHwStatusDir.keys() ) == 0:
            opState = VlanOpState.noResource
         else:
            opState = VlanOpState.disabled
            for hwStatus in ipsgHwStatusDir.itervalues():
               if not hwStatus.permitOnDisabledVlan:
                  opState = VlanOpState.noResource
                  break
               elif not vlan in hwStatus.ipsgDisabledVlan:
                  opState = VlanOpState.incomplete
                  break
         vlanDisabled[ vlan ] = IpsgVlanOperationalState( operationalState=opState )
      return IpsgVlanState( vlanStates=vlanDisabled )

BasicCli.addShowCommandClass( ShowIpsgVlanConfigCmd )

#-------------------------------------------------------------------------------
# show Ip Source Guard config for binding detail
# "show ip verify source detail"
#-------------------------------------------------------------------------------
class ShowIpsgDetailConfigCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show ip verify source detail"
   data = {
            'ip': CliToken.Ip.ipMatcherForShow,
            'verify': CliToken.Verify.verifyMatcherForShow,
            'source': sourceShowNode,
            'detail': 'detail'
          }
   cliModel = IpsgDetailState

   @staticmethod
   def handler( mode, args ):
      detailStates = {}
      ipsgIntfConfig = ipsgConfig.ipsgEnabledIntf
      ipsgVlanConfig = ipsgConfig.ipsgDisabledVlan
      for intfId, bindingList in ipsgTable.ipMacBinding.iteritems():
         # Only display binding entries on IPSG enabled interfaces
         if not intfId in ipsgIntfConfig:
            continue
         for key in bindingList.bindingEntry.iterkeys():
            if ipsgIntfStatus( intfId ) == 'disabled':
               opState = BindingOpState.disabled
            elif key.vlan in ipsgVlanConfig:
               opState = BindingOpState.disabled
            else:
               # The binding entry is considered Active if it is programmed
               # on one of the chips
               opState = BindingOpState.noResource
               if intfId.startswith( 'Port-Channel' ):
                  for intf in lagConfigCli.phyIntf.itervalues():
                     lag = intf.lag
                     if lag and lag.intfId == intfId:
                        break
                  else:
                     opState = BindingOpState.noPortChannelMember
               keyWithIntf = Tac.Value( 'IpMacBinding::KeyWithIntf', key,
                                         intfId )
               for hwStatus in ipsgHwStatusDir.itervalues():
                  if keyWithIntf in hwStatus.ipMacBinding:
                     opState = BindingOpState.active
                     break
            detailEntry = IpsgBindingDetailState( ipAddr=key.ip,
                                                  macAddr=key.mac,
                                                  vlanId=key.vlan,
                                                  state=opState )
            if intfId in detailStates:
               detailStates[ intfId ].intfDetailList.append( detailEntry )
            else:
               intfDetailList = IpsgBindingDetailStateList(
                  intfDetailList = [ detailEntry ] )
               detailStates[ intfId ] = intfDetailList
      
      return IpsgDetailState( detailStates=detailStates )

BasicCli.addShowCommandClass( ShowIpsgDetailConfigCmd )

class IpsgIntfHelper( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del ipsgConfig.ipsgEnabledIntf[ self.intf_.name ]

class IpsgVlanHelper( VlanCli.VlanDependentBase ):
   def destroy( self ):
      del ipsgConfig.ipsgDisabledVlan[ self.vlan_.id_ ]

def Plugin( entityManager ):
   global ipsgConfig, ipsgHwStatusDir, ipsgTable, lagConfigCli

   ipsgConfig = ConfigMount.mount( entityManager,
                                   "security/ipsg/config", "Ipsg::Config", "w" )
   ipsgHwStatusDir = LazyMount.mount( entityManager,
                                      "security/ipsg/hardware/status",
                                      "Tac::Dir", "ri" )
   ipsgTable = LazyMount.mount( entityManager,
                                "security/ipMacBinding",
                                "IpMacBinding::IpMacBindingDir", "r" )
   lagConfigCli = LazyMount.mount( entityManager, "lag/input/config/cli",
                                   "Lag::Input::Config", "r" )
   IntfCli.Intf.registerDependentClass( IpsgIntfHelper )
   VlanCli.Vlan.registerDependentClass( IpsgVlanHelper )
