#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliMatcher
import CliPlugin.IpRibCliLib as IpRibCliLib
import CliPlugin.IpRibCliShow as IpRibCliShow
from CliPlugin.IpAddrMatcher import PREFIX_OVERLAP_AUTOZERO
from CliPlugin.IpGenAddrMatcher import IpGenAddrOrPrefixExprFactory
from CliPlugin.TechSupportCli import techSupportKwMatcher
from CliPlugin.VrfCli import VrfExprFactory, vrfKwForShow
import IpRibModel
import ShowCommand
import Toggles.IpRibLibToggleLib as ipRibTlLib
from CliCommand import Node
# BUG282923 - deprecate ip (make it hidden) and use ipv4 instead
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv4 import ipv4MatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
from ConnectedRouteLeakModel import LeakedConnectedRoutesByVrf

matcherBrief = CliMatcher.KeywordMatcher( 'brief',
      helpdesc='Show a single combined summary of the RIB for all VRFs' )
matcherConnected = CliMatcher.KeywordMatcher( 'connected',
      helpdesc='Display connected route leaking state' )
matcherInternal = CliMatcher.KeywordMatcher( 'internal',
      helpdesc='Dump internal state' )
matcherLeak = CliMatcher.KeywordMatcher( 'leak',
      helpdesc='Show VRF leaking state' )
matcherLoop = CliMatcher.KeywordMatcher( 'loop',
      helpdesc='Recursive route resolution loops' )
matcherLoopRoute = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Routes that are part of recursive route resolution loops' )
matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='Multicast Routing Information Base commands' )
matcherNextHop = CliMatcher.KeywordMatcher( 'next-hop',
      helpdesc='Next-hop vias with their corresponding resolved/unresolved vias' )
matcherPrefix = IpGenAddrOrPrefixExprFactory( ipOverlap=PREFIX_OVERLAP_AUTOZERO,
                                              ip6Overlap=PREFIX_OVERLAP_AUTOZERO )
matcherResolution = CliMatcher.KeywordMatcher( 'resolution',
      helpdesc='Recursive route resolution policy' )
matcherRoute = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Routing Information Base routes' )
matcherResolutionRoute = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Routes that are used for recursive resolution' )
matcherRib = CliMatcher.KeywordMatcher( 'rib',
      helpdesc='Routing Information Base commands' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Routing Information Base summary' )
matcherVrf = VrfExprFactory( helpdesc='Show routes in a VRF' )

protocols = list( IpRibCliLib.ipRibProtocols ) + [ 'host' ]
if not ipRibTlLib.toggleRipMultiAgentEnabled():
   protocols.remove( 'rip' )
matcherProtocolOrHost = CliMatcher.EnumMatcher(
   { IpRibCliLib.ipRibProtocolString.internalToExternal( proto ) : 'Show only %s '
     'routes' % IpRibCliLib.ipRibProtocolString.internalToExternal( proto )
     for proto in protocols } )

#--------------------------------------------------------------------------------
# show rib next-hop ( ip | ipv6 ) dependency
#--------------------------------------------------------------------------------
class ShowNhDependencyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib next-hop ( ip | ipv6 ) dependency'
   data = { 'rib' : matcherRib,
            'next-hop' : matcherNextHop,
            'ip' : ipMatcherForShow,
            'ipv6' : ipv6MatcherForShow,
            'dependency' : 'Recursive dependencies of next-hops',
   }

   handler = IpRibCliShow.showNhDependency
   cliModel = IpRibModel.RecursiveNhDependencyByVrf

BasicCli.addShowCommandClass( ShowNhDependencyCmd )

#--------------------------------------------------------------------------------
# show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#    [ fib policy excluded ]
#--------------------------------------------------------------------------------
class RibRouteIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
               [ fib policy excluded ]'''
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'PREFIX' : matcherPrefix,
      'PROTOCOL' : matcherProtocolOrHost,
      'fib' : 'Match FIB criteria',
      'policy' : 'Match FIB policy',
      'excluded' : 'Show only routes excluded from FIB',
   }

   handler = IpRibCliShow.showIpRoute
   cliModel = IpRibModel.RibRoutesByProtocol

BasicCli.addShowCommandClass( RibRouteIpCmd )

#--------------------------------------------------------------------------------
# show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#--------------------------------------------------------------------------------
class RibRouteIpMulticastCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ]
               [ PROTOCOL ]'''
   data = {
      'rib' : matcherRib,
      'multicast' : matcherMulticast,
      'route' : matcherRoute,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'PREFIX' : matcherPrefix,
      'PROTOCOL' : matcherProtocolOrHost,
   }

   handler = IpRibCliShow.showIpRoute
   cliModel = IpRibModel.RibRoutesByProtocol

BasicCli.addShowCommandClass( RibRouteIpMulticastCmd )

#--------------------------------------------------------------------------------
# show rib route ( ip | ipv6 ) internal [ VRF ]
#--------------------------------------------------------------------------------
class RibRouteIpInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib route ( ip | ipv6 ) internal [ VRF ]'
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'internal' : Node( matcher=matcherInternal, hidden=True ),
      'VRF' : matcherVrf,
   }

   handler = IpRibCliShow.showIpRoute
   cliModel = IpRibModel.RibRoutesByProtocol

BasicCli.addShowCommandClass( RibRouteIpInternalCmd )

#--------------------------------------------------------------------------------
# show rib route summary [ ip | ipv6 ] [ VRF ]
#--------------------------------------------------------------------------------
class ShowRibRouteSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib route summary [ ip | ipv6 ] [ VRF ]'
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'summary' : matcherSummary,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
   }

   handler = IpRibCliShow.showRibRouteSummary
   cliModel = IpRibModel.RibRouteSummary

BasicCli.addShowCommandClass( ShowRibRouteSummary )

#--------------------------------------------------------------------------------
# show rib route summary [ ip | ipv6 ] brief
#--------------------------------------------------------------------------------
class ShowRibRouteSummaryBrief( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib route summary [ ip | ipv6 ] brief'
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'summary' : matcherSummary,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'brief' : matcherBrief,
   }

   handler = IpRibCliShow.showRibRouteSummaryBrief
   cliModel = IpRibModel.RibRouteSummaryForVrf

BasicCli.addShowCommandClass( ShowRibRouteSummaryBrief )

#--------------------------------------------------------------------------------
# show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]
#--------------------------------------------------------------------------------
class RibLoopIpRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]'
   data = {
      'rib' : matcherRib,
      'loop' : matcherLoop,
      'multicast' : matcherMulticast,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'route' : matcherLoopRoute,
      'PREFIX' : matcherPrefix,
   }

   handler = IpRibCliShow.showIpRouteLooped
   cliModel = IpRibModel.LoopedRoutes

BasicCli.addShowCommandClass( RibLoopIpRouteCmd )

#--------------------------------------------------------------------------------
# show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]
#--------------------------------------------------------------------------------
class RibNextHopResolutionRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]'
   data = {
      'rib' : matcherRib,
      'next-hop' : matcherNextHop,
      'resolution' : matcherResolution,
      'route' : matcherResolutionRoute,
      'ipv4' : ipv4MatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
   }

   handler = IpRibCliShow.showResolutionRoutes
   cliModel = IpRibModel.ResolutionRoutes

BasicCli.addShowCommandClass( RibNextHopResolutionRouteCmd )

#--------------------------------------------------------------------------------
# show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]
#--------------------------------------------------------------------------------
class RibNextHopIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]' \
            ' [ detail ]'
   data = {
      'rib' : matcherRib,
      'multicast' : matcherMulticast,
      'next-hop' : matcherNextHop,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'PROTOCOL' : matcherProtocolOrHost,
      'detail' : 'detail',
   }

   handler = IpRibCliShow.showIpRibNextHop
   cliModel = IpRibModel.RibNextHopsByProtocol

BasicCli.addShowCommandClass( RibNextHopIpCmd )

#--------------------------------------------------------------------------------
# show vrf leak connected ( ipv4 | ipv6 )
#--------------------------------------------------------------------------------
class VrfLeakConnectedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vrf leak connected ( ipv4 | ipv6 )'
   data = {
      'vrf' : vrfKwForShow,
      'leak' : matcherLeak,
      'connected' : matcherConnected,
      'ipv4' : ipv4MatcherForShow,
      'ipv6' : ipv6MatcherForShow,
   }

   handler = IpRibCliShow.showVrfLeakConnected
   cliModel = LeakedConnectedRoutesByVrf

BasicCli.addShowCommandClass( VrfLeakConnectedCmd )

#--------------------------------------------------------------------------------
# show tech-support iprib graceful-restart
#--------------------------------------------------------------------------------
class ShowTechIpRibGRCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support iprib graceful-restart'
   data = {
         'tech-support' : techSupportKwMatcher,
         'iprib' : 'Show aggregated status details for IpRib',
         'graceful-restart' : 'Graceful restart information for IpRib agent',
   }

   privileged = True
   handler = IpRibCliShow.showTechIpRibGR

BasicCli.addShowCommandClass( ShowTechIpRibGRCmd )

#--------------------------------------------------------------------------------
# show rib ready [ VRF ]
#--------------------------------------------------------------------------------
class ShowRibReadyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib ready [ VRF ]'
   data = {
      'rib' : matcherRib,
      'ready' : 'Show protocols\' routes ready state',
      'VRF' : VrfExprFactory( helpdesc='Show routes in a VRF', inclAllVrf=True,
                              inclDefaultVrf=True )
   }

   handler = IpRibCliShow.showRibReady
   cliModel = IpRibCliShow.ribReadyVrfModel

BasicCli.addShowCommandClass( ShowRibReadyCmd )
