# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Tac
from CliModel import ( Bool, Enum, List, Model, Str )

typeName = "Tunnel::TunnelTable::TunnelRibNameIdMap"
systemColoredTunnelRibName = Tac.Type( typeName ).systemColoredTunnelRibName

class ResolutionRib( Model ):
   name = Str( help="Name of the resolution RIB." )
   ribType = Enum( [ 'tunnel', 'ip' ], help="Type of the resolution RIB." )
   colored = Bool( optional=True, help="The tunnel RIB is colored." )

   @staticmethod
   def fromTac( tacValue ):
      if tacValue.resolutionRibSource == 'undefined':
         return None

      if ( tacValue.resolutionRibSource == 'systemUnicastRib' or
           tacValue.resolutionRibSource == 'systemConnected' ):
         # Omit 'colored' as it's optional
         params = { 'name' : tacValue.stringValue(), 'ribType' : 'ip' }
      elif tacValue.resolutionRibSource == 'tunnelRib':
         name = tacValue.tunnelRibName
         params = {
               'name' : name,
               'ribType' : 'tunnel',
               'colored' : name == systemColoredTunnelRibName
               }

      return ResolutionRib( **params )

   @staticmethod
   def stringValueIs( stringValue ):
      """
      Convert a string representation of a ResolutionRib into the CAPI type, e.g.
      'system-unicast-rib' or 'tunnel-rib colored foo'.
      """
      name = stringValue.split()[ -1 ]
      ribType = 'tunnel' if 'tunnel-rib' in stringValue else 'ip'
      params = {
            'name' : name,
            'ribType' : ribType,
            }
      if ribType == 'tunnel':
         params[ 'colored' ] = 'colored' in stringValue
      return ResolutionRib( **params )

class ResolutionRibProfileConfig( Model ):
   resolutionMethods = List( valueType=ResolutionRib,
                             help="List of resolution methods" )
   systemDefault = Bool( optional=True,
                         help="System default profile is set for next-hop" )

   def toTac( self ):
      """
      Converts an instance of ResolutionRibProfileConfig into an instance of
      Tac.Type( Rib::ResolutionRibProfileConfig ).
      """
      dictRepr = self.toDict()
      resolutionMethods = dictRepr[ 'resolutionMethods' ]
      systemDefault = dictRepr.get( 'systemDefault', False )
      return ResolutionRibProfileConfig.fromJson( resolutionMethods, systemDefault )

   @staticmethod
   def fromTac( tacValue ):
      """
      Converts an instance of Tac.Type( Rib::ResolutionRibProfileConfig ) into an
      instance of ResolutionRibProfileConfig
      """
      if not isinstance( tacValue,
                         Tac.Type( 'Routing::Rib::ResolutionRibProfileConfig' ) ):
         raise ValueError

      resolutionMethods = filter( None,
                                  ( ResolutionRib.fromTac( v )
                                     for v in tacValue.resolutionMethod.values() ) )
      # If it's a 'system-default' profile, it has no ribType or any real
      # ResolutionRibConfig instances at all, so hardcode the required CAPI model,
      #
      #    { 'systemDefault' : 'true' }
      if tacValue.useDefault:
         assert not resolutionMethods
         return ResolutionRibProfileConfig( systemDefault=True,
                                            resolutionMethods=[] )

      return ResolutionRibProfileConfig( resolutionMethods=resolutionMethods )

   @staticmethod
   def tacResolutionRibProfileConfig( resolutionMethods=None, useDefault=False ):
      """
      This is copied verbatim from
      IpRibLibTypesMixin.IpRibLibTypesMixin.tacResolutionRibProfile. This is to
      workaround the restriction that IpRibLib-testlib mustn't be packaged as part of
      EosImage. BUG514823 tracks deduplicating this.
      """
      resRibProfileConfig = Tac.Value( "Routing::Rib::ResolutionRibProfileConfig",
                                       useDefault=useDefault )
      if not resolutionMethods:
         resolutionMethods = []
      for ( idx, src, tunName ) in resolutionMethods:
         if src != 'tunnelRib' and tunName != '':
            raise ValueError
         config = Tac.Value( "Routing::Rib::ResolutionRibConfig", src )
         config.tunnelRibName = tunName
         resRibProfileConfig.resolutionMethod[ idx ] = config
      return resRibProfileConfig

   @staticmethod
   def fromJson( resolutionMethods, systemDefault=False ):
      """
      Returns a Rib::ResolutionRibProfileConfig from a list of ResolutionRib JSON
      objects represented as a Python dictionary, e.g.

      {
         "resolutionMethods": [
            {
               "ribType": "tunnel",
               "colored": true,
               "name": "system-colored-tunnel-rib"
            },
            {
               "ribType": "tunnel",
               "colored": false,
               "name": "system-tunnel-rib"
            },
            {
               "ribType": "ip",
               "name": "system-unicast-rib"
            }
         ]
      }
      """
      methods = []
      for i, resolutionMethod in enumerate( resolutionMethods ):
         src = None
         tunnelRibName = ''
         if resolutionMethod[ 'ribType' ] == 'tunnel':
            src = 'tunnelRib'
            tunnelRibName = resolutionMethod[ 'name' ]
         elif resolutionMethod[ 'name' ] == 'system-unicast-rib':
            src = 'systemUnicastRib'
         elif resolutionMethod[ 'name' ] == 'system-connected':
            src = 'systemConnected'
         # The CliModelChecker.py generates bogus name / ribType combinations (e.g.
         # { 'ribType': 'ip', name: 'test string' }. It is not possible to specify
         # the contstraints (e.g. 'ribType' == 'ip' => 'name' =~
         # 'system-{unicast-rib,connected}') using the CAPI model types, so be
         # generous, and _silently_ ignore such entries.
         if src in Tac.Type( 'Routing::Rib::ResolutionRibSource' ).attributes:
            methods.append( ( i, src, tunnelRibName ) )
      return ResolutionRibProfileConfig.tacResolutionRibProfileConfig(
            methods, useDefault=systemDefault )

class ResolutionRibProfile( Model ):
   resolutionMethods = List( valueType=ResolutionRib,
                             help="List of resolution methods" )
   error = Str( help="Invalid resolution profile reason", optional=True )

   @staticmethod
   def stringValueIs( stringValue ):
      if stringValue == "No profile set for this next-hop":
         return ResolutionRibProfile( error=stringValue, resolutionMethods=[] )
      methods = [ ResolutionRib.stringValueIs( val )
                  for val in stringValue.split( ',' ) ] if stringValue else []
      return ResolutionRibProfile( resolutionMethods=methods )
