# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CLI expressions and helpers for IpRibLib types.

from CliPlugin.TunnelRibCli import (
   getCustomTunnelRibNames,
   systemTunnelRibName,
   systemColoredTunnelRibName,
)
import Tac
import CliCommand
from CliCommand import hiddenKeyword
import CliMatcher
import CliToken.IpRibLibCliTokens as IpRibLibCliTokens

def getResolutionRibProfileConfig( args ):
   resRibSrcType = Tac.Type( 'Routing::Rib::ResolutionRibSource' )

   # unpack data into ResolutionRibSources
   ribs = []
   if 'IP_PRIMARY' in args:
      assert 'IP_FALL' not in args
      ribs.append( ( args[ 'IP_PRIMARY' ], None ) )

   tRibs = args.get( 'TUNNEL_RIBS', [] )
   if tRibs:
      ribs += tRibs

   if 'IP_FALL' in args:
      assert 'IP_PRIMARY' not in args
      ribs.append( ( args[ 'IP_FALL' ], None ) )

   def PopulateResRibConfig( rib, tunnelName ):
      resRib = None
      if rib == 'tunnel-rib':
         resRib = Tac.Value( 'Routing::Rib::ResolutionRibConfig',
                             resRibSrcType.tunnelRib )
         resRib.tunnelRibName = tunnelName
      elif rib == 'system-connected':
         resRib = Tac.Value( 'Routing::Rib::ResolutionRibConfig',
                             resRibSrcType.systemConnected )
      elif rib == 'system-unicast-rib':
         resRib = Tac.Value( 'Routing::Rib::ResolutionRibConfig',
                             resRibSrcType.systemUnicastRib )
      return resRib

   resRibConfs = [ PopulateResRibConfig( rib, tRib ) for rib, tRib in ribs ]

   # default values come from tacc
   resRibProf = Tac.Value( 'Routing::Rib::ResolutionRibProfileConfig' )
   for i, resRibConf in enumerate( resRibConfs ):
      resRibProf.resolutionMethod[ i ] = resRibConf

   resRibProf.useDefault = 'system-default' in args
   if 'system-default' in args:
      assert not ribs

   return resRibProf


excludeTunnelNames = '^(?!{}$|{}$|{}$)'.format( systemTunnelRibName,
                                                systemColoredTunnelRibName,
                                                'colored' )

tunnelMatcher = CliMatcher.DynamicNameMatcher(
   getCustomTunnelRibNames, 'Name of custom tunnel RIB',
   excludeTunnelNames + r'([A-Za-z0-9_-]+)$', helpname='NAME' )

ipRibsEnumMatcher = CliMatcher.EnumMatcher( {
   'system-connected': 'Resolve using system connected RIB',
   'system-unicast-rib': 'Resolve using system unicast RIB',
} )

# support hidden system-unicast-rib token. See BUG479809
systemConnected = CliMatcher.KeywordMatcher(
      'system-connected',
      helpdesc='Resolve using system connected RIB' )
systemUnicastHidden = hiddenKeyword( 'system-unicast-rib',
                                     helpdesc='Resolve using system unicast RIB' )

# unique expressions for primary and fallback required as the wrapping expression
# will have its tokens replace with these
class IpRibsHiddenUnicastPriExpr( CliCommand.CliExpression ):
   expression = '( PRIMARY_CONNECTED | PRIMARY_UNICAST_HIDDEN )'
   data = { 'PRIMARY_CONNECTED': systemConnected,
            'PRIMARY_UNICAST_HIDDEN': systemUnicastHidden }

class IpRibsHiddenUnicastFallExpr( CliCommand.CliExpression ):
   expression = '( FALL_CONNECTED | FALL_UNICAST_HIDDEN )'
   data = { 'FALL_CONNECTED': systemConnected,
            'FALL_UNICAST_HIDDEN': systemUnicastHidden }

class ResolutionRibsTunnelExpr( CliCommand.CliExpression ):
   expression = '''{ ( tunnel-rib ( %s | TUNNEL_RIB ) ) |
                     ( ctunnel-rib colored %s ) }''' % ( systemTunnelRibName,
                                                         systemColoredTunnelRibName )
   data = {
      'tunnel-rib': CliCommand.singleNode( IpRibLibCliTokens.matcherTunnelRib ),
      'TUNNEL_RIB': CliCommand.singleNode( tunnelMatcher ),
      systemTunnelRibName: \
            CliCommand.singleNode( IpRibLibCliTokens.matcherSystemTunnelRib ),
      'ctunnel-rib': CliCommand.singleNode( IpRibLibCliTokens.matcherTunnelRib ),
      'colored': CliCommand.singleNode( IpRibLibCliTokens.matcherColoredTunnelRib ),
      systemColoredTunnelRibName:
      CliCommand.singleNode( IpRibLibCliTokens.matcherSystemColoredTunnelRib ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      tRibs = []
      # We need to maintain the order of tunnel ribs specified, so we iterate
      # over the argsList
      for i, ( k, _ ) in enumerate( argsList ):
         if k == 'tunnel-rib':
            # Next token will specify the tunnel rib name
            tRibs.append( ( 'tunnel-rib', argsList[ i + 1 ][ 1 ] ) )
         elif k == 'ctunnel-rib':
            # We include additional colored token before tunnel rib name
            assert argsList[ i + 1 ][ 0 ] == 'colored'
            tRibs.append( ( 'tunnel-rib', argsList[ i + 2 ][ 1 ] ) )
      args[ 'TUNNEL_RIBS' ] = tRibs

class ResolutionRibsExpr( CliCommand.CliExpression ):
   expression = '( IP_PRIMARY [ TUNNEL_EXP ] ) | ( TUNNEL_EXP [ IP_FALL ] )'
   data = {
      'IP_PRIMARY': ipRibsEnumMatcher,
      'TUNNEL_EXP': ResolutionRibsTunnelExpr,
      'IP_FALL': ipRibsEnumMatcher,
   }

class ResolutionRibsHiddenUnicastExpr( CliCommand.CliExpression ):
   expression = '( IP_PRIMARY [ TUNNEL_EXP ] ) | ( TUNNEL_EXP [ IP_FALL ] )'
   data = {
      'IP_PRIMARY': IpRibsHiddenUnicastPriExpr,
      'TUNNEL_EXP': ResolutionRibsTunnelExpr,
      'IP_FALL': IpRibsHiddenUnicastFallExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      # map tokens to IP_PRIMARY and IP_FALL and heal hidden unicast to connected
      if 'PRIMARY_CONNECTED' in args:
         args[ 'IP_PRIMARY' ] = args[ 'PRIMARY_CONNECTED' ]
      elif 'PRIMARY_UNICAST_HIDDEN' in args:
         args[ 'IP_PRIMARY' ] = 'system-connected'
      elif 'FALL_CONNECTED' in args:
         args[ 'IP_FALL' ] = args[ 'FALL_CONNECTED' ]
      elif 'FALL_UNICAST_HIDDEN' in args:
         args[ 'IP_FALL' ] = 'system-connected'

