#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#---------------------------------------------------------------------
# This module implements the following show commands:
#
# show rib next-hop ( ip | ipv6 ) dependency
#
# show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#    [ fib policy excluded ]
# show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
# show rib route ( ip | ipv6 ) internal [ VRF ]
#
# show rib route summary [ ip | ipv6 ] [ VRF ]
# show rib route summary [ ip | ipv6 ] brief
#
# show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]
#
# show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]
#
# show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]
#
# show rib ready [ VRF ]
#
# show vrf leak connected ( ipv4 | ipv6 )
#
# show tech-support iprib graceful-restart
#---------------------------------------------------------------------

'''IpRib Show Commands'''

import AgentCommandRequest
import Arnet
import ConnectedRouteLeakModel as Crlm
import IpRib
import IpRibCliLib
import IpRibModel
import QuickTrace
import sys
import SharedMem
import Tac
import TacSigint

from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.ShowRibReadyModels import ( ProtoReadyModel,
                                           RibReadyModel,
                                           RibReadyStateModel )
from CliPlugin.VrfCli import ( ALL_VRF_NAME,
                               DEFAULT_VRF,
                               VrfExecCmdDec,
                               generateVrfCliModel,
                               getVrfNames,
                               vrfExists )
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from IraCommonCli import ( ribReadyHook,
                           ribResolutionRouteHook,
                           ribRouteHook,
                           ribRouteModel )

vrfMounter = IpRibCliLib.IpRibVrfMounter()
ribMounter = IpRibCliLib.IpRibCliMounter( vrfMounter, "rib" )
mribMounter = IpRibCliLib.IpRibCliMounter( vrfMounter, "mrib" )
tunnelRibMounter = IpRibCliLib.TunnelRibMounter()
fibRtStatusMounter = IpRibCliLib.FibRouteStatusMounter( vrfMounter )
nhDependencyMounter = IpRibCliLib.NhDependencyMounter()
allVrfIgpReadyMounter = IpRibCliLib.AllVrfIgpReadyMounter()
allVrfRibReadyMounter = IpRibCliLib.AllVrfRibReadyMounter()
fibReadyDirMounter = IpRibCliLib.FibReadyDirMounter()

entMan = None
shmemEm = None

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt8 = QuickTrace.trace8

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )

ribReadyVrfModel = generateVrfCliModel( RibReadyModel,
                                        'Per VRF Rib ready summary',
                                        uncheckedModel=True )

def getCliMounter( rib="rib" ):
   if rib == "rib":
      return ribMounter
   elif rib == "mrib":
      return mribMounter
   assert False
   return None

def getCliShmemMounter( rib="rib" ):
   return IpRibCliLib.IpRibShmemCliMounter( shmemEm, rib )

#--------------------------------------------------------------------------------
# Implements show rib next-hop ( ip | ipv6 ) dependency
#--------------------------------------------------------------------------------
def showNhDependency( mode, args ):
   nhDepGraph = nhDependencyMounter.getNhDependencyGraph()
   viaStatus = ribMounter.getViaStatus()
   vrfStatus = vrfMounter.getVrfIdStatus()
   inputAf = "ipv4" if "ip" in args else "ipv6"
   nhgEntryStatus = ribMounter.getNhgEntryStatus()

   nhp = Tac.newInstance( "Routing::Rib::NhDependencyPrinter", nhDepGraph,
                          viaStatus, vrfStatus, inputAf, nhgEntryStatus )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()
   nhp.renderNhDependency( fd, outputFormat )
   return IpRibModel.RecursiveNhDependencyByVrf

#--------------------------------------------------------------------------------
# Implements show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#               [ fib policy excluded ]
#            show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ]
#               [ PROTOCOL ]
#            show rib route ( ip | ipv6 ) internal [ VRF ]
#--------------------------------------------------------------------------------
def correctProtocolAndAddrFamily( protocol, af ):
   return ( ( protocol != 'ospf' or af == 'ipv4' ) and
            ( protocol != 'ospf3' or af == 'ipv6' ) )

def showIpRoute( mode, args ):
   rib = 'mrib' if 'multicast' in args else 'rib'
   af = 'ipv4' if 'ip' in args else 'ipv6'
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   prefix = args.get( 'PREFIX' )
   protocolOrHost = args.get( 'PROTOCOL' )
   internal = 'internal' in args
   fibExcluded = 'excluded' in args

   displayHostRoutes = False

   if protocolOrHost is None:
      protocol = Tac.Type( "Routing::Rib::RoutingProtocol" ).routingProtocols
   elif protocolOrHost == 'host':
      protocol = 'connected'
      displayHostRoutes = True
   else:
      protocol = IpRibCliLib.ipRibProtocolString.externalToInternal( protocolOrHost )

   if prefix is None:
      prefix = Tac.Value( "Arnet::IpGenPrefix" )
   else:
      if not isinstance( prefix, str ):
         prefix = prefix.stringValue

      prefix = Arnet.IpGenPrefix( prefix )

   # Support ribd mode
   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
      # Host routes, internal state, and multicast are not (yet) supported. If
      # any of these were specified by the user, an empty result is returned.
      # Call single agent code only if none of these were specified.
      if not displayHostRoutes and not internal and rib == 'rib':
         # Skip ipv4/ospf3 and ipv6/ospf
         if correctProtocolAndAddrFamily( protocol, af ):
            ribRouteHook.notifyExtensions( af, mode, vrfName, protocol, prefix,
                                           fibExcluded=fibExcluded )
      else:
         mode.addError( 'Not supported' )
         return None

      # Deferred Model - return the model class
      return IpRibModel.RibRoutesByProtocol

   if fibExcluded:
      if ( not correctProtocolAndAddrFamily( protocol, af ) or displayHostRoutes or
           internal or rib != 'rib' ):
         mode.addError( "Not supported" )
         return None


   vrfId = vrfMounter.getVrfId( vrfName )
   if vrfId is None:
      return IpRibModel.RibRoutesByProtocol()

   cliMounter = getCliMounter( rib )

   ( routeConfig, routeConfigTrie ) = cliMounter.getRouteConfig( af, vrfName )
   viaSetConfig = cliMounter.getAllViaSetConfig()

   ipv4ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv4' )
   ipv6ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv6' )
   # Evpn vias are in default vrf
   evpnViaConfig = cliMounter.getEvpnViaConfig( DEFAULT_VRF )
   mplsViaConfig = cliMounter.getMplsViaConfig( vrfName )

   ipv4ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv4' )
   ipv6ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv6' )

   viaStatus = cliMounter.getViaStatus()
   viaSetStatus = cliMounter.getAllViaSetStatus()

   winningRouteStatus = cliMounter.getWinningRouteStatus( af, vrfName )
   loopingRouteStatus = cliMounter.getLoopingRouteStatus( af, vrfName )
   nhgEntryStatus = cliMounter.getNhgEntryStatus()
   tunnelFib = cliMounter.getTunnelFib()
   ribConfigBgp = cliMounter.getRibConfigBgp( vrfId )

   routeIpv4Status = fibRtStatusMounter.getFib4RouteStatus( vrfName )
   routeIpv6Status = fibRtStatusMounter.getFib6RouteStatus( vrfName )

   # If the mount paths below are not used anywhere else in ConfigAgent,
   # a delayed auto-unmount of these paths will be triggered when exiting
   # this function (i.e. when cliShmemMounter goes out of scope).
   cliShmemMounter = getCliShmemMounter( rib )
   viaMetricStatus = cliShmemMounter.getViaMetricStatus()
   cliShmemMounter.doClose()

   if ( routeConfig is None or routeConfigTrie is None or viaSetConfig is None or
        ipv4ViaConfig is None or ipv6ViaConfig is None or evpnViaConfig is None or
        viaStatus is None or viaMetricStatus is None or
        winningRouteStatus is None or loopingRouteStatus is None or
        nhgEntryStatus is None or tunnelFib is None or routeIpv4Status is None or
        routeIpv6Status is None ):
      qt0( "Mounts Incomplete" )
      return IpRibModel.RibRoutesByProtocol

   rr = Tac.newInstance( "Routing::Rib::RouteRenderer", vrfName, vrfId, routeConfig,
                         routeConfigTrie, viaSetConfig, ipv4ViaConfig,
                         ipv6ViaConfig, evpnViaConfig, mplsViaConfig,
                         ipv4ViaCfgByProtoByVrf,
                         ipv6ViaCfgByProtoByVrf, viaStatus, viaMetricStatus,
                         winningRouteStatus, loopingRouteStatus, displayHostRoutes,
                         nhgEntryStatus, tunnelFib, None, ribConfigBgp,
                         routeIpv4Status, routeIpv6Status,
                         vrfMounter.getVrfIdStatus(), viaSetStatus,
                         tunnelRibMounter.tunnelRibNameIdMap, False )
   if mode.session_.outputFormatIsJson():
      rr.jsonModelRevision = mode.session_.requestedModelRevision()

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()

   with TacSigint.immediateMode():
      rr.renderRoute( fd, outputFormat, prefix, protocol, fibExcluded )

   if internal:
      print "Internal State for VRF", vrfName
      sys.stdout.flush()
      # Dumps output to the cli instance that's invoking the command. If set to None
      # only cohab tests will be able to see the output (which gets dumped to stdout)
      dumpCtx = Tac.Type( 'Ark::DumpContext' ).dumpContext( fd, 'dumpNormal', False )
      indentCount = 2
      if af == 'ipv4':
         ipv4ViaConfig.dumpState( dumpCtx, indentCount )
      if af == 'ipv6':
         ipv6ViaConfig.dumpState( dumpCtx, indentCount )
      if vrfName == DEFAULT_VRF:
         evpnViaConfig.dumpState( dumpCtx, indentCount )
      viaStatus.dumpState( dumpCtx, indentCount )
      viaMetricStatus.dumpState( dumpCtx, indentCount )
      viaSetConfig.dumpState( dumpCtx, indentCount )
      if routeConfig is not None:
         routeConfig.dumpState( dumpCtx, indentCount )
      winningRouteStatus.dumpState( dumpCtx, indentCount )
      print "Tunnel RIB"
      sys.stdout.flush()
      tunnelRibMounter.tunnelRib.dumpState( dumpCtx, indentCount )

   sys.stdout.flush()

   # Deferred Model - return an empty model
   return IpRibModel.RibRoutesByProtocol

#--------------------------------------------------------------------------------
# Implements show rib route summary [ ip | ipv6 ] [ VRF ]
#            show rib route summary [ ip | ipv6 ] brief
#--------------------------------------------------------------------------------
def getProtocolShowString( protocol ):
   protocolShowStrings = {
      'bgp' : 'BGP',
      'connected' : 'Connected',
      'dynamicPolicy' : 'Dynamic policy',
      'isis' : 'IS-IS',
      'ospf' : 'OSPF',
      'ospf3' : 'OSPFv3',
      'routeInput' : 'Route input',
      'staticConfig' : 'Static',
      'staticRouteCacheConfig' : 'Static Route Cache',
      'vrfLeak' : 'VRF leak',
      'bgpLu' : 'BGP labeled unicast',
      'rip': 'RIP'
   }
   return protocolShowStrings[ protocol ]

def addVrfRouteCountsToRibRouteSummaryForVrf( mode, model, cliMounter, vrfName,
                                              addressFamilies ):
   for addressFamily in addressFamilies:
      ( routeConfig, _ ) = cliMounter.getRouteConfig( addressFamily, vrfName )
      for protocol, protocolRouteConfig in routeConfig.protocolConfig.items():
         if protocol == 'reserved':
            continue
         protocolShowString = getProtocolShowString( protocol )
         protocolRoutes = protocolRouteConfig.route
         if protocolShowString not in model.routeCountPerProtocol:
            model.routeCountPerProtocol[ protocolShowString ] = len( protocolRoutes )
         else:
            model.routeCountPerProtocol[ protocolShowString ] += \
               len( protocolRoutes )

def getRibRouteSummaryForVrf( mode, cliMounter, vrfName, addressFamilies ):
   model = IpRibModel.RibRouteSummaryForVrf()

   addVrfRouteCountsToRibRouteSummaryForVrf( mode, model, cliMounter, vrfName,
                                             addressFamilies )

   return model

def getAddressFamilies( addressFamily ):
   addressFamilies = []
   if addressFamily is None:
      addressFamilies = [ 'ipv4', 'ipv6' ]
   elif addressFamily == 'ip':
      addressFamilies = [ 'ipv4' ]
   else:
      addressFamilies = [ addressFamily ]
   return addressFamilies

def showRibRouteSummary( mode, args ):
   model = IpRibModel.RibRouteSummary()
   vrf = args.get( 'VRF', DEFAULT_VRF )
   vrfs = vrfMounter.getAllVrfNames( mode ) if vrf == ALL_VRF_NAME else [ vrf ]
   addressFamilies = getAddressFamilies( args.get( 'ip' ) or args.get( 'ipv6' ) )
   cliMounter = getCliMounter( 'rib' )

   for vrf in vrfs:
      if vrfMounter.getVrfId( vrf ) is None:
         return model

   for vrf in vrfs:
      summary = getRibRouteSummaryForVrf( mode, cliMounter, vrf,
                                          addressFamilies )
      model.ribRouteSummary[ vrf ] = summary

   return model

def showRibRouteSummaryBrief( mode, args ):
   model = IpRibModel.RibRouteSummaryForVrf()
   allVrfs = vrfMounter.getAllVrfNames( mode )
   addressFamilies = getAddressFamilies( args.get( 'ip' ) or args.get( 'ipv6' ) )
   cliMounter = getCliMounter( 'rib' )

   for vrfName in allVrfs:
      addVrfRouteCountsToRibRouteSummaryForVrf( mode, model, cliMounter, vrfName,
                                                addressFamilies )

   return model

#--------------------------------------------------------------------------------
# Implements show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]
#--------------------------------------------------------------------------------
def showIpRouteLooped( mode, args ):
   rib = 'mrib' if 'multicast' in args else 'rib'
   af = 'ipv4' if 'ip' in args else 'ipv6'
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   prefix = args.get( 'PREFIX' )

   if prefix is None:
      prefix = Tac.Value( "Arnet::IpGenPrefix" )
   else:
      if not isinstance( prefix, str ):
         prefix = prefix.stringValue

      prefix = Arnet.IpGenPrefix( prefix )

   vrfId = vrfMounter.getVrfId( vrfName )
   if vrfId is None:
      return IpRibModel.LoopedRoutes()

   cliMounter = getCliMounter( rib )

   loopingRouteStatus = cliMounter.getLoopingRouteStatus( af, vrfName )

   if loopingRouteStatus is None:
      return IpRibModel.LoopedRoutes()

   rr = Tac.newInstance( "Routing::Rib::RouteRenderer", vrfName, vrfId, None,
                         None, None, None, None, None, None, None, None, None, None,
                         None, loopingRouteStatus, True, None, None, None, None,
                         None, None, vrfMounter.getVrfIdStatus(), None,
                         tunnelRibMounter.tunnelRibNameIdMap, False )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()

   with TacSigint.immediateMode():
      rr.renderLoopRoute( fd, outputFormat, prefix )

   sys.stdout.flush()

   # Deferred Model - return an empty model
   return IpRibModel.LoopedRoutes

#--------------------------------------------------------------------------------
# Implements show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]
#--------------------------------------------------------------------------------
def showResolutionRoutes( mode, args ):
   af = 'ipv4' if 'ipv4' in args else 'ipv6'
   vrfName = args.get( 'VRF', DEFAULT_VRF )

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
      # Support ribd mode. Call Rib agent function showResolutionRibdRoute
      ribResolutionRouteHook.notifyExtensions( af, mode, vrfName )
      return IpRibModel.ResolutionRoutes

   vrfId = vrfMounter.getVrfId( vrfName )
   if vrfId is None:
      return IpRibModel.ResolutionRoutes

   cliMounter = getCliMounter( 'rib' )
   # We don't need the routeConfigTrie
   ( routeConfig, _ ) = cliMounter.getRouteConfig( af, vrfName )
   ipv4ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv4' )
   ipv6ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv6' )
   # Evpn vias are in default vrf
   evpnViaConfig = cliMounter.getEvpnViaConfig( DEFAULT_VRF )
   mplsViaConfig = cliMounter.getMplsViaConfig( vrfName )
   viaSetConfig = cliMounter.getViaSetConfig( vrfName )
   viaStatus = cliMounter.getViaStatus()
   nhResPolicyStatus = cliMounter.getNhResStatus( vrfName )
   nhgEntryStatus = cliMounter.getNhgEntryStatus()
   tunnelFib = cliMounter.getTunnelFib()

   ipv4ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv4' )
   ipv6ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv6' )

   # If the mount paths below are not used anywhere else in ConfigAgent,
   # a delayed auto-unmount of these paths will be triggered when exiting
   # this function (i.e. when cliShmemMounter goes out of scope).
   cliShmemMounter = getCliShmemMounter( 'rib' )
   viaMetricStatus = cliShmemMounter.getViaMetricStatus()
   cliShmemMounter.doClose()

   if ( routeConfig is None or
        ipv4ViaConfig is None or
        ipv6ViaConfig is None or
        evpnViaConfig is None or
        viaSetConfig is None or
        viaStatus is None or
        viaMetricStatus is None or
        nhResPolicyStatus is None or
        nhgEntryStatus is None or
        tunnelFib is None ):
      qt0( "Mounts incomplete" )
      return IpRibModel.ResolutionRoutes

   rr = Tac.newInstance( "Routing::Rib::RouteRenderer", vrfName, vrfId, routeConfig,
                         None, viaSetConfig, ipv4ViaConfig, ipv6ViaConfig,
                         evpnViaConfig, mplsViaConfig, ipv4ViaCfgByProtoByVrf,
                         ipv6ViaCfgByProtoByVrf, viaStatus, viaMetricStatus, None,
                         None, True, nhgEntryStatus, tunnelFib, nhResPolicyStatus,
                         None, None, None, vrfMounter.getVrfIdStatus(), None,
                         tunnelRibMounter.tunnelRibNameIdMap, False )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()
   # BUG282681 - Once we modify NextHopResPolicyConfig/Status to be per-AF,
   #             the appropriate nhResPolicyStatus should be fetched, then we do
   #             not need to pass in af into renderResolutionRoute
   with TacSigint.immediateMode():
      rr.renderResolutionRoute( fd, outputFormat, af )

   sys.stdout.flush()

   # Deferred Model - return an empty Model
   return IpRibModel.ResolutionRoutes

#--------------------------------------------------------------------------------
# Implements show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]
#--------------------------------------------------------------------------------
def showIpRibNextHop( mode, args ):
   rib = 'mrib' if 'multicast' in args else 'rib'
   af = 'ipv4' if 'ip' in args else 'ipv6'
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   protocolOrHost = args.get( 'PROTOCOL' )
   detail = 'detail' in args

   displayHostRoutes = False

   if protocolOrHost is None:
      protocol = Tac.Type( "Routing::Rib::RoutingProtocol" ).routingProtocols
   elif protocolOrHost == 'host':
      protocol = 'connected'
      displayHostRoutes = True
   else:
      protocol = IpRibCliLib.ipRibProtocolString.externalToInternal( protocolOrHost )

   vrfId = vrfMounter.getVrfId( vrfName )
   if vrfId is None:
      return IpRibModel.RibNextHopsByProtocol()

   ipv4ViaConfig = None
   ipv6ViaConfig = None
   evpnViaConfig = None
   mplsViaConfig = None

   cliMounter = getCliMounter( rib )

   if af == 'ipv4':
      ipv4ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv4' )
   elif af == 'ipv6':
      ipv6ViaConfig = cliMounter.getIpViaConfig( vrfName, 'ipv6' )
   else:
      # Evpn vias are in default vrf
      evpnViaConfig = cliMounter.getEvpnViaConfig( DEFAULT_VRF )
      mplsViaConfig = cliMounter.getMplsViaConfig( vrfName )

   ipv4ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv4' )
   ipv6ViaCfgByProtoByVrf = cliMounter.getAllViaConfigPerTransport( 'ipv6' )

   viaStatus = cliMounter.getViaStatus()

   winningRouteStatus = cliMounter.getWinningRouteStatus( af, vrfName )
   loopingRouteStatus = cliMounter.getLoopingRouteStatus( af, vrfName )
   nhgEntryStatus = cliMounter.getNhgEntryStatus()
   tunnelFib = cliMounter.getTunnelFib()

   # If the mount paths below are not used anywhere else in ConfigAgent,
   # a delayed auto-unmount of these paths will be triggered when exiting
   # this function (i.e. when cliShmemMounter goes out of scope).
   cliShmemMounter = getCliShmemMounter( rib )
   viaMetricStatus = cliShmemMounter.getViaMetricStatus()
   cliShmemMounter.doClose()

   if ( ( ipv4ViaConfig is None and ipv6ViaConfig is None and evpnViaConfig is None )
        or ( viaStatus is None or viaMetricStatus is None or
        winningRouteStatus is None or loopingRouteStatus is None or
        nhgEntryStatus is None or tunnelFib is None ) ):
      return IpRibModel.RibNextHopsByProtocol()

   rr = Tac.newInstance( "Routing::Rib::RouteRenderer", vrfName, vrfId, None, None,
                         None, ipv4ViaConfig, ipv6ViaConfig,
                         evpnViaConfig, mplsViaConfig, ipv4ViaCfgByProtoByVrf,
                         ipv6ViaCfgByProtoByVrf, viaStatus, viaMetricStatus,
                         winningRouteStatus, loopingRouteStatus, displayHostRoutes,
                         nhgEntryStatus, tunnelFib, None, None, None, None,
                         vrfMounter.getVrfIdStatus(), None,
                         tunnelRibMounter.tunnelRibNameIdMap, detail )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()

   with TacSigint.immediateMode():
      rr.renderNextHop( fd, outputFormat, protocol )

   sys.stdout.flush()

   # Deferred Model - return an empty model
   return IpRibModel.RibNextHopsByProtocol

#--------------------------------------------------------------------------------
# Implements show rib ready [ VRF ]
#
# This command is implemented in both 'ribd' and 'multi-agent' protocol models.
# Although, in the multi-agent model, this command has been capified as well.
# The output for this command differs a bit too, in both these models.
# The registration for the command was earlier done in gated but has been moved to
# IpRib to prevent dependency issues. The command callback
# method takes care of whether to call the 'ribd' implementation or 'multi-agent'
# implementation based on the protocol model in use.
#
# Sample output of command
# =======================
# Vrf default
# =======================
# Start time : 0:00:43 ago
#
#    State                             StateReached      TimeTaken( in sec )
# --------------------------------- ------------------ ---------------------
#    Fib ready                         True                            2.490
#    ...
#
#    Protocol     Configured     ConfigPending     Ready    TimeTaken( in sec )
# -------------- ------------   ----------------  -------  --------------------
#    connected    True           False             True                   1.700
#    ...
#--------------------------------------------------------------------------------
@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=ribReadyVrfModel )
def showRibReady( mode, args ): # pylint: disable-msg=inconsistent-return-statements
   vrfName = args.get( 'VRF', DEFAULT_VRF )

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd:
      # We are running with protocol model ribd. So the output should be fetched via
      # the method gated has registered as one of the extensions in its CliPlugin
      for hook in ribReadyHook.extensions():
         return hook( mode, vrfName )

   ribReadyModel = RibReadyModel()
   # pylint: disable-msg=protected-access
   ribReadyModel._protocolModel = "multi-agent"

   if not vrfExists( vrfName ):
      mode.addError( "Vrf %s doesn't exist" % ( vrfName ) )
      return

   vrfIgpReady = allVrfIgpReadyMounter.allVrfIgpReady.vrfIgpReady.get( vrfName )
   vrfRibReady = allVrfRibReadyMounter.allVrfRibReady.vrfRibReady.get( vrfName )

   if not vrfIgpReady or not vrfRibReady:
      return

   ribReadyModel._vrf = vrfName # pylint: disable-msg=protected-access
   ribReadyModel.startTime = IpRibCliLib.toUtc( vrfRibReady.startTime )
   ribReadyModel.gracefullyRestarting = vrfRibReady.gracefullyRestarting
   # The routes dict is use by ribd command, but this needs to be set to
   # None so that an empty dict does not show up in the json output
   ribReadyModel.routes = None

   def roundedTimeInterval( readyTime ):
      timeTaken = readyTime - vrfRibReady.startTime
      if timeTaken > 0:
         return round( timeTaken, 3 )
      else:
         # ConnectedRoute and StaticRoute processing is independent of when IpRib
         # starts and startTime is recorded. If ConnectedRoute and StaticRoute have
         # nothing to do and declare themselves ready even before IpRib starts,
         # this situation will occur.
         return 0.0

   stateModel = RibReadyStateModel()
   stateModel.igpReady = vrfIgpReady.ready
   if vrfIgpReady.ready:
      stateModel.igpReadyTime = IpRibCliLib.toUtc( vrfIgpReady.readyTime )
      stateModel.timeTakenForIgpReady = roundedTimeInterval( vrfIgpReady.readyTime )
   stateModel.ribReady = vrfRibReady.vrfReady
   if vrfRibReady.vrfReady:
      stateModel.ribReadyTime = IpRibCliLib.toUtc( vrfRibReady.vrfReadyTime )
      stateModel.timeTakenForRibReady = \
                                    roundedTimeInterval( vrfRibReady.vrfReadyTime )
   stateModel.recursiveResolution = vrfRibReady.vrfRecursiveResolutionReady
   if vrfRibReady.vrfRecursiveResolutionReady:
      stateModel.recursiveResolutionTime = \
                     IpRibCliLib.toUtc( vrfRibReady.vrfRecursiveResolutionReadyTime )
      stateModel.timeTakenForRecursiveResolution = \
                  roundedTimeInterval( vrfRibReady.vrfRecursiveResolutionReadyTime )
   stateModel.fibReady = vrfName in fibReadyDirMounter.fibReadyDir.entryState and \
                         fibReadyDirMounter.fibReadyDir.entity[ vrfName ].vrfReady
   # To make sure that we do not incorrectly display fibReadyTime as negative value
   # when we switch from ribd to multi-agent model, we check that fibReady is set
   # and so is fibReadyTime
   if stateModel.fibReady and vrfRibReady.fibReadyTime:
      stateModel.fibReadyTime = IpRibCliLib.toUtc( vrfRibReady.fibReadyTime )
      stateModel.timeTakenForFibReady = \
                                    roundedTimeInterval( vrfRibReady.fibReadyTime )

   ribReadyModel.states = stateModel

   protocolModels = {}
   for protoEntry in vrfRibReady.vrfRibProtoReady:
      proto = vrfRibReady.vrfRibProtoReady[ protoEntry ]
      protocol = proto.routeProtoName
      if protocol == "staticConfig":
         protocol = "static"
      elif protocol == "ospf3":
         protocol = "ospfv3"
      model = ProtoReadyModel()
      model.configPending = proto.configPending
      model.configured = proto.configured
      model.ready = proto.ready
      if proto.readyTime:
         model.readyTime = IpRibCliLib.toUtc( proto.readyTime )
         model.timeTaken = roundedTimeInterval( proto.readyTime )
      protocolModels[ protocol ] = model

   ribReadyModel.protocols = protocolModels

   return ribReadyModel

#--------------------------------------------------------------------------------
# Implements show vrf leak connected ( ipv4 | ipv6 )
#--------------------------------------------------------------------------------
def showVrfLeakConnected( mode, args ):
   af = 'ipv4' if 'ipv4' in args else 'ipv6'

   crlStatus = ribMounter.getCrLeakStatus( af )
   model = Crlm.LeakedConnectedRoutesByVrf()
   vrfIdStatus = vrfMounter.getVrfIdStatus()
   model.initFromTacModel( crlStatus, vrfIdStatus )

   return model

#--------------------------------------------------------------------------------
# Implements show tech-support iprib graceful-restart
#--------------------------------------------------------------------------------
def showTechIpRibGR( mode, args ):
   AgentCommandRequest.runSocketCommand( entMan, IpRib.agentName(),
                                         'debug', 'DUMP_GR_STATE' )

def Plugin( entityManager ):
   global entMan
   global shmemEm

   entMan = entityManager
   shmemEm = SharedMem.entityManager( sysdbEm=entMan )

   vrfMounter.setEntityManager( entityManager )
   ribMounter.setEntityManager( entityManager )
   mribMounter.setEntityManager( entityManager )
   tunnelRibMounter.setEntityManager( entityManager )
   fibRtStatusMounter.setEntityManager( entityManager )
   nhDependencyMounter.setEntityManager( entityManager )
   allVrfIgpReadyMounter.setEntityManager( entityManager )
   allVrfRibReadyMounter.setEntityManager( entityManager )
   fibReadyDirMounter.setEntityManager( entityManager )
   ribRouteModel[ 'IpRib' ] = IpRibModel
