#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing, QuickTrace
import SharedMem, Smash, SmashLazyMount, LazyMount, Cell
import Shark
from IpLibConsts import DEFAULT_VRF
from GenericReactor import GenericReactor
from Toggles.IpRibLibToggleLib import toggleIpRibMplsViaEnabled

ipRibProtocols = set( Tac.Type( "Routing::Rib::RoutingProtocol" ).attributes )
ipRibProtocols.remove( 'staticRouteCacheConfig' )
ipRibProtocols.remove( 'routingProtocols' )

ipRibProtocolString = Tac.newInstance( "Routing::Rib::RoutingProtocolString" )

ipRibTransport = Tac.Value( "Routing::Rib::Transport" )
tacTypeViaSetStatus = Tac.Type( 'Routing::Rib::ViaSetStatus' )

t0 = Tracing.trace0

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt8 = QuickTrace.trace8


class NhDependencyMounter( object ):
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.nhDepGraph = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getNhDependencyGraph( self ):
      if not self.nhDepGraph:
         readerInfo = SmashLazyMount.mountInfo( 'reader' )
         mountPath = "routing/rib/nextHopDependencyGraph"
         self.nhDepGraph = self.shmemEm.doMount( mountPath,
                           "Routing::Rib::Ip::GlobalNextHopDependencyGraph",
                           readerInfo )
      return self.nhDepGraph

def isRouteConfigFlattened( proto, af, rib="rib" ):
   return Tac.Type( "Routing::RouteConfigLib" )().isAllVrfRouteConfig( proto,
                                                     af, rib )

class IpRibVrfMounter( object ):

   def __init__( self ):

      self.entityManager = None
      self.shmemEm = None
      self.allVrfConfig = None
      self.vrfNameStatus = None
      self.ribMounters = []
      self.vrfReactor = None
      self.vrfIdStatus = None

   def setEntityManager( self, em ):

      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getVrfNames( self, mode ):

      if self.entityManager is None:
         return []

      if self.allVrfConfig is None:
         self.allVrfConfig = LazyMount.mount( self.entityManager, 'ip/vrf/config',
                                              'Ip::AllVrfConfig', 'r' )
      return sorted( self.allVrfConfig.vrf.members() )

   def getAllVrfNames( self, mode ):

      vrflist = self.getVrfNames( mode )
      vrflist.append( DEFAULT_VRF )
      return vrflist

   def getVrfId( self, vrfName ):

      if self.vrfNameStatus is None:
         self.vrfNameStatus = LazyMount.mount( self.entityManager,
                                               Cell.path( 'vrf/vrfNameStatus' ),
                                               'Vrf::VrfIdMap::NameToIdMapWrapper',
                                               'r' )
      if self.vrfNameStatus.nameToIdMap is None:
         return None

      if self.vrfReactor is None:
         self.vrfReactor = GenericReactor( self.vrfNameStatus.nameToIdMap,
                                           [ 'vrfNameToId' ], self.handleVrf )
      return self.vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )

   def getVrfIdStatus( self ):

      if self.vrfIdStatus is None:
         readerInfo = SmashLazyMount.mountInfo( 'reader' )
         mountPath = "vrf/vrfIdMapStatus"
         self.vrfIdStatus = self.shmemEm.doMount( mountPath,
                                                  "Vrf::VrfIdMap::Status",
                                                  readerInfo )
      return self.vrfIdStatus

   def handleVrf( self, reactor, vrfName ):
      for ribMounter in self.ribMounters:
         ribMounter.handleVrf( vrfName )

class TunnelRibMounter( object ):
   def __init__( self ):
      self.entityManager = None
      self._tunnelRib = None
      self._tunnelRibNameIdMap = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def tunnelRib( self ):
      if self._tunnelRib is None:
         self._tunnelRib = SmashLazyMount.mount(
            self.entityManager, 'tunnel/tunnelRibs/status/0',
            'Tunnel::TunnelTable::TunnelRib',
            SmashLazyMount.mountInfo( 'reader' ) )
      return self._tunnelRib

   @property
   def tunnelRibNameIdMap( self ):
      if self._tunnelRibNameIdMap is None:
         self._tunnelRibNameIdMap = LazyMount.mount(
            self.entityManager, 'tunnel/tunnelRibs/tunnelRibNameIdMap',
            'Tunnel::TunnelTable::TunnelRibNameIdMap', 'r' )
         LazyMount.force( self._tunnelRibNameIdMap )
      return self._tunnelRibNameIdMap

class FibRouteStatusMounter( object ):
   def __init__( self, vrfMounter ):
      self.entityManager = None
      self.shmemEm = None
      self.fib4RouteStatus = {}
      self.fib6RouteStatus = {}
      self.vrfMounter = vrfMounter

      self.vrfMounter.ribMounters.append( self )

   def handleVrf( self, vrfName ):
      qt8( "Handling VRF ", qv( vrfName ) )
      if vrfName in self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId:
         return

      def doUnmount( self, pathPrefix, routeStatus ):
         if vrfName in routeStatus and vrfName != DEFAULT_VRF:

            mountPath = '%s/vrf/status/%s' % ( pathPrefix, vrfName )
            self.shmemEm.doUnmount( mountPath )
            del routeStatus[ vrfName ]

      doUnmount( self, 'routing', self.fib4RouteStatus )
      doUnmount( self, 'routing6', self.fib6RouteStatus )

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getFib4RouteStatus( self, vrfName ):
      if vrfName in self.fib4RouteStatus:
         return self.fib4RouteStatus[ vrfName ]

      mountPath = None
      if vrfName == DEFAULT_VRF:
         mountPath = 'routing/status'
      else:
         mountPath = 'routing/vrf/status/%s' % vrfName
      self.fib4RouteStatus[ vrfName ] = self.shmemEm.doMount(
         mountPath, 'Smash::Fib::RouteStatus',
         Smash.mountInfo( 'reader' ) )
      return self.fib4RouteStatus[ vrfName ]

   def getFib6RouteStatus( self, vrfName ):
      if vrfName in self.fib6RouteStatus:
         return self.fib6RouteStatus[ vrfName ]

      mountPath = None
      if vrfName == DEFAULT_VRF:
         mountPath = 'routing6/status'
      else:
         mountPath = 'routing6/vrf/status/%s' % vrfName
      self.fib6RouteStatus[ vrfName ] = self.shmemEm.doMount(
          mountPath, 'Smash::Fib6::RouteStatus',
          Smash.mountInfo( 'reader' ) )
      return self.fib6RouteStatus[ vrfName ]

class IpRibCliMounter( object ):
   '''This class mounts IpRib config and status entities in sysdb/smash.
   It mounts entities the first time they're needed and caches them'''

   def __init__( self, vrfMounter, rib="rib" ):

      assert rib in [ "rib", "mrib" ]

      self.entityManager = None
      self.shmemEm = None
      self.readerInfo = SmashLazyMount.mountInfo( 'reader' )
      self.keyShadowInfo = SmashLazyMount.mountInfo( 'keyshadow' )

      self.vrfMounter = vrfMounter
      # So the VRF mounter can notify us about VRF additions/deletions
      self.vrfMounter.ribMounters.append( self )
      self.rib = rib

      self.routeConfigs = {}
      self.cachedRouteConfig = {}
      self.routeConfigTries = {}
      self.routeConfigTrieSms = {}
      self.viaConfigByProtoByVrf = {}
      self.winningRouteStatuses = {}
      self.loopingRouteStatus = None
      self.tunnelFib = None
      self.nhResPolicyStatus = None
      self.ribConfig = None
      self.ribConfigBgp = None
      self.crLeakStatuses = {}

      # For each AF, we store a dict by VRF Name
      for af in [ 'ipv4', 'ipv6' ]:
         self.routeConfigs[ af ] = {}
         self.routeConfigTries[ af ] = {}
         self.routeConfigTrieSms[ af ] = {}
         self.winningRouteStatuses[ af ] = {}

      self.viaSetConfigByProto = \
         Tac.newInstance( "Routing::Rib::ViaSetConfigByProtocol" )

      vsck = Tac.newInstance( "Routing::Rib::ViaSetConfigKey" )
      self.viaSetConfig = Tac.newInstance( "Routing::Rib::ViaSetConfig", vsck )

      self.viaSetStatusByProto = \
         Tac.newInstance( "Routing::Rib::ViaSetStatusByProtocol" )

      self.viaConfig = {}
      self.transports = [ 'ipv4', 'ipv6', 'evpn', 'mpls' ]
      for transport in self.transports:
         # For each transport, we store a dict by VRF Name
         self.viaConfig[ transport ] = {}

         self.viaConfigByProtoByVrf[ transport ] = \
            Tac.newInstance( "Routing::Rib::Ip::ViaConfigByProtocolByVrf" )
      # We maintain a dict of ViaConfigSplitterSm, keyed by the
      # ViaConfig mount path
      self.viaConfigSplitterSmColl = {}

      self.viaStatus = None
      self.viaMetricStatus = None
      self.vrfNameToViaSetConfigId = {}
      self.nhgEntryStatus = None

   def setEntityManager( self, em ):

      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def handleVrf( self, vrfName ):
      qt8( "Handling VRF ", qv( vrfName ) )
      if vrfName in self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId:
         return

      qt8( "Clearing VRF Name", qv( vrfName ) )

      # Vrf has been deleted. Delete all the objects we cached for this VRF,
      # they will be recreated the next time a CLI involving that VRF is issued.
      for af in [ 'ipv4', 'ipv6' ]:
         if vrfName in self.routeConfigs[ af ]:
            del self.routeConfigs[ af ][ vrfName ]

         if vrfName in self.routeConfigTries[ af ]:
            del self.routeConfigTries[ af ][ vrfName ]

         if vrfName in self.routeConfigTrieSms[ af ]:
            del self.routeConfigTrieSms[ af ][ vrfName ]

         if vrfName in self.winningRouteStatuses[ af ]:
            del self.winningRouteStatuses[ af ][ vrfName ]

      vrfId = None
      for transport in self.transports:
         if vrfName in self.viaConfig[ transport ]:
            if not vrfId:
               vrfId = self.viaConfig[ transport ][ vrfName ].vrfId
            del self.viaConfig[ transport ][ vrfName ]

            if vrfId in self.viaConfigByProtoByVrf[ transport ].protocolConfig:
               del self.viaConfigByProtoByVrf[ transport ].protocolConfig[ vrfId ]

      # Cleanup the viaKeyConfig's for this VRF
      if vrfId is not None:
         for _, splitterSm in self.viaConfigSplitterSmColl.items():
            del splitterSm.viaKeyConfigAllVrf.viaKeyConfigPerVrf[ vrfId ]

      vsci = self.vrfNameToViaSetConfigId.get( vrfName )
      if vsci is None:
         return

      for proto in ipRibProtocols:
         if not self.viaSetConfig.perVrf( proto ):
            continue

         vsck = Tac.newInstance( "Routing::Rib::ViaSetConfigKey", proto, vsci )
         if vsck in self.viaSetConfigByProto.protocolConfig:
            del self.viaSetConfigByProto.protocolConfig[ vsck ]

         del self.viaSetStatusByProto.protocolStatus[ proto ]

   def getRouteConfig( self, af, vrfName ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None, None

      routeConfig = self.routeConfigs[ af ]
      routeConfigTrie = self.routeConfigTries[ af ]

      vrfConfig = routeConfig.get( vrfName )
      vrfConfigTrie = routeConfigTrie.get( vrfName )

      if vrfConfig is None :
         vrfConfig = Tac.newInstance( "Routing::Rib::RouteConfigByProtocol",
                                       vrfName, vrfId )
         vrfConfigTrie = Tac.newInstance(
             "Routing::Rib::RouteConfigTrieByProtocol",
             vrfName )
         vrfConfigTrieSm = Tac.newInstance(
            "Routing::Rib::RouteConfigTrieSmByProtocol",
            vrfName )

         for proto in ipRibProtocols:

            isRCFlattened = isRouteConfigFlattened( proto, af, self.rib )
            if isRCFlattened:
               mountPath = 'routing/' + self.rib + '/routeConfig/' + af + '/' + \
                           proto
               protoConfig = self.cachedRouteConfig.get( mountPath )
               if protoConfig is None:
                  protoConfig = self.shmemEm.doMount( mountPath,
                                     'Routing::Rib::RouteConfig',
                                     self.keyShadowInfo )
                  self.cachedRouteConfig[ mountPath ] = protoConfig
            else:
               mountPath = 'routing/' + self.rib + '/routeConfig/' + af + "/" + \
                           str( vrfId ) + "/" + proto
               protoConfig = self.shmemEm.doMount( mountPath,
                                                   'Routing::Rib::RouteConfig',
                                                   self.keyShadowInfo )
            vrfConfig.protocolConfig.addMember( protoConfig )
            if isRCFlattened:
               vrfConfig.isRouteConfigFlattened.add( proto )
            else:
               vrfConfig.isRouteConfigFlattened.remove( proto )
            protoConfigTrie = vrfConfigTrie.routeConfigTrie.newMember( proto )
            protoConfigTrie.trie = ( proto, )
            protoConfigTrie.trie.af = af
            vrfConfigTrieSm.routeConfigTrieSm.newMember( protoConfig,
                                                         protoConfigTrie,
                                                         vrfId )
            qt8( "Vrf ", qv( vrfName ), " Routes: ", 
                  qv( len( protoConfig.route ) ),
                  " Trie Routes: ", qv( len( protoConfigTrie.trie.trieNode ) ),
                  " Protocol: ", qv( proto ), " VRF ID: ", qv( vrfId ) )
         self.routeConfigs[ af ][ vrfName ] = vrfConfig
         self.routeConfigTries[ af ][ vrfName ] = vrfConfigTrie
         self.routeConfigTrieSms[ af ][ vrfName ] = vrfConfigTrieSm
      else:
         qt8( "Returning existing route config for VRF ", qv( vrfName ),
                             " VRF ID: ", qv( vrfId ) )

      return ( vrfConfig, vrfConfigTrie )

   def getAllViaSetConfig( self ):
      '''Calls getViaSetConfig for all vrfs in order to mount ViaSetConfigs for all
      vrfs. This is required for the vrf leaking scenario where a route is in one
      vrf, and the ViaSetConfig is in another. Returns a ViaSetConfigByProtocol 
      instance, which is a flattened entity of ViaConfigByProtocol accross all vrfs.
      '''
      for vrfName in self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId:
         # We do not care about the return value, we simply need this
         # to mount the collection for all the vrfs.
         _ = self.getViaSetConfig( vrfName )

      return self.viaSetConfigByProto

   def getViaSetConfig( self, vrfName ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None

      for proto in ipRibProtocols:

         viaSetConfigId = None
         if self.viaSetConfig.perVrf( proto ):
            viaSetConfigId = vrfId
            self.vrfNameToViaSetConfigId[ vrfName ] = viaSetConfigId
         else:
            viaSetConfigId = Tac.Value( "Routing::Rib::ViaSetKey" ).allVrfId

         vsck = Tac.newInstance( "Routing::Rib::ViaSetConfigKey", proto,
                                 viaSetConfigId )
         if vsck in self.viaSetConfigByProto.protocolConfig:
            continue

         viaSetConfig = self.viaSetConfigByProto.protocolConfig.newMember( vsck )
         mountPath = self.viaSetConfig.mountPath( self.rib, proto, vrfName )
         qt8( "Mount Path ", qv( mountPath ) )
         smashColl = self.shmemEm.doMount( mountPath,
                                           'Routing::Rib::Smash::ViaSetCollection',
                                           self.readerInfo )
         viaSetConfig.configViaSetColl = smashColl

      return self.viaSetConfigByProto

   def getAllViaSetStatus( self ):
      for proto in ipRibProtocols:
         if proto in self.viaSetStatusByProto.protocolStatus:
            continue
         viaSetStatus = self.viaSetStatusByProto.protocolStatus.newMember( proto )
         mountPath = tacTypeViaSetStatus.mountPath( self.rib, proto )
         smashStatus = self.shmemEm.doMount( mountPath,
                                             'Routing::Rib::Smash::ViaSetStatus',
                                             self.readerInfo )
         viaSetStatus.smashStatus = smashStatus
      return self.viaSetStatusByProto

   def getAllViaConfigPerTransport( self, transport ):
      viaConfigByProtoByVrf = self.viaConfigByProtoByVrf[ transport ]

      for vrfName, vrfId in \
            self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId.items():
         viaConfig = self.getIpViaConfig( vrfName, transport )
         viaConfigByProtoByVrf.protocolConfig[ vrfId ] = viaConfig

      return viaConfigByProtoByVrf

   def getViaConfigByViaType( self, vrfName, transport, viaType ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None

      transportConfig = self.viaConfig.get( transport )
      vrfConfig = transportConfig.get( vrfName )
      if vrfConfig is not None:
         qt8( "Returning existing via config for VRF ", qv( vrfName ) )
         return vrfConfig

      # The Via Config doesn't exist, instantiate a new one

      if viaType == 'ip':
         vcbpType = 'Routing::Rib::Ip::ViaConfigByProtocol'
         vcType = 'Routing::Rib::Ip::ViaConfig'
      elif viaType == 'evpn':
         vcbpType = 'Routing::Rib::Evpn::ViaConfigByProtocol'
         vcType = 'Routing::Rib::Evpn::ViaConfig'
      elif viaType == 'mpls':
         vcbpType = 'Routing::Rib::Mpls::ViaConfigByProtocol'
         vcType = 'Routing::Rib::Mpls::ViaConfig'
      else:
         return None

      vrfConfig = Tac.newInstance( vcbpType, vrfId )

      viaConfigLib = Tac.Type( 'Routing::ViaConfigLib' )

      for proto in ipRibProtocols:
         if viaConfigLib.isAllVrfViaConfig( str( proto ), transport, self.rib ):
            mp = 'routing/' + self.rib + '/viaConfig/' + transport + '/' + proto
            vrfIdArg = Tac.Value( "Vrf::VrfIdMap::VrfId" )
         else:
            mp = 'routing/' + self.rib + '/viaConfig/' + transport + '/' + \
                 str( vrfId ) + '/' + proto
            vrfIdArg = vrfId

         viaConfig = self.shmemEm.doMount( mp, vcType, self.keyShadowInfo )
         viaConfig.rp = proto

         if mp not in self.viaConfigSplitterSmColl:
            if viaType == 'ip':
               if not ( transport == 'ipv4' or transport == 'ipv6' ):
                  return None

               t = getattr( ipRibTransport, transport )
               transportVal = Tac.Value( 'Routing::Rib::Transport', t )
               viaKeyConfigAllVrfKey = Tac.Value(
                                       "Routing::Rib::Ip::ViaKeyConfigAllVrfKey",
                                       proto, transportVal, vrfIdArg,
                                       self.rib == "rib" )
               viaKeyConfigAllVrf = Tac.newInstance(
                                       "Routing::Rib::Ip::ViaKeyConfigAllVrf",
                                       viaKeyConfigAllVrfKey )
               viaConfigSplitterSm = Tac.newInstance(
                                       "Routing::Rib::ViaConfigSplitterSm",
                                       mp, viaConfig, viaKeyConfigAllVrf )
            elif viaType == 'evpn':
               viaKeyConfigAllVrf = Tac.newInstance(
                                    "Routing::Rib::Evpn::ViaKeyConfigAllVrf", mp )
               viaConfigSplitterSm = Tac.newInstance(
                                     "Routing::Rib::EvpnViaConfigSplitterSm",
                                     mp, viaConfig, viaKeyConfigAllVrf )
            elif viaType == 'mpls':
               viaKeyConfigAllVrf = Tac.newInstance(
                                    "Routing::Rib::Mpls::ViaKeyConfigAllVrf", mp )
               viaConfigSplitterSm = Tac.newInstance(
                                     "Routing::Rib::MplsViaConfigSplitterSm",
                                     mp, viaConfig, viaKeyConfigAllVrf )

            qt8( "Instantiated viaKeyConfigAllVrf and SplitterSm for ",
                 qv( mp ) )
            self.viaConfigSplitterSmColl[ mp ] = viaConfigSplitterSm
         else:
            viaConfigSplitterSm = self.viaConfigSplitterSmColl[ mp ]
            viaKeyConfigAllVrf = viaConfigSplitterSm.viaKeyConfigAllVrf
         # This creates a viaKeyConfig for the VRF is not present already.
         viaKeyConfig = viaKeyConfigAllVrf.viaKeyConfigIs( vrfId, viaConfig )
         # The VRF is valid and hence mark it as not tentative
         viaKeyConfig.tentative = False
         vrfConfig.protocolConfig.addMember( viaKeyConfig )

      self.viaConfig[ transport ][ vrfName ] = vrfConfig
      return vrfConfig

   def getIpViaConfig( self, vrfName, transport ):
      return self.getViaConfigByViaType( vrfName, transport, 'ip' )

   def getEvpnViaConfig( self, vrfName ):
      return self.getViaConfigByViaType( vrfName, 'evpn', 'evpn' )

   def getMplsViaConfig( self, vrfName ):
      if not toggleIpRibMplsViaEnabled():
         return None

      return self.getViaConfigByViaType( vrfName, 'mpls', 'mpls' )

   def getWinningRouteStatus( self, af, vrfName ):

      routeStatus = self.winningRouteStatuses[ af ]

      vrfStatus = routeStatus.get( vrfName )
      if vrfStatus is not None:
         return vrfStatus

      vrfStatus = Tac.newInstance( "Routing::Rib::WinningRouteStatusByProtocol",
                                   vrfName )

      for proto in ipRibProtocols:
         mountPath = 'routing/' + self.rib + '/winningRoute/' + af + "/" + \
                     proto
         protoStatus = self.shmemEm.doMount( mountPath,
                                      'Routing::Rib::UnifiedWinningRouteStatus',
                                       self.readerInfo )
         vrfStatus.protocolStatus.addMember( protoStatus )

      self.winningRouteStatuses[ af ][ vrfName ] = vrfStatus

      return vrfStatus

   def getLoopingRouteStatus( self, af, vrfName ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None

      if self.loopingRouteStatus is None:
         path = Cell.path( "routing/" + self.rib + "/allVrfLoopingRouteStatus" )

         self.loopingRouteStatus = \
               LazyMount.mount( self.entityManager, path,
                                'Routing::Rib::AllVrfLoopingRouteStatus', 'r' )
         LazyMount.force( self.loopingRouteStatus )

      if af == "ipv4":
         afStatus = self.loopingRouteStatus.ipv4VrfLoopingRouteStatus
      else:
         afStatus = self.loopingRouteStatus.ipv6VrfLoopingRouteStatus

      return afStatus.get( vrfId )

   def getViaStatus( self ):

      if self.viaStatus is None:
         path = Cell.path( 'routing/' + self.rib + '/viaStatus' )
         self.viaStatus = LazyMount.mount( self.entityManager, path,
                                           'Routing::Rib::ViaStatusByVrf', 'r' )
         LazyMount.force( self.viaStatus )

      return self.viaStatus

   def getNhgEntryStatus( self ):
      if not self.nhgEntryStatus:
         mountPath = 'routing/nexthopgroup/entrystatus'
         self.nhgEntryStatus = self.shmemEm.doMount( mountPath,
                               "NexthopGroup::EntryStatus",
                               self.readerInfo )
      return self.nhgEntryStatus

   def getTunnelFib( self ):

      if self.tunnelFib is None:
         self.tunnelFib = self.shmemEm.doMount(
            'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
            Smash.mountInfo( 'reader' ) )
      return self.tunnelFib

   def getNhResStatus( self, vrfName ):
      if self.nhResPolicyStatus is None:
         path = Cell.path( 'routing/' + self.rib + '/nhResPolicyStatus' )
         self.nhResPolicyStatus = \
            LazyMount.mount( self.entityManager, path,
                             'Routing::Rib::NextHopResPolicyStatusByVrf', 'r' )
         LazyMount.force( self.nhResPolicyStatus )
      return self.nhResPolicyStatus.vrfStatus.get( vrfName )

   def getRibConfig( self ):
      if self.ribConfig is None:
         self.ribConfig = LazyMount.mount( self.entityManager,
                                           'routing/rib/config',
                                           'Tac::Dir', 'ri' )
         LazyMount.force( self.ribConfig )
      return self.ribConfig

   def getRibConfigBgp( self, vrfId ):
      if self.ribConfig is None:
         self.ribConfig = self.getRibConfig()

      if self.ribConfigBgp is None:
         self.ribConfigBgp = self.ribConfig.get( 'bgp' )

      if self.ribConfigBgp:
         return self.ribConfigBgp.vrfConfig.get( vrfId )
      
      return None

   def getCrLeakStatus( self, af ):
      '''Mount connected route leak status if needed and then sort it'''

      status = self.crLeakStatuses.get( af )
      if status is None:
         mountPath = "routing/" if af == "ipv4" else "routing6/"
         mountPath += "connectedRouteLeakStatus"

         status = self.shmemEm.doMount( mountPath,
                                        "Routing::ConnectedRouteLeakStatus",
                                        self.readerInfo )
         self.crLeakStatuses[ af ] = status

      return status

class IpRibShmemCliMounter( object ):
   '''This class mounts IpRib config and status entities from shared-memory.
   It mounts entities the first time they're needed, but doesn't cache them.
   When this object goes out of scope (e.g. when exiting the show command),
   unused entities are auto-unmounted after a pre-determined delay has passed.'''

   def __init__( self, shmemEm, rib ):
      self.rib = rib
      self.shmemMg = SharedMem.CliMountGroup( shmemEm=shmemEm )

   def getViaMetricStatus( self ):
      return self.shmemMg.doMount(
         Cell.path( 'routing/' + self.rib + '/viaMetricStatus' ),
         'Routing::Rib::ViaMetricStatusByVrf',
         Shark.mountInfo( 'shadow' ) )

   def doClose( self ):
      self.shmemMg.doClose( blocking=True )

class AllVrfIgpReadyMounter( object ):
   def __init__( self ):
      self.entityManager = None
      self._allVrfIgpReady = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def allVrfIgpReady( self ):
      if self._allVrfIgpReady is None:
         self._allVrfIgpReady = \
            LazyMount.mount( self.entityManager,
                             Cell.path( "routing/rib/ribReady/status/igpReady" ),
                             "Routing::Rib::AllVrfIgpReady", "r" )
      return self._allVrfIgpReady

class AllVrfRibReadyMounter( object ):
   def __init__( self ):
      self.entityManager = None
      self._allVrfRibReady = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def allVrfRibReady( self ):
      if self._allVrfRibReady is None:
         self._allVrfRibReady = \
            LazyMount.mount( self.entityManager,
                             Cell.path( "routing/rib/ribReady/status/all" ),
                             "Routing::Rib::AllVrfRibReady", "r" )
      return self._allVrfRibReady

class FibReadyDirMounter( object ):
   def __init__( self ):
      self.entityManager = None
      self._fibReadyDir = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def fibReadyDir( self ):
      if self._fibReadyDir is None:
         self._fibReadyDir = LazyMount.mount( self.entityManager,
                                              Cell.path( "routing/fibReady" ),
                                              "Tac::Dir", "ri" )
      return self._fibReadyDir

# Utility functions required for implementing and testing 'show rib ready' command
def toUtc( timestamp ):
   """Convert a timestamp got from Tac::now() to UTC"""
   utcTime = timestamp + Tac.utcNow() - Tac.now() if timestamp else timestamp
   return utcTime
