#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Arnet
import Tac
import Ethernet
import datetime
from CliModel import Model, Dict, List, Bool, Str, Int, Enum
from CliPlugin.IntfModel import Interface
from CliPlugin.IntfCli import Intf
import Toggles.IpLockingToggleLib as ILTL
from ArnetModel import IpGenericAddress, MacAddress
from IpUtils import IpAddress
from TableOutput import TableFormatter, Headings, Format, terminalWidth
from Intf.IntfRange import intfListToCanonical
from collections import namedtuple

LeaseTuple = namedtuple( "Lease", "ip mac source intf installed expirationTime" )

def compareConfigSourceAndIpAddress( lease1, lease2 ):
   """Compares Config Source and two IP addresses numerically."""
   if lease1.source == lease2.source:
      return cmp( IpAddress( lease1.ip ), IpAddress( lease2.ip ) )
   # Lease came from config should be display first
   if lease1.source == 'config':
      return -1
   else:
      return 1

#-------------------------------------------------------------------------------
#
# cmd: show address locking
#
#-------------------------------------------------------------------------------
ipv4InactiveHelpStr = "Reason that Address Locking is not enabled for IPv4"
ipv6InactiveHelpStr = "Reason that Address Locking is not enabled for IPv6"

class IpLockingIntfEnabled( Model ):
   ipv4Enabled = Bool( help="Address Locking is enabled for IPv4" )
   ipv4InactiveReason = Str( help=ipv4InactiveHelpStr, optional=True )
   ipv6Enabled = Bool( help="Address Locking is enabled for IPv6" )
   ipv6InactiveReason = Str( help=ipv6InactiveHelpStr, optional=True )

class IpLockingProtocol( Model ):
   protocol = Enum( values=( "leaseQuery", "arIpQuery" ),
                    help="Protocol used to query server" )

def printServers( model ):
   print( "DHCP servers:" )
   headings = ( ( "Address", "lh" ),
                ( "Query Protocol", "l" ), )
   t = TableFormatter()
   th = Headings( headings )
   th.doApplyHeaders( t )
   f1 = Format( justify='left', wrap=False, minWidth=1 )
   f2 = Format( justify='left', wrap=True, minWidth=1 )
   for server in model.dhcpV4Servers:
      if model.dhcpV4Servers[ server ].protocol == 'arIpQuery':
         t.newRow( server, "Arista IP Query" )
      else:
         t.newRow( server, "Lease Query" )
   t.formatColumns( f1, f2 )
   print( t.output() )

class IpLocking( Model ):
   active = Bool( help="IP Locking active state" )
   inactiveReason = Str( help="Reason that IP Locking is inactive", optional=True )
   if ILTL.toggleIpLockingArIpEnabled():
      dhcpV4Servers = Dict( keyType=IpGenericAddress,
            valueType=IpLockingProtocol,
            help=( "Mapping of servers queried for DHCP v4 leases to the protocol "
                   "used to query them" ) )
   enabledIntfs = Dict( keyType=Interface,
         valueType=IpLockingIntfEnabled,
         help=( "A mapping of interfaces to their IPv4 and IPv6 address locking "
                "enabled status" ) )
   configuredIpv4Intfs = List(
         valueType=Interface,
         help="List of interfaces configured with address locking IPv4" )
   configuredIpv6Intfs = List(
         valueType=Interface,
         help="List of interfaces configured with address locking IPv6" )
   if ILTL.toggleIpLockingEnforcementEnabled():
      ipv6Enforced = Bool( help="Locked-address enforcement is enabled for IPv6" )

   def render( self ):
      active = self.active
      activeStr = 'IP Locking is %s' % ( 'active' if active else 'inactive' )
      if not active:
         activeStr += ' (%s)' % self.inactiveReason
      print( activeStr )

      if ILTL.toggleIpLockingArIpEnabled():
         print( '' )
         printServers( self )

      # TODO: Coallesce interface into ranges.
      t = TableFormatter()
      f1 = Format( justify='left', wrap=False, maxWidth=29 )
      f1.noPadLeftIs( True )
      f2 = Format( justify='left', wrap=True, minWidth=1 )
      f2.noPadLeftIs( True )

      if self.configuredIpv4Intfs:
         spaceLeft = terminalWidth() - len( "Configured IPv4 Interfaces: " )
         ipv4IntfStr = ", ".join( intfListToCanonical( self.configuredIpv4Intfs,
                                                       strLimit=spaceLeft ) )
         t.newRow( 'Configured IPv4 Interfaces:', ipv4IntfStr )
      else:
         t.newRow( 'Configured IPv4 Interfaces:' )

      if self.configuredIpv6Intfs:
         spaceLeft = terminalWidth() - len( "Configured IPv6 Interfaces: " )
         ipv6IntfStr = ", ".join( intfListToCanonical( self.configuredIpv6Intfs,
                                                       strLimit=spaceLeft ) )
         t.newRow( 'Configured IPv6 Interfaces:', ipv6IntfStr )
      else:
         t.newRow( 'Configured IPv6 Interfaces:' )
      t.formatColumns( f1, f2 )
      print ( t.output() )

      dispEnforcementLegend = False
      if active:
         print( "Interface Status" )
         t = TableFormatter()
         headings = ( ( "Interface", "lh" ),
                      ( "IPv4", "l" ),
                      ( "IPv6", "l" ), )
         th = Headings( headings )
         th.doApplyHeaders( t )
         for intf in Arnet.sortIntf( self.enabledIntfs ):
            ipLockingIntfEnabled = self.enabledIntfs[ intf ]
            t.startRow()
            t.newCell( intf )
            if ipLockingIntfEnabled.ipv4Enabled:
               ipv4Str = 'yes'
            else:
               ipv4Str = 'no (%s)' % ipLockingIntfEnabled.ipv4InactiveReason
            if ILTL.toggleIpLockingEnforcementEnabled():
               # We do not support ipv6 address locking if enforcement is
               # enabled for IPv6
               if self.ipv6Enforced:
                  ipv6Str = 'no (%s)' % ipLockingIntfEnabled.ipv6InactiveReason
               elif ipLockingIntfEnabled.ipv6Enabled:
                  ipv6Str = 'yes*'
                  dispEnforcementLegend = True
               else:
                  ipv6Str = 'no (%s)' % ipLockingIntfEnabled.ipv6InactiveReason
            else:
               if ipLockingIntfEnabled.ipv6Enabled:
                  ipv6Str = 'yes'
               else:
                  ipv6Str = 'no (%s)' % ipLockingIntfEnabled.ipv6InactiveReason
            t.newCell( ipv4Str )
            t.newCell( ipv6Str )
         print( t.output() )

      if ILTL.toggleIpLockingEnforcementEnabled() and dispEnforcementLegend:
         print( '* Locked address enforcement is disabled' )

#-------------------------------------------------------------------------------
#
# cmd: show address locking
#
#-------------------------------------------------------------------------------
class IpLockingServers( Model ):
   dhcpV4Servers = Dict( keyType=IpGenericAddress,
         valueType=IpLockingProtocol,
         help=( "Mapping of servers queried for DHCP v4 leases to the protocol "
                "used to query them" ) )

   def render( self ):
      printServers( self )

class Lease( Model ):
   clientMac = MacAddress( help="Client's MAC address" )
   source = Enum( values=( "config", "server" ),
               help="lease source type" )
   installed = Bool( help="This lease is installed in hardware" )
   expirationTime = Int( help="Lease expiration time in UTC" )

class IntfLeases( Model ):
   leases = Dict( keyType=IpGenericAddress,
                  valueType=Lease,
                  help="A mapping of a lease's IP address to its information" )

class LeasesTable( Model ):
   intfLeases = Dict( keyType=Interface,
      valueType=IntfLeases,
      help="A mapping of an interface to its leases" )

   def render( self ):
      t = TableFormatter()
      headings = ( ( "IP Address", "lh" ),
                   ( "MAC Address", "l" ),
                   ( 'Source', "l" ),
                   ( "Interface", "l" ),
                   ( "Installed", "l" ),
                   ( "Expiration Time", "l" ) )
      th = Headings( headings )
      th.doApplyHeaders( t )
      leases = []
      # Generate a flattened list of leases.
      savedUtcTime = Tac.utcNow()
      for intfId, intfLease in self.intfLeases.iteritems():
         for ip, lease in intfLease.leases.iteritems():
            # ExpirationTime in lease is the system time that lease will expire. We
            # want to display the number of seconds that this lease will expire.
            # Negative expirationTime due to lease expire and is in the process
            # of requery should be display as 0.
            expirationTime = max( 0, int( lease.expirationTime - savedUtcTime ) )
            leases.append(
               LeaseTuple(
                  ip=ip, mac=lease.clientMac, source=lease.source,
                  intf=intfId, installed=lease.installed,
                  expirationTime=expirationTime ) )
      leases = sorted( leases, cmp=compareConfigSourceAndIpAddress )

      for lease in leases:
         t.startRow()
         t.newCell( lease.ip )
         t.newCell(
            Ethernet.convertMacAddrCanonicalToDisplay( lease.mac.stringValue ) )
         t.newCell( lease.source )
         t.newCell( Intf.getShortname( lease.intf ) )
         installedString = "installed" if lease.installed else "not installed"
         t.newCell( installedString )
         # Do not display the lease expiration time for the static lease since static
         # lease will not expire.
         if lease.source == 'config':
            continue
         timeString = \
            'in ' + str( datetime.timedelta( seconds=lease.expirationTime ) )
         t.newCell( timeString )
      print( t.output() )
      print ( "Total IP Address for this criterion: %d" % len( leases ) )
