# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliParser
import ConfigMount
import LazyMount
import SharedMem
import Smash
import IntfCli
import Toggles.IpLockingToggleLib as ILTL
import Tac
from BasicCliModes import EnableMode
from CliMode.IpLocking import AddressLockingBaseMode
from CliPlugin.IntfCli import Intf
from CliPlugin.IpAddrMatcher import ipAddrMatcher
from CliPlugin.Ip6AddrMatcher import ip6AddrMatcher
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.IpLockingCliConstants import (
   inactiveReasonHwNotSupported,
   inactiveReasonDisabled,
   inactiveReasonLocalIntfNotConfigured,
   inactiveReasonV4ServerNotConfigured,
   inactiveReasonArpInspectionConfigured,
   inactiveReasonIpsgConfigured,
   inactiveReasonRelayConfigured,
   inactiveReasonDhcpSnoopingConfigured,
   inactiveReasonDhcpServerConfigured,
   inactiveReasonNoInterfaceConfigured,
   inactiveReasonV6EnforcementEnabled,
   )
from CliToken.IpLocking import (
      ipLockingAddressMatcherForConfig,
      ipLockingLockingMatcherForConfig,
      ipLockingAddressMatcherForConfigIf,
      ipLockingLockingMatcherForConfigIf,
      ipLockingIpv4MatcherForConfigIf,
      ipLockingIpv6MatcherForConfigIf,
      ipLockingAddressMatcherForClear,
      ipLockingLockingMatcherForClear,
      ipLockingCountersMatcherForClear,
      )
from CliToken.Clear import clearKwNode
from IpLockingModel import (
   IpLockingIntfEnabled,
   IpLocking,
   IpLockingServers,
   IpLockingProtocol,
   IntfLeases,
   LeasesTable,
   Lease,
   )
from IpLockingCounterModel import (
   IpLockingCountersModel,
   IntfCounters,
   ServerCounters,
   )
from EosDhcpServerLib import dhcpServerActive
from EthIntfCli import EthPhyIntf
from IpLockingLib import intfSupportsIpLocking

# globals
cliConfig = None
cliStatus = None
hwConfig = None
enabledInterfaceStatus = None
leaseStatus = None
staticLeaseTable = None
installedConfig = None
arpInspectionStatus = None
ipsgConfig = None
dhcpRelayStatus = None
dhcpSnoopingStatus = None
dhcpSnooping6Status = None
dhcpServerStatus = None
notConfigured = 'not configured'
notLayer2 = 'not a layer 2 interface'
notSupported = 'not supported'
noDhcpServer = 'no dhcp server'
noLocalIntf = 'no local interface'

def checkOrCreateClearLeaseInfo():
   cliClearLeaseInfo = cliConfig.cliClearLeaseInfo
   if not cliClearLeaseInfo:
      cliConfig.cliClearLeaseInfo = ( "LeaseClearEntity", )

def updateConfiguredAndActive( config ):
   config.configured = \
         bool( config.disabled != config.disabledDefault or
               config.localInterface != config.localInterfaceDefault or
               config.dhcpV4Server or
               config.dhcpV6Server or
               config.staticLease or
               config.reflector or
               config.reflectorRole != config.reflectorRoleDefault or
               ( ILTL.toggleIpLockingMacAgingCleanUpEnabled() and
                 config.macAgingCleanUp != config.macAgingCleanUpDefault ) or
               ( ILTL.toggleIpLockingEnforcementEnabled() and
                 config.ipv6Enforced != config.ipv6EnforcedDefault ) )
   ipv4ConfIntfExists = ipv6ConfIntfExists = False
   for intf in config.interface:
      if config.interface[ intf ].ipv4:
         ipv4ConfIntfExists = True
      if config.interface[ intf ].ipv6:
         ipv6ConfIntfExists = True
   config.active = \
                   bool( config.disabled == config.disabledDefault and
                       ( ( config.localInterface != config.localInterfaceDefault and
                           ( config.dhcpV4Server or config.dhcpV6Server ) and
                           ipv4ConfIntfExists ) or
                         ( config.ipv6Enforced != config.ipv6EnforcedDefault and
                           ipv6ConfIntfExists ) ) )
   config.reflectorActive = \
         bool( config.disabled == config.disabledDefault and
               config.localInterface != config.localInterfaceDefault and
               config.reflectorRole != config.reflectorRoleDefault )
   # HACK: We are changing the default behavior to remove leases upon the
   # corresponding MAC aging out. If the toggle is not enabled, set
   # cliConfig.macAgingCleanUp to False so that we maintain the same previous default
   # behavior when the toggle is turned off.
   if not ILTL.toggleIpLockingMacAgingCleanUpEnabled():
      cliConfig.macAgingCleanUp = False

class IpLockingIntfConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return intfSupportsIpLocking( mode.intf.name )

class AddressLockingMode( AddressLockingBaseMode, BasicCli.ConfigModeBase ):
   name = "Address Locking"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      AddressLockingBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# The "[no|default] address locking" command, in config mode.
#-------------------------------------------------------------------------------
class AddressLockingModeEnter( CliCommand.CliCommandClass ):
   syntax = 'address locking'
   noOrDefaultSyntax = syntax

   data = {
      'address': ipLockingAddressMatcherForConfig,
      'locking': ipLockingLockingMatcherForConfig,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( AddressLockingMode )
      mode.session_.gotoChildMode( childMode )

   # Reset all non-interface attributes to default
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.disabled = cliConfig.disabledDefault
      cliConfig.localInterface = cliConfig.localInterfaceDefault
      cliConfig.reflectorRole = cliConfig.reflectorRoleDefault
      cliConfig.ipv6Enforced = cliConfig.ipv6EnforcedDefault
      cliConfig.dhcpV4Server.clear()
      cliConfig.dhcpV6Server.clear()
      cliConfig.staticLease.clear()
      cliConfig.reflector.clear()
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] disabled" command, in "address locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingDisabled( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax

   data = {
      'disabled': 'Disable IP locking on configured ports',
   }

   @staticmethod
   def handler( mode, args ):
      cliConfig.disabled = True
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.disabled = cliConfig.disabledDefault
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] local-interface <intf>" command, in "address locking"
# config mode.
#-------------------------------------------------------------------------------
class IpLockingLocalInterface( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface [ INTF ]'

   data = {
      'local-interface': 'Configuring local interface',
      'INTF': Intf.matcherWithIpSupport,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      cliConfig.localInterface = intf.name
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.localInterface = cliConfig.localInterfaceDefault
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] dhcp server ipv4 ADDR" command, in "address locking"
# config mode.
#-------------------------------------------------------------------------------
class IpLockingDhcpServer( CliCommand.CliCommandClass ):
   syntax = 'dhcp server ipv4 V4ADDR'
   noOrDefaultSyntax = 'dhcp server ipv4 [ V4ADDR ] ...'

   data = {
      'dhcp': 'Configuration options related to DHCP',
      'server': 'Set DHCP server',
      'ipv4': 'Specify IPv4 DHCP server',
      'V4ADDR': ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      v4addr = args[ 'V4ADDR' ]
      cliConfig.dhcpV4Server[ v4addr ] = 'leaseQuery'
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'V4ADDR' in args:
         v4addr = args[ 'V4ADDR' ]
         del cliConfig.dhcpV4Server[ v4addr ]
      else:
         cliConfig.dhcpV4Server.clear()
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] dhcp server ipv4 ADDR [protocol arista-query]" command, in
# "address locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingServer( CliCommand.CliCommandClass ):
   syntax = 'dhcp server ipv4 V4ADDR [ protocol arista-query ]'
   noOrDefaultSyntax = 'dhcp server ipv4 [ V4ADDR ] [ protocol arista-query ]'

   data = {
      'dhcp': 'Configuration options related to DHCP',
      'server': 'Set DHCP server',
      'ipv4': 'Specify IPv4 DHCP server',
      'V4ADDR': ipAddrMatcher,
      'protocol': 'Query protocol',
      'arista-query': 'Configure source interface',
   }

   @staticmethod
   def handler( mode, args ):
      v4addr = args[ 'V4ADDR' ]
      protocol = 'arIpQuery' if 'arista-query' in args else 'leaseQuery'
      cliConfig.dhcpV4Server[ v4addr ] = protocol
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      v4addr = args.get( 'V4ADDR' )
      if v4addr:
         del cliConfig.dhcpV4Server[ v4addr ]
      else:
         if 'arista-query' in args:
            for server in cliConfig.dhcpV4Server:
               if cliConfig.dhcpV4Server[ server ] == 'arIpQuery':
                  del cliConfig.dhcpV4Server[ server ]
         else:
            cliConfig.dhcpV4Server.clear()
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] reflector address ADDR" command, in "address locking"
# config mode. ADDR can be IPv4 for now ( IPv6 is future work)
#-------------------------------------------------------------------------------
class IpLockingReflectorAddress( CliCommand.CliCommandClass ):
   syntax = 'reflector address V4ADDR'
   noOrDefaultSyntax = 'reflector address [ V4ADDR ] ...'

   data = {
      'reflector': 'Set reflector configuration',
      'address': 'Set reflector address',
      'V4ADDR': ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      addr = args[ 'V4ADDR' ]
      ipGenAddr = Tac.Value( "Arnet::IpGenAddr", str( addr ) )
      cliConfig.reflector[ ipGenAddr ] = True
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'V4ADDR' in args:
         addr = args[ 'V4ADDR' ]
         ipGenAddr = Tac.Value( "Arnet::IpGenAddr", str( addr ) )
         del cliConfig.reflector[ ipGenAddr ]
      else:
         cliConfig.reflector.clear()
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] lease <ipv4-or-ipv6-addr>  mac <m>" command,
# in "address locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingStaticLease( CliCommand.CliCommandClass ):
   syntax = 'lease ( V4ADDR | V6ADDR ) mac MAC_ADDR'
   noOrDefaultSyntax = 'lease ( ( V4ADDR ) | ( V6ADDR ) )...'

   data = {
      'lease': 'Configure static leases',
      'V4ADDR': ipAddrMatcher,
      'V6ADDR': ip6AddrMatcher,
      'mac': 'Configure mac for static lease',
      'MAC_ADDR': macAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'V4ADDR' in args:
         ipGenAddr = Tac.Value( 'Arnet::IpGenAddr', str( args[ 'V4ADDR' ] ) )
      else:
         # IPv6 static lease is not supported yet.
         return
      macAddr = Tac.Value( 'Arnet::EthAddr', stringValue=args[ 'MAC_ADDR' ] )
      if ipGenAddr in cliConfig.staticLease and \
         cliConfig.staticLease[ ipGenAddr ].mac != args[ 'MAC_ADDR' ]:
         del cliConfig.staticLease[ ipGenAddr ]
         cliConfig.staticLease.newMember( ipGenAddr, macAddr, '' )
      else:
         cliConfig.staticLease.newMember( ipGenAddr, macAddr, '' )
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'V4ADDR' not in args:
         return
      ipGenAddr = Tac.Value( 'Arnet::IpGenAddr', str( args[ 'V4ADDR' ] ) )
      if ipGenAddr in cliConfig.staticLease:
         del cliConfig.staticLease[ ipGenAddr ]
      updateConfiguredAndActive( cliConfig )

#----------------------------------------------------------------------------------
# The "[no|default] locked-address ( ipv4 | ipv6 ) enforcement disabled" command in
# "address locking" config mode.
class IpLockingLockedAddressEnforcementDisabled( CliCommand.CliCommandClass ):
   syntax = "locked-address ipv6 enforcement disabled"
   noOrDefaultSyntax = syntax

   data = {
      'locked-address': 'Configuration options for locked addresses',
      'ipv6': 'Configuration for IPv6',
      'enforcement': 'Configure enforcement for locked addresses',
      'disabled': 'Disable enforcement for locked addresses'
   }

   @staticmethod
   def handler( mode, args ):
      cliConfig.ipv6Enforced = False
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.ipv6Enforced = cliConfig.ipv6EnforcedDefault
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] locked-address expiration mac disabled" command in "address
# locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingLockedAddressExpirationMacDisabled( CliCommand.CliCommandClass ):
   syntax = 'locked-address expiration mac disabled'
   noOrDefaultSyntax = syntax

   data = {
      'locked-address': 'Configuration options for locked addresses',
      'expiration': 'Configure expiration mode for locked addresses',
      'mac': 'Configure deauthorizing locked addresses upon MAC aging out',
      'disabled': 'Disable deauthorizing locked addresses upon MAC aging out',
   }

   @staticmethod
   def handler( mode, args ):
      cliConfig.macAgingCleanUp = False
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.macAgingCleanUp = cliConfig.macAgingCleanUpDefault
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] reflector" command, in "address locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingReflectorRole( CliCommand.CliCommandClass ):
   syntax = 'reflector'
   noOrDefaultSyntax = syntax

   data = {
      'reflector': 'Set reflector configuration',
   }

   @staticmethod
   def handler( mode, args ):
      cliConfig.reflectorRole = True
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.reflectorRole = cliConfig.reflectorRoleDefault
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "[no|default] address locking ipv4" command, in "config-if" mode.
# This command is only available on intfSupportsIpLocking interface
#-------------------------------------------------------------------------------
class IpLockingInterfaceEnable( CliCommand.CliCommandClass ):
   syntax = 'address locking { ipv4 | ipv6 }'
   noOrDefaultSyntax = 'address locking [ { ipv4 | ipv6 } ]'

   data = {
      'address': ipLockingAddressMatcherForConfigIf,
      'locking': ipLockingLockingMatcherForConfigIf,
      'ipv4': ipLockingIpv4MatcherForConfigIf,
      'ipv6': ipLockingIpv6MatcherForConfigIf,
   }

   @staticmethod
   def handler( mode, args ):
      intf = cliConfig.newInterface( mode.intf.name )
      if 'ipv4' in args and 'ipv6' in args:
         intf.ipv4 = True
         intf.ipv6 = True
      elif 'ipv4' in args:
         intf.ipv4 = True
         intf.ipv6 = False
      elif 'ipv6' in args:
         intf.ipv4 = False
         intf.ipv6 = True
      updateConfiguredAndActive( cliConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.intf.name not in cliConfig.interface:
         return
      intf = cliConfig.interface[ mode.intf.name ]
      if ( ( 'ipv4' in args and 'ipv6' in args ) or
           ( 'ipv4' not in args and 'ipv6' not in args ) ):
         intf.ipv4 = False
         intf.ipv6 = False
      elif 'ipv4' in args:
         intf.ipv4 = False
      elif 'ipv6' in args:
         intf.ipv6 = False
      if not intf.ipv4 and not intf.ipv6:
         del cliConfig.interface[ mode.intf.name ]
      updateConfiguredAndActive( cliConfig )

#-------------------------------------------------------------------------------
# The "clear address locking lease ( ipv4 V4ADDR | ipv6 V6ADDR | interface INTF | \
#      all )" command, in enable mode.
#-------------------------------------------------------------------------------
class IpLockingClearLease( CliCommand.CliCommandClass ):
   syntax = 'clear address locking lease ( (ipv4 V4ADDR) | (ipv6 V6ADDR) | \
         (interface INTF) | all )'
   data = {
      'clear': clearKwNode,
      'address': ipLockingAddressMatcherForClear,
      'locking': ipLockingLockingMatcherForClear,
      'lease': 'Lease information',
      'ipv4': 'IPv4 address of the lease to be cleared',
      'V4ADDR': ipAddrMatcher,
      'ipv6': 'IPv6 address of the lease to be cleared',
      'V6ADDR': ip6AddrMatcher,
      'interface': 'The interface of the leases to be cleared',
      'INTF': EthPhyIntf.ethMatcher,
      'all': 'The entire lease table',
   }

   @staticmethod
   def handler( mode, args ):
      checkOrCreateClearLeaseInfo()
      # Reset the members of cliClearLeaseInfo everytime it is configured.
      cliConfig.cliClearLeaseInfo.ip = cliConfig.cliClearLeaseInfo.ipDefault
      cliConfig.cliClearLeaseInfo.intf = cliConfig.cliClearLeaseInfo.intfDefault
      cliConfig.cliClearLeaseInfo.clearAll = \
            cliConfig.cliClearLeaseInfo.clearAllDefault
      if 'V4ADDR' in args or 'V6ADDR' in args:
         if 'V4ADDR' in args:
            addr = args[ 'V4ADDR' ]
         else:
            addr = args[ 'V6ADDR' ]
         ipGenAddr = Tac.Value( "Arnet::IpGenAddr", str( addr ) )
         cliConfig.cliClearLeaseInfo.ip = ipGenAddr
      elif 'INTF' in args:
         intf = args[ 'INTF' ]
         cliConfig.cliClearLeaseInfo.intf = intf.name
      elif 'all' in args:
         cliConfig.cliClearLeaseInfo.clearAll = True
      cliConfig.cliClearLeaseInfo.versionNumber += 1

# Remove config when running "default interface <interface>"
class IpLockingIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del cliConfig.interface[ self.intf_.name ]

# --------------------------------------------------------------------------------
# The "clear address locking counters" command, in enable mode.
#---------------------------------------------------------------------------------
class IpLockingClearCounters( CliCommand.CliCommandClass ):
   syntax = 'clear address locking counters'
   data = {
      'clear': clearKwNode,
      'address': ipLockingAddressMatcherForClear,
      'locking': ipLockingLockingMatcherForClear,
      'counters': ipLockingCountersMatcherForClear,
   }

   @staticmethod
   def handler( mode, args ):
      cliConfig.resetCountersTrigger = \
            ( cliConfig.resetCountersTrigger + 1 ) % 256

BasicCli.GlobalConfigMode.addCommandClass( AddressLockingModeEnter )
AddressLockingMode.addCommandClass( IpLockingDisabled )
AddressLockingMode.addCommandClass( IpLockingLocalInterface )
if ILTL.toggleIpLockingArIpEnabled():
   AddressLockingMode.addCommandClass( IpLockingServer )
else:
   AddressLockingMode.addCommandClass( IpLockingDhcpServer )
AddressLockingMode.addCommandClass( IpLockingStaticLease )
if ILTL.toggleIpLockingMacAgingCleanUpEnabled():
   AddressLockingMode.addCommandClass(
         IpLockingLockedAddressExpirationMacDisabled )
if ILTL.toggleIpLockingEnforcementEnabled():
   AddressLockingMode.addCommandClass(
         IpLockingLockedAddressEnforcementDisabled )
if ILTL.toggleIpLockingSyncEnabled():
   AddressLockingMode.addCommandClass( IpLockingReflectorAddress )
   AddressLockingMode.addCommandClass( IpLockingReflectorRole )
EnableMode.addCommandClass( IpLockingClearLease )
if ILTL.toggleIpLockingCountersEnabled():
   EnableMode.addCommandClass( IpLockingClearCounters )
IpLockingIntfConfigModelet.addCommandClass( IpLockingInterfaceEnable )

#-------------------------------------------------------------------------------
# IpLocking show commands helper functions
#-------------------------------------------------------------------------------
def getV4InactiveReason( intf, ipv4Enabled ):
   ipv4InactiveReason = None
   if not ipv4Enabled:
      if not cliConfig.interface[ intf ].ipv4:
         ipv4InactiveReason = notConfigured
      elif cliStatus.getV4ReasonString( intf ) == notLayer2:
         ipv4InactiveReason = notLayer2
      elif cliConfig.localInterface == cliConfig.localInterfaceDefault:
         ipv4InactiveReason = noLocalIntf
      elif not cliConfig.dhcpV4Server and not cliConfig.dhcpV6Server:
         ipv4InactiveReason = noDhcpServer
   return ipv4InactiveReason

def getV6InactiveReason( intf, ipv6Enabled ):
   ipv6InactiveReason = None
   if not ipv6Enabled:
      if cliConfig.ipv6Enforced:
         ipv6InactiveReason = notSupported
      elif not cliConfig.interface[ intf ].ipv6:
         ipv6InactiveReason = notConfigured
      else:
         ipv6InactiveReason = notLayer2
   return ipv6InactiveReason

def doShowIpLocking( mode, args ):
   config = cliConfig
   active = cliConfig.active
   inactiveReason = ''

   configuredIpv4Intfs = [ x for x in cliConfig.interface
                           if cliConfig.interface[ x ].ipv4 ]

   configuredIpv6Intfs = [ x for x in cliConfig.interface
                           if cliConfig.interface[ x ].ipv6 ]

   if not hwConfig.featureSupported:
      active = False
      inactiveReason = inactiveReasonHwNotSupported
   elif arpInspectionStatus.enabled:
      active = False
      inactiveReason = inactiveReasonArpInspectionConfigured
   elif ipsgConfig.ipsgEnabledIntf:
      active = False
      inactiveReason = inactiveReasonIpsgConfigured
   elif dhcpRelayStatus.runControl:
      active = False
      inactiveReason = inactiveReasonRelayConfigured
   elif dhcpSnoopingStatus.enabled or dhcpSnooping6Status.enabled:
      active = False
      inactiveReason = inactiveReasonDhcpSnoopingConfigured
   elif dhcpServerActive( dhcpServerStatus ):
      active = False
      inactiveReason = inactiveReasonDhcpServerConfigured
   elif not config.interface:
      active = False
      inactiveReason = inactiveReasonNoInterfaceConfigured
   elif not active:
      if config.disabled != config.disabledDefault:
         inactiveReason = inactiveReasonDisabled
      elif config.localInterface == config.localInterfaceDefault:
         inactiveReason = inactiveReasonLocalIntfNotConfigured
      elif not config.dhcpV4Server:
         inactiveReason = inactiveReasonV4ServerNotConfigured
      elif config.ipv6Enforced and configuredIpv6Intfs:
         inactiveReason = inactiveReasonV6EnforcementEnabled
   if ILTL.toggleIpLockingArIpEnabled():
      dhcpV4Servers = {}
      for server, protocol in cliConfig.dhcpV4Server.iteritems():
         dhcpV4Servers[ server ] = IpLockingProtocol( protocol=protocol )

   if active:
      ipLockingIntfEnabledDict = {}
      for intf in cliConfig.interface:
         if ( not cliConfig.interface[ intf ].ipv4 or
            cliStatus.getV4ReasonString( intf ) == notLayer2 or
            cliConfig.localInterface == cliConfig.localInterfaceDefault or
            ( not cliConfig.dhcpV4Server and not cliConfig.dhcpV6Server ) ):
            ipv4Enabled = False
         else:
            ipv4Enabled = True
         ipv4InactiveReason = getV4InactiveReason( intf, ipv4Enabled )
         if ILTL.toggleIpLockingEnforcementEnabled:
            ipv6Enabled = True if ( intf in enabledInterfaceStatus.ipv6Interface and
                                  not cliConfig.ipv6Enforced ) else False
            ipv6InactiveReason = getV6InactiveReason( intf, ipv6Enabled )
         else:
            ipv6Enabled = False
            ipv6InactiveReason = notSupported
         ipLockingIntfEnabledDict[ intf ] = IpLockingIntfEnabled(
               ipv4Enabled=ipv4Enabled,
               ipv4InactiveReason=ipv4InactiveReason,
               ipv6Enabled=ipv6Enabled,
               ipv6InactiveReason=ipv6InactiveReason )
      if ( ILTL.toggleIpLockingArIpEnabled() and
           ILTL.toggleIpLockingEnforcementEnabled() ):
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
                           configuredIpv6Intfs=configuredIpv6Intfs,
                           enabledIntfs=ipLockingIntfEnabledDict,
                           dhcpV4Servers=dhcpV4Servers,
                           ipv6Enforced=cliConfig.ipv6Enforced )
      elif ILTL.toggleIpLockingArIpEnabled():
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
               configuredIpv6Intfs=configuredIpv6Intfs,
               enabledIntfs=ipLockingIntfEnabledDict,
               dhcpV4Servers=dhcpV4Servers )
      elif ILTL.toggleIpLockingEnforcementEnabled():
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
                        configuredIpv6Intfs=configuredIpv6Intfs,
                        enabledIntfs=ipLockingIntfEnabledDict,
                        ipv6Enforced=cliConfig.ipv6Enforced )
      else:
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
                        configuredIpv6Intfs=configuredIpv6Intfs,
                        enabledIntfs=ipLockingIntfEnabledDict )
   else:
      if ( ILTL.toggleIpLockingArIpEnabled() and
           ILTL.toggleIpLockingEnforcementEnabled() ):
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
                           configuredIpv6Intfs=configuredIpv6Intfs,
                           dhcpV4Servers=dhcpV4Servers,
                           inactiveReason=inactiveReason,
                           ipv6Enforced=cliConfig.ipv6Enforced )
      elif ILTL.toggleIpLockingArIpEnabled():
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
               configuredIpv6Intfs=configuredIpv6Intfs,
               dhcpV4Servers=dhcpV4Servers, inactiveReason=inactiveReason )
      elif ILTL.toggleIpLockingEnforcementEnabled():
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
                           configuredIpv6Intfs=configuredIpv6Intfs,
                           inactiveReason=inactiveReason,
                           ipv6Enforced=cliConfig.ipv6Enforced )
      else:
         return IpLocking( active=active, configuredIpv4Intfs=configuredIpv4Intfs,
                        configuredIpv6Intfs=configuredIpv6Intfs,
                        inactiveReason=inactiveReason )

def doShowIpLockingServers( mode, args ):
   dhcpV4Servers = {}
   for server, protocol in cliConfig.dhcpV4Server.iteritems():
      dhcpV4Servers[ server ] = IpLockingProtocol( protocol=protocol )
   return IpLockingServers( dhcpV4Servers=dhcpV4Servers )

def leaseInstalled( mac, ip, intfId ):
   # look for installed status in installedConfig         :
   ipMacLeaseKey = Tac.newInstance(
      "IpLocking::IpMacLeaseKey", mac, ip, intfId )
   if ipMacLeaseKey in installedConfig.installedIpMacLease:
      installedIpMacLease = installedConfig.installedIpMacLease[ ipMacLeaseKey ]
      return installedIpMacLease.installedIp and installedIpMacLease.installedArp
   else:
      # Lease is not installed if it hasn't made it into smash yet as the agent
      # should be in the process of installing it.
      return False

def addStaticLeases( intfLeaseDict, filterIntf=None, onlyDisplayInstalled=False ):
   # Build the static lease -> intfId map from installedConfig.
   ipToIntf = {}
   for installedLease in installedConfig.installedIpMacLease:
      if installedLease.ip in cliConfig.staticLease:
         ipToIntf[ installedLease.ip ] = installedLease.intfId

   for staticLease in cliConfig.staticLease.values():
      ip = staticLease.ip
      mac = staticLease.mac
      intfId = ''
      if ip in ipToIntf:
         intfId = ipToIntf[ ip ]

      if filterIntf and intfId != filterIntf.name:
         continue
      if intfId not in intfLeaseDict:
         intfLeaseDict[ intfId ] = {}
      leaseIsInstalled = leaseInstalled( mac, ip, intfId )
      if onlyDisplayInstalled and not leaseIsInstalled:
         continue
      # Set static lease's expiration time to 0 since they won't expire.
      intfLeaseDict[ intfId ][ ip ] = Lease(
         clientMac=mac, source='config', installed=leaseIsInstalled,
         expirationTime=0 )

def addDynamicLease( intfLeaseDict, filterIntf=None, onlyDisplayInstalled=False ):
   for interfaceLeaseSet in leaseStatus.interfaceLeaseSet.itervalues():
      intfId = interfaceLeaseSet.intfId
      if filterIntf and intfId != filterIntf.name:
         continue
      if intfId not in intfLeaseDict:
         intfLeaseDict[ intfId ] = {}
      for ipMacLease in interfaceLeaseSet.ipMacLease.itervalues():
         ip = ipMacLease.ip
         mac = ipMacLease.mac
         expirationTime = int( ipMacLease.expirationTime )
         leaseIsInstalled = leaseInstalled( mac, ip, intfId )
         if onlyDisplayInstalled and not leaseIsInstalled:
            continue
         intfLeaseDict[ intfId ][ ip ] = Lease(
            clientMac=mac, source='server', installed=leaseIsInstalled,
            expirationTime=expirationTime )

def doShowAddressLockingTable( mode, args ):
   filterIntf = args.get( 'INTF' )
   onlyDisplayInstalled = 'installed' in args
   displayStaticLease = 'dynamic' not in args
   displayDynamicLease = 'static' not in args
   intfLeaseDict = {}

   if displayStaticLease:
      addStaticLeases( intfLeaseDict, filterIntf=filterIntf,
                       onlyDisplayInstalled=onlyDisplayInstalled )

   if displayDynamicLease:
      addDynamicLease( intfLeaseDict, filterIntf=filterIntf,
                       onlyDisplayInstalled=onlyDisplayInstalled )

   leaseTableDict = {}
   for ( intfId, leases ) in intfLeaseDict.iteritems():
      leaseTableDict[ intfId ] = IntfLeases( leases=leases )
   return LeasesTable( intfLeases=leaseTableDict )

def getServerCountersModel( serverCounters ):
   return ServerCounters(
      leaseQuery=serverCounters.leaseQuery,
      leaseActiveReceived=serverCounters.leaseActiveReceived,
      leaseActiveDropped=serverCounters.leaseActiveDropped,
      leaseUnknownReceived=serverCounters.leaseUnknownReceived,
      leaseUnknownDropped=serverCounters.leaseUnknownDropped,
      leaseUnassignedReceived=serverCounters.leaseUnassignedReceived,
      leaseUnassignedDropped=serverCounters.leaseUnassignedDropped,
      unknown=serverCounters.unknown,
   )

def getIntfCountersModel( intfCounters ):
   return IntfCounters(
      leaseQuery=intfCounters.leaseQuery,
      leaseActive=intfCounters.leaseActive,
      leaseUnknown=intfCounters.leaseUnknown,
      leaseUnassigned=intfCounters.leaseUnassigned,
   )

def doShowIpLockingCounters( mode, args ):
   intfCounters = {}
   sysdbIntfCounters = cliStatus.intfCounters
   for intf in sysdbIntfCounters:
      intfCounters[ intf ] = getIntfCountersModel( sysdbIntfCounters[ intf ] )
   serverCounters = {}
   sysdbServerCounters = cliStatus.serverCounters
   for server in sysdbServerCounters:
      serverCounters[ server ] = getServerCountersModel(
            sysdbServerCounters[ server ] )
   return IpLockingCountersModel(
      intfCounters=intfCounters, serverCounters=serverCounters )

#-------------------------------------------------------------------------------
# Associate IpLockingIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( IpLockingIntfConfigModelet )

#-------------------------------------------------------------------------------
# Have the Cli agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig
   global cliStatus
   global hwConfig
   global enabledInterfaceStatus
   global leaseStatus
   global installedConfig
   global arpInspectionStatus
   global ipsgConfig
   global dhcpRelayStatus
   global dhcpSnoopingStatus
   global dhcpSnooping6Status
   global dhcpServerStatus

   cliConfig = ConfigMount.mount( entityManager, "iplocking/cliConfig",
                                  "IpLocking::CliConfig", "w" )
   cliStatus = LazyMount.mount( entityManager, "iplocking/cliStatus",
                                "IpLocking::CliStatus", "r" )
   hwConfig = LazyMount.mount( entityManager, "iplocking/hardware/config",
                               "IpLocking::Hardware::Config", "r" )
   enabledInterfaceStatus = LazyMount.mount( entityManager,
         "iplocking/interfaceStatus", "IpLocking::EnabledInterfaceStatus", "r" )
   leaseStatus = LazyMount.mount( entityManager, "iplocking/leaseStatus",
                                  "IpLocking::LeaseStatus", "r" )
   arpInspectionStatus = LazyMount.mount( entityManager,
                                          "security/arpInspection/status",
                                          "ArpInsp::Status", "r" )
   ipsgConfig = LazyMount.mount( entityManager, "security/ipsg/config",
                                 "Ipsg::Config", "r" )
   dhcpRelayStatus = LazyMount.mount( entityManager, "ip/helper/dhcprelay/status",
                                      "Ip::Helper::DhcpRelay::Status", "r" )
   dhcpSnoopingStatus = LazyMount.mount( entityManager,
                                         "bridging/dhcpsnooping/status",
                                         "Bridging::DhcpSnooping::Status", "r" )
   dhcpSnooping6Status = LazyMount.mount( entityManager,
                                          "bridging/dhcpsnooping/dhcp6Status",
                                          "Bridging::DhcpSnooping::Status", "r" )
   dhcpServerStatus = LazyMount.mount( entityManager, "dhcpServer/status",
                                       "DhcpServer::Status", "r" )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   installedConfig = shmemEm.doMount( "iplocking/hardware/installedConfig",
                                      "IpLocking::Hardware::InstalledConfig",
                                      Smash.mountInfo( 'keyshadow' ) )
   IntfCli.Intf.registerDependentClass( IpLockingIntf )
