# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module contains the definition of CliParser rules for matching IP
# addresses and IP address prefixes in their IP version generic forms.
#-------------------------------------------------------------------------------
import re

import Arnet
import CliCommand
import CliMatcher
import CliParser
from CliParserCommon import MatchResult, noMatch
import Ip6AddrMatcher
import IpAddrMatcher

ipGenAddrRe6_ = re.compile( Arnet.Ip6AddrRe )
ipGenAddrRe4_ = re.compile( r'(\d+)\.(\d+)\.(\d+)\.(\d+)$' )

ipGenPrefixRe6_ = re.compile( r'(%s)/(\d{1,3})$' % Arnet.Ip6AddrRe )
ipGenPrefixRe4_ = re.compile( r'([\d.]+)/(\d+)$' )

# Predicates for likely af and whether it not prefix
def isIpv4( str_ ):
   return '.' in str_ and ':' not in str_

def isIpv6( str_ ):
   return ':' in str_

def isPrefix( str_ ):
   return '/' in str_

def hostPrefix( str_ ):
   return "/128" if isIpv6( str_ ) else "/32"

class IpGenAddrMatcher( CliMatcher.Matcher ):
   '''Type of matcher that matches an IP (v4 or v6) address, and evaluates to
   the corresponding Arnet::IpGenAddr value.
   '''
   def __init__( self, helpdesc, helpdesc4=None, helpdesc6=None, **kargs ):
      # A basic regex that is enough to pull out the chracters of an ipv6 address
      # The rest is done in c++
      super( IpGenAddrMatcher, self ).__init__( helpdesc=helpdesc, **kargs )
      self.ipGenAddrRe6_ = ipGenAddrRe6_
      self.ipGenAddrRe4_ = ipGenAddrRe4_

      if helpdesc4 is None:
         helpdesc4 = helpdesc
      if helpdesc6 is None:
         helpdesc6 = helpdesc
      self.ipAddrMatcher_ = IpAddrMatcher.IpAddrMatcher( helpdesc4, partial=True )
      self.ip6AddrMatcher_ = Ip6AddrMatcher.Ip6AddrMatcher( helpdesc6 )

   def match( self, mode, context, token ):
      # Choose based on guessed af. Neither '.' nor ':' results in assuming IPv4
      addrRe = self.ipGenAddrRe6_ if isIpv6( token ) else self.ipGenAddrRe4_
      m = addrRe.match( token )
      if m is None or m.group( 0 ) != token:
         return noMatch
      try:
         addr = Arnet.IpGenAddr( m.group( 0 ) )
      except IndexError:
         return noMatch
      except ValueError:
         raise CliParser.InvalidInputError()
      return MatchResult( addr, token )

   def completions( self, mode, context, token ):
      comp4 = self.ipAddrMatcher_.completions( mode, context, token )
      comp6 = self.ip6AddrMatcher_.completions( mode, context, token )
      return comp4 + comp6

   def __str__( self ):
      return '<IP (v4 or v6) address>'

ipGenAddrMatcher = IpGenAddrMatcher( 'IP address' )

class IpGenPrefixMatcher( CliMatcher.Matcher ):
   '''Type of matcher that matches an IP (v4 or v6) address prefix, and
   evaluates to the corresponding Arnet::IpGenPrefix (or Arnet::IpGenAddrWithMask
   if the addrWithMask option is True. If allowAddr is set it will match IP addresses
   and set the prefixlen to 32/128.
   '''
   def __init__( self, helpdesc, helpdesc4=None, helpdesc6=None,
                 allowAddr=False, addrWithMask=False, **kargs ):
      super( IpGenPrefixMatcher, self ).__init__( helpdesc=helpdesc, **kargs )
      self.allowAddr_ = allowAddr
      self.addrWithMask_ = addrWithMask
      if helpdesc4 is None:
         helpdesc4 = helpdesc
      if helpdesc6 is None:
         helpdesc6 = helpdesc
      self.ipPrefixMatcher_ = IpAddrMatcher.IpPrefixMatcher( helpdesc4,
            partial=True )
      self.ip6PrefixMatcher_ = Ip6AddrMatcher.Ip6PrefixMatcher( helpdesc6 )

      self.ipGenPrefixRe6_ = ipGenPrefixRe6_
      self.ipGenPrefixRe4_ = ipGenPrefixRe4_

   def match( self, mode, context, token ):
      # Is this a prefix? If not append the host prefix
      if token and not isPrefix( token ) and self.allowAddr_:
         token = token + hostPrefix( token )

      # Choose based on guessed af. Neither '.' nor ':' results in assuming IPv4
      addrRe = self.ipGenPrefixRe6_ if isIpv6( token ) else self.ipGenPrefixRe4_
      m = re.match( addrRe, token )
      if m is None or m.group( 0 ) != token:
         return noMatch
      try:
         if self.addrWithMask_:
            addr = Arnet.IpGenAddrWithMask( m.group( 0 ) )
         else:
            addr = Arnet.IpGenPrefix( m.group( 0 ) )
      except IndexError:
         return noMatch
      return MatchResult( addr, token )

   def completions( self, mode, context, token ):
      # Is this a prefix? If not append the host prefix
      if token and not isPrefix( token ) and self.allowAddr_:
         token = token + hostPrefix( token )

      comp4 = self.ipPrefixMatcher_.completions( mode, context, token )
      comp6 = self.ip6PrefixMatcher_.completions( mode, context, token )
      return comp4 + comp6

   def __str__( self ):
      return '<IP (v4 or v6) address prefix>'

class IpGenAddrOrPrefixExprFactory( CliCommand.OrExpressionFactory ):

   def __init__( self, ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_ALLOW,
                 ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_ALLOW, allowAddr=True ):
      CliCommand.OrExpressionFactory.__init__( self )
      ipPrefixMatcher_ = IpAddrMatcher.ipPrefixExpr( 'Match this ip address',
                                                     'Match this subnet mask',
                                                     'Match this IP prefix',
                                                     overlap=ipOverlap )
      ipAddrMatcher_ = IpAddrMatcher.ipAddrMatcher
      ip6PrefixMatcher_ = Ip6AddrMatcher.Ip6PrefixMatcher(
                              'IPv6 prefix', overlap=ip6Overlap )
      ip6AddrMatcher_ = Ip6AddrMatcher.ip6AddrMatcher
      self |= ( "IP_PREFIX", ipPrefixMatcher_ )
      if allowAddr:
         self |= ( "IP_ADDR", ipAddrMatcher_ )
      self |= ( "IP6_PREFIX", ip6PrefixMatcher_ )
      if allowAddr:
         self |= ( "IP6_ADDR", ip6AddrMatcher_ )
