# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#----------------------------------------------------------------
# This module implements Ipv6 Router Advertisement related commands
#----------------------------------------------------------------
'''Configuration commands supported for IPv6'''

import AgentCommandRequest
import BasicCli
import BasicCliModes
import CliMatcher
import CliParser
import CliToken.Ipv6
from CliToken.Clear import clearKwNode
import CliCommand
import CommonGuards
import ConfigMount
import Ipv6RouterAdvtLib
import Ipv6RouterAdvtAgent
import Ip6RaCliModels
import LazyMount
import operator
import SmashLazyMount
import ShowCommand
import Tac

from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IraIp6IntfCli
from CliPlugin import IraCommonCli
from CliPlugin.IraIp6Model import Ip6Status
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.VrfCli import DEFAULT_VRF
from IpUtils import compareIp6Address

ip6Config = None
routerAdvtConfig = None
routerAdvtStatus = None
entityManager = None
allIntfStatusDir = None
routerNeighborRaStatus = None
ip6Status = None

LIFETIME_MAX_VALUE = 2 ** 32 - 1

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
matcherNd = CliMatcher.KeywordMatcher( 'nd',
      helpdesc='Neighbor Discovery / Router Advertisement' )
matcherRa = CliMatcher.KeywordMatcher( 'ra',
      helpdesc='Router Advertisement commands' )
matcherNeighbors = CliMatcher.KeywordMatcher( 'neighbors',
      helpdesc='Neighbors discovered using IPv6 router advertisement' )
matcherInterval = CliMatcher.KeywordMatcher( 'interval',
      helpdesc='Time between RA messages' )
matcherPrefix = CliMatcher.KeywordMatcher( 'prefix',
      helpdesc='Configure global prefix configurations' )
matcherPrefixAddr = Ip6AddrMatcher.Ip6PrefixMatcher( 'IP address prefix' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile',
      helpdesc='Configure a global prefix profile' )
matcherProfileName = CliMatcher.DynamicNameMatcher(
      lambda mode: routerAdvtConfig.prefixProfile,
      helpdesc='Profile name' )
matcherValidLifetime = CliMatcher.IntegerMatcher( 0, LIFETIME_MAX_VALUE,
      helpdesc='Valid Lifetime (secs)' )
matcherValidInfiniteLifetime = CliMatcher.KeywordMatcher( 'infinite',
      helpdesc='Infinite Valid Lifetime' )
matcherPreferredLifetime = CliMatcher.IntegerMatcher( 0, LIFETIME_MAX_VALUE,
      helpdesc='Preferred Lifetime (secs)' )
matcherPreferredInfiniteLifetime = CliMatcher.KeywordMatcher( 'infinite',
      helpdesc='Infinite Preferred Lifetime' )
matcherNoAutoconfig = CliMatcher.KeywordMatcher( 'no-autoconfig',
      helpdesc='No autoconfig flag' )
matcherNoOnlink = CliMatcher.KeywordMatcher( 'no-onlink',
      helpdesc='No onlink flag' )
matcherNoAdvertise = CliMatcher.KeywordMatcher( 'no-advertise',
      helpdesc='Do not advertise the prefix in RA messages.' )
matcherDnsServerLifetime = CliMatcher.IntegerMatcher( 0, LIFETIME_MAX_VALUE,
      helpdesc='Router Advertisement DNS server lifetime value in seconds' )
matcherDnsSuffixLifetime = CliMatcher.IntegerMatcher( 0, LIFETIME_MAX_VALUE,
      helpdesc='Router Advertisement DNS suffix lifetime value in seconds' )

#-------------------------------------------------------------------------------
#Note: Output of "show ipv6 interface" is using spaces instead of tabs hence
#same is being done here.
#-------------------------------------------------------------------------------
def _extraIp6NdRaInfo( intf, result ):
   """Fills in extra fields for the result of 'show ipv6 interface'.

   Args:
     intf: An IPv6 EthIntf object.
     result: An instance of IraIp6Model.Ip6Status
   """

   intfName = intf.name
   dadTransmits = -1
   reachableTime = 30000
   retransmitInterval = 0
   nsFromKernel = 1000
   raInterval = 200000
   raLifetime = 1800
   routerPreference = "Medium"
   hopLimit = 64
   enhancedDad = False
   acceptDad = -1

   # Get default values from kernel
   ip6NdIntfStatus = routerAdvtStatus.intf.get( intfName )
   if ip6NdIntfStatus:
      def getFileData( filename, dataset ):
         if dataset and ( filename in dataset ):
            return dataset[ filename ]
         return None

      if ip6NdIntfStatus.cfgProcFileData:
         cfgFiles = ip6NdIntfStatus.cfgProcFileData.fileData
         data = getFileData( "dad_transmits", cfgFiles )
         if data:
            dadTransmits = int( data )
         data = getFileData( "enhanced_dad", cfgFiles )
         if data:
            enhancedDad = ( int( data ) != 0 )
         data = getFileData( "accept_dad", cfgFiles )
         if data:
            acceptDad = int( data )
         else:
            # If accept_dad is not present, we assume the
            # default value of 1
            acceptDad = 1

      if ip6NdIntfStatus.nbrProcFileData:
         nbrFiles = ip6NdIntfStatus.nbrProcFileData.fileData
         data = getFileData( "base_reachable_time_ms", nbrFiles )
         if data:
            reachableTime = int( data )
         data = getFileData( "retrans_time_ms", nbrFiles )
         if data:
            nsFromKernel = int( data )

   result.ndReachableTime = reachableTime
   result.enhancedDad = enhancedDad
   result.dadAttempts = dadTransmits

   if dadTransmits > 0:
      if acceptDad == 2:
         result.dadStatus = "strict"
      elif acceptDad == 0:
         result.dadStatus = "disabled"
      elif acceptDad == 1:
         result.dadStatus = "enabled"
      else:
         # There are currently only 3 valid values for
         # acceptDad.  Anything else and we don't know
         # what the status should be
         result.dadStatus = "unavailable"
   elif dadTransmits < 0:
      result.dadStatus = "unavailable"
   else:
      result.dadStatus = "unsupported"

   # The reason for the ternary below is that there is no way
   # to access ip6NdIntfConfig.raSuppress right after an interface is IPv6 enabled
   # as config object does not exist until an explicit write to config has been
   # made by the user. This case comes into picture for Loopback interfaces which
   # can be created by user dynamically and its raSuppress flag is set by default.
   # Rest of the interfaces do not have their raSuppress flag set.
   raSuppress = intfName.startswith( ( "Loopback", "Management" ) )
   managedConfig = False
   # Fetch config if present
   # Sysdb/routing6/routerAdvt/config/intf/<Intf>
   # nsRetransmitInterval was moved to Ira::Ip6.tin after 35951 fix, hence check
   # ip6/config/intf/<intf>/nsRetransmitInterval
   if intfName in ip6Config.intf:
      ip6IntfConfig = ip6Config.intf[ intfName ]
      retransmitInterval = ip6IntfConfig.nsRetransmitInterval

   if intfName in routerAdvtConfig.intf:
      ip6NdIntfConfig = routerAdvtConfig.intf.get( intfName )
      raSuppress = ip6NdIntfConfig.raSuppress
      reachableTime = ip6NdIntfConfig.reachableTime
      raInterval = ip6NdIntfConfig.raIntervalMaxMsec
      raLifetime = ip6NdIntfConfig.raLifetime
      routerPreference = ip6NdIntfConfig.routerPreference
      managedConfig = ip6NdIntfConfig.managedConfigFlag
      hopLimit = ip6NdIntfConfig.hopLimit

   result.ndRetransmitInterval = retransmitInterval or nsFromKernel

   # Set ND RA info only when ipv6 unicast routing is enabled
   if routerAdvtStatus.routerAdvtDaemonActive:
      ndInfoDetails = Ip6Status.NeighborDiscoverDetails()
      result.neighborDiscoverDetails = ndInfoDetails
      ndInfoDetails.ndRouterAdvRetransmitInterval = retransmitInterval
      ndInfoDetails.ndRouterAdvInterval = raInterval
      ndInfoDetails.ndAdvertisedReachableTime = reachableTime or 1
      ndInfoDetails.routerPreference = routerPreference.title()
      ndInfoDetails.routerAdvLifetime = raLifetime
      ndInfoDetails.routerAdvHopLimit = hopLimit
      ndInfoDetails.ndRouterAdvSuppressed = raSuppress

   if managedConfig:
      if intfName.startswith( "Loopback" ):
         result.autoConfigStatus = "dhcp"
      else:
         result.autoConfigStatus = "stateful"
   else:
      result.autoConfigStatus = "stateless"

   return result

#-------------------------------------------------------------------------------
# Adds IPv6-ND specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class Ip6NdIntfConfigModelet( CliParser.Modelet ):

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ): # pylint: disable-msg=W0231
      CliParser.Modelet.__init__( self ) # This is a no-op
      self.mode = intfConfigMode
      self.ip6NdIntfConf = Ipv6RouterAdvtLib.Ip6NdIntfConfig(
                              routerAdvtConfig,
                              self.intfName() )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return IraIp6IntfCli.ipv6SupportedOnIntfConfigMode( mode )

   def intfName( self ):
      return self.mode.intf.name

   def printWarningIfNotRoutedPort( self ):
      IraCommonCli.printWarningIfNotRoutedPort( self.mode, self.intfName(),
                                    configType="IPv6" )

   def raConfigIntf( self, create=True ):
      return self.ip6NdIntfConf.ndRaConfigIntf( )

   def raSuppress( self, args ):
      self.printWarningIfNotRoutedPort( )
      self.ip6NdIntfConf.ndRaSuppress( 'all' in args )

   def noRaSuppress( self, args ):
      self.ip6NdIntfConf.noNdRaSuppress()

   def defaultRaSuppress( self, args ):
      if self.intfName().startswith( 'Management' ):
         self.ip6NdIntfConf.ndRaSuppress( True )
      else:
         self.ip6NdIntfConf.noNdRaSuppress()

   def raInterval( self, args ):
      minMsec = None
      if 'msec' in args:
         maxMsec = args[ 'MAX_MSEC' ]
         if 'MIN_MSEC' in args:
            minMsec = args[ 'MIN_MSEC' ]
      else:
         maxMsec = args[ 'MAX_SEC' ] * 1000
         if 'MIN_SEC' in args:
            minMsec = args[ 'MIN_SEC' ] * 1000

      if minMsec:
         if minMsec >= maxMsec:
            self.mode.addError( "Minimum value must be less than maximum value" )
            return
      self.ip6NdIntfConf.ndRaInterval( maxMsec, minMsec )

   def noRaInterval( self, args ):
      self.ip6NdIntfConf.noNdRaInterval()

   def raLifetime( self, args ):
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndRaLifetime( args[ 'LIFETIME' ] )

   def noRaLifetime( self, args ):
      self.ip6NdIntfConf.noNdRaLifetime()

   def hopLimit( self, args ):
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndRaHopLimitIs( args[ 'HOP_LIMIT' ] )

   def noHopLimit( self, args ):
      self.ip6NdIntfConf.noNdRaHopLimitIs()

   def raMtuSuppress( self, args ):
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndRaMtuSuppress()

   def noRaMtuSuppress( self, args ):
      self.ip6NdIntfConf.noNdRaMtuSuppress()

   def managedConfigFlag( self, args ):
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndManagedConfigFlag()

   def noManagedConfigFlag( self, args ):
      self.ip6NdIntfConf.noNdManagedConfigFlag()

   def otherConfigFlag( self, args ):
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndOtherConfigFlag()

   def noOtherConfigFlag( self, args ):
      self.ip6NdIntfConf.noNdOtherConfigFlag()

   def ndReachableTime( self, args ):
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndReachableTime( args[ 'TIME' ] )

   def noNdReachableTime( self, args ):
      self.ip6NdIntfConf.noNdReachableTime()

   def ndRouterPreference( self, args ):
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndRouterPreference( args[ 'PREFERENCE' ] )

   def noNdRouterPreference( self, args ):
      self.ip6NdIntfConf.noNdRouterPreference()

   # Creates a new prefix object. If one already exists, it deletes it and
   # recreates the object.
   def ndIntfConfigPrefix( self, ip6Prefix ):
      self.ip6NdIntfConf.ndIntfConfigPrefix( ip6Prefix )

   def ndPrefixNoAdvertise( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndPrefixNoAdvertise( ip6Prefix )

   def ndPrefixParameters( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      validLifetime, preferredLifetime, flags = processPrefixParameters( args )
      self.printWarningIfNotRoutedPort()
      self.ip6NdIntfConf.ndPrefixParameters( ip6Prefix,
                                             validLifetime=validLifetime,
                                             preferredLifetime=preferredLifetime,
                                             flagsList=flags )

   def ndPrefixProfile( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      profileName = args[ 'PROFILE_NAME' ]
      if profileName not in routerAdvtConfig.prefixProfiles:
         self.mode.addError( "Prefix profile %s does not exist" % profileName )
         return
      self.ip6NdIntfConf.ndPrefixProfile( ip6Prefix, profileName )

   def noNdPrefix( self, args ):
      ip6Prefix = args[ 'IP6PREFIX' ]
      self.ip6NdIntfConf.noNdPrefix( ip6Prefix )

   def dnsServer( self, args ):
      ip6Addr = args[ 'IP6ADDR' ]
      lifetime = args.get( 'LIFETIME' )

      intfConfig = self.raConfigIntf()

      if ip6Addr in intfConfig.dnsServers:
         dnsServerEntry = intfConfig.dnsServers[ ip6Addr ]
      else:
         dnsServerEntry = intfConfig.dnsServers.newMember( ip6Addr )

      if lifetime is None:
         dnsServerEntry.lifetimeConfigured = False
         dnsServerEntry.lifetime = dnsServerEntry.lifetimeDefault
      else:
         dnsServerEntry.lifetimeConfigured = True
         dnsServerEntry.lifetime = lifetime

      return dnsServerEntry

   def noDnsServer( self, args ):
      ip6Addr = args[ 'IP6ADDR' ]
      intfConfig = self.raConfigIntf()
      if not intfConfig:
         return
      if ip6Addr in intfConfig.dnsServers:
         del intfConfig.dnsServers[ ip6Addr ]

   def dnsServersLifetime( self, args ):
      intfConfig = self.raConfigIntf()
      intfConfig.dnsServersLifetimeConfigured = True
      intfConfig.dnsServersLifetime = args[ 'LIFETIME' ]

   def noDnsServersLifetime( self, args ):
      intfConfig = self.raConfigIntf( False )
      if not intfConfig:
         return
      intfConfig.dnsServersLifetimeConfigured = False
      intfConfig.dnsServersLifetime = intfConfig.dnsServersLifetimeDefault

   def dnsSuffix( self, args ):
      suffix = args[ 'SUFFIX' ]
      lifetime = args.get( 'LIFETIME' )
      intfConfig = self.raConfigIntf()
      if suffix in intfConfig.dnsSuffixes:
         dnsSuffixEntry = intfConfig.dnsSuffixes[ suffix ]
      else:
         dnsSuffixEntry = intfConfig.dnsSuffixes.newMember( suffix )
      if lifetime is None:
         dnsSuffixEntry.lifetimeConfigured = False
         dnsSuffixEntry.lifetime = dnsSuffixEntry.lifetimeDefault
      else:
         dnsSuffixEntry.lifetimeConfigured = True
         dnsSuffixEntry.lifetime = lifetime
      return dnsSuffixEntry

   def noDnsSuffix( self, args ):
      suffix = args[ 'SUFFIX' ]
      intfConfig = self.raConfigIntf()
      if not intfConfig:
         return
      del intfConfig.dnsSuffixes[ suffix ]

   def dnsSuffixesLifetime( self, args ):
      intfConfig = self.raConfigIntf()
      if not intfConfig:
         return
      intfConfig.dnsSuffixesLifetimeConfigured = True
      intfConfig.dnsSuffixesLifetime = args[ 'LIFETIME' ]

   def noDnsSuffixesLifetime( self, args ):
      intfConfig = self.raConfigIntf( False )
      if not intfConfig:
         return
      intfConfig.dnsSuffixesLifetimeConfigured = False
      intfConfig.dnsSuffixesLifetime = intfConfig.dnsSuffixesLifetimeDefault

   def networkBootOption( self, args ):
      intfConfig = self.raConfigIntf()
      if not intfConfig:
         return
      intfConfig.networkBootFileUrl = args[ 'URL' ]

   def noNetworkBootOption( self, args ):
      intfConfig = self.raConfigIntf()
      if not intfConfig:
         return
      intfConfig.networkBootFileUrl = ''

#-------------------------------------------------------------------------------
# Associate the Ip6NdIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( Ip6NdIntfConfigModelet )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra ( suppress | disabled ) [ all ]
#--------------------------------------------------------------------------------
class Ipv6NdRaSuppressCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra ( suppress | disabled ) [ all ]'
   noOrDefaultSyntax = 'ipv6 nd ra ( suppress | disabled ) ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'suppress': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'suppress',
            helpdesc='Disable Ipv6 Router advertisement messages on the interface' ),
         deprecatedByCmd='ipv6 nd ra disabled' ),
      'disabled': 'Disable Ipv6 Router advertisement messages on the interface',
      'all': ( 'Disable both Router Advertisement and Router Solicitation '
                'responses on the interface' ),
   }

   handler = Ip6NdIntfConfigModelet.raSuppress
   noHandler = Ip6NdIntfConfigModelet.noRaSuppress
   defaultHandler = Ip6NdIntfConfigModelet.defaultRaSuppress

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaSuppressCmd )

#-------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra interval ( ( MAX_SEC [ MIN_SEC ] ) |
#    ( msec MAX_MSEC [ MIN_MSEC ] ) )
#-------------------------------------------------------------------------------
class Ipv6NdRaIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra interval ( ( MAX_SEC [ MIN_SEC ] ) | ' \
            '( msec MAX_MSEC [ MIN_MSEC ] ) )'
   noOrDefaultSyntax = 'ipv6 nd ra interval ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd' : matcherNd,
      'ra' : matcherRa,
      'interval' : matcherInterval,
      'MAX_SEC' : CliMatcher.IntegerMatcher( 4, 1800,
         helpdesc='Maximum interval (seconds)' ),
      'MIN_SEC' : CliMatcher.IntegerMatcher( 3, 1799,
         helpdesc='Minimum interval (seconds)' ),
      'msec' : 'Interval in units of milliseconds',
      # Accept a lower bound to support older configurations
      # Display the minimal values based on the RFC4861 Section 6.2.1
      'MAX_MSEC' : CliMatcher.IntegerMatcher( 500, 1800000,
         helpdesc='Maximum interval (milliseconds)',
         helpname='<4000-1800000>' ),
      'MIN_MSEC' : CliMatcher.IntegerMatcher( 375, 1799999,
         helpdesc='Minimum interval (milliseconds)',
         helpname='<3000-1800000>' ),
   }

   handler = Ip6NdIntfConfigModelet.raInterval
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noRaInterval

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaIntervalCmd )

#--------------------------------------------------------------------------------
# ipv6 nd ra lifetime LIFETIME
#--------------------------------------------------------------------------------
class Ipv6NdRaLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra lifetime LIFETIME'
   noOrDefaultSyntax = 'ipv6 nd ra lifetime ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'lifetime': ( "Configure 'router lifetime' value in router advertisements "
                     "on this interface( Default: 1800 seconds )" ),
      'LIFETIME': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='Ra lifetime value in Seconds' ),
   }

   handler = Ip6NdIntfConfigModelet.raLifetime
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noRaLifetime

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaLifetimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra mtu suppress
#--------------------------------------------------------------------------------
class Ipv6NdRaMtuSuppressCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra mtu suppress'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'mtu': 'IPv6 RA MTU Option',
      'suppress': 'Suppress IPv6 RA MTU Option',
   }

   handler = Ip6NdIntfConfigModelet.raMtuSuppress
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noRaMtuSuppress

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaMtuSuppressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd managed-config-flag
#--------------------------------------------------------------------------------
class Ipv6NdManagedConfigFlagCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd managed-config-flag'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'managed-config-flag': ( "Set the 'managed address configuration flag' "
                                "in IPv6 router advertisements" ),
   }

   handler = Ip6NdIntfConfigModelet.managedConfigFlag
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noManagedConfigFlag

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdManagedConfigFlagCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd other-config-flag
#--------------------------------------------------------------------------------
class Ipv6NdOtherConfigFlagCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd other-config-flag'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'other-config-flag': ( "Set the 'other stateful configuration' "
                              "flag in IPv6 router advertisements" ),
   }

   handler = Ip6NdIntfConfigModelet.otherConfigFlag
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noOtherConfigFlag

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdOtherConfigFlagCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd reachable-time TIME
#--------------------------------------------------------------------------------
class Ipv6NdReachableTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd reachable-time TIME'
   noOrDefaultSyntax = 'ipv6 nd reachable-time ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'reachable-time': ( 'Configure reachable time - the amount of time that a '
                           'remote IPv6 node is considered reachable' ),
      'TIME': CliMatcher.IntegerMatcher( 0, 0xffffffff, helpdesc='Milliseconds' ),
   }

   handler = Ip6NdIntfConfigModelet.ndReachableTime
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noNdReachableTime

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdReachableTimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd router-preference PREFERENCE
#--------------------------------------------------------------------------------
class Ipv6NdRouterPreferenceCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd router-preference PREFERENCE'
   noOrDefaultSyntax = 'ipv6 nd router-preference ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'router-preference': ( 'Configure default router preference for the router '
                              'on this interface ( Default: medium )' ),
      'PREFERENCE': CliMatcher.EnumMatcher( {
         'high': 'Set default router preference to high',
         'medium': 'Set default router preference to medium',
         'low': 'Set default router preference to low',
      } ),
   }

   handler = Ip6NdIntfConfigModelet.ndRouterPreference
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noNdRouterPreference

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRouterPreferenceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd prefix IP6PREFIX [ ( infinite | VALIDLIFETIME )
#    [ ( infinite | PREFERREDLIFETIME ) ] ] [ { no-autoconfig | no-onlink } ]
#--------------------------------------------------------------------------------
def processPrefixParameters( args ):
   validLifetime = None
   if 'VALID_LIFETIME' in args:
      validLifetime = args[ 'VALID_LIFETIME' ]
   elif 'valid_infinite' in args:
      validLifetime = LIFETIME_MAX_VALUE

   preferredLifetime = None
   if 'PREFERRED_LIFETIME' in args:
      preferredLifetime = args[ 'PREFERRED_LIFETIME' ]
   elif 'preferred_infinite' in args:
      preferredLifetime = LIFETIME_MAX_VALUE

   flags = []
   if 'no-autoconfig' in args:
      flags.append( 'no-autoconfig' )
   if 'no-onlink' in args:
      flags.append( 'no-onlink' )

   return validLifetime, preferredLifetime, flags

class Ipv6NdPrefixCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv6 nd prefix IP6PREFIX [ ( valid_infinite | VALID_LIFETIME )
               [ ( preferred_infinite | PREFERRED_LIFETIME ) ] ]
               [ { no-autoconfig | no-onlink } ]'''

   noOrDefaultSyntax = 'ipv6 nd prefix IP6PREFIX ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd' : matcherNd,
      'prefix' : matcherPrefix,
      'IP6PREFIX' : matcherPrefixAddr,
      'VALID_LIFETIME' : matcherValidLifetime,
      'valid_infinite' : matcherValidInfiniteLifetime,
      'PREFERRED_LIFETIME' : matcherPreferredLifetime,
      'preferred_infinite' : matcherPreferredInfiniteLifetime,
      'no-autoconfig' : matcherNoAutoconfig,
      'no-onlink' : matcherNoOnlink,
   }

   handler = Ip6NdIntfConfigModelet.ndPrefixParameters
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noNdPrefix

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdPrefixCmd )

#--------------------------------------------------------------------------------
# ipv6 nd prefix IP6PREFIX no-advertise
#--------------------------------------------------------------------------------
class Ipv6NdPrefixNoAdvertiseCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd prefix IP6PREFIX no-advertise'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd' : matcherNd,
      'prefix' : matcherPrefix,
      'IP6PREFIX' : matcherPrefixAddr,
      'no-advertise' : matcherNoAdvertise,
   }

   handler = Ip6NdIntfConfigModelet.ndPrefixNoAdvertise

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdPrefixNoAdvertiseCmd )

#--------------------------------------------------------------------------------
# ipv6 nd prefix IP6PREFIX profile PROFILE_NAME
#--------------------------------------------------------------------------------
class Ipv6NdPrefixProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd prefix IP6PREFIX profile PROFILE_NAME'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd' : matcherNd,
      'prefix' : matcherPrefix,
      'IP6PREFIX' : matcherPrefixAddr,
      'profile' : 'Apply profile configurations to prefix',
      'PROFILE_NAME' : matcherProfileName,
   }

   handler = Ip6NdIntfConfigModelet.ndPrefixProfile

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdPrefixProfileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd prefix profile PROFILE_NAME
#    [ ( infinite | VALID_LIFETIME ) [ ( infinite | PREFERRED_LIFETIME ) ] ]
#    [ { no-autoconfig | no-onlink } ]
#--------------------------------------------------------------------------------

def ndPrefixProfileParameters( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   validLifetime, preferredLifetime, flags = processPrefixParameters( args )
   ip6NdConfig = Ipv6RouterAdvtLib.Ip6NdConfig( routerAdvtConfig )
   ip6NdConfig.ndPrefixProfileParameters( profileName,
                                          validLifetime=validLifetime,
                                          preferredLifetime=preferredLifetime,
                                          flagsList=flags )

def noNdPrefixProfile( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   ip6NdConfig = Ipv6RouterAdvtLib.Ip6NdConfig( routerAdvtConfig )
   if not ip6NdConfig.noNdPrefixProfile( profileName ):
      mode.addError( 'First remove profile %s from interfaces '
                     'that have prefixes using it' % profileName )

def defaultNdPrefixProfile( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   ip6NdConfig = Ipv6RouterAdvtLib.Ip6NdConfig( routerAdvtConfig )
   ip6NdConfig.defaultNdPrefixProfile( profileName )

class Ipv6NdPrefixProfileGlobalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd prefix profile PROFILE_NAME ' \
      '[ ( valid_infinite | VALID_LIFETIME ) ' \
      '[ ( preferred_infinite | PREFERRED_LIFETIME ) ] ] ' \
      '[ { no-autoconfig | no-onlink } ]'
   noOrDefaultSyntax = 'ipv6 nd prefix profile PROFILE_NAME ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
      'nd' : matcherNd,
      'prefix' : matcherPrefix,
      'profile' : matcherProfile,
      'PROFILE_NAME' : matcherProfileName,
      'VALID_LIFETIME' : matcherValidLifetime,
      'valid_infinite' : matcherValidInfiniteLifetime,
      'PREFERRED_LIFETIME' : matcherPreferredLifetime,
      'preferred_infinite' : matcherPreferredInfiniteLifetime,
      'no-autoconfig' : matcherNoAutoconfig,
      'no-onlink' : matcherNoOnlink,
   }

   handler = ndPrefixProfileParameters
   noHandler = noNdPrefixProfile
   defaultHandler = defaultNdPrefixProfile

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6NdPrefixProfileGlobalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd prefix profile PROFILE_NAME no-advertise
#--------------------------------------------------------------------------------

class Ipv6NdPrefixProfileNoAdvertiseCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd prefix profile PROFILE_NAME no-advertise'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
      'nd': matcherNd,
      'prefix': matcherPrefix,
      'profile': matcherProfile,
      'PROFILE_NAME': matcherProfileName,
      'no-advertise': matcherNoAdvertise,
   }

   @staticmethod
   def handler( mode, args ):
      ip6NdConfig = Ipv6RouterAdvtLib.Ip6NdConfig( routerAdvtConfig )
      ip6NdConfig.ndPrefixProfileNoAdvertise( args[ 'PROFILE_NAME' ] )

BasicCli.GlobalConfigMode.addCommandClass( Ipv6NdPrefixProfileNoAdvertiseCmd )

#--------------------------------------------------------------------------------
# show ipv6 nd ra neighbors [ VRF | ( interface INTF )
#--------------------------------------------------------------------------------
def constructInterfaceNeighborModel( intfs ):
   neighborsRa = [ x for x  in routerNeighborRaStatus.routerNeighborRa
                   if x.intfId in intfs ]
   intfNeighbors = Ip6RaCliModels.InterfaceNeighbors()

   # interface -> [ RouterNeighbor ]
   # This temporary dict is needed because we can't append to Model's List
   tmpHolder = {}
   for neiRaKey in neighborsRa:
      lastRa = routerNeighborRaStatus.routerNeighborRa[ neiRaKey ].lastRaReceived
      nei = Ip6RaCliModels.RouterNeighbor( address=neiRaKey.neighborAddr,
                                           lastRa=lastRa )
      tmpHolder.setdefault( neiRaKey.intfId, [] ).append( nei )

   # interface -> List( RouterNeighbor )
   # Convert python list to Model List
   tmpIntfNeighbor = {}
   for intf in tmpHolder:
      neighbors = sorted( tmpHolder[ intf ], key=operator.attrgetter( 'address' ),
                          cmp=compareIp6Address )
      tmpIntfNeighbor[ intf ] = Ip6RaCliModels.RouterNeighbors(
                                             routerNeighbors=neighbors )

   intfNeighbors.interfaces = tmpIntfNeighbor
   return intfNeighbors

class ShowIpv6NdRaNeighbors( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 nd ra neighbors [ VRF | ( interface INTF ) ]'
   data = {
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'nd' : matcherNd,
      'ra' : matcherRa,
      'neighbors' : matcherNeighbors,
      'VRF': VrfExprFactory( helpdesc='Show neighbors in a VRF',
                             inclAllVrf=True,
                             inclDefaultVrf=True ),
      'interface' : 'Show neighbors of an interface',
      'INTF' : IntfCli.Intf.matcher
   }
   cliModel = Ip6RaCliModels.VrfNeighbors

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      vrfName = args.get( 'VRF' )
      vrfNeighbors = Ip6RaCliModels.VrfNeighbors()
      if vrfName == 'all':
         # vrf -> InterfaceNeighbors
         # This temporary dict is needed because we need the complete dict to
         # construct the model
         tmpHolder = {}
         for vrf in ip6Status.vrfIp6IntfStatus:
            intfs = ip6Status.vrfIp6IntfStatus[ vrf ].ip6IntfStatus
            intfNeigh = constructInterfaceNeighborModel( intfs )
            if intfNeigh.interfaces:
               tmpHolder[ vrf ] = intfNeigh
         vrfNeighbors.vrfs = tmpHolder
         return vrfNeighbors

      if intf and intf.name in ip6Status.intf:
         # user has specified an interface filter
         vrfName = ip6Status.intf[ intf.name ].vrf
         intfModel = constructInterfaceNeighborModel( [ intf.name ] )
         vrfNeighbors.vrfs = { vrfName : intfModel }
      elif not intf:
         if not vrfName:
            # if user didn't specify any filter, assume default filter
            vrfName = 'default'
         if vrfName in ip6Status.vrfIp6IntfStatus:
            vrfIp6 = ip6Status.vrfIp6IntfStatus[ vrfName ]
            intfs = vrfIp6.ip6IntfStatus
            intfModel = constructInterfaceNeighborModel( intfs )
            vrfNeighbors.vrfs = { vrfName : intfModel }
      return vrfNeighbors

BasicCli.addShowCommandClass( ShowIpv6NdRaNeighbors )

#--------------------------------------------------------------------------------
# clear ipv6 nd ra neighbors [ VRF | ( interface INTF [ IP6_ADDR ) ]
#--------------------------------------------------------------------------------

class ClearIpv6NdRaNeighbors( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 nd ra neighbors [ VRF | ( interface INTF [ IP6_ADDR ] ) ]'
   data = {
      'clear': clearKwNode,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForClear,
      'nd' : matcherNd,
      'ra' : matcherRa,
      'neighbors' : matcherNeighbors,
      'VRF': VrfExprFactory( helpdesc='Clear neighbors in a VRF',
                             inclAllVrf=True,
                             inclDefaultVrf=True ),
      'all': 'Clear neighbors in all VRFs',
      'interface' : 'Clear neighbors of an interface',
      'INTF' : IntfCli.Intf.matcher,
      'IP6_ADDR' : Ip6AddrMatcher.Ip6AddrMatcher( 'IPv6 address of neighbor' )
   }

   @staticmethod
   def handler( mode, args ):
      if not routerAdvtStatus.routerAdvtDaemonActive:
         return
      cmdStr = 'neighbors'
      intf = args.get( 'INTF' )
      vrf = args.get( 'VRF', DEFAULT_VRF )
      if intf:
         cmdStr += ' intf ' + intf.name
         addr = args.get( 'IP6_ADDR' )
         if addr:
            cmdStr += ' ' + addr.stringValue
      else:
         cmdStr += ' vrf ' + vrf

      AgentCommandRequest.runSocketCommand( entityManager, Ipv6RouterAdvtAgent.name,
                                            'clear', cmdStr )

BasicCliModes.EnableMode.addCommandClass( ClearIpv6NdRaNeighbors )

#--------------------------------------------------------------------------------
# show ipv6 nd ra internal state [ VRF | INTF ]
#--------------------------------------------------------------------------------

class Ipv6NdRaInternalStateCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 nd ra internal state [ VRF | INTF ]'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
      'nd': matcherNd,
      'ra': matcherRa,
      'internal': CliCommand.guardedKeyword( 'internal',
         helpdesc='Internal commands for Ipv6RouterAdvt Daemon',
         guard=CommonGuards.standbyGuard ),
      'state': 'Internal state of Ipv6RouterAdvt Daemon',
      'VRF': VrfExprFactory( helpdesc='VRF name',
                             inclDefaultVrf=True,
                             inclAllVrf=True ),
      'INTF': IntfCli.Intf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      vrfName = args.get( 'VRF' )
      if not routerAdvtStatus.routerAdvtDaemonActive:
         return
      cmdStr = 'state'
      if intf:
         cmdStr += ' intf ' + intf.name
      elif vrfName:
         cmdStr += ' vrf ' + vrfName
      else:
         cmdStr += ' vrf default'
      AgentCommandRequest.runSocketCommand( entityManager, Ipv6RouterAdvtAgent.name,
                                             "show", cmdStr )

BasicCli.addShowCommandClass( Ipv6NdRaInternalStateCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra dns-server IP6ADDR [ lifetime LIFETIME ]
#--------------------------------------------------------------------------------
class Ipv6NdRaDnsServerCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra dns-server IP6ADDR [ lifetime LIFETIME ]'
   noOrDefaultSyntax = 'ipv6 nd ra dns-server IP6ADDR ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd' : matcherNd,
      'ra' : matcherRa,
      'dns-server' : 'Configure DNS server',
      'IP6ADDR' : Ip6AddrMatcher.Ip6AddrMatcher(
         helpdesc='IPv6 address of DNS server' ),
      'lifetime' : 'Configure "DNS server address lifetime" value '
                   '(Default: use if-config)',
      'LIFETIME' : matcherDnsServerLifetime,
   }

   handler = Ip6NdIntfConfigModelet.dnsServer
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noDnsServer

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaDnsServerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra dns-servers lifetime LIFETIME
#--------------------------------------------------------------------------------
class Ipv6NdRaDnsServersLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra dns-servers lifetime LIFETIME'
   noOrDefaultSyntax = 'ipv6 nd ra dns-servers lifetime ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'dns-servers' : 'Configure DNS servers',
      'lifetime': 'Configure "DNS servers lifetime" value',
      'LIFETIME': matcherDnsServerLifetime,
   }

   handler = Ip6NdIntfConfigModelet.dnsServersLifetime
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noDnsServersLifetime

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaDnsServersLifetimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra dns-suffix SUFFIX [ lifetime LIFETIME ]
#--------------------------------------------------------------------------------
class Ipv6NdRaDnsSuffixCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra dns-suffix SUFFIX [ lifetime LIFETIME ]'
   noOrDefaultSyntax = 'ipv6 nd ra dns-suffix SUFFIX ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'dns-suffix' : 'Configure DNS suffix',
      'SUFFIX' : CliMatcher.PatternMatcher( pattern='[^:|> ]+',
         helpdesc='DNS domain name suffix', helpname='WORD' ),
      'lifetime' : 'Configure "DNS suffix lifetime" value '
                   '(Default: use if-config)',
      'LIFETIME' : matcherDnsSuffixLifetime,
   }

   handler = Ip6NdIntfConfigModelet.dnsSuffix
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noDnsSuffix

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaDnsSuffixCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra dns-suffixes lifetime LIFETIME
#--------------------------------------------------------------------------------
class Ipv6NdRaDnsSuffixesLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra dns-suffixes lifetime LIFETIME'
   noOrDefaultSyntax = 'ipv6 nd ra dns-suffixes lifetime ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'dns-suffixes': 'Configure DNS suffixes',
      'lifetime': 'Configure "DNS suffixes lifetime" value',
      'LIFETIME' : matcherDnsSuffixLifetime,
   }

   handler = Ip6NdIntfConfigModelet.dnsSuffixesLifetime
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noDnsSuffixesLifetime

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaDnsSuffixesLifetimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra experimental network-boot URL
#--------------------------------------------------------------------------------
class NetworkBootOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra experimental network-boot URL'
   noOrDefaultSyntax = 'ipv6 nd ra experimental network-boot ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'experimental': 'Configure experimental options',
      'network-boot': 'Configure network boot URL',
      'URL': CliMatcher.StringMatcher( helpdesc='Network boot URL',
         helpname='URL' ),
   }

   handler = Ip6NdIntfConfigModelet.networkBootOption
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noNetworkBootOption

Ip6NdIntfConfigModelet.addCommandClass( NetworkBootOptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra hop-limit HOP_LIMIT
#
# Max hop limit value advertised in RA messages, provides separation between
# what the router advertises and what it uses internally. It is the advertised
# recommended maximum hop count to use for packets before packets are dropped.
#--------------------------------------------------------------------------------
class Ipv6NdRaHopLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra hop-limit HOP_LIMIT'
   noOrDefaultSyntax = 'ipv6 nd ra hop-limit ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
      'nd': matcherNd,
      'ra': matcherRa,
      'hop-limit': 'Configure hop limit',
      'HOP_LIMIT': CliMatcher.IntegerMatcher( 0, 255,
         helpdesc=( 'Ra hop limit value. The valid value should be within the '
                    'range(0, 255) with 0 specifying no recommended value' ) ),
   }

   handler = Ip6NdIntfConfigModelet.hopLimit
   noOrDefaultHandler = Ip6NdIntfConfigModelet.noHopLimit

Ip6NdIntfConfigModelet.addCommandClass( Ipv6NdRaHopLimitCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd ra consistency-check default
#
# Enables router advertisement consistency logging globally. Disabled by default.
#--------------------------------------------------------------------------------
class Ipv6NdRaConsistencyCheckDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd ra consistency-check default'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
      'nd': matcherNd,
      'ra': matcherRa,
      'consistency-check': 'Router advertisement consistency logging options',
      'default': 'Enable router advertisement consistency logging',
   }

   @staticmethod
   def handler( mode, args ):
      routerAdvtConfig.raConsistency = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      routerAdvtConfig.raConsistency = routerAdvtConfig.raConsistencyDefault

BasicCli.GlobalConfigMode.addCommandClass( Ipv6NdRaConsistencyCheckDefaultCmd )

#---------------------------------------------------------------------------------
# Ip6RouterAdvtIntf class is registered with IntfCli.Intf class to register
# callbacks. Destroy method of Ip6RouterAdvtIntfclass is called by Intf class when
# an interface object is deleted. Intf class will create a new instance of
# Ip6RouterAdvtIntf and call destroy method on it.
# ---------------------------------------------------------------------------------

class Ip6RouterAdvtIntf( IntfCli.IntfDependentBase ):
   def __init__( self, intf, sysdbRoot ):
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   #--------------------------------------------------------------------------------
   # Destroys the Ipv6 RouterAdvt IntfConfig object for this interface if it exists.
   #--------------------------------------------------------------------------------
   def setDefault( self ):
      intfName = self.intf_.name # pylint:disable-msg=E1101
      del routerAdvtConfig.intf[ intfName ]

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( em ):
   global entityManager
   global routerAdvtConfig, routerAdvtStatus
   global allIntfStatusDir, ip6Config
   global routerNeighborRaStatus, ip6Status

   entityManager = em
   ip6Config = LazyMount.mount( em, "ip6/config", "Ip6::Config", "r" )
   routerAdvtConfig = ConfigMount.mount( em, "routing6/routerAdvt/config",
                                         "RouterAdvt::Config", "w" )
   routerAdvtStatus = LazyMount.mount( em, "routing6/routerAdvt/status",
                                       "RouterAdvt::Status", "r" )
   allIntfStatusDir = LazyMount.mount( em, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   ip6Status = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   mountPath = Tac.Type( "RouterAdvt::Smash::RouterNeighborRaStatus" ).smashPath
   routerNeighborRaStatus = SmashLazyMount.mount( em, mountPath,
                                      "RouterAdvt::Smash::RouterNeighborRaStatus",
                                      SmashLazyMount.mountInfo( 'reader' ) )
   IntfCli.Intf.registerDependentClass( Ip6RouterAdvtIntf )

   IraIp6IntfCli.extraIp6NdRaInfoHook = _extraIp6NdRaInfo
