# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module contains the definition of CliParser rules for matching IP
# addresses and IP address prefixes.
#-------------------------------------------------------------------------------
import re
from socket import AF_INET6

import Arnet
import CliCommand
import CliMatcher
import CliParser
from CliParserCommon import MatchResult, noMatch
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import Tac

def _createPrefixAutozero( mode, match ):
   return Tac.Value( 'Arnet::Ip6Prefix',
                     address=match.subnet.address,
                     len=match.len )

def _createPrefixRejectInvalid( mode, match ):
   if match and match.validAsPrefix:
      return Tac.Value( 'Arnet::Ip6Prefix',
                        address=match.address, len=match.len )
   else:
      return None

class Ip6AddrMatcher( CliMatcher.Matcher ):
   def __init__( self, helpdesc, inverse=False, checkContinuous=False,
         **kargs ):
      super( Ip6AddrMatcher, self ).__init__( helpdesc=helpdesc, **kargs )
      completionText = 'A:B:C:D:E:F:G:H'
      self.completion_ = CliParser.Completion( completionText, helpdesc, False )
      self.inverse_ = inverse
      self.checkContinuous_ = checkContinuous

      # Strings found in the exception of an incomplete ipv6 address
      self.ip6AddrIncompleteRe_ = \
         re.compile( '(incomplete ipv6 address|expected)' )

   def match( self, mode, context, token ):
      m = Arnet.Ip6AddrCompiledRe.match( token )
      if m is None or m.group( 0 ) != token:
         return noMatch

      # we match an ipv4 address, so we shouldn't try to make an Arnet Ip6Addr obj
      ipMatch = Arnet.IpAddrCompiledRe.match( token )
      if ipMatch is not None and ipMatch.group( 0 ) == token:
         return noMatch

      try:
         addr = Arnet.Ip6Addr( m.group( 0 ) )
      except IndexError:
         return noMatch
      if self.checkContinuous_:
         try:
            Arnet.Mask( addr.stringValue, inverse=self.inverse_,
                        addrFamily=AF_INET6 )
         except ValueError:
            return noMatch
      return MatchResult( addr, token )

   def completions( self, mode, context, token ):
      m = Arnet.Ip6AddrCompiledRe.match( token )
      if m is None or m.group( 0 ) != token:
         return []
      try:
         Arnet.Ip6Addr( m.group( 0 ) )
      except IndexError, e:
         m = self.ip6AddrIncompleteRe_.search( str( e ) )
         if m is None:
            return []
      except: # pylint: disable=bare-except
         return []
      return [ self.completion_ ]

   def __str__( self ):
      return '<IPv6 address>'

class Ip6PrefixMatcher( CliMatcher.Matcher ):
   '''Type of matcher that matches an IPv6 address prefix, and
   evaluates to the corresponding Arnet::Ip6AddrWithMask value.
   '''
   prefixRe_ = re.compile( r'(%s)/(\d{1,3})$' % Arnet.Ip6AddrRe )
   completionRe_ = re.compile( r'(?:(%s)(?:/(?:(\d{1,3}))?)?)?$' %
                               Arnet.Ip6AddrRe )

   def __init__( self, helpdesc, overlap=IpAddrMatcher.PREFIX_OVERLAP_ALLOW,
         **kargs ):
      super( Ip6PrefixMatcher, self ).__init__( helpdesc=helpdesc, **kargs )
      self.completion_ = CliParser.Completion( 'A:B:C:D:E:F:G:H/I', helpdesc,
                                               False )
      self.ip6AddrMatcher_ = Ip6AddrMatcher( helpdesc )
      self.prefixLenMatcher_ = CliMatcher.IntegerMatcher( 0, 128,
                                          helpdesc='Length of the prefix in bits' )
      self.overlap = overlap

   def match( self, mode, context, token ):
      m = self.prefixRe_.match( token )
      if m is None or m.group( 0 ) != token:
         return noMatch
      addr = self.ip6AddrMatcher_.match( mode, context, m.group( 1 ) )
      if addr is noMatch:
         return noMatch
      if self.prefixLenMatcher_.match( mode, context, m.group( 2 ) ) is noMatch:
         return noMatch
      result = Arnet.Ip6AddrWithMask( m.group( 0 ) )
      if self.overlap == IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO:
         result = _createPrefixAutozero( mode, result )
         if result is not None:
            result = Arnet.Ip6AddrWithMask( result.address, result.len )
      elif self.overlap == IpAddrMatcher.PREFIX_OVERLAP_REJECT:
         result = _createPrefixRejectInvalid( mode, result )
         if result is not None:
            result = Arnet.Ip6AddrWithMask( result.address, result.len )
      return MatchResult( result, token )

   def completions( self, mode, context, token ):
      m = self.completionRe_.match( token )
      if m is None or m.group( 0 ) != token:
         return []
      if m.group( 1 ) is not None:
         if self.ip6AddrMatcher_.completions( mode, context, m.group( 1 ) ) == []:
            return []
      if m.group( 2 ) is not None:
         if self.prefixLenMatcher_.completions( mode, context, m.group( 2 ) ) == []:
            return []
      return [ self.completion_ ]

   def __str__( self ):
      return '<IP address prefix>'

# Returns a valid prefix without overlap
class Ip6PrefixValidMatcher( CliMatcher.WrapperMatcher ):
   def __init__( self, helpdesc, overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT,
                 **kwargs ):
      if overlap == IpAddrMatcher.PREFIX_OVERLAP_REJECT:
         transformMatch = _createPrefixRejectInvalid
      elif overlap == IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO:
         transformMatch = _createPrefixAutozero
      else:
         raise ValueError( 'Unsupported overlap resolution type' )

      matcher = Ip6PrefixMatcher( helpdesc, overlap=overlap, value=transformMatch,
            **kwargs )
      CliMatcher.WrapperMatcher.__init__( self, matcher )

ip6AddrMatcher = Ip6AddrMatcher( 'IPv6 address' )
ip6PrefixMatcher = Ip6PrefixMatcher( 'IPv6 prefix' )

def _ip6PrefixExpr( name, addrdesc, maskdesc, prefixdesc, fullMaskValid=True,
                    fullMaskGuard=None, inverseMask=False,
                    overlap=IpAddrMatcher.PREFIX_OVERLAP_ALLOW,
                    maskCheckContinuous=True ):
   if fullMaskGuard:
      assert fullMaskValid

   prefixMatcher = Ip6PrefixMatcher( prefixdesc,
                                     value=lambda mode, match:
                                     Arnet.Ip6AddrWithFullMask(
                                        match.address,
                                        Arnet.ip6MaskFromPrefix( match.len ) ),
                                     overlap=overlap )

   class Ip6PrefixExpr( CliCommand.CliExpression ):
      expression = name
      data = { name : prefixMatcher }

   addrName = name + '_ADDR'
   maskName = name + '_MASK'

   class Ip6AddrWithPrefixOrFullMaskExpr( CliCommand.CliExpression ):
      expression = '%s | ( %s %s )' % ( name, addrName, maskName )
      data = { name : prefixMatcher,
               addrName : CliCommand.Node(
                  Ip6AddrMatcher( helpdesc=addrdesc ),
                  guard=fullMaskGuard ),
               maskName : Ip6AddrMatcher( helpdesc=maskdesc, inverse=inverseMask,
                                          checkContinuous=maskCheckContinuous )
              }

      @staticmethod
      def adapter( mode, args, argsList ):
         if addrName in args:
            ip6Addr = args.pop( addrName )
            ip6Mask = args.pop( maskName )
            # handle iteration
            isList = isinstance( ip6Addr, list )
            if not isList:
               ip6Addr = [ ip6Addr ]
               ip6Mask = [ ip6Mask ]
            newAddr = []
            for a, m in zip( ip6Addr, ip6Mask ):
               if inverseMask:
                  m = Tac.ValueConst( "Arnet::Ip6Addr",
                                      m.word0 ^ 0xffffffff,
                                      m.word1 ^ 0xffffffff,
                                      m.word2 ^ 0xffffffff,
                                      m.word3 ^ 0xffffffff )
               if overlap == IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO:
                  a = Tac.ValueConst( "Arnet::Ip6Addr",
                                      a.word0 & m.word0,
                                      a.word1 & m.word1,
                                      a.word2 & m.word2,
                                      a.word3 & m.word3 )
               newAddr.append( Arnet.Ip6AddrWithFullMask( a, m ) )
            if not isList:
               newAddr = newAddr[ 0 ]
            args[ name ] = newAddr

   if fullMaskValid:
      return Ip6AddrWithPrefixOrFullMaskExpr
   else:
      return Ip6PrefixExpr

class Ip6PrefixExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, addrdesc, maskdesc, prefixdesc, **kwargs ):
      self.addrdesc_ = addrdesc
      self.maskdesc_ = maskdesc
      self.prefixdesc_ = prefixdesc
      self.kwargs_ = kwargs
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      return _ip6PrefixExpr( name, self.addrdesc_, self.maskdesc_, self.prefixdesc_,
                            **self.kwargs_ )

def ip6PrefixExpr( addrdesc, maskdesc, prefixdesc, **kwargs ):
   return Ip6PrefixExprFactory( addrdesc, maskdesc, prefixdesc, **kwargs )

def ip6AddrWithPrefixOrFullMaskExpr( name, desc, fullMaskValid,
                                     fullMaskGuard, inverseMask=True,
                                     maskCheckContinuous=False ):
   addrdesc = '%s IPv6 address' % desc
   maskdesc = 'IPv6 %s address mask' % desc
   prefixdesc = 'IPv6 %s address prefix' % desc
   return _ip6PrefixExpr( name, addrdesc, maskdesc, prefixdesc,
                          fullMaskValid=fullMaskValid,
                          fullMaskGuard=fullMaskGuard, inverseMask=inverseMask,
                          maskCheckContinuous=maskCheckContinuous )

class Ip6AddrOrPrefixExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.overlap_ = overlap

   def generate( self, name ):
      '''Accept either an IPv6 address or a prefix'''
      class Ip6AddrOrPrefixExpr( CliCommand.CliExpression ):
         expression = 'IP6_ADDR_%s | IP6_PREFIX_%s' % ( name, name )
         data = {
            'IP6_ADDR_%s' % name : CliCommand.Node(
               matcher=Ip6AddrMatcher( 'Match this IPv6 address' ), alias=name ),
            'IP6_PREFIX_%s' % name : CliCommand.Node(
               matcher=Ip6PrefixValidMatcher( overlap=self.overlap_,
                  helpdesc='IPv6 address prefix' ),
               alias=name )
         }

      return Ip6AddrOrPrefixExpr
