# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This module implements the "[no] snmp trap link-change" command in config-if mode
# and the snmp mib ifmib ifindex command in enable mode.

import Arnet
import CliParser
import Tac
import ConfigMount
import Tracing
import CliModel
from IntfCli import IntfConfigMode
import IntfCli
from IntfModels import Interface
import CliMatcher
import CliCommand
import LazyMount

t0 = Tracing.trace0

snmpIntfConfig = None
dynamicIfIndexStatus = None

class SnmpIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   # pylint: disable-msg=W0231
   def __init__( self, intfConfigMode ):
      CliParser.Modelet.__init__( self )
      self.intfConfigMode = intfConfigMode
      
   @staticmethod
   def shouldAddModeletRule( mode ):
      return True

   @staticmethod
   def linkChangeTrapEnableIs( intf, enable ):
      t0( "Enabling" if enable else "Disabling", "link traps for", intf.name )
      intfId = Tac.Value( 'Arnet::IntfId', stringValue=intf.name )
      if intfId in snmpIntfConfig.trapConfig:
         cfg = Tac.nonConst( snmpIntfConfig.trapConfig[ intfId ] )
         cfg.linkStatusEnable = enable
      else:
         cfg = Tac.Value( 'Snmp::Interface::TrapConfig',
                          linkStatusEnable=enable )
      # pylint: disable-msg=E1103
      if cfg is not None and cfg != Tac.Value( 'Snmp::Interface::TrapConfig' ):
         snmpIntfConfig.trapConfig[ intfId ] = cfg
      else:
         del snmpIntfConfig.trapConfig[ intfId ]

   def enableLinkChangeTrap( self, args ):
      if CliCommand.isDefaultCmd( args ):
         self.setDefault( self.intfConfigMode.intf )
      else:
         self.linkChangeTrapEnableIs( self.intfConfigMode.intf,
                                      not CliCommand.isNoCmd( args ) )

   @staticmethod
   def setDefault( intf ):
      SnmpIntfModelet.linkChangeTrapEnableIs( intf, True )

IntfConfigMode.addModelet( SnmpIntfModelet )

#--------------------------------------------------------------------------------
# [ no | default ] snmp trap link-change
# Legacy:
# [ no | default ] snmp trap link-status
#--------------------------------------------------------------------------------
class SnmpTrapLinkChangeCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp trap ( link-change | link-status )'
   noOrDefaultSyntax = syntax
   data = {
      'snmp': 'Set interface SNMP parameters',
      'trap': 'Enable interface SNMP notifications',
      'link-change': 'Enable linkUp/linkDown SNMP notifications',
      'link-status': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                            'link-status',
                            helpdesc='Enable linkUp/linkDown SNMP notifications' ),
                         deprecatedByCmd='snmp trap link-change' ),
   }

   handler = SnmpIntfModelet.enableLinkChangeTrap
   noOrDefaultHandler = SnmpIntfModelet.enableLinkChangeTrap

SnmpIntfModelet.addCommandClass( SnmpTrapLinkChangeCmd )

#-------------------------------------------------------------------------------
# The "show snmp mib ifmib ifindex [interfaceName]" command, in "enable" mode.
#-------------------------------------------------------------------------------
class SnmpMibIfIndex( CliModel.Model ):
   ifIndex = CliModel.Dict( keyType=Interface, valueType=int, 
                            help="Mapping between interface name and the "
                                 "corresponding SNMP interface index" )
   pending = CliModel.List( valueType=Interface,
                            help="Interfaces for which an ifIndex value "
                                 "has not yet been allocated" )

   def render( self ):
      for interface in Arnet.sortIntf( self.ifIndex ):
         print "%s: Ifindex = %d" % ( interface, self.ifIndex[ interface ] )
      for interface in Arnet.sortIntf( self.pending ):
         print "%s: pending" % ( interface, )

def showSnmpIfIndex( mode, args ):
   intf = args.get( 'INTF' )
   result = SnmpMibIfIndex()
   intfs = IntfCli.Intf.getAll( mode, intf )
   if not intfs: 
      return result
   intfs = [ i for i in intfs if i.lookup() ]
   if not intfs: 
      return result

   for i in intfs:
      details = Tac.newInstance( "IntfSnmp::IntfDetails" )
      details.status = i.status()
      if details.ifIndex == details.ifIndexDynamic():
         result.pending.append( i.name )
      else:
         result.ifIndex[ i.name ] = details.ifIndex

   if intf is None:
      # If we're listing all values, add in the ifIndex values from the
      # dynamic collection, just in case they have been assigned but do
      # not currently exist in the system.
      for i, ifIndex in dynamicIfIndexStatus.dynamicIfIndex.iteritems():
         result.ifIndex[ i ] = ifIndex

   return result

class SnmpIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      SnmpIntfModelet.setDefault( self.intf_ )

def Plugin( entityManager ):
   global snmpIntfConfig
   global dynamicIfIndexStatus
   snmpIntfConfig = ConfigMount.mount( entityManager, 'snmp/interface/config',
                                       'Snmp::Interface::Config', 'w' )
   IntfCli.Intf.registerDependentClass( SnmpIntf )
   dynamicIfIndexStatus = LazyMount.mount( entityManager, 'interface/dynamicIfIndex',
                                           'Interface::DynamicIfIndexStatus', 'r' )
   # Mount the dynamic ifIndex collection, so that
   # "show snmpmib ifmib ifindex" can access it.
   # This entity is ephemeral: it just registers
   # the right mounts.  It mounts the same collection
   # as above, but also instantiates the necessary
   # singleton.
   _ = Tac.newInstance( 'Interface::DynamicIfIndexMounter', 'diim',
                        entityManager.cEm_ )
