#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet.IntCmdLib
import BasicCli
import CliMatcher
import FileUrl
import Intf
import IntfCli
import ShowCommand
import Url
from CliPlugin.FileCli import ( runningConfigAfterShowKw,
                                startupConfigAfterShowKw,
                                allKwMatcher, detailKwMatcher )

def _intfNames( intfs ):
   intfNames = []
   for intf in intfs:
      if isinstance( intf, Intf.IntfRange.IntfList ):
         intfNames.extend( intf.intfNames() )
      else:
         intfNames.append( intf.name )
   return intfNames

def intfFilter( mode, url, intfNames ):
   '''This is a helper function for the show running-config interface and
   show startup-config interface commands.  The commands pass in the url
   to open to get to the config along with a list of Intf or IntfRange
   objects.  We use the Arnet IntCmdLib to do the work of matching the right
   lines from the config file.'''
   try:
      f = url.open()
      try:
         (matchedLines, error) = \
            Arnet.IntCmdLib.matchIntCmds( intfNames, f.xreadlines(), subIntf=True )
         if error:
            mode.addError( error )
         else:
            for line in matchedLines:
               print line,
      finally:
         f.close()
   except EnvironmentError, e:
      mode.addError( "Error displaying %s (%s)" % (url.url, e.strerror) )

#####################################################################################
## Implementation of the show running-config interface command
##     show running-config [all] [detail] interface <intf-range> [<intf-range> ... ]
#####################################################################################

showConfigIntfKwMatcher = CliMatcher.KeywordMatcher(
   'interfaces',
   helpdesc='Filter config to include only the given interfaces' )

class ShowRunningConfigIntf( ShowCommand.ShowCliCommandClass ):
   '''Filter the running config to include only the given interfaces.'''

   syntax = ( "show running-config [ all ] [ detail ] interface { INTF }"
              " [ profile expanded ]" )
   data = {
      "running-config" : runningConfigAfterShowKw,
      "all" : allKwMatcher,
      "detail" : detailKwMatcher,
      "interface" : showConfigIntfKwMatcher,
      "INTF" : IntfCli.Intf.rangeMatcher,
      "profile": "Option about interface profiles",
      "expanded": "Show interface config with profile content expanded",
      }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      intfs = _intfNames( args[ 'INTF' ] )
      showExpanded = args.get( 'expanded', False )
      url = None
      if 'all' in args:
         url = FileUrl.localRunningConfigAll( *Url.urlArgsFromMode( mode ),
                                              showDetail=( 'detail' in args ),
                                              showProfileExpanded=showExpanded,
                                              intfFilter=intfs )
      else:
         url = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ),
                                           showProfileExpanded=showExpanded,
                                           intfFilter=intfs )
      intfFilter( mode, url, intfs )

BasicCli.addShowCommandClass( ShowRunningConfigIntf )

#####################################################################################
## Implementation of the show startup-config interface command
##     show startup-config interface <intf-range> [<intf-range> ... ]
#####################################################################################
class ShowStartupConfigIntf( ShowCommand.ShowCliCommandClass ):
   '''Filter the startup config to include only the given interfaces.'''

   syntax = "show startup-config interface { INTF }"
   data = {
      "startup-config" : startupConfigAfterShowKw,
      "interface" : showConfigIntfKwMatcher,
      "INTF" : IntfCli.Intf.rangeMatcher
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      startupConfigUrl = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
      intfFilter( mode, startupConfigUrl, _intfNames( args[ 'INTF' ] ) )

BasicCli.addShowCommandClass( ShowStartupConfigIntf )

