# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import cStringIO
import sys
import tempfile

import BasicCli
import Cli
from CliMode.IntfProfile import IntfProfileMode
import CliCommand
import CliMatcher
import CliParser
import CliSave
import CliSession
import CliToken.Cli
import ConfigMount
import ConfigSessionCommon
from CliPlugin.EthIntfCli import EthIntfModelet
import CliPlugin.SessionCli as SessionCli
import IntfCli
import IntfProfileCliLib
import LazyMount
import SessionUrlUtil
import ShowCommand
import Tac
import Url

#--------------------------------------------------------------
# CLI commands/modes:
# The "interface profile <profileName>" group-change mode:
# (config)# interface profile FOO
# (config-intf-profile-FOO)# command <COMMAND-1>
# (config-intf-profile-FOO)# command <COMMAND-2>
# (config-intf-profile-FOO)# exit
#
# "[ no | default] profile PROFILE" command in config-if mode
#--------------------------------------------------------------

profileConfig = None
ethPhyIntfConfigDir = None

tokenProfileName = CliMatcher.DynamicNameMatcher( lambda mode: profileConfig.profile,
                                                  'Interface profile name' )

def getProfileConfig( profileName ):
   return profileConfig.profile.get( profileName )

def getProfileCmds( profileName ):
   profConfig = getProfileConfig( profileName )
   if profConfig:
      return profConfig.command.values()
   return []

def getProfileListCmds( profileList ):
   """Return a list of all the commands defined in all the profiles
   in the profileList."""
   pCmds = []
   if profileList:
      for profile in profileList:
         pCmds += getProfileCmds( profile )
   return pCmds

def getModifiedProfileCmds( intf, profileNames, newProfCmds ):
   """Return the profile commands for intf, replacing the commands associated
   with @profileNames with the commands in @newProfCmds."""
   profAppCfg = profileConfig.intfToAppliedProfile.get( intf )
   pCmds = []
   if profAppCfg:
      for profile in profAppCfg.profile.values():
         if profile not in profileNames:
            pCmds += getProfileCmds( profile )
         else:
            pCmds += newProfCmds
   return pCmds

def getAppliedProfileCmds( intf ):
   """Return the profile commands that were applied to the interface."""
   profAppCfg = profileConfig.intfToAppliedProfile.get( intf )
   if profAppCfg:
      return profAppCfg.profileCmdsApplied.splitlines()
   return []

def getIntfRunningConfig( mode, intfNames, sessionName=None ):
   """Return a dictionary mapping each intf in @intfName to its configuration."""
   runningCfg = cStringIO.StringIO()
   if mode.session_.inConfigSession() or sessionName:
      if sessionName is None:
         sessionName = CliSession.currentSession( mode.entityManager )
      SessionUrlUtil.saveSessionConfig( mode.entityManager, sessionName, runningCfg,
                                        False, False,
                                        showProfileExpanded=True )
   else:
      CliSave.saveRunningConfig( mode.entityManager, runningCfg, False, False, 
                                 showProfileExpanded=True )
   # TODO: running config has intfFilter option which we should be using. 
   # We are not using it here though because session config does not have it.
   # Convert to intfFilter option when it gets added to saveSessionConfig.
   return IntfProfileCliLib.filterIntfConfig( runningCfg.getvalue(), intfNames )

def makeUrlFromFile( mode, fileObj, content ):
   fileObj.write( content )
   fileObj.flush()
   return Url.parseUrl( 'file:%s' % fileObj.name, 
                        Url.Context( *Url.urlArgsFromMode( mode ) ) )

def applyConfigToSession( mode, config, sessionName=None ):
   em = mode.entityManager
   if not mode.session_.inConfigSession():
      CliSession.enterSession( sessionName, entityManager=em )
   with ConfigMount.ConfigMountDisabler( disable=False ):
      errors = Cli.loadConfigFromFile( config.splitlines(), em,
                                       initialModeClass=SessionCli.ConfigSessionMode,
                                       disableAaa=mode.session.disableAaa_,
                                       disableGuards=mode.session.disableGuards_,
                                       startupConfig=mode.session.startupConfig_,
                                       skipConfigCheck=mode.session.skipConfigCheck_,
                                       isEapiClient=mode.session.isEapiClient_,
                                       aaaUser=mode.session_.aaaUser(),
                                       privLevel=mode.session_.privLevel_ )
   if errors:
      mode.addWarning( "Not all profile commands could be applied." )
   if sessionName:
      CliSession.exitSession( em )

def applyInitialConfig( mode, config ):
   """ Apply the config, assumed to consist of the rollback config and the
   profile commands, to a new or existing config session.
   Returns a tuple of ( whether application succeeded, sessionName to commit ). """
   applyToExistingSession = mode.session_.inConfigSession()
   if applyToExistingSession:
      applyConfigToSession( mode, config )
   else:
      try:
         with tempfile.NamedTemporaryFile( prefix="intfProf" ) as intfProfFile:
            surl = makeUrlFromFile( mode, intfProfFile, config )
            session, err = ConfigSessionCommon.configReplaceSession( mode, surl,
                                                                     replace=False )
            if err:
               if ConfigSessionCommon.LOAD_CONFIG_ERROR in err:
                  err = "Errors in profile commands."
               mode.addError( err )
               return False, None
            else:
               return True, session
      except ( IOError, OSError ) as e:
         mode.addError( e )
         return False, None
   return True, None

def applyProfileConfig( mode, intfToProfileCmds ):
   """ Given a mapping of interfaces to profile commands to apply to the interface,
   apply the profile commands to the interfaces. 
   Returns a boolean indicating success."""
   currIntfCfg = getIntfRunningConfig( mode, intfToProfileCmds )
   profileCfg = ''
   nonProfileCfg = ''
   for intfName, newProfCmds in intfToProfileCmds.items():
      oldProfileCmds = getAppliedProfileCmds( intfName )
      if oldProfileCmds == newProfCmds:
         continue
      appProfCfg = IntfProfileCliLib.AppliedProfileConfig( 
                                          intfName, 
                                          currIntfCfg.get( intfName, '' ),
                                          oldProfileCmds, 
                                          newProfCmds )
      profileCfg += appProfCfg.newProfileConfig()
      nonProfileCfg += appProfCfg.nonProfileConfig()

   if not profileCfg and not nonProfileCfg:
      return True
   profileCfg += "\nend"
   nonProfileCfg += "\nend"

   # Create a session (or use existing one), and rollback interfaces and apply
   # profile commands
   success, session = applyInitialConfig( mode, profileCfg )
   if not success:
      return False

   # Save profile commands config for updating profileCmdsApplied later
   profCmdsApplied = getIntfRunningConfig( mode, intfToProfileCmds, session )
  
   # Apply the non-profile commands (commands that are applied to interfaces
   # that are not associated with a profile)
   applyConfigToSession( mode, nonProfileCfg, session )

   # Commit if we aren't in a config session
   if session:
      err = CliSession.commitSession( mode.entityManager, sessionName=session )
      if err:
         mode.addError( err )
         return False
   
   # Update profileCmdsApplied
   for intfName in intfToProfileCmds:
      profileAppCfg = profileConfig.newIntfToAppliedProfile( intfName )
      profileAppCfg.profileCmdsApplied = profCmdsApplied.get( intfName, '' )
   
   return success

def updateProfileConfig( intfName, profileNames ):
   if not profileNames:
      del profileConfig.intfToAppliedProfile[ intfName ]
      return
   profileAppCfg = profileConfig.newIntfToAppliedProfile( intfName )
   profileAppCfg.profile.clear()
   for profile in profileNames:
      profileAppCfg.profile.enq( profile )

#---------------------------------------------------------------
# The "interface profile <profileName>" mode
#
# IntfProfileConfigMode is a Group-Change Mode that supports:
# (config)# interface profile FOO
# (config-intf-profile-FOO)# command <COMMAND-1>
# (config-intf-profile-FOO)# command <COMMAND-2>
# (config-intf-profile-FOO)# exit
#
# Group-change mode commands:
# (config-intf-profile-FOO)# abort
# (config-intf-profile-FOO)# show active
# (config-intf-profile-FOO)# show pending
# (config-intf-profile-FOO)# show diff
#---------------------------------------------------------------
class IntfProfileConfigMode( IntfProfileMode, BasicCli.ConfigModeBase ):
   name = 'interface profile configuration'
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      self.profile = Tac.newInstance( 'IntfProfile::Profile', profileName )
      self.currProfile = getProfileConfig( profileName )
      copyProfileConfig( self.currProfile, self.profile )

      IntfProfileMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def commitProfile( self ):
      profile = profileConfig.profile.newMember( self.profileName )
      copyProfileConfig( self.profile, profile )

   def getValidateCommandsConfig( self, intf, cmds ):
      config = "default interface {0}\ninterface {0}\n".format( intf )
      intfConfig = "\n".join( cmds )
      return config + intfConfig + "\nend"

   def validateCommands( self, intf, cmds ):
      if not cmds:
         return [] 
      currSession = None
      # Save session name so we can re-enter later
      if self.session_.inConfigSession():
         currSession = CliSession.currentSession( self.entityManager )
      try:
         config = self.getValidateCommandsConfig( intf, cmds )
         with tempfile.NamedTemporaryFile( prefix="intfProfValid" ) as intfProfFile:
            surl = makeUrlFromFile( self, intfProfFile, config )
            session, err = ConfigSessionCommon.configReplaceSession( self, surl,
                                                                     replace=False,
                                                                     noCommit=True )
            if currSession:
               CliSession.enterSession( currSession, 
                                        entityManager=self.entityManager )
            if err:
               if ConfigSessionCommon.LOAD_CONFIG_ERROR in err:
                  # Raising AlreadyHandledError here will stop us from exiting
                  # the mode
                  self.addError( "Errors in profile commands. Please fix the errors"
                                 " before exiting the mode, or abort." )
                  raise CliParser.AlreadyHandledError()
               else:
                  self.addError( err )
            else:
               savedCmds = getIntfRunningConfig( self, [ intf ], 
                                                 sessionName=session )
               CliSession.abortSession( self.entityManager, sessionName=session )
               savedCmds = savedCmds.get( intf, '' ).splitlines()
               self.checkNestedConfigMode( savedCmds )
               # Even after checking for nested config mode, some show up if
               # there is no config (like evpn ethernet-segment) so remove the "!"
               savedCmds = [ cmd for cmd in savedCmds if cmd != "!" ]
               self.checkMissingCommands( savedCmds )
               return savedCmds
      except ( IOError, OSError ) as e:
         self.addError( e )
      return []

   def checkNestedConfigMode( self, cmds ):
      errorMsg = ( "Nested config mode: '%s' is not supported."
                   " Please remove the commands or abort." )
      badCmd = None
      for idx, cmd in enumerate( cmds ):
         if cmd.startswith( "   " ):
            badCmd = cmds[ idx - 1 ]
            self.addError( errorMsg % badCmd )
            raise CliParser.AlreadyHandledError()
 
   def checkMissingCommands( self, canonicalCmds ):
      if self.profile:
         if len( self.profile.command ) > len( canonicalCmds ):
            self.addWarning( "Some commands were not saved in the profile. Reasons"
                             " for this occurring include configuring conflicting"
                             " commands in a profile or commands that set default"
                             " values." )

   def onExit( self ):
      if self.needsUpdate():
         appliedIntfs = {} # Mapping of intf name to list of profile names
         intfProfCmds = {} # Mapping of intf name to profile cmds to apply
         newProfCmds = self.profile.command.values()
         for intfName, config in profileConfig.intfToAppliedProfile.iteritems():
            profileList = config.profile.values()
            if self.profileName in profileList:
               appliedIntfs[ intfName ] = profileList
               intfProfCmds[ intfName ] = getModifiedProfileCmds( 
                                             intfName,
                                             [ self.profileName ],
                                             newProfCmds )
         intf = self.getTestIntf( appliedIntfs )
         if intf is None:
            # There are no interfaces to use to test profile commands
            self.addWarning( "Unable to validate profile commands." )
         else:
            pCmds = self.profile.command.values()
            canonicalCmds = self.validateCommands( intf, pCmds )
            self.saveValidatedCommands( canonicalCmds )
         success = applyProfileConfig( self, intfProfCmds )
         self.commitProfile()
         if success:
            for intfName, profList in appliedIntfs.iteritems():
               updateProfileConfig( intfName, profList )
         else:
            self.addWarning( "Failed to apply profile commands to interfaces." )
      BasicCli.ConfigModeBase.onExit( self )

   def getTestIntf( self, appliedIntfs ):
      if appliedIntfs:
         return appliedIntfs.keys()[ 0 ]
      # If we can't test the profile commands on an interface that the profile is
      # actually applied on, pick any ethernet interface that exists
      for intfName in ethPhyIntfConfigDir:
         if intfName.startswith( "Ethernet" ):
            return intfName
      return None
 
   def saveValidatedCommands( self, cmds ):
      if self.profile:
         self.profile.command.clear()
         for cmd in cmds:
            self.profile.command.enq( cmd )

   def abort( self ):
      self.profile = None
      self.session_.gotoParentMode()

   def needsUpdate( self ):
      if self.profile is None:
         return False # We aborted
      if self.currProfile is None:
         return True
      return self.currProfile.command.values() != self.profile.command.values()

   def addProfileCommand( self, command ):
      if self.profile is not None:
         if command in self.profile.command.values():
            return
         if command.startswith( "profile " ):
            self.addError( "Nested profiles are not supported." )
            return
         self.profile.command.enq( command )

   def removeProfileCommand( self, command ):
      if self.profile:
         for idx, cmd in self.profile.command.iteritems():
            if cmd == command:
               del self.profile.command[ idx ]

def copyProfileConfig( srcProfile, dstProfile ):
   if srcProfile:
      dstProfile.command.clear()
      for profileCmd in srcProfile.command.itervalues():
         dstProfile.command.enq( profileCmd )

class EnterIntfProfileCommandClass( CliCommand.CliCommandClass ):
   syntax = """interface profile <profileName>"""
   noOrDefaultSyntax = """interface profile <profileName>"""

   data = { "interface": IntfCli.interfaceKwMatcher,
            "profile": "Configure interface profiles",
            "<profileName>": tokenProfileName }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ '<profileName>' ]
      childMode = mode.childMode( IntfProfileConfigMode, profileName=profileName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ '<profileName>' ]
      lastMode = mode.session_.modeOfLastPrompt()
      if ( isinstance( lastMode, IntfProfileConfigMode ) and lastMode.profile
           and lastMode.profileName == profileName ):
         # If we are deleting ourselves, make sure we don't write it back when
         # we exit.
         lastMode.profile = None
      if profileName not in profileConfig.profile:
         return
      appliedIntfs = {} # Mapping of intf name to list of profile names
      intfProfCmds = {} # Mapping of intf name to profile cmds to apply
      for intfName, config in profileConfig.intfToAppliedProfile.iteritems():
         profileList = config.profile.values()
         if profileName in profileList:
            appliedIntfs[ intfName ] = profileList
            intfProfCmds[ intfName ] = getModifiedProfileCmds( intfName,
                                                               [ profileName ],
                                                               [] )
      success = applyProfileConfig( mode, intfProfCmds )
      del profileConfig.profile[ profileName ]
      if success:
         for intfName, profileList in appliedIntfs.iteritems():
            updateProfileConfig( intfName, profileList )
      else:
         mode.addWarning( "Failed to remove profile commands from interfaces." )

BasicCli.GlobalConfigMode.addCommandClass( EnterIntfProfileCommandClass )

class ConfigureIntfProfile( CliCommand.CliCommandClass ):
   syntax = """command COMMAND"""
   noOrDefaultSyntax = """command COMMAND"""

   data = { "command": "Add commands to the profile",
            "COMMAND": CliMatcher.StringMatcher( helpname='LINE',
                                                 helpdesc='Command string' )
   }

   @staticmethod
   def handler( mode, args ):
      command = args[ 'COMMAND' ]
      mode.addProfileCommand( command )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      command = args[ 'COMMAND' ]
      mode.removeProfileCommand( command )

IntfProfileConfigMode.addCommandClass( ConfigureIntfProfile )

class AbortIntfProfile( CliCommand.CliCommandClass ):
   syntax = """abort"""
   data = { "abort": CliToken.Cli.abortMatcher }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

IntfProfileConfigMode.addCommandClass( AbortIntfProfile )

#-----------------------------------------------------------------
# "[ no | default] profile PROFILE" command in config-if mode
# Attach an interface profile to an actual interface
#-----------------------------------------------------------------
class ApplyIntfProfile( CliCommand.CliCommandClass ):
   syntax = "profile { PROFILE }"
   noOrDefaultSyntax = "profile [ { PROFILE } ]"
   data = { "profile": "Apply an interface profile to this interface",
            "PROFILE": tokenProfileName }

   @staticmethod
   def handler( mode, args ):
      profileList = args[ 'PROFILE' ]
      newProfCmds = getProfileListCmds( profileList )
      intfName = mode.intf.name
      oldProfileCmds = getAppliedProfileCmds( intfName )
      if ( newProfCmds == oldProfileCmds or
           applyProfileConfig( mode, { intfName : newProfCmds } ) ):
         updateProfileConfig( intfName, profileList )
      else:
         mode.addError( "Unable to apply profiles to interface." )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      removeList = args.get( 'PROFILE' )
      intfName = mode.intf.name
      profAppCfg = profileConfig.intfToAppliedProfile.get( intfName )
      if not profAppCfg:
         return
     
      # Figure out which commands to apply based on the profiles we
      # want to remove
      currProfList = profAppCfg.profile.values()
      newProfCmds = []
      newProfList = []
      if removeList:
         newProfList = [ profile for profile in currProfList
                                 if profile not in removeList ]
         if newProfList == currProfList:
            # The profiles to remove are not configured, no changes necessary.
            return
         newProfCmds = getModifiedProfileCmds( intfName, removeList, [] )

      # Apply the commands
      oldProfileCmds = getAppliedProfileCmds( mode.intf.name )
      if ( oldProfileCmds != newProfCmds and
           not applyProfileConfig( mode, { intfName: newProfCmds } ) ):
         mode.addWarning( "Failed to remove profile commands from interface." )
      updateProfileConfig( intfName, newProfList )

EthIntfModelet.addCommandClass( ApplyIntfProfile )

#-----------------------------------------------------------------
# Making sure "default interface" removes the profile
#-----------------------------------------------------------------
class IntfProfileCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del profileConfig.intfToAppliedProfile[ self.intf_.name ]

#-----------------------------------------------------------------
# show active|pending|diff for IntfProfileConfigMode
#-----------------------------------------------------------------
def _showList( profile, profileName, output=None ):
   if output is None:
      output = sys.stdout
   if profile is None or not profile.command:
      return
   output.write( 'interface profile %s\n' % profileName )
   for oCmd in profile.command.values():
      output.write( '   command %s\n' % oCmd )

def _showDiff( mode ):
   # generate diff between active and pending
   import difflib
   activeOutput = cStringIO.StringIO()
   _showList( mode.currProfile, mode.profileName, output=activeOutput )
   pendingOutput = cStringIO.StringIO()
   _showList( mode.profile, mode.profileName, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print '\n'.join( list( diff ) )

class ActivePendingDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ active | diff | pending ]'
   data = {
      'active' : 'Show the list in the current running-config',
      'pending' : 'Show pending list in this session',
      'diff' : 'Show the difference between active and pending list',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      if 'active' in args:
         return _showList( mode.currProfile, mode.profileName )
      elif 'diff' in args:
         return _showDiff( mode )
      else:
         return _showList( mode.profile, mode.profileName )

IntfProfileConfigMode.addShowCommandClass( ActivePendingDiffCmd )

def Plugin( entityManager ):
   global profileConfig
   global ethPhyIntfConfigDir

   profileConfig = ConfigMount.mount( entityManager, "interface/profile/config", 
                                      "IntfProfile::Config", "w" )
   ethPhyIntfConfigDir = LazyMount.mount( entityManager,
                                          "interface/config/eth/phy/all",
                                          "Interface::AllEthPhyIntfConfigDir", "r" )
   IntfCli.Intf.registerDependentClass( IntfProfileCleaner )
