#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import Arnet
import TableOutput
import Tac
from ArnetModel import MacAddress
from CliMode.Intf import IntfMode
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import Float
from CliModel import List
from CliModel import Submodel
from CliModel import Bool
from IntfModels import Interface
import CliExtensions
import Ethernet

_hardwareTypeMap = { "portChannel" : "Port-Channel",
                     "ethernet" : "Ethernet",
                     "vlan" : "Vlan",
                     "loopback" : "Loopback",
                     "tunnel" : "Tunnel",
                     "test" : "Test",
                     "vxlan" : "Vxlan",
                     "subinterface" : "Subinterface",
                     "fabric" : "Fabric",
                     "cpu" : "Cpu",
                     "application" : "Application FPGA connector",
                     "switch": "Switch", }

prettyIntfStatusMap = { 'uninitDown' : 'uninit down',
                        'adminDown' : 'admin down',
                        'up' : 'up',
                        'down' : 'down' }

def formatInterfaceStatus( model ):
   """Helper function to render interface status.

   Arg:
     - model: An instance of a model that has at least two attributes:
       an `interfaceStatus' and a `lineProtocolStatus', such as in the
       IntfStatus model.
   """
   if model.interfaceStatus == "connected":
      intfStatus = "up"
   elif model.interfaceStatus == "disabled":
      intfStatus = "administratively down"
   else:
      intfStatus = "down"
   return ( "%s, line protocol is %s (%s)" % ( intfStatus, 
                                               model.lineProtocolStatus.lower(), 
                                               model.interfaceStatus ) )

def formatShortIntfStatus( model ):
   """
   Helper function to render a short interface status.

   Arg:
     - model: An instance of a model that has `interfaceStatus' attribute
       such as in the IntfStatus model.
   """
   if model.interfaceStatus == "uninitialized":
      return "uninit down"

   if model.interfaceStatus == "disabled" or model.interfaceStatus == "admin":
      return "admin down"

   if model.interfaceStatus == "connected":
      return "up"

   return "down"

def formatMac( macAddr ):
   return Ethernet.convertMacAddrToDisplay( macAddr.stringValue )

def intfOperStatusToEnum( st ):
   # Converts "intfOperFooBar" to "fooBar", for example.
   return re.sub( "intfOper([A-Z])", lambda m: m.group( 1 ).lower(), st )
intfOperStatuses = [ intfOperStatusToEnum( s ) for s in \
                        Tac.Type( 'Interface::IntfOperStatus' ).attributes ]
#--------------------------------------------------------------------------------
# EAPI Models
#--------------------------------------------------------------------------------   
class InterfaceAddressIp4Base( Model ):
   # All IPv4 address models that can be returned inherit from this
   pass

class InterfaceAddressIp6Base( Model ):
   # All IPv6 address models that can be returned inherit from this
   pass

class InterfaceStatus( Model ):
   # This is also present as the intfStatuses key - but it
   # useful to repeat it here, so that IntfStatus is self-
   # contained.
   name = Interface( help="Name of the interface" )
   forwardingModel = Enum( values=( "recirculation", "quietDataLink", "unauthorized",
                                    "dataLink", "bridged", "routed" ),
                            help="Recirculation -- the interface recirculates "
                            "packets. Protocols do not run on this port, "
                            "QuietDataLink -- the interface forwards packets "
                            "but does not participate in bridging or routing and "
                            "protocols do not run on this interface, " 
                            "Unauthorized -- the interface does not participate "
                            "in bridging or routing and only EAPOL runs on this "
                            "interface, "
                            "DataLink -- the interface forwards packets "
                            "given to it but is not participating in bridging or "
                            "routing operations, "
                            "Routed -- the interface can route packets, "
                            "Bridged -- the interface can bridge packets but not "
                            "route them" )
   lineProtocolStatus = Enum( help="Line protocol status",
                              values=intfOperStatuses )
   interfaceStatus = Enum( values=( 'disabled', 'connected', 'notconnect', 'unknown',
                                    'admin', 'errdisabled', 'uninitialized',
                                    'inactive', 'maint-down' ),
                           help="Connection status of the interface" )
   maintenanceEnterTime = Float( help="Time at which interface went under "
                               "maintenance", optional=True )
   hardware = Enum( values=_hardwareTypeMap.keys(), help="Hardware Description" )

   # Should not be a list but defined as such for historical reasons.
   # Original intent for the list was to contain multiple instances
   # of multiple address classes. But JSON does not return class types
   # for list elements which makes it difficult for applications to
   # process said list. Adding a class type enum to the base class
   # is not backwards compatible. Hence the list has always been in
   # practice a single instance of an IPv4 address submodel.
   interfaceAddress = List( valueType=InterfaceAddressIp4Base,
                            help="IPv4 addresses for this interface" )

   interfaceAddressIp6 = Submodel( valueType=InterfaceAddressIp6Base,
                                   help="IPv6 addresses for this interface",
                                   optional=True )
   
   # Loopback interfaces don't have MAC addresses.
   physicalAddress = MacAddress( help="Physical Layer Address", optional=True )
   burnedInAddress = MacAddress( help="Burned-in Address", optional=True )
   description = Str( help="Configured interface description" )
   bandwidth = Int( help="Bandwidth (bit)" )
   mtu = Int( help="MTU (bytes)" )
   l3MtuConfigured = Bool( help="Per-interface MTU is set under interface " \
                                "configuration mode", default=False )
   l2Mru = Int( help="L2 MRU (bytes)", default=0 )
   lastStatusChangeTimestamp = Float( help="Timestamp of last status change", 
                                      optional=True )
   
   def renderHeader( self ):
      # pylint: disable-msg=E1101
      print "%s is %s" % ( self.name.stringValue, formatInterfaceStatus( self ) )
      
   def renderHardware( self ):
      output = "  Hardware is %s" % _hardwareTypeMap[ self.hardware ]
      if self.physicalAddress is not None:
         output += ", address is %s" % formatMac( self.physicalAddress )
      if self.burnedInAddress is not None:
         output += " (bia %s)" %  formatMac( self.burnedInAddress )
      print output
      
      if self.description:
         print "  Description: %s" % self.description         
                
   def renderMtuMruAndBw( self ):
      tokens = []
      # Process MTU
      if self.forwardingModel == "routed":
         mtuToken = "IP MTU %d bytes" % self.mtu
         mtuToken += " (default)"  if not self.l3MtuConfigured else ""
         tokens.append( mtuToken )
      elif self.mtu is not None:
         tokens.append( "Ethernet MTU %d bytes" % self.mtu )
      else:
         assert not self.bandwidth, repr( self.bandwidth )

      # Process L2 MRU
      if self.l2Mru:
         tokens.append( "Ethernet MRU %d bytes" % self.l2Mru )

      # Process Bandwidth
      if self.bandwidth:
         tokens.append( "BW %d kbit" % ( self.bandwidth / 1000 ) )

      if tokens:
         output = "  "
         output += ", ".join( tokens )
         print output

   def timeString( self, elapsed, output, units ):
      assert isinstance( elapsed, int )
      if units:
         _ , unit = units[ -1 ]
         assert ( unit == 1 ), "Last unit must have 1 as a unit value"
      if ( elapsed > 0 ) and units:
         localName, localUnits = units[0] 
         value = elapsed % localUnits if len( units ) > 1 else elapsed
         if( value > 0 ):
            output = ( "%d %s"%( value, localName + ( "s" if value > 1 else "" ) )
                        + ( ( ", " + output ) if output else "" ) )
         return( self.timeString( elapsed//localUnits, 
                 output, units[1:] ) )
      else: 
         return( output ) 
      
   def renderUptime( self ):
      if self.lastStatusChangeTimestamp is None:
         return
      
      timeDelta = int( Tac.utcNow() - self.lastStatusChangeTimestamp )
      uptimeStatus = "  Up " if self.lineProtocolStatus == "up" else "  Down "
      uptimeString = self.timeString( timeDelta, "",
                                      [ ( "second", 60 ), ( "minute", 60 ),
                                        ( "hour", 24 ), ( "day", 365 ),
                                        ( "year", 1 ) ] )
      print uptimeStatus + uptimeString
      
   def renderMaintEnterTime( self ):
      if self.maintenanceEnterTime is None:
         return

      timeDelta = int( Tac.utcNow() - self.maintenanceEnterTime )
      maintenanceEnterTimeString = self.timeString( timeDelta, "",
                                                    [ ( "second", 60 ),
                                                      ( "minute", 60 ),
                                                      ( "hour", 24 ),
                                                      ( "day", 365 ),
                                                      ( "year", 1 ) ] )
      print "  Under maintenance for %s" % maintenanceEnterTimeString

   def renderInterfaceAddress( self ):
      for address in self.interfaceAddress:
         address.render()
      if self.interfaceAddressIp6:
         self.interfaceAddressIp6.render()

   def render( self ):
      self.renderHeader()
      self.renderHardware()
      self.renderInterfaceAddress()
      self.renderMtuMruAndBw()
      self.renderUptime()
      self.renderMaintEnterTime()

# Container for all interfaces
class InterfaceStatuses( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfaceStatus,
                      help="Interface Status" )

   def render( self ):
      for key in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ key ].render()

class InterfaceConfigSanity( InterfaceStatus ):
   
   # Default behavior; override Intf.getConfigSanityModel to provide an
   # InterfaceStatus model
   def render( self ):
      print ( 'Configuration sanity checking for interface %s is not '
         'supported.' ) % str( self.name )

class InterfaceCountersBase( Model ):
   _name = Interface( help="Name of the interface" )
   
   _l3Counters = Bool( help="True if counters are for Layer 3 interface",
                        default=False )
   _tunnelIntfCounters = Bool( help="True if counters are for static "
                                    "Tunnel interface", default=False )

   # some interfaces ( SubIntfs for example ) might not support
   # Ingress/Egress/BUM counters
   _ingressCounters = Bool( help="True if interface supports ingress counters",
                            default=True )
   _egressCounters = Bool( help="True if interface supports egress counters",
                           default=True )

   def l3Counters( self ):
      return self._l3Counters

   def tunnelIntfCounters( self ):
      return self._tunnelIntfCounters

   def renderHeader( self, direction='in' ):
      pass

   def renderIncoming( self ):
      pass

   def renderOutgoing( self ):
      pass 

   def counterSupported( self, direction ):
      return self._ingressCounters if direction == 'in' else self._egressCounters

# Interface Counters
class InterfaceCounters( InterfaceCountersBase ):

   inOctets = Int( help="Input octets" )
   inUcastPkts = Int( help="Input unicast packets" )
   inMulticastPkts = Int( help="Input multicast packets" )
   inBroadcastPkts = Int( help="Input broadcast packets  " )
   inDiscards = Int( help="Input packets with errors  ", optional=True )
   inTotalPkts = Int( help="Input packets", optional=True )

   outOctets = Int( help="Output octets" )
   outUcastPkts = Int( help="Output unicast packets" )
   outMulticastPkts = Int( help="Output multicast packets" )
   outBroadcastPkts = Int( help="Output broadcast packets" )
   outDiscards = Int( help="Output packets with errors  ", optional=True )

   lastUpdateTimestamp = Float( help="Time of last update", optional=True )

   # some interfaces ( L3 interfaces on T2 for example ) might only
   # support unicast and multicast counters, but not broadcast
   _umCounters = Bool( 
      help="True if interface supports Unicast and Multicast counters",
      default=False )
   _bumCounters = Bool( help="True if interface supports BUM counters",
                        default=True )

   def printRow( self, port, totalOctets, ucastPkts, mcastPkts=None, 
                 bcastPkts=None ):
      def checkValue( value ):
         return value if value is not None else 'n/a'
      
      if self._bumCounters:
         counterDisplayFormat = "%-20s %18s %15s %15s %15s"
         print counterDisplayFormat % ( port, checkValue( totalOctets ),
                                        checkValue( ucastPkts ),
                                        checkValue( mcastPkts ),
                                        checkValue( bcastPkts ) )
      elif self._umCounters:
         counterDisplayFormat = "%-20s %18s %15s %15s"
         print counterDisplayFormat % ( port, checkValue( totalOctets ),
                                        checkValue( ucastPkts ),
                                        checkValue( mcastPkts ) )
      else:
         counterDisplayFormat = "%-20s %18s %15s"
         print counterDisplayFormat % ( port, checkValue( totalOctets ),
                                        checkValue( ucastPkts ) )
   
   def renderHeader( self, direction='in' ):
      if self._l3Counters:
         heading = "L3 Interface"
      elif self._tunnelIntfCounters:
         heading = "Tunnel"
      else:
         heading = "Port"
      if direction == 'in':
         if not self._ingressCounters:
            return
         if self._bumCounters:
            self.printRow( heading, "InOctets", "InUcastPkts", "InMcastPkts",
                           "InBcastPkts" )
         elif self._umCounters:
            self.printRow( heading, "InOctets", "InUcastPkts", "InMcastPkts" )
         else:
            self.printRow( heading, "InOctets", "InPkts" )
      elif direction == 'out':
         if not self._egressCounters:
            return
         if self._bumCounters:
            self.printRow( heading, "OutOctets", "OutUcastPkts", "OutMcastPkts",
                           "OutBcastPkts" )
         elif self._umCounters:
            self.printRow( heading, "OutOctets", "OutUcastPkts", "OutMcastPkts" )
         else:
            self.printRow( heading, "OutOctets", "OutPkts" )
      else:
         assert False, "Unhandled direction"

   def renderIncoming( self ):
      if not self._ingressCounters:
         return
      if self._bumCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.inOctets, 
                        self.inUcastPkts, self.inMulticastPkts,
                        self.inBroadcastPkts )
      elif self._umCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.inOctets, 
                        self.inUcastPkts, self.inMulticastPkts )
      else:
         self.printRow( IntfMode.getShortname( self._name ), self.inOctets,
                        self.inUcastPkts )

   def renderOutgoing( self ):
      if not self._egressCounters:
         return
      if self._bumCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.outOctets,
                        self.outUcastPkts, self.outMulticastPkts,
                        self.outBroadcastPkts )
      elif self._umCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.outOctets,
                        self.outUcastPkts, self.outMulticastPkts )
      else:
         self.printRow( IntfMode.getShortname( self._name ), self.outOctets,
                        self.outUcastPkts )

# Container base class
class InterfacesCountersBase( Model ):
   _outputType = Enum( values=( 'all', 'incoming', 'outgoing' ),
                       default='all', help="Which counters to show" )

   def getInterfaces( self ):
      raise NotImplementedError

   def getInterfaceCounters( self, intf ):
      raise NotImplementedError

   def l3Counters( self, intf ):
      return False

   def tunnelIntfCounters( self, intf ):
      return False

   def partition( self, intfs ):
      # partition intfs into list of intfs that
      # have counters rendered separately
      intfList = []
      subIntfList = []
      l3IntfList = []
      tunnelIntfList = []
      for intf in intfs:
         if self.l3Counters( intf ):
            # subIntf will be accounted here if they support only l3 counters
            l3IntfList.append( intf )
         elif '.' in intf: 
            subIntfList.append( intf )
         elif self.tunnelIntfCounters( intf ):
            tunnelIntfList.append( intf )
         else:
            intfList.append( intf )
      return [ intfList, subIntfList, l3IntfList, tunnelIntfList ]

   def renderWithHeader( self, intfList, direction='in' ):
      prevValueType = None
      for key in intfList:
         intfCounter = self.getInterfaceCounters( key )
         if not intfCounter.counterSupported( direction ):
            continue
         valueType = intfCounter.__class__
         if valueType != prevValueType:
            prevValueType = valueType
            intfCounter.renderHeader( direction )
         if direction == 'in':
            intfCounter.renderIncoming()
         else:
            intfCounter.renderOutgoing()

   def render( self ):
      intfs = self.getInterfaces()
      if not intfs:
         return

      sortedIntfs = Arnet.sortIntf( intfs )
      partitionedIntfs = self.partition( sortedIntfs )
      for intfList in partitionedIntfs:
         if intfList and partitionedIntfs.index( intfList ) != 0:
            # print empty line for separation only if intfList is not empty
            print

         if self._outputType != 'outgoing':
            self.renderWithHeader( intfList )

         if intfList and self._outputType == 'all':
            print

         if self._outputType != 'incoming':
            self.renderWithHeader( intfList, direction='out' )

# Container class
class InterfacesCounters( InterfacesCountersBase ):
   interfaces = Dict( keyType=Interface, valueType=InterfaceCountersBase,
                      help="Interface Counters" )

   def getInterfaces( self ):
      return self.interfaces.keys()

   def getInterfaceCounters( self, intf ):
      return self.interfaces[ intf ]

   def l3Counters( self, intf ):
      return self.interfaces[ intf ].l3Counters()

   def tunnelIntfCounters( self, intf ):
      return self.interfaces[ intf ].tunnelIntfCounters()

class InterfaceCountersRateBase( Model ):
   _name = Interface( help="Name of the interface" )

# Interface rate counters
class InterfaceCountersRate( InterfaceCountersRateBase ):
   description = Str( help="Port description" )
   interval = Int( help="Interval in seconds" )

   inBpsRate = Float( help="Input bps rate" )
   inPktsRate = Float( help="Input packets rate percentage", optional=True )
   inPpsRate = Float( help="Input pps rate" )

   outBpsRate = Float( help="Output bps rate" )
   outPktsRate = Float( help="Output packets rate percentage", optional=True )
   outPpsRate = Float( help="Output Kpps rate" )

   lastUpdateTimestamp = Float( help="Time of last update", optional=True )

   def renderHeader( self ):
      print "%-9s %-11.11s %5s  %8s %6s %8s  %8s %6s %8s" % ( 'Port', 'Name',
                                               'Intvl', 'In Mbps', '%', 'In Kpps',
                                               'Out Mbps', '%', 'Out Kpps' )

   def renderRates( self ):
      formatStr = "%-9s %-11.11s %2d:%02d  %8.1f %6s %8d  %8.1f %6s %8d"
      minutes = self.interval / 60
      seconds = self.interval % 60
      inMbpsRate = self.inBpsRate / 1000.0 / 1000.0
      outMbpsRate = self.outBpsRate / 1000.0 / 1000.0
      inRatePctStr = ( "%.1f%%" % self.inPktsRate 
         if self.inPktsRate is not None else '-' )
      outRatePctStr = ( "%.1f%%" % self.outPktsRate
         if self.outPktsRate is not None else '-' )
      print formatStr % ( IntfMode.getShortname( self._name ),
                       self.description, minutes, seconds, inMbpsRate,
                       inRatePctStr, self.inPpsRate / 1000, 
                       outMbpsRate, outRatePctStr, self.outPpsRate / 1000 )
   
class InterfaceCountersRates( Model ):

   interfaces = Dict( keyType=Interface, valueType=InterfaceCountersRateBase,
                 help="Mapping between an interface rate counters and model" )

   def render( self ):
      if not self.interfaces:
         return

      prevValueType = None
      for key in Arnet.sortIntf( self.interfaces ):
         valueType = self.interfaces[ key ].__class__
         if valueType != prevValueType:
            self.interfaces[ key ].renderHeader()
            prevValueType = valueType
         self.interfaces[ key ].renderRates()

# Interface Counters Discards
class InterfaceCounterDiscards( Model ):
   _name = Interface( help="Name of the interface" )
   inDiscards = Int( help="Input packets discarded" )
   outDiscards = Int( help="Output packets discarded", optional=True )

# Container class
class InterfacesCountersDiscards( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfaceCounterDiscards,
                      help="Interface Discard Counters" )

   inDiscardsTotal = Int( help="Total input packets discarded", default=0 )
   outDiscardsTotal = Int( help="Total output packets discarded", optional=True )

   def render( self ):
      if len( self.interfaces ) == 0:
         return
      
      # Set up the column formatting objects.
      fl = TableOutput.Format( justify="left" )
      fl.noPadLeftIs( True )
      fr = TableOutput.Format( justify="right" )

      headings = ( "Port", "InDiscards", "OutDiscards" )

      table = TableOutput.createTable( headings )
      table.formatColumns( fl, fr, fr )

      for key in Arnet.sortIntf( self.interfaces ):
         if self.interfaces[ key ].outDiscards is not None:
            outDiscards = self.interfaces[ key ].outDiscards
         else:
            outDiscards = "N/A"
         table.newRow( IntfMode.getShortname( key ),
                       self.interfaces[ key ].inDiscards,
                       outDiscards )

      # If discard statistics are shown for two or more interfaces, then
      # also display a row showing summary statistics for all interfaces.
      if len( self.interfaces ) >= 2:
         row = [ "Totals", self.inDiscardsTotal ]

         # Only include the "outDiscards" sum if there is at least one
         # interface that supports it. Otherwise, show "N/A".
         if self.outDiscardsTotal is not None:
            row.append( self.outDiscardsTotal )
         else:
            row.append( "N/A" )

         sep = "---------"
         table.newRow( sep, sep, sep )
         table.newRow( *row )

      print table.output()
      
# Interface Traffic-class Counters
class InterfaceTrafficClassCounters( Model ):
   inOctets = Int( help="Input octets" )
   inPkts = Int( help="Input packets" )
   inUcastPkts = Int( help="Input unicast packets", optional=True )
   inMcastPkts = Int( help="Input multicast packets", optional=True )
   inBcastPkts = Int( help="Input broadcast packets", optional=True )

# Container class for traffic classes
class InterfaceTrafficClassesCounters( Model ):
   _name = Interface( help="Name of the interface" )
   _bumCounters = Bool( help="True if interface supports BUM counters",
                        default=True )
   _egressCounters = Bool( help="True if interface supports egress counters",
                           default=False )
   trafficClasses = Dict( keyType=int, valueType=InterfaceTrafficClassCounters,
         help="A mapping between a traffic class (in range 0 to 7)"
              " and its counters" )

   def formatter( self ):
      return '%-26s %7s %18s %18s %18s %18s' if self._bumCounters else \
             '%-26s %7s %18s %18s'

   def headerFormatter( self ):
      return self.formatter() + '\n%-26s %-7s\n' + ( '-' * 72 )

   def renderHeader( self, direction='in' ):
      if direction == 'in':
         print 'Traffic-Class Counters'
         if self._bumCounters:
            print self.headerFormatter() % (
             'Ingress', 'Traffic', 'InOctets', 'InUcastPkts',
                                   'InMcastPkts', 'InBcastPkts',
             'Port', 'Class' )
         else:
            print self.headerFormatter() % (
             'Ingress', 'Traffic', 'Pkts', 'Octets',
             'Port', 'Class' )
      elif self._egressCounters:
         # Implement when we support egress counters
         pass

   def renderIncoming( self ):
      for tc, tcCounter in sorted( self.trafficClasses.iteritems() ):
         tcStr = 'TC%d' % tc
         # pylint: disable-msg=E1101
         if self._bumCounters:
            print self.formatter() % (
             self._name.shortName, tcStr, tcCounter.inOctets, tcCounter.inUcastPkts,
             tcCounter.inMcastPkts, tcCounter.inBcastPkts )
         else:
            print self.formatter() % (
             self._name.shortName, tcStr,
             tcCounter.inPkts, tcCounter.inOctets )
         # pylint: enable-msg=E1101

   def renderOutgoing( self ):
      # Implement when we support egress counters
      pass

   def counterSupported( self, direction ):
      return True if direction == 'in' else self._egressCounters

# Container class
class InterfacesTrafficClassesCounters( InterfacesCountersBase ):
   interfaces = Dict( keyType=Interface, valueType=InterfaceTrafficClassesCounters,
         help="A mapping between an interface and its traffic class counters" )

   def getInterfaces( self ):
      return self.interfaces.keys()

   def getInterfaceCounters( self, intf ):
      return self.interfaces[ intf ]

class InterfacesDescriptions( Model ):
   class InterfaceDescription( Model ):
      description = InterfaceStatus.description
      lineProtocolStatus = InterfaceStatus.lineProtocolStatus
      interfaceStatus = Enum( values=prettyIntfStatusMap.keys(),
                              help='Connection status of the interface' )
      
   interfaceDescriptions = Dict( keyType=Interface, valueType=InterfaceDescription,
                                 help='Mapping between an interface and its '
                                      'description' )
   
   def render( self ):
      if not self.interfaceDescriptions:
         return 
      
      fmt = '%-30s %-14s %-18s %s'
      print fmt % ( 'Interface', 'Status', 'Protocol', 'Description' )
      for intf in Arnet.sortIntf( self.interfaceDescriptions ):
         interfaceDescription = self.interfaceDescriptions[ intf ]
         print fmt % ( IntfMode.getShortname( intf ), 
                       prettyIntfStatusMap[ interfaceDescription.interfaceStatus ],
                       interfaceDescription.lineProtocolStatus.lower(), 
                       interfaceDescription.description ) 

# Error counters for interfaces   
class ErrorCounters( Model ):
   fcsErrors = Int( help="FCS errors" )
   alignmentErrors = Int( help="Alignment errors" )
   symbolErrors = Int( help="Symbol Errors" )
   inErrors = Int( help="Input errors" )
   frameTooShorts = Int( help="Frame too short errors" )
   frameTooLongs = Int( help="Frame too long errors" )
   outErrors = Int( help="Output errors" )

#Class that holds error counters for given interfaces
class InterfacesErrorCounters( Model ):

   interfaceErrorCounters = Dict( keyType=Interface, valueType=ErrorCounters,
                                  help="Mapping between an interface and error "
                                       "counters" )

   def render( self ):
      if not self.interfaceErrorCounters:
         return
      fmt = '%-10s %11s %8s %8s %8s %8s %8s %8s'
      print fmt % ( 'Port', 'FCS' , 'Align', 'Symbol', 'Rx',
                    'Runts', 'Giants', 'Tx' )
      for intf in Arnet.sortIntf( self.interfaceErrorCounters ):
         x = self.interfaceErrorCounters[ intf ]
         print fmt % ( IntfMode.getShortname( intf ), x.fcsErrors, 
                          x.alignmentErrors, x.symbolErrors, x.inErrors,
                          x.frameTooShorts, x.frameTooLongs, x.outErrors )

# Half-duplex error counters for interfaces
class HalfDuplexErrorCounters( Model ):
   singleCollisionFrames = Int( help="Single collision frames" )
   multipleCollisionFrames = Int( help="Multiple collision frames" )
   deferredTransmissions  = Int( help="Deferred transmissions" )
   lateCollisions = Int( help="Late collisions" )
   excessiveCollisions = Int( help="Excessive collisions" )

#Class that holds half-duplex error counters for given interfaces
class InterfacesHalfDuplexErrorCounters( Model ):

   intfHalfDuplexErrorCounters = Dict( keyType=Interface,
                                       valueType=HalfDuplexErrorCounters,
                                       help="Mapping between an interface and "
                                            "half-duplex error counters" )

   def render( self ):
      if not self.intfHalfDuplexErrorCounters:
         return
      # Set up the column formatting objects.
      fl = TableOutput.Format( justify="left" )
      fr = TableOutput.Format( justify="right" )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr.padLimitIs( True )

      headings = ( "\nInterface", "Single   \nCollision", "Multiple \nCollision",
                   "Late     \nCollision", "Excessive\nCollision",
                   "Deferred    \nTransmission", )

      table = TableOutput.createTable( headings )
      table.formatColumns( fl, fr, fr, fr, fr )
      for intf in Arnet.sortIntf( self.intfHalfDuplexErrorCounters ):
         counter = self.intfHalfDuplexErrorCounters[ intf ]
         table.newRow( IntfMode.getShortname( intf ),
                       counter.singleCollisionFrames,
                       counter.multipleCollisionFrames,
                       counter.lateCollisions,
                       counter.excessiveCollisions,
                       counter.deferredTransmissions )
      print( table.output() )

# Acl drop counters for interfaces
class InterfacesCounterAclDrop( Model ):
   _name = Interface( help="Name of the interface" )
   aclDrops = Int( help="Packets denied by an ACL" )

#Class that holds acl drop counters for given interfaces
class InterfacesCountersAclDrop( Model ):

   interfaces = Dict( keyType=Interface,
                      valueType=InterfacesCounterAclDrop,
                      help="Mapping between an interface and ACL drop counters" )

   def render( self ):
      if len( self.interfaces ) == 0:
         return

      # Set up the column formatting objects.
      fl = TableOutput.Format( justify="left" )
      fl.noPadLeftIs( True )
      fr = TableOutput.Format( justify="right" )

      headings = ( "Port", "InAclDrops" )

      table = TableOutput.createTable( headings )
      table.formatColumns( fl, fr )
      for intf in Arnet.sortIntf( self.interfaces ):
         x = self.interfaces[ intf ]
         table.newRow( IntfMode.getShortname( intf ),
                       x.aclDrops )
      print table.output()

interfaceInfoHook = CliExtensions.CliHook()
