# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliPlugin.MaintenanceCliLib import (
   defaultProfile,
   linecardBuiltinGroupPrefix,
   maintenanceKwMatcher,
   Profile,
   profileMatcher,
   profileNameMatcher,
   reservedProfileName,
)
from CliPlugin.MaintenanceGroupCli import MaintenanceGroup
from CliPlugin.MaintenanceMode import MaintenanceConfigMode
from CliPlugin.MaintenanceModels import profilesCleanupHook
from CliPlugin.IntfGroupCli import IntfBuiltinGroup
import ConfigMount
from IntfGroupLib import IntfGroupConfigMode, IntfBuiltinGroupConfigMode
from IntfMaintenanceProfileConfigMode import IntfMaintenanceProfileConfigMode
import Tac

# pylint: disable-msg=E1103

globalIntfProfileConfigDir = None
globalDefaultIntfProfile = None

matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Configure interface' )
intfProfileNameMatcher = profileNameMatcher( lambda mode:
                                          globalIntfProfileConfigDir.config.keys() )

intfProfileType = "profileTypeInterface"

#--------------------------------------------------------------------------------
# [ no | default ] maintenance profile interface PROFILE_NAME
#--------------------------------------------------------------------------------
class MaintenanceProfileIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'maintenance profile interface PROFILE_NAME'
   noOrDefaultSyntax = 'maintenance profile interface [ PROFILE_NAME ]'
   data = {
      'maintenance' : maintenanceKwMatcher,
      'profile' : profileMatcher,
      'interface' : matcherInterface,
      'PROFILE_NAME' : intfProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName.lower() == reservedProfileName:
         mode.addError( "The profile name '%s' is reserved." % profileName )
         return
      profileKey = Tac.Value( "Maintenance::Profile::ProfileKey", 
                              type = intfProfileType,
                              name = profileName )
      if isinstance( mode.group(), list ):
         for _group in mode.group():
            assert( isinstance( _group, IntfBuiltinGroup ) and
                    _group.name().startswith( linecardBuiltinGroupPrefix ) )
            maintenanceGroup = MaintenanceGroup( _group.key() )
            maintenanceGroup.addProfile( profileKey )
      else:
         maintenanceGroup = MaintenanceGroup( mode.group().key() )
         maintenanceGroup.addProfile( profileKey )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args.get( 'PROFILE_NAME' )
      if isinstance( mode.group(), list ):
         for _group in mode.group():
            assert( isinstance( _group, IntfBuiltinGroup ) and
                    _group.name().startswith( linecardBuiltinGroupPrefix ) )
            maintenanceGroup = MaintenanceGroup( _group.key() )
            maintenanceGroup.remProfile( intfProfileType, profileName )
      else:
         maintenanceGroup = MaintenanceGroup( mode.group().key() )
         maintenanceGroup.remProfile( intfProfileType, profileName )

IntfGroupConfigMode.addCommandClass( MaintenanceProfileIntfCmd )
IntfBuiltinGroupConfigMode.addCommandClass( MaintenanceProfileIntfCmd )

#
# IntfProfile class: Holds cli state for each configured Interface profile
#
class IntfProfile( Profile ):
   def __init__( self, profileName ):
      self.name_ = profileName
      Profile.__init__( self )

   def addProfile( self, profileName ):
      if profileName not in globalIntfProfileConfigDir.config:
         profileConfig = globalIntfProfileConfigDir.newConfig( profileName )
         Profile.addProfile( self, profileConfig.key )

   def remProfile( self, profileName ):
      profileConfig = globalIntfProfileConfigDir.config.get( profileName )
      if profileConfig:
         Profile.remProfile( self, profileConfig.key )
         del globalIntfProfileConfigDir.config[ profileName ]

   def name( self ):
      return self.name_

#--------------------------------------------------------------------------------
# [ no | default ] profile interface PROFILE_NAME [ default ]
#--------------------------------------------------------------------------------
def _delIntfProfileConfig( mode, profileName ):
   intfProfile = IntfProfile( profileName )
   intfProfile.remProfile( profileName )

class ProfileInterfaceProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'profile interface PROFILE_NAME [ default ]'
   noOrDefaultSyntax = syntax
   data = {
      'profile' : profileMatcher,
      'interface' : matcherInterface,
      'PROFILE_NAME' : intfProfileNameMatcher,
      'default' : 'Default profile',
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName.lower() == reservedProfileName:
         mode.addError( "The profile name '%s' is reserved." % profileName )
         return

      if 'default' in args:
         globalDefaultIntfProfile.profileName = profileName
      else:
         intfProfile = IntfProfile( profileName )
         intfProfile.addProfile( profileName )
         childMode = mode.childMode( IntfMaintenanceProfileConfigMode,
                                     intfProfile=intfProfile )
         mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if 'default' in args:
         if profileName == globalDefaultIntfProfile.profileName:
            globalDefaultIntfProfile.profileName = defaultProfile
      else:
         _delIntfProfileConfig( mode, profileName )

MaintenanceConfigMode.addCommandClass( ProfileInterfaceProfilenameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] rate-monitoring ( ( load-interval LOAD_INTERVAL ) |
#                                    ( threshold THRESHOLD ) )
#--------------------------------------------------------------------------------
# instantiate temp LoadInterval object so that we can get the min/max values
LoadInterval = Tac.Type( 'IntfMaintenanceProfile::Config' )

class RateMonitoringCmd( CliCommand.CliCommandClass ):
   syntax = ( 'rate-monitoring ( ( load-interval LOAD_INTERVAL ) | '
                                '( threshold THRESHOLD ) )' )
   noOrDefaultSyntax = ( 'rate-monitoring [ load-interval | threshold ]' )
   data = {
      'rate-monitoring' : 'Traffic Rate Monitoring Configuration',
      'load-interval' : 'Rate Monitoring period in secs',
      'LOAD_INTERVAL' : CliMatcher.IntegerMatcher( LoadInterval.minLoadInterval,
         LoadInterval.maxLoadInterval, helpdesc='Number of seconds' ),
      'threshold' : 'Rate Monitoring threshold in kbps',
      'THRESHOLD' : CliMatcher.IntegerMatcher( 1, 0xffffffff,
         helpdesc='Threshold in kbps' ),
   }

   @staticmethod
   def handler( mode, args ):
      profileName = mode.intfProfile_.name_
      profileConfig = globalIntfProfileConfigDir.config.get( profileName )
      if not profileConfig:
         return

      if 'LOAD_INTERVAL' in args:
         profileConfig.rateMonLoadInterval = args[ 'LOAD_INTERVAL' ]
      elif 'THRESHOLD' in args:
         profileConfig.rateMonThresholdInKbps = args[ 'THRESHOLD' ]
      else:
         assert False, "rateMonOp has to be either loadInterval or threshold"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = mode.intfProfile_.name_
      profileConfig = globalIntfProfileConfigDir.config.get( profileName )
      if not profileConfig:
         return

      if 'load-interval' in args:
         profileConfig.rateMonLoadInterval = profileConfig.defaultLoadInterval
      elif 'threshold' in args:
         profileConfig.rateMonThresholdInKbps = profileConfig.defaultRateMonThreshold
      else:
         # Set both load-interval and threshold to defaults
         profileConfig.rateMonLoadInterval = profileConfig.defaultLoadInterval
         profileConfig.rateMonThresholdInKbps = profileConfig.defaultRateMonThreshold

IntfMaintenanceProfileConfigMode.addCommandClass( RateMonitoringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown max-delay MAX_DELAY
#--------------------------------------------------------------------------------
class IntfShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown max-delay MAX_DELAY'
   noOrDefaultSyntax = 'shutdown ...'
   data = {
      'shutdown' : 'Administratively shut off the interface',
      'max-delay' : 'Maximum duration to wait for traffic to drain before shutdown',
      'MAX_DELAY' : CliMatcher.IntegerMatcher( 0, 0xffffffff,
         helpdesc='Duration in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      profileName = mode.intfProfile_.name_
      profileConfig = globalIntfProfileConfigDir.config.get( profileName )
      if not profileConfig:
         return
      profileConfig.shutdown = True
      profileConfig.shutdownMaxDelay = args[ 'MAX_DELAY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = mode.intfProfile_.name_
      profileConfig = globalIntfProfileConfigDir.config.get( profileName )
      if not profileConfig:
         return
      profileConfig.shutdown = False
      profileConfig.shutdownMaxDelay = profileConfig.defaultShutdownMaxDelay

IntfMaintenanceProfileConfigMode.addCommandClass( IntfShutdownCmd )

def intfProfileCleanup( mode ):
   for profileName in globalIntfProfileConfigDir.config:
      if profileName != defaultProfile:
         _delIntfProfileConfig( mode, profileName )
   globalDefaultIntfProfile.profileName = defaultProfile

def Plugin( entityManager ):
   global globalIntfProfileConfigDir
   global globalDefaultIntfProfile
   profilesCleanupHook.addExtension( intfProfileCleanup )
   globalIntfProfileConfigDir = ConfigMount.mount( entityManager,
      'maintenance/profile/config/interface',
      'IntfMaintenanceProfile::ConfigDir', 'w' )
   globalDefaultIntfProfile = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/default/interface',
      'Maintenance::Profile::DefaultProfile', 'w' )
