# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

import IntfInteractionsReferenceLib as IxnRef
import Tracing
from TypeFuture import TacLazyType

# The generated default trace-handle is something like
# CliPlugin::IntfInteractionReferenceMatcher, so I'm changing that.
traceHandle = Tracing.Handle( "IntfInteractionReferenceMatcher" )
t0 = traceHandle.trace0

XcvrType = TacLazyType( "Xcvr::XcvrType" )

class InteractionReferenceId( object ):
   """
   Bundles a set of criteria that map to a particular
   IntfInteractionReference-based class.

   Members (optional unless otherwise specified)
   ---------------------------------------------
   phys: set( string ) (mandatory*)
      Set of strings describing the phys between the front-panel and the
      switch-chip, inclusive. Strings come from descriptions in L1Topo.
   xcvrSlot: string (mandatory*)
      Basically an alias for xcvrStatus.xcvrType
   productRe: string
      Match against entMib model name
   portRange: iterable of ints
      Ints correspond to front-panel port numbers/transceiver slot IDs. This
      criteria combined with the previous leads to the most explicit matching.
      Should be reserved for the special-est unique-est of cases.
   
   * 'phys' and 'xcvrSlot' will be "mandatory" for products that support L1Topo
     ("mandatory" in the sense that we won't assert out, but you won't get the
     match you want). For products that don't, matching will be done differently.
   """
   def __init__( self, phys=None, xcvrSlot="", productRe="",
                 portRange=range( 0 ) ):
      self.phys = phys or set()
      self.xcvrSlot = xcvrSlot
      self.productRe = productRe
      self.portRange = portRange

   def __repr__( self ):
      argString = ""
      if self.phys:
         argString += "phys=%s, " % self.phys
      if self.xcvrSlot:
         argString += "xcvrSlot=%s, " % self.xcvrSlot
      if self.productRe:
         argString += "productRe=%s, " % self.productRe
      if self.portRange:
         argString += "portRange=%s, " % self.portRange
      return "<InteractionReferenceId: %s>" % argString

################################################################################
# Reference Registry
#
# This is a central place to register InteractionReferenceIds to
# IntfInteractionReference-derived classes. The matching logic will favor the
# most specific RefId in the case that there are multiple matches.
Id = InteractionReferenceId  # will be used as key very often...
_intfInteractionReferenceRegistry = {
   # Jericho2 Template Types
   Id( set( ( "crt50216gb", "Blackhawk" ) ),
       XcvrType.qsfpPlus ): IxnRef.BabbageQsfp28Pair,
   Id( set( ( "cms42550gb", "Blackhawk" ) ),
       XcvrType.qsfpPlus ): IxnRef.BabbageQsfp28Pair,
   Id( set( ( "Blackhawk", ) ), XcvrType.qsfpPlus ): IxnRef.QsfpMixedRate,
   Id( set( ( "Blackhawk", ) ), XcvrType.osfp ): IxnRef.EightLaneIntfs,
   Id( set( ( "Blackhawk", ) ), XcvrType.qsfpDd ): IxnRef.EightLaneIntfs,

   # Strata Template Types
   Id( set( ( "crt50216gb", "tscbh" ) ),
       XcvrType.qsfpCmis ): IxnRef.BabbageQsfp28Pair,
   Id( set( ( "crt50216rt", "tscbh" ) ), XcvrType.osfp ): IxnRef.EightLaneIntfs,
   Id( set( ( "crt50216rt", "tscbh" ) ), XcvrType.qsfpDd ): IxnRef.EightLaneIntfs,
   Id( set( ( "tscbh", ) ), XcvrType.osfp ): IxnRef.EightLaneIntfs,
   Id( set( ( "tscbh", ) ), XcvrType.qsfpDd ): IxnRef.EightLaneIntfs,
   Id( set( ( "tscm", ) ), XcvrType.sfpPlus ): IxnRef.Sfp,

   # Non-L1-Topo Template Types (i.e. super-hacky-fun-stuff)

   # Capitola ports:
   # * Et1-6,31-42,67-72 are standard QSFP+
   # * Et7-10,27-30,43-46,63-66 are 40G-only QSFP+
   # * Et11-26,47-62 are Sesto2 port pairs; master-slave are odd-even pairs
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA?-C72",
       portRange=( range( 1, 7 ) + range( 31, 43 ) +
                   range( 67, 73 ) ) ): IxnRef.BasicQsfp,
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA?-C72",
       portRange=( range( 7, 11 ) + range( 27, 31 ) + range( 43, 47 ) +
                   range( 63, 67 ) ) ): IxnRef.Qsfp40gOnly,
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA?-C72",
       portRange=( range( 11, 27 ) + range( 47, 63 ) ) ): IxnRef.Sesto2Pair,

   # Nice ports:
   # * Et1-6,8,10,12,26,28,30-36 are standard QSFP+
   # * Et7,9,11,25,27,29 are standard QSFP28 (but use the same reference as QSFP+)
   # * Et13-24 are Sesto2 port pairs
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA-C36S",
       portRange=( range( 1, 13 ) + range( 25, 37 ) ) ): IxnRef.BasicQsfp,
   Id( xcvrSlot=XcvrType.qsfpPlus, productRe="DCS-7280QRA-C36S",
       portRange=( range( 13, 25 ) ) ): IxnRef.Sesto2Pair,
}

################################################################################
# API to match criteria with reference
def findMatch( refId ):
   match = None
   # Find a baseline first by matching against phys + xcvr-slot. Products that
   # don't support L1Topo are expected to have refId.phys==set().
   matchingKeys = [ k for k in _intfInteractionReferenceRegistry
                    if k.phys == refId.phys and k.xcvrSlot == refId.xcvrSlot ]
   if len( matchingKeys ) > 1:
      # Need to refine matches with additional criteria
      t0( refId, "initially matched to", matchingKeys )
      # Filter first based on product
      productReMatchKeys = [ key for key in matchingKeys
                             if re.match( key.productRe, refId.productRe ) ]
      if productReMatchKeys:
         t0( refId, "product matched to", productReMatchKeys[ 0 ].productRe )
      # Filter further on port-range
      portMatchedKeys = [ key for key in productReMatchKeys if
                          set( refId.portRange ).issubset( set( key.portRange ) ) ]
      if len( portMatchedKeys ) == 1:
         match = _intfInteractionReferenceRegistry[ portMatchedKeys[ 0 ] ]
   elif len( matchingKeys ) == 1:
      match = _intfInteractionReferenceRegistry[ matchingKeys[ 0 ] ]
   # The "else" case means that there is no match. In the grand scheme of things,
   # it means that serdes domains with this reference ID won't appear in the
   # command output.
   t0( refId, "matched to", match )
   return match

