# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin.GroupLib import Group
import CliPlugin.MaintenanceCliLib as MaintenanceCliLib
from CliPlugin.MaintenanceCliLib import linecardBuiltinGroupPrefix
from CliPlugin.MaintenanceCliLib import intfGroupType, builtinGroupType
from CliPlugin.MaintenanceGroupCli import MaintenanceGroup
from CliPlugin.MaintenanceMode import MaintenanceUnitConfigMode
from CliPlugin.MaintenanceModels import intfMaintenanceGroupCleanupHook
import ConfigMount
import Intf
import IntfGroupLib
import LazyMount
import MultiRangeRule
import Tac

globalIntfGroupConfigDir = None
globalIntfGroupStatusDir = None

#
# IntfBuiltinGroup class: Holds cli state for each configured builtin interface 
# group
#
class IntfBuiltinGroup( Group ):
   def __init__( self, groupName ):
      self.name_ = groupName
      groupKey = Tac.Value( 'Group::GroupKey', intfGroupType, groupName )
      Group.__init__( self, groupKey )
      globalIntfGroupConfigDir.newConfig( self.name_ )
      group = globalIntfGroupConfigDir.config.get( self.name_ )
      group.origin = builtinGroupType
      Group.addGroup( self )
      
   def name( self ):
      return self.name_

   def remGroup( self ):
      maintenanceGroup = MaintenanceGroup( self.key_ )
      maintenanceGroup.remGroup()
      if self.name_ in globalIntfGroupConfigDir.config:
         del globalIntfGroupConfigDir.config[ self.name_ ]
         Group.remGroup( self )
#
# IntfGroup class: Holds cli state for each configured interface group
#
class IntfGroup( Group ):
   def __init__( self, groupName ):
      self.name_ = groupName
      groupKey = Tac.Value( 'Group::GroupKey', intfGroupType, groupName )
      Group.__init__( self, groupKey )

   def addGroup( self ):
      globalIntfGroupConfigDir.newConfig( self.name_ )
      Group.addGroup( self )

   def remGroup( self ):
      maintenanceGroup = MaintenanceGroup( self.key_ )
      maintenanceGroup.remGroup()
      if self.name_ in globalIntfGroupConfigDir.config:
         del globalIntfGroupConfigDir.config[ self.name_ ]
         Group.remGroup( self )

   def name( self ):
      return self.name_

   def addMember_( self, intfId ):
      groupConfig = globalIntfGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      groupConfig.intf[ intfId ] = True

   def remMember_( self, intfId ):
      groupConfig = globalIntfGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      del groupConfig.intf[ intfId ]

   def addOrRemMember_( self, intfId, remove=False ):
      if not remove:
         self.addMember_( intfId )
      else:
         self.remMember_( intfId )

   def addOrRemIntf_( self, intfList, remove=False ):
      if isinstance( intfList, Intf.IntfRange.IntfList ):
         for intfName in intfList.intfNames():
            intfId = Tac.Value( "Arnet::IntfId", intfName )
            self.addOrRemMember_( intfId, remove )
      else:
         intfId = Tac.Value( "Arnet::IntfId", intfList.name )
         self.addOrRemMember_( intfId, remove )

   def addIntf( self, intfList ):
      self.addOrRemIntf_( intfList )

   def remIntf( self, intfList ):
      self.addOrRemIntf_( intfList, remove=True )

   def addProfile( self, profileKey ):
      groupConfig = globalIntfGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      groupConfig.profile[ profileKey ] = True

   def remProfile( self, profileType ):
      groupConfig = globalIntfGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      for profile in groupConfig.profile.keys():
         if profile.type == profileType:
            profileKey = Tac.Value( "Group::ProfileKey", type=profileType,
                                    name=profile.name )
            del groupConfig.profile[ profileKey ]

matcherGroup = CliMatcher.KeywordMatcher( 'group', helpdesc='Configure Group' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Configure Interface Group' )
intfGroupNameMatcher = MaintenanceCliLib.groupNameMatcher(
      lambda mode: globalIntfGroupConfigDir.config )

def _handleLinecardBuiltinGroup( mode, groupName ):
   linecardList = []
   intfBuiltinGroupList = []
   for linecard in groupName:
      linecardList.append( int( linecard.label ) )
      intfBuiltinGroup = IntfBuiltinGroup( linecard.name )
      intfBuiltinGroupList.append( intfBuiltinGroup )
   linecardModeName = linecardBuiltinGroupPrefix + \
                      MultiRangeRule.multiRangeToCanonicalString( linecardList )
   childMode = mode.childMode( IntfGroupLib.IntfBuiltinGroupConfigMode,
                               intfBuiltinGroup=intfBuiltinGroupList,
                               builtinModeName=linecardModeName )
   return childMode                               

def isLinecardBuiltinGroup( linecardGroupList ):
   """
   Checks whether the given linecardGroupList contains linecards or not
   """
   if isinstance( linecardGroupList, list ):
      for linecard in linecardGroupList:
         assert linecard.name.startswith( linecardBuiltinGroupPrefix )
      return True
   return False

def addIntfGroupConfig( mode, groupName ):
   """
   Handle group interface config add.
   """
   if isLinecardBuiltinGroup( groupName ):
      childMode = _handleLinecardBuiltinGroup( mode, groupName )
   elif groupName == MaintenanceCliLib.allEthIntfBuiltinGroupName:
      intfBuiltinGroup = IntfBuiltinGroup( groupName )
      childMode = mode.childMode( IntfGroupLib.IntfBuiltinGroupConfigMode, 
                                  intfBuiltinGroup=intfBuiltinGroup,
                                  builtinModeName=groupName )
   else:   
      intfGroup = IntfGroup( groupName )
      intfGroup.addGroup()
      childMode = mode.childMode( IntfGroupLib.IntfGroupConfigMode,
            intfGroup=intfGroup )
   mode.session_.gotoChildMode( childMode )

def _handleDelIntfGroupConfig( mode, groupName ):
   if( groupName.startswith( linecardBuiltinGroupPrefix ) or
       groupName == MaintenanceCliLib.allEthIntfBuiltinGroupName ):
      intfBuiltinGroup = IntfBuiltinGroup( groupName )
      intfBuiltinGroup.remGroup()
   else:
      intfGroup = IntfGroup( groupName )
      intfGroup.remGroup()

def delIntfGroupConfig( mode, groupName ):
   """
   Handle group interface config deletion.
   """
   if isLinecardBuiltinGroup( groupName ):
      for linecard in groupName:
         _handleDelIntfGroupConfig( mode, linecard.name )
   else:
      _handleDelIntfGroupConfig( mode, groupName )

class GroupNameExpr( CliCommand.CliExpression ):
   expression = 'AllEthernetInterface | GROUP'
   data = {
      'AllEthernetInterface' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            MaintenanceCliLib.allEthIntfBuiltinGroupName,
            helpdesc='AllEthernetInterface builtin group' ),
         alias='GROUP' ),
      'GROUP' : intfGroupNameMatcher
   }

#--------------------------------------------------------------------------------
# [ no | default ] group interface GROUP
#--------------------------------------------------------------------------------
# Linecard Interface group name rule is added in MaintenanceMode package to avoid
# circular dependency between Intf and Fru package having Linecard slots in system.
class GroupInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'group interface GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'group' : matcherGroup,
      'interface' : matcherInterface,
      'GROUP' : GroupNameExpr,
   }

   @staticmethod
   def handler( mode, args ):
      return addIntfGroupConfig( mode, args[ 'GROUP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return delIntfGroupConfig( mode, args[ 'GROUP' ] )

BasicCli.GlobalConfigMode.addCommandClass( GroupInterfaceCmd )

def _addGroup( mode, groupName ):
   groupKey = Tac.Value( "Group::GroupKey", type=intfGroupType, name=groupName )
   mode.maintenanceUnit_.addGroup( groupKey )

def _remGroup( mode, groupName ):
   groupKey = Tac.Value( "Group::GroupKey", type=intfGroupType, name=groupName )
   mode.maintenanceUnit_.remGroup( groupKey )

def addLinecardGroupToUnit( mode, linecardGroupName ):
   """
   Add Builtin Linecard Group names to the unit config
   """
   assert isLinecardBuiltinGroup( linecardGroupName )
   for linecard in linecardGroupName:
      _addGroup( mode, linecard.name )

def remLinecardGroupFromUnit( mode, linecardGroupName ):
   """
   Remove Builtin Linecard Group names to the unit config
   """
   assert isLinecardBuiltinGroup( linecardGroupName )
   for linecard in linecardGroupName:
      _remGroup( mode, linecard.name )

# Linecard Interface group name rule is added in MaintenanceMode package to avoid
# circular dependency between Intf and Fru package having Linecard slots in system.
class MaintenanceGroupInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'group interface GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'group' : matcherGroup,
      'interface' : matcherInterface,
      'GROUP' : GroupNameExpr,
   }

   @staticmethod
   def handler( mode, args ):
      return _addGroup( mode, args[ 'GROUP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return _remGroup( mode, args[ 'GROUP' ] )

MaintenanceUnitConfigMode.addCommandClass( MaintenanceGroupInterfaceCmd )

def Plugin( entityManager ):
   global globalIntfGroupConfigDir
   global globalIntfGroupStatusDir
   intfMaintenanceGroupCleanupHook.addExtension( delIntfGroupConfig )
   globalIntfGroupConfigDir = ConfigMount.mount( entityManager,
      'group/config/interface', 'IntfGroup::ConfigDir', 'w' )
   globalIntfGroupStatusDir = LazyMount.mount( entityManager,
      'group/status/interface', 'IntfGroup::StatusDir', 'r' )

